#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvrecbin.h"
#include "edvrecbinfio.h"
#include "edvrecbinfop.h"
#include "endeavour.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"

#include "images/pdi_file01_20x20.xpm"
#include "images/pdi_file02_20x20.xpm"
#include "images/pdi_file03_20x20.xpm"
#include "images/pdi_file04_20x20.xpm"
#include "images/pdi_file05_20x20.xpm"
#include "images/pdi_file06_20x20.xpm"
#include "images/pdi_folder_32x32.xpm"
#include "images/pdi_folderfile_32x32.xpm"
#include "images/pdi_trash01_20x20.xpm"
#include "images/pdi_trash02_20x20.xpm"
#include "images/pdi_trash03_20x20.xpm"
#include "images/pdi_trash04_20x20.xpm"
#include "images/pdi_trash05_20x20.xpm"
#include "images/pdi_trash06_20x20.xpm"

#include "images/icon_trash_32x32.xpm"
#include "images/icon_trash_empty_32x32.xpm"


/*
 *      Return values have the following error meanings:
 *
 *      0       Success (no error)
 *      -1      General error
 *      -2      Ambiguous, not found, other error.
 *      -3      Systems error (out of memory/out of disk space)
 *      -4      User responded with "Cancel"
 *      -5      User responded with "No" or response is not available.
 *      -6      Call would cause reentry.
 */

const gchar *EDVRecBinFOPGetError();

/*
static gbool EDVRecBinObjectsSame(
        const struct stat *src_stat_buf,
        const struct stat *tar_stat_buf
);
 */
static gbool EDVRecBinNoDeleteCheck(
	edv_core_struct *core_ptr, GtkWidget *toplevel,
	gbool interactive, gbool *yes_to_all,
	const gchar *path, struct stat *stat_buf
);

static void EDVRecBinFOPMapProgressDialogRecover(
	const gchar *label, gfloat progress_value, GtkWidget *toplevel,
	gbool force_remap
);
static void EDVRecBinFOPMapProgressDialogDelete(
        const gchar *label, gfloat progress_value, GtkWidget *toplevel,
	gbool force_remap
);
static void EDVRecBinFOPMapProgressDialogPurge(
        const gchar *label, gfloat progress_value, GtkWidget *toplevel,
	gbool force_remap
);
static gint EDVRecBinFOPRecycleProgressPercentCB(
        gpointer client_data, gfloat percent
);
static gint EDVRecBinFOPRecycleProgressCB(
        gpointer client_data, gulong pos, gulong total
);

gint EDVRecBinFOPRecover(
        edv_core_struct *core_ptr,
	guint index,			/* Recycled object to recover. */
	const gchar *tar_obj,		/* Can be NULL. */
	gchar **new_obj_rtn,		/* Dynamically allocated. */
	GtkWidget *toplevel,		/* Can be NULL. */
        gbool show_progress, gbool interactive,
        gbool *yes_to_all
);

static guint EDVRecBinFOPDeleteIterate(
        edv_core_struct *core_ptr,
        const gchar *recycled_index_file,
        const gchar *path,
        guint **index_rtn, gint *total_index_rtns,
        GtkWidget *toplevel,            /* Can be NULL. */
        gbool show_progress, gbool interactive,
        gbool *yes_to_all, gint *status, gint delete_method
);
gint EDVRecBinFOPDelete(
	edv_core_struct *core_ptr,
	const gchar *path,		/* Disk object to delete. */
        guint **index_rtn, gint *total_index_rtns,
	GtkWidget *toplevel,		/* Can be NULL. */
	gbool show_progress, gbool interactive,
	gbool *yes_to_all
);
gint EDVRecBinFOPPurge(
        edv_core_struct *core_ptr,
        guint index,
        GtkWidget *toplevel,            /* Can be NULL. */
        gfloat progress_value,          /* Can be -1.0. */
        gbool show_progress, gbool interactive,
        gbool *yes_to_all
);
gint EDVRecBinFOPPurgeAll(
        edv_core_struct *core_ptr,
        GtkWidget *toplevel,            /* Can be NULL. */
        gbool show_progress, gbool interactive,
        gbool *yes_to_all
);

static const gchar *last_error = NULL;


/*
 *      Returns the last error message or NULL if there was no error.
 *
 *      The returned pointer must not be deallocated.
 */
const gchar *EDVRecBinFOPGetError()
{
        return(last_error);
}


#if 0
/*
 *      Checks if the stats of the two objects are the same by checking
 *      both their devices and inodes.
 */
static gbool EDVRecBinObjectsSame(
        const struct stat *src_stat_buf,
        const struct stat *tar_stat_buf
)
{
        if((src_stat_buf == NULL) || (tar_stat_buf == NULL))
            return(FALSE);

        if((src_stat_buf->st_dev == tar_stat_buf->st_dev) &&
           (src_stat_buf->st_ino == tar_stat_buf->st_ino)
        )
            return(TRUE);
        else
            return(FALSE);
}
#endif


/*
 *	Checks if the given object specified by path and stat_buf may
 *	not be deleted because it is a critical part of either this
 *	process or the system's function.
 *
 *	Returns TRUE if the object may not be deleted.
 */
static gbool EDVRecBinNoDeleteCheck(
        edv_core_struct *core_ptr, GtkWidget *toplevel,
        gbool interactive, gbool *yes_to_all,
        const gchar *path, struct stat *stat_buf
)
{
typedef struct {
 gchar *path;
 gchar *message;
 gint error_level;
} no_delete_struct;
	gint i;
	gbool no_delete;
	const gchar *cstrptr, *cstrptr2;
	no_delete_struct *nd_ptr = NULL;
	no_delete_struct no_delete_list[] = EDV_NO_DELETE_LIST;


	/* Check for valid inputs, return TRUE if they are invalid to
	 * prevent calling operation from proceeding.
	 */
	if((core_ptr == NULL) || (path == NULL))
	    return(TRUE);

	/* Iterate through no delete list, checking if an entry has a
	 * path that matches the given path.
	 */
	i = 0;
	no_delete = FALSE;
	while(no_delete_list[i].path != NULL)
	{
	    nd_ptr = &no_delete_list[i];

	    /* Given object path matches one in the no delete list? */
	    if(!strcmp(nd_ptr->path, path))
	    {
		/* Handle by error level of this no delete entry. */
		switch(nd_ptr->error_level)
		{
		  case 0:	/* Only warn user. */
		    if(interactive)
		    {


		    }
		    break;

		  case 1:	/* Query user for no delete. */
                    if(interactive && !(*yes_to_all))
                    {
			gint status;
                        gchar *buf = g_strdup_printf(
"The object `%s' should not be deleted, reason:\n\
\n\
%s.\n\
\n\
Are you sure you want to delete this object?",
                            path,
                            (nd_ptr->message != NULL) ?
                                nd_ptr->message :
                                "Object is marked `no_delete' by this process"
                        );

                        CDialogSetTransientFor(toplevel);
                        status = CDialogGetResponse(
                            "Delete Warning", buf, NULL,
                            CDIALOG_ICON_ERROR,
                            CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_YES_TO_ALL |
			    CDIALOG_BTNFLAG_NO,
			    CDIALOG_BTNFLAG_NO
                        );
                        CDialogSetTransientFor(NULL);

                        g_free(buf);

			switch(status)
			{
			  case CDIALOG_RESPONSE_YES_TO_ALL:
			    *yes_to_all = TRUE;
			  case CDIALOG_RESPONSE_YES:
			    break;

			  default:
			    no_delete = TRUE;
			    break;
			}
                    }
		    else
		    {
			no_delete = TRUE;
		    }
		    break;

		  default:	/* No delete under any case. */
		    no_delete = TRUE;
		    if(interactive)
		    {
			gchar *buf = g_strdup_printf(
"The object `%s' cannot be deleted, reason:\n\
\n\
%s.\n",
			    path,
			    (nd_ptr->message != NULL) ?
				nd_ptr->message :
				"Object is marked `no_delete' by this process"
			);

			CDialogSetTransientFor(toplevel);
			CDialogGetResponse(
			    "Delete Failed", buf, NULL,
			    CDIALOG_ICON_ERROR,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);

			g_free(buf);
		    }
		    break;
		}
		/* Stop itterating through no delete list and return
		 * response.
		 */
		return(no_delete);
	    }

	    i++;
	}


	/* Check for other runtime paths. */

        /* Special notation? */
	cstrptr = strrchr(path, DIR_DELIMINATOR);
	if(cstrptr != NULL)
	    cstrptr++;
	if(cstrptr != NULL)
	{
	    if(!strcmp(cstrptr, ".") ||
               !strcmp(cstrptr, "..")
	    )
	    {
                if(interactive)
                {
                    gchar *buf = g_strdup_printf(
"Special object notations such as `%s' may not be\n\
used to reffer to an object that is to be deleted.",
			cstrptr
		    );

                    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        "Delete Failed", buf, NULL,
                        CDIALOG_ICON_ERROR,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);

                    g_free(buf);
                }
                return(TRUE);
	    }
	}

	/* User's home directory? */
	cstrptr = getenv("HOME");
	if(cstrptr != NULL)
	{
	    if(!strcmp(cstrptr, path))
	    {
                if(interactive)
                {
                    gchar *buf = g_strdup_printf(
"The object `%s' cannot be deleted, reason:\n\
\n\
User's home directory.\n",
                        path
		    );

		    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        "Delete Failed", buf, NULL,
                        CDIALOG_ICON_ERROR,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);

		    g_free(buf);
		}
		return(TRUE);
	    }
	}

        /* Object is a recycled objects system compoent object?
	 *
	 * Check this by getting the parent of the recycled objects
	 * index file. This parent directory will be checked to see if it
	 * is the parent of the given object. If it is then we know that
	 * the given object is in within the recycled objects directory
	 * and thus an object of the recycled objects system which cannot
	 * be deleted.
	 */
        cstrptr = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );
	cstrptr2 = GetParentDir(cstrptr);
        if(cstrptr2 != NULL)
        {
            if(EDVIsParentPath(cstrptr2, path))
            {
                if(interactive)
                {
                    gchar *buf = g_strdup_printf(
"The object `%s' cannot be deleted, reason:\n\
\n\
Object is used in the recycled objects system.\n",
                        path
                    );

                    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        "Delete Failed", buf, NULL,
                        CDIALOG_ICON_ERROR,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);

                    g_free(buf);
                }
                return(TRUE);
            }
        }




	return(FALSE);
}


/*
 *      Maps the progress dialog as needed in animation mode for recover.
 */
static void EDVRecBinFOPMapProgressDialogRecover(
        const gchar *label, gfloat progress_value, GtkWidget *toplevel,
	gbool force_remap
)
{
        u_int8_t        **start_icon_data[3],
                        **icon_data[6],
                        **end_icon_data[3];

        /* Already mapped? */
        if(ProgressDialogIsQuery())
        {
            /* Check if the progress dialog needs to be unmapped and
             * remapped again.
             */
            if(force_remap)
            {
                ProgressDialogBreakQuery(FALSE);
            }
            else
            {
                /* Already mapped and does not need unmapping, so just
                 * update the progress message.
                 */
                ProgressDialogUpdate(
                    NULL, label, NULL, NULL,
                    progress_value, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
                );
                return;
            }
        }

        ProgressDialogSetTransientFor(toplevel);

        start_icon_data[0] = (u_int8_t **)icon_trash_32x32_xpm;
        start_icon_data[1] = (u_int8_t **)icon_trash_32x32_xpm;
        start_icon_data[2] = (u_int8_t **)icon_trash_32x32_xpm;
        icon_data[0] = (u_int8_t **)pdi_file01_20x20_xpm;
        icon_data[1] = (u_int8_t **)pdi_file02_20x20_xpm;
        icon_data[2] = (u_int8_t **)pdi_file03_20x20_xpm;
        icon_data[3] = (u_int8_t **)pdi_file04_20x20_xpm;
        icon_data[4] = (u_int8_t **)pdi_file05_20x20_xpm;
        icon_data[5] = (u_int8_t **)pdi_file06_20x20_xpm;
        end_icon_data[0] = (u_int8_t **)pdi_folder_32x32_xpm;
        end_icon_data[1] = (u_int8_t **)pdi_folder_32x32_xpm;
        end_icon_data[2] = (u_int8_t **)pdi_folderfile_32x32_xpm;

        ProgressDialogMapAnimation(
            "Recovering",
            label,
            "Stop",
            start_icon_data, 3,
            icon_data, 6,
            end_icon_data, 3,
            500,
            10000
        );
        ProgressDialogUpdate(
            NULL, NULL, NULL, NULL,
            progress_value, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
        );
}

/*
 *      Maps the progress dialog as needed in animation mode for deleting.
 */
static void EDVRecBinFOPMapProgressDialogDelete(
        const gchar *label, gfloat progress_value, GtkWidget *toplevel,
	gbool force_remap
)
{
        u_int8_t        **start_icon_data[3],
                        **icon_data[6],
                        **end_icon_data[3];

        /* Already mapped? */
        if(ProgressDialogIsQuery())
        {
            /* Check if the progress dialog needs to be unmapped and
             * remapped again.
             */
            if(force_remap)
            {
                ProgressDialogBreakQuery(FALSE);
            }
            else
            {
                /* Already mapped and does not need unmapping, so just
                 * update the progress message.
                 */
                ProgressDialogUpdate(
                    NULL, label, NULL, NULL,
                    progress_value, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
                );
                return;
            }
        }

        ProgressDialogSetTransientFor(toplevel);

        start_icon_data[0] = (u_int8_t **)pdi_folderfile_32x32_xpm;
        start_icon_data[1] = (u_int8_t **)pdi_folder_32x32_xpm;
        start_icon_data[2] = (u_int8_t **)pdi_folder_32x32_xpm;
        icon_data[0] = (u_int8_t **)pdi_file01_20x20_xpm;
        icon_data[1] = (u_int8_t **)pdi_file02_20x20_xpm;
        icon_data[2] = (u_int8_t **)pdi_file03_20x20_xpm;
        icon_data[3] = (u_int8_t **)pdi_file04_20x20_xpm;
        icon_data[4] = (u_int8_t **)pdi_file05_20x20_xpm;
        icon_data[5] = (u_int8_t **)pdi_file06_20x20_xpm;
        end_icon_data[0] = (u_int8_t **)icon_trash_32x32_xpm;
        end_icon_data[1] = (u_int8_t **)icon_trash_32x32_xpm;
        end_icon_data[2] = (u_int8_t **)icon_trash_32x32_xpm;

        ProgressDialogMapAnimation(
            "Deleting",
            label,
            "Stop",
            start_icon_data, 3,
            icon_data, 6,
            end_icon_data, 3,
            500,
            10000
        );
        ProgressDialogUpdate(
            NULL, NULL, NULL, NULL,
            progress_value, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
        );
}

/*
 *      Maps the progress dialog as needed in animation mode for purging.
 */
static void EDVRecBinFOPMapProgressDialogPurge(
        const gchar *label, gfloat progress_value, GtkWidget *toplevel,
	gbool force_remap
)
{
        u_int8_t        **start_icon_data[3],
                        **icon_data[6],
                        **end_icon_data[3];

        /* Already mapped? */
        if(ProgressDialogIsQuery())
        {
            /* Check if the progress dialog needs to be unmapped and
             * remapped again.
             */
            if(force_remap)
            {
                ProgressDialogBreakQuery(FALSE);
            }
            else
            {
                /* Already mapped and does not need unmapping, so just
                 * update the progress message.
                 */
                ProgressDialogUpdate(
                    NULL, label, NULL, NULL,
                    progress_value, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
                );
                return;
            }
        }

        ProgressDialogSetTransientFor(toplevel);

        start_icon_data[0] = (u_int8_t **)icon_trash_32x32_xpm;
        start_icon_data[1] = (u_int8_t **)icon_trash_empty_32x32_xpm;
        start_icon_data[2] = (u_int8_t **)icon_trash_empty_32x32_xpm;
        icon_data[0] = (u_int8_t **)pdi_trash01_20x20_xpm;
        icon_data[1] = (u_int8_t **)pdi_trash02_20x20_xpm;
        icon_data[2] = (u_int8_t **)pdi_trash03_20x20_xpm;
        icon_data[3] = (u_int8_t **)pdi_trash04_20x20_xpm;
        icon_data[4] = (u_int8_t **)pdi_trash05_20x20_xpm;
        icon_data[5] = (u_int8_t **)pdi_trash06_20x20_xpm;
        end_icon_data[0] = (u_int8_t **)NULL;
        end_icon_data[1] = (u_int8_t **)NULL;
        end_icon_data[2] = (u_int8_t **)NULL;

        ProgressDialogMapAnimation(
            "Purging",
            label,
            "Stop",
            start_icon_data, 3,
            icon_data, 6,
            end_icon_data, 3,
            500,
            10000
        );
        ProgressDialogUpdate(
            NULL, NULL, NULL, NULL,
            progress_value, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
        );
}


/*
 *	Recycle recover or delete progress callback.
 */
static gint EDVRecBinFOPRecycleProgressPercentCB(
        gpointer client_data, gfloat percent
)
{
	gint status = 0;


	if(ProgressDialogIsQuery())
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		percent,
		EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		status = -1;
	}

	return(status);
}

static gint EDVRecBinFOPRecycleProgressCB(
        gpointer client_data, gulong pos, gulong total
)
{
	return(EDVRecBinFOPRecycleProgressPercentCB(
	    client_data,
	    ((total > 0) && (pos >= 0)) ?
		((gfloat)pos / (gfloat)total) : -1.0
	));
}


/*
 *	Recovers the recycled object specified by the given index and
 *	recreates it as a disk object. On success *new_obj_rtn will be
 *	a dynamically allocated string containing the newly created
 *	disk object.
 *
 *	The calling function must deallocate the returned string in
 *	*new_obj_rtn.
 */
gint EDVRecBinFOPRecover(
        edv_core_struct *core_ptr,
        guint index,                    /* Recycled object to recover. */
        const gchar *tar_obj,           /* Can be NULL. */
        gchar **new_obj_rtn,            /* Dynamically allocated. */
        GtkWidget *toplevel,            /* Can be NULL. */
        gbool show_progress, gbool interactive,
        gbool *yes_to_all
)
{
	static gint status;
	const gchar *cstrptr;
	const gchar *recycled_index_file;
	gchar *ltar_obj = NULL;
        gulong time_start = (gulong)time(NULL);
	edv_recbin_object_struct *obj = NULL;
	struct stat lstat_buf;


	/* Reset last error message. */
	last_error = NULL;


	if((core_ptr == NULL) || (index == 0))
	{
	    last_error = "Bad input value";
	    return(-1);
	}

#define DO_FREE_LOCALS	\
{ \
 g_free(ltar_obj); \
 ltar_obj = NULL; \
\
 EDVRecBinObjectDelete(obj); \
 obj = NULL; \
}

        /* Get path to recycled objects index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );
        if(recycled_index_file == NULL)
	{
	    DO_FREE_LOCALS
	    last_error = "Unable to get recycled objects index file";
            return(-1);
	}

	/* Get statistics of recycled object. */
	obj = EDVRecBinObjectStat(recycled_index_file, index);
	if(obj == NULL)
	{
	    DO_FREE_LOCALS
            last_error = "Unable to get statistics of recycled object";
            return(-1);
	}


	/* Formulate full target path. */
	if(tar_obj == NULL)
	{
	    /* Recover to target path not given so generate one from
	     * the original path.
	     */
	    if((obj->original_path == NULL) || (obj->name == NULL))
	    {
                DO_FREE_LOCALS
                last_error =
"Unable to get original path of recycled object";
		return(-1);
	    }

	    cstrptr = PrefixPaths(obj->original_path, obj->name);
	    ltar_obj = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	}
	else
	{
	    /* Given recovery path is not NULL, it is assumed to be a
	     * directory so postfix the recycled object's name to it.
	     */
	    cstrptr = PrefixPaths(tar_obj, obj->name);
	    ltar_obj = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
	}
	if(ltar_obj == NULL)
	{
	    DO_FREE_LOCALS
	    last_error =
"Unable to generate recovery path of recycled object";
	    return(-1);
	}


	/* Target object must not exist. */
	if(!lstat(ltar_obj, &lstat_buf))
	{
            DO_FREE_LOCALS
            last_error =
"An object with the same name already exists at the specified recovery location";
            return(-1);
	}


	if(show_progress)
	{
	    gchar *p1 = EDVCopyShortenPath(
		obj->name, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
	    );
	    gchar *p2 = EDVCopyShortenPath(
		ltar_obj, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
	    );
	    gchar *buf = g_strdup_printf(
"Recovering:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n",
		p1, p2
	    );

	    EDVRecBinFOPMapProgressDialogRecover(
		buf, 0.0, toplevel, FALSE
	    );

	    g_free(buf);
	    g_free(p1);
	    g_free(p2);
	}


	/* Do recovery. */
	status = EDVRecBinDiskObjectRecover(
	    recycled_index_file,
	    index,
	    ltar_obj,
	    show_progress ? EDVRecBinFOPRecycleProgressCB : NULL,
	    core_ptr
	);
	if(status == -4)
	{
	    /* User aborted recovery. */
	}
        else if(status)
        {
	    /* Recovery failed. */
	    last_error = EDVRecBinFIOGetError();
        }
        else
        {
            /* Remove recycled object entry from the recycled objects
             * index file.
             */
            EDVRecBinIndexRemove(
                recycled_index_file, index
            );

	    /* Update newly recovered object path. */
	    if(new_obj_rtn != NULL)
	    {
		g_free(*new_obj_rtn);
		*new_obj_rtn = ltar_obj;
		ltar_obj = NULL;
	    }
        }

        /* Record history. */
        EDVAppendHistory(
            core_ptr,
	    EDV_HISTORY_RECYCLED_OBJECT_RECOVER,
            time_start, (gulong)time(NULL),
            status,
            obj->name,		/* Source name. */
            (new_obj_rtn != NULL) ? *new_obj_rtn : ltar_obj,
            last_error          /* Use last_error as comment if not NULL. */
        );


        /* Deallocate locally allocated memory. */
	DO_FREE_LOCALS
#undef DO_FREE_LOCALS

	return(status);
}


/*
 *	Called by EDVRecBinFOPDelete() to delete the object specified by
 *	path to the recycled objects directory.
 *
 *	Returns the index of the recycled object or 0 if no recycled
 *	object index is available, this can occure if the delete
 *	method is set to purge.
 */
static guint EDVRecBinFOPDeleteIterate(
        edv_core_struct *core_ptr,
	const gchar *recycled_index_file,
	const gchar *path,
	guint **index_rtn, gint *total_index_rtns,
	GtkWidget *toplevel,            /* Can be NULL. */
        gbool show_progress, gbool interactive,
        gbool *yes_to_all, gint *status,
	gint delete_method
)
{
	gbool no_delete;
	guint index;
	gint strc;
	gchar **strv;
	const gchar *cstrptr;
	struct stat lstat_buf, stat_buf;


	/* Error or user aborted from prior call? */
	if(*status)
	    return(0);

        /* Get local stats of the given disk object to be deleted. */
        if(lstat(path, &lstat_buf))
        {
            *status = -2;
            last_error =
"Unable to get statistics of the object to be deleted";
            return(0);
        }

#if 0
        /* If effective user ID of this process does not own object then
         * the object cannot be deleted.
         */
        if(!EDVCheckIsOwner(core_ptr, lstat_buf.st_uid))
        {
            *status = -2;
            last_error =
"Effective user id of this process does not own the object";
            return(0);
        }
#endif

        /* Get destination statistics of this object and check if it
         * may not be deleted.
         */
        if(stat(path, &stat_buf))
            no_delete = EDVRecBinNoDeleteCheck(
                core_ptr, toplevel, interactive, yes_to_all, path, NULL
            );
        else
            no_delete = EDVRecBinNoDeleteCheck(
                core_ptr, toplevel, interactive, yes_to_all, path, &stat_buf
            );
        if(no_delete)
        {
            *status = -2;
	    /* Set last error only if not interactive since calling
	     * EDVRecBinNoDeleteCheck() would have already displayed 
	     * appropriate error message.
	     */
	    if(!interactive)
                last_error =
"Object is part of the function of this process or system and may not be deleted";
            return(0);
        }


        /* If the given object to be deleted is locally a directory, then
	 * get contents of the directory and delete recursivly.
	 */
	if(S_ISDIR(lstat_buf.st_mode))
	    strv = GetDirEntNames2(path, &strc);
	else
	    strv = NULL;
        if(strv != NULL)
	{
            gint i;
            const gchar *cstrptr2;
            gchar *child_path;


            /* Sort strings. */
            strv = StringQSort(strv, strc);
            if(strv == NULL)
            {
		*status = -2;
                last_error = "Unable to sort directory entries";
                return(0);
            }


            /* Iterate through each string. */
            child_path = NULL;
            for(i = 0; i < strc; i++)
            {
#define FREE_AND_CONTINUE	{	\
 g_free(child_path);			\
 child_path = NULL;			\
 g_free(strv[i]);			\
 strv[i] = NULL;			\
 continue;				\
}
                cstrptr = strv[i];
		if(cstrptr == NULL)
		    FREE_AND_CONTINUE

                /* Error encountered or user abort? */
                if(*status)
                    FREE_AND_CONTINUE

                /* Skip special dir notations. */
                if(!strcmp(cstrptr, "..") ||
                   !strcmp(cstrptr, ".")
                )
                    FREE_AND_CONTINUE

                /* Formulate full path to child. */
                cstrptr2 = PrefixPaths(path, cstrptr);
                if(cstrptr2 == NULL)
                    FREE_AND_CONTINUE
                child_path = g_strdup(cstrptr2);


		/* Delete this object and all its children. */
		index = EDVRecBinFOPDeleteIterate(
		    core_ptr, recycled_index_file, child_path,
		    index_rtn, total_index_rtns,
		    toplevel, show_progress, interactive,
		    yes_to_all, status, delete_method
		);
		/* Above return index is ignored. */

                FREE_AND_CONTINUE
#undef FREE_AND_CONTINUE
            }

            /* Deallocate pointer array to directory entry strings. */
	    g_free(strv);
	    strv = NULL;
	    strc = 0;
	}

        /* Error or user aborted while deleting any child objects? */
        if(*status)
            return(0);

	/* Update progress to display the path of this object that is
	 * about to be deleted?
	 */
	if(show_progress)
	{
	    gchar *p1 = EDVCopyShortenPath(
		path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
	    );
	    gchar *buf = g_strdup_printf(
"Deleting:\n\
\n\
    %s\n",
		p1
	    );

	    EDVRecBinFOPMapProgressDialogDelete(
		buf, 0.0, toplevel, FALSE
	    );

	    g_free(buf);
	    g_free(p1);

	    if(ProgressDialogStopCount() > 0)
	    {
		*status = -4;
		return(0);
	    }
	}

	/* At this point, check if we need to delete the object by
	 * moving it to the recycled objects directory or permanently
	 * purging the object.
	 */
	if(delete_method == EDV_DELETE_METHOD_PURGE)
	{
	    /* Permanently purge this object. */

            /* Get system permissions of object. */
            mode_t m = lstat_buf.st_mode;

	    /* Delete by object type. */
	    /* Directory? */
	    if(S_ISDIR(m))
	    {
                if(rmdir(path))
                {
                    switch(errno)
                    {
                      case ENOTEMPTY:
                        last_error = "Directory is not empty";
                        break;
                      default:
                        last_error = "Unable to remove directory";
                        break;
                    }
		    *status = -1;
                }
	    }
	    /* All else, assume it can be unlink'ed. */
	    else
	    {
                if(unlink(path))
                {
                    switch(errno)
                    {
                      case EFAULT:
                        last_error = "Segmentation fault";
                        break;
                      case EACCES:
                        last_error =
 "Object's permissions or location does not permit it to be deleted";
                        break;
                      case EPERM:
                        last_error =
 "You do not own that object in the tempory files location";
                        break;
                      case ENAMETOOLONG:
                        last_error = "Object's path name is too long";
                        break;
                      case ENOENT:
                        last_error =
 "A compoent of the object's path does not exist";
                        break;
                      case ENOTDIR:
                        last_error =
 "A compoent of the object's path is not a directory";
                        break;
                      case EISDIR:
                        last_error =
 "Object's path reffers to a directory object";
                        break;
                      case ENOMEM:
                        last_error = "System is out of memory";
                        break;
                      case EROFS:
                        last_error =
 "The object exists on a read-only filesystem";
                        break;
                      case ELOOP:
                        last_error =
 "Too many symbolic links encountered in the object's path";
                        break;
                      case EIO:
                        last_error = "An input/output error occured";
                        break;
                      default:
                        last_error = "Unable to delete file";
                        break;
                    }
                    *status = -1;
	        }
	    }

            index = 0;
	    if(*status == 0)
	    {
                    /* Append this index to the index array return. */
                    if((index_rtn != NULL) && (total_index_rtns != NULL))
                    {
                        gint i;

                        if(*total_index_rtns < 0)
                            *total_index_rtns = 0;

                        i = *total_index_rtns;
                        *total_index_rtns = i + 1;

                        *index_rtn = (guint *)g_realloc(
                            *index_rtn,
                            *total_index_rtns * sizeof(guint)
                        );
                        if(*index_rtn == NULL)
                        {
                            *total_index_rtns = 0;
                        }
                        else
                        {
                            (*index_rtn)[i] = index;
                        }
                    }
	    }
	}
	else
	{
	    /* All else, move this object to the recycled objects
	     * directory.
	     */

	    /* Get system permissions of object. */
	    mode_t m = lstat_buf.st_mode;

	    /* Create a new tempory recycled object structure. */
	    edv_recbin_object_struct *obj = EDVRecBinObjectNew();
	    if(obj == NULL)
		return(0);

	    /* Set values from object to new recycled object structure. */
	    cstrptr = strrchr(path, DIR_DELIMINATOR);
	    if(cstrptr == NULL)
		cstrptr = path;
	    else
		cstrptr++;
	    g_free(obj->name);
	    obj->name = g_strdup(cstrptr);

	    cstrptr = GetParentDir(path);
	    if(cstrptr == NULL)
		cstrptr = "/";
	    g_free(obj->original_path);
	    obj->original_path = g_strdup(cstrptr);

	    obj->date_deleted = time(NULL);

	    obj->type = EDVObjectGetTypeFromStatMode(m);

	    obj->permissions = EDVObjectGetPermissionsFromStatMode(m);

	    obj->access_time = lstat_buf.st_atime;
	    obj->modify_time = lstat_buf.st_mtime;
	    obj->change_time = lstat_buf.st_ctime;

	    obj->owner_id = lstat_buf.st_uid;
	    obj->group_id = lstat_buf.st_gid;

	    obj->size = lstat_buf.st_size;

	    /* Add object to recycled index file. */
	    index = EDVRecBinIndexAdd(recycled_index_file, obj);
	    if(index != 0)
	    {
		/* Remove the actual disk object and place it into the
		 * recycled objects directory.
		 */
		*status = EDVRecBinDiskObjectDelete(
		    recycled_index_file,
		    index,
		    path,
		    show_progress ? EDVRecBinFOPRecycleProgressCB : NULL,
		    core_ptr
		);
		/* User aborted operation? */
		if(*status == -4)
		{
		    EDVRecBinIndexRemove(recycled_index_file, index);
		    index = 0;
		}
		/* Other error occured? */
		else if(*status)
		{
		    EDVRecBinIndexRemove(recycled_index_file, index);
		    index = 0;
		    last_error = EDVRecBinFIOGetError();
		}
		/* Successfully deleted object. */
		else
		{
		    /* Append this index to the index array return. */
		    if((index_rtn != NULL) && (total_index_rtns != NULL))
		    {
			gint i;

			if(*total_index_rtns < 0)
			    *total_index_rtns = 0;

			i = *total_index_rtns;
			*total_index_rtns = i + 1;

			*index_rtn = (guint *)g_realloc(
			    *index_rtn,
			    *total_index_rtns * sizeof(guint)
			);
			if(*index_rtn == NULL)
			{
			    *total_index_rtns = 0;
			}
			else
			{
			    (*index_rtn)[i] = index;
			}
		    }
		}
	    }
	    else
	    {
		last_error = "Unable to generate new recycled object index";
		*status = -1;
	    }

	    /* Delete tempory recycled object structure, it is no
	     * longer needed.
	     */
	    EDVRecBinObjectDelete(obj);
	}

	return(index);
}

/*
 *	Deletes the disk object specified by the full path and places it
 *	into the recycle bin. On success *index_rtn will be set to a
 *	dynamically allocated string containing an array of recycled
 *	object indexes.
 *
 *      The calling function must deallocate the returned array in
 *      *index_rtn.
 */
gint EDVRecBinFOPDelete(
        edv_core_struct *core_ptr,
        const gchar *path,              /* Disk object to delete. */
        guint **index_rtn, gint *total_index_rtns,
        GtkWidget *toplevel,            /* Can be NULL. */
        gbool show_progress, gbool interactive,
        gbool *yes_to_all
)
{
        static gint status;
	guint index;
	const gchar *recycled_index_file;
	gint delete_method;
        gulong time_start = (gulong)time(NULL);
	gchar *lsrc_obj = NULL;
	gchar num_str[80];


        /* Reset last error message. */
        last_error = NULL;


	/* Reset returns. */
	if(index_rtn != NULL)
	    *index_rtn = NULL;
	if(total_index_rtns != NULL)
	    *total_index_rtns = 0;


        if((core_ptr == NULL) || (path == NULL))
        {
            last_error = "Bad input value";
            return(-1);
        }

#define DO_FREE_LOCALS  \
{ \
 g_free(lsrc_obj); \
 lsrc_obj = NULL; \
}

        /* Get path to recycled objects index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );
        if(recycled_index_file == NULL)
        {
            DO_FREE_LOCALS
            last_error = "Unable to get recycled objects index file";
            return(-1);
        }

	/* Get delete method, so we know if we should just move the
	 * object to the recycled objects directory or permanently
	 * purge it.
	 */
	delete_method = EDVCFGItemListGetValueI(
            core_ptr->cfg_list, EDV_CFG_PARM_DELETE_METHOD
        );


	/* Get copy of path of object to be deleted. */
	lsrc_obj = g_strdup(path);
	if(lsrc_obj == NULL)
	{
            DO_FREE_LOCALS
            last_error = "Memory allocation error";
            return(-3);
	}


	status = 0;

	/* Do delete iteration(s). */
	index = EDVRecBinFOPDeleteIterate(
	    core_ptr, recycled_index_file, lsrc_obj,
	    index_rtn, total_index_rtns,
	    toplevel, show_progress, interactive,
	    yes_to_all, &status, delete_method
	);

        /* Record history. */
	sprintf(num_str, "%i", index);
        EDVAppendHistory(
            core_ptr,
            EDV_HISTORY_DISK_OBJECT_DELETE,
            time_start, (gulong)time(NULL),
            status,
            lsrc_obj,		/* Source full path. */
	    num_str,		/* Target recycled object as index string. */
            last_error          /* Use last_error as comment if not NULL. */
        );


        /* Deallocate locally allocated memory. */
	DO_FREE_LOCALS
#undef DO_FREE_LOCALS

	return(status);
}


/*
 *	Purges the given recycled object specified by index from the
 *	recycled objects directory.
 *
 *	If and only if the progress_value is non negative, then the 
 *	progress dialog will be updated with an internally calculated
 *	value, otherwise the given progress_value is passed to the
 *	progress dialog.
 */
gint EDVRecBinFOPPurge(
	edv_core_struct *core_ptr,
	guint index,
	GtkWidget *toplevel,		/* Can be NULL. */
	gfloat progress_value,		/* Can be -1.0. */
	gbool show_progress, gbool interactive,
	gbool *yes_to_all
)
{
        static gint status;
	const gchar *recycled_index_file;
        gulong time_start = (gulong)time(NULL);
        edv_recbin_object_struct *obj = NULL;
	gchar num_str[80];


        /* Reset last error message. */
        last_error = NULL;

        if((core_ptr == NULL) || (index == 0))
        {
            last_error = "Bad input value";
            return(-1);
        }

#define DO_FREE_LOCALS	\
{ \
 EDVRecBinObjectDelete(obj); \
 obj = NULL; \
}

        /* Get path to recycled objects index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );
        if(recycled_index_file == NULL)
        {
            DO_FREE_LOCALS
            last_error = "Unable to get recycled objects index file";
            return(-1);
        }

        /* Get statistics of recycled object. */
        obj = EDVRecBinObjectStat(recycled_index_file, index);


        if(show_progress)
        {
	    gchar *buf;

	    if((obj != NULL) ? (obj->name != NULL) : FALSE)
	    {
                gchar *p1 = EDVCopyShortenPath(
                    obj->name, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                );
		buf = g_strdup_printf(
"Purging:\n\
\n\
    %s\n",
		    p1
		);
		g_free(p1);
	    }
	    else
	    {
		buf = g_strdup_printf(
"Purging:\n\
\n\
    #%i\n",
		    index
		);
	    }

            EDVRecBinFOPMapProgressDialogPurge(
                buf, MAX(progress_value, 0.0), toplevel, FALSE
            );

            g_free(buf);

	    /* If the given progress value is not negative then explicitly
	     * call the progress callback with values simulating the given
	     * progress value.
	     */
	    if(progress_value >= 0.0)
	    {
		if(EDVRecBinFOPRecycleProgressPercentCB(
		    core_ptr, progress_value
		))
		{
		    DO_FREE_LOCALS
		    status = -4;
		    return(status);
		}
	    }
        }


	/* Remove recycled object from the recycled objects directory. */
        status = EDVRecBinDiskObjectPurge(
            recycled_index_file,
            index,
	    ((progress_value < 0.0) && show_progress) ?
		EDVRecBinFOPRecycleProgressCB : NULL,
 	    core_ptr
        );
        if(status)
        {
	    last_error = EDVRecBinFIOGetError();
        }
        else
        {
            /* Remove recycled object entry from the recycled objects
             * index file.
             */
            EDVRecBinIndexRemove(
                recycled_index_file, index
            );
        }


        /* Record history. */
        sprintf(num_str, "%i", index);
        EDVAppendHistory(
            core_ptr,
            EDV_HISTORY_RECYCLED_OBJECT_PURGE,
            time_start, (gulong)time(NULL),
            status,
            num_str,		/* Source index string. */
	    NULL,		/* No target since purging into nothingness. */
            last_error          /* Use last_error as comment if not NULL. */
        );


        /* Deallocate locally allocated memory. */
        DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}


/*
 *      Purges all recycled objects from the recycled objects directory
 *	including the recycled objects index file.
 *
 *      If and only if the progress_value is non negative, then the
 *      progress dialog will be updated with an internally calculated
 *      value, otherwise the given progress_value is passed to the
 *      progress dialog.
 */
gint EDVRecBinFOPPurgeAll(
        edv_core_struct *core_ptr,
        GtkWidget *toplevel,            /* Can be NULL. */
        gbool show_progress, gbool interactive,
        gbool *yes_to_all
)
{
        static gint status;
        const gchar *recycled_index_file;
        gulong time_start = (gulong)time(NULL);


        /* Reset last error message. */
        last_error = NULL;

        if(core_ptr == NULL)
        {
            last_error = "Bad input value";
            return(-1);
        }

#define DO_FREE_LOCALS  \
{ \
}

        /* Get path to recycled objects index file. */
        recycled_index_file = EDVCFGItemListGetValueS(
            core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
        );
        if(recycled_index_file == NULL)
        {
            DO_FREE_LOCALS
            last_error = "Unable to get recycled objects index file";
            return(-1);
        }


        if(show_progress)
        {
            gchar *buf = g_strdup_printf(
"Purging all contents from the Recycle Bin..."
	    );

            EDVRecBinFOPMapProgressDialogPurge(
                buf, 0.0, toplevel, FALSE
            );

            g_free(buf);
        }


	/* Purge all objects found in the recycled objects directory,
	 * including the recycled objects index file.
	 */
	status = EDVRecBinDiskObjectPurgeAll(
	    recycled_index_file, 
	    show_progress ? EDVRecBinFOPRecycleProgressCB : NULL,
	    core_ptr
	);
	if(status)
	    last_error = EDVRecBinFIOGetError();


        /* Record history. */
        EDVAppendHistory(
            core_ptr,
            EDV_HISTORY_RECYCLED_OBJECT_PURGE_ALL,
            time_start, (gulong)time(NULL),
            status,
            NULL,		/* No source index string to imply all. */
            NULL,		/* No target since purging into nothingness. */
            last_error          /* Use last_error as comment if not NULL. */
        );

	/* Deallocate locally allocated memory. */
	DO_FREE_LOCALS
#undef DO_FREE_LOCALS

        return(status);
}
