#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvobj.h"
#include "findbar.h"
#include "endeavour.h"
#include "edvfind.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"

#include "images/icon_stop_20x20.xpm"


static gint EDVFindBarProgressCB(
        const gchar *path, gfloat progress, gpointer data
);
static void EDVFindBarMatchExcerptCB(
        const gchar *path, const struct stat *lstat_buf, const gchar *excerpt,
	gpointer data
);
static void EDVFindBarMatchCB(
        const gchar *path, const struct stat *lstat_buf, gpointer data
);


static void EDVFindBarFindCB(GtkWidget *widget, gpointer data);
static void EDVFindBarStopCB(GtkWidget *widget, gpointer data);
static gint EDVFindBarCrossingCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

void EDVFindBarSetSearch(
        edv_findbar_struct *fb, const gchar *s,
        gbool record_history
);
gint EDVFindBarCurrentFindOP(edv_findbar_struct *fb);

edv_findbar_struct *EDVFindBarNew(
        gpointer core_ptr, GtkWidget *parent,
	gint role,
        void (*status_message_cb)(gpointer, const gchar *, gpointer),
        void (*status_progress_cb)(gpointer, gfloat, gpointer),
        gchar *(*get_location_cb)(gpointer, gpointer),
        void (*start_cb)(gpointer, gpointer),
        void (*end_cb)(gpointer, gint, gpointer),
        void (*match_cb)(
	    const gchar *, const struct stat *, const gchar *, gpointer),
        gpointer client_data
);
void EDVFindBarUpdateMenus(edv_findbar_struct *fb);
void EDVFindBarMap(edv_findbar_struct *fb);
void EDVFindBarUnmap(edv_findbar_struct *fb);
void EDVFindBarDelete(edv_findbar_struct *fb);



/*
 *	Progress callback, used for EDVFindObjectByName() and similar
 *	functions.
 */
static gint EDVFindBarProgressCB(
	const gchar *path, gfloat progress, gpointer data
)
{
        edv_findbar_struct *fb = (edv_findbar_struct *)data;
        if(fb == NULL)
            return(0);

	if(fb->stop_count > 0)
	    return(1);

	if(fb->status_message_cb != NULL)
	    fb->status_message_cb(fb, path, fb->client_data);
        if(fb->status_progress_cb != NULL)
            fb->status_progress_cb(fb, progress, fb->client_data);

	while(gtk_events_pending() > 0)
	    gtk_main_iteration();

	return(0);
}

/*
 *      Got match callback, used for EDVFindObjectByName() and similar
 *      functions.
 */
static void EDVFindBarMatchExcerptCB(
        const gchar *path, const struct stat *lstat_buf, const gchar *excerpt,
        gpointer data
)
{
        edv_findbar_struct *fb = (edv_findbar_struct *)data;
        if(fb == NULL)
            return;

        if(fb->match_cb != NULL)
            fb->match_cb(path, lstat_buf, excerpt, fb->client_data);

        while(gtk_events_pending() > 0)
            gtk_main_iteration();
}

static void EDVFindBarMatchCB(
	const gchar *path, const struct stat *lstat_buf, gpointer data
)
{
	EDVFindBarMatchExcerptCB(path, lstat_buf, NULL, data);
}

/*
 *	Find "activate" signal callback.
 *
 *	This starts the search process.
 */
static void EDVFindBarFindCB(GtkWidget *widget, gpointer data)
{
	static gbool reenterent = FALSE;
	gbool case_sensitive = FALSE;
	gint op_code;
	const gchar *cstrptr;
	gchar *location, *search_string;
	gint total_matches;
	GtkWidget *w;
	edv_core_struct *core_ptr;
	edv_findbar_struct *fb = (edv_findbar_struct *)data;
	if(fb == NULL)
	    return;

	if(fb->processing)
	    return;

	core_ptr = (edv_core_struct *)fb->core_ptr;
	if(core_ptr == NULL)
	    return;

	if(reenterent)
	    return;
	else
	    reenterent = TRUE;


	/* Reset values for upcomming find operation. */
        fb->processing = TRUE;
	fb->stop_count = 0;
        EDVFindBarUpdateMenus(fb);


	/* Get find operation. */
	op_code = EDVFindBarCurrentFindOP(fb);

	/* Get copy of location. */
	cstrptr = NULL;
	switch(fb->role)
	{
	  case EDV_FINDBAR_ROLE_DISK_OBJECT:
          case EDV_FINDBAR_ROLE_ARCHIVE_OBJECT:
	    if(fb->get_location_cb != NULL)
		cstrptr = fb->get_location_cb(fb, fb->client_data);
	    break;
	  case EDV_FINDBAR_ROLE_RECYCLED_OBJECT:
	    cstrptr = EDVCFGItemListGetValueS(
		core_ptr->cfg_list, EDV_CFG_PARM_FILE_RECYCLED_INDEX
	    );
	    break;
	}
	location = g_strdup((cstrptr != NULL) ? cstrptr : "/");

	/* Get copy of search string. */
	cstrptr = NULL;
	w = fb->search_combo;
	if(w != NULL)
	{
	    GtkEntry *entry = (GtkEntry *)GTK_COMBO(w)->entry;
	    cstrptr = gtk_entry_get_text(entry);
	}
	search_string = g_strdup((cstrptr != NULL) ? cstrptr : "");
        /* No wild cards in search string? */
        if((strchr(search_string, '*') == NULL) &&
           (strchr(search_string, '?') == NULL) &&
           (op_code == EDV_FINDBAR_FIND_OBJECT_NAME)
        )
        {
            /* Tack on two '*' characters to the beginning and end of
             * the search string.
             */
            gchar *strptr = g_strdup_printf("*%s*", search_string);
            g_free(search_string);
            search_string = strptr;
        }
        /* Record search string history. */
        EDVFindBarSetSearch(fb, search_string, TRUE);

        /* Case sensitive. */
        w = fb->case_sensitive_check;
        if(w != NULL)
            case_sensitive = GTK_TOGGLE_BUTTON(w)->active;


	/* Call start find callback to notify about start of find. */
	if(fb->start_cb != NULL)
	    fb->start_cb(fb, fb->client_data);


        /* Perform find procedure by find operation code. */
	total_matches = 0;
        switch(op_code)
        {
          case EDV_FINDBAR_FIND_OBJECT_NAME:
	    switch(fb->role)
	    {
	      case EDV_FINDBAR_ROLE_DISK_OBJECT:
                total_matches = EDVFindObjectByName(
                    location, search_string,
                    FALSE,			/* Recursive. */
                    case_sensitive,
                    fb,
                    EDVFindBarProgressCB,
                    EDVFindBarMatchCB
                );
		break;

              case EDV_FINDBAR_ROLE_RECYCLED_OBJECT:
		total_matches = EDVFindRecycledObjectByName(
		    location,		/* Recycled objects index file. */
 		    search_string,
		    case_sensitive,
		    fb,
		    EDVFindBarProgressCB,
		    EDVFindBarMatchCB
		);
		break;

              case EDV_FINDBAR_ROLE_ARCHIVE_OBJECT:
                total_matches = EDVFindArchiveObjectByName(
		    core_ptr,
                    location,		/* Archive. */
                    search_string,
                    case_sensitive,
                    fb,
                    EDVFindBarProgressCB,
                    EDVFindBarMatchCB
                );
		break;
	    }
            break;

          case EDV_FINDBAR_FIND_OBJECT_CONTENT:
            switch(fb->role)
            {
              case EDV_FINDBAR_ROLE_DISK_OBJECT:
                total_matches = EDVFindObjectByContent(
                    location, search_string,
                    FALSE,			/* Recursive. */
                    case_sensitive,
                    fb,
                    EDVFindBarProgressCB,
                    EDVFindBarMatchExcerptCB
                );
		break;

              case EDV_FINDBAR_ROLE_RECYCLED_OBJECT:
		total_matches = EDVFindRecycledObjectByContent(
		    location,		/* Recycled objects index file. */
		    search_string,
		    case_sensitive,
		    fb,
		    EDVFindBarProgressCB,
                    EDVFindBarMatchExcerptCB
                );
                break;

              case EDV_FINDBAR_ROLE_ARCHIVE_OBJECT:
		/* Finding archive object by content is not supported yet. */
                total_matches = 0;
                break;
            }
            break;
        }


        /* Tally. */
        if(fb->status_progress_cb != NULL)
            fb->status_progress_cb(fb, 0.0, fb->client_data);

        if(total_matches > 0)
        {
            gchar *buf = g_strdup_printf(
                "Matched %i object%s",
                total_matches,
                (total_matches == 1) ? "" : "s"
            );
	    if(fb->status_message_cb != NULL)
		fb->status_message_cb(fb, buf, fb->client_data);
            g_free(buf);
        }
        else
        {
	    if(fb->status_message_cb != NULL)
		fb->status_message_cb(fb, "No objects found", fb->client_data);
        }


	/* Mark find bar as done processing. */
	fb->processing = FALSE;

	EDVFindBarUpdateMenus(fb);


        /* Call end find callback to notify about end of find. */
        if(fb->end_cb != NULL)
            fb->end_cb(fb, total_matches, fb->client_data);


	/* Deallocate local memory. */
	g_free(location);
	location = NULL;
	g_free(search_string);
	search_string = NULL;

	reenterent = FALSE;
}

/*
 *	Stop callback.
 */
static void EDVFindBarStopCB(GtkWidget *widget, gpointer data)
{
        edv_findbar_struct *fb = (edv_findbar_struct *)data;
        if(fb == NULL)
            return;

	fb->stop_count++;
}

/*
 *      find bar widget "enter_notify_event" or "leave_notify_event"
 *      signal callback.
 */
static gint EDVFindBarCrossingCB(
        GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
        static gbool reenterent = FALSE;
        gint status = FALSE;
        gint etype;
        GtkWidget *w;
        const gchar *mesg = NULL;
        edv_findbar_struct *fb = (edv_findbar_struct *)data;
        if((widget == NULL) || (crossing == NULL) || (fb == NULL))
            return(status);

        if(reenterent)
            return(status);
        else
            reenterent = TRUE;

        /* Get event type. */
        etype = crossing->type;

        w = widget;


        /* Handle by event type. */
        switch(etype)
        {
          case GDK_ENTER_NOTIFY:
            if(w == fb->case_sensitive_check)
#ifdef PROG_LANGUAGE_ENGLISH
		mesg =
"Specifies if string matches should be case sensitive or not";
#endif
#ifdef PROG_LANGUAGE_SPANISH
		mesg =
"Especifica si los iguales de cuerda no deben ser el casos sensibles ni";
#endif
#ifdef PROG_LANGUAGE_FRENCH
		mesg =
"Spcifie si les allumettes de ficelle devraient tre des cas sensibles ou pas";
#endif
            else if(w == fb->stop_btn)
#ifdef PROG_LANGUAGE_ENGLISH
		mesg =
"Specifies if the search process should recurse into subdirectories";
#endif
#ifdef PROG_LANGUAGE_SPANISH
		mesg =
"Especifica si el proceso de la bsqueda debe recurse en el subdirectories";
#endif
#ifdef PROG_LANGUAGE_FRENCH
		mesg =
"Spcifie si le procd de recherche doit recurse dans subdirectories";
#endif
            break;
        }

        if(fb->status_message_cb != NULL)
            fb->status_message_cb(
                fb, mesg, fb->client_data
            );

        reenterent = FALSE;
        return(status);
}

/*
 *      Sets the search combo of the given find bar to the value of
 *      the string s.
 *
 *      If record_history is TRUE then the current value will be recorded
 *      on the combo's list before setting of the new value.
 *      Duplicate values will not be recorded.
 */
void EDVFindBarSetSearch(
        edv_findbar_struct *fb, const gchar *s,
        gbool record_history
)
{
        GtkCombo *combo;
        gchar *new_s;


        if((fb == NULL) || (s == NULL))
            return;

        combo = (GtkCombo *)fb->search_combo;
        if(combo == NULL)
            return;

        /* Copy given path as the new path. */
        new_s = g_strdup(s);

        /* Check for no change in value. */
        if(new_s != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
            if(cstrptr != NULL)
            {
                /* No change in value? */
                if(!strcmp(cstrptr, new_s))
                {
                    g_free(new_s);
                    new_s = NULL;
                    return;
                }
            }
        }

        /* Get old value from combo's entry and record it on the list. */
        if(record_history)
        {
            gchar *old_s;
            const gchar *cstrptr = gtk_entry_get_text(GTK_ENTRY(combo->entry));
            old_s = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
            if(old_s != NULL)
            {
                GUIComboAddItem(combo, old_s);
                g_free(old_s);
                old_s = NULL;
            }
        }

        /* Set new value on entry and deallocate coppied new_s. */
        if(new_s != NULL)
        {
            gtk_entry_set_text(GTK_ENTRY(combo->entry), new_s);
            g_free(new_s);
            new_s = NULL;
        }
}


/*
 *      Returns the current find operation code, one of EDV_FINDBAR_FIND_*.
 */
gint EDVFindBarCurrentFindOP(edv_findbar_struct *fb)
{
        gint i;
        const gchar *value, *value2;
        GList *glist;
        GtkCombo *combo;


        if(fb == NULL)
            return(EDV_FINDBAR_FIND_OBJECT_NAME);

        combo = (GtkCombo *)fb->find_op_combo;
        if(combo == NULL)
            return(EDV_FINDBAR_FIND_OBJECT_NAME);

        value = gtk_entry_get_text(GTK_ENTRY(combo->entry));
        glist = (GList *)GUIComboGetList(combo);
        if((value == NULL) || (glist == NULL))
            return(EDV_FINDBAR_FIND_OBJECT_NAME);

        i = 0;
        while(glist != NULL)
        {
            value2 = (const gchar *)glist->data;
            if((value2 != NULL) ? !strcasecmp(value2, value) : FALSE)
                return(i);

            i++;
            glist = glist->next;
        }

        return(EDV_FINDBAR_FIND_OBJECT_NAME);
}



/*
 *	Create a new find bar.
 */
edv_findbar_struct *EDVFindBarNew(
        gpointer core_ptr, GtkWidget *parent,
	gint role,
        void (*status_message_cb)(gpointer, const gchar *, gpointer),
        void (*status_progress_cb)(gpointer, gfloat, gpointer),
        gchar *(*get_location_cb)(gpointer, gpointer),
	void (*start_cb)(gpointer, gpointer),
	void (*end_cb)(gpointer, gint, gpointer),
        void (*match_cb)(
            const gchar *, const struct stat *, const gchar *, gpointer
	),
	gpointer client_data
)
{
        gint border_minor = 2;
	gpointer combo_rtn;
	GList *glist;
        GtkWidget *w, *parent2;
	GtkCombo *combo;
	edv_findbar_struct *fb = (edv_findbar_struct *)g_malloc0(
            sizeof(edv_findbar_struct)
        );
        if(fb == NULL)
            return(fb);

        /* Reset values. */
        fb->initialized = TRUE;
        fb->map_state = FALSE;
	fb->processing = FALSE;
	fb->core_ptr = core_ptr;
	fb->stop_count = 0;
	fb->role = role;

	fb->status_message_cb = status_message_cb;
	fb->status_progress_cb = status_progress_cb;
	fb->get_location_cb = get_location_cb;
	fb->start_cb = start_cb;
	fb->end_cb = end_cb;
	fb->match_cb = match_cb;
	fb->client_data = client_data;


        /* Create toplevel. */
	fb->toplevel = w = gtk_hbox_new(FALSE, border_minor);
        gtk_container_border_width(GTK_CONTAINER(w), 2);
        gtk_container_add(GTK_CONTAINER(parent), w);
        parent2 = w;


        /* Create glist containing a list of find operation names, where
         * the index of each name corresponds to one of EDV_FINDBAR_FIND_*.
         */
        glist = NULL;
#ifdef PROG_LANGUAGE_ENGLISH
        glist = g_list_append(glist, "Object Name");
        glist = g_list_append(glist, "Object Content");
#endif
#ifdef PROG_LANGUAGE_SPANISH
        glist = g_list_append(glist, "El Nombre Objetivo");
        glist = g_list_append(glist, "El Contenido Objetivo");
#endif
#ifdef PROG_LANGUAGE_FRENCH
        glist = g_list_append(glist, "Opposer le Nom");
        glist = g_list_append(glist, "Opposer le Contenu");
#endif

        /* Find operation combo. */
        w = (GtkWidget *)GUIComboCreate(
#ifdef PROG_LANGUAGE_ENGLISH
            "Find:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Hallazgo:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Dcouverte:",
#endif
            "",
            glist,
            2,          /* Maximum items. */
            &combo_rtn,
            fb,
            EDVFindBarFindCB,
            NULL
        );
        fb->find_op_combo = (GtkWidget *)combo_rtn;
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        w = fb->find_op_combo;
        combo = GTK_COMBO(w);
        gtk_widget_set_usize(w, 150, -1);

        g_list_free(glist);

        w = combo->entry;
        gtk_entry_set_editable(GTK_ENTRY(w), FALSE);
/*
        glist = GUIComboGetList(combo);
        glist = g_list_nth(glist, find_op);
        if((glist != NULL) ? (glist->data != NULL) : FALSE)
            gtk_entry_set_text(GTK_ENTRY(w), (const gchar *)glist->data);
 */

        /* Search string combo. */
        glist = NULL;
        w = (GtkWidget *)GUIComboCreate(
#ifdef PROG_LANGUAGE_ENGLISH
            "Matching:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Parejo:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Egaler:",
#endif
            "",
            glist,
            20,         /* Maximum items. */
            &combo_rtn,
            fb,
            EDVFindBarFindCB,
            NULL
        );
        fb->search_combo = (GtkWidget *)combo_rtn;
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_widget_show(w);

        w = fb->search_combo;
        combo = GTK_COMBO(w);
	gtk_combo_set_case_sensitive(combo, TRUE);

        w = combo->entry;
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
	EDVEntrySetDND((edv_core_struct *)core_ptr, w);
/*
        if(search_string != NULL)
            gtk_entry_set_text(GTK_ENTRY(w), search_string);
 */

        /* Case sensitive check. */
        fb->case_sensitive_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
            "Case Sensitive"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El Caso Sensible"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Le Cas Sensible"
#endif
        );
        GTK_TOGGLE_BUTTON(w)->active = FALSE;
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVFindBarCrossingCB), fb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVFindBarCrossingCB), fb
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Specifies if string matches should be case sensitive or not"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Especifica si los iguales de cuerda no deben ser el casos sensibles ni"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Spcifie si les allumettes de ficelle devraient tre des cas sensibles ou pas"
#endif
        );
        gtk_widget_show(w);

	w = gtk_vseparator_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

        /* Stop button. */
        fb->stop_btn = w = GUIButtonPixmap(
            (u_int8_t **)icon_stop_20x20_xpm
        );
/*	gtk_widget_set_usize(w, 20 + 2, 20 + 2); */
        gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFindBarStopCB), fb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "enter_notify_event",
            GTK_SIGNAL_FUNC(EDVFindBarCrossingCB), fb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "leave_notify_event",
            GTK_SIGNAL_FUNC(EDVFindBarCrossingCB), fb
        );
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Stop the current find procedure"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Pare el procedimiento actual del hallazgo"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Arrter la procdure de dcouverte actuelle"
#endif
	);
        gtk_widget_show(w);



	EDVFindBarUpdateMenus(fb);

	return(fb);
}


/*
 *	Updates all menus and widgets on the find bar.
 */
void EDVFindBarUpdateMenus(edv_findbar_struct *fb)
{
        gbool sensitive;
        GtkWidget *w;
        edv_core_struct *core_ptr;


        if(fb == NULL)
            return;

        if(!fb->initialized)
            return;

        core_ptr = (edv_core_struct *)fb->core_ptr;
        if(core_ptr == NULL)
            return;

#define DO_SET_SENSITIVE        \
{ \
 if(w != NULL) \
  gtk_widget_set_sensitive(w, sensitive); \
}
#define DO_SHOW \
{ \
 if(w != NULL) \
  gtk_widget_show(w); \
}
#define DO_HIDE \
{ \
 if(w != NULL) \
  gtk_widget_hide(w); \
}
#define DO_SET_CHECK_STATE      \
{ \
 if((w != NULL) ? GTK_IS_CHECK_MENU_ITEM(w) : FALSE) \
  GTK_CHECK_MENU_ITEM(w)->active = state; \
}

        /* Update buttons. */
        sensitive = fb->processing;
        w = fb->stop_btn;
        DO_SET_SENSITIVE

        sensitive = !fb->processing;
        w = fb->search_combo;
        DO_SET_SENSITIVE
        w = fb->case_sensitive_check;
        DO_SET_SENSITIVE
        w = fb->find_op_combo;
        DO_SET_SENSITIVE

#undef DO_SHOW
#undef DO_HIDE
#undef DO_SET_SENSITIVE
#undef DO_SET_CHECK_STATE
}

/*
 *	Maps the find bar.
 */
void EDVFindBarMap(edv_findbar_struct *fb)
{
	GtkWidget *w;

        if(fb == NULL)
            return;

	if(!fb->map_state)
	{
	    w = fb->toplevel;
	    if(w != NULL)
		gtk_widget_show(w);

	    fb->map_state = TRUE;
	}
}

/*
 *	Unmaps the find bar.
 */
void EDVFindBarUnmap(edv_findbar_struct *fb)
{
        GtkWidget *w;

        if(fb == NULL)
            return;

        if(fb->map_state)
        {
            w = fb->toplevel;
            if(w != NULL)
                gtk_widget_hide(w);

            fb->map_state = FALSE;
        }
}

/*
 *	Deallocates all resources on the given find bar structure and
 *	deallocates the find bar structure itself.
 */
void EDVFindBarDelete(edv_findbar_struct *fb)
{
        GtkWidget **w;


        if(fb == NULL)
            return;

        if(fb->initialized)
        {
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}
	    w = &fb->find_op_combo;
            DO_DESTROY_WIDGET
	    w = &fb->search_combo;
            DO_DESTROY_WIDGET
	    w = &fb->case_sensitive_check;
            DO_DESTROY_WIDGET
	    w = &fb->stop_btn;
            DO_DESTROY_WIDGET

	    w = &fb->toplevel;
	    DO_DESTROY_WIDGET

#undef DO_DESTROY_WIDGET
	}

        /* Deallocate find bar structure itself. */
        g_free(fb);
}
