#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "progressdialog.h"
#include "pulist.h"

#include "edvtypes.h"
#include "edvid.h"
#include "edvobj.h"
#include "edvcfg.h"
#include "endeavour.h"
#include "edvfop.h"
#include "edvcb.h"
#include "edvop.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "fopdialog.h"
#include "config.h"

#include "images/icon_cancel_20x20.xpm"
#include "images/icon_browse_20x20.xpm"
#include "images/icon_move_file_20x20.xpm"
#include "images/icon_copy_file_20x20.xpm"
#include "images/icon_link2_20x20.xpm"
#include "images/icon_move_file_32x32.xpm"
#include "images/icon_copy_file_32x32.xpm"
#include "images/icon_link2_32x32.xpm"
#include "images/icon_chmod_20x20.xpm"
#include "images/icon_chmod_32x32.xpm"
#include "images/icon_owned_20x20.xpm"
#include "images/icon_owned_32x32.xpm"


static gint EDVFOPDialogDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVFOPDialogProcessCB(GtkWidget *widget, gpointer data);
static void EDVFOPDialogCancelCB(GtkWidget *widget, gpointer data);
static void EDVFOPDialogBrowseCB(GtkWidget *widget, gpointer data);

static void EDVFOPDialogOwnerMapPUListCB(GtkWidget *widget, gpointer data);
static void EDVFOPDialogGroupMapPUListCB(GtkWidget *widget, gpointer data);

edv_fopdialog_struct *EDVFOPDialogNew(gpointer core_ptr);
void EDVFOPDialogReset(edv_fopdialog_struct *d);
void EDVFOPDialogMapValues(
	edv_fopdialog_struct *d,
	gint op,		/* One ofEDV_FOPDIALOG_OP_*. */
	edv_object_struct **src_obj,	/* Source list, will be coppied. */
        gint src_obj_total,
	const gchar *src_dir,
        GtkWidget *toplevel
);
void EDVFOPDialogUnmap(edv_fopdialog_struct *d);
void EDVFOPDialogDelete(edv_fopdialog_struct *d);


#define FOPDIALOG_BTN_WIDTH	(100 + (2 * 3))
#define FOPDIALOG_BTN_HEIGHT	(30 + (2 * 3))


/*
 *	"delete_event" signal callback.
 */
static gint EDVFOPDialogDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_fopdialog_struct *d = (edv_fopdialog_struct *)data;
	if(d == NULL)
	    return(FALSE);

	EDVFOPDialogCancelCB(NULL, d);

	return(TRUE);
}

/*
 *	Opaque process callback for move, copy, link, chmod, and chown 
 *	operatios.
 */
static void EDVFOPDialogProcessCB(GtkWidget *widget, gpointer data)
{
	gint i, op;
	edv_object_struct **object, *obj;
	gint total_objects;
	gchar *tar_path;
	guint permissions = 0x00000000;
	gint owner_id = 0, group_id = 0;
	GtkWidget *w, *toplevel;
	edv_core_struct *core_ptr;
        edv_fopdialog_struct *d = (edv_fopdialog_struct *)data;
        if(d == NULL)
	    return;

	core_ptr = (edv_core_struct *)d->core_ptr;
	if(core_ptr == NULL)
	    return;

        /* Check and warn if write protect is enabled. */
        if(EDVCheckWriteProtect(core_ptr, TRUE))
            return;


	/* Get operation code. */
	op = d->op;

	/* Get reference toplevel widget for transiency of dialogs that
	 * will be used during the file operation.
	 */
	toplevel = d->ref_toplevel;

	/* Get source list of disk object structures. */
	total_objects = d->src_obj_total;
	object = d->src_obj;

	/* Get copy of parsed and simplified target path. */
	w = d->target_entry;
	tar_path = (w != NULL) ?
	    EDVCopyEvaluateInputPath(
		d->src_dir,
		gtk_entry_get_text(GTK_ENTRY(w))
	    ) : NULL;

	/* Get permissions. */
	w = d->target_chmod_parent;
	if(w != NULL)
	{
	    w = d->target_uread_check;
	    if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
		permissions |= EDV_PERMISSION_UREAD;
            w = d->target_uwrite_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_UWRITE;
            w = d->target_uexecute_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_UEXECUTE;

            w = d->target_gread_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_GREAD;
            w = d->target_gwrite_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_GWRITE;
            w = d->target_gexecute_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_GEXECUTE;

            w = d->target_aread_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_AREAD;
            w = d->target_awrite_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_AWRITE;
            w = d->target_aexecute_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_AEXECUTE;

            w = d->target_setuid_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_SETUID;
            w = d->target_setgid_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_SETGID;
            w = d->target_sticky_check;
            if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
                permissions |= EDV_PERMISSION_STICKY;

	}

        /* Get owner and group. */
        w = d->target_chown_parent;
        if(w != NULL)
        {
            /* Get user id from user entry. */
            w = d->target_owner_entry;
            if(w != NULL)
            {
                owner_id = EDVUIDNameToUID(
                    core_ptr->uid, core_ptr->total_uids,
                    gtk_entry_get_text(GTK_ENTRY(w))
                );
            }

            /* Get group id from group entry. */
            w = d->target_group_entry;
            if(w != NULL)
            {
                group_id = EDVGIDNameToGID(
                    core_ptr->gid, core_ptr->total_gids,
                    gtk_entry_get_text(GTK_ENTRY(w))
                );
            }
        }


        /* Unmap dialog. */
        EDVFOPDialogUnmap(d);


	/* Process. */
	if(1)
	{
            gbool yes_to_all = FALSE;
	    gbool need_emit_modified = FALSE;
	    gint status;
	    const gchar *src_path;
	    gchar *new_path;
	    const gchar *error_mesg;
	    gint src_lstat_result;
	    struct stat lstat_buf;


	    /* Iterate through source objects. */
	    for(i = 0; i < total_objects; i++)
	    {
		obj = object[i];
		if(obj == NULL)
		    continue;

		src_path = obj->full_path;
		if(src_path == NULL)
		    continue;

		/* Handle by operation type. */
                status = -1;
                new_path = NULL;
		error_mesg = NULL;
		switch(op)
		{
		  case EDV_FOPDIALOG_OP_COPY:
                    status = EDVFOPCopy(
                        core_ptr, src_path, tar_path,
                        &new_path, toplevel,
                        TRUE, TRUE,
                        &yes_to_all
                    );
                    break;

                  case EDV_FOPDIALOG_OP_MOVE:
                    status = EDVFOPMove(
                        core_ptr, src_path, tar_path,
                        &new_path, toplevel,
                        TRUE, TRUE,
                        &yes_to_all
                    );
                    break;

                  case EDV_FOPDIALOG_OP_LINK:
                    status = EDVFOPLink(
                        core_ptr, src_path, tar_path,
                        &new_path, toplevel,
                        TRUE, TRUE,
                        &yes_to_all
                    );
                    break;

                  case EDV_FOPDIALOG_OP_CHMOD:
		    if(1)
		    {
			gfloat progress = (total_objects > 0) ?
			    ((gfloat)i / (gfloat)total_objects) : 0.0;
			gchar *p1 = EDVCopyShortenPath(
			    src_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                        );
                        gchar *buf = g_strdup_printf(
"Changing permissions of:\n\
\n\
    %s\n",
                            p1
                        );

		        if(ProgressDialogIsQuery())
			{
			    ProgressDialogUpdate(
				NULL, buf, NULL, NULL,
				progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
			    );
			}
			else
			{
			    ProgressDialogSetTransientFor(toplevel);
			    ProgressDialogMap(
				"Changing Permissions",
				buf,
				(const u_int8_t **)icon_chmod_32x32_xpm,
				"Stop"
			    );
                            ProgressDialogUpdate(
                                NULL, NULL, NULL, NULL,
                                progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
                            );
			}
			g_free(buf);
			g_free(p1);
		    }
		    if(ProgressDialogStopCount() > 0)
			status = -4;
		    else
			status = EDVFOPChmod(
			    core_ptr, src_path,
			    permissions,
			    toplevel,
			    FALSE, TRUE, &yes_to_all
			);
		    need_emit_modified = TRUE;
		    break;

                  case EDV_FOPDIALOG_OP_CHOWN:
                    if(1)
                    {
                        gfloat progress = (total_objects > 0) ?
                            ((gfloat)i / (gfloat)total_objects) : 0.0;
                        gchar *p1 = EDVCopyShortenPath(
                            src_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
                        );
                        gchar *buf = g_strdup_printf(
"Chowning:\n\
\n\
    %s\n",
                            p1
                        );

                        if(ProgressDialogIsQuery())
                        {
                            ProgressDialogUpdate(
                                NULL, buf, NULL, NULL,
                                progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
                            );
                        }
                        else
                        {
                            ProgressDialogSetTransientFor(toplevel);
                            ProgressDialogMap(
                                "Chowning",
                                buf,
                                (const u_int8_t **)icon_owned_32x32_xpm,
                                "Stop"
                            );
                            ProgressDialogUpdate(
                                NULL, NULL, NULL, NULL,
                                progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
                            );
                        }
                        g_free(buf);
                        g_free(p1);
                    }
                    if(ProgressDialogStopCount() > 0)
                        status = -4;
                    else
			status = EDVFOPChown(
			    core_ptr, src_path,
			    owner_id, group_id,
			    toplevel,
			    FALSE, TRUE, &yes_to_all
			);
		    need_emit_modified = TRUE;
		    break;

		  default:
		    error_mesg =
"Internal error: Unsupported disk object operation";
 		    break;
                }

                /* Get error message (if any) that might have occured
                 * in the above operation.
                 */
		if(error_mesg == NULL)
		    error_mesg = EDVFOPGetError();
                if((error_mesg != NULL) && (status != -4))
                {
                    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        "Operation Error",
                        error_mesg,
                        NULL,
                        CDIALOG_ICON_ERROR,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
                }


                /* Check if new_path is valid and that no error
                 * occured. This implies the operation was successful
                 * and we must emit a object added signal to all
                 * windows.
                 */
                if(((new_path != NULL) && !status) ?
                    !lstat(new_path, &lstat_buf) : FALSE
                )
                {
/*                  browser->processing = FALSE; */
                    EDVObjectAddedEmit(
                        core_ptr, new_path, &lstat_buf
                    );
/*                  browser->processing = TRUE; */
                }


		/* Get local stats for the source object (possibly
		 * overwriting lstat_buf of any data obtained from the
		 * local stats of new_path above).
		 */
		src_lstat_result = lstat(src_path, &lstat_buf);

		/* Need to emit an object modified signal? (For cases of
		 * chmod or chown operations?
		 */
		if(need_emit_modified && !status)
		{
		    if(!src_lstat_result)
			EDVObjectModifiedEmit(
			    core_ptr, src_path, src_path,
			    &lstat_buf
			);
		}

		/* Check if the object no longer exists locally, if it
		 * no longer exists then emit an object removed signal to
		 * all of endeavour's resources.
		 *
		 * Note that this will work out if the operation was a
		 * copy, link, chmod, or chown since it dosen't emit an
		 * object removed signal if the object still exists.
		 */
		if(src_lstat_result)
                    EDVObjectRemovedEmit(core_ptr, src_path);


                /* Deallocate copy of the new disk object's path. */
                g_free(new_path);
                new_path = NULL;

                /* Error occured and it was not a user `no' or `not
                 * available' response?
                 */
                if(status && (status != -5))
                {
                    /* Skip rest of the objects due to error. */
                    break;
                }
	    }

            /* Unmap progress dialog, it may have been mapped if any
             * operations occured in the above loop.
             */
            ProgressDialogBreakQuery(TRUE);
            ProgressDialogSetTransientFor(NULL);
	}

	/* Deallocate copy of target path. */
	g_free(tar_path);
	tar_path = NULL;

	/* From this point on these variables are invalid, and they
	 * will be deallocated when EDVFOPDialogReset() is called further
	 * below.
	 */
	total_objects = 0;
	object = NULL;

        /* Deallocate source disk objects structure list and current
         * source directory on the dialog.
         */
        EDVFOPDialogReset(d);
}

/*
 *      Cancel callback.
 */
static void EDVFOPDialogCancelCB(GtkWidget *widget, gpointer data)
{
        edv_fopdialog_struct *d = (edv_fopdialog_struct *)data;
        if(d == NULL)
            return;

	/* Unmap dialog. */
	EDVFOPDialogUnmap(d);

        /* Deallocate source disk objects structure list and current
         * source directory on the dialog.
         */
        EDVFOPDialogReset(d);
}

/*
 *	Browse callback.
 */
static void EDVFOPDialogBrowseCB(GtkWidget *widget, gpointer data)
{
        gbool status;
        GtkWidget *toplevel;
        fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
        gint total_ftypes = 0;
        gchar **path_rtn = NULL;
        gint total_path_rtns = 0;
        GtkEntry *entry = (GtkEntry *)data;
        if(entry == NULL)
            return;


        if(FileBrowserIsQuery())
            return;

        toplevel = gtk_widget_get_toplevel(GTK_WIDGET(entry));

        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );

        /* Query user for target object. */
        FileBrowserSetTransientFor(toplevel);
        status = FileBrowserGetResponse(
            "Select Target Object",
            "Select", "Cancel",
            gtk_entry_get_text(entry),          /* Startup path. */
            ftype, total_ftypes,
            &path_rtn, &total_path_rtns,
            &ftype_rtn
        );
        FileBrowserSetTransientFor(NULL);

        /* Got user response? */
        if(status)
        {
            const gchar *new_path = (total_path_rtns > 0) ?
                path_rtn[0] : NULL;
            if((new_path != NULL) ? (*new_path != '\0') : FALSE)
                gtk_entry_set_text(entry, new_path);
        }

        /* Deallocate file types list. */
        FileBrowserDeleteTypeList(ftype, total_ftypes);
}


/*
 *      Maps the popup list for the owner entry.
 */
static void EDVFOPDialogOwnerMapPUListCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        const gchar *value;
        GtkEntry *entry;
        edv_core_struct *core_ptr;
        edv_fopdialog_struct *d = (edv_fopdialog_struct *)data;
        if(d == NULL)
            return;

        core_ptr = (edv_core_struct *)d->core_ptr;
        if(core_ptr == NULL)
            return;

        entry = (GtkEntry *)d->target_owner_entry;
        if(entry == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Block input and get value. */
        value = PUListMapQuery(
            core_ptr->users_pulist,
            gtk_entry_get_text(entry),
            15,
            PULIST_RELATIVE_CENTER,
            GTK_WIDGET(entry), widget
        );
        /* Got value? */
        if(value != NULL)
            gtk_entry_set_text(entry, value);

        reenterent = FALSE;
}

/*
 *      Maps the popup list for the group entry.
 */
static void EDVFOPDialogGroupMapPUListCB(GtkWidget *widget, gpointer data)
{
        static gbool reenterent = FALSE;
        const gchar *value;
        GtkEntry *entry;
        edv_core_struct *core_ptr;
        edv_fopdialog_struct *d = (edv_fopdialog_struct *)data;
        if(d == NULL)
            return;

        core_ptr = (edv_core_struct *)d->core_ptr;
        if(core_ptr == NULL)
            return;

        entry = (GtkEntry *)d->target_group_entry;
        if(entry == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Block input and get value. */
        value = PUListMapQuery(
            core_ptr->groups_pulist,
            gtk_entry_get_text(entry),
            15,
            PULIST_RELATIVE_CENTER,
            GTK_WIDGET(entry), widget
        );
        /* Got value? */
        if(value != NULL)
            gtk_entry_set_text(entry, value);

        reenterent = FALSE;
}


/*
 *	Creates a new edv_fopdialog_struct.
 */
edv_fopdialog_struct *EDVFOPDialogNew(gpointer core_ptr)
{
	gint border_minor = 2, border_major = 5;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w, *parent, *parent2, *parent3, *parent4,
			*parent5, *parent6, *parent7;
	edv_fopdialog_struct *d = (edv_fopdialog_struct *)g_malloc0(
	    sizeof(edv_fopdialog_struct)
	);
	if(d == NULL)
	    return(d);


	/* Reset values. */
	d->initialized = TRUE;
	d->map_state = FALSE;
	d->core_ptr = core_ptr;
	d->ref_toplevel = NULL;
	d->op = EDV_FOPDIALOG_OP_MOVE;
	d->src_obj = NULL;
	d->src_obj_total = 0;
	d->src_dir = NULL;

        /* Keyboard accelerator group. */
        d->accelgrp = accelgrp = gtk_accel_group_new();

        /* Toplevel. */
        d->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
        gtk_widget_realize(w);
        window = w->window;
        if(window != NULL)
        {
            gdk_window_set_decorations(
                window,
                GDK_DECOR_TITLE | GDK_DECOR_MENU | GDK_DECOR_MINIMIZE
            );
            gdk_window_set_functions(
                window,
                GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
            );
        }
        gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(EDVFOPDialogDeleteEventCB), d
        );
        gtk_container_border_width(GTK_CONTAINER(w), 0);
        gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
        parent = w;


        /* Main vbox. */
        d->main_vbox = w = gtk_vbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);
        parent = w;


        /* Hbox to hold icon and message label. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, border_major);
        gtk_widget_show(w);
        parent2 = w;


        /* Vbox to hold icon fixed. */
        w = gtk_vbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent3 = w;

        d->icon_fixed = w = gtk_fixed_new();
        d->icon_pm = NULL;
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
        gtk_widget_realize(w);
        gtk_widget_show(w);


	/* Vbox for label and entry. */
	w = gtk_vbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
        gtk_widget_show(w);
        parent3 = w;

        /* Label hbox. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent4 = w;
        /* Label. */
        d->label = w = gtk_label_new("Are you sure?");
        gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


	/* Hbox for target widgets. */
	w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
	/* Label alignment. */
	w = gtk_alignment_new(1.0, 0.0, 0.0, 0.0);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
        /* Label. */
        w = gtk_label_new("To:");
        gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_container_add(GTK_CONTAINER(parent5), w);
        gtk_widget_show(w);


	/* Target path entry. */
	d->target_entry = w = gtk_entry_new();
	gtk_widget_set_usize(w, 350, -1);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "activate",
            GTK_SIGNAL_FUNC(EDVFOPDialogProcessCB), d
        );
	EDVEntrySetDND((edv_core_struct *)core_ptr, w);

	/* Browse button. */
	d->target_browse_btn = w = (GtkWidget *)GUIButtonPixmap(
	    (u_int8_t **)icon_browse_20x20_xpm
	);
	gtk_widget_set_usize(w, 20 + 2, 20 + 2);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFOPDialogBrowseCB), d->target_entry
        );

        /* Vbox for target chmod labels and check buttons. */
        d->target_chmod_parent = w = gtk_vbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        parent5 = w;

        /* Labels hbox. */
        w = gtk_hbox_new(TRUE, border_major);
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent6 = w;
        w = gtk_label_new("Owner");
        gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
        gtk_widget_show(w);
        w = gtk_label_new("Group");
        gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
        gtk_widget_show(w);
        w = gtk_label_new("Other");
        gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
        gtk_widget_show(w);

        /* Check buttons hbox. */
        w = gtk_hbox_new(TRUE, border_major);
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent6 = w;

        /* Owner check button set hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
        gtk_widget_show(w);
        parent7 = w;
        /* Read check button. */
        d->target_uread_check = w = gtk_check_button_new_with_label("R");
        gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Write check button. */
        d->target_uwrite_check = w = gtk_check_button_new_with_label("W");
        gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Execute check button. */
        d->target_uexecute_check = w = gtk_check_button_new_with_label("X");
        gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Group check button set hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
        gtk_widget_show(w);
        parent7 = w;
        /* Read check button. */
        d->target_gread_check = w = gtk_check_button_new_with_label("R");
        gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Write check button. */
        d->target_gwrite_check = w = gtk_check_button_new_with_label("W");
        gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Execute check button. */
        d->target_gexecute_check = w = gtk_check_button_new_with_label("X");
        gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Other check button set hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
        gtk_widget_show(w);
        parent7 = w;
        /* Read check button. */
        d->target_aread_check = w = gtk_check_button_new_with_label("R");
        gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Write check button. */
        d->target_awrite_check = w = gtk_check_button_new_with_label("W");
        gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        /* Execute check button. */
        d->target_aexecute_check = w = gtk_check_button_new_with_label("X");
        gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
        gtk_widget_show(w);



        /* Hbox for setuid, setgid, and sticky permission check buttons. */
        w = gtk_hbox_new(TRUE, border_major);
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent6 = w;

        /* Setuid check button. */
        d->target_setuid_check = w = gtk_check_button_new_with_label("SetUID");
        gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
        gtk_widget_show(w);

        /* Setgid check button. */
        d->target_setgid_check = w = gtk_check_button_new_with_label("SetGID");
        gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
        gtk_widget_show(w);

        /* Sticky check button. */
        d->target_sticky_check = w = gtk_check_button_new_with_label("Sticky");
        gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
        gtk_widget_show(w);


        /* Vbox for target owned label, entry, and buttons. */
        d->target_chown_parent = w = gtk_vbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        parent5 = w;

        /* Owner entry hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent6 = w;
        /* Prefix label. */
        w = gtk_alignment_new(1.0, 0.5, 0.0, 0.0);
        gtk_widget_set_usize(w, 60, -1);
        gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent7 = w;
        w = gtk_label_new("Owner:");
        gtk_container_add(GTK_CONTAINER(parent7), w);
        gtk_widget_show(w);
        /* Owner entry. */
 	d->target_owner_entry = w = gtk_entry_new();
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(w, 250, -1);
        gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        /* Popup list map button. */
	d->target_owner_btn = w = PUListNewMapButton(
            EDVFOPDialogOwnerMapPUListCB, d
        );
        gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Group entry hbox. */
        w = gtk_hbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent6 = w;
        /* Prefix label. */
        w = gtk_alignment_new(1.0, 0.5, 0.0, 0.0);
        gtk_widget_set_usize(w, 60, -1);
        gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent7 = w;
        w = gtk_label_new("Group:");
        gtk_container_add(GTK_CONTAINER(parent7), w);
        gtk_widget_show(w);
        /* Group entry. */
        d->target_group_entry = w = gtk_entry_new();
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_widget_set_usize(w, 250, -1);
        gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        /* Popup list map button. */
        d->target_group_btn = w = PUListNewMapButton(
            EDVFOPDialogGroupMapPUListCB, d
        );
        gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
        gtk_widget_show(w);





        /* Separator. */
        w = gtk_hseparator_new();
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


        /* Buttons hbox. */
        w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Move button. */
        d->move_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_move_file_20x20_xpm, "Move", NULL
        );
        gtk_widget_set_usize(w, FOPDIALOG_BTN_WIDTH, FOPDIALOG_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFOPDialogProcessCB), d
        );

        /* Copy button. */
        d->copy_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_copy_file_20x20_xpm, "Copy", NULL
        );
        gtk_widget_set_usize(w, FOPDIALOG_BTN_WIDTH, FOPDIALOG_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFOPDialogProcessCB), d
        );

        /* Link button. */
        d->link_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_link2_20x20_xpm, "Link", NULL
        );
        gtk_widget_set_usize(w, FOPDIALOG_BTN_WIDTH, FOPDIALOG_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFOPDialogProcessCB), d
        );

        /* ChMod button. */
        d->chmod_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_chmod_20x20_xpm, "ChMod", NULL
        );
        gtk_widget_set_usize(w, FOPDIALOG_BTN_WIDTH, FOPDIALOG_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFOPDialogProcessCB), d
        );

        /* Chown button. */
        d->chown_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_owned_20x20_xpm, "Chown", NULL
        );
        gtk_widget_set_usize(w, FOPDIALOG_BTN_WIDTH, FOPDIALOG_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFOPDialogProcessCB), d
        );

        /* Cancel button. */
        d->cancel_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_cancel_20x20_xpm, "Cancel", NULL
        );
        gtk_widget_set_usize(w, FOPDIALOG_BTN_WIDTH, FOPDIALOG_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVFOPDialogCancelCB), d
        );
        gtk_accel_group_add(
            accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
	gtk_widget_show(w);



	return(d);
}


/*
 *	Deallocates source disk object structure list and current
 *	source directory on the given dialog.
 */
void EDVFOPDialogReset(edv_fopdialog_struct *d)
{
	gint i;


	if(d == NULL)
	    return;

	/* Begin deallocating memory. */

	/* Source disk object structures. */
        for(i = 0; i < d->src_obj_total; i++)
            EDVObjectDelete(d->src_obj[i]);
        g_free(d->src_obj);
        d->src_obj = NULL;
        d->src_obj_total = 0;

	/* Source directory. */
        g_free(d->src_dir);
        d->src_dir = NULL;
}

/*
 *	Maps the operation dialog with respect to the given inputs.
 *
 *	The given list of src_obj must contain atleast one object, this
 *	list will not be modified by this function.
 */
void EDVFOPDialogMapValues(
        edv_fopdialog_struct *d,
        gint op,                /* One ofEDV_FOPDIALOG_OP_*. */
        edv_object_struct **src_obj,    /* Source list, will be coppied. */
        gint src_obj_total,
        const gchar *src_dir,
	GtkWidget *toplevel
)
{
	gint i;
	gbool need_warn_chmod_link = FALSE;
	const gchar *title = NULL;
	gchar *buf = NULL;
	const gchar *src_obj_full_path, *src_obj_name;
	guint8 **icon_data = NULL;
	edv_object_struct *obj;
	edv_core_struct *core_ptr;
	gbool map_browse_btn = FALSE;
	GtkWidget *w, *parent, *button = NULL, *target_w = NULL;


	if(d == NULL)
	    return;

	core_ptr = (edv_core_struct *)d->core_ptr;
	if(core_ptr == NULL)
	    return;

        /* Deallocate source disk objects structure list and current
         * source directory on the dialog.
         */
        EDVFOPDialogReset(d);


	/* If there is exactly one object in the source object list then
	 * get its full path and name. Otherwise set the single object
	 * full path and name to NULL.
	 */
	if(src_obj_total == 1)
	{
	    obj = src_obj[0];
	    src_obj_name = (obj != NULL) ? obj->name : NULL;
	    src_obj_full_path = (obj != NULL) ? obj->full_path : NULL;
	}
	else
	{
	    obj = NULL;
            src_obj_name = NULL;
            src_obj_full_path = NULL;
        }


	/* Set operation code. */
	d->op = op;

	/* Update widgets based on operation code. */
	switch(op)
	{
	  case EDV_FOPDIALOG_OP_MOVE:
	    title = "Move";
	    icon_data = (guint8 **)icon_move_file_32x32_xpm;
	    if(src_obj_name != NULL)
		buf = g_strdup_printf(
"Move:\n\
\n\
    %s\n",
		    src_obj_name
		);
	    else
		buf = g_strdup_printf(
"Move %i objects\n",
		    src_obj_total
                );
	    button = d->move_btn;
	    target_w = d->target_entry;
	    map_browse_btn = TRUE;
	    break;

	  case EDV_FOPDIALOG_OP_COPY:
	    title = "Copy";
	    icon_data = (guint8 **)icon_copy_file_32x32_xpm;
	    if(src_obj_name != NULL)
		buf = g_strdup_printf(
"Copy:\n\
\n\
    %s\n",
		    src_obj_name
                );
	    else
		buf = g_strdup_printf(
"Copy %i objects\n",
		    src_obj_total
		);
            button = d->copy_btn;
            target_w = d->target_entry;
	    map_browse_btn = TRUE;
	    break;

	  case EDV_FOPDIALOG_OP_LINK:
	    title = "Link";
            icon_data = (guint8 **)icon_link2_32x32_xpm;
	    if(src_obj_name != NULL)
		buf = g_strdup_printf(
"Link:\n\
\n\
    %s\n",
		    src_obj_name
		);
	    else
		buf = g_strdup_printf(
"Link %i objects\n",
		    src_obj_total
		);
            button = d->link_btn;
            target_w = d->target_entry;
	    map_browse_btn = TRUE;
	    break;

          case EDV_FOPDIALOG_OP_CHMOD:
            title = "Change Permissions";
            icon_data = (guint8 **)icon_chmod_32x32_xpm;
            if(src_obj_name != NULL)
                buf = g_strdup_printf(
"Change permissions of:\n\
\n\
    %s\n",
                    src_obj_name
                );
            else
                buf = g_strdup_printf(
"Change permissions of %i objects\n",
                    src_obj_total
                );
            button = d->chmod_btn;
            target_w = d->target_chmod_parent;
	    break;

          case EDV_FOPDIALOG_OP_CHOWN:
            title = "Chown";
            icon_data = (guint8 **)icon_owned_32x32_xpm;
            if(src_obj_name != NULL)
                buf = g_strdup_printf(
"Chown:\n\
\n\
    %s\n",
                    src_obj_name
                );
            else
                buf = g_strdup_printf(
"Chown %i objects\n",
                    src_obj_total
                );
            button = d->chown_btn;
            target_w = d->target_chown_parent;
            break;

	}

	/* Update icon. */
	parent = d->icon_fixed;
	if((icon_data != NULL) && (parent != NULL) && (d->toplevel != NULL))
	{
	    gint width, height;
	    GdkWindow *window;
            GdkBitmap *mask;
	    GdkPixmap *pixmap;
	    GtkStyle *style;


	    /* Get style, first try getting style of parent, then toplevel,
	     * and if all else fails get the default.
	     */
	    style = gtk_widget_get_style(parent);
	    if(style == NULL)
                style = gtk_widget_get_style(d->toplevel);
            if(style == NULL)
                style = gtk_widget_get_default_style();

	    /* Get GdkWindow from dialog toplevel widget. */
	    window = d->toplevel->window;

            if((style != NULL) && (window != NULL))
	    {
		/* Create new pixmap and mask pair from icon_data. */
		pixmap = gdk_pixmap_create_from_xpm_d(
                    window, &mask,
                    &style->bg[GTK_STATE_NORMAL],
                    (gchar **)icon_data
                );
                if(pixmap != NULL)
                {
		    /* Get size of new pixmap and mask pair. */
		    gdk_window_get_size((GdkWindow *)pixmap, &width, &height);

		    /* Get icon GtkPixmap, create it as needed. */
		    w = d->icon_pm;
		    if(w == NULL)
		    {
			d->icon_pm = w = gtk_pixmap_new(pixmap, mask);
			gtk_fixed_put(GTK_FIXED(parent), w, 0, 0);
			gtk_widget_show(w);
		    }
		    else
		    {
			gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
		    }

		    /* Adjust size of fixed widget to fit pixmap. */
		    gtk_widget_set_usize(parent, width, height);
		    gtk_widget_queue_resize(parent);
		    gtk_widget_shape_combine_mask(parent, mask, 0, 0);

		    /* Unref newly loaded pixmap and mask pair, the refcounts
		     * are no longer needed.
		     */
		    gdk_pixmap_unref(pixmap);
		    pixmap = NULL;
		    if(mask != NULL)
		    {
			gdk_bitmap_unref(mask);
			mask = NULL;
		    }
		}
	    }

	    /* Update toplevel GdkWindow's WM icon. */
	    GUISetWMIcon(window, (u_int8_t **)icon_data);
	}

	/* Update label. */
	if(buf != NULL)
	{
	    w = d->label;
	    if(w != NULL)
		gtk_label_set_text(GTK_LABEL(w), buf);

	    /* Deallocate message, it is no longer needed. */
	    g_free(buf);
	    buf = NULL;
	}


	/* Update source directory. */
        g_free(d->src_dir);
        d->src_dir = (src_dir != NULL) ? g_strdup(src_dir) : NULL;

	/* Update target entry. */
	w = d->target_entry;
	if((w != NULL) && (w == target_w))
	{
	    gtk_entry_set_text(
		GTK_ENTRY(w), (src_dir != NULL) ? src_dir : ""
	    );
            gtk_widget_grab_focus(w);
            gtk_widget_grab_default(w);
	}


	/* Copy the given list of disk objects to our dialog's list of
	 * disk objects (copy both the pointer array and each object
	 * structure).
	 */
	d->src_obj_total = src_obj_total;
	if(d->src_obj_total > 0)
	{
	    /* Allocate our list of disk object structures pointer
	     * array.
	     */
	    d->src_obj = (edv_object_struct **)g_malloc0(
		d->src_obj_total * sizeof(edv_object_struct *)
	    );
	    if(d->src_obj == NULL)
	    {
		d->src_obj_total = 0;
	    }
	    else
	    {
		/* Make a copy of each disk object structure from the
		 * given list of source disk objects.
		 */
		for(i = 0; i < d->src_obj_total; i++)
		    d->src_obj[i] = EDVObjectCopy(src_obj[i]);
	    }
	}
	else
	{
	    d->src_obj_total = 0;
	}


	/* Update permissions? */
	w = d->target_chmod_parent;
	if((w != NULL) && (w == target_w))
	{
	    /* Connect permissions from the list of objects coppied to our
	     * dialog structure. Gather accumulative permissions mask to
	     * contain all permissions that the objects posess.
	     */
	    guint permissions = 0x00000000;
	    edv_object_struct *obj_ptr;

	    /* Iterate through objects. */
	    for(i = 0; i < d->src_obj_total; i++)
	    {
		obj_ptr = d->src_obj[i];
		if(obj_ptr == NULL)
		    continue;

		/* Do a quick check to see if the object to change
		 * permissions of is a symbolic link. If this is a symbolic
		 * link then a warning needs to be issued about changing
		 * a symbolic link's permissions that will actually change
		 * its destination permissions.
		 */
		if(obj_ptr->type == EDV_OBJECT_TYPE_LINK)
		    need_warn_chmod_link = TRUE;

		/* Gather permissions from this object. */
		permissions |= obj_ptr->permissions;
	    }

	    /* Begin setting permission check widgets. */
	    w = d->target_uread_check;
	    if(w != NULL)
		GTK_TOGGLE_BUTTON(w)->active =
		    (permissions & EDV_PERMISSION_UREAD) ? TRUE : FALSE;
            w = d->target_uwrite_check;
            if(w != NULL)
                GTK_TOGGLE_BUTTON(w)->active =
                    (permissions & EDV_PERMISSION_UWRITE) ? TRUE : FALSE;
            w = d->target_uexecute_check;
            if(w != NULL)
                GTK_TOGGLE_BUTTON(w)->active =
                    (permissions & EDV_PERMISSION_UEXECUTE) ? TRUE : FALSE;

            w = d->target_gread_check;
            if(w != NULL)
                GTK_TOGGLE_BUTTON(w)->active =
                    (permissions & EDV_PERMISSION_GREAD) ? TRUE : FALSE;
            w = d->target_gwrite_check;
            if(w != NULL)
                GTK_TOGGLE_BUTTON(w)->active =
                    (permissions & EDV_PERMISSION_GWRITE) ? TRUE : FALSE;
            w = d->target_gexecute_check;
            if(w != NULL)
                GTK_TOGGLE_BUTTON(w)->active =
                    (permissions & EDV_PERMISSION_GEXECUTE) ? TRUE : FALSE;

            w = d->target_aread_check;
            if(w != NULL)
                GTK_TOGGLE_BUTTON(w)->active =
                    (permissions & EDV_PERMISSION_AREAD) ? TRUE : FALSE;
            w = d->target_awrite_check;
            if(w != NULL)
                GTK_TOGGLE_BUTTON(w)->active =
                    (permissions & EDV_PERMISSION_AWRITE) ? TRUE : FALSE;
            w = d->target_aexecute_check;
            if(w != NULL)
                GTK_TOGGLE_BUTTON(w)->active =
                    (permissions & EDV_PERMISSION_AEXECUTE) ? TRUE : FALSE;

            w = d->target_setuid_check;
            if(w != NULL)
                GTK_TOGGLE_BUTTON(w)->active =
                    (permissions & EDV_PERMISSION_SETUID) ? TRUE : FALSE;
            w = d->target_setgid_check;
            if(w != NULL)
                GTK_TOGGLE_BUTTON(w)->active =
                    (permissions & EDV_PERMISSION_SETGID) ? TRUE : FALSE;
            w = d->target_sticky_check;
            if(w != NULL)
                GTK_TOGGLE_BUTTON(w)->active =
                    (permissions & EDV_PERMISSION_STICKY) ? TRUE : FALSE;
	}

        /* Update owner and group? */
        w = d->target_chown_parent;
        if((w != NULL) && (w == target_w))
        {
	    w = d->target_owner_entry;
	    if(w != NULL)
		gtk_entry_set_text(
		    GTK_ENTRY(w),
		    EDVUIDGetNameFromUID(
			core_ptr->uid, core_ptr->total_uids,
			(obj != NULL) ?
			    obj->owner_id : core_ptr->effective_user_id,
			NULL
		    )
		);

            w = d->target_group_entry;
            if(w != NULL)
                gtk_entry_set_text(
                    GTK_ENTRY(w),
                    EDVGIDGetNameFromGID(
                        core_ptr->gid, core_ptr->total_gids,
                        (obj != NULL) ?
                            obj->group_id : core_ptr->effective_group_id,
			NULL
                    )
                );
	}


	/* Unmap all buttons. */
	w = d->move_btn;
	if(w != NULL)
	    gtk_widget_hide(w);
        w = d->copy_btn;
        if(w != NULL)
            gtk_widget_hide(w);
        w = d->link_btn;
        if(w != NULL)
            gtk_widget_hide(w);
        w = d->chmod_btn;
        if(w != NULL)
            gtk_widget_hide(w);
        w = d->chown_btn;
        if(w != NULL)
            gtk_widget_hide(w);

	/* Check if button to be mapped is valid and map it. */
	if(button != NULL)
	    gtk_widget_show(button);


	/* Unmap all target widgets. */
	w = d->target_entry;
        if(w != NULL)
            gtk_widget_hide(w);
        w = d->target_chmod_parent;
        if(w != NULL)
            gtk_widget_hide(w);
        w = d->target_chown_parent;
        if(w != NULL)
            gtk_widget_hide(w);

        /* Check if target widget to be mapped is valid and map it. */
        if(target_w != NULL)
            gtk_widget_show(target_w);

	/* Map/unmap browse button. */
	w = d->target_browse_btn;
        if(w != NULL)
	{
	    if(map_browse_btn)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	/* Need to print warning about changing permissions on link
	 * objects?
	 */
	if(need_warn_chmod_link)
	{
            CDialogSetTransientFor(toplevel);
            CDialogGetResponse(
"Changing Permissions Warning",
"Changing the permissions on a symbolic link object\n\
will effectively change the permissions of its destination\n\
object.",
"One or more of the selected object(s) are of type\n\
symbolic link. Symbolic links do not have permissions,\n\
instead their permissions are determined by the permissions\n\
of the destination object. So changing the permissions on a\n\
symbolic link will effectively change the permissions of its\n\
destination object.",
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	}


	/* Update toplevel. */
	w = d->toplevel;
	if(w != NULL)
	{
	    /* Update title? */
	    if(title != NULL)
		gtk_window_set_title(GTK_WINDOW(w), title);

	    /* If given toplevel is not NULL then set modal. */
	    if((toplevel != NULL) ? GTK_IS_WINDOW(toplevel) : FALSE)
	    {
                gtk_window_set_modal(GTK_WINDOW(w), TRUE);
                gtk_window_set_transient_for(
                    GTK_WINDOW(w), GTK_WINDOW(toplevel)
                );
		d->ref_toplevel = toplevel;
	    }

	    /* Map toplevel. */
	    gtk_widget_show_raise(w);

	    d->map_state = TRUE;
	}


}

/*
 *	Unmaps the file operation dialog.
 */
void EDVFOPDialogUnmap(edv_fopdialog_struct *d)
{
        GtkWidget *w;


        if(d == NULL)
            return;

        if(!d->initialized)
            return;

	w = d->toplevel;
	if(w == NULL)
	    return;

	/* Need to unmap the dialog? */
        if(d->map_state)
        {
	    gtk_widget_hide(w);
	    d->map_state = FALSE;
        }

        /* Unset modal and transiency of the dialog. */
        gtk_window_set_modal(GTK_WINDOW(w), FALSE);
	gtk_window_set_transient_for(GTK_WINDOW(w), NULL);
	d->ref_toplevel = NULL;
}

/*
 *	Deallocates all resources on the given fop dialog including the
 *	structure itself.
 */
void EDVFOPDialogDelete(edv_fopdialog_struct *d)
{
	GtkWidget **w;


	if(d == NULL)
	    return;

	/* Deallocate source disk objects structure list and source
	 * directory on the dialog.
	 */
	EDVFOPDialogReset(d);

        if(d->initialized)
        {
#define DO_DESTROY_WIDGET	\
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	    /* Reset transiency toplevel, do not destroy it (shared). */
	    d->ref_toplevel = NULL;


	    /* Begin destroying widgets. */
	    w = &d->icon_pm;
	    DO_DESTROY_WIDGET
	    w = &d->icon_fixed;
	    DO_DESTROY_WIDGET

	    w = &d->toplevel;
            DO_DESTROY_WIDGET

            if(d->accelgrp != NULL)
            {
                gtk_accel_group_unref(d->accelgrp);
                d->accelgrp = NULL;
            }

#undef DO_DESTROY_WIDGET
        }

	/* Deallocate structure itself. */
	g_free(d);
}
