#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../../include/fio.h"
#include "../../include/disk.h"
#include "../../include/prochandle.h"

#include "../guiutils.h"

#include "../edvtypes.h"
#include "../edvdevices.h"

#include "formatmanager.h"
#include "formatcb.h"

#include "config.h"

#include "../images/icon_run_20x20.xpm"
#include "../images/icon_stop_20x20.xpm"
#include "../images/icon_clear_20x20.xpm"
#include "../images/icon_close_20x20.xpm"
#include "../images/icon_floppy_48x48.xpm"

#include "disk_icon_00_48x48.xpm"
#include "disk_icon_01_48x48.xpm"
#include "disk_icon_02_48x48.xpm"


void FormatManagerSetDevices(
        format_manager_struct *fm,
	edv_device_struct **device, gint total_devices
);
void FormatManagerAppendMessage(
        format_manager_struct *fm,
	GdkFont *font, GdkColor *color_fg, GdkColor *color_bg,
	const gchar *text,
        gboolean allow_auto_scroll
);

format_manager_struct *FormatManagerNew(edv_context_struct *ctx);
void FormatManagerUpdateMenus(format_manager_struct *fm);
void FormatManagerSetBusy(format_manager_struct *fm, gboolean is_busy);
void FormatManagerSetProgress(
        format_manager_struct *fm, gfloat value,
        gboolean allow_gtk_iteration
);
void FormatManagerMap(format_manager_struct *fm);
void FormatManagerUnmap(format_manager_struct *fm);
void FormatManagerDelete(format_manager_struct *fm);


#define FORMAT_MANAGER_WIDTH	640
#define FORMAT_MANAGER_HEIGHT	480

#define FORMAT_MANAGER_BTN_WIDTH	(100 + (2 * 3))
#define FORMAT_MANAGER_BTN_HEIGHT	(30 + (2 * 3))

#define EDV_STATUS_BAR_HEIGHT   26


/*
 *	Loads the given list of devices to the format manager window's
 *	devices listing and devices clist.
 *
 *	The given list of devices should not be referenced again
 *	after this call.
 */
void FormatManagerSetDevices(
        format_manager_struct *fm,
        edv_device_struct **device, gint total_devices
)
{
	gint i, row, column;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	const gchar *cell_text;
	GtkCList *clist;
	edv_device_struct *dev_ptr;
	gchar *strv[4];

#define DO_DEALLOCATE_INPUT_LIST	\
{ \
 for(i = 0; i < total_devices; i++) \
  EDVDeviceDelete(device[i]); \
 g_free(device); \
}

	if(fm == NULL)
	{
	    DO_DEALLOCATE_INPUT_LIST
	    return;
	}

	clist = (GtkCList *)fm->devices_clist;
	if(clist == NULL)
	{
	    DO_DEALLOCATE_INPUT_LIST
	    return;
	}

	gtk_clist_freeze(clist);


	/* Clear clist and deallocate previoust list of devices (if
	 * any).
	 */
	gtk_clist_clear(clist);

	for(i = 0; i < fm->total_devices; i++)
	    EDVDeviceDelete(fm->device[i]);
	g_free(fm->device);


	/* Assign the given list of devices to the format manager's
	 * listing of devices.
	 */
	fm->device = device;
	fm->total_devices = total_devices;

	/* Update the clist's listing of devices. */
	for(i = 0; i < 4; i++)
	    strv[i] = "";
	for(i = 0; i < fm->total_devices; i++)
	{
            dev_ptr = fm->device[i];
            if(dev_ptr == NULL)
                continue;

	    /* Hide this device from general listing? */
	    if(dev_ptr->unlisted)
		continue;

            /* Append a new clist row. */
            row = gtk_clist_append(clist, strv);

	    /* Realize device structure so we can have use of its
	     * icon pixmap and mask pairs.
	     */
	    EDVDeviceRealize(dev_ptr, FALSE);

	    /* Get small icon pixmap and mask pair for this row. */
	    pixmap = dev_ptr->small_pixmap[
		EDV_DEVICE_ICON_STATE_STANDARD
	    ];
	    mask = dev_ptr->small_mask[
		EDV_DEVICE_ICON_STATE_STANDARD
	    ];


	    /* Set name cell. */
	    column = 0;
	    cell_text = dev_ptr->name;
	    if((cell_text == NULL) && (dev_ptr->device_path != NULL))
	    {
		cell_text = strrchr(dev_ptr->device_path, DIR_DELIMINATOR);
		if(cell_text != NULL)
                    cell_text++;
                else
                    cell_text = dev_ptr->device_path;
            }
            if(cell_text == NULL)
                cell_text = "(null)";

            if(pixmap != NULL)
                gtk_clist_set_pixtext(
                    clist, row, column, cell_text,
                    EDV_LIST_PIXMAP_TEXT_SPACING,
                    pixmap, mask
                );
            else
                gtk_clist_set_text(
                    clist, row, column, cell_text
                );

	    /* Set device path cell. */
	    column = 1;
	    cell_text = dev_ptr->device_path;
	    if(cell_text == NULL)
                cell_text = "(null)";

            gtk_clist_set_text(
                clist, row, column, cell_text
            );


	    /* Set mount path cell. */
	    column = 2;
	    cell_text = dev_ptr->mount_path;
            if(cell_text == NULL)
                cell_text = "(null)";

            gtk_clist_set_text(
                clist, row, column, cell_text
            );


            /* Set fstype cell. */
	    column = 3;
            gtk_clist_set_text(
                clist, row, column,
                EDVDeviceGetFSStringFromNumber(dev_ptr->fs_type)
            );


	    /* Set row data of this new row to be a gint indicating
	     * the device listing index of the device that this row
	     * is displaying.
	     */
	    gtk_clist_set_row_data(clist, row, (gpointer)i);
	}

	gtk_clist_thaw(clist);

	FormatManagerUpdateMenus(fm);

#undef DO_DEALLOCATE_INPUT_LIST
}

/*
 *	Appends the given text to the output text on the format manager
 *	window.
 */
void FormatManagerAppendMessage(
        format_manager_struct *fm,
        GdkFont *font, GdkColor *color_fg, GdkColor *color_bg,
        const gchar *text,
	gboolean allow_auto_scroll
)
{
	GtkWidget *w;
	GtkEditable *editable;
	GtkText *text_w;


	if((fm == NULL) || (text == NULL))
	    return;

	w = fm->output_text;
	if(w == NULL)
	    return;

        editable = GTK_EDITABLE(w);
        text_w = GTK_TEXT(w);

	gtk_text_insert(
	    text_w, font, color_fg, color_bg, text, -1
	);

	if(allow_auto_scroll)
	{
	    GtkAdjustment *adj = text_w->vadj;

	    /* Need to scroll down to show text not visible? */
	    if(((adj->upper - adj->lower) > adj->page_size) &&
	       ((adj->value + adj->page_size) < adj->upper)
	    )
	    {
                adj->value = adj->upper - adj->page_size;
                if(adj->value < adj->lower)
                    adj->value = adj->lower;
                gtk_signal_emit_by_name(
                    GTK_OBJECT(adj), "value_changed"
                );
            }
        }
}


/*
 *	Creates a new format manager window.
 */
format_manager_struct *FormatManagerNew(edv_context_struct *ctx)
{
        gint border_major = 5, border_minor = 2;
	gint i;
	GList *glist;
        GdkColor *c;
	GdkColormap *colormap;
	GtkAdjustment *adj;
        GdkWindow *window;
	GtkStyle *style;
	gpointer entry_rtn, combo_rtn;
        GtkWidget	*w, *parent, *parent2, *parent3, *parent4,
			*parent5, *main_vbox;
        GtkAccelGroup *accelgrp;
	GtkEditable *editable;
	GtkText *text;
        GtkCList *clist;
	GtkCombo *combo;
        gchar *heading[4];
        format_manager_struct *fm = (format_manager_struct *)g_malloc0(
            sizeof(format_manager_struct)
        );
        if(fm == NULL)
            return(NULL);


        /* Reset values. */
        fm->initialized = TRUE;
        fm->map_state = FALSE;
	fm->ctx = ctx;
	fm->busy_count = 0;
	fm->queued_device_to_format = NULL;
	fm->format_stage = FORMAT_STAGE_FORMAT;
	fm->format_toid = (guint)-1;
	fm->format_pid = 0;
	fm->stdout_fp = NULL;
	fm->stderr_fp = NULL;
	fm->stdout_path = NULL;
	fm->stderr_path = NULL;
        fm->stop_count = 0;
	fm->progress_pos_last = -1.0;
	fm->device = NULL;
	fm->total_devices = 0;

	/* Load cursors. */
	fm->busy_cur = gdk_cursor_new(GDK_WATCH);
	fm->text_cur = gdk_cursor_new(GDK_XTERM);

        /* Keyboard accelerator group. */
        fm->accelgrp = accelgrp = gtk_accel_group_new();

	/* Load fonts. */
	fm->text_font = gdk_font_load(
	    PROG_GTK_FONT_NAME_TEXT_EDITABLE
	);

        /* Toplevel. */
        fm->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
        gtk_widget_set_usize(w, FORMAT_MANAGER_WIDTH, FORMAT_MANAGER_HEIGHT);
        gtk_widget_realize(w);
        window = w->window;
        if(window != NULL)
        {
            gdk_window_set_decorations(
                window,
                GDK_DECOR_TITLE | GDK_DECOR_MENU | GDK_DECOR_MINIMIZE
            );
            gdk_window_set_functions(
                window,
                GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
            );
            GUISetWMIcon(window, (u_int8_t **)icon_floppy_48x48_xpm);
        }
        gtk_window_set_title(GTK_WINDOW(w), PROG_NAME);
        gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(FormatManagerCloseCB),
            (gpointer)fm
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "destroy",
            GTK_SIGNAL_FUNC(FormatManagerDestroyCB),
            (gpointer)fm
        );
        gtk_container_border_width(GTK_CONTAINER(w), 0);
        gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
        parent = w;

        /* Main vbox. */
        fm->main_vbox = main_vbox = w = gtk_vbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);
        parent = w;

	/* Hbox for devices clist, start and stop buttons. */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Vbox for devices clist and format parameters. */
	w = gtk_vbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent3 = w;

	/* Scrolled window for devices clist. */
        w = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(
            GTK_SCROLLED_WINDOW(w),
            GTK_POLICY_AUTOMATIC,
            GTK_POLICY_AUTOMATIC
        );
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent4 = w;

        /* Devices clist. */
#ifdef PROG_LANGUAGE_ENGLISH
        heading[0] = g_strdup("Name");
        heading[1] = g_strdup("Device Path");
        heading[2] = g_strdup("Mount Path");
        heading[3] = g_strdup("File System");
#endif
#ifdef PROG_LANGUAGE_SPANISH
        heading[0] = g_strdup("El Nombre");
        heading[1] = g_strdup("El Sendero Del Artefacto");
        heading[2] = g_strdup("Monte Sendero");
        heading[3] = g_strdup("Archive Sistema");
#endif
#ifdef PROG_LANGUAGE_FRENCH
        heading[0] = g_strdup("Nom");
        heading[1] = g_strdup("Sentier D'appareil");
        heading[2] = g_strdup("Monter Le Sentier");
        heading[3] = g_strdup("Classer Le Systme");
#endif
        fm->devices_clist = w = gtk_clist_new_with_titles(4, heading);
        g_free(heading[0]);
        g_free(heading[1]);
        g_free(heading[2]);
        g_free(heading[3]);
        clist = GTK_CLIST(w);
        gtk_widget_add_events(
            w,
            GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
            GDK_BUTTON_PRESS_MASK
        );
/*
        gtk_signal_connect(
            GTK_OBJECT(w), "key_press_event",
            GTK_SIGNAL_FUNC(EDVArchiverKeyEventCB), fm
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "key_release_event",
            GTK_SIGNAL_FUNC(EDVArchiverKeyEventCB), fm
        );
 */
	gtk_signal_connect_after(
            GTK_OBJECT(w), "button_press_event",
            GTK_SIGNAL_FUNC(FormatManagerButtonPressEventCB), fm
        );
        gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
        gtk_clist_set_column_width(clist, 0, 130);
        gtk_clist_set_column_width(clist, 1, 130);
        gtk_clist_set_column_width(clist, 2, 130);
        gtk_clist_set_column_width(clist, 3, 50);
        gtk_clist_column_titles_passive(clist);
        gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
        gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
        gtk_widget_set_usize(w, -1, 150);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_signal_connect(
            GTK_OBJECT(w), "select_row",
            GTK_SIGNAL_FUNC(FormatManagerSelectRowCB), fm
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "unselect_row",
            GTK_SIGNAL_FUNC(FormatManagerUnselectRowCB), fm
        );
        gtk_widget_show(w);


	/* Devices clist menu. */
	if(1)
        {
#define DO_ADD_MENU_ITEM_LABEL	\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_LABEL, accel_group, \
  icon, label, accel_key, accel_mods, (void **)&fw, \
  mclient_data, func_cb \
 ); \
}
#define DO_ADD_MENU_SEP		\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}

	    GtkWidget *menu = (GtkWidget *)GUIMenuCreate();
	    u_int8_t **icon;
	    const gchar *label;
	    gint accel_key;
	    guint accel_mods;
	    gpointer accel_group = NULL;
	    gpointer mclient_data = fm;
	    void (*func_cb)(GtkWidget *w, gpointer);
	    gpointer fw;


            icon = (u_int8_t **)icon_run_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Start";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Comienzo";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Dbut";
#endif
            accel_key = 0;
            accel_mods = 0;
            func_cb = FormatManagerStartCB;
            DO_ADD_MENU_ITEM_LABEL
            fm->devices_start_mi = w;

            icon = (u_int8_t **)icon_stop_20x20_xpm;
#ifdef PROG_LANGUAGE_ENGLISH
            label = "Stop";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            label = "Parada";
#endif
#ifdef PROG_LANGUAGE_FRENCH
            label = "Arrt";
#endif
            accel_key = 0;
            accel_mods = 0;
            func_cb = FormatManagerStopCB;
            DO_ADD_MENU_ITEM_LABEL
            fm->devices_stop_mi = w;




	    fm->devices_menu = menu;
#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SEP
	}





	/* Frame for format parameters. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Options"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Opciones"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Options"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent4 = w;

        w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);
        parent4 = w;

	w = gtk_hbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent5 = w;

        /* Quick format. */
        fm->quick_format_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Quick Format"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "El Formato Rpido"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Format Rapide"
#endif
        );
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
"Check this to skip the format stage and only create the file system"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Verifique esto saltarse la etapa de formato y slo crear sistema del archivo"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Vrifier ceci pour sauter l'tape de format et cre seulement systme de fichier"
#endif
        );
        gtk_widget_show(w);
        gtk_widget_show(w);
        GTK_TOGGLE_BUTTON(w)->active = FALSE;


        /* List of supported file system types. */
        glist = NULL;
        glist = g_list_append(glist, "360 kb");
        glist = g_list_append(glist, "720 kb");
        glist = g_list_append(glist, "1200 kb");
        glist = g_list_append(glist, "1440 kb");
        /* File system type combo. */
        w = (GtkWidget *)GUIComboCreate(
#ifdef PROG_LANGUAGE_ENGLISH
            "Capacity:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Capacidad:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Capacit:",
#endif
            "",
            glist,
            4,		/* Maximum items. */
            &combo_rtn,
            fm, NULL, NULL
        );
        fm->capacity_combo = (GtkWidget *)combo_rtn;
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        w = fm->capacity_combo;
        combo = GTK_COMBO(w);
        gtk_widget_set_usize(w, 80, -1);

        g_list_free(glist);

        w = combo->entry;
        gtk_entry_set_editable(GTK_ENTRY(w), FALSE);
	gtk_entry_set_text(GTK_ENTRY(w), "1440 kb");
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
"Select the actual capacity of the drive"
#endif
#ifdef PROG_LANGUAGE_SPANISH
""
#endif
#ifdef PROG_LANGUAGE_FRENCH
""
#endif
        );

	/* Volume label entry. */
	w = GUIPromptBar(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
            "Volume Label:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Etiqueta Volumen:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Etiquette Volume:",
#endif
	    NULL, &entry_rtn
	);
	fm->volume_label_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent5), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
	w = (GtkWidget *)entry_rtn;
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
"Enter the volume label to be placed on the media after format\
 (up to 11 characters)"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Entre la etiqueta del volumen para ser colocada en los medios despus de formato"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Entrer l'tiquette de volume tre place sur le presse aprs le format"
#endif
        );


	/* Hbox for second row of options. */
        w = gtk_hbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent5 = w;

	/* List of supported file system types. */
	glist = NULL;
        glist = g_list_append(glist, EDV_FS_TYPE_MSDOS_NAME);
        glist = g_list_append(glist, EDV_FS_TYPE_EXT2_NAME);
        glist = g_list_append(glist, EDV_FS_TYPE_MINIX_NAME);
        /* File system type combo. */
        w = (GtkWidget *)GUIComboCreate(
#ifdef PROG_LANGUAGE_ENGLISH
            "File System Type:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Archive El Tipo De Sistema:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Classer Le Type De Systme:",
#endif
            "",
            glist,
            3,		/* Maximum items. */
            &combo_rtn,
            fm, NULL, NULL
        );
        fm->file_system_type_combo = (GtkWidget *)combo_rtn;
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        w = fm->file_system_type_combo;
        combo = GTK_COMBO(w);
        gtk_widget_set_usize(w, 80, -1);

        g_list_free(glist);

        w = combo->entry;
        gtk_entry_set_editable(GTK_ENTRY(w), FALSE);
        gtk_entry_set_text(GTK_ENTRY(w), EDV_FS_TYPE_MSDOS_NAME);
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
"Select the type of file system to be created on the media"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Escoja el tipo de sistema de archivo para ser creado en los medios"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Choisir le type de systme de fichier tre cr sur le presse"
#endif
        );


	/* Verbose. */
        fm->verbose_check = w = gtk_check_button_new_with_label(
#ifdef PROG_LANGUAGE_ENGLISH
            "Verbose"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Detallado"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Verbose"
#endif
        );
        gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Check this display additional detailed messages during format"
#endif
#ifdef PROG_LANGUAGE_SPANISH
	    "Verifique este despliegue los mensajes adicionales durante formato"
#endif
#ifdef PROG_LANGUAGE_FRENCH
	    "Vrifier cette exposition ces messages dtaills pendant le format"
#endif
        );
        gtk_widget_show(w);
        GTK_TOGGLE_BUTTON(w)->active = TRUE;



        /* Vbox for buttons. */
        w = gtk_vbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

        /* Hbox for disk animation icon drawing area. */
        w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

	/* Disk animation icon drawing area. */
	fm->disk_anim_da = w = gtk_drawing_area_new();
        gtk_drawing_area_size(GTK_DRAWING_AREA(w), 48, 48);
        gtk_widget_add_events(w, GDK_EXPOSURE_MASK);
        gtk_signal_connect(
            GTK_OBJECT(w), "expose_event",
            GTK_SIGNAL_FUNC(FormatManagerDiskIconExposeCB), fm
        );
        gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_realize(w);
        gtk_widget_show(w);
	/* Graphics context for disk animation icon. */
	fm->gc = gdk_gc_new(w->window);
	/* Pixmap buffer for disk animation icon. */
	fm->disk_anim_pm = gdk_pixmap_new(w->window, 48, 48, -1);


	/* Vbox for start and stop buttons. */
	w = gtk_vbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;


        /* Start button. */
        fm->start_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_run_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Start",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Comienzo",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Dbut",
#endif
	    NULL
        );
        gtk_widget_set_usize(
            w,
            FORMAT_MANAGER_BTN_WIDTH,
            FORMAT_MANAGER_BTN_HEIGHT
        );
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Start formatting the media in the selected device"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "El comienzo formatting la media el artefacto escogido"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Le dbut pour formatting le media l'appareil choisi"
#endif
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(FormatManagerStartCB),
            (gpointer)fm
        );
        gtk_accel_group_add(
            accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_accel_group_add(
            accelgrp, GDK_3270_Enter, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_accel_group_add(
            accelgrp, GDK_KP_Enter, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_accel_group_add(
            accelgrp, GDK_ISO_Enter, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_widget_show(w);

        /* Stop button. */
        fm->stop_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_stop_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Stop",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Parada",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Arrt",
#endif
	    NULL
        );
        gtk_widget_set_usize(
            w,
            FORMAT_MANAGER_BTN_WIDTH,
            FORMAT_MANAGER_BTN_HEIGHT
        );
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Stop the current formatting process"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Pare la corriente el proceso que formatting"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Arrter le courant procd qui formatting"
#endif
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(FormatManagerStopCB),
            (gpointer)fm
        );
        gtk_accel_group_add(
            accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_widget_show(w);



	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Hbox for output text and buttons. */
        w = gtk_hbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_widget_show(w);
        parent2 = w;

        /* Frame for output text. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
            "Results"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Resultados"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Rsultats"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
        gtk_widget_show(w);
        parent3 = w;

        /* Table for text and scroll bar widgets. */
        w = gtk_table_new(2, 2, FALSE);
        gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
        gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_widget_show(w);
        parent3 = w;

	fm->output_text = w = gtk_text_new(NULL, NULL);
	editable = (GtkEditable *)w;
	text = (GtkText *)w;
        gtk_text_set_editable(text, FALSE);
        gtk_text_set_word_wrap(text, TRUE);
        gtk_table_attach(
            GTK_TABLE(parent3), w,
            0, 1, 0, 1,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL,
            0, 0
        );
	fm->colormap = colormap = gtk_widget_get_colormap(w);
	if(colormap != NULL)
	    gdk_colormap_ref(colormap);
        gtk_widget_show(w);

        parent4 = gtk_vscrollbar_new(GTK_TEXT(w)->vadj);
        gtk_table_attach(
            GTK_TABLE(parent3), parent4,
            1, 2, 0, 1,
            GTK_FILL,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL,
            0, 0
        );
        gtk_widget_realize(w);
	gdk_window_set_cursor(w->window, fm->text_cur);
        gtk_widget_show(parent4);


        /* Allocate colors. */
        c = &fm->error_color;
        c->red = 1.0 * (guint16)-1;
        c->green = 0.0 * (guint16)-1;
        c->blue = 0.0 * (guint16)-1;
        gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);










        /* Vbox for buttons. */
        w = gtk_vbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

        /* Vbox for clear button. */
        w = gtk_vbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

        /* Clear button. */
        fm->clear_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_clear_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Clear",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Claro",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Clair",
#endif
	    NULL
        );
        gtk_widget_set_usize(
            w,
            FORMAT_MANAGER_BTN_WIDTH,
            FORMAT_MANAGER_BTN_HEIGHT
        );
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Clear results"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Los resultados claros"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Rsultats clairs"
#endif
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(FormatManagerClearCB),
            (gpointer)fm
        );
        gtk_widget_show(w);

        /* Vbox for close button. */
        w = gtk_vbox_new(FALSE, border_minor);
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent4 = w;

        /* Close button. */
        fm->close_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_close_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Close",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Cierre",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Proche",
#endif
	    NULL
        );
        gtk_widget_set_usize(
	    w,
	    FORMAT_MANAGER_BTN_WIDTH,
	    FORMAT_MANAGER_BTN_HEIGHT
        );
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_end(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        GUISetWidgetTip(
            w,
#ifdef PROG_LANGUAGE_ENGLISH
            "Close this window"
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Cierre esta ventana"
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Fermer cette fentre"
#endif
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(FormatManagerCloseBtnCB),
            (gpointer)fm
        );
        gtk_accel_group_add(
            accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_widget_show(w);



	/* Status bar frame. */
	w = gtk_frame_new(NULL);
        gtk_widget_set_usize(w, -1, EDV_STATUS_BAR_HEIGHT);
        gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
        gtk_container_border_width(GTK_CONTAINER(w), 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
        parent2 = w;

        /* Table in main frame. */
        w = gtk_table_new(1, 1, FALSE);
        gtk_container_add(GTK_CONTAINER(parent2), w);
        gtk_widget_show(w);
        parent2 = w;

        /* Progress bar. */
        adj = (GtkAdjustment *)gtk_adjustment_new(0, 1, 100, 0, 0, 0);
        fm->progress_bar = w = gtk_progress_bar_new_with_adjustment(adj);
        gtk_widget_set_usize(w, -1, 20);
        gtk_progress_bar_set_orientation(
            GTK_PROGRESS_BAR(w), GTK_PROGRESS_LEFT_TO_RIGHT
        );
        gtk_progress_bar_set_bar_style(
            GTK_PROGRESS_BAR(w), GTK_PROGRESS_CONTINUOUS
        );
        gtk_progress_set_activity_mode(
            GTK_PROGRESS(w), FALSE
        );
	gtk_progress_set_show_text(GTK_PROGRESS(w), FALSE);
	gtk_progress_bar_update(GTK_PROGRESS_BAR(w), 0.0);
        gtk_table_attach(
            GTK_TABLE(parent2), w,
            0, 1, 0, 1,
            GTK_SHRINK | GTK_EXPAND | GTK_FILL,
            GTK_SHRINK,
            border_minor, border_minor
        );
        gtk_widget_show(w);



	/* Load disk animation icons. */
	fm->last_disk_icon_num = 0;

        style = gtk_widget_get_style(fm->disk_anim_da);
	fm->total_disk_icons = 3;
	fm->disk_icon_pixmap = (GdkPixmap **)g_malloc0(
	    fm->total_disk_icons * sizeof(GdkPixmap *)
	);
        fm->disk_icon_mask = (GdkBitmap **)g_malloc0(
            fm->total_disk_icons * sizeof(GdkBitmap *)
        );

	i = 0;
	fm->disk_icon_pixmap[i] = gdk_pixmap_create_from_xpm_d(
            window, &fm->disk_icon_mask[i],
            &style->bg[GTK_STATE_NORMAL],
            (gchar **)disk_icon_00_48x48_xpm
        );
        i = 1;
        fm->disk_icon_pixmap[i] = gdk_pixmap_create_from_xpm_d(
            window, &fm->disk_icon_mask[i],
            &style->bg[GTK_STATE_NORMAL],
            (gchar **)disk_icon_01_48x48_xpm
        );
        i = 2;
        fm->disk_icon_pixmap[i] = gdk_pixmap_create_from_xpm_d(
            window, &fm->disk_icon_mask[i],
            &style->bg[GTK_STATE_NORMAL],
            (gchar **)disk_icon_02_48x48_xpm
        );





	FormatManagerUpdateMenus(fm);

	return(fm);
}

/*
 *	Updates widgets to reflect local data values.
 */
void FormatManagerUpdateMenus(format_manager_struct *fm)
{
        gboolean sensitive;
        GtkWidget *w;
        GtkCList *clist;


	if(fm == NULL)
	    return;

        if(!fm->initialized)
            return;

#define DO_SET_SENSITIVE        \
{ \
 if(w != NULL) \
  gtk_widget_set_sensitive(w, sensitive); \
}

	sensitive = (fm->format_toid != (guint)-1) ? FALSE : TRUE;
	w = fm->start_btn;
	DO_SET_SENSITIVE
	w = fm->devices_start_mi;
	DO_SET_SENSITIVE

        sensitive = (fm->format_toid != (guint)-1) ? TRUE : FALSE;
        w = fm->stop_btn;
        DO_SET_SENSITIVE
        w = fm->devices_stop_mi;
        DO_SET_SENSITIVE


	w = fm->devices_clist;
	clist = (GtkCList *)w;







#undef DO_SET_SENSITIVE
}

/*
 *      Marks the format manager window as busy or ready.
 */
void FormatManagerSetBusy(format_manager_struct *fm, gboolean is_busy)
{
        GdkCursor *cur;
        GtkWidget *w;


        if(fm == NULL)
            return;

        if(!fm->initialized)
            return;

        w = fm->toplevel;
        if(w != NULL)
        {
            if(is_busy)
            {
                /* Increase busy count. */
                fm->busy_count++;

                /* If already busy then don't change anything. */
                if(fm->busy_count > 1)
                    return;

                cur = fm->busy_cur;
            }
            else
            {
                /* Reduce busy count. */
                fm->busy_count--;
                if(fm->busy_count < 0)
                    fm->busy_count = 0;

                /* If still busy do not change anything. */
                if(fm->busy_count > 0)
                    return;

                cur = NULL;	/* Use default cursor. */
            }

            /* Update toplevel window's cursor. */
            if(w->window != NULL)
            {
                gdk_window_set_cursor(w->window, cur);
                gdk_flush();
            }
        }
}

/*
 *	Updates the progress bar on the format manager window.
 */
void FormatManagerSetProgress(
        format_manager_struct *fm, gfloat value,
        gboolean allow_gtk_iteration
)
{
        GtkWidget *w;
        GtkProgress *progress;


        if(fm == NULL)
            return;

        if(!fm->initialized)
            return;

        w = fm->progress_bar;
        if(w == NULL)
            return;
        progress = GTK_PROGRESS(w);


        /* Negative? Implies just do activity. */
        if(value < 0.0)
        {
            GtkAdjustment *adj = progress->adjustment;


            /* Get new value based on unknown position. */
            value = gtk_progress_get_value(progress) + 1.0;
            if(value > adj->upper)
                value = adj->lower;

            /* Update progress widget for `activity mode' (unknown limit)
             * and set new value.
             */
            gtk_progress_set_activity_mode(progress, TRUE);
            gtk_progress_set_show_text(progress, FALSE);
            gtk_progress_set_value(progress, value);

            /* Reset last progress position to -1.0, indicating no last
             * position since in unknown time limit we don't keep track of
             * the last position.
             */
            fm->progress_pos_last = -1.0;

            /* Iterate through GTK main function until all events are
             * handled so that we ensure this progress bar setting is
             * updated.
             */
            while((gtk_events_pending() > 0) && allow_gtk_iteration)
                gtk_main_iteration();

            return;
        }

        /* Clip value to 0.0, 1.0. */
        if(value > 1.0)
            value = 1.0;

        /* Reset last progress position if it is greater than the given
         * value, implying the progress wraped back to the beginning.
         */
        if(fm->progress_pos_last > value)
            fm->progress_pos_last = -1.0;

        /* Check if percent did not sufficiently change (use 0.01
         * increments).
         */
        if((fm->progress_pos_last > 0.0) &&
           ((value - fm->progress_pos_last) < 0.01)
        )
            return;

        /* Update progress. */
        gtk_progress_set_activity_mode(progress, FALSE);
        gtk_progress_set_format_string(
            progress, "%p%%"
        );
        gtk_progress_set_show_text(
            progress, (value > 0.0) ? TRUE : FALSE
        );
        gtk_progress_bar_update(GTK_PROGRESS_BAR(w), value);

        /* Record last position. */
        fm->progress_pos_last = value;

        /* Iterate through GTK main function until all events are
         * handled so that we ensure this progress bar setting is
         * updated.
         */
        while((gtk_events_pending() > 0) && allow_gtk_iteration)
            gtk_main_iteration();
}

/*
 *	Maps the format manager window.
 */
void FormatManagerMap(format_manager_struct *fm)
{
        GtkWidget *w;

        if(fm == NULL)
            return;

        if(!fm->initialized)
            return;

	/* Grab focus on the start button. */
	w = fm->start_btn;
	if(w != NULL)
	{
	    gtk_widget_grab_focus(w);
	    gtk_widget_grab_default(w);
	}

	w = fm->toplevel;
	gtk_widget_show_raise(w);
	fm->map_state = TRUE;
}

/*
 *      Unmaps the format manager window.
 */
void FormatManagerUnmap(format_manager_struct *fm)
{
        GtkWidget *w;

        if(fm == NULL)
            return;

        if(!fm->initialized)
            return;

        w = fm->toplevel;
        if(w == NULL)
            return;

        if(fm->map_state)
        {
            fm->map_state = FALSE;
            gtk_widget_hide(w);
        }
}

/*
 *      Destroys format manager widgets and deallocates resources
 *	including the structure itself.
 */
void FormatManagerDelete(format_manager_struct *fm)
{
	gint i;
	GdkColor *c;
	GdkColormap *colormap;
	GdkFont **font;
	GdkCursor **cur;
	GtkWidget **w;


        if(fm == NULL)
            return;

        if(fm->initialized)
        {
#define DO_DESTROY_CURSOR	\
{ \
 if((*cur) != NULL) \
 { \
  GdkCursor *tmp_cur = *cur; \
  (*cur) = NULL; \
  gdk_cursor_destroy(tmp_cur); \
 } \
}
#define DO_UNREF_FONT		\
{ \
 if((*font) != NULL) \
 { \
  GdkFont *tmp_font = *font; \
  (*font) = NULL; \
  gdk_font_unref(tmp_font); \
 } \
}
#define DO_FREE_COLOR		\
{ \
 gdk_colormap_free_colors(colormap, c, 1); \
 memset(c, 0x00, sizeof(GdkColor)); \
}
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}
	    colormap = fm->colormap;

	    if(fm->queued_device_to_format != NULL)
	    {
		g_list_free(fm->queued_device_to_format);
		fm->queued_device_to_format = NULL;
	    }

	    if(fm->format_toid != (guint)-1)
	    {
		gtk_timeout_remove(fm->format_toid);
		fm->format_toid = (guint)-1;
	    }
	    if(fm->format_pid != 0)
	    {
		kill(fm->format_pid, SIGTERM);
		fm->format_pid = 0;
	    }

	    FClose(fm->stdout_fp);
	    fm->stdout_fp = NULL;
	    FClose(fm->stderr_fp);
	    fm->stderr_fp = NULL;

	    if(fm->stdout_path != NULL)
		unlink(fm->stdout_path);
	    g_free(fm->stdout_path);
	    fm->stdout_path = NULL;

            if(fm->stderr_path != NULL)
                unlink(fm->stderr_path);
	    g_free(fm->stderr_path);
	    fm->stderr_path = NULL;


	    for(i = 0; i < fm->total_disk_icons; i++)
	    {
		if(fm->disk_icon_pixmap[i] != NULL)
		    gdk_pixmap_unref(fm->disk_icon_pixmap[i]);
		if(fm->disk_icon_mask[i] != NULL)
                    gdk_bitmap_unref(fm->disk_icon_mask[i]);
	    }
	    g_free(fm->disk_icon_pixmap);
	    fm->disk_icon_pixmap = NULL;
	    g_free(fm->disk_icon_mask);
            fm->disk_icon_mask = NULL;
	    fm->total_disk_icons = 0;

	    fm->devices_start_mi = NULL;
	    fm->devices_stop_mi = NULL;
	    w = &fm->devices_menu;
	    DO_DESTROY_WIDGET

	    w = &fm->devices_clist;
            DO_DESTROY_WIDGET
            w = &fm->quick_format_check;
            DO_DESTROY_WIDGET
            w = &fm->capacity_combo;
            DO_DESTROY_WIDGET
	    w = &fm->volume_label_entry;
	    DO_DESTROY_WIDGET
            w = &fm->file_system_type_combo;
	    DO_DESTROY_WIDGET
            w = &fm->verbose_check;
	    DO_DESTROY_WIDGET
            w = &fm->output_text;
            DO_DESTROY_WIDGET
	    w = &fm->progress_bar;
	    DO_DESTROY_WIDGET

	    if(fm->disk_anim_pm != NULL)
		gdk_pixmap_unref(fm->disk_anim_pm);
	    fm->disk_anim_pm = NULL;
            w = &fm->disk_anim_da;
            DO_DESTROY_WIDGET


            w = &fm->start_btn;
            DO_DESTROY_WIDGET
            w = &fm->stop_btn;
            DO_DESTROY_WIDGET
            w = &fm->clear_btn;
            DO_DESTROY_WIDGET
            w = &fm->close_btn;
            DO_DESTROY_WIDGET

            w = &fm->toplevel;
            DO_DESTROY_WIDGET

            if(fm->accelgrp != NULL)
            {
                gtk_accel_group_unref(fm->accelgrp);
                fm->accelgrp = NULL;
            }

	    if(fm->gc != NULL)
	    {
		gdk_gc_unref(fm->gc);
		fm->gc = NULL;
	    }

	    c = &fm->error_color;
	    DO_FREE_COLOR

	    if(fm->colormap != NULL)
	    {
		gdk_colormap_unref(fm->colormap);
		fm->colormap = NULL;
	    }

	    cur = &fm->busy_cur;
	    DO_DESTROY_CURSOR
            cur = &fm->text_cur;
            DO_DESTROY_CURSOR

	    font = &fm->text_font;
	    DO_UNREF_FONT

	    for(i = 0; i < fm->total_devices; i++)
		EDVDeviceDelete(fm->device[i]);
	    g_free(fm->device);
	    fm->device = NULL;
	    fm->total_devices = 0;

#undef DO_DESTROY_WIDGET
#undef DO_DESTROY_CURSOR
#undef DO_UNREF_FONT
#undef DO_FREE_COLOR
        }

        /* Deallocate the structure itself. */
        g_free(fm);
}

