/*
	Endeavour Configuration List Management

	Management for sequential lists of configuration items, each
	item in this list has a type, parameter, and value. Each of which
	can be dynamically and abstractly treated for easier coding of
	parsing routines.

 */

#ifndef EDVCFG_H
#define EDVCFG_H

#include <glib.h>
#include "edvtypes.h"


/*
 *	Configuration item structure, contains three (3) values as
 *	follows; a parameter, a data type, and a value. Each member
 *	must be a size of 4 bytes, so the size of the structure must be
 *	exactly 3 * 4 = 12 bytes.
 */
typedef struct {
#define EDV_CFG_ITEM_TYPE_NONE		0	/* Value is NULL. */
#define EDV_CFG_ITEM_TYPE_INT8		1	/* gint8 */
#define EDV_CFG_ITEM_TYPE_UINT8		2	/* guint8 */
#define EDV_CFG_ITEM_TYPE_INT16		3	/* gint16 */
#define EDV_CFG_ITEM_TYPE_UINT16	4	/* guint16 */
#define EDV_CFG_ITEM_TYPE_INT32		5	/* gint32 */
#define EDV_CFG_ITEM_TYPE_UINT32	6	/* guint32 */
#define EDV_CFG_ITEM_TYPE_INT64		7	/* gint64 */
#define EDV_CFG_ITEM_TYPE_UINT64	8	/* guint64 */
#define EDV_CFG_ITEM_TYPE_FLOAT		9	/* gfloat */
#define EDV_CFG_ITEM_TYPE_DOUBLE	10	/* gdouble */
#define EDV_CFG_ITEM_TYPE_STRING	11	/* gchar * */

#define EDV_CFG_ITEM_TYPE_COLOR		20	/* edv_color_struct */
#define EDV_CFG_ITEM_TYPE_INTLIST	21	/* edv_intlist_struct */

	gint type;	/* One of EDV_CFG_ITEM_TYPE_*, specifies value type. */
	gchar *parameter;
	gpointer value;

} edv_cfg_item_struct;


/* Configuration item IO. */
extern edv_cfg_item_struct *EDVCFGItemNew(
	gint type, const gchar *parameter
);
extern gpointer EDVCFGItemGetValue(
	const edv_cfg_item_struct *ci, gint *type_rtn
);
extern void EDVCFGItemSetValue(edv_cfg_item_struct *ci, const gpointer value);
extern void EDVCFGItemResetValue(edv_cfg_item_struct *ci);
extern void EDVCFGItemReset(edv_cfg_item_struct *ci);
extern void EDVCFGItemDelete(edv_cfg_item_struct *ci);


/* List utilities. */
extern edv_cfg_item_struct *EDVCFGItemListCopyList(
        const edv_cfg_item_struct *list
);
extern void EDVCFGItemListDeleteList(edv_cfg_item_struct *list);
extern gint EDVCFGItemListMatchParameter(
	const edv_cfg_item_struct *list, const gchar *parameter
);

/* Value fetching. */
extern gpointer EDVCFGItemListMatchGetValue(
	const edv_cfg_item_struct *list, const gchar *parameter,
	gint *type_rtn
);
extern gint EDVCFGItemListGetValueI(
	const edv_cfg_item_struct *list, const gchar *parameter
);
extern glong EDVCFGItemListGetValueL(
	const edv_cfg_item_struct *list, const gchar *parameter
);
extern gulong EDVCFGItemListGetValueUL(
        const edv_cfg_item_struct *list, const gchar *parameter
);
extern gfloat EDVCFGItemListGetValueF(
	const edv_cfg_item_struct *list, const gchar *parameter
);
extern gdouble EDVCFGItemListGetValueD(
        const edv_cfg_item_struct *list, const gchar *parameter
);
extern gchar *EDVCFGItemListGetValueS(
        const edv_cfg_item_struct *list, const gchar *parameter
);
extern edv_color_struct *EDVCFGItemListGetValueColor(
        const edv_cfg_item_struct *list, const gchar *parameter
);
extern edv_intlist_struct *EDVCFGItemListGetValueIntList(
        const edv_cfg_item_struct *list, const gchar *parameter
);

/* Value setting. */
extern edv_cfg_item_struct *EDVCFGItemListSetValue(
	edv_cfg_item_struct *list, const gchar *parameter,
	const gpointer value, gbool create_as_needed
);
extern edv_cfg_item_struct *EDVCFGItemListSetValueI(
        edv_cfg_item_struct *list, const gchar *parameter,
        gint value, gbool create_as_needed
);
extern edv_cfg_item_struct *EDVCFGItemListSetValueL(
        edv_cfg_item_struct *list, const gchar *parameter,
        glong value, gbool create_as_needed
);
extern edv_cfg_item_struct *EDVCFGItemListSetValueUL(
        edv_cfg_item_struct *list, const gchar *parameter,
        gulong value, gbool create_as_needed
);
extern edv_cfg_item_struct *EDVCFGItemListSetValueF(
        edv_cfg_item_struct *list, const gchar *parameter,
        gfloat value, gbool create_as_needed
);
extern edv_cfg_item_struct *EDVCFGItemListSetValueS(
        edv_cfg_item_struct *list, const gchar *parameter,
        const gchar *value, gbool create_as_needed
);
extern edv_cfg_item_struct *EDVCFGItemListSetValueColor(
        edv_cfg_item_struct *list, const gchar *parameter,
        const edv_color_struct *value, gbool create_as_needed
);
extern edv_cfg_item_struct *EDVCFGItemListSetValueIntList(
        edv_cfg_item_struct *list, const gchar *parameter,
        const edv_intlist_struct *value, gbool create_as_needed
);


#endif	/* EDVCFG_H */
