#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "fprompt.h"
#include "editclist.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "edvmimetypes.h"
#include "mimetypeswin.h"
#include "endeavour.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvcfglist.h"
#include "config.h"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_mimetypes_48x48.xpm"


static void EDVMimeTypesEditIconUpdateDisplay(
        edv_mimetype_editwin_icon_struct *ei, gint state
);
static void EDVMimeTypesEditIconChangeCB(
        GtkWidget *widget, gpointer data
);
static void EDVMimeTypesEditIconPrevCB(
        GtkWidget *widget, gpointer data
);
static void EDVMimeTypesEditIconNextCB(
        GtkWidget *widget, gpointer data
);

static gint EDVMimeTypesEditWinDeleteEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVMimeTypesEditWinClassChangedCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypesEditWinHandlerChangedCB(
        GtkWidget *widget, gpointer data
);
static void EDVMimeTypesEditWinOKCB(
        GtkWidget *widget, gpointer data
);
static void EDVMimeTypesEditWinApplyCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypesEditWinCancelCB(
        GtkWidget *widget, gpointer data
);

static gint EDVMimeTypesEditWinGetClass(
        edv_mimetype_editwin_struct *ew
);
static gint EDVMimeTypesEditWinGetHandler(
        edv_mimetype_editwin_struct *ew
);

gint EDVMimeTypesEditWinVerifyValues(
        edv_mimetype_struct **list, gint total,
        gint mt_num,
        gbool verbose, GtkWidget *toplevel
);

void EDVMimeTypesEditWinFetchValues(
	edv_mimetype_editwin_struct *ew, gint mt_num
);

void EDVMimeTypesEditWinMimeTypeRemovedCB(
        edv_mimetype_editwin_struct *ew, gint mt_num
);

static edv_mimetype_editwin_icon_struct *EDVMimeTypesEditIconNew(
	gpointer editwin, GtkWidget *parent, const gchar *title,
	gint btn_width, gint btn_height
);
static void EDVMimeTypesEditIconDelete(edv_mimetype_editwin_icon_struct *ei);

edv_mimetype_editwin_struct *EDVMimeTypesEditWinNew(
        gpointer core_ptr, gpointer listwin
);
void EDVMimeTypesEditWinMap(
        edv_mimetype_editwin_struct *ew
);
void EDVMimeTypesEditWinUnmap(
        edv_mimetype_editwin_struct *ew
);
void EDVMimeTypesEditWinDelete(
        edv_mimetype_editwin_struct *ew
);


#define EDITWIN_WIDTH	500
#define EDITWIN_HEIGHT	-1

#define EDITWIN_BTN_WIDTH	(100 + (2 * 3))
#define EDITWIN_BTN_HEIGHT	(30 + (2 * 3))

#define EDITWIN_TITLE	"MIME Type"



/*
 *	Updates the icon displayed on the button of the given edit window
 *	icon structure with the corresponding pixmap and mask pair of
 *	the given state.
 */
static void EDVMimeTypesEditIconUpdateDisplay(
	edv_mimetype_editwin_icon_struct *ei, gint state
)
{
        const gchar *label_str = "";
        GdkPixmap *pixmap;
        GdkBitmap *mask;
	GtkWidget *w, *parent;


	if(ei == NULL)
	    return;

        if((state < 0) || (state >= EDV_MIMETYPE_TOTAL_ICON_STATES))
	    return;

	/* Get references to pixmap and mask pair specified by the given
	 * state. 
	 */
	pixmap = ei->pixmap[state];
	mask = ei->mask[state];

	/* Get button widget as the parent to create GtkPixmap in. */
	parent = ei->button;
	if(parent == NULL)
	    return;

	/* Referenced pixmap valid? */
	if(pixmap != NULL)
        {
	    /* Referenced pixmap is valid, now check if the GtkPixmap that
	     * is to be used to display the pixmap has been created yet.
	     * if not then create a new one.
	     */
	    w = ei->gtk_pixmap;
	    if(w == NULL)
	    {
		ei->gtk_pixmap = w = gtk_pixmap_new(pixmap, mask);
		gtk_container_add(GTK_CONTAINER(parent), w);
		gtk_widget_show(w);
	    }
	    else
	    {
		/* GtkPixmap exists, so just update it. */
		gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
		gtk_widget_show(w);
	    }
        }
	else
	{
	    /* No referenced pixmap, so unmap the the GtkPixmap. */
	    w = ei->gtk_pixmap;
	    if(w != NULL)
	    {
		gtk_widget_hide(w);
	    }
	}


	/* Update state label, get the currect label string to display
	 * for the state label based on the given state.
	 */
        switch(state)
        {
          case EDV_MIMETYPE_ICON_STATE_EXTENDED:
	    label_str =
#ifdef PROG_LANGUAGE_ENGLISH
		"Extended";
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"Extendido";
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Etendu";
#endif
	    break;
          case EDV_MIMETYPE_ICON_STATE_SELECTED:
	    label_str =
#ifdef PROG_LANGUAGE_ENGLISH
		"Selected";
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Escogi";
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Choisi";
#endif
            break;
          case EDV_MIMETYPE_ICON_STATE_STANDARD:
	    label_str =
#ifdef PROG_LANGUAGE_ENGLISH
		"Standard";
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Estndar";
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Norme";
#endif
            break;
	}
	w = ei->state_label;
	if(w != NULL)
	    gtk_label_set_text(GTK_LABEL(w), label_str);

}

/*
 *	Edit window icon change callback.
 */
static void EDVMimeTypesEditIconChangeCB(
        GtkWidget *widget, gpointer data
)
{
	static gbool reenterent = FALSE;
	gint state, status;
	const gchar *cstrptr, *old_path;
	gchar *old_parent_path;
	GdkWindow *window;
	GtkAdjustment *adj;
	GtkStyle *style;
	GtkWidget *w, *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	edv_core_struct *core_ptr;
        edv_mimetype_editwin_struct *ew;
        edv_mimetype_editwin_icon_struct *ei =
            (edv_mimetype_editwin_icon_struct *)data;
        if(ei == NULL)
            return;

	if(FileBrowserIsQuery())
	    return;

	adj = ei->adj;
	if(adj == NULL)
	    return;

	/* Get icon state (standard, opened, or extended) and make sure it
	 * is in bounds.
	 */
	state = (gint)adj->value;
	if((state < 0) || (state >= EDV_MIMETYPE_TOTAL_ICON_STATES))
	    return;

        ew = (edv_mimetype_editwin_struct *)ei->editwin;
        if(ew == NULL)
            return;

        /* Get values from MIME Types edit window's toplevel widget. */
        toplevel = w = ew->toplevel;
        if(w == NULL)
            return;
        style = gtk_widget_get_style(w);
        window = w->window;
        if((style == NULL) || (window == NULL))
            return;

	core_ptr = (edv_core_struct *)ew->core_ptr;
	if(core_ptr == NULL)
	    return;

	if(reenterent)
	    return;
	else
	    reenterent = TRUE;


	/* Get current path to the icon file for this state (can be
	 * NULL).
	 */
	old_path = ei->path[state];
	if(old_path != NULL)
	{
	    cstrptr = GetParentDir(old_path);
	    old_parent_path = (cstrptr != NULL) ?
		g_strdup(cstrptr) : NULL;
	}
	else
	{
	    old_parent_path = NULL;
	}

	/* Create file types list. */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    ".xpm", "XPM Image"
	);
        FileBrowserTypeListNew(
            &ftype, &total_ftypes,
            "*.*", "All files"
        );

	/* Current old_path must be considered invalid at this point. */
	old_path = NULL;


	/* Query user for new icon file. */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Select Icon",
	    "Select", "Cancel",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Icono Selecto",
            "Selecto", "Cancela",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Choisir L'Icne",
            "Choisir", "Annuler",
#endif
	    old_parent_path,
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
        FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{
	    const gchar *new_path = (total_path_rtns > 0) ?
		path_rtn[0] : NULL;


	    if((new_path != NULL) ? (*new_path != '\0') : FALSE)
	    {
		GdkPixmap *old_pixmap = ei->pixmap[state];
		GdkBitmap *old_mask = ei->mask[state];


		/* Load new pixmap and mask. */
		ei->pixmap[state] = gdk_pixmap_create_from_xpm(
		    window, &ei->mask[state],
		    &style->bg[GTK_STATE_NORMAL],
		    new_path
		);
		if(ei->pixmap[state] != NULL)
		{
		    /* New pixmap and mask loaded successfully. */

		    /* Unref old pixmap and mask. */
		    if(old_pixmap != NULL)
		    {
			gdk_pixmap_unref(old_pixmap);
			old_pixmap = NULL;
		    }
		    if(old_mask != NULL)
		    {
			gdk_bitmap_unref(old_mask);
			old_mask = NULL;
		    }

		    /* Set new icon file path. */
		    g_free(ei->path[state]);
		    ei->path[state] = g_strdup(new_path);

		    /* Update displayed icon on the given edit icon
		     * structure.
		     */
		    EDVMimeTypesEditIconUpdateDisplay(ei, state);
		}
		else
		{
		    /* Load failed, unref current pixmap and mask pair. */
 		    gchar *buf;


                    /* Unref current pixmap and mask. */
                    if(old_pixmap != NULL)
                    {
                        gdk_pixmap_unref(old_pixmap);
                        old_pixmap = NULL;
                    }
                    if(old_mask != NULL)
                    {
                        gdk_bitmap_unref(old_mask);
                        old_mask = NULL;
                    }

		    /* Mark that current pixmap and mask pair have been
		     * unloaded.
		     */
		    ei->pixmap[state] = NULL;
                    ei->mask[state] = NULL;

		    /* Unset path. */
		    g_free(ei->path[state]);
		    ei->path[state] = NULL;

                    /* Update displayed icon on the given edit icon
                     * structure.
                     */
                    EDVMimeTypesEditIconUpdateDisplay(ei, state);


		    /* Notify user about failed load. */
		    buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to load icon file:\n\
\n\
    %s\n\
\n\
Please verify that the specified path is correct and\n\
the file is a valid xpm image file.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de cargar el archivo de icono:\n\
\n\
    %s\n\
\n\
Verifique por favor que el sendero especificado es\n\
correcto el archivo es un archivo vlido de la\n\
imagen de xpm.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour charger le fichier d'icne:\n\
\n\
    %s\n\
\n\
S'il vous plat vrifier que le sentier spcifi\n\
est exact et le fichier est un fichier d'image de\n\
xpm valide.",
#endif
			new_path
		    );
		    CDialogSetTransientFor(toplevel);
		    CDialogGetResponse(
			"Load Failed", buf, NULL,
			CDIALOG_ICON_ERROR,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		    );
		    CDialogSetTransientFor(NULL);
		    g_free(buf);
		}
	    }
	}


	/* Deallocate file types list. */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	/* Deallocate coppied parent path. */
	g_free(old_parent_path);
	old_parent_path = NULL;


	reenterent = FALSE;
}

/*
 *      Prev icon edit window icon callback.
 */
static void EDVMimeTypesEditIconPrevCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
	GtkAdjustment *adj;
        edv_mimetype_editwin_struct *ew;
        edv_mimetype_editwin_icon_struct *ei =
            (edv_mimetype_editwin_icon_struct *)data;
        if(ei == NULL)
            return;

        ew = (edv_mimetype_editwin_struct *)ei->editwin;
        if(ew == NULL)
            return;

	adj = ei->adj;
	if(adj == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Increment adjustment position. */
        if(adj->value > adj->lower)
            adj->value -= adj->step_increment;
        if(adj->value < adj->lower)
            adj->value = adj->lower;

	EDVMimeTypesEditIconUpdateDisplay(ei, (gint)adj->value);

        reenterent = FALSE;
}

/*
 *	Next icon edit window icon callback.
 */
static void EDVMimeTypesEditIconNextCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
        GtkAdjustment *adj;
	edv_mimetype_editwin_struct *ew;
	edv_mimetype_editwin_icon_struct *ei =
	    (edv_mimetype_editwin_icon_struct *)data;
	if(ei == NULL)
	    return;

	ew = (edv_mimetype_editwin_struct *)ei->editwin;
	if(ew == NULL)
	    return;

        adj = ei->adj;
        if(adj == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Increment adjustment position. */
        if(adj->value < adj->upper)
            adj->value += adj->step_increment;
        if(adj->value > adj->upper)
            adj->value = adj->upper;

        EDVMimeTypesEditIconUpdateDisplay(ei, (gint)adj->value);

        reenterent = FALSE;
}


/*
 *	MIME Types edit window "delete_event" signal callback.
 */
static gint EDVMimeTypesEditWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_mimetype_editwin_struct *ew = (edv_mimetype_editwin_struct *)data;
	if(ew == NULL)
	    return(TRUE);

	EDVMimeTypesEditWinCancelCB(NULL, ew);

	return(TRUE);
}

/*
 *	MIME Types edit window class_combo "changed" callback.
 *
 *	This will update the value_label and value_entry widgets.
 */
static void EDVMimeTypesEditWinClassChangedCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
	gint mt_class;
	GtkWidget *w;
	gbool sensitive = TRUE;
	const gchar *value_label_str = "";
        edv_mimetype_editwin_struct *ew = (edv_mimetype_editwin_struct *)data;
        if(ew == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	/* Get new class value from class_combo. */
	mt_class = EDVMimeTypesEditWinGetClass(ew);

	/* Update widgets depending on class. */
	switch(mt_class)
	{
	  case EDV_MIMETYPE_CLASS_SYSTEM:
	    sensitive = FALSE;
#ifdef PROG_LANGUAGE_ENGLISH
	    value_label_str = "Value:";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            value_label_str = "El Valor:";
#endif

#ifdef PROG_LANGUAGE_FRENCH
            value_label_str = "Valeur:";
#endif
	    break;

          case EDV_MIMETYPE_CLASS_FORMAT:
            sensitive = TRUE;
#ifdef PROG_LANGUAGE_ENGLISH
            value_label_str = "Extensions:";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            value_label_str = "Las Extensiones:";
#endif

#ifdef PROG_LANGUAGE_FRENCH
            value_label_str = "Extensions:";
#endif
            break;

          case EDV_MIMETYPE_CLASS_PROGRAM:
            sensitive = TRUE;
#ifdef PROG_LANGUAGE_ENGLISH
            value_label_str = "Location:";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            value_label_str = "La Ubicaci:";
#endif

#ifdef PROG_LANGUAGE_FRENCH
            value_label_str = "Emplacement:";
#endif
            break;

          case EDV_MIMETYPE_CLASS_UNIQUE:
            sensitive = TRUE;
#ifdef PROG_LANGUAGE_ENGLISH
            value_label_str = "Location:";
#endif
#ifdef PROG_LANGUAGE_SPANISH
            value_label_str = "La Ubicaci:";
#endif

#ifdef PROG_LANGUAGE_FRENCH
            value_label_str = "Emplacement:";
#endif
            break;
	}

	w = ew->value_label;
	if(w != NULL)
	    gtk_label_set_text(GTK_LABEL(w), value_label_str);

	w = ew->value_entry;
	if(w != NULL)
	    gtk_widget_set_sensitive(w, sensitive);

	/* Need to resize since label value changed which means size 
	 * allocations may change as well.
	 */
	gtk_widget_queue_resize(ew->toplevel);

	reenterent = FALSE;
}

/*
 *	MIME Types edit window handler_combo "changed" callback.
 *
 *	This will update the sensitivity or mapping of some widgets
 *	such as the command editable clist.
 */
static void EDVMimeTypesEditWinHandlerChangedCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
        gint handler;
        GtkWidget *w;
        gbool sensitive = TRUE;
	editclist_struct *editclist;
        edv_mimetype_editwin_struct *ew = (edv_mimetype_editwin_struct *)data;
        if(ew == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

        /* Get new handler value from handler_combo. */
	handler = EDVMimeTypesEditWinGetHandler(ew);

        /* Update widgets depending on class. */
        switch(handler)
        {
          case EDV_MIMETYPE_HANDLER_COMMAND:
            sensitive = TRUE;
            break;

	  default:
	    sensitive = FALSE;
	    break;
        }

	editclist = ew->editclist;
	if(editclist != NULL)
	{
	    w = editclist->toplevel;
	    if(w != NULL)
		gtk_widget_set_sensitive(w, sensitive);
	}

        reenterent = FALSE;
}

/*
 *      MIME Types edit window OK button callback.
 */
static void EDVMimeTypesEditWinOKCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_mimetype_editwin_struct *ew = (edv_mimetype_editwin_struct *)data;
        if(ew == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	EDVMimeTypesEditWinApplyCB(NULL, ew);
        EDVMimeTypesEditWinUnmap(ew);

        reenterent = FALSE;
}

/*
 *      MIME Types edit window Apply button callback.
 */
static void EDVMimeTypesEditWinApplyCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
	GtkEntry *entry;
	GtkCList *clist;
	edv_core_struct *core_ptr;
	gint mt_num;
	edv_mimetype_struct *mt_ptr;
	edv_mimetype_listwin_struct *lw;
	edv_mimetype_editwin_icon_struct *ei;
        edv_mimetype_editwin_struct *ew = (edv_mimetype_editwin_struct *)data;
        if(ew == NULL)
            return;

        core_ptr = (edv_core_struct *)ew->core_ptr;
        if(core_ptr == NULL)
            return;

	lw = (edv_mimetype_listwin_struct *)ew->listwin;
	if(lw == NULL)
	    return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	/* Get MIME Type index and pointer to structure on the core's
	 * list.
	 */
	mt_num = ew->mt_num;
	if((mt_num >= 0) && (mt_num < core_ptr->total_mimetypes))
	    mt_ptr = core_ptr->mimetype[mt_num];
	else
	    mt_ptr = NULL;

	/* MIME Type no longer exists on core structure's list of MIME 
	 * Types?
	 */
	if(mt_ptr == NULL)
	{
            CDialogSetTransientFor(ew->toplevel);
            CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
"Apply Failed",
"This MIME Type no longer exists, these values will be discarded.",
                NULL,
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Aplique Fallado",
"Este MIME Type no ms largo existe, estos valores se tirarn.",
                NULL,
#endif
#ifdef PROG_LANGUAGE_FRENCH
"S'appliquer Echou",
"Ce MIME Type non plus long existe, ces valeurs seront rejetes.",
                NULL,
#endif
                CDIALOG_ICON_ERROR,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);

	    reenterent = FALSE;
	    return;
	}

	/* MIME Type marked as read only? */
	if(mt_ptr->read_only)
	{
            gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"MIME Type `%s' cannot be edited, because it\n\
was either created internally or loaded from a global\n\
configuration.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"MIME Type `%s' no puede ser redactado,\n\
porque era o creado internamente o cargado de una\n\
configuracin global.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"MIME Type `%s' ne peut pas tre diter,\n\
parce que c'tait ou cr intrieurement ou charg\n\
d'une configuration globale.",
#endif
                mt_ptr->type
            );

            CDialogSetTransientFor(ew->toplevel);
            CDialogGetResponse(
#ifdef PROG_LANGUAGE_ENGLISH
		"Apply Failed",
		buf,
                NULL,
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Aplique Fallado",
                buf,
                NULL,
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "S'appliquer Echou",
                buf,
                NULL,
#endif
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);

	    g_free(buf);

            reenterent = FALSE;
            return;
	}


	/* Begin applying values. */

	/* Class. */
	mt_ptr->mt_class = EDVMimeTypesEditWinGetClass(ew);

	/* Type. */
	entry = (GtkEntry *)ew->type_entry;
	if(entry != NULL)
	{
	    const gchar *cstrptr = gtk_entry_get_text(entry);
	    if(cstrptr != NULL)
	    {
		g_free(mt_ptr->type);
		mt_ptr->type = g_strdup(cstrptr);
	    }
	}

        /* Value. */
        entry = (GtkEntry *)ew->value_entry;
        if(entry != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(entry);
            if(cstrptr != NULL)
            {
                g_free(mt_ptr->value);
                mt_ptr->value = g_strdup(cstrptr);
            }
        }

        /* Description. */
        entry = (GtkEntry *)ew->description_entry;
        if(entry != NULL)
        {
            const gchar *cstrptr = gtk_entry_get_text(entry);
            if(cstrptr != NULL)
            {
                g_free(mt_ptr->description);
                mt_ptr->description = g_strdup(cstrptr);
            }
        }

        /* Handler. */
        mt_ptr->handler = EDVMimeTypesEditWinGetHandler(ew);

	/* Commands. */
	clist = (GtkCList *)EditCListGetCList(ew->editclist);
	if(clist != NULL)
	{
	    gint i;
	    gchar *strptr;


	    /* Delete old command and command names. */
	    for(i = 0; i < mt_ptr->total_commands; i++)
	    {
		g_free(mt_ptr->command[i]);
		g_free(mt_ptr->command_name[i]);
	    }

	    /* Adjust total commands on MIME Type structure and reallocate
	     * the command and command_name pointer arrays to the new size.
	     */
	    mt_ptr->total_commands = clist->rows;
	    if(mt_ptr->total_commands > 0)
	    {
		mt_ptr->command = (gchar **)g_realloc(
		    mt_ptr->command,
		    mt_ptr->total_commands * sizeof(gchar *)
		);
                mt_ptr->command_name = (gchar **)g_realloc(
                    mt_ptr->command_name,
                    mt_ptr->total_commands * sizeof(gchar *)
                );
		if((mt_ptr->command == NULL) ||
                   (mt_ptr->command_name == NULL)
		)
		{
		    g_free(mt_ptr->command);
		    mt_ptr->command = NULL;
		    g_free(mt_ptr->command_name);
		    mt_ptr->command_name = NULL;
		    mt_ptr->total_commands = 0;
		}
	    }
	    else
	    {
		mt_ptr->total_commands = 0;
		g_free(mt_ptr->command);
		mt_ptr->command = NULL;
		g_free(mt_ptr->command_name);
		mt_ptr->command_name = NULL;
	    }

	    /* Get command and command name values from clist. */
	    if(mt_ptr->total_commands > 0)
	    {
		for(i = 0; i < clist->rows; i++)
		{
		    /* Get text from each cell, we know that the cells
		     * are of type GTK_CELLTYPE_TEXT
		     */
		    strptr = NULL;
		    gtk_clist_get_text(
			clist, i, 0, &strptr
		    );
		    mt_ptr->command_name[i] = (strptr != NULL) ?
			g_strdup(strptr) : NULL;

                    strptr = NULL;
                    gtk_clist_get_text(
                        clist, i, 1, &strptr
                    );
                    mt_ptr->command[i] = (strptr != NULL) ?
                        g_strdup(strptr) : NULL;
		}
	    }
	}

	/* Begin applying icons file paths. */

	/* Small icon. */
        ei = ew->icon_small;
        if(ei != NULL)
        {
	    gint i;
	    const gchar *cstrptr;

	    for(i = 0; i < EDV_MIMETYPE_TOTAL_ICON_STATES; i++)
	    {
		cstrptr = ei->path[i];

		g_free(mt_ptr->small_icon_file[i]);
		mt_ptr->small_icon_file[i] = (cstrptr != NULL) ?
		    g_strdup(cstrptr) : NULL;
	    }
	}

        /* Medium icon. */
        ei = ew->icon_medium;
        if(ei != NULL)
        {
            gint i;
            const gchar *cstrptr;

            for(i = 0; i < EDV_MIMETYPE_TOTAL_ICON_STATES; i++)
            {
                cstrptr = ei->path[i];

                g_free(mt_ptr->medium_icon_file[i]);
                mt_ptr->medium_icon_file[i] = (cstrptr != NULL) ?
                    g_strdup(cstrptr) : NULL;
            }
        }

        /* Large icon. */
        ei = ew->icon_large;
        if(ei != NULL)
        {
            gint i;
            const gchar *cstrptr;

            for(i = 0; i < EDV_MIMETYPE_TOTAL_ICON_STATES; i++)
            {
                cstrptr = ei->path[i];

                g_free(mt_ptr->large_icon_file[i]);
                mt_ptr->large_icon_file[i] = (cstrptr != NULL) ?
                    g_strdup(cstrptr) : NULL;
            }
        }


        /* Do verification procedure, to check if all values are set
         * properly on the MIME Type and to print any warning messages.
         */
	EDVMimeTypesEditWinVerifyValues(
	    core_ptr->mimetype, core_ptr->total_mimetypes,
	    mt_num,
	    TRUE, ew->toplevel
	);


	/* Re-realize the MIME Type. */
	EDVMimeTypeRealize(mt_ptr, TRUE);

        /* Send MIME Type modified signal to all of endeavour's resources. */
        EDVMimeTypeModifiedEmit(core_ptr, mt_num, mt_ptr);


        reenterent = FALSE;
}

/*
 *      MIME Types edit window Cancel button callback.
 */
static void EDVMimeTypesEditWinCancelCB(
        GtkWidget *widget, gpointer data
)
{
        static gbool reenterent = FALSE;
        edv_mimetype_editwin_struct *ew = (edv_mimetype_editwin_struct *)data;
        if(ew == NULL)
            return;

        if(reenterent)
            return;
        else
            reenterent = TRUE;

	EDVMimeTypesEditWinUnmap(ew);

        reenterent = FALSE;
}

/*
 *	Returns one of EDV_MIMETYPE_CLASS_* depending on the value
 *	contained on class_combo that corresponds to an index who
 *	is a value of EDV_MIMETYPE_CLASS_*.
 */
static gint EDVMimeTypesEditWinGetClass(
        edv_mimetype_editwin_struct *ew
)
{
	gint mt_class = 0;
	GtkCombo *combo;


	if(ew == NULL)
	    return(mt_class);

	combo = (GtkCombo *)ew->class_combo;
	if(combo != NULL)
	{
            GList *glist = (GList *)GUIComboGetList(combo);
            const gchar *cstrptr = gtk_entry_get_text(
                GTK_ENTRY(combo->entry)
            );


            /* Iterate through combo's list of strings, incrementing
             * mt_class as each iteration corresponds to one index.
             */
            if(cstrptr != NULL)
            {
                while(glist != NULL)
                {
                    if(glist->data != NULL)
                    {
                        if(!strcasecmp((const gchar *)glist->data, cstrptr))
                            break;
                    }
                    mt_class++;
                    glist = glist->next;
                }
            }
	}

	return(mt_class);
}

/*
 *      Returns one of EDV_MIMETYPE_HANDLER_* depending on the value
 *      contained on handler_combo that corresponds to an index who
 *      is a value of EDV_MIMETYPE_HANDLER_*.
 */
static gint EDVMimeTypesEditWinGetHandler(
        edv_mimetype_editwin_struct *ew
)
{
        gint handler = 0;
        GtkCombo *combo;


        if(ew == NULL)
            return(handler);

        combo = (GtkCombo *)ew->handler_combo;
        if(combo != NULL)
        {
            GList *glist = (GList *)GUIComboGetList(combo);
            const gchar *cstrptr = gtk_entry_get_text(
                GTK_ENTRY(combo->entry)
            );


            /* Iterate through combo's list of strings, incrementing
             * handler as each iteration corresponds to one index.
             */
            if(cstrptr != NULL)
            {
                while(glist != NULL)
                {
                    if(glist->data != NULL)
                    {
                        if(!strcasecmp((const gchar *)glist->data, cstrptr))
                            break;
                    }
                    handler++;
                    glist = glist->next;
                }
            }
        }

        return(handler);
}


/*
 *	This function is used to verify if the applied values to a
 *	MIME Type given by mt_num in the list of MIME Types has all
 *	the required information and if it has possible conflicts
 *	or missing references.
 *
 *	Returns non-zero if there was a problem.
 */
gint EDVMimeTypesEditWinVerifyValues(
        edv_mimetype_struct **list, gint total,
        gint mt_num,
	gbool verbose, GtkWidget *toplevel
)
{
	gint i, status = 0;
	const gchar *title =
#ifdef PROG_LANGUAGE_ENGLISH
		"MIME Type Warning";
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "MIME Type Advertir";
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "MIME Type Avertissement";
#endif
	edv_mimetype_struct *mt_ptr;


	if(list == NULL)
	    return(status);

	/* Get pointer to MIME Type. */
	if((mt_num >= 0) && (mt_num < total))
	    mt_ptr = list[mt_num];
	else
	    mt_ptr = NULL;
	if(mt_ptr == NULL)
	{
	    status = -1;
	    if(verbose)
	    {
		gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"MIME Type with index #%i is not in the list",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"MIME Type con el ndice #%i no est en la lista",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"MIME Type avec l'index #%i n'est pas dans la liste",
#endif

		    mt_num
		);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    title, buf, NULL,
		    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
		g_free(buf);
	    }
	    return(status);
	}

	/* Handle by MIME Type class. */
	switch(mt_ptr->mt_class)
	{
	  case EDV_MIMETYPE_CLASS_SYSTEM:

	    break;

	  case EDV_MIMETYPE_CLASS_FORMAT:
#if 0
/* Obsolete now that we allow more than just '.' characters. */
            /* Check if value is defined as a set of extensions. */
	    if((mt_ptr->value != NULL) ?
                (strchr(mt_ptr->value, '.') == NULL) : TRUE
            )
            {
                status = -2;
                if(verbose)
                {
                    gchar *buf;
		    if(mt_ptr->value == NULL)
			buf = g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
"The file type extensions list is not set, this MIME\n\
Type can never be matched."
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Las extensiones del tipo del archivo listan no es\n\
puesto, este MIME Type nunca se puede emparejar."
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Les extensions de type de fichier numrent n'est\n\
pas rgl, ce MIME Type ne peut jamais tre gal."
#endif
			);
		    else
			buf = g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
"The file type extensions list must be set as a\n\
space-separated list of file type extensions such\n\
as \".txt .text .doc\""
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Las extensiones del tipo del archivo listan debe\n\
ser puesto como una lista separada de espacio de\n\
extensiones de tipo de archivo tal como\n\
\".txt .text .doc\""
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Les extensions de type de fichier numrent doivent\n\
tre rgles comme une liste d'espace-spar\n\
d'extensions de type de fichier tel que\n\
\".txt .text .doc\""
#endif
			);
                    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        title, buf, NULL,
                        CDIALOG_ICON_WARNING,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
                    g_free(buf);
                }
	    }
#endif
	    if(mt_ptr->handler == EDV_MIMETYPE_HANDLER_COMMAND)
	    {
	      /* Check all defined commands on this file format MIME Type
	       * and make sure they reffer to valid MIME Types (if the
	       * command reffers to another MIME Type).
	       */
	      for(i = 0; i < mt_ptr->total_commands; i++)
	      {
		const gchar *cmd = mt_ptr->command[i];
		if(cmd == NULL)
		    continue;

		/* Does the command not start with a path deliminator?
		 * Suggesting that it is a reference to another MIME
		 * Type?
		 */
		if(*cmd != DIR_DELIMINATOR)
		{
		    if(EDVMimeTypeMatchListByType(
			list, total, NULL, cmd, TRUE
		    ) == NULL
		    )
		    {
			/* Referenced MIME Type not found. */
			status = -2;

			if(verbose)
			{
			    gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Command `%s' reffers to a non-existent MIME Type\n\
called `%s'.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Reffers de orden `%s' a un MIME Type inexistente\n\
llam `%s'.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Ordonner reffers  `%s'  un MIME Type inexistant\n\
a appel `%s'.",
#endif
				mt_ptr->command_name[i],
				cmd
			    );
			    CDialogSetTransientFor(toplevel);
                            CDialogGetResponse(
                                title, buf, NULL,
                                CDIALOG_ICON_WARNING,
                                CDIALOG_BTNFLAG_OK,
                                CDIALOG_BTNFLAG_OK
                            );
                            CDialogSetTransientFor(NULL);
                            g_free(buf);
                        }

		    }
		}
	      }
	    }
	    break;

	  case EDV_MIMETYPE_CLASS_PROGRAM:
            /* Check if value is defined and reffers to a valid disk
             * object that is executable.
             */
            if((mt_ptr->value != NULL) ?
                access(mt_ptr->value, X_OK) : TRUE
            )
            {
                status = -2;
                if(verbose)
                {
                    gchar *buf;
                    if(mt_ptr->value == NULL)
                        buf = g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
"Location of application object not set"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"La ubicacin de objeto de aplicacin no conjunto"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"L'emplacement d'objet d'application pas srie"
#endif
                        );
                    else
                        buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Invalid application object:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El objeto invlido de la aplicacin:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"L'objet d'application d'invalide:\n\
\n\
    %s\n",
#endif
                            mt_ptr->value
                        );
                    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        title, buf,
#ifdef PROG_LANGUAGE_ENGLISH
"The application object is not defined, does not exist,\n\
or is not set executable. Also, make sure that you\n\
specify a full path to the application object.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El objeto de la aplicacin no se define, no existe, ni\n\
no es puesto executable. Tambin, se cerciora que usted\n\
especifica un sendero repleto al objeto de la\n\
aplicacin.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"L'objet d'application n'est pas dfini, ne pas\n\
existe, ou n'est pas rgl executable. Aussi, s'assure\n\
que vous spcifiez un sentier plein  l'objet\n\
d'application.",
#endif
                        CDIALOG_ICON_WARNING,
                        CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
                    g_free(buf);
                }
	    }
	    /* Check if the handler for this application class MIME Type is
	     * set to EDV_MIMETYPE_HANDLER_COMMAND and if it is then make
	     * sure that atleast one command is defined.
	     */
	    if((mt_ptr->handler == EDV_MIMETYPE_HANDLER_COMMAND) ?
		(mt_ptr->total_commands < 1) : FALSE
	    )
	    {
		status = -2;

                if(verbose)
                {
                    gchar *buf = g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
"No command has been defined on this application\n\
class MIME Type, this MIME Type will not be able to\n\
execute the application it reffers to."
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Ninguna orden se ha definido en esta clase de la\n\
aplicacin MIME Type, este MIME Type no ser capaz de\n\
ejecutar la aplicacin reffers a."
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Aucun ordre a t dfini sur ce MIME Type de\n\
classe d'application, ce MIME Type ne pourra pas\n\
excuter l'application il reffers ."
#endif
		    );
                    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        title, buf,
			NULL,
                        CDIALOG_ICON_WARNING,
                        CDIALOG_BTNFLAG_OK,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
                    g_free(buf);
                }
	    }
	    break;

	  case EDV_MIMETYPE_CLASS_UNIQUE:
	    /* Check if value is defined and reffers to a valid disk 
	     * object.
	     */
	    if((mt_ptr->value != NULL) ?
		access(mt_ptr->value, F_OK) : TRUE
	    )
	    {
		status = -2;
		if(verbose)
		{
		    gchar *buf;
		    if(mt_ptr->value == NULL)
			buf = g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
"Location of unique object not set"
#endif
#ifdef PROG_LANGUAGE_SPANISH
"La ubicacin de objeto extraordinario no conjunto"
#endif
#ifdef PROG_LANGUAGE_FRENCH
"L'emplacement d'objet unique pas srie"
#endif
			);
		    else
			buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unable to find object:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Incapaz de encontrar objeto:\n\
\n\
    %s\n",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Incapable pour trouver l'objet:\n\
\n\
    %s\n",
#endif
			    mt_ptr->value
			);
                    CDialogSetTransientFor(toplevel);
                    CDialogGetResponse(
                        title, buf,
#ifdef PROG_LANGUAGE_ENGLISH
"The unique object is not defined or does not appear\n\
to exist. If it is defined and exists, then check if\n\
its permissions do not allow it to be accessed.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"El objeto extraordinario no se define ni no aparece\n\
existir. Si se define y existe, entonces cheque si\n\
sus permisos no permiten ser conseguido acceso a.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"L'objet unique n'est pas dfini ou n'apparat pas\n\
d'exister. S'il est dfini et existe, alors le chque\n\
si ses permissions ne permettent pas il tre\n\
accessed.",
#endif
                        CDIALOG_ICON_WARNING,
                        CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                        CDIALOG_BTNFLAG_OK
                    );
                    CDialogSetTransientFor(NULL);
		    g_free(buf);
		}
	    }
	    break;

	  default:
	    status = -2;
	    if(verbose)
            {
                gchar *buf = g_strdup_printf(
#ifdef PROG_LANGUAGE_ENGLISH
"Unsupported MIME Type class code `%i'.",
#endif
#ifdef PROG_LANGUAGE_SPANISH
"Cdigo `%i' no apoyado de clase de MIME Type.",
#endif
#ifdef PROG_LANGUAGE_FRENCH
"Le code `%i' non soutenu de classe de MIME Type.",
#endif
                    mt_ptr->mt_class
                );
                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
                    title, buf, NULL,
                    CDIALOG_ICON_ERROR,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
		g_free(buf);
            }
	    break;
	}

	return(status);
}

/*
 *	Loads the MIME Type values from the structure specified by the index
 *	mt_num on the core structure on the given edit window.
 */
void EDVMimeTypesEditWinFetchValues(
        edv_mimetype_editwin_struct *ew, gint mt_num
)
{
	gbool sensitive;
	GtkWidget *w;
	GtkEntry *entry;
	GtkCombo *combo;
	GtkCList *clist;
	edv_mimetype_struct *mt_ptr;
	edv_core_struct *core_ptr;
	edv_mimetype_editwin_icon_struct *ei;


	if(ew == NULL)
	    return;

	core_ptr = (edv_core_struct *)ew->core_ptr;
	if(core_ptr == NULL)
	    return;

	/* Get pointer to MIME Type structure on core's MIME Types list. */
	if((mt_num < 0) || (mt_num >= core_ptr->total_mimetypes))
	    return;
	else
	    mt_ptr = core_ptr->mimetype[mt_num];
	if(mt_ptr == NULL)
	    return;


	/* Begin fetching values. */

	/* Set MIME Type index on core structure from the given mt_num. */
	ew->mt_num = mt_num;

	/* Class. */
	combo = (GtkCombo *)ew->class_combo;
	if(combo != NULL)
	{
	    gint i = 0;
	    const gchar *class_str = NULL;
	    GList *glist = (GList *)GUIComboGetList(combo);


	    entry = GTK_ENTRY(combo->entry);

	    /* Find class string as the index value from the MIME Type
	     * structure's class in the combo's list of strings.
	     */
	    while(glist != NULL)
	    {
		if(i == mt_ptr->mt_class)
		{
		    class_str = (const gchar *)glist->data;
		    break;
		}

		i++;
		glist = glist->next;
	    }

	    /* Check if a class string was matched, in which case we
	     * update the combo's entry widget value with it.
	     */
	    if(class_str != NULL)
		gtk_entry_set_text(entry, class_str);
	}

        /* Type. */
        entry = (GtkEntry *)ew->type_entry;
        if(entry != NULL)
        {
	    gtk_entry_set_text(
		entry,
		(mt_ptr->type != NULL) ? mt_ptr->type : ""
	    );
	}

        /* Value. */
        entry = (GtkEntry *)ew->value_entry;
        if(entry != NULL)
        {
            gtk_entry_set_text(
                entry,
                (mt_ptr->value != NULL) ? mt_ptr->value : ""
            );
        }

        /* Description. */
        entry = (GtkEntry *)ew->description_entry;
        if(entry != NULL)
        {
            gtk_entry_set_text(
                entry,
                (mt_ptr->description != NULL) ? mt_ptr->description : ""
            );
        }


        /* Handler. */
        combo = (GtkCombo *)ew->handler_combo;
        if(combo != NULL)
        {
            gint i = 0;
            const gchar *handler_str = NULL;
            GList *glist = (GList *)GUIComboGetList(combo);


            entry = GTK_ENTRY(combo->entry);

            /* Find handler string as the index value from the MIME Type
             * structure's handler in the combo's list of strings.
             */
            while(glist != NULL)
            {
                if(i == mt_ptr->handler)
                {
                    handler_str = (const gchar *)glist->data;
                    break;
                }

                i++;
                glist = glist->next;
            }

            /* Check if a handler string was matched, in which case we
             * update the combo's entry widget value with it.
             */
            if(handler_str != NULL)
                gtk_entry_set_text(entry, handler_str);
        }


	/* Commands. */
	clist = (GtkCList *)EditCListGetCList(ew->editclist);
	if(clist != NULL)
	{
	    gint i, new_row;
	    gchar **strv;


	    strv = (gchar **)g_malloc0(clist->columns * sizeof(gchar *));

	    gtk_clist_freeze(clist);
	    gtk_clist_clear(clist);

	    for(i = 0; i < mt_ptr->total_commands; i++)
	    {
		if(clist->columns >= 2)
		{
		    strv[0] = mt_ptr->command_name[i];
		    strv[1] = mt_ptr->command[i];
		}
		new_row = gtk_clist_append(clist, strv);
	    }

	    gtk_clist_thaw(clist);

	    g_free(strv);
	}


	/* Begin loading icons. */

#define DO_LOAD_ICON	\
{ \
 for(i = 0; i < EDV_MIMETYPE_TOTAL_ICON_STATES; i++) \
 { \
  /* Increase ref count on source pixmap because it is about to be \
   * transfered. \
   */ \
  if(pixmap[i] != NULL) \
   gdk_pixmap_ref(pixmap[i]); \
\
  /* Unref old pixmap if any on the edit window icon structure. */ \
  if(ei->pixmap[i] != NULL) \
   gdk_pixmap_unref(ei->pixmap[i]); \
\
  /* Transfer source pixmap to target pixmap. */ \
  ei->pixmap[i] = pixmap[i]; \
\
\
  /* Increase ref count on source bitmap because it is about to be \
   * transfered.\
   */ \
  if(mask[i] != NULL) \
   gdk_bitmap_ref(mask[i]); \
\
  /* Unref old bitmap if any on the edit window icon structure. */ \
  if(ei->mask[i] != NULL) \
   gdk_bitmap_unref(ei->mask[i]); \
\
  /* Transfer source bitmap to target bitmap. */ \
  ei->mask[i] = mask[i]; \
\
\
  /* Deallocate old icon file path and copy new icon file path. */ \
  g_free(ei->path[i]); \
  ei->path[i] = (path[i] != NULL) ? g_strdup(path[i]) : NULL; \
\
 } \
}

	/* Small icons. */
	ei = ew->icon_small;
	if(ei != NULL)
	{
	    gint i;
	    GdkPixmap **pixmap = mt_ptr->small_pixmap;
	    GdkBitmap **mask = mt_ptr->small_mask;
	    gchar **path = mt_ptr->small_icon_file;

	    if(ei->adj != NULL)
		ei->adj->value = 0.0;

	    DO_LOAD_ICON

	    if(ei->adj != NULL)
		EDVMimeTypesEditIconUpdateDisplay(
		    ei, (gint)ei->adj->value
		);
	}

        /* Medium icons. */
        ei = ew->icon_medium;
        if(ei != NULL)
        {
            gint i;
            GdkPixmap **pixmap = mt_ptr->medium_pixmap;
            GdkBitmap **mask = mt_ptr->medium_mask;
            gchar **path = mt_ptr->medium_icon_file;

            if(ei->adj != NULL)
                ei->adj->value = 0.0;

            DO_LOAD_ICON

            if(ei->adj != NULL)
                EDVMimeTypesEditIconUpdateDisplay(
                    ei, (gint)ei->adj->value
                );
        }

        /* Large icons. */
        ei = ew->icon_large;
        if(ei != NULL)
        {
            gint i;
            GdkPixmap **pixmap = mt_ptr->large_pixmap;
            GdkBitmap **mask = mt_ptr->large_mask;
            gchar **path = mt_ptr->large_icon_file;

            if(ei->adj != NULL)
                ei->adj->value = 0.0;

            DO_LOAD_ICON

            if(ei->adj != NULL)
                EDVMimeTypesEditIconUpdateDisplay(
                    ei, (gint)ei->adj->value
                );
        }

#undef DO_LOAD_ICON

	if(ew->toplevel != NULL)
	    gtk_widget_queue_resize(ew->toplevel);

#define DO_SET_SENSITIVE	\
{ \
 if(w != NULL) \
  gtk_widget_set_sensitive(w, sensitive); \
}

	/* Update button sensitivities. */
	sensitive = !mt_ptr->read_only;
	w = ew->ok_btn;
	DO_SET_SENSITIVE
	w = ew->apply_btn;
	DO_SET_SENSITIVE

#undef DO_SET_SENSITIVE

	/* Update menus on edit clist. */
	EditCListUpdateMenus(ew->editclist);
}


/*
 *      Notifies the given MIME Types list window that the given MIME Type
 *      has been removed.
 */
void EDVMimeTypesEditWinMimeTypeRemovedCB(
        edv_mimetype_editwin_struct *ew, gint mt_num
)
{
        if(ew == NULL)
            return;

        /* If the removed MIME Type is the MIME Type that this edit window
	 * is displaying values for then this edit window needs to be
	 * unmapped.
	 */
        if(ew->mt_num == mt_num)
        {
            ew->mt_num = -1;
            EDVMimeTypesEditWinUnmap(ew);
        }
}


/*
 *	Creates a new edit window icon structure and parents it to the
 *	given GtkBox parent.
 */
static edv_mimetype_editwin_icon_struct *EDVMimeTypesEditIconNew(
        gpointer editwin, GtkWidget *parent, const gchar *title,
	gint btn_width, gint btn_height
)
{
	gint bw = 20, bh = 20;
	GtkAdjustment *adj;
	GtkWidget *w, *parent2, *parent3, *parent4;
	edv_mimetype_editwin_icon_struct *ei = (edv_mimetype_editwin_icon_struct *)g_malloc0(
	    sizeof(edv_mimetype_editwin_icon_struct)
	);
	if(ei == NULL)
	    return(ei);

	/* Reset values. */
	ei->editwin = editwin;

	/* Create adjustment for keeping track of displayed icon. */
	ei->adj = adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0,	/* Value. */
	    0.0,	/* Lower. */
	    (gfloat)EDV_MIMETYPE_TOTAL_ICON_STATES - 1.0,	/* Upper. */
	    1.0, 1.0,	/* Step and page increments. */
	    1.0		/* Page size. */
	);

	/* Create toplevel. */
	ei->toplevel = w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

        /* Hbox for title label. */
        w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;
	/* Label. */
	w = gtk_label_new(title);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


        /* Hbox for button. */
        w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

	/* Create button. */
	ei->button = w = gtk_button_new();
	gtk_widget_set_usize(w, btn_width, btn_height);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVMimeTypesEditIconChangeCB), ei
        );
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
        gtk_widget_show(w);
	parent3 = w;
/*
	ei->gtk_pixmap = w = gtk_pixmap_new(NULL, NULL);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);
 */


	/* Hbox for buttons and labels. */
	w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;


	/* Left button. */
        ei->left_btn = w = gtk_button_new();
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVMimeTypesEditIconPrevCB), ei
	);
        gtk_widget_set_usize(w, bw, bh);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
        /* Arrow. */
        w = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);

	/* Frame for state label. */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_widget_set_usize(w, 80, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* State label. */
	ei->state_label = w = gtk_label_new("");
	gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);

        /* Right button. */
        ei->right_btn = w = gtk_button_new();
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVMimeTypesEditIconNextCB), ei
        );
        gtk_widget_set_usize(w, bw, bh);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
        /* Arrow. */
        w = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);


	return(ei);
}

/*
 *	Deallocates all resources of the given edit window icon structure
 *	and deallocates the structure itself.
 */
static void EDVMimeTypesEditIconDelete(edv_mimetype_editwin_icon_struct *ei)
{
	gint i;
	GtkWidget **w;
	GdkPixmap *pixmap;
	GdkBitmap *mask;


	if(ei == NULL)
	    return;

#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	/* Unref all loaded pixmap and mask pairs. */
	for(i = 0; i < EDV_MIMETYPE_TOTAL_ICON_STATES; i++)
        {
	    pixmap = ei->pixmap[i];
	    ei->pixmap[i] = NULL;
	    if(pixmap != NULL)
		gdk_pixmap_unref(pixmap);

	    mask = ei->mask[i];
            ei->mask[i] = NULL;
            if(mask != NULL)
                gdk_bitmap_unref(mask);
	}

	/* Destroy all widgets. */
	w = &ei->gtk_pixmap;
	DO_DESTROY_WIDGET

	w = &ei->button;
	DO_DESTROY_WIDGET
	w = &ei->left_btn;
	DO_DESTROY_WIDGET
	w = &ei->right_btn;
	DO_DESTROY_WIDGET
        w = &ei->state_label;
	DO_DESTROY_WIDGET
        w = &ei->toplevel;
	DO_DESTROY_WIDGET

	if(ei->adj != NULL)
	{
	    gtk_object_unref(GTK_OBJECT(ei->adj));
	    ei->adj = NULL;
	}

#undef DO_DESTROY_WIDGET

        /* Deallocate all icon paths. */
        for(i = 0; i < EDV_MIMETYPE_TOTAL_ICON_STATES; i++)
        {
	    g_free(ei->path[i]);
	    ei->path[i] = NULL;
	}

	/* Deallocate structure itself. */
	g_free(ei);
}


/*
 *	Creates a new MIME Types edit window.
 */
edv_mimetype_editwin_struct *EDVMimeTypesEditWinNew(
        gpointer core_ptr, gpointer listwin
)
{
        gint /*border_minor = 2, */ border_major = 5;
        gchar *heading[2];
	gpointer combo_rtn, entry_rtn, label_rtn;
	GList *glist;
        GdkWindow *window;
        GtkAccelGroup *accelgrp;
        GtkWidget *w, *parent, *parent2, *parent3, *parent4;
	GtkCombo *combo;
        GtkCList *clist;
        edv_mimetype_editwin_struct *ew = (edv_mimetype_editwin_struct *)g_malloc0(
            sizeof(edv_mimetype_editwin_struct)
        );
        if(ew == NULL)
            return(ew);


        /* Reset values. */
        ew->initialized = TRUE;
        ew->map_state = FALSE;
        ew->core_ptr = core_ptr;
	ew->listwin = listwin;
	ew->mt_num = -1;


        /* Keyboard accelerator group. */
        ew->accelgrp = accelgrp = gtk_accel_group_new();


        /* Toplevel. */
        ew->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_set_usize(w, EDITWIN_WIDTH, EDITWIN_HEIGHT);
        gtk_widget_realize(w);
        gtk_window_set_title(GTK_WINDOW(w), EDITWIN_TITLE);
        window = w->window;
        if(window != NULL)
        {
            gdk_window_set_decorations(
                window,
                GDK_DECOR_TITLE | GDK_DECOR_MENU | GDK_DECOR_MINIMIZE
            );
            gdk_window_set_functions(
                window,
                GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
            );
            GUISetWMIcon(window, (u_int8_t **)icon_mimetypes_48x48_xpm);
        }
        gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(EDVMimeTypesEditWinDeleteEventCB), ew
        );
        gtk_container_border_width(GTK_CONTAINER(w), 0);
        gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
        parent = w;


        /* Main vbox. */
        ew->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);
        parent = w;


	/* Hbox to split things into two columns. */
        w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent2 = w;


	/* Left side vbox. */
	w = gtk_vbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent3 = w;


	/* Class frame. */
	w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
		"Class"
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"La Clase"
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Classe"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
        parent4 = w;

        w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);
        parent4 = w;

	/* Create class strings glist. */
	glist = NULL;
	glist = g_list_append(glist, g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
		"System Object Type"
#endif
#ifdef PROG_LANGUAGE_SPANISH
		"El Tipo Del Objeto Del Sistema"
#endif
#ifdef PROG_LANGUAGE_FRENCH
		"Le Type D'Objet De Systme"
#endif
	));
	glist = g_list_append(glist, g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
		"File Format"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Archive Formato"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Classer Le Format"
#endif
	));
        glist = g_list_append(glist, g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
		"Application"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "La Aplicacin"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Application"
#endif
	));
        glist = g_list_append(glist, g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
		"Unique Object"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "El Objeto Extraordinario"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Objet Unique"
#endif
	));

	/* Class combo. */
	w = (GtkWidget *)GUIComboCreate(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
	    "File Format",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Archive Formato",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Classer Le Format",
#endif
	    glist,		/* Initial glist of items. */
	    4,			/* Maximum items. */
	    &combo_rtn,
	    ew,
	    NULL,
	    NULL
	);
	combo = (GtkCombo *)combo_rtn;
	ew->class_combo = (GtkWidget *)combo_rtn;
	gtk_entry_set_editable(GTK_ENTRY(combo->entry), FALSE);
        gtk_signal_connect(
            GTK_OBJECT(combo->entry), "changed",
            GTK_SIGNAL_FUNC(EDVMimeTypesEditWinClassChangedCB), ew
        );
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

	/* Deallocate class strings glist. */
	g_list_foreach(glist, (GFunc)g_free, NULL);
	g_list_free(glist);


        /* Identification frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
		"Identification"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Identificacin"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Identification"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

        w = gtk_vbox_new(FALSE, border_major);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);
        parent4 = w;

	/* Type entry. */
	w = (GtkWidget *)GUIPromptBar(
	    NULL,
#ifdef PROG_LANGUAGE_ENGLISH
		"Type:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "El Tipo:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Type:",
#endif
	    NULL, &entry_rtn
	);
	ew->type_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        EDVEntrySetDND((edv_core_struct *)core_ptr, (GtkWidget *)entry_rtn);
        gtk_widget_show(w);

        /* Value entry. */
        w = (GtkWidget *)GUIPromptBar(
            NULL,
#ifdef PROG_LANGUAGE_ENGLISH
		"Value:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "El Valor:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Valeur:",
#endif
	     &label_rtn, &entry_rtn
        );
	ew->value_label = (GtkWidget *)label_rtn;
        ew->value_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        EDVEntrySetDND((edv_core_struct *)core_ptr, (GtkWidget *)entry_rtn);
        gtk_widget_show(w);

        /* Description entry. */
        w = (GtkWidget *)GUIPromptBar(
            NULL,
#ifdef PROG_LANGUAGE_ENGLISH
		"Description:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "La Descripcin:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Description:",
#endif
	    NULL, &entry_rtn
        );
        ew->description_entry = (GtkWidget *)entry_rtn;
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        EDVEntrySetDND((edv_core_struct *)core_ptr, (GtkWidget *)entry_rtn);
        gtk_widget_show(w);


        /* Handler frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
		"Handler"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "El Tratante"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Agent"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;

        w = gtk_vbox_new(FALSE, border_major);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);
        parent4 = w;


        /* Create handlers strings glist. */
        glist = NULL;
        glist = g_list_append(glist, g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
		"Command"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "La Orden"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Ordre"
#endif
	));
        glist = g_list_append(glist, g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
		"Endeavour Archiver"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Endeavour Archiver"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Endeavour Archiver"
#endif
	));
        glist = g_list_append(glist, g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
		"Endeavour Image Browser"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Endeavour Imagine A Examinador"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Endeavour Browser D'Image"
#endif
	));
        glist = g_list_append(glist, g_strdup(
#ifdef PROG_LANGUAGE_ENGLISH
		"Endeavour Recycle Bin"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Endeavour El Cajn De La Recirculacin"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Endeavour Recycler L'Huche"
#endif
	));

        /* Class combo. */
        w = (GtkWidget *)GUIComboCreate(
#ifdef PROG_LANGUAGE_ENGLISH
	    "Handle By:",	/* Label. */
	    "Command",		/* Initial value. */
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Maneje Por:",	/* Label. */
            "La Orden",		/* Initial value. */
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Contrler Par:",	/* Label. */
            "Ordre",		/* Initial value. */
#endif
            glist,              /* Initial glist of items. */
            4,                  /* Maximum items. */
            &combo_rtn,
            ew,
            NULL,
            NULL
        );
        combo = (GtkCombo *)combo_rtn;
        ew->handler_combo = (GtkWidget *)combo_rtn;
        gtk_entry_set_editable(GTK_ENTRY(combo->entry), FALSE);
        gtk_signal_connect(
            GTK_OBJECT(combo->entry), "changed",
            GTK_SIGNAL_FUNC(EDVMimeTypesEditWinHandlerChangedCB), ew
        );
        gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
        gtk_widget_show(w);

        /* Deallocate class strings glist. */
        g_list_foreach(glist, (GFunc)g_free, NULL);
        g_list_free(glist);


	/* Commands editable clist. */
#ifdef PROG_LANGUAGE_ENGLISH
        heading[0] = g_strdup("Name");
        heading[1] = g_strdup("Command");
#endif
#ifdef PROG_LANGUAGE_SPANISH
        heading[0] = g_strdup("El Nombre");
        heading[1] = g_strdup("La Orden");
#endif
#ifdef PROG_LANGUAGE_FRENCH
        heading[0] = g_strdup("Nom");
        heading[1] = g_strdup("Ordre");
#endif
	ew->editclist = EditCListNew(
	    parent4, -1, 120,
	    FALSE, FALSE, 0,
	    heading, 2,
	    NULL,
	    ew
	);
        g_free(heading[0]);
        g_free(heading[1]);
        clist = (GtkCList *)EditCListGetCList(ew->editclist);
        gtk_clist_set_column_width(clist, 0, 80);
        gtk_clist_set_column_width(clist, 1, 230);
        gtk_clist_column_titles_passive(clist);
        gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
        gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	EditCListMap(ew->editclist);



        /* Right side vbox. */
        w = gtk_vbox_new(FALSE, border_major);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent3 = w;

        /* Icons frame. */
        w = gtk_frame_new(
#ifdef PROG_LANGUAGE_ENGLISH
		"Icons"
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Iconos"
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Icnes"
#endif
	);
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent4 = w;

        w = gtk_vbox_new(FALSE, border_major);
        gtk_container_border_width(GTK_CONTAINER(w), border_major);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);
        parent4 = w;


	/* Create icon structures. */
	ew->icon_large = EDVMimeTypesEditIconNew(
	    ew, parent4,
#ifdef PROG_LANGUAGE_ENGLISH
		"Large 48x48:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Grande 48x48:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Grand 48x48:",
#endif
	    54, 54
	);
        ew->icon_medium = EDVMimeTypesEditIconNew(
            ew, parent4,
#ifdef PROG_LANGUAGE_ENGLISH
		"Medium 32x32:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "El Medio 32x32:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Milieu 32x32:",
#endif
	    38, 38
        );
        ew->icon_small = EDVMimeTypesEditIconNew(
            ew, parent4,
#ifdef PROG_LANGUAGE_ENGLISH
		"Small 20x20:",
#endif
#ifdef PROG_LANGUAGE_SPANISH
                "Pequeo 20x20:",
#endif
#ifdef PROG_LANGUAGE_FRENCH
                "Petit 20x20:",
#endif
	    26, 26
        );


        /* Horizontal separator. */
        w = gtk_hseparator_new();
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


	/* Hbox for buttons. */
	w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
	parent2 = w;

        /* OK button. */
        ew->ok_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_ok_20x20_xpm, "OK", NULL
        );
        gtk_widget_set_usize(w, EDITWIN_BTN_WIDTH, EDITWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVMimeTypesEditWinOKCB), ew
        );
        gtk_widget_show(w);

        /* Apply button. */
        ew->apply_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_select_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Apply",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Aplique",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "S'appliquer",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, EDITWIN_BTN_WIDTH, EDITWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVMimeTypesEditWinApplyCB), ew
        );
        gtk_widget_show(w);

        /* Cancel button. */
        ew->cancel_btn = w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_cancel_20x20_xpm,
#ifdef PROG_LANGUAGE_ENGLISH
            "Cancel",
#endif
#ifdef PROG_LANGUAGE_SPANISH
            "Cancele",
#endif
#ifdef PROG_LANGUAGE_FRENCH
            "Annuler",
#endif
	    NULL
        );
        gtk_widget_set_usize(w, EDITWIN_BTN_WIDTH, EDITWIN_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EDVMimeTypesEditWinCancelCB), ew
        );
        gtk_accel_group_add(
            accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        gtk_widget_show(w);


	return(ew);
}

/*
 *	Maps the MIME Types edit window.
 */
void EDVMimeTypesEditWinMap(
        edv_mimetype_editwin_struct *ew
)
{
        GtkWidget *w;

        if(ew == NULL)
            return;

	/* Set OK button as the default on map. */
        w = ew->ok_btn;
        if(w != NULL)
        {
            gtk_widget_grab_focus(w);
            gtk_widget_grab_default(w);
        }

	w = ew->toplevel;
	gtk_widget_show_raise(w);
	ew->map_state = TRUE;
}

/*
 *	Unmaps the MIME Types edit window.
 */
void EDVMimeTypesEditWinUnmap(
        edv_mimetype_editwin_struct *ew
)
{
	GtkWidget *w;

	if(ew == NULL)
	    return;

	w = ew->toplevel;
	if(w == NULL)
	    return;

	if(ew->map_state)
	{
	    gtk_widget_hide(w);
	    ew->map_state = FALSE;
	}
}

/*
 *	Deallocates all resources of the given MIME Types edit window and
 *	deallocates the structure itself.
 */
void EDVMimeTypesEditWinDelete(
        edv_mimetype_editwin_struct *ew
)
{
        GtkWidget **w;


        if(ew == NULL)
            return;

        if(ew->initialized)
        {
#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	    /* Begin destroying widgets. */

	    EDVMimeTypesEditIconDelete(ew->icon_small);
	    ew->icon_small = NULL;
            EDVMimeTypesEditIconDelete(ew->icon_medium);
	    ew->icon_medium = NULL;
            EDVMimeTypesEditIconDelete(ew->icon_large);
	    ew->icon_large = NULL;

	    EditCListDelete(ew->editclist);
	    ew->editclist = NULL;

	    w = &ew->ok_btn;
	    DO_DESTROY_WIDGET
            w = &ew->apply_btn;
            DO_DESTROY_WIDGET
	    w = &ew->cancel_btn;
	    DO_DESTROY_WIDGET

	    w = &ew->class_combo;
	    DO_DESTROY_WIDGET
            w = &ew->description_entry;
            DO_DESTROY_WIDGET
            w = &ew->value_label;
            DO_DESTROY_WIDGET
            w = &ew->value_entry;
            DO_DESTROY_WIDGET
            w = &ew->type_entry;
            DO_DESTROY_WIDGET
            w = &ew->handler_combo;
            DO_DESTROY_WIDGET

            w = &ew->toplevel;
            DO_DESTROY_WIDGET

            if(ew->accelgrp != NULL)
            {
                gtk_accel_group_unref(ew->accelgrp);
                ew->accelgrp = NULL;
            }

#undef DO_DESTROY_WIDGET
        }

        /* Deallocate structure itself. */
	g_free(ew);
}
