#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "cdialog.h"

#include "edvtypes.h"
#include "edvcfg.h"
#include "optwin.h"
#include "endeavour.h"
#include "edvcb.h"
#include "config.h"


static void OptWinFetchWidgetValueFromCFG(
	edv_core_struct *core_ptr, edv_cfg_wref_struct *wref
);
static void OptWinSetWidgetValueToCFG(
        edv_core_struct *core_ptr, edv_cfg_wref_struct *wref
);

void OptWinDoFetch(edv_optwin_struct *optwin);
void OptWinDoApply(edv_optwin_struct *optwin);



/*
 *	Nexus for fetching a value from the configruation list to the
 *	given widget.
 */
static void OptWinFetchWidgetValueFromCFG(
        edv_core_struct *core_ptr, edv_cfg_wref_struct *wref
)
{
	gint i, cfg_type;
	const gchar *cfg_parm;
	edv_cfg_item_struct *cfg_item;
	GtkWidget *w;


	if((core_ptr == NULL) || (wref == NULL))
	    return;

	cfg_parm = wref->cfg_parm;
	if(cfg_parm == NULL)
	    return;


	/* Match configuration item from the configuration list on the
	 * given core structure.
	 */
	i = EDVCFGItemListMatchParameter(
	    core_ptr->cfg_list, cfg_parm
	);
	if(i < 0)
	    return;

	cfg_item = &core_ptr->cfg_list[i];
	cfg_type = cfg_item->type;


	/* Do special handling first. */

	/* Stack list? */
	if(wref->slist != NULL)
	{
	    if(cfg_type == EDV_CFG_ITEM_TYPE_INTLIST)
	    {
		edv_intlist_struct *intlist =
		    (edv_intlist_struct *)cfg_item->value;
		stack_list_struct *slist = wref->slist;


		/* Update stack list's source list from items in its own
		 * cache which were defined when the stack list was
		 * created.
		 */
		StackListItemClearSrc(slist);
		StackListItemSetAllFromCacheSrc(slist);
		/* Remove any occurance of identical items in the intlist
		 * from the source list that do now allow multiple
		 * coppies. This is so that no item already on the target
		 * list will appear on the source list.
		 */
		if(intlist != NULL)
                {
                    for(i = 0; i < intlist->total; i++)
			StackListItemRemoveByIDSrc(
			    slist, intlist->i[i], TRUE
                        );
                }

		/* Add items to the stack list's target list from the
		 * intlist where each value is an id.
		 */
		StackListItemClearTar(slist);
		if(intlist != NULL)
		{
		    for(i = 0; i < intlist->total; i++)
			StackListItemAppendTar(
			    slist, intlist->i[i]
			);
		}
	    }

	    return;
	}


	/* Standard widget handling, get widget that is to hold value. */
	w = wref->w;
	if(w == NULL)
	    return;

	/* Handle by widget type. */
	if(GTK_IS_SPIN_BUTTON(w))
	{
	    GtkSpinButton *spin_button = GTK_SPIN_BUTTON(w);

	    if(cfg_type == EDV_CFG_ITEM_TYPE_STRING)
	    {
		const gchar *values = (const gchar *)cfg_item->value;
		if(values != NULL)
		    gtk_entry_set_text(GTK_ENTRY(w), values);
	    }
	    else
	    {
		gfloat valuef = EDVCFGItemListGetValueF(
		    core_ptr->cfg_list, cfg_parm
		);
		gtk_spin_button_set_value(spin_button, valuef);
	    }
	}
	else if(GTK_IS_COMBO(w))
	{
	    GtkCombo *combo = GTK_COMBO(w);

            if(cfg_type == EDV_CFG_ITEM_TYPE_STRING)
            {
                const gchar *values = (const gchar *)cfg_item->value;
                if(values != NULL)
                    gtk_entry_set_text(GTK_ENTRY(combo->entry), values);
            }
	    else
	    {
		/* All else assume handling as an enumeration, where
		 * the configuration value is an item index on the
		 * combo's list of strings.
		 */
		const gchar *new_text;
		gint n = EDVCFGItemListGetValueI(
		    core_ptr->cfg_list, cfg_parm
		);
		GList *glist = (GList *)GUIComboGetList(combo);

		/* Get glist pointer to the index specified by the
		 * configuration value.
		 */
		glist = g_list_nth(glist, n);
		new_text = (const gchar *)((glist != NULL) ?
		    glist->data : NULL
		);
		if(new_text != NULL)
		    gtk_entry_set_text(GTK_ENTRY(combo->entry), new_text);
	    }
	}
	else if(GTK_IS_ENTRY(w))
	{
            GtkEntry *entry = GTK_ENTRY(w);

            if(cfg_type == EDV_CFG_ITEM_TYPE_STRING)
            {
                const gchar *values = (const gchar *)cfg_item->value;
                if(values != NULL)
                    gtk_entry_set_text(entry, values);
            }
            else
            {
		gchar num_str[80];

		*num_str = '\0';
		switch(cfg_type)
		{
		  case EDV_CFG_ITEM_TYPE_INT8:
                  case EDV_CFG_ITEM_TYPE_UINT8:
		  case EDV_CFG_ITEM_TYPE_INT16:
		  case EDV_CFG_ITEM_TYPE_UINT16:
		  case EDV_CFG_ITEM_TYPE_INT32:
		  case EDV_CFG_ITEM_TYPE_UINT32:
		    sprintf(
			num_str, "%i",
			EDVCFGItemListGetValueI(
			    core_ptr->cfg_list, cfg_parm
			)
		    );
		    break;

                  case EDV_CFG_ITEM_TYPE_INT64:
                  case EDV_CFG_ITEM_TYPE_UINT64:
		    sprintf(
			num_str, "%ld",
			EDVCFGItemListGetValueL(
			    core_ptr->cfg_list, cfg_parm
			)
		    );
		    break;

		  case EDV_CFG_ITEM_TYPE_FLOAT:
                    sprintf(
                        num_str, "%f",
                        EDVCFGItemListGetValueF(
                            core_ptr->cfg_list, cfg_parm
                        )
                    );
                    break;

                  case EDV_CFG_ITEM_TYPE_DOUBLE:
                    sprintf(
                        num_str, "%f",
                        EDVCFGItemListGetValueD(
                            core_ptr->cfg_list, cfg_parm
                        )
                    );
                    break;
		}
		gtk_entry_set_text(entry, num_str);
            }
	}
        else if(GTK_IS_RADIO_BUTTON(w))
        {
	    gint valuei;
	    GtkToggleButton *toggle_button = GTK_TOGGLE_BUTTON(w);

	    valuei = EDVCFGItemListGetValueI(core_ptr->cfg_list, cfg_parm);
	    if(valuei == wref->radio_value)
		toggle_button->active = TRUE;
	    else
		toggle_button->active = FALSE;
        }
	else if(GTK_IS_TOGGLE_BUTTON(w))
	{
            gint valuei;
            GtkToggleButton *toggle_button = GTK_TOGGLE_BUTTON(w);

            valuei = EDVCFGItemListGetValueI(core_ptr->cfg_list, cfg_parm);
	    toggle_button->active = valuei ? TRUE : FALSE;
	}
	else if(GTK_IS_RANGE(w))
	{
	    GtkRange *range = GTK_RANGE(w);
	    GtkAdjustment *adj = range->adjustment;
	    if(adj != NULL)
		adj->value = EDVCFGItemListGetValueF(
		    core_ptr->cfg_list, cfg_parm
		);
	}
	else if(GTK_IS_DRAWING_AREA(w))
	{



	}
	else
	{
		/* Unsupported widget. */
	}
}

/*
 *      Nexus for setting a value from the given widget to the configuration
 *	list.
 *
 *      Inputs assumed valid.
 */
static void OptWinSetWidgetValueToCFG(
        edv_core_struct *core_ptr, edv_cfg_wref_struct *wref
)
{
        gint i, cfg_type;
        const gchar *cfg_parm;
        edv_cfg_item_struct *cfg_item;
        GtkWidget *w;


        if((core_ptr == NULL) || (wref == NULL))
            return;

        cfg_parm = wref->cfg_parm;
        if(cfg_parm == NULL)
            return;


        /* Match configuration item from the configuration list on the
         * given core structure.
         */
        i = EDVCFGItemListMatchParameter(
            core_ptr->cfg_list, cfg_parm
        );
        if(i < 0)
            return;

        cfg_item = &core_ptr->cfg_list[i];
        cfg_type = cfg_item->type;


        /* Do special handling first. */

        /* Stack list? */
        if(wref->slist != NULL)
        {
            if(cfg_type == EDV_CFG_ITEM_TYPE_INTLIST)
            {
		gint *new_id;
		gint total_new_ids;
                edv_intlist_struct *intlist =
                    (edv_intlist_struct *)cfg_item->value;
                stack_list_struct *slist = wref->slist;


		/* Get listing of ids in the stack list's target list and
		 * set those ids to the intlist.
		 */
		new_id = StackListItemGetTar(slist, &total_new_ids);
		if((new_id != NULL) && (intlist != NULL))
		{
		    /* Dealloate old intlist. */
		    g_free(intlist->i);
		    intlist->i = NULL;

		    /* Set new intlist to match the number of ids in the
		     * stack list's target list.
		     */
		    intlist->total = total_new_ids;
		    if(intlist->total > 0)
		    {
			intlist->i = (gint *)g_malloc0(
			    intlist->total * sizeof(gint)
			);
			if(intlist->i == NULL)
			    intlist->total = 0;
			else
			    memcpy(
				intlist->i, new_id,
				intlist->total * sizeof(gint)
			    );
		    }
		    else
		    {
			intlist->total = 0;
		    }
		}

		/* Deallocate returned id list from slist's target list. */
		g_free(new_id);
		new_id = NULL;
		total_new_ids = 0;
	    }
	}



        /* Standard widget handling, get widget that is to hold value. */
        w = wref->w;
        if(w == NULL)
            return;

        /* Handle by widget type. */
        if(GTK_IS_SPIN_BUTTON(w))
        {
            GtkSpinButton *spin_button = GTK_SPIN_BUTTON(w);

            if(cfg_type == EDV_CFG_ITEM_TYPE_STRING)
            {
                const gchar *values = gtk_entry_get_text(GTK_ENTRY(w));
		if(values != NULL)
		    EDVCFGItemListSetValueS(
			core_ptr->cfg_list, cfg_parm,
			values, FALSE
		    );
            }
            else
            {
		if(spin_button->digits > 0)
		{
		    gfloat valuef = gtk_spin_button_get_value_as_float(
			spin_button
		    );
		    EDVCFGItemListSetValueF(
			core_ptr->cfg_list, cfg_parm,
                        valuef, FALSE
                    );
		}
		else
		{
                    gfloat valuei = gtk_spin_button_get_value_as_int(
                        spin_button
                    );
                    EDVCFGItemListSetValueI(
                        core_ptr->cfg_list, cfg_parm,
                        valuei, FALSE
                    );
                }
            }
        }
        else if(GTK_IS_COMBO(w))
        {
            GtkCombo *combo = GTK_COMBO(w);

            if(cfg_type == EDV_CFG_ITEM_TYPE_STRING)
            {
                const gchar *values = gtk_entry_get_text(GTK_ENTRY(combo->entry));
                if(values != NULL)
                    EDVCFGItemListSetValueS(
                        core_ptr->cfg_list, cfg_parm,
                        values, FALSE
                    );
            }
            else
            {
                /* All else assume handling as an enumeration, where
		 * the current string value should match a string in the
		 * combo's list of strings. The matched string index
		 * value will be used as the new configuration value.
                 */
                const gchar *sel_text = gtk_entry_get_text(
		    GTK_ENTRY(combo->entry)
		);
		gint string_index = 0;
                GList *glist = (GList *)GUIComboGetList(combo);


		while(glist != NULL)
		{
		    if((glist->data != NULL) && (sel_text != NULL))
		    {
			if(!strcasecmp((const gchar *)glist->data, sel_text))
			    break;
		    }
		    glist = glist->next;
		    string_index++;
		}

		EDVCFGItemListSetValueI(
		    core_ptr->cfg_list, cfg_parm, string_index, FALSE
		);
            }
	}
        else if(GTK_IS_ENTRY(w))
        {
            GtkEntry *entry = GTK_ENTRY(w);

            if(cfg_type == EDV_CFG_ITEM_TYPE_STRING)
            {
                const gchar *values = gtk_entry_get_text(entry);
                if(values != NULL)
                    EDVCFGItemListSetValueS(
                        core_ptr->cfg_list, cfg_parm,
                        values, FALSE
                    );
            }
	    else
	    {
		const gchar *cstrptr;
                const gchar *values = gtk_entry_get_text(entry);
                if(values != NULL)
		{
		    /* Decimal in the string? */
		    cstrptr = strchr(values, '.');
		    if(cstrptr != NULL)
			EDVCFGItemListSetValueF(
                            core_ptr->cfg_list, cfg_parm,
                            atof(values), FALSE
                        );
		    else
                        EDVCFGItemListSetValueI(
                            core_ptr->cfg_list, cfg_parm,
                            atoi(values), FALSE
                        );
		}
	    }
        }
        else if(GTK_IS_RADIO_BUTTON(w))
        {
            GtkToggleButton *toggle_button = GTK_TOGGLE_BUTTON(w);

	    if(toggle_button->active)
		EDVCFGItemListSetValueI(
		    core_ptr->cfg_list, cfg_parm,
		    wref->radio_value, FALSE
		);
	}
        else if(GTK_IS_TOGGLE_BUTTON(w))
        {
            GtkToggleButton *toggle_button = GTK_TOGGLE_BUTTON(w);

	    EDVCFGItemListSetValueI(
		core_ptr->cfg_list, cfg_parm,
		toggle_button->active ? TRUE : FALSE, FALSE
	    );
        }
        else if(GTK_IS_RANGE(w))
        {
	    GtkRange *range = GTK_RANGE(w);
	    GtkAdjustment *adj = range->adjustment;
	    switch(cfg_type)
	    {
	      case EDV_CFG_ITEM_TYPE_FLOAT:
	      case EDV_CFG_ITEM_TYPE_DOUBLE:
		EDVCFGItemListSetValueF(
		    core_ptr->cfg_list, cfg_parm,
		    adj->value, FALSE
		);
		break;
	      default:
                EDVCFGItemListSetValueI(
                    core_ptr->cfg_list, cfg_parm,
                    (gint)adj->value, FALSE
                );
                break;
	    }
	}
        else if(GTK_IS_DRAWING_AREA(w))
        {



        }
        else
        {
            /* Unsupported widget. */
        }
}

/*
 *	Fetches values from configuration list filtered by the wrefs on
 *	the given optwin and placed into the widgets.
 */
void OptWinDoFetch(edv_optwin_struct *optwin)
{
	gint i;
	edv_core_struct *core_ptr;


	if(optwin == NULL)
	    return;

	core_ptr = (edv_core_struct *)optwin->core_ptr;
	if(core_ptr == NULL)
	    return;

	/* Iterate through all wrefs. */
	for(i = 0; i < optwin->total_wrefs; i++)
	    OptWinFetchWidgetValueFromCFG(core_ptr, optwin->wref[i]);
}

/*
 *	Applies values from the widgets specified by the wrefs on the
 *	given optwin into the configuration list.
 */
void OptWinDoApply(edv_optwin_struct *optwin)
{
        gint i;
        edv_core_struct *core_ptr;


        if(optwin == NULL)
            return;

        core_ptr = (edv_core_struct *)optwin->core_ptr;
        if(core_ptr == NULL)
            return;

        /* Iterate through all wrefs. */
        for(i = 0; i < optwin->total_wrefs; i++)
	    OptWinSetWidgetValueToCFG(
		core_ptr, optwin->wref[i]
	    );

	/* Send out reconfigured signal to all resources. */
	EDVReconfiguredEmit(core_ptr);
}
