#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "../guiutils.h"
#include "../cdialog.h"
#include "../pdialog.h"

#include "../edvtypes.h"
#include "../edvcfglist.h"
#include "../edvdevices.h"
#include "../edvdevicesfio.h"
#include "../lib/endeavour2.h"

#include "ziptool.h"
#include "ziptoolcb.h"
#include "config.h"


static void ZipToolsSignalCB(int s)
{
        switch(s)
        {
          case SIGINT:
          case SIGTERM:
          case SIGSEGV:
            exit(1);
            break;
        }
}

static gint ZipToolsMainTOCB(gpointer data)
{
	zip_tools_struct *zt = (zip_tools_struct *)data;
	if((zt != NULL) ? !zt->map_state : TRUE)
	{
	    gtk_main_quit();
	    return(FALSE);
	}
	else
	{
	    return(TRUE);
	}
}

int main(int argc, char *argv[])
{
	gboolean initialized_gtk = FALSE;
	gint i;
	zip_tools_struct *zt;
	const gchar *cstrptr, *arg_ptr, *startup_device = "/dev/sda4";
        gchar *devices_ini_file;
	edv_device_struct **device = NULL;
	gint total_devices = 0;
	edv_context_struct *ctx;


        /* Set up time zone. */
        tzset();

        /* Set up signal callbacks. */
        signal(SIGINT, ZipToolsSignalCB);
        signal(SIGTERM, ZipToolsSignalCB);
        signal(SIGKILL, ZipToolsSignalCB);
        signal(SIGSEGV, ZipToolsSignalCB);
        signal(SIGSTOP, ZipToolsSignalCB);
        signal(SIGCONT, ZipToolsSignalCB);
        signal(SIGPIPE, ZipToolsSignalCB);


	/* Parse arguments. */
	for(i = 1; i < argc; i++)
	{
	    arg_ptr = argv[i];
	    if(arg_ptr == NULL)
		continue;

	    /* Help. */
	    if(!strcasecmp(arg_ptr, "--help") ||
	       !strcasecmp(arg_ptr, "-help") ||
               !strcasecmp(arg_ptr, "--h") ||
               !strcasecmp(arg_ptr, "-h") ||
               !strcasecmp(arg_ptr, "-?")
	    )
	    {
		printf("%s", PROG_HELP_MESG);
		return(0);
	    }
            /* Version. */
            else if(!strcasecmp(arg_ptr, "--version") ||
                    !strcasecmp(arg_ptr, "-version") ||
                    !strcasecmp(arg_ptr, "--v") ||
                    !strcasecmp(arg_ptr, "-v")
            )
            {
                printf("%s %s\n%s", PROG_NAME, PROG_VERSION, PROG_COPYRIGHT);
		return(0);
            }
	    /* All else assume startup device. */
	    else
	    {
		startup_device = arg_ptr;
	    }
	}



        /* Initialize GTK+ as needed. */
        if(!initialized_gtk)
        {
            if(!gtk_init_check(&argc, &argv))
            {
                fprintf(
                    stderr,
 "This program requires X.\n"
                );
                return(1);
            }
            initialized_gtk = TRUE;
        }

        /* Initialize GDK RGB buffers system. */
        gdk_rgb_init();

        /* Initialize dialogs. */
        CDialogInit();
	PDialogInit();

	/* Initialize Endeavour context. */
	ctx = EDVContextNew();
	EDVContextLoadConfigurationFile(ctx, NULL);

        /* Load devices. */
        cstrptr = EDVGetS(ctx, EDV_CFG_PARM_FILE_DEVICES);
        devices_ini_file = (cstrptr != NULL) ? g_strdup(cstrptr) : NULL;
        EDVDevicesListLoadFromFile(
            devices_ini_file,
            &device, &total_devices,
            NULL, NULL
        );

	/* Check if no devices have been loaded, suggesting that
	 * either no device references exist in the devices.ini or
	 * that the file does not exist.
	 */
	if(total_devices == 0)
	{
	    gchar *buf = g_strdup_printf(
"No devices were found in the file:\n\
\n\
    %s\n\
\n\
It is possible that no device references were defined\n\
or that the device references file does not exist.",
		devices_ini_file
	    );
            CDialogSetTransientFor(NULL);
            CDialogGetResponse(
                "No Devices Found",
                buf,
"You should run Endeavour Mark II and configure the\n\
device references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);
	    g_free(buf);
	}


	/* Create first and only fsck manager window. */
	zt = ZipToolsNew(ctx);
	if(zt != NULL)
	{
            ZipToolsMap(zt);

            ZipToolsSetBusy(zt, TRUE);

            /* Find startup device from devices list. */
            if(startup_device != NULL)
            {
		edv_device_struct *dev_ptr;

		for(i = 0; i < total_devices; i++)
		{
		    dev_ptr = device[i];
		    if(dev_ptr == NULL)
			continue;

		    /* This device's path matches the startup device? */
		    if((dev_ptr->device_path != NULL) ?
		       !strcmp(dev_ptr->device_path, startup_device) : FALSE
		    )
		    {
			/* Transfer this device from the list to the
			 * zip tools window and mark it NULL on our list
			 * since it should not be referenced again.
			 */
			ZipToolsSetDevice(zt, dev_ptr);
			ZipToolsRefreshDevice(zt, dev_ptr);
			ZipToolsUpdateMenus(zt);
			device[i] = dev_ptr = NULL;
			break;
		    }
		}
		/* No such device? */
		if((i >= total_devices) && (total_devices > 0))
		{
		    gchar *buf = g_strdup_printf(
"Could not find device:\n\
\n\
    %s\n\
\n\
It is possible that no device references were defined\n\
or that the device references file does not exist.",
			startup_device
		    );
		    CDialogSetTransientFor(zt->toplevel);
		    CDialogGetResponse(
			"Device Not Found",
			buf,
"You should run Endeavour Mark II and configure the\n\
device references by going to Device->Devices...\n\
and then exit Endeavour Mark II to ensure that\n\
the changes have been saved. Afterwards run this\n\
program again.",
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
			CDIALOG_BTNFLAG_OK
		    );
		    CDialogSetTransientFor(NULL);
		    g_free(buf);

		}
	    }

	    ZipToolsSetBusy(zt, FALSE);


	    gtk_timeout_add(
		1000,		/* ms */
		(GtkFunction)ZipToolsMainTOCB,
		zt
	    );
	    gtk_main();

	    ZipToolsDelete(zt);
	}

	/* Deallocate list of devices (just in case a fsck manager
	 * was not created.
	 */
	for(i = 0; i < total_devices; i++)
	    EDVDeviceDelete(device[i]);
        g_free(device);
        device = NULL;
        total_devices = 0;

        g_free(devices_ini_file);
        devices_ini_file = NULL;

	EDVContextSync(ctx);
	EDVContextDelete(ctx);
	ctx = NULL;

        /* Shutdown dialogs. */
	PDialogShutdown();
        CDialogShutdown();

	return(0);
}
