/*
                                  MIME Types
 */

#ifndef EDVMIMETYPES_H
#define EDVMIMETYPES_H

#include <gtk/gtk.h>
#include "edvtypes.h"


/*
 *	MIME Type Icon States:
 */
typedef enum{
	EDV_MIMETYPE_ICON_STATE_STANDARD,
	EDV_MIMETYPE_ICON_STATE_SELECTED,	/* Directories=Expanded
						 * Other=Selected */
	EDV_MIMETYPE_ICON_STATE_EXTENDED,	/* Links=Broken
						 * Other=Active/Opened/Reserved	 */
	EDV_MIMETYPE_ICON_STATE_HIDDEN
} edv_mimetype_icon_state;
#define EDV_MIMETYPE_TOTAL_ICON_STATES		4


/*
 *	Standard MIME Type Strings:
 *
 *	These are strings used for the member type of the
 *	edv_mimetype_struct
 *
 *	MIME Types of these types are created internally at startup by
 *	the program (not loaded from file) and always of the category
 *	EDV_MIMETYPE_CLASS_SYSTEM
 */
#define EDV_MIMETYPE_STR_UNKNOWN		"unknown"
#define EDV_MIMETYPE_STR_FILE			"file/regular"
#define EDV_MIMETYPE_STR_FILE_EXECUTABLE	"file/executable"
#define EDV_MIMETYPE_STR_DIRECTORY		"directory/regular"
#define EDV_MIMETYPE_STR_LINK			"link/symbolic"
#define EDV_MIMETYPE_STR_DEVICE_BLOCK		"device/block"
#define EDV_MIMETYPE_STR_DEVICE_CHARACTER	"device/character"
#define EDV_MIMETYPE_STR_FIFO			"fifo"
#define EDV_MIMETYPE_STR_SOCKET			"socket"


/*
 *	MIME Type:
 */
typedef struct {

	edv_mimetype_class	mt_class;

	/* The MIME Type's value, it is determined by the class:
	 *
	 * mt_class			value is a
	 * --------                     ----------
	 * EDV_MIMETYPE_CLASS_SYSTEM	*Ignored*
	 * EDV_MIMETYPE_CLASS_FORMAT	Space separated list of extensions
	 * EDV_MIMETYPE_CLASS_PROGRAM	Path to the program object
	 * EDV_MIMETYPE_CLASS_UNIQUE	Path to the unique object
	 */
	gchar		*value;

	/* The name of the MIME Type */
	gchar		*type;

	/* A one line verbose description of the MIME Type */
	gchar		*description;


	/* Indicates if this MIME Type has been realized, meaning that
	 * its icons and related resources have been initialized/loaded
	 */
	gboolean	realized;

	/* Indicates that this MIME Type is either created internally
	 * or loaded from a global configuration
	 *
	 * Either of which cannot be modified or removed if this is
	 * set TRUE
	 */
	gboolean	read_only;

	/* Small 20x20 icons */
	GdkPixmap	*small_pixmap[EDV_MIMETYPE_TOTAL_ICON_STATES];
	GdkBitmap	*small_mask[EDV_MIMETYPE_TOTAL_ICON_STATES];
	gchar		*small_icon_file[EDV_MIMETYPE_TOTAL_ICON_STATES];

	/* Medium 32x32 icons */
	GdkPixmap	*medium_pixmap[EDV_MIMETYPE_TOTAL_ICON_STATES];
	GdkBitmap	*medium_mask[EDV_MIMETYPE_TOTAL_ICON_STATES];
	gchar		*medium_icon_file[EDV_MIMETYPE_TOTAL_ICON_STATES];

	/* Large 48x48 icons */
	GdkPixmap	*large_pixmap[EDV_MIMETYPE_TOTAL_ICON_STATES];
	GdkBitmap	*large_mask[EDV_MIMETYPE_TOTAL_ICON_STATES];
	gchar		*large_icon_file[EDV_MIMETYPE_TOTAL_ICON_STATES];


	/* Handler, one of EDV_MIMETYPE_HANDLER_*
	 *
	 * This is basically to determine if a command should be used
	 * to open the object associated with this MIME Type (in the
	 * case of EDV_MIMETYPE_HANDLER_COMMAND) or to use one of
	 * Endeavour's windows to open the object
	 */
	edv_mimetype_handler	handler;

	/* Command strings and corresponding command name strings
	 *
	 * The first index is usually considered the default
	 *
	 * Command are only used if the handler is set to
	 * EDV_MIMETYPE_HANDLER_COMMAND
	 *
	 * The intention of each command is related by the mt_class as
	 * follows:
	 *
	 * mt_class			value is a
	 * --------			----------
	 * EDV_MIMETYPE_CLASS_SYSTEM	*Ignored*
	 * EDV_MIMETYPE_CLASS_FORMAT	Program commands or program aliases
	 * EDV_MIMETYPE_CLASS_PROGRAM	Program commands (no aliases)
	 * EDV_MIMETYPE_CLASS_UNIQUE	Program commands or program aliases
	 */
	gchar		**command;
	gchar		**command_name;
	gint		total_commands;

	/* Time stamps */
	gulong		access_time,	/* In systime seconds */
			modify_time,
			change_time;

} edv_mimetype_struct;
#define EDV_MIMETYPE(p)		((edv_mimetype_struct *)(p))


extern edv_mimetype_struct *EDVMimeTypeMatchListByType(
	edv_mimetype_struct **list, gint total,
	gint *n,
	const gchar *type, gboolean case_sensitive
);
extern gchar *EDVMimeTypeGetCommandByName(
	edv_mimetype_struct *m, const gchar *name
);

extern edv_mimetype_struct *EDVMimeTypeNew(
	edv_mimetype_class mt_class,	/* One of EDV_MIMETYPE_CLASS_* */
	const gchar *value,
	const gchar *type,		/* MIME Type name */
	const gchar *description	/* Short verbose description */
);

extern void EDVMimeTypeLoadSmallIconsData(
	edv_mimetype_struct *m, guint8 ***data
);
extern void EDVMimeTypeLoadMediumIconsData(
	edv_mimetype_struct *m, guint8 ***data
);
extern void EDVMimeTypeLoadLargeIconsData(
	edv_mimetype_struct *m, guint8 ***data
);

extern void EDVMimeTypeRealize(
	edv_mimetype_struct *m, gboolean force_rerealize
);

extern void EDVMimeTypeDelete(edv_mimetype_struct *m);


#endif	/* EDVMIMETYPES_H */
