/*
				MIME Types
 */

#ifndef EDVMIMETYPES_H
#define EDVMIMETYPES_H

#include <glib.h>
#include "edvtypes.h"


/*
 *	MIME Type Icon States:
 */
typedef enum{
	EDV_MIMETYPE_ICON_STATE_STANDARD,
	EDV_MIMETYPE_ICON_STATE_SELECTED,	/* Directories = Expanded
						 * Other = Selected */
	EDV_MIMETYPE_ICON_STATE_EXTENDED,	/* Links = Broken
						 * Other = Active/Opened/Reserved */
	EDV_MIMETYPE_ICON_STATE_HIDDEN
} edv_mimetype_icon_state;
#define EDV_MIMETYPE_TOTAL_ICON_STATES		4


/*
 *	Standard MIME Type Strings:
 *
 *	These are strings used for member type of the
 *	edv_mimetype_struct
 *
 *	MIME Types of these types are created internally at startup by
 *	the program (not loaded from file) and always of the category
 *	EDV_MIMETYPE_CLASS_SYSTEM
 */
#define EDV_MIMETYPE_STR_UNKNOWN		"unknown"
#define EDV_MIMETYPE_STR_FILE			"file/regular"
#define EDV_MIMETYPE_STR_FILE_EXECUTABLE	"file/executable"
#define EDV_MIMETYPE_STR_DIRECTORY		"directory/regular"
#define EDV_MIMETYPE_STR_LINK			"link/symbolic"
#define EDV_MIMETYPE_STR_DEVICE_BLOCK		"device/block"
#define EDV_MIMETYPE_STR_DEVICE_CHARACTER	"device/character"
#define EDV_MIMETYPE_STR_FIFO			"fifo"
#define EDV_MIMETYPE_STR_SOCKET			"socket"


/*
 *	MIME Type:
 */
typedef struct {

	edv_mimetype_class	mt_class;

	/* Value, its contents is determined by mt_class:
	 *
	 * mt_class			value is a
	 * --------                     ----------
	 * EDV_MIMETYPE_CLASS_SYSTEM	*Ignored*
	 * EDV_MIMETYPE_CLASS_FORMAT	Space separated list of extensions
	 * EDV_MIMETYPE_CLASS_PROGRAM	Path to the program object
	 * EDV_MIMETYPE_CLASS_UNIQUE	Path to the unique object
	 */
	gchar		*value;

	/* The MIME Type's type (aka the MIME Type's name) */
	gchar		*type;

	/* A one line verbose description */
	gchar		*description;

	/* Small 20x20 icons */
	gchar		*small_icon_file[EDV_MIMETYPE_TOTAL_ICON_STATES];

	/* Medium 32x32 icons */
	gchar		*medium_icon_file[EDV_MIMETYPE_TOTAL_ICON_STATES];

	/* Large 48x48 icons */
	gchar		*large_icon_file[EDV_MIMETYPE_TOTAL_ICON_STATES];

	/* Handler, determines if a command should be used to open
	 * objects assicated with this MIME Type (in the case of member
	 * handler = EDV_MIMETYPE_HANDLER_COMMAND) or to use one of
	 * Endeavour's windows to open the object
	 */
	edv_mimetype_handler	handler;

	/* Commands and corresponding command names
	 *
	 * The first command is usually considered the default
	 *
	 * Commands are only used if member handler =
	 * EDV_MIMETYPE_HANDLER_COMMAND
	 *
	 * The intention of each command is related by the mt_class as
	 * follows:
	 *
	 * mt_class			value is a
	 * --------			----------
	 * EDV_MIMETYPE_CLASS_SYSTEM	*Ignored*
	 * EDV_MIMETYPE_CLASS_FORMAT	Program commands or program aliases
	 * EDV_MIMETYPE_CLASS_PROGRAM	Program command (no aliases)
	 * EDV_MIMETYPE_CLASS_UNIQUE	Program commands or program aliases
	 *
	 * If the command is not an absolute path then it is considered
	 * an alias, an alias is the type (the name) of a MIME Type
	 * who's member mt_class = EDV_MIMETYPE_CLASS_PROGRAM
	 */
	gchar		**command;
	gchar		**command_name;
	gint		total_commands;

	/* Time stamps */
	gulong		access_time,	/* In systime seconds */
			modify_time,
			change_time;

} edv_mimetype_struct;
#define EDV_MIMETYPE(p)		((edv_mimetype_struct *)(p))


/*
 *      Creates a new MIME Type.
 */
extern edv_mimetype_struct *EDVMimeTypeNew(
	edv_mimetype_class mt_class,	/* One of EDV_MIMETYPE_CLASS_* */
	const gchar *value,
	const gchar *type,		/* MIME Type name */
	const gchar *description	/* Short verbose description */
);

/*
 *      Coppies the MIME Type.
 */
extern edv_mimetype_struct *EDVMimeTypeCopy(const edv_mimetype_struct *m);

/*
 *      Deletes the MIME Type.
 */
extern void EDVMimeTypeDelete(edv_mimetype_struct *m);


#endif	/* EDVMIMETYPES_H */
