/*
                      Endeavour Utility Functions

           These functions do not depend on Endeavour or GTK.
 */

#ifndef EDVUTILS_H
#define EDVUTILS_H

#include <glib.h>


/*
 *	Returns a statically allocated string describing the specified
 *	numeric value with ',' deliminators at the thousands.
 */
extern const gchar *EDVSizeStrDelim(gulong i);

/*
 *	Returns a dynamically allocated string with all occurances of
 *	the string token replaced with the string val in the string fmt.
 */
extern gchar *EDVStrSub(
	const gchar *fmt, const gchar *token, const gchar *val
);

/*
 *	Simplifies the path by removing any tailing path deliminators
 *	reducing any occurances of "/..".
 */
extern void EDVSimplifyPath(gchar *path);

/*
 *	Checks if the specified parent path is a parent or grand parent
 *	of the specified child path.                             
 *
 *	Both paths must be absolute and simplified.
 */
extern gboolean EDVIsParentPath(const gchar *parent, const gchar *child);

/*
 *	Checks if at least one of the specified list of space-separated
 *	extensions matches the extension of the specified name.
 *
 *	Wild cards such as '*' and '?' are accepted in the
 *	space-separated extensions list.
 *
 *	The following example will return TRUE.
 *
 *	name	= "myfile.doc"
 *	ext	= ".txt .doc *rc Makefile*"
 */
extern gboolean EDVIsExtension(const gchar *name, const gchar *ext);

/*
 *	Returns a dynamically allocated copy of the path that is no
 *	longer than the specified max characters.
 *
 *	If the specified path is longer than max then the shortened
 *	return string will contain a "..." prefix.
 */
extern gchar *EDVShortenPath(const gchar *path, gint max);

/*
 *	Returns a dynamically allocated evaluated path based on the
 *	specified path.
 *
 *	The parent path should be an absolute path indicating the parent
 *	or current path. The parent path is only used if the specified
 *	path is not an absolute path. If the parent path is NULL then
 *	the toplevel directory will be used as the parent path.
 *
 *	The path will be evaulated as follows:
 *
 *	Check for home directory "~" prefix and substitute home
 *	directory needed.
 *
 *	Simplified, all occurances of ".." evaluated.
 *
 *	Tailing directory deliminators removed.
 */
extern gchar *EDVEvaluatePath(const gchar *parent, const gchar *path);

/*
 *	Changes the current working directory.
 *
 *	Returns non-zero on error.
 */
extern gint EDVCHDir(const gchar *path);

/*
 *	Returns a dynamically allocated string describing the current
 *	working directory.
 */
extern gchar *EDVGetCWD(void);

/*
 *	Returns a dynamically allocated string describing a unique file
 *	name in the specified dir that is gauranteed to not exist.
 */
extern gchar *EDVTmpName(const gchar *dir);

/*
 *	Returns a dynamically allocated string path describing the
 *	completed absolute path to the program found from the PATH
 *	environment variable for the program specified name or NULL if
 *	there is no match.
 */
extern gchar *EDVWhich(const gchar *name);

/*
 *	Checks if the link specified by path has a destination that
 *	may possibly be infinately recursive.
 */
extern gboolean EDVIsLinkInfinatelyRecursive(const gchar *path);

/*
 *	Returns the device major and minor numbers parsed from the
 *	specified (dev_t)rdev value (obtained from stat()).
 */
extern void EDVGetDeviceNumbers(gint rdev, gint *major, gint *minor);

/*
 *	Returns the (dev_t)rdev value (obtained from stat()) parsed
 *	from specified device major and minor numbers.
 */
extern gint EDVFormatDeviceNumbers(gint major, gint minor);

/*
 *	Checks if the process speecified by pid is still running.
 */
extern gboolean EDVProcessIsRunning(gint pid);

/*
 *	Executes the command specified by cmd in the background.
 *
 *	Returns the process ID or -1 on error.
 */
extern gint EDVSystem(const gchar *cmd);

/*
 *	Executes the command specified by cmd and blocks until it has
 *	exited.
 *
 *	If status is not NULL then the status of the command be
 *	returned.
 *
 *	Returns the process ID or -1 on error.
 */
extern gint EDVSystemBlock(const gchar *cmd, gint *status);

/*
 *	Opens the file specified by path and returns its contents as
 *	a list of dynamically allocated strings in a GList.
 *
 *	If max_lines is positive then no more than max_lines will be
 *	read from the file.
 */
extern GList *EDVOpenFileGList(const gchar *path, gint max_lines);

/*
 *	Writes each string in the specified GList to the file
 *	specified by path.
 */
extern void EDVSaveFileGList(const gchar *path, const GList *glist);


#endif	/* EDVUTILS_H */
