#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"

#include "edvtypes.h"
#include "cfg.h"
#include "edvobj.h"
#include "edvrecbin.h"
#include "statusbar.h"
#include "recbin.h"
#include "recbincb.h"
#include "recbinopcb.h"
#include "recbindnd.h"
#include "recbincontents.h"
#include "endeavour.h"
#include "edvop.h"
#include "edvcb.h"
#include "edvutils.h"
#include "edvutilsgtk.h"
#include "edvlistseek.h"
#include "edvcfglist.h"
#include "config.h"



void EDVRecBinContentsItemDestroyCB(gpointer data);

static gint EDVRecBinCListColumnSortDateNexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	gint sort_code
);
static gint EDVRecBinCListColumnSortDateAccessCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVRecBinCListColumnSortDateModifyCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVRecBinCListColumnSortDateChangeCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVRecBinCListColumnSortDateDeleteCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);

gint EDVRecBinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
gint EDVRecBinKeyEventCB(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
);
gint EDVRecBinButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);

void EDVRecBinHandleChildAttachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);
void EDVRecBinHandleChildDetachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);

void EDVRecBinResizeColumnCB(
	GtkCList *clist, gint column, gint width, gpointer data
);
void EDVRecBinClickColumnCB(
	GtkCList *clist, gint column, gpointer data
);
void EDVRecBinSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
void EDVRecBinUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);

void EDVRecBinWriteProtectChangedCB(
	edv_recbin_struct *recbin, gboolean state
);
void EDVRecBinRecycledObjectAddedNotifyCB(
	edv_recbin_struct *recbin, guint index
);
void EDVRecBinRecycledObjectModifiedNotifyCB(
	edv_recbin_struct *recbin, guint index
);
void EDVRecBinRecycledObjectRemovedNotifyCB(
	edv_recbin_struct *recbin, guint index
);
void EDVRecBinReconfiguredNotifyCB(edv_recbin_struct *recbin);
void EDVRecBinMimeTypeAddedCB(
	edv_recbin_struct *recbin,
	gint mt_num, edv_mimetype_struct *mt
);
void EDVRecBinMimeTypeModifiedCB(
	edv_recbin_struct *recbin,
	gint mt_num, edv_mimetype_struct *mt
);
void EDVRecBinMimeTypeRemovedCB(
	edv_recbin_struct *recbin, gint mt_num
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Recycle Bin contents GtkCList item "destroy" signal callback.
 */
void EDVRecBinContentsItemDestroyCB(gpointer data)
{
	EDVRecBinObjectDelete(EDV_RECBIN_OBJECT(data));
}


/*
 *      Returns the sort code for the rows sorted by date.
 *
 *      The recycled object structures are obtained from each row and the
 *      dates are compared.
 */
static gint EDVRecBinCListColumnSortDateNexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	gint sort_code
)
{
	gint sort_column;
	const GtkCListRow *row1 = (const GtkCListRow *)ptr1;
	const GtkCListRow *row2 = (const GtkCListRow *)ptr2;
	edv_recbin_object_struct *obj1, *obj2;


	if((clist == NULL) || (row1 == NULL) || (row2 == NULL))
	    return(-1);

	sort_column = clist->sort_column;
	if((sort_column < 0) || (sort_column >= clist->columns))
	    return(-1);

	obj1 = EDV_RECBIN_OBJECT(row1->data);
	obj2 = EDV_RECBIN_OBJECT(row2->data);
	if((obj1 == NULL) || (obj2 == NULL))
	    return(-1);

	switch(sort_code)
	{
	  case 0:       /* Access time */
	    if(obj1->access_time <= obj2->access_time)
		    return((gint)(obj1->access_time < obj2->access_time));
		else
		    return(-1);
	    break;

	  case 1:       /* Modify time */
	    if(obj1->modify_time <= obj2->modify_time)
		    return((gint)(obj1->modify_time < obj2->modify_time));
		else
		    return(-1);
	    break;

	  case 2:       /* Change time */
	    if(obj1->change_time <= obj2->change_time)
		    return((gint)(obj1->change_time < obj2->change_time));
		else
		    return(-1);
	    break;

	  case 3:	/* Delete time */
	    if(obj1->date_deleted <= obj2->date_deleted)
		    return((gint)(obj1->date_deleted < obj2->date_deleted));
		else
		    return(-1);
	    break;
	}

	return(-1);
}

static gint EDVRecBinCListColumnSortDateAccessCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVRecBinCListColumnSortDateNexus(
	    clist, ptr1, ptr2, 0
	));
}

static gint EDVRecBinCListColumnSortDateModifyCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVRecBinCListColumnSortDateNexus(
	    clist, ptr1, ptr2, 1
	));
}

static gint EDVRecBinCListColumnSortDateChangeCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVRecBinCListColumnSortDateNexus(
	    clist, ptr1, ptr2, 2
	));
}

static gint EDVRecBinCListColumnSortDateDeleteCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVRecBinCListColumnSortDateNexus(
	    clist, ptr1, ptr2, 3
	));
}


/*
 *	Recycle Bin toplevel GtkWidget "delete_event" signal callback.
 */
gint EDVRecBinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if(recbin == NULL)
	    return(TRUE);

	if(recbin->processing)
	    return(TRUE);

	EDVRecBinOPClose(recbin);

	return(TRUE);
}

/*
 *	Recycle Bin "key_press_event" or "key_release_event" signal
 *	callback.
 */
gint EDVRecBinKeyEventCB(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = FALSE;
	gboolean is_press;
	gint etype;
	guint keyval, state;
	GtkWidget *w, *focus_widget;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core_ptr;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((widget == NULL) || (key == NULL) || (recbin == NULL))
	    return(status);

	if(recbin->processing)
	    return(status);

	core_ptr = EDV_CORE(recbin->core_ptr);
	if(core_ptr == NULL)
	    return(status);

	w = recbin->toplevel;
	focus_widget = (w != NULL) ? GTK_WINDOW(w)->focus_widget : NULL;
	cfg_list = core_ptr->cfg_list;
	etype = key->type;
	is_press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

/* Stop emit of signal */
#define DO_STOP_KEY_SIGNAL_EMIT	{		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  is_press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	/* If the focus_widget is not a GtkEditable then check if the
	 * keyval is an accelerator key before all subsequence checks
	 */
	if((focus_widget != NULL) ?
	    !GTK_IS_EDITABLE(focus_widget) : TRUE
	)
	{
	    edv_recbin_op op = (edv_recbin_op)EDVMatchAccelKeyOPID(
		cfg_list, EDV_CFG_PARM_RECBIN_ACCELERATOR_KEYS,
		keyval, state
	    );
	    if(op > 0)
	    {
		if(is_press)
		{
		    edv_recbin_opid_struct *opid = EDVRecBinMatchOPID(
			recbin, op
		    );
		    if(opid != NULL)
			EDVRecBinOPCB(NULL, -1, opid);
		}
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		return(status);
	    }
	}

	/* Check which widget this signal is for */

	/* Contents List */
	if(widget == recbin->contents_clist)
	{
	    GtkCList *clist = GTK_CLIST(widget);
	    gint row = EDVCListGetSelectedLast(clist, NULL);

	    switch(keyval)
	    {
	      case GDK_space:
	      case GDK_KP_Space:
		row = clist->focus_row;
		if((row >= 0) && (row < clist->rows) && is_press)
		{
		    gboolean already_selected = FALSE;

		    /* Check if this row is already selected */
		    GList *glist = clist->selection;
		    while(glist != NULL)
		    {
			if(row == (gint)glist->data)
			{
			    already_selected = TRUE;
			    break;
			}
			glist = g_list_next(glist);
		    }

		    gtk_clist_freeze(clist);
		    if(already_selected)
			gtk_clist_unselect_row(clist, row, 0);
		    else
			gtk_clist_select_row(clist, row, 0);
		    gtk_clist_thaw(clist);
		}
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	      default:
		/* For all other alphanumeric character keys and while
		 * no modifier keys are held, attempt to seek to the
		 * item who's name starts with the letter of the key
		 * that was pressed
		 */
		if(isalnum((int)keyval) && is_press &&
                   !(state & GDK_CONTROL_MASK) &&
                   !(state & GDK_SHIFT_MASK)    
		)
		{
		    /* Get the column that is displaying the object
		     * name as column_type_name
		     */
		    gint column_type_name = -1;
		    const cfg_item_struct *cfg_list = core_ptr->cfg_list;
		    const cfg_intlist_struct *intlist = EDV_GET_INTLIST(
			EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
		    );
		    if(intlist != NULL)
		    {
			/* Iterate through column type intlist */
			gint i = 0;
			GList *glist = intlist->list;
			while(glist != NULL)
			{
			    if((edv_recbin_column_type)glist->data ==
				EDV_RECBIN_COLUMN_TYPE_NAME
			    )
			    {
				column_type_name = i;
				break;
			    }
			    i++;
			    glist = g_list_next(glist);
			}
		    }

		    gtk_clist_freeze(clist);
		    EDVCListSeekCharacter(
			clist, column_type_name, 0, keyval
		    );
		    gtk_clist_thaw(clist);

		    DO_STOP_KEY_SIGNAL_EMIT
		    status = TRUE;
		}
		break;
	    }
	}

	return(status);
#undef DO_STOP_KEY_SIGNAL_EMIT
}

/*
 *	Recycle Bin GtkWidget "button_press_event" signal callback.
 */
gint EDVRecBinButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	edv_core_struct *core_ptr;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((widget == NULL) || (recbin == NULL))
	    return(status);

	if(recbin->processing)
	    return(status);

	core_ptr = EDV_CORE(recbin->core_ptr);
	if(core_ptr == NULL)
	    return(status);

	/* Get event type */
	etype = button->type;

	/* Check which widget this signal is for */
	if(widget == recbin->contents_clist)
	{
	    gint row, column;
	    gint rows_selected = 0, selected_row = -1;
	    GList *glist;
	    GtkCList *clist = GTK_CLIST(widget);


	    /* Find row and column based on given coordinates */
	    if(!gtk_clist_get_selection_info(
		clist, button->x, button->y, &row, &column
	    ))
	    {
		row = -1;
		column = 0;
	    }

	    /* Get number of selected rows and highest selected row */
	    glist = clist->selection;
	    while(glist != NULL)
	    {
		rows_selected++;
		selected_row = (gint)glist->data;
		glist = g_list_next(glist);
	    }

	    /* Handle by button number */
	    switch(button->button)
	    {
	      case 3:
		if(etype == GDK_BUTTON_PRESS)
		{
		    GtkMenu *menu;

		    /* Select item before mapping menu? */
		    if(CFGItemListGetValueI(
			core_ptr->cfg_list, EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS
		    ) && (row >= 0) && (row < clist->rows))
		    {
			/* Select the row that the button was pressed over.
			 * if no key modifiers are held then this will also
			 * unselect all previously selected rows.
			 */
			gtk_clist_freeze(clist);
			if(!(button->state & GDK_CONTROL_MASK) &&
			   !(button->state & GDK_SHIFT_MASK)
			)
			    gtk_clist_unselect_all(clist);
			clist->focus_row = row;
			gtk_clist_select_row(clist, row, 0);
			gtk_clist_thaw(clist);
		    }

		    /* Update all menus and map right click menu */
		    EDVRecBinUpdateMenus(recbin);
		    menu = (GtkMenu *)recbin->contents_clist_menu;
		    if(menu != NULL)
			gtk_menu_popup(
			    menu, NULL, NULL,
			    NULL, NULL,
			    button->button, button->time
			);
		}
		status = TRUE;
		break;

	      case 2:
		if(etype == GDK_BUTTON_PRESS)
		{
		    if((row >= 0) && (row < clist->rows))
		    {
/* Renaming not allowed */
		    }
		}
		break;

	      case 1:
		/* Double click? */
		if(etype == GDK_2BUTTON_PRESS)
		{
		    if((row >= 0) && (row < clist->rows))
		    {
			/* Do recover recycled object */
			EDVRecBinOPRecover(recbin);
			status = TRUE;
		    }
		}
		break;
	    }

	    if(etype == GDK_BUTTON_PRESS)
		gtk_widget_grab_focus(widget);
	}

	return(status);
}


/*
 *      GtkHandleBox "child_attached" signal callback.
 */
void EDVRecBinHandleChildAttachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((handle_box == NULL) || (recbin == NULL))
	    return;

	gtk_widget_queue_resize(
	    gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}

/*
 *      GtkHandleBox "child_detached" signal callback.
 */
void EDVRecBinHandleChildDetachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((handle_box == NULL) || (recbin == NULL))
	    return;

	gtk_widget_queue_resize(
	    gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}


/*
 *      Recycle Bin GtkCList "resize_column" signal callback.
 */
void EDVRecBinResizeColumnCB(
	GtkCList *clist, gint column, gint width, gpointer data
)
{
	edv_core_struct *core_ptr;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((clist == NULL) || (recbin == NULL))
	    return;

	if(recbin->processing)
	    return;

	core_ptr = EDV_CORE(recbin->core_ptr);
	if(core_ptr == NULL)
	    return;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == recbin->contents_clist)
	{
	    edv_recbin_column_type column_type = EDV_RECBIN_COLUMN_TYPE_NAME;
	    cfg_intlist_struct *column_types_intlist, *column_width_intlist;

	    /* Get column_type from the given column index */
	    column_types_intlist = CFGItemListGetValueIntList(
		core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
	    );
	    if(column_types_intlist != NULL)
	    {
		column_type = (edv_recbin_column_type)g_list_nth_data(
		    column_types_intlist->list, column
		);
	    }

	    /* Get column widths intlist */
	    column_width_intlist = CFGItemListGetValueIntList(
		core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN_WIDTH
	    );
	    if(column_width_intlist != NULL)
	    {
		GList *glist = g_list_nth(
		    column_width_intlist->list, (guint)column_type
		);
		if(glist != NULL)
		    glist->data = (gpointer)width;
		else
		    g_printerr(
"EDVRecBinResizeColumnCB(): Warning:\
 Specified column type %i not in list of column widths.\n",
			column_type
		    );
	    }
	}
}

/*
 *      Recycle Bin GtkCList "click_column" signal callback.
 */
void EDVRecBinClickColumnCB(
	GtkCList *clist, gint column, gpointer data
)
{
	edv_core_struct *core_ptr;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((clist == NULL) || (recbin == NULL))
	    return;

	if(recbin->processing)
	    return;

	core_ptr = EDV_CORE(recbin->core_ptr);
	if(core_ptr == NULL)
	    return;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == recbin->contents_clist)
	{
	    cfg_intlist_struct *column_types_intlist;
	    GtkCListCompareFunc cmp_func = NULL;
	    GtkCListCompareFunc cmp_func_str =
		 (GtkCListCompareFunc)EDVCListColumnSortStringCB;
	    GtkCListCompareFunc cmp_func_num =
		(GtkCListCompareFunc)EDVCListColumnSortNumericCB;


	    EDVRecBinSetBusy(recbin, TRUE);
	    GUIBlockInput(recbin->toplevel, TRUE);
	    recbin->processing = TRUE;


	    /* Get column types mapping */
	    column_types_intlist = CFGItemListGetValueIntList(
		core_ptr->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
	    );
	    if(column_types_intlist != NULL)
	    {
		edv_recbin_column_type column_type = (edv_recbin_column_type)g_list_nth_data(
		    column_types_intlist->list,
		    column
		);
		switch(column_type)
		{
		  case EDV_RECBIN_COLUMN_TYPE_NAME:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_SIZE:
		    cmp_func = cmp_func_num;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_TYPE:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_PERMISSIONS:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_OWNER:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_GROUP:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_DATE_ACCESS:
		    cmp_func = EDVRecBinCListColumnSortDateAccessCB;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_DATE_MODIFIED:
		    cmp_func = EDVRecBinCListColumnSortDateModifyCB;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_DATE_CHANGED:
		    cmp_func = EDVRecBinCListColumnSortDateChangeCB;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_DATE_DELETED:
		    cmp_func = EDVRecBinCListColumnSortDateDeleteCB;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_LINKED_TO:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_ORIGINAL_LOCATION:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_INDEX:
		    cmp_func = cmp_func_num;
		    break;
		}
	    }


	    gtk_clist_freeze(clist);

	    /* Set sort column settings on the clist */
	    if(column != clist->sort_column)
		gtk_clist_set_sort_column(clist, column);
	    else
		gtk_clist_set_sort_type(
		    clist,
		    (clist->sort_type == GTK_SORT_ASCENDING) ?
			GTK_SORT_DESCENDING : GTK_SORT_ASCENDING
		);
	    if(cmp_func != NULL)
		gtk_clist_set_compare_func(clist, cmp_func);

	    /* Sort rows */
	    gtk_clist_sort(clist);

	    gtk_clist_thaw(clist);

	    recbin->processing = FALSE;
	    GUIBlockInput(recbin->toplevel, FALSE);
	    EDVRecBinSetBusy(recbin, FALSE);

	    EDVRecBinUpdateMenus(recbin);
	}
}

/*
 *      Recycle Bin GtkCList "select_row" signal callback.
 */
void EDVRecBinSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((clist == NULL) || (recbin == NULL))
	    return;

	if(recbin->processing)
	    return;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == recbin->contents_clist)
	{
	    edv_core_struct *core_ptr = EDV_CORE(recbin->core_ptr);

	    /* Get total number of objects selected */
	    const gint nselected = g_list_length(clist->selection);

	    /* Get object */
	    edv_recbin_object_struct *obj = EDV_RECBIN_OBJECT(
		gtk_clist_get_row_data(clist, row)
	    );

	    /* Update selected row */
	    recbin->contents_clist_selected_row = row;

	    /* Update DND icon */
	    EDVRecBinContentsDNDSetIcon(recbin, row, column);

	    if(obj != NULL)
	    {
		/* Update status bar message */
		if(!STRISEMPTY(obj->name))
		{
		    gchar *buf, *size_str = NULL;
		    const gchar *type_str = NULL;

		    /* Get object type string and size string */
		    switch(obj->type)
		    {
		      case EDV_OBJECT_TYPE_UNKNOWN:
			break;
		      case EDV_OBJECT_TYPE_FILE:
			type_str = "File";
			size_str = g_strdup_printf(
			    " (%s byte%s)",
			    EDVGetObjectSizeStr(
				EDV_CORE(recbin->core_ptr),
				obj->size
			    ),
			    (obj->size == 1) ? "" : "s"
			);
			break;
		      case EDV_OBJECT_TYPE_DIRECTORY:
			type_str = "Directory";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_LINK:
			type_str = "Link";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_DEVICE_BLOCK:
			type_str = "Block device";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
			type_str = "Character device";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_FIFO:
			type_str = "FIFO Pipe";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_SOCKET:
			type_str = "Socket";
			size_str = NULL;
			break;
		    }

		    /* Set status bar message */
		    if(nselected > 1)
			buf = g_strdup_printf(
			    "%s objects selected",
			    EDVGetObjectSizeStr(core_ptr, nselected)
			);
		    else if(!strcmp(obj->name, ".."))
			buf = g_strdup_printf(
			    "Parent directory selected"
			);
		    else
			buf = g_strdup_printf(
			    "%s \"%s\" selected%s",
			    type_str, obj->name,
			    (size_str != NULL) ? size_str : ""
			);
		    EDVStatusBarMessage(
			recbin->status_bar, buf, FALSE
		    );
		    g_free(buf);
		    g_free(size_str);
		}
		else
		{
		    EDVStatusBarMessage(
			recbin->status_bar,
			"Object with NULL name selected",
			FALSE
		    );
		}
	    }

	    /* Check if selected row is fully visible, if not then adjust
	     * scroll position to try and make it visible
	     */
	    if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    clist,
		    row, -1,	/* Row, column */
		    0.5f, 0.0f	/* Row, column */
		);

/*	    EDVRecBinSetTitle(recbin); */
	    EDVRecBinUpdateMenus(recbin);
	}
}

/*
 *      Recycle Bin GtkCList "unselect_row" signal callback.
 */
void EDVRecBinUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((clist == NULL) || (recbin == NULL))
	    return;

	if(recbin->processing)
	    return;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == recbin->contents_clist)
	{
	    gchar *buf;
	    edv_core_struct *core_ptr = EDV_CORE(recbin->core_ptr);

	    /* Get total number of objects selected */
	    const gint nselected = g_list_length(clist->selection);

	    /* Update status bar message */
	    if(nselected > 0)
		buf = g_strdup_printf(
		    "%s object%s selected",
		    EDVGetObjectSizeStr(core_ptr, nselected),
		    (nselected == 1) ? "" : "s"  
		);
	    else
		buf = STRDUP("No objects selected");
	    EDVStatusBarMessage(
		recbin->status_bar, buf, FALSE
	    );
	    g_free(buf);

	    EDVRecBinUpdateMenus(recbin);
	}
}


/*
 *	Write protect changed callback.
 */
void EDVRecBinWriteProtectChangedCB(
	edv_recbin_struct *recbin, gboolean state
)
{
	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	EDVRecBinUpdateMenus(recbin);
}

/*
 *	Recycled object added callback.
 */
void EDVRecBinRecycledObjectAddedNotifyCB(
	edv_recbin_struct *recbin, guint index
)
{
	edv_core_struct *core_ptr;

	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	core_ptr = EDV_CORE(recbin->core_ptr);
	if(core_ptr == NULL)
	    return;

	/* Contents List */
	EDVRecBinContentsObjectAddedNotify(recbin, index);

	/* Update menus for the purpose of updating the Recycle Bin
	 * toplevel window's WM icon. Do this only if there is a change
	 * in the number of recycled objects from 0.
	 */
	if(core_ptr->last_recbin_items != recbin->last_recbin_items)
	    EDVRecBinUpdateMenus(recbin);
}

/*
 *	Recycled object modified callback.
 */
void EDVRecBinRecycledObjectModifiedNotifyCB(
	edv_recbin_struct *recbin, guint index
)
{
	edv_core_struct *core_ptr;

	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	core_ptr = EDV_CORE(recbin->core_ptr);
	if(core_ptr == NULL)
	    return;

	/* Contents List */
	EDVRecBinContentsObjectModifiedNotify(recbin, index);

	/* Update menus for the purpose of updating the Recycle Bin
	 * toplevel window's WM icon. Do this only if there is a
	 * change in the number of recycled objects from 0
	 */
	if(core_ptr->last_recbin_items != recbin->last_recbin_items)
	    EDVRecBinUpdateMenus(recbin);
}

/*
 *	Recycled object removed callback.
 */
void EDVRecBinRecycledObjectRemovedNotifyCB(
	edv_recbin_struct *recbin, guint index
)
{
	edv_core_struct *core_ptr;

	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	core_ptr = EDV_CORE(recbin->core_ptr);
	if(core_ptr == NULL)
	    return;

	/* Contents List */
	EDVRecBinContentsObjectRemovedNotify(recbin, index);

	/* Update menus for the purpose of updating the Recycle Bin
	 * toplevel window's WM icon. Do this only if there is a change
	 * in the number of recycled objects from 0
	 */
	if(core_ptr->last_recbin_items != recbin->last_recbin_items)
	    EDVRecBinUpdateMenus(recbin);
}


/*
 *	Reconfigured callback.
 */
void EDVRecBinReconfiguredNotifyCB(edv_recbin_struct *recbin)
{
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkWidget *w;
	const cfg_item_struct *cfg_list;
	edv_statusbar_struct *status_bar;
	edv_core_struct *core_ptr;

	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	core_ptr = EDV_CORE(recbin->core_ptr);
	if(core_ptr == NULL)
	    return;

	cfg_list = core_ptr->cfg_list;
	standard_rcstyle = core_ptr->standard_rcstyle;
	lists_rcstyle = core_ptr->lists_rcstyle;


	/* Reset last state markers so that resources get explicitly
	 * checked due to reconfiguring
	 */
	recbin->last_recbin_items = -1;
	recbin->last_write_protect_state = -1;


	/* Update title */
	EDVRecBinSetTitle(recbin);

	/* Update Accelkey Labels */
	EDVRecBinAccelkeysRegenerate(recbin);

	/* Regenerate Tool Bar */
	EDVRecBinToolBarRegenerate(recbin);

	/* Show tool bar? */
	w = recbin->tool_bar_handle;
	if(w != NULL)
	{
	    recbin->tool_bar_map_state = CFGItemListGetValueI(
		cfg_list, EDV_CFG_PARM_RECBIN_SHOW_TOOL_BAR
	    );
	    if(recbin->tool_bar_map_state)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	/* Show find bar? */
	w = recbin->find_bar_handle;
	if(w != NULL)
	{
	    recbin->find_bar_map_state = CFGItemListGetValueI(
		cfg_list, EDV_CFG_PARM_RECBIN_SHOW_FIND_BAR
	    );
	    if(recbin->find_bar_map_state)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	/* Show status bar? */
	status_bar = recbin->status_bar;
	if(status_bar != NULL)
	{
	    recbin->status_bar_map_state = CFGItemListGetValueI(
		cfg_list, EDV_CFG_PARM_RECBIN_SHOW_STATUS_BAR
	    );
	    if(recbin->status_bar_map_state)
		EDVStatusBarMap(status_bar);
	    else
		EDVStatusBarUnmap(status_bar);
	}


	/* Update RC styles */
	w = recbin->toplevel;
	if((w != NULL) && (standard_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, standard_rcstyle);
	w = recbin->contents_clist;
	if((w != NULL) && (lists_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, lists_rcstyle);
	w = recbin->contents_clist_menu;
	if((w != NULL) && (standard_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, standard_rcstyle);

	/* Realize listings */
	EDVRecBinContentsRealizeListing(recbin);

	/* Update menus */
	EDVRecBinUpdateMenus(recbin);

	/* Notify toplevel to resize */
	w = recbin->toplevel;
	if(w != NULL)
	    gtk_widget_queue_resize(w);
}


/*
 *	MIME Type added callback.
 */
void EDVRecBinMimeTypeAddedCB(
	edv_recbin_struct *recbin,
	gint mt_num, edv_mimetype_struct *mt
)
{
	/* Treat a MIME Type added the same as it would be for a MIME
	 * Type modified, forward signal to the MIME Type modified
	 * callback
	 */
	EDVRecBinMimeTypeModifiedCB(recbin, mt_num, mt);
}

/*
 *	MIME Type modified callback.
 */
void EDVRecBinMimeTypeModifiedCB(
	edv_recbin_struct *recbin,
	gint mt_num, edv_mimetype_struct *mt
)
{
	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	/* Realize listings */
	EDVRecBinContentsRealizeListing(recbin);

/*      EDVRecBinUpdateMenus(recbin); */
}

/*
 *	MIME Type removed callback.
 */
void EDVRecBinMimeTypeRemovedCB(
	edv_recbin_struct *recbin, gint mt_num
)
{
	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	/* Realize listings */
	EDVRecBinContentsRealizeListing(recbin);

/*	EDVRecBinUpdateMenus(recbin); */
}
