#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#if defined(HAVE_REGEX)
# include <regex.h>
#else
# include <fnmatch.h>
#endif
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "progressdialog.h"
#include "fprompt.h"

#include "edv_types.h"
#include "edv_date.h"
#include "cfg.h"
#include "edv_id.h"
#include "edv_obj.h"
#include "edv_obj_op.h"
#include "edv_mime_type.h"
#include "browser.h"
#include "browser_cb.h"
#include "browser_dir_tree.h"
#include "browser_contents_list.h"
#include "endeavour2.h"
#include "edv_open.h"
#include "edv_cb.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"

#include "edv_cfg_list.h"
#include "config.h"


typedef struct _EDVBrowserContentsListFPromptData	EDVBrowserContentsListFPromptData;
#define EDV_BROWSER_CONTENTS_LIST_FPROMPT_DATA(p)	((EDVBrowserContentsListFPromptData *)(p))


static edv_object_struct *EDVBrowserNewErrorObject(
	const gchar *path, const gchar *error_msg
);

/* Column Setting */
static void EDVBrowserContentsResetColumns(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist
);

/* Cell Setting */
static void EDVBrowserContentsSetCellName(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
);
static void EDVBrowserContentsSetCellSize(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size, const gboolean hide_link_size,
        const edv_size_format size_format,
        const gulong block_size
);
static void EDVBrowserContentsSetCellType(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
);
static void EDVBrowserContentsSetCellPermissions(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column, gboolean hide_link_permissions
);
static void EDVBrowserContentsSetCellOwner(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
);
static void EDVBrowserContentsSetCellGroup(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
);
static void EDVBrowserContentsSetCellDateAccess(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column,
	edv_date_relativity date_relativity, const gchar *date_format
);
static void EDVBrowserContentsSetCellDateModified(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column,
	edv_date_relativity date_relativity, const gchar *date_format
);
static void EDVBrowserContentsSetCellDateChanged(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column,
	edv_date_relativity date_relativity, const gchar *date_format
);
static void EDVBrowserContentsSetCellHardLinks(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
);
static void EDVBrowserContentsSetCellLinkedTo(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
);
static void EDVBrowserContentsSetCellDevice(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
);
static void EDVBrowserContentsSetCellIndex(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
);
static void EDVBrowserContentsSetCellDeviceType(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
);
static void EDVBrowserContentsSetCellBlockSize(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
);
static void EDVBrowserContentsSetCellBlocks(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
);
static void EDVBrowserContentsSetRow(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row
);

static gint EDVBrowserContentsAppendObject(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist,
	edv_object_struct *obj
);
static void EDVBrowserContentsAppendListing(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist,
	const gchar *path,
	const gboolean update_status_bar
);

/* Selection */
GList *EDVBrowserContentsGetSelectedPaths(edv_browser_struct *browser);

/* Finding */
gint EDVBrowserContentsFindRowByIndex(
	edv_browser_struct *browser,
	const gulong device_index, const gulong index
);
gint EDVBrowserContentsFindRowByPath(
	edv_browser_struct *browser, const gchar *path
);

/* Realize Listing */
void EDVBrowserContentsRealizeListing(edv_browser_struct *browser);

/* Get Listing */
void EDVBrowserContentsGetListing(
	edv_browser_struct *browser, const gchar *path,
	const gboolean update_status_bar
);
void EDVBrowserContentsClear(edv_browser_struct *browser);

/* Opening */
void EDVBrowserContentsOpen(
	edv_browser_struct *browser,
	const gint row, const gint column,
	const guint state		/* Modifier keys */
);
void EDVBrowserContentsOpenWith(
	edv_browser_struct *browser, const gint row, const gint column
);

/* Renaming */
static void EDVBrowserContentsFPromptRenameApplyCB(
	gpointer data, const gchar *value
);
static void EDVBrowserContentsFPromptRenameCancelCB(gpointer data);
void EDVBrowserContentsPromptRename(
	edv_browser_struct *browser, const gint row, const gint column
);

/* Object Callbacks */
void EDVBrowserContentsObjectAddedNotify(
	edv_browser_struct *browser, const gchar *path,
	const struct stat *lstat_buf
);
void EDVBrowserContentsObjectModifiedNotify(
	edv_browser_struct *browser, const gchar *path,
	const gchar *new_path,
	const struct stat *lstat_buf
);
void EDVBrowserContentsObjectRemovedNotify(
	edv_browser_struct *browser, const gchar *path
);

/* Mount Callbacks */
void EDVBrowserContentsMountNotify(
	edv_browser_struct *browser, edv_device_struct *dev,
	gboolean is_mounted
);


struct _EDVBrowserContentsListFPromptData {
	edv_browser_struct	*browser;
	GtkWidget	*clist;
	gint		row,
			column;
};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new error object.
 *
 *	The path specifies the full path to the error object.
 */
static edv_object_struct *EDVBrowserNewErrorObject(
	const gchar *path, const gchar *error_msg
)
{
	edv_object_struct *obj = EDVObjectNew();
	if(obj == NULL)
	    return(NULL);

	obj->type = EDV_OBJECT_TYPE_ERROR;
	EDVObjectSetPath(obj, path);
	EDVObjectUpdateLinkFlags(obj);
	if(!STRISEMPTY(error_msg))
	    EDVObjectPropSet(
		obj,
		EDV_OBJECT_PROP_NAME_ERROR,
		error_msg,
		TRUE
	    );

	return(obj);
}


/*
 *	Resets the Contents List's column headings to the names and 
 *	ordering specified by the configuration. 
 *
 *	All inputs assumed valid.
 */
static void EDVBrowserContentsResetColumns(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist
)
{
	gint i, width;
	const gchar *title = NULL;
	GList *glist;
	GtkJustification justify = GTK_JUSTIFY_LEFT;
	GtkRcStyle *lists_rcstyle = core->lists_rcstyle;
	cfg_intlist_struct	*column_types_intlist,
				*column_width_intlist;
	const cfg_item_struct *cfg_list = core->cfg_list;
	edv_browser_column_type column_type;

	/* Get the column types mapping */
	column_types_intlist = EDV_GET_INTLIST(
	    EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
	    return;

	/* Get the column widths */
	column_width_intlist = EDV_GET_INTLIST(
	    EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN_WIDTH
	);
	if(column_width_intlist == NULL)
	    return;

	gtk_clist_freeze(clist);

	/* Update the clist's column settings */
	gtk_clist_column_titles_active(clist);
	gtk_clist_column_titles_show(clist);
	gtk_clist_set_auto_sort(clist, FALSE);
	gtk_clist_set_sort_type(clist, GTK_SORT_DESCENDING);

#if 0
/* Already set */
	/* Change clist selection mode to GTK_SELECTION_EXTENDED
	 *
	 * The selection mode can change whenever the contents list is
	 * updated
	 */
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
#endif

	/* Iterate through each column */
	for(i = 0, glist = column_types_intlist->list;
	    glist != NULL;
	    i++, glist = g_list_next(glist)
	)
	{
	    column_type = (edv_browser_column_type)glist->data;

	    /* Get the width for this column type */
	    width = (gint)g_list_nth_data(
		column_width_intlist->list,
		(guint)column_type
	    );

	    /* Get column title and justification  based on the
	     * column type
	     */
	    switch(column_type)
	    {
	      case EDV_BROWSER_COLUMN_TYPE_NAME:
		title = "Name";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_SIZE:
		title = "Size";
		justify = GTK_JUSTIFY_RIGHT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_TYPE:
		title = "Type";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_PERMISSIONS:
		title = "Permissions";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_OWNER:
		title = "Owner";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_GROUP:
		title = "Group";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_DATE_ACCESS:
		title = "Date Access";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_DATE_MODIFIED:
		title = "Date Modified";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_DATE_CHANGED:
		title = "Date Changed";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_HARD_LINKS:
		title = "Hard Links";
		justify = GTK_JUSTIFY_RIGHT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_LINKED_TO:
		title = "Linked To";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_DEVICE_INDEX:
		title = "Device Index";
		justify = GTK_JUSTIFY_RIGHT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_INDEX:
		title = "Index";
		justify = GTK_JUSTIFY_RIGHT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_DEVICE_TYPE:
		title = "Device Type";
		justify = GTK_JUSTIFY_RIGHT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_BLOCK_SIZE:
		title = "Block Size";
		justify = GTK_JUSTIFY_RIGHT;
		break;
	      case EDV_BROWSER_COLUMN_TYPE_BLOCKS:
		title = "Blocks";
		justify = GTK_JUSTIFY_RIGHT;
		break;
	    }

	    gtk_clist_set_column_visibility(clist, i, TRUE);
	    gtk_clist_set_column_auto_resize(
		clist, i, FALSE
	    );
	    gtk_clist_set_column_title(clist, i, title);
	    gtk_clist_set_column_width(clist, i, width);
	    gtk_clist_set_column_justification(clist, i, justify);
	}
	/* Reset the rest of the columns */
	for(; i < clist->columns; i++)
	    gtk_clist_set_column_visibility(clist, i, FALSE);

	/* Set RC style after column headings have been mapped */
	if(lists_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		GTK_WIDGET(clist), lists_rcstyle
	    );

	gtk_clist_thaw(clist);
}

/*
 *	Sets the object name for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellName(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
)
{
	const gchar *name = obj->name;
	GdkPixmap *pixmap, *pixmap_exp, *pixmap_ext, *pixmap_hid;
	GdkBitmap *mask, *mask_exp, *mask_ext, *mask_hid;
	GtkStyle	**cell_styles = browser->cell_style,
			*style = NULL;

	/* Get icon */
	EDVMatchObjectIcon(
	    core->device, core->total_devices,
	    core->mimetype, core->total_mimetypes,
	    obj->type,
	    obj->full_path,
	    EDV_OBJECT_IS_LINK_VALID(obj),
	    obj->permissions,
	    0,			/* Small icons */
	    &pixmap, &mask,
	    &pixmap_exp, &mask_exp,
	    &pixmap_ext, &mask_ext,
	    &pixmap_hid, &mask_hid
	);
	/* Get style and alternate icon based on any type-specific
	 * conditions
	 */
	switch(obj->type)
	{
	  case EDV_OBJECT_TYPE_UNKNOWN:
	    break;
	  case EDV_OBJECT_TYPE_FILE:
	    /* Hidden */
	    if(EDVIsObjectHidden(obj))
	    {
		if(pixmap_hid != NULL)
		{
		    pixmap = pixmap_hid;
		    mask = mask_hid;
		}
	    }
	    break;
	  case EDV_OBJECT_TYPE_DIRECTORY:
	    /* Hidden */
	    if(EDVIsObjectHidden(obj))
	    {
		if(pixmap_hid != NULL)
		{
		    pixmap = pixmap_hid;
		    mask = mask_hid;
		}
	    }
	    /* Special directory notations? */
	    if(!strcmp(name, ".."))
	    {
		if(browser->folder_parent_pixmap != NULL)
		{
		    pixmap = browser->folder_parent_pixmap;
		    mask = browser->folder_parent_mask;
		}
	    }
#if 0
	    /* Home directory? */
	    if((!STRISEMPTY(obj->full_path) &&
		!STRISEMPTY(core->home_dir)) ?
		!strcmp(obj->full_path, core->home_dir) : FALSE
	    )
	    {
		if(browser->folder_home_pixmap != NULL)
		{
		    pixmap = browser->folder_home_pixmap;
		    mask = browser->folder_home_mask;
		}
	    }
#endif
	    /* Directory not accessable? */
	    if(!EDVIsObjectAccessable(core, obj))
	    {
		if(pixmap_ext != NULL)
		{
		    pixmap = pixmap_ext;
		    mask = mask_ext;
		}
		if(style == NULL)
		    style = cell_styles[
			EDV_BROWSER_CELL_STYLE_NO_ACCESS
		    ];
	    }
	    break;
	  case EDV_OBJECT_TYPE_LINK:
	    /* Dangling? */
	    if(!EDV_OBJECT_IS_LINK_VALID(obj))
	    {
		if(pixmap_ext != NULL)
		{
		    pixmap = pixmap_ext;
		    mask = mask_ext;
		}
		if(style == NULL)
		    style = cell_styles[
		        EDV_BROWSER_CELL_STYLE_DANGLING_LINK
		    ];
	    }
	    /* Infinately recursive? */
	    if(EDV_OBJECT_IS_LINK_TAR_GRAND_PARENT(obj))
	    {
		if(style == NULL)
		    style = cell_styles[
			EDV_BROWSER_CELL_STYLE_RECURSIVE_LINK
		    ];
	    }
	    break;
	  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
	    /* Hidden */
	    if(EDVIsObjectHidden(obj))
	    {
		if(pixmap_hid != NULL)
		{
		    pixmap = pixmap_hid;
		    mask = mask_hid;
		}
	    }
	    break;
	  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
	    /* Hidden */
	    if(EDVIsObjectHidden(obj))
	    {
		if(pixmap_hid != NULL)
		{
		    pixmap = pixmap_hid;
		    mask = mask_hid;
		}
	    }
	    break;
	  case EDV_OBJECT_TYPE_FIFO:
	    /* Hidden */
	    if(EDVIsObjectHidden(obj))
	    {
		if(pixmap_hid != NULL) 
		{
		    pixmap = pixmap_hid;
		    mask = mask_hid;
		}
	    }
	    break;
	  case EDV_OBJECT_TYPE_SOCKET:
	    /* Hidden */
	    if(EDVIsObjectHidden(obj))
	    {
		if(pixmap_hid != NULL)
		{
		    pixmap = pixmap_hid;
		    mask = mask_hid;
		}
	    }
	    break;
	  case EDV_OBJECT_TYPE_ERROR:
	    break;
	}

	/* Set cell */
	gtk_clist_freeze(clist);
	if(pixmap != NULL)
	     gtk_clist_set_pixtext(
		clist, row, column,
		(name != NULL) ? name : "(null)",
		EDV_LIST_PIXMAP_TEXT_SPACING,
		pixmap, mask
	    );
	else
	    gtk_clist_set_text(
		clist, row, column,
		(name != NULL) ? name : "(null)"
	    );
	gtk_clist_set_cell_style(clist, row, column, style);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *      Sets the object size for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellSize(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size, const gboolean hide_link_size,
        const edv_size_format size_format,
        const gulong block_size
)
{
	const gint border_minor = 2;
	const gchar *s;
	const edv_object_type type = obj->type;

	if(hide_dir_size && (type == EDV_OBJECT_TYPE_DIRECTORY))
	    s = "";
	else if(hide_link_size && (type == EDV_OBJECT_TYPE_LINK))
	    s = "";
	else if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
		(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
	    s = "";
	else
	    s = EDVSizeStrFormat(
		obj->size, size_format, block_size, ',', TRUE
	    );

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, (s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the object type for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellType(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
)
{
	gchar *type_str;

	/* Get MIME Type type string for the given object */
	EDVMatchObjectTypeString(
	    core->mimetype, core->total_mimetypes,
	    obj->type, obj->permissions,
	    obj->full_path,
	    &type_str
	);

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, (type_str != NULL) ? type_str : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *      Sets the object permissions for the specified cell on the given
 *	clist.
 */
static void EDVBrowserContentsSetCellPermissions(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column, gboolean hide_link_permissions
)
{
	edv_permission_flags permissions = obj->permissions;
	gchar s[80];

	if(hide_link_permissions && (obj->type == EDV_OBJECT_TYPE_LINK))
	    *s = '\0';
	else
	    g_snprintf(
		s, sizeof(s),
		"%c%c%c%c%c%c%c%c%c",
	    (permissions & EDV_PERMISSION_UREAD) ? 'r' : '-',
	    (permissions & EDV_PERMISSION_UWRITE) ? 'w' : '-',
	    (permissions & EDV_PERMISSION_SETUID) ?
		'S' :
		((permissions & EDV_PERMISSION_UEXECUTE) ? 'x' : '-'),
	    (permissions & EDV_PERMISSION_GREAD) ? 'r' : '-',
	    (permissions & EDV_PERMISSION_GWRITE) ? 'w' : '-',
	    (permissions & EDV_PERMISSION_SETGID) ?
		'G' :
		((permissions & EDV_PERMISSION_GEXECUTE) ? 'x' : '-'),
	    (permissions & EDV_PERMISSION_AREAD) ? 'r' : '-',
	    (permissions & EDV_PERMISSION_AWRITE) ? 'w' : '-',
	    (permissions & EDV_PERMISSION_STICKY) ?
		'T' :
		((permissions & EDV_PERMISSION_AEXECUTE) ? 'x' : '-')
	    );

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, s
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *      Sets the object owner for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellOwner(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
)
{
	/* Get owner name from object's user id */
	gchar *owner_name = EDVUIDGetNameFromUID(
	    core->uids_list,
	    obj->owner_id, NULL
	);

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column,
	    (owner_name != NULL) ? owner_name : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);

	g_free(owner_name);
}

/*
 *      Sets the object group for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellGroup(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
)
{
	/* Get group name from object's group id */
	gchar *group_name = EDVGIDGetNameFromGID(
	    core->gids_list,
	    obj->group_id, NULL
	);

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column,
	    (group_name != NULL) ? group_name : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);

	g_free(group_name);
}

/*
 *	Sets the object access date for the specified cell on the given
 *	clist.
 */
static void EDVBrowserContentsSetCellDateAccess(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column,
	edv_date_relativity date_relativity, const gchar *date_format
)
{
	const gchar *s;

	/* Get date string for last access date */
	const gulong t = obj->access_time;
	if(t > 0)
	    s = EDVDateFormatString(
		t, date_format, date_relativity
	    );
	else
	    s = "";

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, s
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *      Sets the object modified date for the specified cell on the given
 *      clist.
 */
static void EDVBrowserContentsSetCellDateModified(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column,
	edv_date_relativity date_relativity, const gchar *date_format
)
{
	const gchar *s;

	/* Get date string for last modify date */
	const gulong t = obj->modify_time;
	if(t > 0)
	    s = EDVDateFormatString(
		t, date_format, date_relativity
	    );
	else
	    s = "";

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, s
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *      Sets the object change date for the specified cell on the given
 *      clist.
 */
static void EDVBrowserContentsSetCellDateChanged(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column,
	edv_date_relativity date_relativity, const gchar *date_format
)
{
	const gchar *s;

	/* Get date string for last change date */
	const gulong t = obj->change_time;
	if(t > 0)
	    s = EDVDateFormatString(
		t, date_format, date_relativity
	    );
	else
	    s = "";

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, s
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *      Sets the object hard links for the specified cell on the given
 *      clist.
 */
static void EDVBrowserContentsSetCellHardLinks(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	gchar num_str[80];

	/* Get string for number of hard links */
	g_snprintf(
	    num_str, sizeof(num_str),
	    "%i",
	    obj->hard_links
	);

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, num_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the object link destination for the specified cell on the
 *	given clist.
 */
static void EDVBrowserContentsSetCellLinkedTo(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
)
{
	const gchar	*path = (obj->full_path != NULL) ?
			    obj->full_path : "",
			*link_value = (obj->link_target != NULL) ?
			    obj->link_target : "";
	GdkBitmap	*mask = NULL,
			*mask_ext = NULL,
			*mask_hid = NULL;
	GdkPixmap	*pixmap = NULL,
			*pixmap_ext = NULL,
			*pixmap_hid = NULL;
	GtkStyle	*style = NULL,
			**cell_styles = browser->cell_style;

	/* Begin checking for which GtkStyle to be used */

	/* Dangling link?
	 *
	 * Note that we use the no access style for the link
	 * destination of a dangling link. The actual dangling link
	 * style is only used for the name of a dangling link
	 */
	if(!EDV_OBJECT_IS_LINK_VALID(obj))
	{
	    if(style == NULL)
		style = cell_styles[EDV_BROWSER_CELL_STYLE_NO_ACCESS];
	}
	if(EDV_OBJECT_IS_LINK_TAR_GRAND_PARENT(obj))
	{
	    if(style == NULL)
		style = cell_styles[EDV_BROWSER_CELL_STYLE_RECURSIVE_LINK];
	}

	/* Get link value */
	link_value = (obj->link_target != NULL) ? obj->link_target : "";
	if(EDV_OBJECT_IS_LINK_VALID(obj))
	{
	    struct stat lstat_buf;
	    gchar	*parent = g_dirname(path),
			*dest_path = EDVEvaluatePath(parent, link_value);

	    if((dest_path != NULL) ?
		!lstat((const char *)dest_path, &lstat_buf) : FALSE
	    )
	    {
		edv_object_struct *dest_obj = EDVObjectNew();
		EDVObjectSetPath(dest_obj, dest_path);
		EDVObjectSetStat(dest_obj, &lstat_buf);
		EDVObjectUpdateLinkFlags(dest_obj);

		/* Get icon for the link destination */
		EDVMatchObjectIcon(
		    core->device, core->total_devices,
		    core->mimetype, core->total_mimetypes,
		    dest_obj->type,
		    dest_obj->full_path,
		    EDV_OBJECT_IS_LINK_VALID(dest_obj),
		    dest_obj->permissions,
		    0,				/* Small icons */
		    &pixmap, &mask,
		    NULL, NULL,
		    &pixmap_ext, &mask_ext,
		    &pixmap_hid, &mask_hid
		);
		/* Checking if an alternate GtkStyle and icon should
		 * be used based on certain object types
		 */
		switch(dest_obj->type)
		{
		  case EDV_OBJECT_TYPE_UNKNOWN:
		    break;
		  case EDV_OBJECT_TYPE_FILE:
		    /* Hidden */
		    if(EDVIsObjectHidden(obj))
		    {
			if(pixmap_hid != NULL)
			{
			    pixmap = pixmap_hid;
			    mask = mask_hid;
			}
		    }
		    break;
		  case EDV_OBJECT_TYPE_DIRECTORY:
		    /* Hidden */
		    if(EDVIsObjectHidden(obj))
		    {
			if(pixmap_hid != NULL)
			{
			    pixmap = pixmap_hid;
			    mask = mask_hid;
			}
		    }
		    /* Special directory notations? */
		    if(!STRISEMPTY(dest_obj->name) ?
			!strcmp(dest_obj->name, "..") : FALSE
		    )
		    {
			if(browser->folder_parent_pixmap != NULL)
			{
			    pixmap = browser->folder_parent_pixmap;
			    mask = browser->folder_parent_mask;
			}
		    }
#if 0
		    /* Home directory? */
		    if((!STRISEMPTY(dest_obj->full_path) &&
			!STRISEMPTY(core->home_dir)) ?
			!strcmp(dest_obj->full_path, core->home_dir) : FALSE
		       )
		    {
			if(browser->folder_home_pixmap != NULL)
			{
			    pixmap = browser->folder_home_pixmap;
			    mask = browser->folder_home_mask;
			}
		    }
#endif
		    /* Directory not accessable? */
		    if(!EDVIsObjectAccessable(core, dest_obj))
		    {
			if(pixmap_ext != NULL)
			{
			    pixmap = pixmap_ext;
			    mask = mask_ext;
			}
			if(style == NULL)
			    style = cell_styles[
				EDV_BROWSER_CELL_STYLE_NO_ACCESS
			    ];
		    }
		    break;
		  case EDV_OBJECT_TYPE_LINK:
		    if(EDV_OBJECT_IS_LINK_TAR_GRAND_PARENT(dest_obj))
		    {
			if(style == NULL)
			    style = cell_styles[
				EDV_BROWSER_CELL_STYLE_RECURSIVE_LINK
			    ];
		    }
		    break;
		  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
		    /* Hidden */
		    if(EDVIsObjectHidden(obj))
		    {
			if(pixmap_hid != NULL)
			{
			    pixmap = pixmap_hid;
			    mask = mask_hid;
			}
		    }
		    break;
		  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
		    /* Hidden */
		    if(EDVIsObjectHidden(obj))
		    {
			if(pixmap_hid != NULL)
			{
			    pixmap = pixmap_hid;
			    mask = mask_hid;
			}
		    }
		    break;
		  case EDV_OBJECT_TYPE_FIFO:
		    /* Hidden */
		    if(EDVIsObjectHidden(obj))
		    {
			if(pixmap_hid != NULL)
			{
			    pixmap = pixmap_hid;
			    mask = mask_hid;
			}
		    }
		    break;
		  case EDV_OBJECT_TYPE_SOCKET:
		    /* Hidden */
		    if(EDVIsObjectHidden(obj))
		    {
			if(pixmap_hid != NULL)
			{
			    pixmap = pixmap_hid;
			    mask = mask_hid;
			}
		    }
		    break;
		  case EDV_OBJECT_TYPE_ERROR:
		    break;
		}

		EDVObjectDelete(dest_obj);
	    }
	    g_free(dest_path);
	    g_free(parent);
	}

	/* Set cell */
	gtk_clist_freeze(clist);
	if(pixmap != NULL)
	    gtk_clist_set_pixtext(
		clist, row, column,
		link_value,
		EDV_LIST_PIXMAP_TEXT_SPACING,
		pixmap, mask
	    );
	else
	    gtk_clist_set_text(
		clist, row, column, link_value
	    );
	gtk_clist_set_cell_style(clist, row, column, style);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the object's device index or the specified cell on
 *	the given clist.
 */
static void EDVBrowserContentsSetCellDevice(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	gchar num_str[80];

	/* Get string for device */
	g_snprintf(
	    num_str, sizeof(num_str),
	    "%ld",
	    obj->device_index
	);

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, num_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the object's index for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellIndex(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	gchar num_str[80];

	g_snprintf(
	    num_str, sizeof(num_str),
	    "%ld",
	    obj->index
	);

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, num_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *      Sets the object device type for the specified cell on the given
 *	clist.
 */
static void EDVBrowserContentsSetCellDeviceType(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	const edv_object_type type = obj->type;
	gint major, minor;
	gchar num_str[2 * 80];

	/* Is object a block or character device? */
	if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
	   (type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
	{
	    EDVGetDeviceNumbers(
		(dev_t)obj->device_type, &major, &minor
	    );
	    g_snprintf(
		num_str, sizeof(num_str),
		"%i, %i",
		major, minor
	    );
	}
	else
	{
	    *num_str = '\0';
	}

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, num_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *      Sets the object block size for the specified cell on the given
 *	clist.
 */
static void EDVBrowserContentsSetCellBlockSize(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	gchar num_str[80];

	/* Get string for block size */
	g_snprintf(
	    num_str, sizeof(num_str),
	    "%ld",
	    obj->block_size
	);

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, num_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *      Sets the object blocks for the specified cell on the given clist.
 */
static void EDVBrowserContentsSetCellBlocks(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	gchar num_str[80];

	/* Get string for number of blocks */
	g_snprintf(
	    num_str, sizeof(num_str),
	    "%ld",
	    obj->blocks
	);

	/* Set cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, num_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the Contents List's row to the values of the object.
 *
 *	The clist specifies the Contents List.
 *
 *	The obj specifies the object.
 *
 *	The row specifies the row.
 *
 *	All inputs assumed valid.
 */
static void EDVBrowserContentsSetRow(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist, edv_object_struct *obj,
	const gint row
)
{
	gint i;
	gulong block_size;
	gboolean hide_dir_size, hide_link_size, hide_link_permissions;
	const gchar *date_format;
	GList *glist;
	edv_size_format size_format;
	edv_date_relativity date_relativity;
	edv_browser_column_type column_type;
	cfg_intlist_struct *column_types_intlist;
	const cfg_item_struct *cfg_list = core->cfg_list;

	/* Get the column types mapping */
	column_types_intlist = EDV_GET_INTLIST(
	    EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
	    return;

	/* Get the display options */
	size_format = (edv_size_format)EDV_GET_I(
	    EDV_CFG_PARM_SIZE_FORMAT
	);
	block_size = EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE);
	date_relativity = (edv_date_relativity)EDV_GET_I(
	    EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);
	hide_dir_size = EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_HIDE_DIR_SIZE);
	hide_link_size = EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_HIDE_LINK_SIZE);
	hide_link_permissions = EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_HIDE_LINK_PERMISSIONS);

	gtk_clist_freeze(clist);

	/* Iterate through each column */
	for(i = 0, glist = column_types_intlist->list;
	    glist != NULL;
	    i++, glist = g_list_next(glist)
	)
	{
	    column_type = (edv_browser_column_type)glist->data;
	    switch(column_type)
	    {
	      case EDV_BROWSER_COLUMN_TYPE_NAME:
		EDVBrowserContentsSetCellName(
		    core, browser, clist, obj,
		    row, i
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_SIZE:
		EDVBrowserContentsSetCellSize(
		    core, browser, clist, obj,
		    row, i, hide_dir_size, hide_link_size,
		    size_format, block_size
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_TYPE:
		EDVBrowserContentsSetCellType(
		    core, browser, clist, obj,
		    row, i
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_PERMISSIONS:
		EDVBrowserContentsSetCellPermissions(
		    core, browser, clist, obj,
		    row, i, hide_link_permissions
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_OWNER:
		EDVBrowserContentsSetCellOwner(
		    core, browser, clist, obj,
		    row, i
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_GROUP:
		EDVBrowserContentsSetCellGroup(
		    core, browser, clist, obj,
		    row, i
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_DATE_ACCESS:
		EDVBrowserContentsSetCellDateAccess(
		    core, browser, clist, obj,
		    row, i, date_relativity, date_format
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_DATE_MODIFIED:
		EDVBrowserContentsSetCellDateModified(
		    core, browser, clist, obj,
		    row, i, date_relativity, date_format
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_DATE_CHANGED:
		EDVBrowserContentsSetCellDateChanged(
		    core, browser, clist, obj,
		    row, i, date_relativity, date_format
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_HARD_LINKS:
		EDVBrowserContentsSetCellHardLinks(
		    core, browser, clist, obj,
		    row, i
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_LINKED_TO:
		EDVBrowserContentsSetCellLinkedTo(
		    core, browser, clist, obj,
		    row, i
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_DEVICE_INDEX:
		EDVBrowserContentsSetCellDevice(
		    core, browser, clist, obj,
		    row, i
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_INDEX:
		EDVBrowserContentsSetCellIndex(
		    core, browser, clist, obj,
		    row, i
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_DEVICE_TYPE:
		EDVBrowserContentsSetCellDeviceType(
		    core, browser, clist, obj,
		    row, i
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_BLOCK_SIZE:
		EDVBrowserContentsSetCellBlockSize(
		    core, browser, clist, obj,
		    row, i
		);
		break;
	      case EDV_BROWSER_COLUMN_TYPE_BLOCKS:
		EDVBrowserContentsSetCellBlocks(
		    core, browser, clist, obj,
		    row, i
		);
		break;
	    }
	}

	gtk_clist_thaw(clist);
}

/*
 *	Appends the object to the Contents List.
 *
 *	The clist specifies the Contents List.
 *
 *	The obj specifies the object who's values will be used to
 *	append a new row on the Contents List and will be transfered
 *	to the Content List's row data. The obj should not be
 *	referenced again after this call.
 *
 *	Returns the new row index or -1 on error.
 *
 *	All inputs assumed valid.
 */
static gint EDVBrowserContentsAppendObject(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist,
	edv_object_struct *obj
)
{
	const gint ncolumns = MAX(clist->columns, 1);
	gint i, new_row;
	gchar **strv;

	gtk_clist_freeze(clist);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	/* Append the new row */
	new_row = gtk_clist_append(clist, strv);

	/* Delete the row cell values */
	g_free(strv);

	/* Failed to append row? */
	if(new_row < 0)
	{
	    gtk_clist_thaw(clist);
	    EDVObjectDelete(obj);
	    return(-1);
	}

	/* Set the new row's values */
	EDVBrowserContentsSetRow(
	    core, browser, clist,
	    obj,
	    new_row
	);

	/* Set this object as the new row's data */
	gtk_clist_set_row_data_full(
	    clist, new_row,
	    obj, EDVBrowserContentsItemDestroyCB
	);

	gtk_clist_thaw(clist);

	return(new_row);
}

/*
 *	Appends the objects from a specified location to the Contents
 *	List.
 *
 *	The clist specifies the Contents List.
 *
 *	The path specifies the full path to the location.
 *
 *	If update_status_bar is TRUE then the status bar will be
 *	updated during this operation.
 *
 *	All inputs assumed valid.
 */
static void EDVBrowserContentsAppendListing(
	edv_core_struct *core, edv_browser_struct *browser,
	GtkCList *clist,
	const gchar *path,
	const gboolean update_status_bar
)
{
#ifdef HAVE_REGEX
	regex_t *regex_filter;
#endif
	gboolean hide_object_hidden, hide_object_noaccess;
	const gchar *filter = browser->contents_list_filter;
	gint nobjs;
	gchar **names_list;
	const cfg_item_struct *cfg_list = core->cfg_list;
	edv_status_bar_struct *sb = browser->status_bar;

	hide_object_hidden = !EDV_GET_B(
	    EDV_CFG_PARM_BROWSER_SHOW_OBJECT_HIDDEN
	);
	hide_object_noaccess = !EDV_GET_B(
	    EDV_CFG_PARM_BROWSER_SHOW_OBJECT_NOACCESS
	);

#if defined(HAVE_REGEX)
	/* Compile the regex search criteria */
	if(STRISEMPTY(filter) ?
	    FALSE : strcmp(filter, "*")
	)
	{
	    regex_filter = (regex_t *)g_malloc(sizeof(regex_t));
	    if(regcomp(
		regex_filter,
		filter,
#ifdef REG_EXTENDED
		REG_EXTENDED |		/* Use POSIX extended regex */
#endif
		REG_NOSUB		/* Do not report subpattern matches */
	    ))
	    {
		g_free(regex_filter);
		regex_filter = NULL;
	    }
	}
	else
	{
	    regex_filter = NULL;
	}
#else
	if(STRISEMPTY(filter) ?
	    TRUE : !strcmp(filter, "*")
	)
	    filter = NULL;
#endif

	/* Get directory entries from the location specified by path */
	names_list = GetDirEntNames2(path, &nobjs);
	if(names_list != NULL)
	{
	    /* Sort the directory contents alphabetically by name */
	    names_list = StringQSort(names_list, nobjs);
	    if(names_list != NULL)
	    {
		struct stat stat_buf;
		gint	i,
			nobjs_loaded = 0, last_progress_percent = -1;
		const gchar *name;
		gchar *full_path = NULL;
		edv_object_struct *obj;
#define UPDATE_PROGRESS	{				\
 if(update_status_bar && (nobjs > 0)) {			\
  const gint progress_percent = nobjs_loaded * 100 / nobjs; \
  if(progress_percent > last_progress_percent) {	\
   EDVStatusBarProgress(				\
    sb,							\
    (gfloat)progress_percent / 100.0f,			\
    TRUE						\
   );							\
   last_progress_percent = progress_percent;		\
  }							\
 }							\
}
		/* Iterate through the directory contents and pick
		 * out just the directories for this first iteration
		 */
		for(i = 0; i < nobjs; i++)
		{
#define FREE_AND_CONTINUE	{	\
 g_free(full_path);			\
 full_path = NULL;			\
					\
 /* Do not delete names_list[i] on this	\
  * first iteration			\
  */					\
					\
 continue;				\
}

		    name = names_list[i];
		    if(name == NULL)
			FREE_AND_CONTINUE

		    /* Skip special dir notations */
		    if(!strcmp(name, "."))
		    {
			nobjs_loaded++;
			UPDATE_PROGRESS
			FREE_AND_CONTINUE
		    }

		    /* Complete a full path to this object */
		    full_path = STRDUP(PrefixPaths(path, name));
		    if(full_path == NULL)
		    {
			g_free(names_list[i]);
			names_list[i] = NULL;
			FREE_AND_CONTINUE
		    }

		    /* Get the destination object's stats */
		    if(stat((const char *)full_path, &stat_buf))
		    {
			/* Failed to get the statistics of this
			 * object but do not add it as an error
			 * object on this directory iteration, it
			 * will be added later in the second
			 * iteration
			 */
			FREE_AND_CONTINUE
		    }

#ifdef S_ISDIR
		    /* Skip if the destination is not a directory */
		    if(!S_ISDIR(stat_buf.st_mode))
			FREE_AND_CONTINUE
#endif

		    /* Get this directory's local stats */
		    if(lstat((const char *)full_path, &stat_buf))
		    {
			/* Unable to get the directory's local stats
			 * so add it as an error object
			 */
			const gint error_code = (gint)errno;
			obj = EDVBrowserNewErrorObject(full_path, g_strerror(error_code));
			if(obj != NULL)
			    EDVBrowserContentsAppendObject(
				core, browser, clist,
				obj
			    );
			nobjs_loaded++;
			UPDATE_PROGRESS
			g_free(names_list[i]);
			names_list[i] = NULL;
			FREE_AND_CONTINUE
		    }

		    /* Create a new object, set its values from the
		     * stats, and append it with a new row on the
		     * Contents List
		     */
		    obj = EDVObjectNew();
		    if(obj != NULL)
		    {
			EDVObjectSetPath(obj, full_path);
			EDVObjectSetStat(obj, &stat_buf);
			EDVObjectUpdateLinkFlags(obj);

			/* Skip hidden or no access objects? */
			if((hide_object_hidden ?
			       EDVIsObjectHidden(obj) : FALSE) ||
			   (hide_object_noaccess ?
			       !EDVIsObjectAccessable(core, obj) : FALSE)
			)
			{
			    EDVObjectDelete(obj);
			}
			else
			{
			    /* Append/transfer this object to the listing */
			    EDVBrowserContentsAppendObject(
				core, browser, clist,
				obj
			    );
			}
		    }

		    nobjs_loaded++;
		    UPDATE_PROGRESS
		    g_free(names_list[i]);
		    names_list[i] = NULL;
		    FREE_AND_CONTINUE
#undef FREE_AND_CONTINUE
		}

		/* Now iterate through all other objects, skipping
		 * directories
		 *
		 * This iteration will delete all strings in names_list
		 *
		 * Do not reset last_progress_percent, leave it as the
		 * last value from the previous loop
		 */
		full_path = NULL;
		for(i = 0; i < nobjs; i++)
		{
#define FREE_AND_CONTINUE	{	\
 g_free(full_path);			\
 full_path = NULL;			\
					\
 g_free(names_list[i]);			\
 names_list[i] = NULL;			\
					\
 continue;				\
}
		    name = names_list[i];
		    if(name == NULL)
			FREE_AND_CONTINUE

		    /* Skip special dir notations */
		    if(!strcmp(name, "."))
			FREE_AND_CONTINUE

		    /* Filter check */
#if defined(HAVE_REGEX)
		    if(regex_filter != NULL)
		    {
			if(regexec(
			    regex_filter,
			    name,
			    0, NULL,
			    0
			) == REG_NOMATCH)
			{
			    nobjs_loaded++;
			    UPDATE_PROGRESS
			    FREE_AND_CONTINUE
			}
		    }
#else
		    if(filter != NULL)
		    {
			if(fnmatch(filter, name, 0) == FNM_NOMATCH)
			{
			    nobjs_loaded++;
			    UPDATE_PROGRESS
			    FREE_AND_CONTINUE
			}
		    }
#endif

		    /* Complete a full path to this object */
		    full_path = STRDUP(PrefixPaths(path, name));
		    if(full_path == NULL)
			FREE_AND_CONTINUE

		    /* Get the object's local stats */
		    if(lstat((const char *)full_path, &stat_buf))
		    {
			/* Unable to get the object's local stats
			 * so add it as an error object
			 */
			const gint error_code = (gint)errno;
			obj = EDVBrowserNewErrorObject(full_path, g_strerror(error_code));
			if(obj != NULL)
			    EDVBrowserContentsAppendObject(
				core, browser, clist,
				obj                  
			    );
			nobjs_loaded++;
			UPDATE_PROGRESS
			FREE_AND_CONTINUE
		    }

		    /* Create a new object, set its values from the
		     * stats, and append it with a new row on the
		     * Contents List
		     */
		    obj = EDVObjectNew();
		    if(obj != NULL)
		    {
			EDVObjectSetPath(obj, full_path);
			EDVObjectSetStat(obj, &stat_buf);
			EDVObjectUpdateLinkFlags(obj);

			/* Skip hidden objects */
			if((hide_object_hidden ?
			       EDVIsObjectHidden(obj) : FALSE)
/*                         ||
			   (hide_object_noaccess ?
			        !EDVIsObjectAccessable(core, obj) : FALSE)
 */
			)
			{
			    EDVObjectDelete(obj);
			}
			else
			{
			    /* Append/transfer this object to the listing */
			    EDVBrowserContentsAppendObject(
				core, browser, clist,
				obj
			    );
			}
		    }

		    nobjs_loaded++;
		    UPDATE_PROGRESS
		    FREE_AND_CONTINUE
#undef FREE_AND_CONTINUE
		}

		/* At this point all strings in names_list have been deleted */
#undef UPDATE_PROGRESS
	    }

	    /* Delete the string pointer array, each string should
	     * already be deleted
	     */
	    g_free(names_list);
	}

#ifdef HAVE_REGEX
	if(regex_filter != NULL)
	{
	    regfree(regex_filter);
	    g_free(regex_filter);
	}
#endif
}


/* 
 *	Gets the selected list of paths.
 *
 *	Returns a GList of gchar * strings describing the full paths
 *	of the selected objects or NULL on error.
 */
GList *EDVBrowserContentsGetSelectedPaths(edv_browser_struct *browser)
{
	GList *glist, *paths_list;
	GtkCList *clist;
	edv_object_struct *obj;

	if(browser == NULL)
	    return(NULL);

	clist = GTK_CLIST(browser->contents_clist);

	/* Create the selected paths list */
	paths_list = NULL;
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    obj = EDV_OBJECT(gtk_clist_get_row_data(
		clist, (gint)glist->data
	    ));
	    if(obj == NULL)
		continue;

	    if(STRISEMPTY(obj->full_path))
		continue;

	    paths_list = g_list_append(
		paths_list,
		STRDUP(obj->full_path)
	    );
	}

	return(paths_list);
}


/*
 *	Finds the row who's object matches the location index.
 *
 *	The device_index specifies the index of the device that the
 *	object resides on.
 *
 *	The index specifies the index on the device that the object
 *	resides at.
 *
 *	Returns the matched row or negative on error.
 */
gint EDVBrowserContentsFindRowByIndex(
	edv_browser_struct *browser, 
	const gulong device_index, const gulong index
)
{
	gint i, n;
	GtkCList *clist;
	edv_object_struct *obj;

	if(browser == NULL)
	    return(-2);

	clist = GTK_CLIST(browser->contents_clist);

	n = clist->rows;
	for(i = 0; i < n; i++)
	{
	    obj = EDV_OBJECT(gtk_clist_get_row_data(clist, i));
	    if(obj == NULL)
		continue;

	    if((obj->device_index == device_index) &&
	       (obj->index == index)
	    )
		return(i);
	}

	return(-1);
}

/*
 *	Finds the row who's object matches the path.
 *
 *	The path specifies the full path of the object.
 *
 *	Returns the matched row or negative on error.
 */
gint EDVBrowserContentsFindRowByPath(
	edv_browser_struct *browser, const gchar *path
)
{
	gint i, n;
	GtkCList *clist;
	edv_object_struct *obj;

	if((browser == NULL) || STRISEMPTY(path))
	    return(-2);

	clist = GTK_CLIST(browser->contents_clist);

	n = clist->rows;
	for(i = 0; i < n; i++)
	{
	    obj = EDV_OBJECT(gtk_clist_get_row_data(clist, i));
	    if(obj == NULL)
		continue;

	    if(obj->full_path == NULL)
		continue;

	    /* Full paths match? */
	    if(!strcmp((const char *)obj->full_path, (const char *)path))
		return(i);
	}

	return(-1);
}

/*
 *	Updates all rows on the Contents List with each row's
 *	associated object.
 *
 *	This should be called when MIME Types or Devices have been
 *	added/modified/removed or after reconfiguration.
 *
 *	This function should not be used to get a new listing, use
 *	EDVBrowserContentsGetListing() instead.
 */
void EDVBrowserContentsRealizeListing(edv_browser_struct *browser)
{
	gint i, n;
	GtkCList *clist;
	edv_object_struct *obj;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;

	gtk_clist_freeze(clist);

	/* Reset the columns */
	EDVBrowserContentsResetColumns(core, browser, clist);

	/* Update the rows */
	n = clist->rows;
	for(i = 0; i < n; i++)
	{
	    obj = EDV_OBJECT(gtk_clist_get_row_data(clist, i));
	    if(obj == NULL)
		continue;

	    EDVBrowserContentsSetRow(
		core, browser, clist,
		obj,
		i			/* Row */
	    );
	}

	gtk_clist_thaw(clist);
}


/*
 *	Clears the Contents List, resets the columns, and gets a new
 *	listing of objects.
 *
 *	The path specifies the full path to the location.
 *
 *	If update_status_bar is TRUE then the status bar will be updated
 *	during this operation.
 */
void EDVBrowserContentsGetListing(
	edv_browser_struct *browser, const gchar *path,
	const gboolean update_status_bar
)
{
	GtkWidget *w;
	GtkCList *clist;
	edv_status_bar_struct *sb;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	w = browser->contents_clist;
	clist = GTK_CLIST(w);
	sb = browser->status_bar;
	core = browser->core;

	/* Report the initial progress? */
	if(update_status_bar)
	{
	    EDVStatusBarMessage(
		sb,
		"Loading directory contents...",
		FALSE
	    );
	    EDVStatusBarProgress(sb, 0.0f, TRUE);
	}

	gtk_clist_freeze(clist);

	/* Clear the listing */
	EDVBrowserContentsClear(browser);

	/* Reset the columns */
	EDVBrowserContentsResetColumns(core, browser, clist);

	/* Append the list of objects to the listing */
	if(!STRISEMPTY(path))
	    EDVBrowserContentsAppendListing(
		core, browser, clist,
		path,				/* Location */
		update_status_bar
	    );

	gtk_clist_thaw(clist);

	/* Report the final progress? */
	if(update_status_bar)
	{
	    EDVStatusBarMessage(
		sb,
		"Directory contents loaded",
		FALSE
	    );
	    EDVStatusBarProgress(sb, 0.0f, TRUE);
	}
}

/*
 *	Clears the Contents List.
 */
void EDVBrowserContentsClear(edv_browser_struct *browser)
{
	GtkCList *clist;

	if(browser == NULL)
	    return;

	clist = GTK_CLIST(browser->contents_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);
}


/*
 *	Opens the object.
 *
 *	The row and column specify the object displayed on the
 *	Contents List. If row and column are -1 then the selected
 *	object(s) will be used instead.
 *
 *	The state specifies the current modifier key states.
 */
void EDVBrowserContentsOpen(
	edv_browser_struct *browser,
	const gint row, const gint column,
	const guint state
)
{
	GList *paths_list;
	GtkWidget *toplevel;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	toplevel = browser->toplevel;
	core = browser->core;

	/* Use selected object(s)? */
	if(row < 0)
	{
	    paths_list = EDVBrowserContentsGetSelectedPaths(browser);
	}
	else
	{
	    GtkCList *clist = GTK_CLIST(browser->contents_clist);
	    edv_object_struct *obj = EDV_OBJECT(
		gtk_clist_get_row_data(clist, row)
	    );
	    if(obj == NULL)
		return;

	    if(STRISEMPTY(obj->full_path))
		return;

	    paths_list = NULL;
	    paths_list = g_list_append( 
		paths_list,
		STRDUP(obj->full_path)
	    );
	}

	/* Check if there is only one path and it refers to a directory */
	if(g_list_length(paths_list) == 1)
	{
	    struct stat stat_buf;
	    gchar *path = EDVEvaluatePath(
		NULL, (gchar *)paths_list->data
	    );
	    if((path != NULL) ? stat((const char *)path, &stat_buf) : TRUE)
	    {
		g_list_foreach(paths_list, (GFunc)g_free, NULL);
		g_list_free(paths_list);
		g_free(path);
		return;
	    }
#ifdef S_ISDIR
	    if(S_ISDIR(stat_buf.st_mode))
	    {
		/* Change directory and return */
		EDVBrowserGotoDirectoryCB(browser, path);

		g_list_foreach(paths_list, (GFunc)g_free, NULL);
		g_list_free(paths_list);
		g_free(path);

		return;
	    }
#endif
	}

	if(paths_list != NULL)
	{
	    gchar	*stdout_path_rtn = NULL,
			*stderr_path_rtn = NULL;
	    const gchar *command_name = NULL;

	    if(state & GDK_CONTROL_MASK)
		command_name = "edit";
	    else if(state & GDK_SHIFT_MASK)
		command_name = "edit";

	    EDVBrowserSetBusy(browser, TRUE);
	    EDVOpen(
		core,
		paths_list,		/* Path */
		command_name,		/* Command Name */
		toplevel,		/* Toplevel */
		TRUE,			/* Verbose */
		&stdout_path_rtn,
		&stderr_path_rtn
	    );
	    EDVBrowserSetBusy(browser, FALSE);

	    g_free(stdout_path_rtn);
	    g_free(stderr_path_rtn);

	    g_list_foreach(paths_list, (GFunc)g_free, NULL);
	    g_list_free(paths_list);
	}
}

/*
 *	Maps the Open With list to open the object.
 *
 *	The row and column specify the object displayed on the
 *	Contents List. If row and column are -1 then the selected
 *	object(s) will be used instead.
 */
void EDVBrowserContentsOpenWith(
	edv_browser_struct *browser,
	const gint row, const gint column
)
{
	GList *paths_list;
	GtkWidget *toplevel;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	toplevel = browser->toplevel;
	core = browser->core;

	/* Use selected object paths? */
	if(row < 0)
	{
	    /* Get list of selected object paths */
	    paths_list = EDVBrowserContentsGetSelectedPaths(browser);
	}
	else
	{   
	    GtkCList *clist = GTK_CLIST(browser->contents_clist);
	    edv_object_struct *obj = EDV_OBJECT(
		gtk_clist_get_row_data(clist, row)
	    );
	    if(obj == NULL)
		return;

	    if(STRISEMPTY(obj->full_path))
		return;

	    paths_list = NULL;
	    paths_list = g_list_append(
		paths_list,
		STRDUP(obj->full_path)
	    );
	}

	if(paths_list != NULL)
	{
	    gchar       *stdout_path_rtn = NULL,
			*stderr_path_rtn = NULL;

	    EDVOpenWith(
		core,
		paths_list,		/* Paths List */
		NULL,			/* Command Name */
		toplevel,		/* Toplevel */
		TRUE,			/* Verbose */
		&stdout_path_rtn,
		&stderr_path_rtn
	    );

	    g_free(stdout_path_rtn);
	    g_free(stderr_path_rtn);

	    g_list_foreach(paths_list, (GFunc)g_free, NULL);
	    g_list_free(paths_list);
	}
}


/*
 *	FPrompt rename apply callback.
 */
static void EDVBrowserContentsFPromptRenameApplyCB(
	gpointer data, const gchar *value
)
{
	gint row;
	GtkCList *clist;
	edv_status_bar_struct *sb;
	edv_browser_struct *browser;
	EDVBrowserContentsListFPromptData *d = EDV_BROWSER_CONTENTS_LIST_FPROMPT_DATA(data);
	if(d == NULL)
	    return;

	browser = d->browser;
	clist = GTK_CLIST(d->clist);
	row = d->row;
	sb = browser->status_bar;

	/* Inputs valid? */
	if((browser != NULL) && (clist != NULL) && (row >= 0) &&
	   (value != NULL)
	)
	{
	    GtkWidget *toplevel = browser->toplevel;
	    edv_core_struct *core = browser->core;
	    edv_object_struct *obj = EDV_OBJECT(
		gtk_clist_get_row_data(clist, row)
	    );

	    /* Check if the selected object's disk object structure
	     * is valid
	     */
	    if((obj != NULL) ? (obj->full_path != NULL) : FALSE)
	    {
		gboolean yes_to_all = FALSE;
		const gchar *error_msg;
		gchar *old_full_path = STRDUP(obj->full_path);
		GList *modified_paths_list;

		/* Rename */
		EDVObjectOPRename(
		    core,
		    old_full_path, value,
		    &modified_paths_list,
		    toplevel,
		    FALSE,			/* Do not show progress */
		    TRUE,			/* Interactive */
		    &yes_to_all
		);

		/* Unmap the progress dialog */
		ProgressDialogBreakQuery(FALSE);
		ProgressDialogSetTransientFor(NULL);

		/* Check for errors */
		error_msg = EDVObjectOPGetError(core);
		if(!STRISEMPTY(error_msg))
		{
		    /* Report the error */
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Rename Error",
			error_msg,
			NULL,
			toplevel
		    );
		}

		/* Report the modified objects */
		if(modified_paths_list != NULL)
		{
		    struct stat lstat_buf;
		    const gchar *modified_path;
		    GList *glist;

		    for(glist = modified_paths_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			modified_path = (const gchar *)glist->data;
			if(modified_path == NULL)
			    continue;

			if(!lstat((const char *)modified_path, &lstat_buf))
			{
			    gchar *msg = g_strdup_printf(
				"Object \"%s\" renamed to \"%s\"",
				g_basename(old_full_path),
				g_basename(modified_path)
			    );
			    EDVStatusBarMessage(sb, msg, FALSE);
			    g_free(msg);

			    EDVObjectModifiedEmit(
				core,
				old_full_path,
				modified_path,
				&lstat_buf
			    );
			}
		    }

		    if(modified_paths_list != NULL)
		    {
			g_list_foreach(
			    modified_paths_list, (GFunc)g_free, NULL
			);
			g_list_free(modified_paths_list);
		    }
		}
		else
		{   
		    /* Did not get the modified object path so this
		     * implies failure
		     */
		    EDVStatusBarMessage(
			sb,
			"Rename object failed",
			FALSE
		    );
		}

		g_free(old_full_path);
	    }
	}

	g_free(d);
}

/*
 *	FPrompt rename cancel callback.
 */
static void EDVBrowserContentsFPromptRenameCancelCB(gpointer data)
{
	EDVBrowserContentsListFPromptData *d = EDV_BROWSER_CONTENTS_LIST_FPROMPT_DATA(data);
	if(d == NULL)
	    return;

	g_free(d);
}

/*
 *	Prompts to rename the object.
 *
 *	The row specifies the row index.
 *
 *	The column specifies which column to rename. If column is
 *	-1 then it implies any column is valid.
 */
void EDVBrowserContentsPromptRename(
	edv_browser_struct *browser, const gint row, const gint column
)
{
	gint cx, cy, px, py, pwidth, pheight, name_column;
	GtkWidget *toplevel;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	cfg_intlist_struct *column_type_intlist;
	edv_object_struct *obj;
	edv_core_struct *core;

	if((browser == NULL) || FPromptIsQuery())
	    return;

	toplevel = browser->toplevel;
	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	    return;

	EDVBrowserSyncData(browser);

	/* Row does not exist? */
	if((row < 0) || (row >= clist->rows))
	    return;

	/* Find which column is displaying the name */
	name_column = -1;
	column_type_intlist = EDV_GET_INTLIST(
	    EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN
	);
	if(column_type_intlist != NULL)
	{
	    gint i = 0;
	    GList *glist;

	    for(glist = column_type_intlist->list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		if((edv_browser_column_type)glist->data ==
		    EDV_BROWSER_COLUMN_TYPE_NAME
		)
		{
		    name_column = i;
		    break;
		}
		i++;
	    }
	}
	/* No column displaying the name? */
	if(name_column < 0)
	    return;

	/* Get the cell's geometry */
	if(!gtk_clist_get_cell_geometry(
	    clist, name_column, row,
	    &cx, &cy, &pwidth, &pheight
	))
	    return;

	/* Get the root window relative coordinates */
	px = 0;
	py = 0;
	gdk_window_get_deskrelative_origin(
	    clist->clist_window, &px, &py
	);
	px += cx + 0;
	py += cy - 2;		/* Move up a little */

	/* Get the object */
	obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));
	if(obj == NULL)
	    return;

	/* Check if the object's name is a special notation that
	 * may not be renamed
	 */
	if(!STRISEMPTY(obj->name))
	{
	    const gchar *name = obj->name;
	    if(!strcmp(name, ".") || !strcmp(name, "..") ||
	       !strcmp(name, "/")
	    )
		return;
	}

	/* Is the specified column the name column or -1? */
	if((column == name_column) || (column < 0))
	{
	    gchar *value = STRDUP(obj->name);
	    EDVBrowserContentsListFPromptData *d = EDV_BROWSER_CONTENTS_LIST_FPROMPT_DATA(
		g_malloc0(sizeof(EDVBrowserContentsListFPromptData))
	    );
	    if(d != NULL)
	    {
		d->browser = browser;
		d->clist = GTK_WIDGET(clist);
		d->row = row;
		d->column = column;
	    }

	    /* Map floating prompt to change values */
	    FPromptSetTransientFor(browser->toplevel);
	    FPromptSetPosition(px, py);
	    FPromptMapQuery(
		NULL,			/* No label */
		value,
		NULL,			/* No tooltip message */
		FPROMPT_MAP_NO_MOVE,	/* Map code */
		pwidth, -1,		/* Width and height */
		0,			/* No buttons */
		d,			/* Callback data */
		NULL,			/* No browse callback */
		EDVBrowserContentsFPromptRenameApplyCB,
		EDVBrowserContentsFPromptRenameCancelCB
	    );

	    g_free(value);
	}
}


/*
 *      This should be called whenever a new object has been added, it
 *      will add a new row as needed to represent the new object.
 *
 *      The given path must be an absolute path to the object.
 */
void EDVBrowserContentsObjectAddedNotify(
	edv_browser_struct *browser, const gchar *path,
	const struct stat *lstat_buf
)
{
	gint row;
	gchar *cur_path, *parent_path;
	GtkCList *clist;
	edv_core_struct *core;

	if((browser == NULL) || STRISEMPTY(path) || (lstat_buf == NULL))
	    return;

	cur_path = STRDUP(EDVBrowserCurrentLocation(browser));
	if(cur_path == NULL)
	    return;

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;

	/* Is the current location the same as the added object? */
	if(!strcmp((const char *)cur_path, (const char *)path))
	{
	    /* Reget the listing */
	    EDVBrowserContentsGetListing(browser, path, TRUE);
	    g_free(cur_path);
	    return;
	}

	/* Update the object who's path matches the added path */
	row = EDVBrowserContentsFindRowByPath(browser, path);
	if(row > -1)
	{
	    /* Update this object */
	    edv_object_struct *obj = EDV_OBJECT(
		gtk_clist_get_row_data(clist, row)
	    );
	    if(obj != NULL)
	    {
		EDVObjectSetPath(obj, path);
		EDVObjectSetStat(obj, lstat_buf);
		EDVObjectUpdateLinkFlags(obj);

		/* Update the Contents List row with the new values
		 * from this object
		 */
		EDVBrowserContentsSetRow(
		    core, browser, clist,
		    obj,
		    row
		);
	    }
	    g_free(cur_path);
	    return;
	}

	/* Get the parent directory of the added object */
	parent_path = g_dirname(path);
	if(parent_path != NULL)
	{
	    /* Is the parent directory of the added object the same as
	     * the current location?
	     */
	    if(!strcmp((const char *)parent_path, (const char *)cur_path))
	    {
		/* Add this object to the list */
		edv_object_struct *obj = EDVObjectNew();
		if(obj != NULL)
		{
		    gint new_row;

		    EDVObjectSetPath(obj, path);
		    EDVObjectSetStat(obj, lstat_buf);
		    EDVObjectUpdateLinkFlags(obj);

		    /* Append/transfer this object to the listing */
		    new_row = EDVBrowserContentsAppendObject(
			core, browser, clist, obj
		    );
		}
	    }
	    g_free(parent_path);
	}

	g_free(cur_path);
}

/*
 *	Object modified callback.
 */
void EDVBrowserContentsObjectModifiedNotify(
	edv_browser_struct *browser, const gchar *path,
	const gchar *new_path,
	const struct stat *lstat_buf
)
{
	gint row;
	gchar *cur_path;
	GtkCList *clist;
	edv_core_struct *core;

	if((browser == NULL) || STRISEMPTY(path) || (lstat_buf == NULL))
	    return;

	if(new_path == NULL)
	    new_path = path;

	cur_path = STRDUP(EDVBrowserCurrentLocation(browser));
	if(cur_path == NULL)
	    return;

	clist = GTK_CLIST(browser->contents_clist);
	core = browser->core;

	/* Is the new modified path the same as the current location? */
	if(!strcmp((const char *)new_path, (const char *)cur_path))
	{
	    /* Reget the listing */
	    EDVBrowserContentsGetListing(browser, new_path, TRUE);
	    g_free(cur_path);
	    return;
	}

	/* Update the object who's path matches the modified path */
	row = EDVBrowserContentsFindRowByPath(browser, path);
	if(row > -1)
	{
	    /* Update this object */
	    edv_object_struct *obj = EDV_OBJECT(
		gtk_clist_get_row_data(clist, row)
	    );
	    if(obj != NULL)
	    {
		EDVObjectSetPath(obj, new_path);	/* Use the new path */
		EDVObjectSetStat(obj, lstat_buf);
		EDVObjectUpdateLinkFlags(obj);

		/* Update the Contents List row with the new values
		 * from this object
		 */
		EDVBrowserContentsSetRow(
		    core, browser, clist,
		    obj,
		    row
		);
	    }
	}

	g_free(cur_path);
}

/*
 *	Object removed callback/
 */
void EDVBrowserContentsObjectRemovedNotify(
	edv_browser_struct *browser, const gchar *path
)
{
	gchar *cur_path;
	GtkCList *clist;

	if((browser == NULL) || STRISEMPTY(path))
	    return;

	cur_path = STRDUP(EDVBrowserCurrentLocation(browser));
	if(cur_path == NULL)
	    return;

	clist = GTK_CLIST(browser->contents_clist);

	/* Is the removed path the same as the current location? */
	if(!strcmp((const char *)path, (const char *)cur_path))
	{
	    /* Clear the list */
	    EDVBrowserContentsClear(browser);
	}
	else
	{
	    /* Remove all the row who's object's path matches the
	     * removed path
	     */
	    gint row = EDVBrowserContentsFindRowByPath(browser, path);
	    if(row > -1)
	    {
		gtk_clist_freeze(clist);
		do {
		    gtk_clist_remove(clist, row);
		    row = EDVBrowserContentsFindRowByPath(browser, path);
		} while(row > -1);
		gtk_clist_thaw(clist);
	    }
	}

	g_free(cur_path);
}


/*
 *      This should be called whenever a device has been mounted or
 *	unmounted.
 */
void EDVBrowserContentsMountNotify(
	edv_browser_struct *browser, edv_device_struct *dev,
	gboolean is_mounted
)
{
	gchar *cur_path = NULL, *mount_path = NULL;
	GtkCList *clist;

	if((browser == NULL) || (dev == NULL))
	    return;

	clist = GTK_CLIST(browser->contents_clist);

	/* Get the current location and mount path */
	cur_path = STRDUP(EDVBrowserCurrentLocation(browser));
	mount_path = STRDUP(dev->mount_path);
	if((cur_path == NULL) || (mount_path == NULL))
	{
	    g_free(cur_path);
	    g_free(mount_path);
	    return;
	}

	/* Need to simplify coppies of current and mount paths */
	EDVSimplifyPath(cur_path);
	EDVSimplifyPath(mount_path);

	/* Check if mount path was the current location, if it was
	 * then the Contents List needs to be updated
	 */
	if(!strcmp((const char *)cur_path, (const char *)mount_path))
	{
	    /* Reget the listing */
	    EDVBrowserContentsGetListing(browser, cur_path, TRUE);
	}

	g_free(cur_path);
	g_free(mount_path);
}
