#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <time.h>
#include <signal.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <unistd.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "cdialog.h"
#include "progressdialog.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_archive_obj.h"
#include "edv_archive_add.h"
#include "edv_archive_add_tar.h"
#include "edv_archive_add_xar.h"
#include "edv_archive_add_zip.h"
#include "endeavour2.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "animations/file01_20x20.xpm"
#include "animations/file02_20x20.xpm"
#include "animations/file03_20x20.xpm"
#include "animations/file04_20x20.xpm"
#include "animations/file05_20x20.xpm"
#include "animations/file06_20x20.xpm"
#include "animations/folderfile_32x32.xpm"
#include "animations/package_32x32.xpm"
#include "animations/packagefile_32x32.xpm"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Error Message */
const gchar *EDVArchAddGetError(edv_core_struct *core);
void EDVArchAddCopyErrorMessage(
	edv_core_struct *core, const gchar *msg
);

/* Progress Dialog */
void EDVArchAddMapProgressDialog(
	const gchar *label, const gfloat progress,
	GtkWidget *toplevel, const gboolean force_remap
);
void EDVArchAddMapProgressDialogUnknown(
	const gchar *label, GtkWidget *toplevel,
	const gboolean force_remap
);

/* Calculate Total Size */
gulong EDVArchAddCalculateTotalSize(
	const gchar *arch_path, const gchar *path,
	gint *nobjs,
	const gboolean recursive, const gboolean dereference_links
);

/* Add To Archive */
static gint EDVArchAddARJ(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
);
static gint EDVArchAddLHA(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
);
static gint EDVArchAddRAR(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
);
gint EDVArchAdd(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define UNLINK(p)	(((p) != NULL) ? (gint)unlink((const char *)(p)) : -1)
#define INTERRUPT(i)	(((i) > 0) ? (gint)kill((int)(i), SIGINT) : -1)

#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))


static gchar *G_STRCAT(gchar *s, const gchar *s2)
{
	if(s != NULL) {
	    if(s2 != NULL) {
		gchar *sr = g_strconcat(s, s2, NULL);
		g_free(s);
		s = sr;
	    }
	} else {
	    if(s2 != NULL)
		s = STRDUP(s2);
	    else
		s = STRDUP("");
	}
	return(s);
}


/*
 *	Returns the last error message as a statically allocated
 *	string or NULL if there was no previous error.
 */
const gchar *EDVArchAddGetError(edv_core_struct *core)
{
	return((core != NULL) ? core->archive_last_error : NULL);
}


/*
 *	Coppies the error message to the core's archive_last_error_buf
 *	and sets the core's archive_last_error to point to it.
 */
void EDVArchAddCopyErrorMessage(
	edv_core_struct *core, const gchar *msg
)
{
	if(core == NULL)
	    return;

	core->archive_last_error = NULL;

	g_free(core->archive_last_error_buf);
	core->archive_last_error_buf = STRDUP(msg);

	core->archive_last_error = core->archive_last_error_buf;
}


/*
 *	Maps the progress dialog as needed in animation mode for
 *	adding.
 */
void EDVArchAddMapProgressDialog(
	const gchar *label, const gfloat progress,
	GtkWidget *toplevel, const gboolean force_remap
)
{
	guint8	**start_icon_data[3],
		**icon_data[6],
		**end_icon_data[3];

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
	    /* Check if the progress dialog needs to be unmapped and
	     * remapped again
	     */
	    if(force_remap)
	    {
		ProgressDialogBreakQuery(FALSE);
	    }
	    else
	    {
		/* Already mapped and does not need unmapping, so just
		 * update the progress message
		 */
		ProgressDialogUpdate(
		    NULL, label, NULL, NULL,
		    progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
		return;
	    }
	}

	ProgressDialogSetTransientFor(toplevel);

	start_icon_data[0] = (guint8 **)folderfile_32x32_xpm;
	start_icon_data[1] = (guint8 **)folderfile_32x32_xpm;
	start_icon_data[2] = (guint8 **)folderfile_32x32_xpm;
	icon_data[0] = (guint8 **)file01_20x20_xpm;
	icon_data[1] = (guint8 **)file02_20x20_xpm;
	icon_data[2] = (guint8 **)file03_20x20_xpm;
	icon_data[3] = (guint8 **)file04_20x20_xpm;
	icon_data[4] = (guint8 **)file05_20x20_xpm;
	icon_data[5] = (guint8 **)file06_20x20_xpm;
	end_icon_data[0] = (guint8 **)package_32x32_xpm;
	end_icon_data[1] = (guint8 **)package_32x32_xpm;
	end_icon_data[2] = (guint8 **)packagefile_32x32_xpm;

	ProgressDialogMapAnimation(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Agregar",
	    label,
	    "Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Addition",
	    label,
	    "Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Hinzufgen",
	    label,
	    "Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "L'Aggiunta",
	    label,
	    "Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Toevoegen",
	    label,
	    "Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Adicionar",
	    label,
	    "Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Tilfying",
	    label,
	    "Stans",
#else
	    "Adding",
	    label,
	    "Stop",
#endif
	    start_icon_data, 3,
	    icon_data, 6,
	    end_icon_data, 3,
	    EDV_DEF_PROGRESS_DLG_ANIM_INT,
	    EDV_DEF_PROGRESS_DLG_ANIM_INC
	);
	ProgressDialogUpdate(
	    NULL, NULL, NULL, NULL,
	    progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}

/*
 *	Maps the progress dialog as needed in animation mode for adding
 *	with an unknown progress value (activity mode).
 */
void EDVArchAddMapProgressDialogUnknown(
	const gchar *label, GtkWidget *toplevel,
	const gboolean force_remap
)
{
	guint8	**start_icon_data[3],
		**icon_data[6],
		**end_icon_data[3];

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
	    /* Check if the progress dialog needs to be unmapped and
	     * remapped again
	     */
	    if(force_remap)
	    {
		ProgressDialogBreakQuery(FALSE);
	    }
	    else
	    {
		/* Already mapped and does not need unmapping, so just
		 * update the progress message
		 */
		ProgressDialogUpdateUnknown(
		    NULL, label, NULL, NULL, TRUE
		);
		return;
	    }
	}

	ProgressDialogSetTransientFor(toplevel);

	start_icon_data[0] = (guint8 **)folderfile_32x32_xpm;
	start_icon_data[1] = (guint8 **)folderfile_32x32_xpm;
	start_icon_data[2] = (guint8 **)folderfile_32x32_xpm;
	icon_data[0] = (guint8 **)file01_20x20_xpm;
	icon_data[1] = (guint8 **)file02_20x20_xpm;
	icon_data[2] = (guint8 **)file03_20x20_xpm;
	icon_data[3] = (guint8 **)file04_20x20_xpm;
	icon_data[4] = (guint8 **)file05_20x20_xpm;
	icon_data[5] = (guint8 **)file06_20x20_xpm;
	end_icon_data[0] = (guint8 **)package_32x32_xpm;
	end_icon_data[1] = (guint8 **)package_32x32_xpm;
	end_icon_data[2] = (guint8 **)packagefile_32x32_xpm;

	ProgressDialogMapAnimation(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Agregar",
	    label,
	    "Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Addition",
	    label,
	    "Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Hinzufgen",
	    label,
	    "Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "L'Aggiunta",
	    label,
	    "Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Toevoegen",
	    label,
	    "Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Adicionar",
	    label,
	    "Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Tilfying",
	    label,
	    "Stans",
#else
	    "Adding",
	    label,
	    "Stop",
#endif
	    start_icon_data, 3,
	    icon_data, 6,
	    end_icon_data, 3,
	    EDV_DEF_PROGRESS_DLG_ANIM_INT,
	    EDV_DEF_PROGRESS_DLG_ANIM_INC
	);
	ProgressDialogUpdateUnknown(
	    NULL, NULL, NULL, NULL, TRUE
	);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}


/*
 *	Calculates the size and number of objects of the specified
 *	path.
 *
 *	If recursive is TRUE and the specified path is a directory
 *	then the sizes and number of all the objects within that
 *	directory will be counted.
 *
 *	If dereference_links is TRUE and the specified path is a link
 *	then the size of the link's target will be counted if it
 *	leads to a file. Otherwise the link's size is counted.
 *
 *	The *nobjs will be added with the number of objects
 *	counted.
 *
 *	Returns the total size of all the objects of the specified
 *	path. Only the sizes of files and links are counted.
 */
gulong EDVArchAddCalculateTotalSize(
	const gchar *arch_path, const gchar *path,
	gint *nobjs,
	const gboolean recursive, const gboolean dereference_links
)
{
	mode_t m;
	struct stat stat_buf;
	gulong subtotal = 0l;

	if(STRISEMPTY(path))
	    return(subtotal);

	/* Skip the archive itself */
	if(!strcmp((const char *)arch_path, path))
	    return(subtotal);

	if(lstat((const char *)path, &stat_buf))
	    return(subtotal);

	m = stat_buf.st_mode;

	/* Count this object */
	*nobjs = (*nobjs) + 1;

	/* Directory? */
#ifdef S_ISDIR
	if(S_ISDIR(m) && recursive)
#else
	if(FALSE)
#endif
	{
	    gint i, strc;
	    const gchar *name;
	    gchar *full_path;
	    gchar **strv = (gchar **)GetDirEntNames2((const char *)path, (int *)&strc);
	    for(i = 0; i < strc; i++)
	    {
		name = strv[i];
		if(name == NULL)
		    continue;

		/* Ignore current and parent directory notations */
		if(!strcmp((const char *)name, ".") ||
		   !strcmp((const char *)name, "..")
		)
		{
		    g_free(strv[i]);
		    continue;
		}

		full_path = g_strconcat(
		    path, G_DIR_SEPARATOR_S, name, NULL
		);
		subtotal += EDVArchAddCalculateTotalSize(
		    arch_path, full_path,
		    nobjs,
		    recursive, dereference_links
		);
		g_free(full_path);
		g_free(strv[i]);
	    }
	    g_free(strv);
	}
	/* Link? */
#ifdef S_ISLNK
	else if(S_ISLNK(m))
#else
	else if(FALSE)
#endif
	{
	    /* Dereference this link? */
	    if(dereference_links)
	    {
		/* Get the destination object's stats */
		if(!stat((const char *)path, &stat_buf))
		{
		    const mode_t m = stat_buf.st_mode;

		    /* Destination is a directory? */
#ifdef S_ISDIR
		    if(S_ISDIR(m))
#else
		    if(FALSE)
#endif
		    {
			gint i, strc;
			const gchar *name;
			gchar *full_path;
			gchar **strv = (gchar **)GetDirEntNames2((const char *)path, (int *)&strc);
			for(i = 0; i < strc; i++)
			{
			    name = strv[i];
			    if(name == NULL)
				continue;

			    /* Ignore current and parent directory notations */
			    if(!strcmp((const char *)name, ".") ||
			       !strcmp((const char *)name, "..")
			    )
			    {
				g_free(strv[i]);
				continue;
			    }

			    full_path = g_strconcat(
				path, G_DIR_SEPARATOR_S, name, NULL
			    );
			    subtotal += EDVArchAddCalculateTotalSize(
				arch_path, full_path,
				nobjs,
				recursive, dereference_links
			    );
			    g_free(full_path);
			    g_free(strv[i]);
			}
			g_free(strv);
		    }
		    /* Destination is a file? */
#ifdef S_ISREG
		    else if(S_ISREG(stat_buf.st_mode))
#else
		    else if(FALSE)
#endif
		    {
			subtotal += (gulong)stat_buf.st_size;
		    }
		}
	    }
	    else
	    {
		/* Not dereferencing links so count this link's size */
		subtotal += (gulong)stat_buf.st_size;
	    }
	}
	/* File? */
#ifdef S_ISREG
	else if(S_ISREG(m))
#else
	else if(FALSE)
#endif
	{
	    subtotal += (gulong)stat_buf.st_size;
	}

	return(subtotal);
}


/*
 *	Add object to a ARJ archive.
 *
 *	Inputs assumed valid.
 */
static gint EDVArchAddARJ(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
)
{
	const gchar *prog_arj = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_ARJ
	);
	FILE *fp;
	struct stat arch_path_stat_buf;
	gint status, arch_path_stat_result, p, nobjs;
	gulong total_size;
	gchar	*parent_dir = NULL,
		*pwd = NULL,
		*cmd = NULL,
		*stdout_path = NULL,
		*stderr_path = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(parent_dir);				\
 g_free(cmd);					\
 g_free(stdout_path);				\
 g_free(stderr_path);				\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  EDVSetCWD(pwd);				\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record the previous working directory and set the new
	 * working directory
	 */
	pwd = EDVGetCWD();
	parent_dir = g_dirname(arch_path);
	EDVSetCWD(parent_dir);

	/* If the ARJ archive exists and is empty then it must
	 * be removed first before adding objects to it
	 */
	arch_path_stat_result = stat((const char *)arch_path, &arch_path_stat_buf);
	if(!arch_path_stat_result)
	{
	    if(arch_path_stat_buf.st_size == 0l)
		UNLINK(arch_path);
	}

	/* Format add to archive command and calculate the number
	 * of objects and the total size
	 */
	status = 0;
	p = 0;
	nobjs = 0;
	total_size = 0l;
	if(cmd == NULL)
	{
	    const gchar *tpath;
	    GList *glist;

	    cmd = g_strdup_printf(
		"\"%s\" a -i -y -m%i %s\"%s\"",
		prog_arj,
		CLIP(compression * 4 / 100, 0, 4),
		recursive ? "-r " : "",
		arch_path
	    );
	    for(glist = tar_paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		tpath = (const gchar *)glist->data;
		if(STRISEMPTY(tpath))
		    continue;

		/* Count the size and number of objects */
		total_size += EDVArchAddCalculateTotalSize(
		    arch_path, tpath,
		    &nobjs,
		    recursive, dereference_links
		);

		/* Seek tpath to the relative path as needed */
		if(EDVIsParentPath(parent_dir, tpath))
		{
		    tpath += STRLEN(parent_dir);
		    while(*tpath == G_DIR_SEPARATOR)
			tpath++;
		}

		cmd = G_STRCAT(cmd, " \"");
		cmd = G_STRCAT(cmd, tpath);
		cmd = G_STRCAT(cmd, "\"");
	    }
	    if(cmd == NULL)
	    {
		core->archive_last_error = "Unable to generate the add command.";
		CLEANUP_RETURN(-1);
	    }

	    /* Generate the output file paths */
	    stdout_path = EDVTmpName(NULL);
	    stderr_path = EDVTmpName(NULL);

	    /* Execute the add object to archive command */
	    p = (gint)ExecOE(
		(const char *)cmd,
		(const char *)stdout_path,
		(const char *)stderr_path
	    );
	    if(p <= 0)
	    {
		core->archive_last_error = "Unable to execute the add command.";
		CLEANUP_RETURN(-1);
	    }

	    /* Delete the command */
	    g_free(cmd);
	    cmd = NULL;
	}

	/* Open the output file for reading */
	fp = fopen((const char *)stdout_path, "rb");
	if(fp != NULL)
	{
	    gint buf_pos = 0, nobjs_added = 0;
	    gfloat progress = 0.0f;
	    gchar buf[10000];
	    gboolean need_break = FALSE;

	    /* Begin reading the output file */
	    while(TRUE)
	    {
		/* Update progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			INTERRUPT(p);
			p = 0;
			status = -4;
			break;
		    }
		}


		/* Check if there is new data to be read from the
		 * output file
		 */
		if(FPending(fp))
		{
		    gint c;
		    gboolean got_complete_line = FALSE;

		    /* Copy all available data from the current output
		     * file position to its end to the line buffer buf
		     */
		    while(TRUE)
		    {
			c = (gint)fgetc(fp);
			if((int)c == EOF)
			{
			    clearerr(fp);
			    break;
			}

			if(ISCR(c))
			{
			    got_complete_line = TRUE;
			    if(buf_pos < sizeof(buf))
				buf[buf_pos] = '\0';
			    else
				buf[sizeof(buf) - 1] = '\0';
			    buf_pos = 0;
			    break;
			}

			if(buf_pos < sizeof(buf))
			{
			    buf[buf_pos] = c;
			    buf_pos++;
			}
		    }
		    if(got_complete_line &&
		       (strcasepfx(buf, "Adding") ||
		        strcasepfx(buf, "Replacing")
		       )
		    )
		    {
			/* Update the progress dialog's label */
			gchar *s = buf, *s2;
			const gchar *added_path;

			/* Set s to the start of the loaded line buffer
			 * and seek past the first "Adding:" prefix
			 */
			while(ISBLANK(*s))
			    s++;
			while(!ISBLANK(*s) && (*s != '\0'))
			    s++;
			while(ISBLANK(*s))
			    s++;

			/* Cap the space character after path */
			s2 = strchr(s, ' ');
			if(s2 != NULL)
			    *s2 = '\0';

			added_path = s;

			/* Append path to the list of new paths added
			 * to the archive
			 */
			if(new_paths_list_rtn != NULL)
			    *new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn, STRDUP(added_path)
			    );

			/* Update the progress dialog's label? */
			if(show_progress && !STRISEMPTY(added_path))
			{
			    gchar	*p1 = EDVShortenPath(
				added_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*p2 = EDVShortenPath(
				arch_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Agregar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Addition:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hinzufgen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aggiunta:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoegen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicionar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfying:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Adding:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
				, p1, p2
			    );
			    EDVArchAddMapProgressDialog(
				msg, progress, toplevel, FALSE
			    );
			    g_free(msg);
			    g_free(p1);
			    g_free(p2);
			}

			nobjs_added++;
			progress = (nobjs > 0) ?
			    ((gfloat)nobjs_added / (gfloat)nobjs) : 0.0f;

			continue;
		    }
		}

		if(need_break)
		    break;

		/* Check if the add process has exited, if it has then
		 * we set need_break to TRUE. Which will be tested on
		 * the next loop if there is still no more data to be
		 * read
		 */
		if(!ExecProcessExists(p))
		    need_break = TRUE;

		usleep(8000);
	    }

	    fclose(fp);
	}

	/* Remove the output files */
	UNLINK(stdout_path);
	UNLINK(stderr_path);

	/* Report the final progress? */
	if(show_progress && (status == 0) && ProgressDialogIsQuery())
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		1.0f, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Add object to a LHA archive.
 *
 *	Inputs assumed valid.
 */
static gint EDVArchAddLHA(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
)
{
	const gchar *prog_lha = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_LHA
	);
	FILE *fp;
	gint status, p, nobjs;
	gulong total_size;
	gchar   *parent_dir = NULL,
		*pwd = NULL,
		*cmd = NULL,
		*stdout_path = NULL,
		*stderr_path = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(parent_dir);				\
 g_free(cmd);					\
 g_free(stdout_path);				\
 g_free(stderr_path);				\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  EDVSetCWD(pwd);				\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record the previous working directory and set the new
	 * working directory
	 */
	pwd = EDVGetCWD();
	parent_dir = g_dirname(arch_path);
	EDVSetCWD(parent_dir);

	/* Format add to archive command and calculate the number
	 * of objects and the total size
	 */
	status = 0;
	p = 0;
	nobjs = 0;
	total_size = 0l;
	if(cmd == NULL)
	{
	    const gchar *tpath;
	    GList *glist;

	    cmd = g_strdup_printf(
		"\"%s\" -aq1fo%i \"%s\"",
		prog_lha,
		CLIP((compression * 2 / 100) + 5, 5, 7),
		arch_path
	    );
	    for(glist = tar_paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		tpath = (const gchar *)glist->data;
		if(STRISEMPTY(tpath))
		    continue;

		/* Count the size and number of objects */
		total_size += EDVArchAddCalculateTotalSize(
		    arch_path, tpath,
		    &nobjs,
		    recursive, dereference_links
		);

		/* Seek tpath to the relative path as needed */
		if(EDVIsParentPath(parent_dir, tpath))
		{
		    tpath += STRLEN(parent_dir);
		    while(*tpath == G_DIR_SEPARATOR)
			tpath++;
		}

		cmd = G_STRCAT(cmd, " \"");
		cmd = G_STRCAT(cmd, tpath);
		cmd = G_STRCAT(cmd, "\"");
	    }
	    if(cmd == NULL)
	    {
		core->archive_last_error = "Unable to generate the add command.";
		CLEANUP_RETURN(-1);
	    }

	    /* Generate the output file paths */
	    stdout_path = EDVTmpName(NULL);
	    stderr_path = EDVTmpName(NULL);

	    /* Execute the add object to archive command */
	    p = (gint)ExecOE((const char *)cmd, (const char *)stdout_path, (const char *)stderr_path);
	    if(p <= 0)
	    {
		core->archive_last_error = "Unable to execute the add command.";
		CLEANUP_RETURN(-1);
	    }

	    /* Delete the command */
	    g_free(cmd);
	    cmd = NULL;
	}

	/* Open the output file for reading */
	fp = fopen((const char *)stdout_path, "rb");
	if(fp != NULL)
	{
	    gint buf_pos = 0, nobjs_added = 0;
	    gfloat progress = 0.0f;
	    gchar buf[10000];
	    gboolean need_break = FALSE;

	    /* Begin reading the output file */
	    while(TRUE)
	    {
		/* Update progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			INTERRUPT(p);
			p = 0;
			status = -4;
			break;
		    }
		}

		/* Check if there is new data to be read from the
		 * output file
		 */
		if(FPending(fp))
		{
		    gint c;
		    gboolean got_complete_line = FALSE;

		    /* Copy all available data from the current output
		     * file position to its end to the line buffer buf
		     */
		    while(TRUE)
		    {
			c = (gint)fgetc(fp);
			if(c == EOF)
			{
			    clearerr(fp);
			    break;
			}

			if(ISCR(c))
			{
			    got_complete_line = TRUE;
			    if(buf_pos < sizeof(buf))
				buf[buf_pos] = '\0';
			    else
				buf[sizeof(buf) - 1] = '\0';
			    buf_pos = 0;
			    break;
			}

			if(buf_pos < sizeof(buf))
			{
			    buf[buf_pos] = c;
			    buf_pos++;
			}
		    }
		    /* Got a complete line from the output file and the
		     * progress dialog is mapped?
		     */
		    if(got_complete_line)
		    {
			/* Update the progress dialog label */
			gchar *s = buf, *s2;

			while(ISBLANK(*s))
			    s++;

			/* Cap the end of the path deliminator */
			s2 = (gchar *)strstr((char *)s, " :");
			if(s2 != NULL)
			{
			    const gchar *added_path;

			    *s2 = '\0';

			    added_path = s;

			    /* Append path to the list of new paths
			     * added to the archive
			     */
			    if(new_paths_list_rtn != NULL)
			        *new_paths_list_rtn = g_list_append(
				    *new_paths_list_rtn, STRDUP(added_path)
				);

			    /* Update the progress dialog's label? */
			    if(show_progress && !STRISEMPTY(added_path))
			    {
			        gchar	*p1 = EDVShortenPath(
				    added_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			        ),
					*p2 = EDVShortenPath(
				    arch_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
				),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Agregar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Addition:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hinzufgen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aggiunta:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoegen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicionar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfying:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Adding:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
				    , p1, p2
				);
				EDVArchAddMapProgressDialog(
				    msg, progress, toplevel, FALSE
				);
				g_free(msg);
				g_free(p1);
				g_free(p2);
			    }

			    nobjs_added++;
			    progress = (nobjs > 0) ?
				((gfloat)nobjs_added / (gfloat)nobjs) : 0.0f;
			}

			continue;
		    }
		}

		if(need_break)
		    break;

		/* Check if the add process has exited, if it has then
		 * we set need_break to TRUE. Which will be tested on
		 * the next loop if there is still no more data to be
		 * read
		 */
		if(!ExecProcessExists(p))
		    need_break = TRUE;

		usleep(8000);
	    }

	    fclose(fp);
	}

	/* Remove the output files */
	UNLINK(stdout_path);
	UNLINK(stderr_path);

	/* Report the final progress? */
	if(show_progress && (status == 0) && ProgressDialogIsQuery())
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		1.0f, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Add object to a RAR archive.
 *
 *	Inputs assumed valid.
 */
static gint EDVArchAddRAR(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
)
{
	const gchar *prog_rar = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_RAR
	);
	FILE *fp;
	struct stat arch_path_stat_buf;
	gint status, arch_path_stat_result, p, nobjs;
	gulong total_size;
	gchar	*parent_dir = NULL,
		*pwd = NULL,
		*cmd = NULL,
		*stdout_path = NULL,
		*stderr_path = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(parent_dir);				\
 g_free(cmd);					\
 g_free(stdout_path);				\
 g_free(stderr_path);				\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  EDVSetCWD(pwd);				\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record the previous working directory and set the new
	 * working directory
	 */
	pwd = EDVGetCWD();
	parent_dir = g_dirname(arch_path);
	EDVSetCWD(parent_dir);

	/* If the RAR archive exists and is empty then it must
	 * be removed first before adding objects to it
	 */
	arch_path_stat_result = stat((const char *)arch_path, &arch_path_stat_buf);
	if(!arch_path_stat_result)
	{
	    if(arch_path_stat_buf.st_size == 0l)
		UNLINK(arch_path);
	}

	/* Format add to archive command and calculate the number
	 * of objects and the total size
	 */
	status = 0;
	p = 0;
	nobjs = 0;
	total_size = 0l;
	if(cmd == NULL)
	{
	    const gchar *tpath;
	    GList *glist;

	    /* Format the base command */
	    cmd = g_strdup_printf(
		"\"%s\" a%s \"-p%s\" -m%i%s -y \"%s\"",
		prog_rar,
		dereference_links ? "" : " -ol",
		(STRISEMPTY(password)) ? "-" : password,
		CLIP(compression * 5 / 100, 0, 5),
		recursive ? " -r" : "",
		arch_path
	    );

	    /* Append the objects to extract to the command */
	    for(glist = tar_paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		tpath = (const gchar *)glist->data;
		if(STRISEMPTY(tpath))
		    continue;

		/* Count the size and number of objects */
		total_size += EDVArchAddCalculateTotalSize(
		    arch_path, tpath,
		    &nobjs,
		    recursive, dereference_links
		);

		/* Seek tpath to the relative path as needed */
		if(EDVIsParentPath(parent_dir, tpath))
		{
		    tpath += STRLEN(parent_dir);
		    while(*tpath == G_DIR_SEPARATOR)
			tpath++;
		}

		cmd = G_STRCAT(cmd, " \"");
		cmd = G_STRCAT(cmd, tpath);
		cmd = G_STRCAT(cmd, "\"");
	    }
	    if(cmd == NULL)
	    {
		core->archive_last_error = "Unable to generate the add command.";
		CLEANUP_RETURN(-1);
	    }

	    /* Generate the output file paths */
	    stdout_path = EDVTmpName(NULL);
	    stderr_path = EDVTmpName(NULL);

	    /* Execute the add object to archive command */
	    p = (gint)ExecOE((const char *)cmd, (const char *)stdout_path, (const char *)stderr_path);
	    if(p <= 0)
	    {
		core->archive_last_error = "Unable to execute the add command.";
		CLEANUP_RETURN(-1);
	    }

	    /* Delete the command */
	    g_free(cmd);
	    cmd = NULL;
	}

	/* Open the output file for reading */
	fp = fopen((const char *)stdout_path, "rb");
	if(fp != NULL)
	{
	    gint buf_pos = 0, nobjs_added = 0;
	    gfloat progress = 0.0f;
	    gchar buf[10000];
	    gboolean need_break = FALSE;

	    /* Begin reading the output file */
	    while(TRUE)
	    {
		/* Update progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			INTERRUPT(p);
			p = 0;
			status = -4;
			break;
		    }
		}

		/* Check if there is new data to be read from the
		 * output file
		 */
		if(FPending(fp))
		{
		    gint c;
		    gboolean got_complete_line = FALSE;

		    /* Copy all available data from the current output
		     * file position to its end to the line buffer buf
		     */
		    while(TRUE)
		    {
			c = (gint)fgetc(fp);
			if((int)c == EOF)
			{
			    clearerr(fp);
			    break;
			}

			if(ISCR(c))
			{
			    got_complete_line = TRUE;
			    if(buf_pos < sizeof(buf))
				buf[buf_pos] = '\0';
			    else
				buf[sizeof(buf) - 1] = '\0';
			    buf_pos = 0;
			    break;
			}

			if(buf_pos < sizeof(buf))
			{
			    buf[buf_pos] = c;
			    buf_pos++;
			}
		    }
		    /* Got a complete line from the output file and the
		     * progress dialog is mapped?
		     */
		    if(got_complete_line &&
		       (strcasepfx(buf, "Adding  ") ||
		        strcasepfx(buf, "Updating  ")
		       )
		    )
		    {
			/* Update progress dialog label */
			gchar *s = buf, *s2;
			const gchar *added_path;

			/* Set s to the start of the loaded line buffer
			 * and seek past the "Adding" or "Updating"
			 * prefix
			 */
			while(ISBLANK(*s))
			    s++;
			while(!ISBLANK(*s) && (*s != '\0'))
			    s++;
			while(ISBLANK(*s))
			    s++;

			/* Cap s at the end deliminator "  " */
			s2 = strstr(s, "  ");
			if(s2 == NULL)
			    s2 = strchr(s, ' ');
			if(s2 == NULL)
			    s2 = strchr(s, '\t');
			if(s2 != NULL)
			    *s2 = '\0';

			added_path = s;

			/* Append path to the list of new paths added
			 * to the archive
			 */
			if(new_paths_list_rtn != NULL)
			    *new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn, STRDUP(added_path)
			    );

			/* Update the progress dialog's label? */
			if(show_progress && !STRISEMPTY(added_path))
			{
			    gchar	*p1 = EDVShortenPath(
				added_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*p2 = EDVShortenPath(
				arch_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Agregar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Addition:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hinzufgen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aggiunta:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoegen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicionar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfying:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Adding:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
				, p1, p2
			    );
			    EDVArchAddMapProgressDialog(
				msg, progress, toplevel, FALSE
			    );
			    g_free(msg);
			    g_free(p1);
			    g_free(p2);
			}

			nobjs_added++;
			progress = (nobjs > 0) ?
			    ((gfloat)nobjs_added / (gfloat)nobjs) : 0.0f;

			continue;
		    }
		}

		if(need_break)
		    break;

		/* Check if the add process has exited, if it has then
		 * we set need_break to TRUE. Which will be tested on
		 * the next loop if there is still no more data to be
		 * read
		 */
		if(!ExecProcessExists(p))
		    need_break = TRUE;

		usleep(8000);
	    }

	    fclose(fp);
	}

	/* Remove the output files */
	UNLINK(stdout_path);
	UNLINK(stderr_path);

	/* Report the final progress? */
	if(show_progress && (status == 0) && ProgressDialogIsQuery())
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		1.0f, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Adds the object(s) to the archive.
 *
 *	The arch_path specifies the archive.
 *
 *	The tar_paths_list specifies the list of objects to add to the
 *	archive.
 *
 *	If new_paths_list_rtn is not NULL then a list of paths
 *	describing the objects that have been added to the archive will
 *	be returned. The calling function must delete the returned list
 *	and each string.
 *
 *	If password is not NULL then a password will be used to
 *	encrypt the objects being added to the archive (if the archive
 *	format supports encryption).
 *
 *	If recursive is TRUE then all the objects in any directory
 *	object specified in tar_paths_list will be added as well.
 *
 *	The compression specifies the compression level from 0 to
 *	100 where 0 is no compression and 100 is maximum compression.
 *	This value has different affects on different archive formats.
 *
 *	If dereference_links is TRUE then the link's target will
 *	be added instead of the link itself.
 */
gint EDVArchAdd(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
)
{
	gint status;
	const gulong time_start = (gulong)time(NULL);
	const gchar *s, *arch_name;
	gchar *path;
	GList *glist, *ltar_paths_list = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 if(ltar_paths_list != NULL) {			\
  g_list_foreach(				\
   ltar_paths_list, (GFunc)g_free, NULL		\
  );						\
  g_list_free(ltar_paths_list);			\
 }						\
						\
 return(_v_);					\
}

	if(ProgressDialogIsQuery())
	{
	    EDVArchAddCopyErrorMessage(
		core,
"An operation is already in progress, please try again later."
	    );
	    return(-6);
	}

	/* Reset the last error message */
	EDVArchAddCopyErrorMessage(core, NULL);

	/* Reset returns */
	if(new_paths_list_rtn != NULL)
	    *new_paths_list_rtn = NULL;

	if((core == NULL) || STRISEMPTY(arch_path) ||
	   (yes_to_all == NULL)
	)
	{
	    EDVArchAddCopyErrorMessage(core, "Invalid value.");
	    return(-2);
	}

	/* No objects to add? */
	if(tar_paths_list == NULL)
	    return(0);

	arch_name = g_basename(arch_path);

	/* Make a copy of the list of objects to be added to the
	 * archive as ltar_paths_list
	 */
	for(glist = tar_paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    s = (const gchar *)glist->data;
	    if(STRISEMPTY(s))
		continue;

	    path = STRDUP(s);
	    if(path == NULL)
		continue;

	    /* Simplify target object path notation, stripping
	     * tailing deliminators
	     */
	    EDVSimplifyPath(path);

	    ltar_paths_list = g_list_append(ltar_paths_list, path);
	}
	if(ltar_paths_list == NULL)
	{
	    CLEANUP_RETURN(-1);
	}

	/* Begin adding the target objects to the archive. The adding
	 * method will be determined by taking the extension of the
	 * archive's name
	 */

	/* ARJ Archive */
	if(EDVIsExtension(arch_name, ".arj"))
	{
	    status = EDVArchAddARJ(
		core, arch_path,
		ltar_paths_list,
		new_paths_list_rtn,
		toplevel, show_progress, interactive, yes_to_all,
		recursive, compression, dereference_links
	    );
	}
	/* LHA Archive */
	else if(EDVIsExtension(arch_name, ".lha"))
	{
	    status = EDVArchAddLHA(
		core, arch_path,
		ltar_paths_list,
		new_paths_list_rtn,
		toplevel, show_progress, interactive, yes_to_all,
		recursive, compression, dereference_links
	    );
	}
	/* RAR Archive */
	else if(EDVIsExtension(arch_name, ".rar"))
	{
	    status = EDVArchAddRAR(
		core, arch_path,
		ltar_paths_list,
		new_paths_list_rtn,
		password,
		toplevel, show_progress, interactive, yes_to_all,
		recursive, compression, dereference_links
	    );
	}
	/* Tape Archive (Compressed) */
	else if(EDVIsExtension(arch_name, ".tar.Z"))
	{
	    status = EDVArchAddTar(
		core, arch_path,
		ltar_paths_list,
		new_paths_list_rtn,
		toplevel, show_progress, interactive, yes_to_all,
		recursive, compression, dereference_links,
		TRUE,		/* Is compress compressed */
		FALSE,		/* Not gzip compressed */
		FALSE		/* Not bzip2 compressed */
	    );
	}
	/* Tape Archive (GZip) */
	else if(EDVIsExtension(arch_name, ".tgz .tar.gz"))
	{
	    status = EDVArchAddTar(
		core, arch_path,
		ltar_paths_list,
		new_paths_list_rtn,
		toplevel, show_progress, interactive, yes_to_all,
		recursive, compression, dereference_links,
		FALSE,		/* Not compress compressed */
		TRUE,		/* Is gzip compressed */
		FALSE		/* Not bzip2 compressed */
	    );
	}
	/* Tape Archive (BZip2) */
	else if(EDVIsExtension(arch_name, ".tar.bz2"))
	{
	    status = EDVArchAddTar(
		core, arch_path,
		ltar_paths_list,
		new_paths_list_rtn,
		toplevel, show_progress, interactive, yes_to_all,
		recursive, compression, dereference_links,
		FALSE,		/* Not compress compressed */
		FALSE,		/* Not gzip compressed */
		TRUE		/* Is bzip2 compressed */
	    );
	}
	/* Tape Archive */
	else if(EDVIsExtension(arch_name, ".tar"))
	{
	    status = EDVArchAddTar(
		core, arch_path,
		ltar_paths_list,
		new_paths_list_rtn,
		toplevel, show_progress, interactive, yes_to_all,
		recursive, compression, dereference_links,
		FALSE,		/* Not compress compressed */
		FALSE,		/* Not gzip compressed */
		FALSE		/* Not bzip2 compressed */
	    );
	}
	/* X Archive */
	else if(EDVIsExtension(arch_name, ".xar"))
	{
	    status = EDVArchAddXArchive(
		core, arch_path,
		ltar_paths_list,
		new_paths_list_rtn,
		password,
		toplevel, show_progress, interactive, yes_to_all,
		recursive, compression, dereference_links
	    );
	}
	/* PKZip Archive */
	else if(EDVIsExtension(arch_name, ".xpi .zip"))
	{
	    status = EDVArchAddPKZip(
		core, arch_path,
		ltar_paths_list,
		new_paths_list_rtn,
		password,
		toplevel, show_progress, interactive, yes_to_all,
		recursive, compression, dereference_links
	    );
	}
	else
	{
	    core->archive_last_error = "Unsupported archive format.";
	    status = -2;
	}

	/* Record history */
	if(status != 0)
	{
	    const gulong time_end = (gulong)time(NULL);
	    const gchar *first_src_obj = (ltar_paths_list != NULL) ?
		(const gchar *)ltar_paths_list->data : NULL;

	    EDVAppendHistory(
		core,
		EDV_HISTORY_ARCHIVE_OBJECT_ADD,
		time_start, time_end,
		status,
		first_src_obj,		/* Source */
		arch_path,		/* Target */
		core->archive_last_error		/* Comment */
	    );
	}
	else
	{
	    const gulong time_end = (gulong)time(NULL);

	    for(glist = ltar_paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
		EDVAppendHistory(
		    core,
		    EDV_HISTORY_ARCHIVE_OBJECT_ADD,
		    time_start, time_end,
		    status,
		    (const gchar *)glist->data,	/* Source */
		    arch_path,			/* Target */
		    core->archive_last_error			/* Comment */
		);
	}

	/* Need to flush disk changes since the archive may have been
	 * modified on another process and the changes have not reached
	 * our process yet
	 */
	sync();

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
