#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <time.h>
#include <signal.h>
#include <sys/stat.h>
#ifdef HAVE_LIBZIP
# include <zip.h>
#endif
#include <gtk/gtk.h>
#include <unistd.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "cdialog.h"
#include "progressdialog.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_archive_obj.h"
#include "edv_archive_add.h"
#include "edv_archive_add_zip.h"
#include "endeavour2.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"


#ifdef HAVE_LIBZIP
typedef struct _EDVArchAddZipSourceData	EDVArchAddZipSourceData;
#define EDV_ARCH_ADD_ZIP_SOURCE_DATA(p)	((EDVArchAddZipSourceData *)(p))
#endif


#ifdef HAVE_LIBZIP
static struct zip *EDVOpenZipWriting(
	const gchar *arch_path,
	gint *zip_error_code, gint *sys_error_code
);
static EDVArchAddZipSourceData *EDVArchAddZipSourceDataNew(void);
static void EDVArchAddZipSourceDataDelete(EDVArchAddZipSourceData *d);
static ssize_t EDVArchAddZipSourceCB(
	void *user_data, void *data,
	size_t len, enum zip_source_cmd cmd
);
static gint EDVArchAddZipIterateFile(
	edv_core_struct *core, gint *status,
	const gchar *arch_path, struct zip *archive,
	const gchar *password,
	const gchar *parent,		/* Parent directory of the archive */
	const gchar *path,		/* The object to add to the archive */
	struct stat *stat_buf,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress, const gint compression
);
static gint EDVArchAddZipIterateDirectory(
	edv_core_struct *core, gint *status,
	const gchar *arch_path, struct zip *archive,
	const gchar *password,
	const gchar *parent,		/* Parent directory of the archive */
	const gchar *path,		/* The object to add to the archive */
	struct stat *stat_buf,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress
);
static gint EDVArchAddZipIterateLink(
	edv_core_struct *core, gint *status,
	const gchar *arch_path, struct zip *archive,
	const gchar *password,
	const gchar *parent,		/* Parent directory of the archive */
	const gchar *path,		/* The object to add to the archive */
	struct stat *stat_buf,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress, const gint compression
);
static void EDVArchAddZipIterate(
	edv_core_struct *core, gint *status,
	const gchar *arch_path, struct zip *archive,
	const gchar *password,
	const gchar *parent,		/* Parent directory of the archive */
	const gchar *path,		/* The object to add to the archive */
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
);
#endif	/* HAVE_LIBZIP */
gint EDVArchAddPKZip(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
);


#ifdef HAVE_LIBZIP
/*
 *	libzip source data:
 *
 *	Used in EDVArchAddZipSourceCB() to pass and record data
 *	relavent to the object being added to the archive.
 */
struct _EDVArchAddZipSourceData {
	edv_core_struct	*core;

	gint		*status;	/* Our current status */
	int		zip_error[2];	/* Zip error codes */

	GtkWidget	*toplevel;
	gboolean	show_progress;

	gchar		*arch_path;	/* Archive's path */

	GList		**new_paths_list_rtn;

	gulong		*cur_size,	/* Total size of all the
					 * objects added thus far in
					 * bytes */
			total_size;	/* Total size of all the
					 * objects to be added in
					 * bytes */

	gchar		*src_path,	/* Current object's path */
			*tar_path;	/* Path of object in the archive */
	edv_object_type	type;		/* Current object's type */
	gulong		this_size;	/* Current object's size in
					 * bytes (same as stat_buf.st_size) */
	struct stat	stat_buf;	/* Current object's struct stat */
	FILE		*fp;		/* File pointer to the current
					 * object */
	gulong		bytes_read;	/* Bytes read for the current
					 * object */
};
#endif	/* HAVE_LIBZIP */


#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define UNLINK(p)	(((p) != NULL) ? (gint)unlink((const char *)(p)) : -1)
#define INTERRUPT(i)	(((i) > 0) ? (gint)kill((int)(i), SIGINT) : -1)


#ifdef HAVE_LIBZIP
/*
 *	Opens the PKZip Archive for writing.
 *
 *	Returns the handle to the PKZip Archive.
 */
static struct zip *EDVOpenZipWriting(
	const gchar *arch_path,
	gint *zip_error_code, gint *sys_error_code
)
{
	struct stat stat_buf;
	struct zip *archive;

	/* If the PKZip Archive exists but has a size of 0 bytes then
	 * it must be deleted first or else libzip will return an
	 * error stating that it is not a valid PKZip Archive
	 */
	if(!stat((const char *)arch_path, &stat_buf))
	{
	    if(stat_buf.st_size == 0l)
		unlink(arch_path);
	}

	archive = zip_open(arch_path, ZIP_CREATE, zip_error_code);
	if(sys_error_code != NULL)
	    *sys_error_code = errno;

	return(archive);
}

/*
 *	Creates the libzip source data.
 */
static EDVArchAddZipSourceData *EDVArchAddZipSourceDataNew(void)
{
	return((EDVArchAddZipSourceData *)g_malloc0(
	    sizeof(EDVArchAddZipSourceData)
	));
}

/*
 *	Deletes the libzip source data.
 *
 *	If the source file is still opened then it will be closed.
 */
static void EDVArchAddZipSourceDataDelete(EDVArchAddZipSourceData *d)
{
	if(d == NULL)
	    return;

	g_free(d->arch_path);
	g_free(d->src_path);
	g_free(d->tar_path);
	if(d->fp != NULL)
	    fclose(d->fp);
	g_free(d);
}

/*
 *	libzip source callback.
 *
 *	For adding objects of any type to the PKZip Archive by
 *	responding to libzip commands.
 */
static ssize_t EDVArchAddZipSourceCB(
	void *user_data, void *data,
	size_t len, enum zip_source_cmd cmd
)
{
	edv_core_struct *core;
	EDVArchAddZipSourceData *d = EDV_ARCH_ADD_ZIP_SOURCE_DATA(user_data);
	if(d == NULL)
	    return(-1);

	core = d->core;

	switch(cmd)
	{
	  case ZIP_SOURCE_OPEN:
	    /* Previously aborted? */
	    if(*d->status == -4)
		return(0);	/* Return successfully opened (even though not) */

	    /* Update the progress dialog to display the object
	     * being added to the archive?
	     */
	    if(d->show_progress)
	    {
		gchar	*p1 = EDVShortenPath(
		    d->src_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
		),
			*p2 = EDVShortenPath(
		    d->arch_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
		),
			*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Agregar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Addition:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hinzufgen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aggiunta:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoegen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicionar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfying:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Adding:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
		    , p1, p2
		);
		EDVArchAddMapProgressDialog(
		    msg,
		    (d->total_size > 0l) ?
	((gfloat)(*d->cur_size) / (gfloat)d->total_size) : 0.0f,
		    d->toplevel, FALSE
		);
		g_free(msg);
		g_free(p1);
		g_free(p2);
		if(ProgressDialogIsQuery())
		{
		    if(ProgressDialogStopCount() > 0)
		    {
			*d->status = -4;
			return(0);
		    }
		}
	    }

	    /* Open by object type */
	    switch(d->type)
	    {
	      case EDV_OBJECT_TYPE_UNKNOWN:
		{
		    int *zip_error = d->zip_error;
		    zip_error[0] = ZIP_ER_OPEN;
		    zip_error[1] = EINVAL;
		}
		core->archive_last_error = "Unknown object type.";
		*d->status = -2;
		return(-1);
		break;

	      case EDV_OBJECT_TYPE_FILE:
		/* Is the file already opened? */
		if(d->fp != NULL)
		{
		    /* Rewind it */
		    rewind(d->fp);
		    d->bytes_read = 0l;
		}
		/* Is the source file known? */
		else if(d->src_path != NULL)
		{
		    /* Open the source file for reading */
		    d->fp = fopen((const char *)d->src_path, "rb");
		    if(d->fp == NULL)
		    {
			/* Return success to libzip and only indicate
			 * local error because otherwise libzip will
			 * not add subsequent objects and revert the
			 * entire archive
			 */
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			    d->src_path, d->arch_path, g_strerror(error_code)
			);
			EDVArchAddCopyErrorMessage(core, msg);
			g_free(msg);
			*d->status = -1;
			return(0);
		    }
		    d->bytes_read = 0l;
		}
		else
		{
		    int *zip_error = d->zip_error;
		    zip_error[0] = ZIP_ER_INTERNAL;
		    zip_error[1] = EINVAL;
		    core->archive_last_error = "Source file not specified.";
		    return(-1);
		}
		break;

	      case EDV_OBJECT_TYPE_DIRECTORY:
	      case EDV_OBJECT_TYPE_LINK:
	      case EDV_OBJECT_TYPE_DEVICE_BLOCK:
	      case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
	      case EDV_OBJECT_TYPE_FIFO:
	      case EDV_OBJECT_TYPE_SOCKET:
		d->bytes_read = 0l;
		break;

	      case EDV_OBJECT_TYPE_ERROR:
		/* Ignore error objects */
		d->bytes_read = 0l;
		break;
	    }

	    /* Add this object to the list of objects added */
	    if(d->new_paths_list_rtn != NULL)
		*d->new_paths_list_rtn = g_list_append(
		    *d->new_paths_list_rtn, STRDUP(d->src_path)
		);

	    return(0);
	    break;

	  case ZIP_SOURCE_READ:
	    /* Previously aborted? */
	    if(*d->status == -4)
		return(0);	/* Return 0 bytes read (which is a success) */

	    /* Report progress? */
	    if(d->show_progress && ProgressDialogIsQuery())
	    {
		ProgressDialogUpdate(
		    NULL, NULL, NULL, NULL,
		    (d->total_size > 0l) ?
	((gfloat)(*d->cur_size) / (gfloat)d->total_size) : 0.0f,
		    EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
		/* User aborted? */
		if(ProgressDialogStopCount() > 0)
		{
		    *d->status = -4;
		    return(0);
		}
	    }

	    /* Read by object type */
	    switch(d->type)
	    {
	      case EDV_OBJECT_TYPE_UNKNOWN:
		{
		    int *zip_error = d->zip_error;
		    zip_error[0] = ZIP_ER_READ;
		    zip_error[1] = EINVAL;
		}
		core->archive_last_error = "Unknown object type.";
		*d->status = -2;
		return(-1);
		break;

	      case EDV_OBJECT_TYPE_FILE:
		/* Was the file opened? */
		if(d->fp != NULL)
		{
		    gint units_read;

		    /* Read the requested length of bytes from the
		     * file into the specified buffer
		     */
		    if(len > 0)
			units_read = (gint)fread(
			    data, sizeof(guint8), (size_t)len, d->fp
			);
		    else
			units_read = 0;

		    /* Read error? */
		    if(units_read < 0)
		    {
			/* Return success to libzip and only indicate
			 * local error because otherwise libzip will
			 * not add subsequent objects and revert the
			 * entire archive
			 */
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			    d->src_path, d->arch_path, g_strerror(error_code)
			);
			EDVArchAddCopyErrorMessage(core, msg);
			g_free(msg);
			*d->status = -1;
			return(0);	/* Return no bytes read */
		    }

		    /* Increment the number of bytes read for this
		     * object
		     */
		    d->bytes_read += (gulong)(units_read * sizeof(guint8));

		    /* Increment the total number of bytes read for
		     * all the files to be added to the archive
		     */
		    *d->cur_size = (*d->cur_size) +
			(gulong)(units_read * sizeof(guint8));

		    /* Return the number of bytes read */
		    return((ssize_t)(units_read * sizeof(guint8)));
		}
		else
		{
		    /* A file may not have been able to be opened but
		     * we should not return error to libzip because
		     * libzip will not add subsequent objects and
		     * revert the entire archive so instead return 0
		     * bytes read
		     */
		    return(0);	/* Return no bytes read */
		}
		break;

	      case EDV_OBJECT_TYPE_DIRECTORY:
		return(0);
		break;

	      case EDV_OBJECT_TYPE_LINK:
		/* Is the path to the source file known? */
		if(d->src_path != NULL)
		{
		    gint bytes_read, value_len;
		    const gulong prev_pos = d->bytes_read;
		    gchar *value;

		    /* Already read this or no need to? */
		    if(prev_pos >= d->this_size)
			return(0);	/* Return no bytes read */

		    /* Get the link's target value */
		    value = EDVGetLinkTarget(d->src_path);
		    if(value == NULL)
		    {
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			    d->src_path, d->arch_path, g_strerror(error_code)
			);
			EDVArchAddCopyErrorMessage(core, msg);
			g_free(msg);
			*d->status = -1;
			return(0);	/* Return no bytes read */
		    }

		    value_len = STRLEN(value);

		    /* Copy the link's target value to libzip's
		     * read buffer and update bytes_read to indicate
		     * the number of bytes coppied to it
		     */
		    if((data != NULL) && (len > 0))
		    {
			const gchar	*src = value + prev_pos,
					*src_end = value + value_len;
			gchar	*tar = (gchar *)data,
				*tar_end = tar + len;

			bytes_read = 0;		/* Reset */

			while((src < src_end) && (tar < tar_end))
			{
			    *tar = *src;
			    tar++;
			    src++;
			    bytes_read++;
			}

			/* Increment the number of bytes read for
			 * this object
			 */
			d->bytes_read += (gulong)bytes_read;

			/* Increment the total number of bytes read
			 * for all the files to be added to the
			 * archive
			 */
			*d->cur_size = (*d->cur_size) + (gulong)bytes_read;
		    }
		    else
		    {
			bytes_read = 0;
		    }

		    /* Delete the link's target value */
		    g_free(value);

		    /* Return the number of bytes read */
		    return((ssize_t)bytes_read);
		}
		else
		{
		    int *zip_error = d->zip_error;
		    zip_error[0] = ZIP_ER_INTERNAL;
		    zip_error[1] = EINVAL;
		    core->archive_last_error = "Source file not specified.";
		    return(-1);
		}
		break;

	      case EDV_OBJECT_TYPE_DEVICE_BLOCK:
	      case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
	      case EDV_OBJECT_TYPE_FIFO:
	      case EDV_OBJECT_TYPE_SOCKET:
		return(0);
		break;

	      case EDV_OBJECT_TYPE_ERROR:
		/* Do not report any data read for error objects */
		return(0);
		break;
	    }
	    break;

	  case ZIP_SOURCE_CLOSE:
	    /* Close the object as needed, regardless of if it was
	     * opened or not or if there was an error
	     */
	    if(d->fp != NULL)
	    {
		fclose(d->fp);
		d->fp = NULL;
	    }

	    /* Report progress only if there were no errors and
	     * and not user aborted
	     */
	    if(d->show_progress && ProgressDialogIsQuery() &&
	       (*d->status == 0)
	    )
	    {
		ProgressDialogUpdate(
		    NULL, NULL, NULL, NULL,
		    (d->total_size > 0l) ?
	((gfloat)(*d->cur_size) / (gfloat)d->total_size) : 0.0f,
		    EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
	    }

	    return(0);
	    break;

	  case ZIP_SOURCE_STAT:
	    if((data != NULL) && (len >= sizeof(struct zip_stat)))
	    {
		struct stat *src_stat = &d->stat_buf;
		struct zip_stat *zstat = (struct zip_stat *)data;

		/* Set the zip stat values
		 *
		 * Although not explicitly stated in the libzip
		 * documentation as of libzip 0.7, libzip's code
		 * suggests that only members; crc, mtime, size,
		 * comp_size and comp_method are to be set
		 */
		zstat->crc = 0;
		zstat->mtime = src_stat->st_mtime;
		switch(d->type)
		{
		  case EDV_OBJECT_TYPE_UNKNOWN:
		    zstat->size = 0l;
		    break;
		  case EDV_OBJECT_TYPE_FILE:
		    zstat->size = src_stat->st_size;
		    break;
		  case EDV_OBJECT_TYPE_DIRECTORY:
		    zstat->size = 0l;
		    break;
		  case EDV_OBJECT_TYPE_LINK:
		    zstat->size = src_stat->st_size;
		    break;
		  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
		  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
		  case EDV_OBJECT_TYPE_FIFO:
		  case EDV_OBJECT_TYPE_SOCKET:
		    zstat->size = 0l;
		    break;
		  case EDV_OBJECT_TYPE_ERROR:
		    zstat->size = 0l;
		    break;
		}
		zstat->comp_size = -1;
		zstat->comp_method = ZIP_CM_STORE;
		zstat->encryption_method = ZIP_EM_NONE;

		/* We return the size of struct zip_stat instead of 0
		 * to indicate success, even though the libzip
		 * documentation which states that we should return 0
		 * for success, this is due to the fact that libzip,
		 * actually, expects the size of struct zip_stat and
		 * 0 or -1 are both errors in this case
		 */
		return((ssize_t)sizeof(struct zip_stat));
	    }
	    else
	    {
		int *zip_error = d->zip_error;
		zip_error[0] = ZIP_ER_INTERNAL;
		zip_error[1] = EINVAL;
		core->archive_last_error =
"libzip requested to obtain statistics without providing a buffer.";
		*d->status = -3;
		return(-1);
	    }
	    break;

	  case ZIP_SOURCE_ERROR:
	    if((data != NULL) && (len >= (2 * sizeof(int))))
	    {
		memcpy(data, d->zip_error, (2 * sizeof(int)));
		return((ssize_t)(2 * sizeof(int)));
	    }
	    else
	    {
		int *zip_error = d->zip_error;
		zip_error[0] = ZIP_ER_INTERNAL;
		zip_error[1] = EINVAL;
		return(-1);
	    }
	    break;

	  case ZIP_SOURCE_FREE:
	    EDVArchAddZipSourceDataDelete(d);
	    return(0);
	    break;
	}

	return(-1);
}

/*
 *	Adds the file specified by path to the PKZip Archive.
 *
 *	Called by EDVArchAddZipIterate().
 */
static gint EDVArchAddZipIterateFile(
	edv_core_struct *core, gint *status,
	const gchar *arch_path, struct zip *archive,
	const gchar *password,
	const gchar *parent,		/* Parent directory of the archive */
	const gchar *path,		/* The object to add to the archive */
	struct stat *stat_buf,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress, const gint compression
)
{
	struct zip_source *zsrc;
	gint i;
	gchar *tar_path;
	EDVArchAddZipSourceData *d;

	/* Generate the path of the file within the archive */
	if(EDVIsParentPath(parent, path))
	{
	    const char *s = path + STRLEN(parent);
	    while(*s == G_DIR_SEPARATOR)
		s++;
	    tar_path = STRDUP(s);
	}
	else
	{
	    tar_path = STRDUP(path);
	}
	if(tar_path == NULL)
	{
	    core->archive_last_error = "Memory allocation error.";
	    *status = -3;
	    return(*status);
	}

	/* Create a new and set up the libzip source data */
	d = EDVArchAddZipSourceDataNew();
	if(d == NULL)
	{
	    g_free(tar_path);
	    core->archive_last_error = "Memory allocation error.";
	    *status = -3;
	    return(*status);
	}
	d->core = core;
	d->status = status;
	d->zip_error[0] = ZIP_ER_OK;
	d->zip_error[1] = 0;
	d->toplevel = toplevel;
	d->show_progress = show_progress;
	d->arch_path = STRDUP(arch_path);
	d->new_paths_list_rtn = new_paths_list_rtn;
	d->cur_size = cur_size;
	d->total_size = total_size;

	d->src_path = STRDUP(path);
	d->tar_path = STRDUP(tar_path);
	d->type = EDV_OBJECT_TYPE_FILE;
	d->this_size = (gulong)stat_buf->st_size;
	memcpy(&d->stat_buf, stat_buf, sizeof(struct stat));
	d->fp = NULL;
	d->bytes_read = 0l;

	/* Check if the object already exists in the PKZip Archive */
	i = zip_name_locate(archive, tar_path, 0);

	/* Set the libzip source function */
	zsrc = zip_source_function(archive, EDVArchAddZipSourceCB, d);
	if(zsrc == NULL)
	{
	    gchar *msg = g_strdup_printf(
"Unable to set the libzip source function.\n\
\n\
%s.",
		zip_strerror(archive)
	    );
	    EDVArchAddCopyErrorMessage(core, msg);
	    g_free(msg);
	    EDVArchAddZipSourceDataDelete(d);
	    g_free(tar_path);
	    *status = -1;
	    return(*status);
	}

	/* Replace or add the file? */
	if(i > -1)
	{
	    /* Replace the existing file */
	    if(zip_replace(archive, i, zsrc) < 0)
	    {
		gchar *msg = g_strdup_printf(
"Unable to replace the PKZip Archive object #%i.\n\
\n\
%s.",
		    i + 1, zip_strerror(archive)
		);
		EDVArchAddCopyErrorMessage(core, msg);
		g_free(msg);
		zip_source_free(zsrc);
		g_free(tar_path);
		*status = -1;
		return(*status);
	    }
	}
	else
	{
	    /* Add the file */
	    if(zip_add(archive, tar_path, zsrc) < 0)
	    {
		gchar *msg = g_strdup_printf(
"Unable to add a new PKZip Archive object.\n\
\n\
%s.",
		    zip_strerror(archive)
		);
		EDVArchAddCopyErrorMessage(core, msg);
		g_free(msg);
		zip_source_free(zsrc);
		g_free(tar_path);
		*status = -1;
		return(*status);
	    }
	}

	g_free(tar_path);

	return(0);
}

/*
 *	Adds the directory specified by path to the PKZip Archive.
 *
 *	Does not add any objects within the directory into the archive.
 *
 *	Called by EDVArchAddZipIterate().
 */
static gint EDVArchAddZipIterateDirectory(
	edv_core_struct *core, gint *status,
	const gchar *arch_path, struct zip *archive,
	const gchar *password,
	const gchar *parent,		/* Parent directory of the archive */
	const gchar *path,		/* The object to add to the archive */
	struct stat *stat_buf,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress
)
{
	struct zip_source *zsrc;
	gint i;
	gchar *tar_path;
	EDVArchAddZipSourceData *d;

	/* Generate the path of the file within the archive
	 *
	 * All directories in PKZip Archives must have a tailing
	 * deliminator character to denote that it is a directory
	 */
	if(EDVIsParentPath(parent, path))
	{
	    const char *s = path + STRLEN(parent);
	    while(*s == G_DIR_SEPARATOR)
		s++;
	    tar_path = g_strconcat(s, G_DIR_SEPARATOR_S, NULL);
	}
	else
	{
	    tar_path = g_strconcat(path, G_DIR_SEPARATOR_S, NULL);
	}
	if(tar_path == NULL)
	{
	    core->archive_last_error = "Memory allocation error.";
	    *status = -3;
	    return(*status);
	}

	/* Create a new and set up the libzip source data */
	d = EDVArchAddZipSourceDataNew();
	if(d == NULL)
	{
	    g_free(tar_path);
	    core->archive_last_error = "Memory allocation error.";
	    *status = -3;
	    return(*status);
	}
	d->core = core;
	d->status = status;
	d->zip_error[0] = ZIP_ER_OK;
	d->zip_error[1] = 0;
	d->toplevel = toplevel;
	d->show_progress = show_progress;
	d->arch_path = STRDUP(arch_path);
	d->new_paths_list_rtn = new_paths_list_rtn;
	d->cur_size = cur_size;
	d->total_size = total_size;

	d->src_path = STRDUP(path);
	d->tar_path = STRDUP(tar_path);
	d->type = EDV_OBJECT_TYPE_DIRECTORY;
	d->this_size = (gulong)stat_buf->st_size;
	memcpy(&d->stat_buf, stat_buf, sizeof(struct stat));
	d->fp = NULL;
	d->bytes_read = 0l;

	/* Check if the object already exists in the PKZip Archive */
	i = zip_name_locate(archive, tar_path, 0);

	/* Set the libzip source function */
	zsrc = zip_source_function(archive, EDVArchAddZipSourceCB, d);
	if(zsrc == NULL)
	{
	    gchar *msg = g_strdup_printf(
"Unable to set the libzip source function.\n\
\n\
%s.",
		zip_strerror(archive)
	    );
	    EDVArchAddCopyErrorMessage(core, msg);
	    g_free(msg);
	    EDVArchAddZipSourceDataDelete(d);
	    g_free(tar_path);
	    *status = -1;
	    return(*status);
	}

	/* Does the object already exist within the archive? */
	if(i > -1)
	{
	    /* Replace the object */
	    if(zip_replace(archive, i, zsrc) < 0)
	    {
		gchar *msg = g_strdup_printf(
"Unable to replace the PKZip Archive object #%i.\n\
\n\
%s.",
		    i + 1, zip_strerror(archive)
		);
		EDVArchAddCopyErrorMessage(core, msg);
		g_free(msg);
		zip_source_free(zsrc);
		g_free(tar_path);
		*status = -1;
		return(*status);
	    }
	}
	else
	{
	    /* Add the directory */
	    if(zip_add(archive, tar_path, zsrc) < 0)
	    {
		gchar *msg = g_strdup_printf(
"Unable to add a new PKZip Archive object.\n\
\n\
%s.",
		    zip_strerror(archive)
		);
		EDVArchAddCopyErrorMessage(core, msg);
		g_free(msg);
		zip_source_free(zsrc);
		g_free(tar_path);
		*status = -1;
		return(*status);
	    }
	}

	g_free(tar_path);

	return(0);
}

/*
 *	Adds the link specified by path to the PKZip Archive.
 *
 *	The link is not dereferenced.
 *
 *	Called by EDVArchAddZipIterate().
 */
static gint EDVArchAddZipIterateLink(
	edv_core_struct *core, gint *status,
	const gchar *arch_path, struct zip *archive,
	const gchar *password,
	const gchar *parent,		/* Parent directory of the archive */
	const gchar *path,		/* The object to add to the archive */
	struct stat *stat_buf,
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress, const gint compression
)
{
	struct zip_source *zsrc;
	gint i;
	gchar *tar_path;
	EDVArchAddZipSourceData *d;

	/* Generate the path of the file within the archive
	 *
	 * All links in PKZip Archives must have a tailing '@'
	 * character to denote that it is a link
	 */
	if(EDVIsParentPath(parent, path))
	{
	    const char *s = path + STRLEN(parent);
	    while(*s == G_DIR_SEPARATOR)
		s++;
	    tar_path = g_strconcat(s, "@", NULL);
	}
	else
	{
	    tar_path = g_strconcat(path, "@", NULL);
	}
	if(tar_path == NULL)
	{
	    core->archive_last_error = "Memory allocation error.";
	    *status = -3;
	    return(*status);
	}

	/* Create a new and set up the libzip source data */
	d = EDVArchAddZipSourceDataNew();
	if(d == NULL)
	{
	    g_free(tar_path);
	    core->archive_last_error = "Memory allocation error.";
	    *status = -3;
	    return(*status);
	}
	d->core = core;
	d->status = status;
	d->zip_error[0] = ZIP_ER_OK;
	d->zip_error[1] = 0;
	d->toplevel = toplevel;
	d->show_progress = show_progress;
	d->arch_path = STRDUP(arch_path);
	d->new_paths_list_rtn = new_paths_list_rtn;
	d->cur_size = cur_size;
	d->total_size = total_size;

	d->src_path = STRDUP(path);
	d->tar_path = STRDUP(tar_path);
	d->type = EDV_OBJECT_TYPE_LINK;
	d->this_size = (gulong)stat_buf->st_size;
	memcpy(&d->stat_buf, stat_buf, sizeof(struct stat));
	d->fp = NULL;
	d->bytes_read = 0l;

	/* Check if the object already exists in the PKZip Archive */
	i = zip_name_locate(archive, tar_path, 0);

	/* Set the libzip source function */
	zsrc = zip_source_function(archive, EDVArchAddZipSourceCB, d);
	if(zsrc == NULL)
	{
	    gchar *msg = g_strdup_printf(
"Unable to set the libzip source function.\n\
\n\
%s.",
		zip_strerror(archive)
	    );
	    EDVArchAddCopyErrorMessage(core, msg);
	    g_free(msg);
	    EDVArchAddZipSourceDataDelete(d);
	    g_free(tar_path);
	    *status = -1;
	    return(*status);
	}

	/* Does the object already exist within the archive? */
	if(i > -1)
	{
	    /* Replace the object */
	    if(zip_replace(archive, i, zsrc) < 0)
	    {
		gchar *msg = g_strdup_printf(
"Unable to replace the PKZip Archive object #%i.\n\
\n\
%s.",
		    i + 1, zip_strerror(archive)
		);
		EDVArchAddCopyErrorMessage(core, msg);
		g_free(msg);
		zip_source_free(zsrc);
		g_free(tar_path);
		*status = -1;
		return(*status);
	    }
	}
	else
	{
	    /* Add the link */
	    if(zip_add(archive, tar_path, zsrc) < 0)
	    {
		gchar *msg = g_strdup_printf(
"Unable to add a new PKZip Archive object.\n\
\n\
%s.",
		    zip_strerror(archive)
		);
		EDVArchAddCopyErrorMessage(core, msg);
		g_free(msg);
		zip_source_free(zsrc);
		g_free(tar_path);
		*status = -1;
		return(*status);
	    }
	}

	g_free(tar_path);

	return(0);
}

/*
 *	Adds the object specified by path to the PKZip Archive.
 *
 *	Called by EDVArchAddPKZip().
 */
static void EDVArchAddZipIterate(
	edv_core_struct *core, gint *status,
	const gchar *arch_path, struct zip *archive,
	const gchar *password,
	const gchar *parent,		/* Parent directory of the archive */
	const gchar *path,		/* The object to add to the archive */
	GList **new_paths_list_rtn,
	gulong *cur_size, const gulong total_size,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
)
{
	mode_t m;
	struct stat stat_buf;

	if(STRISEMPTY(path))
	    return;

	/* Skip the archive itself */
	if(!strcmp((const char *)arch_path, path))
	    return;

	/* Check if this object does not exist */
	if(lstat((const char *)path, &stat_buf))
	{
	    const gint error_code = (gint)errno;
	    gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
		path, arch_path, g_strerror(error_code)
	    );
	    EDVArchAddCopyErrorMessage(core, msg);
	    g_free(msg);
	    *status = -1;
	    return;
	}

	m = stat_buf.st_mode;

/* Reports an add error to the user and queries the user to
 * continue adding objects
 */
#define REPORT_ERROR_QUERY_CONTINUE_ADDING	{	\
 if(interactive) {					\
  if(!(*yes_to_all)) {					\
   /* Query the user to continue adding */		\
   gint response;					\
   gchar *msg = g_strdup_printf(			\
"An error occured while adding the object:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
Continue adding subsequent objects?",			\
    path, arch_path					\
   );							\
   EDVPlaySoundError(core);				\
   CDialogSetTransientFor(toplevel);			\
   response = CDialogGetResponse(			\
    "Add Failed",					\
    msg,						\
    NULL,						\
    CDIALOG_ICON_ERROR,					\
    CDIALOG_BTNFLAG_YES |				\
     CDIALOG_BTNFLAG_YES_TO_ALL |			\
     CDIALOG_BTNFLAG_NO,				\
    CDIALOG_BTNFLAG_YES					\
   );							\
   g_free(msg);						\
   CDialogSetTransientFor(NULL);			\
							\
   /* Stop adding? */					\
   if((response == CDIALOG_RESPONSE_NO) ||		\
      (response == CDIALOG_RESPONSE_CANCEL) ||		\
      (response == CDIALOG_RESPONSE_NOT_AVAILABLE)	\
   )							\
   {							\
    *status = -5;					\
    return;						\
   }							\
							\
   if(response == CDIALOG_RESPONSE_YES_TO_ALL)		\
    *yes_to_all = TRUE;					\
  }							\
 }							\
}

	/* Directory? */
#ifdef S_ISDIR
	if(S_ISDIR(m))
#else
	if(FALSE)
#endif
	{
	    /* Add this directory to the PKZip Archive (but not the
	     * objects within it)
	     */
	    if(EDVArchAddZipIterateDirectory(
		core, status,
		arch_path, archive,
		password,
		parent,		/* Parent directory of the archive */
		path,		/* The object to add to the archive */
		&stat_buf,
		new_paths_list_rtn,
		cur_size, total_size,
		toplevel,
		show_progress
	    ))
	    {
		REPORT_ERROR_QUERY_CONTINUE_ADDING
	    }

	    /* Recurse into this directory? */
	    if(recursive && (*status != -4) && (*status != -5))
	    {
		/* Add the objects within this directory into the
		 * PKZip Archive
		 */
		gint i, strc;
		const gchar *name;
		gchar **strv = GetDirEntNames2((const char *)path, (int *)&strc);
		StringQSort(strv, strc);
		for(i = 0; i < strc; i++)
		{
		    name = strv[i];
		    if(name == NULL)
			continue;

		    /* Ignore current and parent directory notations */
		    if(!strcmp((const char *)name, ".") ||
		       !strcmp((const char *)name, "..")
		    )
		    {
			g_free(strv[i]);
			continue;
		    }

		    if((*status != -4) && (*status != -5))
		    {
			gchar *full_path = STRDUP(PrefixPaths(
			    (const char *)path, (const char *)name
			));
			EDVArchAddZipIterate(
			    core, status,
			    arch_path, archive,
			    password,
			    parent,
			    full_path,
			    new_paths_list_rtn,
			    cur_size, total_size,
			    toplevel,
			    show_progress, interactive, yes_to_all,
			    recursive, compression, dereference_links
			);
			g_free(full_path);
		    }
		    g_free(strv[i]);
		}
		g_free(strv);
	    }
	}
	/* Link? */
#ifdef S_ISLNK
	else if(S_ISLNK(m))
#else
	else if(FALSE)
#endif
	{
	    /* Dereference this link? */
	    if(dereference_links)
	    {
		struct stat stat_buf;

		/* Get the destination object's stats */
		if(stat((const char *)path, &stat_buf))
		{
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"Unable to add:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
%s.",
			path, arch_path, g_strerror(error_code)
		    );
	            EDVArchAddCopyErrorMessage(core, msg);
	            g_free(msg);
		    *status = -1;
		    return;
		}
		else
		{
		    const mode_t m = stat_buf.st_mode;

		    /* Destination is a directory? */
#ifdef S_ISDIR
		    if(S_ISDIR(m))
#else
		    if(FALSE)
#endif
		    {
			/* Add this directory to the PKZip Archive (but
			 * not the objects within it)
			 */
			if(EDVArchAddZipIterateDirectory(
			    core, status,
			    arch_path, archive,
			    password,
			    parent,	/* Parent directory of the archive */
			    path,	/* The object to add to the archive */
			    &stat_buf,
			    new_paths_list_rtn,
			    cur_size, total_size,
			    toplevel,
			    show_progress
			))
			{                    
			    REPORT_ERROR_QUERY_CONTINUE_ADDING
			}

			/* Recurse into this directory? */
			if(recursive && (*status != -4) && (*status != -5))
			{
			    /* Add the objects within this directory
			     * into the PKZip Archive
			     */
			    gint i, strc;
			    const gchar *name;
			    gchar **strv = GetDirEntNames2(path, &strc);
			    StringQSort(strv, strc);
			    for(i = 0; i < strc; i++)
			    {
				name = strv[i];
				if(name == NULL)
				    continue;

				/* Ignore current and parent directory notations */
				if(!strcmp((const char *)name, ".") ||
				   !strcmp((const char *)name, "..")
				)
				{
				    g_free(strv[i]);
				    continue;
				}

				if((*status != -4) && (*status != -5))
				{
				    gchar *full_path = STRDUP(PrefixPaths(
					(const char *)path, (const char *)name
				    ));
				    EDVArchAddZipIterate(
					core, status,
					arch_path, archive,
					password,
					parent,
					full_path,
					new_paths_list_rtn,
					cur_size, total_size,
					toplevel,
					show_progress, interactive, yes_to_all,
					recursive, compression, dereference_links
				    );
				    g_free(full_path);
				}
				g_free(strv[i]);
			    }
			    g_free(strv);
			}
		    }
		    /* Destination is a file? */
#ifdef S_ISREG
		    else if(S_ISREG(m))
#else
		    else if(FALSE)
#endif
		    {
			/* Add this file to the archive */
			if(EDVArchAddZipIterateFile(
			    core, status,
			    arch_path, archive,
			    password,
			    parent,	/* Parent directory of the archive */
			    path,	/* The object to add to the archive */
			    &stat_buf,
			    new_paths_list_rtn,
			    cur_size, total_size,
			    toplevel,
			    show_progress, compression
			))
			{
			    REPORT_ERROR_QUERY_CONTINUE_ADDING
			}
		    }
		    /* Destination is some other type of object
		     * (not a file, directory, or link)
		     */
		    else
		    {
			/* Ignore this object type */
		    }
		}
	    }
	    else
	    {
		/* Add this link to the archive (do not dereference) */
		if(EDVArchAddZipIterateLink(
		    core, status,
		    arch_path, archive,
		    password,
		    parent,	/* Parent directory of the archive */
		    path,	/* The object to add to the archive */
		    &stat_buf,
		    new_paths_list_rtn,
		    cur_size, total_size,
		    toplevel,
		    show_progress, compression
		))
		{
		    REPORT_ERROR_QUERY_CONTINUE_ADDING
		}
	    }
	}
	/* File? */
#ifdef S_ISREG
	else if(S_ISREG(m))
#else
	else if(FALSE)
#endif
	{
	    /* Add this file to the archive */
	    if(EDVArchAddZipIterateFile(
		core, status,
		arch_path, archive,
		password,
		parent,		/* Parent directory of the archive */
		path,		/* The object to add to the archive */
		&stat_buf,
		new_paths_list_rtn,
		cur_size, total_size,
		toplevel,
		show_progress, compression
	    ))
	    {
		REPORT_ERROR_QUERY_CONTINUE_ADDING
	    }
	}
	/* Destination is some other type of object */
	else
	{
	    /* Ignore this object type */
	}

#undef REPORT_ERROR_QUERY_CONTINUE_ADDING
}
#endif	/* HAVE_LIBZIP */

/*
 *	Add object to a PKZip Archive.
 *
 *	Inputs assumed valid.
 */
gint EDVArchAddPKZip(
	edv_core_struct *core, const gchar *arch_path,
	GList *tar_paths_list,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean recursive, const gint compression,
	const gboolean dereference_links
)
{
#ifdef HAVE_LIBZIP
	struct zip *archive;
	gint status, zip_error_code, sys_error_code, nobjs;
	gulong cur_size, total_size;
	gchar *parent;
	GList *glist;

	/* Get the parent directory of the archive, this will be the
	 * base directory in which all objects added into the PKZip
	 * archive will be relative to (have their parent paths striped
	 * of if their parent paths match this parent path)
	 */
	parent = g_dirname(arch_path);
	if(parent == NULL)
	{
	    core->archive_last_error = "Unable to obtain the parent directory of the archive.";
	    return(-1);
	}

	/* Calculate the total uncompressed size of all the objects to
	 * be added to the archive
	 */
	nobjs = 0;
	total_size = 0l;
	for(glist = tar_paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	    total_size += EDVArchAddCalculateTotalSize(
		arch_path, (const gchar *)glist->data,
		&nobjs,
		recursive, dereference_links
	    );

	/* Open the PKZip Archive for writing */
	archive = EDVOpenZipWriting(
	    arch_path,
	    &zip_error_code, &sys_error_code
	);
	if(archive == NULL)
	{
	    gchar *msg, err_msg[1024];
	    zip_error_to_str(
		err_msg, sizeof(err_msg),
		zip_error_code, sys_error_code
	    );
	    msg = g_strdup_printf(
"Unable to open the PKZip Archive for writing:\n\
\n\
    %s\n\
\n\
%s.",
		arch_path, err_msg
	    );
	    EDVArchAddCopyErrorMessage(core, msg);
	    g_free(msg);
	    g_free(parent);
	    return(-1);
	}

	/* Iterate through the list of objects to add */
	status = 0;
	cur_size = 0l;
	for(glist = tar_paths_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    EDVArchAddZipIterate(
		core, &status,
		arch_path, archive,
		password,
		parent,
		(const gchar *)glist->data,	/* Full path to the object to be added */
		new_paths_list_rtn,
		&cur_size, total_size,
		toplevel,
		show_progress, interactive, yes_to_all,
		recursive, compression, dereference_links
	    );
	    if((status == -4) || (status == -5))
		break;
	}

	/* Write/flush changes to the PKZip Archive and close it
	 *
	 * This is where libzip actually adds the object to the archive
	 * by calling our EDVArchAddZipSourceCB() write callback function
	 *
	 * The progress will be reported and the status and cur_size
	 * will be updated by calls to EDVArchAddZipSourceCB() made in
	 * zip_close()
	 */
	if(zip_close(archive))
	{
	    /* If not interrupted during the write and close then
	     * set the status to indicate error, otherwise the error
	     * was that the user aborted
	     */
	    if((status == 0) || (status != -4))
	    {
		gchar *msg = g_strdup_printf(
"Unable to close the PKZip Archive:\n\
\n\
    %s\n\
\n\
%s.",
		    arch_path, zip_strerror(archive)
		);
		EDVArchAddCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }
	}

	g_free(parent);

	return(status);
#else
	const gchar *prog_zip = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_ZIP
	);
	FILE *fp;
	struct stat arch_path_stat_buf;
	gint status, arch_path_stat_result;
	gint p = 0;
	gchar	*parent_dir = NULL,
		*pwd = NULL,
		*cmd = NULL,
		*stdout_path = NULL,
		*stderr_path = NULL;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(parent_dir);				\
 g_free(cmd);					\
 g_free(stdout_path);				\
 g_free(stderr_path);				\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  EDVSetCWD(pwd);				\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record the previous working directory and set the new
	 * working directory
	 */
	pwd = EDVGetCWD();
	parent_dir = g_dirname(arch_path);
	EDVSetCWD(parent_dir);

	/* If the PKZip Archive exists and is empty then it must
	 * be removed first before adding objects to it
	 */
	arch_path_stat_result = stat((const char *)arch_path, &arch_path_stat_buf);
	if(!arch_path_stat_result)
	{
	    if(arch_path_stat_buf.st_size == 0l)
		UNLINK(arch_path);
	}

	status = 0;

	/* Format add object to archive command */
	if(cmd == NULL)
	{
	    gchar *s;
	    const gchar *tpath;
	    GList *glist;

	    cmd = g_strdup_printf(
		"\"%s\" -%i %s %s \"%s\"",
		prog_zip,
		CLIP(compression * 9 / 100, 0, 9),
		recursive ? "-r" : "",
		dereference_links ? "" : "-y",
		arch_path
	    );
	    for(glist = tar_paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		tpath = (const gchar *)glist->data;
		if(STRISEMPTY(tpath))
		    continue;

		/* Seek tpath to the relative path as needed */
		if(EDVIsParentPath(parent_dir, tpath))
		{
		    tpath += STRLEN(parent_dir);
		    while(*tpath == G_DIR_SEPARATOR)
			tpath++;
		}

		s = g_strconcat(
		    cmd, " \"", tpath, "\"", NULL
		);
		g_free(cmd);
		cmd = s;
	    }
	    if(cmd == NULL)
	    {
		core->archive_last_error = "Unable to generate the add command.";
		CLEANUP_RETURN(-1);
	    }

	    /* Generate the output file paths */
	    stdout_path = EDVTmpName(NULL);
	    stderr_path = EDVTmpName(NULL);

	    /* Execute the add object to archive command */
	    p = (gint)ExecOE(
		(const char *)cmd,
		(const char *)stdout_path,
		(const char *)stderr_path
	    );
	    if(p <= 0)
	    {
		core->archive_last_error = "Unable to execute the add command.";
		CLEANUP_RETURN(-1);
	    }

	    /* Delete the command */
	    g_free(cmd);
	    cmd = NULL;
	}

	/* Open the output file for reading */
	fp = fopen((const char *)stdout_path, "rb");
	if(fp != NULL)
	{
	    gint buf_pos = 0;
	    gchar buf[10000];
	    gboolean need_break = FALSE;

	    /* Begin reading the output file */
	    while(TRUE)
	    {
		/* Update progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdateUnknown(
			NULL, NULL, NULL, NULL, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			INTERRUPT(p);
			p = 0;

			status = -4;
			break;
		    }
		}

		/* Check if there is new data to be read from the
		 * output file
		 */
		if(FPending(fp))
		{
		    gint c;
		    gboolean got_complete_line = FALSE;

		    /* Copy all available data from the current output
		     * file position to its end to the line buffer buf
		     */
		    while(TRUE)
		    {
			c = (gint)fgetc(fp);
			if((int)c == EOF)
			{
			    clearerr(fp);
			    break;
			}

			if(ISCR(c))
			{
			    got_complete_line = TRUE;
			    if(buf_pos < sizeof(buf))
				buf[buf_pos] = '\0';
			    else
				buf[sizeof(buf) - 1] = '\0';
			    buf_pos = 0;
			    break;
			}

			if(buf_pos < sizeof(buf))
			{
			    buf[buf_pos] = c;
			    buf_pos++;
			}
		    }
		    if(got_complete_line)
		    {
			/* Update progress dialog label */
			gchar *s = buf, *s2;
			const gchar *added_path;

			/* Seek s past the first prefix */
			while(ISBLANK(*s))
			    s++;
			while(!ISBLANK(*s) && (*s != '\0'))
			    s++;
			while(ISBLANK(*s))
			    s++;

			/* Cap the space character after path */
			s2 = strchr(s, ' ');
			if(s2 != NULL)
			    *s2 = '\0';

			added_path = s;

			/* Append path to the list of new paths added
			 * to the archive
			 */
			if(new_paths_list_rtn != NULL)
			    *new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn, STRDUP(added_path)
			    );

			/* Update the progress dialog's label? */
			if(show_progress && !STRISEMPTY(added_path))
			{
			    gchar	*p1 = EDVShortenPath(
				added_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*p2 = EDVShortenPath(
				arch_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Agregar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Addition:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hinzufgen:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aggiunta:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoegen:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicionar:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfying:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Adding:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
				, p1, p2
			    );
			    EDVArchAddMapProgressDialogUnknown(
				msg, toplevel, FALSE
			    );
			    g_free(msg);
			    g_free(p1);
			    g_free(p2);
			}
			continue;
		    }
		}

		if(need_break)
		    break;

		/* Check if the add process has exited, if it has then
		 * we set need_break to TRUE. Which will be tested on
		 * the next loop if there is still no more data to be
		 * read
		 */
		if(!ExecProcessExists(p))
		    need_break = TRUE;

		usleep(8000);
	    }

	    fclose(fp);
	}

	/* Remove the output files */
	UNLINK(stdout_path);
	UNLINK(stderr_path);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
#endif	/* !HAVE_LIBZIP */
}
