#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "progressdialog.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_list_cb.h"
#include "edv_date.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_history.h"
#include "edv_history_list.h"
#include "history_win.h"
#include "endeavour2.h"
#include "edv_cb.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_trash_20x20.xpm"
#include "images/icon_search_20x20.xpm"
#include "images/icon_executable_20x20.xpm"
#include "images/icon_new_20x20.xpm"
#include "images/icon_copy_file_20x20.xpm"
#include "images/icon_move_file_20x20.xpm"
#include "images/icon_link2_20x20.xpm"
#include "images/icon_chmod_20x20.xpm"
#include "images/icon_owned_20x20.xpm"
#include "images/icon_time_stamp_20x20.xpm"
#include "images/icon_download_file_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_recover_20x20.xpm"
#include "images/icon_purge_20x20.xpm"
#include "images/icon_archive_add_20x20.xpm"
#include "images/icon_archive_extract_20x20.xpm"
#include "images/icon_aid_20x20.xpm"

#include "images/icon_object_misc_20x20.xpm"
#include "images/icon_object_misc_32x32.xpm"

#include "images/icon_error_20x20.xpm"
#include "images/icon_error_32x32.xpm"

#include "images/icon_reload_20x20.xpm"
#include "images/icon_clear_20x20.xpm"
#include "images/icon_save_as_20x20.xpm"
#include "images/icon_save_32x32.xpm"
#include "images/icon_close_20x20.xpm"

#include "images/icon_trace_48x48.xpm"


/* Callbacks */
static void EDVHistoryListItemDestroyCB(gpointer data);
static gint EDVHistoryListProgressCB(
	gpointer data, gulong pos, gulong max
);

static gint EDVHistoryWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint EDVHistoryWinKeyEventCB(
	GtkWidget *widget, GdkEventKey *key, gpointer data
);
static gint EDVHistoryWinButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
static void EDVHistoryWinSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void EDVHistoryWinUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static gint EDVHistoryWinCommentsTextEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVHistoryWinFindCB(
	GtkWidget *widget, gpointer data
);
static void EDVHistoryWinRefreshCB(
	GtkWidget *widget, gpointer data
);
static void EDVHistoryWinSaveAsCB(
	GtkWidget *widget, gpointer data
);
static void EDVHistoryWinClearCB(
	GtkWidget *widget, gpointer data
);
static void EDVHistoryWinCloseCB(
	GtkWidget *widget, gpointer data
);

void EDVHistoryWinWriteProtectChangedCB(
	edv_history_win_struct *lw, gboolean state
);
void EDVHistoryWinHistoryAddedCB(
	edv_history_win_struct *lw,
	gint h_num, edv_history_struct *h
);
void EDVHistoryWinHistoryRemovedCB(
	edv_history_win_struct *lw,
	gint h_num
);
void EDVHistoryWinReconfiguredNotifyCB(
	edv_history_win_struct *lw
);


/* Operations */
void EDVHistoryWinClear(edv_history_win_struct *lw);
static gint EDVHistoryWinSave(
	edv_history_win_struct *lw, const gchar *filename,
	const gboolean append
);
void EDVHistoryWinSaveAs(edv_history_win_struct *lw);

void EDVHistoryWinUpdateDisplay(
	edv_history_win_struct *lw, edv_history_struct *h
);
static void EDVHistoryWinSetRow(
	edv_history_win_struct *lw,
	const gint row, edv_history_struct *h
);
void EDVHistoryWinUpdateList(edv_history_win_struct *lw);

void EDVHistoryWinSyncConfiguration(edv_history_win_struct *lw);

void EDVHistoryWinListStylesRegenerate(edv_history_win_struct *lw);

edv_history_win_struct *EDVHistoryWinNew(edv_core_struct *core);
void EDVHistoryWinUpdateMenus(edv_history_win_struct *lw);
void EDVHistoryWinSetBusy(
	edv_history_win_struct *lw, const gboolean is_busy
);
gboolean EDVHistoryWinIsMapped(edv_history_win_struct *lw);
void EDVHistoryWinMap(edv_history_win_struct *lw);
void EDVHistoryWinUnmap(edv_history_win_struct *lw);
void EDVHistoryWinDelete(edv_history_win_struct *lw);


#define LISTWIN_WIDTH			640
#define LISTWIN_HEIGHT			520
#define LISTWIN_DISPLAY_HEIGHT		165

#define LISTWIN_TITLE			"History"


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	GtkCList item "destroy" signal callback.
 */
static void EDVHistoryListItemDestroyCB(gpointer data)
{
	EDVHistoryDelete(EDV_HISTORY(data));
}


/*
 *	Progress callback.
 */
static gint EDVHistoryListProgressCB(
	gpointer data, gulong pos, gulong max
)
{
	edv_history_win_struct *lw = EDV_HISTORY_WIN(data);
	if(lw == NULL)
	    return(-1);

	if(ProgressDialogIsQuery())
	{
	    if(ProgressDialogStopCount() > 0)
		return(-1);

	    if(max > 0l)
		ProgressDialogUpdate(
		    NULL, NULL, NULL, NULL,
		    CLIP((gfloat)pos / (gfloat)max, 0.0f, 1.0f),
		    EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
	}

	return(0);
}


/*
 *	GtkWindow "delete_event" signal callback.
 */
static gint EDVHistoryWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_history_win_struct *lw = EDV_HISTORY_WIN(data);
	if(lw == NULL)
	    return(FALSE);

	if(lw->freeze_count > 0)
	    return(FALSE);

	EDVHistoryWinCloseCB(widget, lw);

	return(TRUE);
}

/*
 *	History list window "key_press_event" or "key_release_event"
 *	signal callback.
 */
static gint EDVHistoryWinKeyEventCB(
	GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	guint keyval, state;
	gboolean press;
	edv_core_struct *core;
	edv_history_win_struct *lw = EDV_HISTORY_WIN(data);
	if(lw == NULL)
	    return(status);

	if(lw->freeze_count > 0)
	    return(status);

	core = lw->core;
	etype = key->type;
	press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

/* Macro to emit a signal stop for a key press or release depending
 * on the current event's type
 */
#define STOP_SIGNAL_EMIT(_w_)	{		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(_w_),				\
  press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	if(lw->history_clist == widget)
	{
	    GtkCList *clist = GTK_CLIST(widget);
	    gint row = EDVCListGetSelectedLast(clist, NULL);

	    /* Handle by key value */
	    switch(keyval)
	    {
#if 0
	      case GDK_Return:
	      case GDK_KP_Enter:
	      case GDK_ISO_Enter:
	      case GDK_3270_Enter:
		if(press)
		{
		    EDVHistoryWinEditCB(NULL, lw);
		}
		STOP_SIGNAL_EMIT(widget);
		status = TRUE;
		break;
#endif

#if 0
	      case GDK_Delete:
		if(press)
		{
		    EDVHistoryWinRemoveCB(NULL, lw);
		}
		STOP_SIGNAL_EMIT(widget);
		status = TRUE;
		break;
#endif
	      case GDK_space:
	      case GDK_KP_Space:
		row = clist->focus_row;
		if((row >= 0) && (row < clist->rows) && press)
		{
		    gboolean already_selected = (g_list_find(
			clist->selection, (gpointer)row
		    ) != NULL) ? TRUE : FALSE;
		    gtk_clist_freeze(clist);
		    if(already_selected)
			gtk_clist_unselect_row(clist, row, 0);
		    else
			gtk_clist_select_row(clist, row, 0);
		    gtk_clist_thaw(clist);
		}
		STOP_SIGNAL_EMIT(widget);
		status = TRUE;
		break;
	    }
	}

	return(status);
#undef STOP_SIGNAL_EMIT
}

/*
 *	History list window "button_press_event" or
 *	"button_release_event" signal callback.
 */
static gint EDVHistoryWinButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;
	edv_history_win_struct *lw = EDV_HISTORY_WIN(data);
	if(lw == NULL)
	    return(status);

	if(lw->freeze_count > 0)
	    return(status);

	core = lw->core;
	cfg_list = core->cfg_list;

	if(widget == lw->history_clist)
	{
	    gint row, column;
	    GtkCList *clist = GTK_CLIST(widget);

	    if(clist->clist_window != ((GdkEventAny *)button)->window)
		return(status);

	    /* Find row and column based on given coordinates */
	    if(!gtk_clist_get_selection_info(
		clist, (gint)button->x, (gint)button->y, &row, &column
	    ))
	    {
		row = -1;
		column = 0;
	    }

	    /* Handle by event type */
	    switch((gint)button->type)
	    {
	      case GDK_BUTTON_PRESS:
		/* Handle by button number */
		switch(button->button)
		{
		  case GDK_BUTTON3:	/* Map right-click menu */
		    /* Select row before mapping menu? */
		    if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
		       (row >= 0) && (row < clist->rows)
		    )
		    {
			gtk_clist_freeze(clist);
			gtk_clist_select_row(clist, row, column);
			gtk_clist_thaw(clist);
		    }

		    /* Map right-click menu */
		    gtk_menu_popup(
			GTK_MENU(lw->menu), NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );

		    status = TRUE;
		    break;
		}
		break;
	    }
	}

	return(status);
}

/*
 *	History list window "select_row" signal callback.
 */
static void EDVHistoryWinSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_core_struct *core;
	edv_history_win_struct *lw = EDV_HISTORY_WIN(data);
	if(lw == NULL)
	    return;

	if(lw->freeze_count > 0)
	    return;

	core = lw->core;

	/* Check which clist this event is for */
	if(GTK_WIDGET(clist) == lw->history_clist)
	{
	    /* Check if selected row is fully visible, if not then
	     * adjust scroll position to try and make it visible
	     */
	    if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    clist,
		    row, -1,	/* Row, column */
		    0.5f, 0.0f	/* Row, column */
		);

	    /* Update displayed history */
	    EDVHistoryWinUpdateDisplay(
		lw,
		EDV_HISTORY(gtk_clist_get_row_data(clist, row))
	    );

	    EDVHistoryWinUpdateMenus(lw);
	}
}

/*
 *	History list window "unselect_row" signal callback.
 */
static void EDVHistoryWinUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_core_struct *core;
	edv_history_win_struct *lw = EDV_HISTORY_WIN(data);
	if(lw == NULL)
	    return;

	if(lw->freeze_count > 0)
	    return;

	core = lw->core;

	/* Check which clist this event is for */
	if(GTK_WIDGET(clist) == lw->history_clist)
	{
	    EDVHistoryWinUpdateDisplay(lw, NULL);
	    EDVHistoryWinUpdateMenus(lw);
	}
}

/*
 *	Comments GtkText event signal callback.
 */
static gint EDVHistoryWinCommentsTextEventCB(
        GtkWidget *widget, GdkEvent *event, gpointer data
)
{
        gint status = FALSE;
        GdkEventButton *button;
        GtkText *text;
	edv_history_win_struct *lw = EDV_HISTORY_WIN(data);
        if((widget == NULL) || (event == NULL) || (lw == NULL))
            return(status);

        if(lw->freeze_count > 0)
            return(status);

        text = GTK_TEXT(widget);

        switch((gint)event->type)
        {
          case GDK_BUTTON_PRESS:
            button = (GdkEventButton *)event;
            switch(button->button)
            {
              case GDK_BUTTON4:
                /* Scroll up */
                if(text->vadj != NULL)
                {
                    GtkAdjustment *adj = text->vadj;
                    const gfloat inc = MAX(
                        (0.25f * adj->page_size),
                        adj->step_increment
                    );
                    gfloat v = adj->value - inc;
                    if(v > (adj->upper - adj->page_size))
                        v = adj->upper - adj->page_size;
                    if(v < adj->lower)
                        v = adj->lower;
                    gtk_adjustment_set_value(adj, v);
                }
                /* Need to mark the GtkText button as 0 or else it will
                 * keep marking
                 */
                text->button = 0;
                gtk_grab_remove(widget);
                gtk_signal_emit_stop_by_name(
                    GTK_OBJECT(widget), "button_press_event"
                );
                status = TRUE;
                break;
              case GDK_BUTTON5:
                /* Scroll down */
                if(text->vadj != NULL)
                {
                    GtkAdjustment *adj = text->vadj;
                    const gfloat inc = MAX(
                        (0.25f * adj->page_size),
                        adj->step_increment
                    );
                    gfloat v = adj->value + inc;
                    if(v > (adj->upper - adj->page_size))
                        v = adj->upper - adj->page_size;
                    if(v < adj->lower)
                        v = adj->lower;
                    gtk_adjustment_set_value(adj, v);
                }
                /* Need to mark the GtkText button as 0 or else it will
                 * keep marking
                 */
                text->button = 0;
                gtk_grab_remove(widget);
                gtk_signal_emit_stop_by_name(
                    GTK_OBJECT(widget), "button_press_event"
                );
                status = TRUE;
                break;
            }
            break;

          case GDK_BUTTON_RELEASE:
            button = (GdkEventButton *)event;
            switch(button->button)
            {
              case GDK_BUTTON4:
                /* Need to mark the GtkText button as 0 or else it will
                 * keep marking
                 */
                text->button = 0;
                gtk_grab_remove(widget);
                gtk_signal_emit_stop_by_name(
                    GTK_OBJECT(widget), "button_release_event"
                );
                status = TRUE;
                break;
              case GDK_BUTTON5:
                /* Need to mark the GtkText button as 0 or else it will
                 * keep marking
                 */
                text->button = 0;
                gtk_grab_remove(widget);
                gtk_signal_emit_stop_by_name(
                    GTK_OBJECT(widget), "button_release_event"
                );
                status = TRUE;
                break;
            }
            break;
        }

        return(status);
}


/*
 *	History find entry "activate" signal callback.
 */
static void EDVHistoryWinFindCB(
	GtkWidget *widget, gpointer data
)
{
	gint find_iterations = 0;
	gint row, nrows, column, ncolumns, sel_row;
	const gchar *expression;
	GtkEntry *entry;
	GtkCList *clist;

	gchar *cell_text;
	guint8 spacing;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	edv_history_win_struct *lw = EDV_HISTORY_WIN(data);
	if(lw == NULL)
	    return;

	if(lw->freeze_count > 0)
	    return;

	entry = GTK_ENTRY(lw->find_entry);
	clist = GTK_CLIST(lw->history_clist);
	ncolumns = clist->columns;
	nrows = clist->rows;

	/* Get string to match for from entry */
	expression = gtk_entry_get_text(entry);
	if(STRISEMPTY(expression))
	    return;

	/* Get selected row + 1 or 0 if none */
	sel_row = EDVCListGetSelectedLast(clist, NULL);
	if(sel_row < 0)
	    sel_row = 0;
	else
	    sel_row++;

	/* Iterate through rows, checking each Type column to see if
	 * there are any partial patches
	 */
	do
	{
	    /* Iterate from selected row to end */
	    for(row = sel_row; row < nrows; row++)
	    {
		/* Iterate through all cells on this row */
		for(column = 0; column < ncolumns; column++)
		{
		    /* Begin fetching current cell's text by its type */
		    cell_text = NULL;
		    switch(gtk_clist_get_cell_type(clist, row, column))
		    {
		      case GTK_CELL_TEXT:
			gtk_clist_get_text(clist, row, column, &cell_text);
			break;
		      case GTK_CELL_PIXTEXT:
			gtk_clist_get_pixtext(
			    clist, row, column, &cell_text,
			    &spacing, &pixmap, &mask
			);
			break;
		      case GTK_CELL_PIXMAP:
		      case GTK_CELL_WIDGET:
		      case GTK_CELL_EMPTY:
			break;
		    }
		    /* Got cell text? */
		    if(!STRISEMPTY(cell_text))
		    {
			/* Find string found inside cell text string? */
			if(strcasestr(
			    (const char *)cell_text,
			    (const char *)expression
			) != NULL)
			    break;
		    }
		}
		/* If column iteration broke before all columns were
		 * iterated through then that implies a match was made
		 */
		if(column < ncolumns)
		    break;
	    }
	    /* Got match? */
	    if(row < nrows)
		break;
	    else
		find_iterations++;

	    /* Reset sel_row to 0 so that find starts at beginning */
	    sel_row = 0;

	} while(find_iterations < 2);

	/* Got match? */
	if(row < nrows)
	{
	    /* Select new matched row */
	    gtk_clist_freeze(clist);
	    gtk_clist_unselect_all(clist);
	    gtk_clist_select_row(clist, row, 0);
	    gtk_clist_thaw(clist);
	}
}

/*
 *	History refresh button callback.
 */
static void EDVHistoryWinRefreshCB(
	GtkWidget *widget, gpointer data
)
{
	gint last_row;
	gfloat last_x, last_y;
	GtkAdjustment *hadj, *vadj;
	GtkCList *clist;
	edv_history_win_struct *lw = EDV_HISTORY_WIN(data);
	if(lw == NULL)
	    return;

	if(lw->freeze_count > 0)
	    return;

	clist = GTK_CLIST(lw->history_clist);

	EDVHistoryWinSetBusy(lw, TRUE);

	/* Record last list scroll position */
	hadj = clist->hadjustment;
	last_x = (hadj != NULL) ? hadj->value : 0.0f;
	vadj = clist->vadjustment;
	last_y = (vadj != NULL) ? vadj->value : 0.0f;
	last_row = EDVCListGetSelectedLast(clist, NULL);

	/* Update the list */
	EDVHistoryWinUpdateList(lw);

	/* Scroll back to the original position */
	EDVScrollCListToPosition(clist, last_x, last_y);
	gtk_clist_select_row(clist, last_row, -1);

	EDVHistoryWinUpdateMenus(lw);

	EDVHistoryWinSetBusy(lw, FALSE);
}

/*
 *	History list window save button callback.
 */
static void EDVHistoryWinSaveAsCB(
	GtkWidget *widget, gpointer data
)
{
	edv_history_win_struct *lw = EDV_HISTORY_WIN(data);
	if(lw == NULL)
	    return;

	if(lw->freeze_count > 0)
	    return;

	EDVHistoryWinSaveAs(lw);
}

/*
 *	History list window clear callback.
 */
static void EDVHistoryWinClearCB(
	GtkWidget *widget, gpointer data
)
{
	edv_history_win_struct *lw = EDV_HISTORY_WIN(data);
	if(lw == NULL)
	    return;

	if(lw->freeze_count > 0)
	    return;

	EDVHistoryWinClear(lw);
}

/*
 *	History list window close button callback.
 */
static void EDVHistoryWinCloseCB(
	GtkWidget *widget, gpointer data
)
{
	edv_history_win_struct *lw = EDV_HISTORY_WIN(data);
	if(lw == NULL)
	    return;

	if(lw->freeze_count > 0)
	    return;

	EDVHistoryWinSyncConfiguration(lw);
	EDVHistoryWinUnmap(lw);
}


/*
 *	Write protect changed signal callback.
 */
void EDVHistoryWinWriteProtectChangedCB(
	edv_history_win_struct *lw, gboolean state
)
{
	if(!EDVHistoryWinIsMapped(lw))
	    return;

	EDVHistoryWinUpdateMenus(lw);
}

/*
 *	History event added signal callback.
 */
void EDVHistoryWinHistoryAddedCB(
	edv_history_win_struct *lw,
	gint h_num, edv_history_struct *h
)
{
	gint i, row, ncolumns;
	gchar **strv;
	GtkAdjustment *adj;
	GtkCList *clist;

	if(!EDVHistoryWinIsMapped(lw))
	    return;

	clist = GTK_CLIST(lw->history_clist);
	ncolumns = MAX(clist->columns, 1);

	gtk_clist_freeze(clist);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	/* Append row? */
	if(h_num < 0)
	    row = gtk_clist_append(clist, strv);
	else
	    row = gtk_clist_insert(clist, h_num, strv);
	if(row > -1)
	{
	    gint width;
	    GtkCListColumn *column;

	    EDVHistoryWinSetRow(lw, row, h);

	    for(i = 0; i < ncolumns; i++)
	    {
		column = &clist->column[i];
		if(!column->visible)
		    continue;

		width = gtk_clist_optimal_column_width(clist, i);
		gtk_clist_set_column_width(clist, i, width);
	    }
	}

	gtk_clist_thaw(clist);

	/* Scroll to end */
	adj = clist->vadjustment;
	if((adj != NULL) && (h_num < 0))
	    gtk_adjustment_set_value(
		adj,
		MAX(
		    (adj->upper - adj->page_size),
		    adj->lower
		)
	    );

	EDVHistoryWinUpdateMenus(lw);

	/* Delete the row cell values */
	g_free(strv);
}

/*
 *	History event removed signal callback.
 */
void EDVHistoryWinHistoryRemovedCB(
	edv_history_win_struct *lw,
	gint h_num
)
{
	gint row;
	GtkCList *clist;

	if(!EDVHistoryWinIsMapped(lw))
	    return;

	clist = GTK_CLIST(lw->history_clist);

	gtk_clist_freeze(clist);

	/* Remove last row? */
	if(h_num < 0)
	{
	    row = clist->rows - 1;
	    if(row > -1)
		gtk_clist_remove(clist, row);
	}
	/* Row to remove in bounds? */
	else if(h_num < clist->rows)
	{
	    row = h_num;
	    gtk_clist_remove(clist, row);
	}

	gtk_clist_thaw(clist);

	EDVHistoryWinUpdateMenus(lw);
}

/*
 *	Reconfigured notify signal callback.
 */
void EDVHistoryWinReconfiguredNotifyCB(
	edv_history_win_struct *lw
)
{
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkWidget *w;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(lw == NULL)
	    return;

	core = lw->core;
	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	/* Update the RC styles */
	w = lw->toplevel;
	if((w != NULL) && (standard_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, standard_rcstyle);
	w = lw->history_clist;
	if((w != NULL) && (lists_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, lists_rcstyle);
	EDVHistoryWinListStylesRegenerate(lw);

	EDVHistoryWinUpdateMenus(lw);
}


/*
 *	Clears the history.
 */
void EDVHistoryWinClear(edv_history_win_struct *lw)
{
	gint response;
	const gchar *s;
	GtkWidget *toplevel;
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(lw == NULL)
	    return;

	toplevel = lw->toplevel;
	core = lw->core;
	cfg_list = core->cfg_list;

	EDVHistoryWinSetBusy(lw, TRUE);

	/* Confirm clear */
	lw->freeze_count++;
	EDVPlaySoundQuestion(core);
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Claro",
"Usted est seguro que usted quiere borrar todos acontecimientos\n\
de la historia?",
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer Effacer",
"Etes-vous sr que vous voulez effacer tous vnements de l'historique?",
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie Klar",
"Sind sie sicher sie alles geschichte ereignisse wollen lschen?",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Chiaro",
"Lei sono sicuro che lei vuole cancellare tutta gli avvenimenti\n\
di storia?",
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Helder",
"Bent u zeker u alle geschiedenis gebeurtenissen wil schrappen?",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Claro",
"Esto seguro quer anular todos acontecimentos de histria?",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Klar",
"Er De sikker de stryker all historiebegivenheter?",
#else
"Confirm Clear",
"Are you sure you want to delete all history events?",
#endif
	    NULL,
	    CDIALOG_ICON_QUESTION,
	    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
	    CDIALOG_BTNFLAG_YES
	);
	CDialogSetTransientFor(NULL);
	lw->freeze_count--;

	if(response != CDIALOG_RESPONSE_YES)
	{
	    EDVHistoryWinSetBusy(lw, FALSE);
	    return;
	}

	/* Begin clearing */

	/* Delete the history events file */
	s = EDV_GET_S(EDV_CFG_PARM_FILE_HISTORY);
	if(!STRISEMPTY(s))
	    unlink((const char *)s);

	EDVPlaySoundCompleted(core);

	/* Update the list */
	EDVHistoryWinUpdateList(lw);
	EDVHistoryWinUpdateDisplay(lw, NULL);
	EDVHistoryWinUpdateMenus(lw);

	EDVHistoryWinSetBusy(lw, FALSE);
}

/*
 *	Saves the current history list to file.
 *
 *	If append is TRUE then the history list will be appended to
 *	the specified file.
 */
static gint EDVHistoryWinSave(
	edv_history_win_struct *lw, const gchar *filename,
	const gboolean append
)
{
	FILE *fp;
	gboolean aborted;
	gint row, nrows;
	GtkCList *clist;
	edv_history_struct *h;

	if((lw == NULL) || STRISEMPTY(filename))
	    return(-2);

	clist = GTK_CLIST(lw->history_clist);
	nrows = clist->rows;

	/* Open history file for writing */
	if(append)
	    fp = fopen((const char *)filename, "ab");
	else
	    fp = fopen((const char *)filename, "wb");
	if(fp == NULL)
	    return(-1);

	aborted = FALSE;

	if(EDVHistoryListProgressCB(lw, 0l, (gulong)nrows))
	    aborted = TRUE;

	/* Iterate through each row, saving the the information of
	 * each cell on each row
	 */
	if(!aborted)
	{
	    for(row = 0; row < nrows; row++)
	    {
		if(EDVHistoryListProgressCB(
		    lw,
		    (gulong)(row + 1),
		    (gulong)nrows
		))
		{
		    aborted = TRUE;
		    break;
		}

		h = EDV_HISTORY(gtk_clist_get_row_data(clist, row));

		EDVHistoryAppendToStream(fp, h);
	    }
	}

	/* Close the history file */
	fclose(fp);

	if(!aborted)
	    EDVHistoryListProgressCB(lw, (gulong)nrows, (gulong)nrows);

	return(0);
}

/*
 *	Prompts the user to save the history to an alternate file.
 */
void EDVHistoryWinSaveAs(edv_history_win_struct *lw)
{
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	edv_core_struct *core;

	if((lw == NULL) || FileBrowserIsQuery())
	    return;

	toplevel = lw->toplevel;
	core = lw->core;

	EDVHistoryWinSetBusy(lw, TRUE);

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    ".log", "Log files"
	);
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the file to save to */
	lw->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Salve La Historia Como",
"Salve",
"Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
"Enregister l'historique sous",
"Epargner",
"Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
"Auer Geschichte Als",
"Sparen",
"Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Risparmiare La Storia Come",
"Risparmiare",
"Annullare",
#elif defined(PROG_LANGUAGE_DUTCH)
"Behalve Geschiedenis Als",
"Red",
"Annuleer",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Exceto Histria Como",
"Poupe",
"Cancelamento",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Untatt Historie Som",
"Untatt",
"Kanseller",
#else
"Save History As",
"Save",
"Cancel",
#endif
	    NULL,               /* Use last path */
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
	lw->freeze_count--;

	/* Got user response? */
	if(response)
	{
	    const gchar *new_path = (npaths > 0) ?
		paths_list[0] : NULL;
	    const gchar *file_type = (ftype_rtn != NULL) ?
		ftype_rtn->ext : NULL;

	    while(!STRISEMPTY(new_path) && !STRISEMPTY(file_type))
	    {
		struct stat stat_buf, lstat_buf;
		gboolean object_exists = FALSE;
		gchar *msg;

		/* File already exists? */
		if(!stat((const char *)new_path, &stat_buf))
		{
		    gboolean need_break = FALSE;
		    gint response;
		    gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Escriba para reemplazar existir el archivo:\n\
\n\
    %s"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ecraser le fichier existant:\n\
\n\
    %s"
#elif defined(PROG_LANGUAGE_GERMAN)
"berschreiben sie existierend ablegt:\n\
\n\
    %s"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Sovrascrivere il file di esistere:\n\
\n\
    %s"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beschrijf bestaad dossier:\n\
\n\
    %s"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Overwrite existe arquivo:\n\
\n\
    %s"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Overskriv finnes arkiv:\n\
\n\
    %s"
#else
"Overwrite existing file:\n\
\n\
    %s"
#endif
			, new_path
		    );

		    EDVPlaySoundWarning(core);
		    CDialogSetTransientFor(toplevel);
		    response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Escriba Para Reemplazar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer craser"
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie berschreibt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Sovrascrivere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Beschrijft"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Overwrite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Overskriver"
#else
"Confirm Overwrite"
#endif
			, msg,
			NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_NO
		    );
		    g_free(msg);
		    CDialogSetTransientFor(NULL);
		    switch(response)
		    {
		      case CDIALOG_RESPONSE_YES:
		      case CDIALOG_RESPONSE_YES_TO_ALL:
		      case CDIALOG_RESPONSE_OK:
			break;
		      default:
			need_break = TRUE;
			break;
		    }

		    /* Mark that the object we are saving to currently
		     * exists
		     */
		    object_exists = TRUE;

		    if(need_break)
			break;
		}


		/* Format progress dialog message */
		msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Salvar la historia para archivar:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Enregister l'historique sous:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Sparende Geschichte abzulegen:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La storia di risparmio di schedare:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Reddende geschiedenis te archiveren:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Que poupando histria arquivar:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sparende historie arkivere:\n\
\n\
    %s\n"
#else
"Saving history to file:\n\
\n\
    %s\n"
#endif
		    , new_path
		);

		/* Map the progress dialog */
		ProgressDialogSetTransientFor(toplevel);
		ProgressDialogMap(
		    "Saving History",
		    msg,
		    (const guint8 **)icon_save_32x32_xpm,
		    "Stop"
		);

		g_free(msg);

		gdk_flush();

		/* Begin saving history using a format based on the
		 * selected extension
		 */
#if 0
		if(!strcmp((const char *)file_type, ".log"))
#else
		if(TRUE)
#endif
		{
		    EDVHistoryWinSave(lw, new_path, TRUE);
		}

		/* Notify about the save */
		if(!lstat((const char *)new_path, &lstat_buf))
		{
		    if(object_exists)
			EDVObjectModifiedEmit(
			    core, new_path, new_path, &lstat_buf
			);
		    else
			EDVObjectAddedEmit(
			    core, new_path, &lstat_buf
			);
		}

		break;	/* Break since this level is one big while() loop */
	    }
	}	/* Got user response? */

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	/* Reset due to possible file related change */
	FileBrowserReset();

	EDVHistoryWinUpdateMenus(lw);

	EDVHistoryWinSetBusy(lw, FALSE);
}


/*
 *	Updates the displayed history event.
 *
 *	The h specifies the history event to display for or NULL to
 *	clear the display.
 */
void EDVHistoryWinUpdateDisplay(
	edv_history_win_struct *lw, edv_history_struct *h
)
{
	const gint	border_major = 5,
			border_minor = 2;
	edv_history_type type;
	edv_date_relativity date_relativity;
	const gchar *date_format;
	GtkRcStyle *rcstyle;
	GtkWidget *w, *parent, *parent2, *parent3, *parent4;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(lw == NULL)
	    return;

	core = lw->core;
	cfg_list = core->cfg_list;
	date_relativity = (edv_date_relativity)EDV_GET_I(
	    EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);

	/* Get the previous display client GtkVBox and destroy it */
	w = lw->display_client;
	if(w != NULL)
	{
	    gtk_widget_destroy(w);
	    lw->display_client = w = NULL;
	}

	/* Get the display parent GtkWidget */
	parent = lw->display_parent;
	if(parent == NULL)
	    return;

	/* No history event? */
	if(h == NULL)
	{
	    /* Create the new client GtkVBox */
	    lw->display_client = w = gtk_vbox_new(TRUE, 0);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Display a generic message */
	    w = gtk_label_new(
"(No Event Selected)"
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	    rcstyle = gtk_rc_style_new();
	    rcstyle->font_name = STRDUP(
"-adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-iso8859-1"
	    );
	    gtk_widget_modify_style(w, rcstyle);
	    GTK_RC_STYLE_UNREF(rcstyle);
	    gtk_widget_show(w);

	    return;
	}

	/* Create the new client GtkVBox */
	lw->display_client = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	type = h->type;

	if(TRUE)
	{
	    const gchar *s;
	    GdkBitmap *mask;
	    GdkPixmap *pixmap;

	    /* Operation Heading GtkHBox */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Get the icon and the name of the operation */
	    switch(type)
	    {
	      case EDV_HISTORY_SYNC_DISKS:
		s = "Sync Disks";
		pixmap = lw->other_operation_pixmap;
		mask = lw->other_operation_mask;
		break;
	      case EDV_HISTORY_SYNC_RECBIN:
		s = "Sync Recycle Bin";
		pixmap = lw->other_operation_pixmap;
		mask = lw->other_operation_mask;
		break;

	      case EDV_HISTORY_DISK_OBJECT_CREATE:
		s = "Create";
		pixmap = lw->object_create_pixmap;
		mask = lw->object_create_mask;
		break;
	      case EDV_HISTORY_DISK_OBJECT_COPY:
		s = "Copy";
		pixmap = lw->object_copy_pixmap;
		mask = lw->object_copy_mask;
		break;
	      case EDV_HISTORY_DISK_OBJECT_MOVE:
		s = "Move";
		pixmap = lw->object_move_pixmap;
		mask = lw->object_move_mask;
		break;
	      case EDV_HISTORY_DISK_OBJECT_LINK:
		s = "Link";
		pixmap = lw->object_link_pixmap;
		mask = lw->object_link_mask;
		break;
	      case EDV_HISTORY_DISK_OBJECT_CHOWN:
		s = "Change Ownership";
		pixmap = lw->object_chown_pixmap;
		mask = lw->object_chown_mask;
		break;
	      case EDV_HISTORY_DISK_OBJECT_CHMOD:
		s = "Change Permissions";
		pixmap = lw->object_chmod_pixmap;
		mask = lw->object_chmod_mask;
		break;
	      case EDV_HISTORY_DISK_OBJECT_CHTIME:
		s = "Change Time Stamps";
		pixmap = lw->object_chtime_pixmap;
		mask = lw->object_chtime_mask;
		break;
	      case EDV_HISTORY_DISK_OBJECT_DOWNLOAD:
		s = "Download";
		pixmap = lw->object_download_pixmap;
		mask = lw->object_download_mask;
		break;
	      case EDV_HISTORY_DISK_OBJECT_DELETE:
		s = "Delete";
		pixmap = lw->object_delete_pixmap;
		mask = lw->object_delete_mask;
		break;

	      case EDV_HISTORY_RECYCLED_OBJECT_RECOVER:
		s = "Recover";
		pixmap = lw->recobj_recover_pixmap;
		mask = lw->recobj_recover_mask;
		break;
	      case EDV_HISTORY_RECYCLED_OBJECT_PURGE:
		s = "Purge";
		pixmap = lw->recobj_purge_pixmap;
		mask = lw->recobj_purge_mask;
		break;
	      case EDV_HISTORY_RECYCLED_OBJECT_PURGE_ALL:
		s = "Purge All";
		pixmap = lw->recobj_purge_pixmap;
		mask = lw->recobj_purge_mask;
		break;

	      case EDV_HISTORY_ARCHIVE_OBJECT_ADD:
		s = "Archive Add";
		pixmap = lw->arcobj_add_pixmap;
		mask = lw->arcobj_add_mask;
		break;
	      case EDV_HISTORY_ARCHIVE_OBJECT_EXTRACT:
		s = "Archive Extract";
		pixmap = lw->arcobj_extract_pixmap;
		mask = lw->arcobj_extract_mask;
		break;
	      case EDV_HISTORY_ARCHIVE_OBJECT_DELETE:
		s = "Archive Delete";
		pixmap = lw->arcobj_delete_pixmap;
		mask = lw->arcobj_delete_mask;
		break;
	      case EDV_HISTORY_ARCHIVE_FIX:
		s = "Archive Fix";
		pixmap = lw->archive_fix_pixmap;
		mask = lw->archive_fix_mask;
		break;

	      default:
		s = "Other Operation";
		pixmap = lw->other_operation_pixmap;
		mask = lw->other_operation_mask;
		break;
	    }

	    /* Create the operation icon */
	    if(pixmap != NULL)
	    {
		w = gtk_pixmap_new(pixmap, mask);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	    }

	    /* Create the operation label */
	    w = gtk_label_new(s);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    rcstyle = gtk_rc_style_new();
	    rcstyle->font_name = STRDUP(
"-adobe-helvetica-medium-r-normal-*-14-*-*-*-*-*-iso8859-1"
	    );
	    gtk_widget_modify_style(w, rcstyle);
	    GTK_RC_STYLE_UNREF(rcstyle);
	    gtk_widget_show(w);

	    /* Error label needed? */
	    if(h->status != 0)
	    {
		gint state;
		gchar *buf, *s2;
		const GdkColor	*c,
				cs_red = EDV_GDKCOLOR_RED,
				cs_orange = EDV_GDKCOLOR_ORANGE;

		/* Get the error string */
		switch(h->status)
		{
		  case -1:
		    s = "Error";
		    s2 = NULL;
		    c = &cs_red;
		    break;
		  case -2:
		    s = "Invalid Value";
		    s2 = NULL;
		    c = &cs_red;
		    break;
		  case -3:
		    s = "System Error";
		    s2 = NULL;
		    c = &cs_red;
		    break;
		  case -4:
		    s = "User Aborted";
		    s2 = NULL;
		    c = &cs_orange;
		    break;
		  case -5:
		    s = "User Canceled";
		    s2 = NULL;
		    c = &cs_orange;
		    break;
		  case -6:
		    s = "Operation In Progress";
		    s2 = NULL;
		    c = &cs_orange;
		    break;
		  default:
		    s = "Unknown Error";
		    s2 = g_strdup_printf("%i", h->status);
		    c = &cs_red;
		    break;
		}

		/* Label */
		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(
"-adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-iso8859-1"
		);
		state = GTK_STATE_NORMAL;
		rcstyle->color_flags[state] =
		    GTK_RC_FG | GTK_RC_TEXT;
		memcpy(&rcstyle->fg[state], c, sizeof(GdkColor));
		memcpy(&rcstyle->text[state], c, sizeof(GdkColor));
		state = GTK_STATE_INSENSITIVE;
		rcstyle->color_flags[state] =
		    GTK_RC_FG | GTK_RC_TEXT;
		memcpy(&rcstyle->fg[state], &cs_orange, sizeof(GdkColor));
		memcpy(&rcstyle->text[state], &cs_orange, sizeof(GdkColor));

		buf = g_strdup_printf(
		    "%s%s",
		    s,
		    (s2 != NULL) ? s2 : ""
		);
		w = gtk_label_new(buf);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_CENTER);
		gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
		gtk_widget_modify_style(w, rcstyle);
		GTK_RC_STYLE_UNREF(rcstyle);
		gtk_widget_show(w);

		g_free(buf);
		g_free(s2);
	    }

#if 0
	    /* Event number */
	    if(TRUE)
	    {

	    }
#endif

	    /* Create the date label */
	    if(h->time_start > 0l)
	    {
		s = EDVDateFormatString(
		    h->time_start,
		    date_format, date_relativity
		);
		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
		);
		w = gtk_label_new(s);
		gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		GTK_RC_STYLE_UNREF(rcstyle);
		gtk_widget_show(w);
	    }
	}


	/* Create the GtkHBox for the source and target? */
	if(!STRISEMPTY(h->src_path) ||
	   !STRISEMPTY(h->tar_path)
	)
	{
	    w = gtk_hbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	}

	/* Create the source icon and label? */
	if(!STRISEMPTY(h->src_path))
	{
	    gchar	*path = STRDUP(h->src_path),
			*name = EDVShortenPath(
		    (path != NULL) ? g_basename(path) : NULL, 25
			);
	    GdkBitmap *mask = NULL;
	    GdkPixmap *pixmap = NULL;

	    /* Get the icon pixmap & mask */
	    if(path != NULL)
	    {
		if(g_path_is_absolute(path))
		{
		    struct stat lstat_buf;
		    if(!lstat((const char *)path, &lstat_buf))
		    {
			edv_object_struct *obj = EDVObjectNew();
			if(obj != NULL)
			{
			    EDVObjectSetPath(obj, path);
			    EDVObjectSetStat(obj, &lstat_buf);
			    EDVMatchObjectIcon(
				core->device, core->total_devices,
				core->mimetype, core->total_mimetypes,
				obj->type,
				obj->full_path,
				TRUE,		/* Assume link valid */
				obj->permissions,
				1,			/* Medium icons */
				&pixmap, &mask,
				NULL, NULL,
				NULL, NULL,
				NULL, NULL
			    );
			    EDVObjectDelete(obj);
			}
		    }
		    else
		    {
			EDVMatchObjectIcon(
			    core->device, core->total_devices,
			    core->mimetype, core->total_mimetypes,
			    EDV_OBJECT_TYPE_FILE,
			    path,
			    TRUE,		/* Assume link valid */
			    0,			/* No permissions */
			    1,			/* Medium icons */
			    &pixmap, &mask,
			    NULL, NULL,
			    NULL, NULL,
			    NULL, NULL
			);
		    }
		}
		else if((type == EDV_HISTORY_SYNC_RECBIN) ||
			(type == EDV_HISTORY_DISK_OBJECT_CREATE) ||
			(type == EDV_HISTORY_DISK_OBJECT_COPY) ||
			(type == EDV_HISTORY_DISK_OBJECT_MOVE) ||
			(type == EDV_HISTORY_DISK_OBJECT_LINK) ||
			(type == EDV_HISTORY_DISK_OBJECT_CHOWN) ||
			(type == EDV_HISTORY_DISK_OBJECT_CHMOD) ||
			(type == EDV_HISTORY_DISK_OBJECT_CHTIME) ||
			(type == EDV_HISTORY_DISK_OBJECT_DELETE) ||
			(type == EDV_HISTORY_DISK_OBJECT_DOWNLOAD) ||
			(type == EDV_HISTORY_RECYCLED_OBJECT_RECOVER) ||
			(type == EDV_HISTORY_RECYCLED_OBJECT_PURGE) ||
			(type == EDV_HISTORY_RECYCLED_OBJECT_PURGE_ALL) ||
			(type == EDV_HISTORY_ARCHIVE_OBJECT_ADD) ||
			(type == EDV_HISTORY_ARCHIVE_OBJECT_EXTRACT) ||
			(type == EDV_HISTORY_ARCHIVE_OBJECT_DELETE) ||
			(type == EDV_HISTORY_ARCHIVE_FIX)
		)
		{
		    EDVMatchObjectIcon(
			core->device, core->total_devices,
			core->mimetype, core->total_mimetypes,
			EDV_OBJECT_TYPE_FILE,
			path,
			TRUE,			/* Assume link valid */
			0,			/* No permissions */
			1,			/* Medium icons */
			&pixmap, &mask,
			NULL, NULL,
			NULL, NULL,
			NULL, NULL
		    );
		}
	    }
	    /* If no icon was obtained then use a generic icon for
	     * these operations
	     */
	    if(pixmap == NULL)
	    {
		edv_mime_type_struct *m;
		switch(type)
		{
		  case EDV_HISTORY_SYNC_RECBIN:
		  case EDV_HISTORY_DISK_OBJECT_CREATE:
		  case EDV_HISTORY_DISK_OBJECT_COPY:
		  case EDV_HISTORY_DISK_OBJECT_MOVE:
		  case EDV_HISTORY_DISK_OBJECT_LINK:
		  case EDV_HISTORY_DISK_OBJECT_CHOWN:
		  case EDV_HISTORY_DISK_OBJECT_CHMOD:
		  case EDV_HISTORY_DISK_OBJECT_CHTIME:
		  case EDV_HISTORY_DISK_OBJECT_DELETE:
		  case EDV_HISTORY_DISK_OBJECT_DOWNLOAD:
		  case EDV_HISTORY_RECYCLED_OBJECT_RECOVER:
		  case EDV_HISTORY_RECYCLED_OBJECT_PURGE:
		  case EDV_HISTORY_RECYCLED_OBJECT_PURGE_ALL:
		  case EDV_HISTORY_ARCHIVE_OBJECT_ADD:
		  case EDV_HISTORY_ARCHIVE_OBJECT_EXTRACT:
		  case EDV_HISTORY_ARCHIVE_OBJECT_DELETE:
		  case EDV_HISTORY_ARCHIVE_FIX:
		    m = EDVMimeTypesListMatchType(
			core->mimetype, core->total_mimetypes,
			NULL,
			EDV_MIME_TYPE_TYPE_INODE_FILE,
			FALSE
		    );
		    if(m != NULL)
		    {
			pixmap = m->medium_pixmap[
			    EDV_MIME_TYPE_ICON_STATE_STANDARD
			];
			mask = m->medium_mask[
			    EDV_MIME_TYPE_ICON_STATE_STANDARD
			];
		    }
		    break;
		  default:
		    break;
		}
	    }

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;

	    /* Create the icon GtkPixmap */
	    if(pixmap != NULL)
	    {
		w = gtk_pixmap_new(pixmap, mask);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	    }

	    /* Create the label */
	    rcstyle = gtk_rc_style_new();
	    rcstyle->font_name = STRDUP(
"-adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-iso8859-1"
	    );
	    w = gtk_label_new(name);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_modify_style(w, rcstyle);
	    GTK_RC_STYLE_UNREF(rcstyle);
	    gtk_widget_show(w);

	    g_free(name);
	    g_free(path);
	}
	/* Create the target icon and label? */
	if(!STRISEMPTY(h->tar_path))
	{
	    gchar	*path = STRDUP(h->tar_path),
			*name = EDVShortenPath(
		(path != NULL) ? g_basename(path) : NULL, 25
			);
	    GdkBitmap *mask = NULL;
	    GdkPixmap *pixmap = NULL;

	    /* Get the icon pixmap & mask */
	    if(path != NULL)
	    {
		if(g_path_is_absolute(path))
		{
		    struct stat lstat_buf;
		    if(!lstat((const char *)path, &lstat_buf))
		    {
			edv_object_struct *obj = EDVObjectNew();
			if(obj != NULL)
			{
			    EDVObjectSetPath(obj, path);
			    EDVObjectSetStat(obj, &lstat_buf);
			    EDVMatchObjectIcon(
				core->device, core->total_devices,
				core->mimetype, core->total_mimetypes,
				obj->type,
				obj->full_path,
				TRUE,		/* Assume link valid */
				obj->permissions,
				1,		/* Medium icons */
				&pixmap, &mask,
				NULL, NULL,
				NULL, NULL,
				NULL, NULL
			    );
			    EDVObjectDelete(obj);
			}
		    }
		    else
		    {
			EDVMatchObjectIcon(
			    core->device, core->total_devices,
			    core->mimetype, core->total_mimetypes,
			    EDV_OBJECT_TYPE_FILE,
			    path,
			    TRUE,		/* Assume link valid */
			    0,			/* No permissions */
			    1,			/* Medium icons */
			    &pixmap, &mask,
			    NULL, NULL,
			    NULL, NULL,
			    NULL, NULL
			);
		    }
		}
		else if((type == EDV_HISTORY_DISK_OBJECT_CREATE) ||
			(type == EDV_HISTORY_DISK_OBJECT_COPY) ||
			(type == EDV_HISTORY_DISK_OBJECT_MOVE) ||
			(type == EDV_HISTORY_DISK_OBJECT_LINK) ||
			(type == EDV_HISTORY_DISK_OBJECT_DOWNLOAD) ||
			(type == EDV_HISTORY_RECYCLED_OBJECT_RECOVER) ||
			(type == EDV_HISTORY_RECYCLED_OBJECT_PURGE) ||
			(type == EDV_HISTORY_RECYCLED_OBJECT_PURGE_ALL) ||
			(type == EDV_HISTORY_ARCHIVE_OBJECT_ADD) ||
			(type == EDV_HISTORY_ARCHIVE_OBJECT_EXTRACT) ||
			(type == EDV_HISTORY_ARCHIVE_OBJECT_DELETE)
		)
		{
		    EDVMatchObjectIcon(
			core->device, core->total_devices,
			core->mimetype, core->total_mimetypes,
			EDV_OBJECT_TYPE_FILE,
			path,
			TRUE,			/* Assume link valid */
			0,			/* No permissions */
			1,			/* Medium icons */
			&pixmap, &mask,
			NULL, NULL,
			NULL, NULL,
			NULL, NULL
		    );
		}
	    }
	    /* If no icon was obtained then use a generic icon for
	     * these operations
	     */
	    if(pixmap == NULL)
	    {
		edv_mime_type_struct *m;
		switch(type)
		{
		  case EDV_HISTORY_DISK_OBJECT_CREATE:
		  case EDV_HISTORY_DISK_OBJECT_COPY:
		  case EDV_HISTORY_DISK_OBJECT_MOVE:
		  case EDV_HISTORY_DISK_OBJECT_LINK:
		  case EDV_HISTORY_DISK_OBJECT_DOWNLOAD:
		  case EDV_HISTORY_RECYCLED_OBJECT_RECOVER:
		  case EDV_HISTORY_RECYCLED_OBJECT_PURGE:
		  case EDV_HISTORY_RECYCLED_OBJECT_PURGE_ALL:
		  case EDV_HISTORY_ARCHIVE_OBJECT_ADD:
		  case EDV_HISTORY_ARCHIVE_OBJECT_EXTRACT:
		  case EDV_HISTORY_ARCHIVE_OBJECT_DELETE:
		    m = EDVMimeTypesListMatchType(
			core->mimetype, core->total_mimetypes,
			NULL,
			EDV_MIME_TYPE_TYPE_INODE_FILE,
			FALSE
		    );
		    if(m != NULL)
		    {
			pixmap = m->medium_pixmap[
			    EDV_MIME_TYPE_ICON_STATE_STANDARD
			];
			mask = m->medium_mask[
			    EDV_MIME_TYPE_ICON_STATE_STANDARD
			];
		    }
		    break;
		  default:
		    break;
		}
	    }

	    /* Create deliminator? */
	    if(!STRISEMPTY(h->src_path))
	    {
		w = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	    }

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;

	    /* Create the icon GtkPixmap */
	    if(pixmap != NULL)
	    {
		w = gtk_pixmap_new(pixmap, mask);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	    }

	    /* Create label */
	    rcstyle = gtk_rc_style_new();
	    rcstyle->font_name = STRDUP(
"-adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-iso8859-1"
	    );
	    w = gtk_label_new(name);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_modify_style(w, rcstyle);
	    GTK_RC_STYLE_UNREF(rcstyle);
	    gtk_widget_show(w);

	    g_free(name);
	    g_free(path);
	}


	/* Source and target locations */
	if(TRUE)
	{
	    const gchar	*src_path = h->src_path,
			*tar_path = h->tar_path;
	    gchar *text;
	    GdkBitmap *mask;
            GdkPixmap *pixmap;

	    w = gtk_hbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Get the source location icon & label */
	    if(type == EDV_HISTORY_RECYCLED_OBJECT_RECOVER)
	    {
		text = STRDUP("Recycle Bin");
		mask = lw->recycle_bin_mask;
		pixmap = lw->recycle_bin_pixmap;
	    }
	    else if((type == EDV_HISTORY_RECYCLED_OBJECT_PURGE) ||
		    (type == EDV_HISTORY_RECYCLED_OBJECT_PURGE_ALL)
	    )
	    {
		text = STRDUP("Recycle Bin");
		mask = lw->recycle_bin_mask;
		pixmap = lw->recycle_bin_pixmap;
	    }
	    else if((src_path != NULL) ?
		    g_path_is_absolute(src_path) : FALSE
	    )
	    {
		struct stat lstat_buf;
		gchar *path = g_dirname(src_path);

		text = EDVShortenPath(path, 40);

		/* Get the icon pixmap & mask */
		if(!lstat((const char *)path, &lstat_buf))
		{
		    edv_object_struct *obj = EDVObjectNew();
		    if(obj != NULL)
		    {
			EDVObjectSetPath(obj, path);
			EDVObjectSetStat(obj, &lstat_buf);
			EDVMatchObjectIcon(
			    core->device, core->total_devices,
			    core->mimetype, core->total_mimetypes,
			    obj->type,
			    obj->full_path,
			    TRUE,		/* Assume link valid */
			    obj->permissions,
			    0,			/* Small icons */
			    &pixmap, &mask,
			    NULL, NULL,
			    NULL, NULL,
			    NULL, NULL
			);
			EDVObjectDelete(obj);
		    }
		}
		else
		{
		    edv_mime_type_struct *m = EDVMimeTypesListMatchType(
			core->mimetype, core->total_mimetypes,
			NULL,
			EDV_MIME_TYPE_TYPE_INODE_DIRECTORY,
			FALSE
		    );
		    if(m != NULL)
		    {
			const edv_mime_type_icon_state state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
			pixmap = m->small_pixmap[state];
			mask = m->small_mask[state];
		    }
		}

		g_free(path);
	    }
	    else
	    {
		text = NULL;
		mask = NULL;
		pixmap = NULL;
	    }

	    /* Create the source location icon & label? */
	    if(text != NULL)
	    {
		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent4 = w;

		/* Create the icon GtkPixmap */
		if(pixmap != NULL)
		{
		    w = gtk_pixmap_new(pixmap, mask);
		    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		    gtk_widget_show(w);
		}

		/* Create the GtkLabel */
		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
		);
		w = gtk_label_new(text);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		GTK_RC_STYLE_UNREF(rcstyle);
		gtk_widget_show(w);

		g_free(text);
	    }

	    /* Get the target location icon and label */
	    if((tar_path != NULL) ?
	       g_path_is_absolute(tar_path) : FALSE
	    )
	    {
		struct stat lstat_buf;
		gchar *path = g_dirname(tar_path);

		text = EDVShortenPath(path, 40);

		/* Get the icon pixmap & mask */
		if(!lstat((const char *)path, &lstat_buf))
		{
		    edv_object_struct *obj = EDVObjectNew();
		    if(obj != NULL)
		    {
			EDVObjectSetPath(obj, path);
			EDVObjectSetStat(obj, &lstat_buf);
			EDVMatchObjectIcon(
			    core->device, core->total_devices,
			    core->mimetype, core->total_mimetypes,
			    obj->type,
			    obj->full_path,
			    TRUE,		/* Assume link valid */
			    obj->permissions,
			    0,			/* Small icons */
			    &pixmap, &mask,
			    NULL, NULL,
			    NULL, NULL,
			    NULL, NULL
			);
			EDVObjectDelete(obj);
		    }
		}
		else
		{
		    edv_mime_type_struct *m = EDVMimeTypesListMatchType(
			core->mimetype, core->total_mimetypes,
			NULL,
			EDV_MIME_TYPE_TYPE_INODE_DIRECTORY,
			FALSE
		    );
		    if(m != NULL)
		    {
			const edv_mime_type_icon_state state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
			pixmap = m->small_pixmap[state];
			mask = m->small_mask[state];
		    }
		}

		g_free(path);
	    }
	    else
	    {
		text = NULL;
		mask = NULL;
		pixmap = NULL;
	    }

	    /* Create the source location icon & label? */
	    if(text != NULL)
	    {
		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent4 = w;

		/* Create the icon GtkPixmap */
		if(pixmap != NULL)
		{
		    w = gtk_pixmap_new(pixmap, mask);
		    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		    gtk_widget_show(w);
		}

		/* Create the GtkLabel */
		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
		);
		w = gtk_label_new(text);
		gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		GTK_RC_STYLE_UNREF(rcstyle);
		gtk_widget_show(w);

		g_free(text);
	    }
	}


	/* Duration */
	if(h->time_end > 0l)
	{
	    gchar *text;
	    const gchar *s;

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    s = EDVDateStringDuration(
		MAX(h->time_end - h->time_start, 0)
	    );
	    text = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"La Duracin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dure"
#elif defined(PROG_LANGUAGE_GERMAN)
"Zeitdauer"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Durata"
#elif defined(PROG_LANGUAGE_DUTCH)
"Duur"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Durao"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Varighet"
#else
"Duration"
#endif
		    ": %s", s
	    );
	    rcstyle = gtk_rc_style_new();
	    rcstyle->font_name = STRDUP(
"-adobe-helvetica-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
	    );
	    w = gtk_label_new(text);
	    g_free(text);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_modify_style(w, rcstyle);
	    GTK_RC_STYLE_UNREF(rcstyle);
	    gtk_widget_show(w);
	}


	/* Comments */
	if(!STRISEMPTY(h->comments))
	{
	    GtkEditable *editable;
	    GtkText *text;

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Display the error icon? */
	    if(h->status != 0)
	    {
		/* Icon GtkVBox */
		w = gtk_vbox_new(TRUE, 0);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent4 = w;

		if(lw->error_medium_pixmap != NULL)
		{
		    /* Icon GtkPixmap */
		    w = gtk_pixmap_new(
			lw->error_medium_pixmap,
			lw->error_medium_mask
		    );
		    gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
		    gtk_widget_show(w);
		}
	    }

	    /* Comments text GtkVBox */
	    w = gtk_vbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent4 = w;

	    /* Comments text and scroll bars GtkTable */
	    w = gtk_table_new(2, 2, FALSE);
	    gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
	    gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Comments GtkText */
	    rcstyle = gtk_rc_style_new();
	    rcstyle->font_name = g_strdup(
#if defined(PROG_LANGUAGE_POLISH)
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-2"
#else
"-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-iso8859-1"
#endif
	    );
	    w = gtk_text_new(NULL, NULL);
	    editable = GTK_EDITABLE(w);
	    text = GTK_TEXT(w);
	    gtk_widget_set_usize(w, -1, 50);
	    text->default_tab_width = 8;
	    gtk_text_set_editable(text, FALSE);
	    gtk_text_set_word_wrap(text, TRUE);
	    gtk_table_attach(
		GTK_TABLE(parent4), w,
		0, 1, 0, 1,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(EDVHistoryWinCommentsTextEventCB), lw
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(EDVHistoryWinCommentsTextEventCB), lw
	    );
	    gtk_widget_modify_style(w, rcstyle);
	    GTK_RC_STYLE_UNREF(rcstyle);
/*	    gdk_window_set_cursor(text->text_area, lw->text_cur); */
	    GUIEditableEndowPopupMenu(w, GUI_EDITABLE_POPUP_MENU_READ_ONLY);
            gtk_text_freeze(text);
            gtk_text_insert(text, NULL, NULL, NULL, h->comments, -1);
            gtk_text_thaw(text);
	    gtk_widget_show(w);
	    /* Vertical GtkScrollBar */
	    w = gtk_vscrollbar_new(text->vadj);
	    gtk_table_attach(
		GTK_TABLE(parent4), w,
		1, 2, 0, 1,
		GTK_FILL,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		0, 0
	    );
	    gtk_widget_show(w);
	}

}


/*
 *	Sets the row on the history list window's clist to the values
 *	specified by the given history structure.
 */
static void EDVHistoryWinSetRow(
	edv_history_win_struct *lw,
	const gint row, edv_history_struct *h
)
{
	gint nrows, ncolumns;
	const gchar *date_format;
	const gchar *type_name;
	GdkPixmap *pixmap;
	GdkBitmap *mask; 
	GtkCList *clist;
	edv_history_type type;
	edv_date_relativity date_relativity;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if((lw == NULL) || (h == NULL))
	    return;

	clist = GTK_CLIST(lw->history_clist);
	ncolumns = clist->columns;
	nrows = clist->rows;
	core = lw->core;
	cfg_list = core->cfg_list;
	date_relativity = (edv_date_relativity)EDV_GET_I(
	    EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);

	/* Get operation type and icon */
	type = h->type;
	switch(type)
	{
	  case EDV_HISTORY_UNKNOWN:
	    type_name = "Other";
	    pixmap = lw->other_operation_pixmap;
	    mask = lw->other_operation_mask;
	    break;

	  case EDV_HISTORY_SYNC_DISKS:
	    type_name = "Sync Disks";
	    pixmap = lw->other_operation_pixmap;
	    mask = lw->other_operation_mask;
	    break;
	  case EDV_HISTORY_SYNC_RECBIN:
	    type_name = "Sync Recycle Bin";
	    pixmap = lw->other_operation_pixmap;
	    mask = lw->other_operation_mask;
	    break;

	  case EDV_HISTORY_DISK_OBJECT_CREATE:
	    type_name = "Create";
	    pixmap = lw->object_create_pixmap;
	    mask = lw->object_create_mask;
	    break;
	  case EDV_HISTORY_DISK_OBJECT_COPY:
	    type_name = "Copy";
	    pixmap = lw->object_copy_pixmap;
	    mask = lw->object_copy_mask;
	    break;
	  case EDV_HISTORY_DISK_OBJECT_MOVE:
	    type_name = "Move";
	    pixmap = lw->object_move_pixmap;
	    mask = lw->object_move_mask;
	    break;
	  case EDV_HISTORY_DISK_OBJECT_LINK:
	    type_name = "Link";
	    pixmap = lw->object_link_pixmap;
	    mask = lw->object_link_mask;
	    break;
	  case EDV_HISTORY_DISK_OBJECT_CHOWN:
	    type_name = "Change Ownership";
	    pixmap = lw->object_chown_pixmap;
	    mask = lw->object_chown_mask;
	    break;
	  case EDV_HISTORY_DISK_OBJECT_CHMOD:
	    type_name = "Change Permissions";
	    pixmap = lw->object_chmod_pixmap;
	    mask = lw->object_chmod_mask;
	    break;
	  case EDV_HISTORY_DISK_OBJECT_CHTIME:
	    type_name = "Change Time Stamps";
	    pixmap = lw->object_chtime_pixmap;
	    mask = lw->object_chtime_mask;
	    break;
	  case EDV_HISTORY_DISK_OBJECT_DOWNLOAD:
	    type_name = "Download";
	    pixmap = lw->object_download_pixmap;
	    mask = lw->object_download_mask;
	    break;
	  case EDV_HISTORY_DISK_OBJECT_DELETE:
	    type_name = "Delete";
	    pixmap = lw->object_delete_pixmap;
	    mask = lw->object_delete_mask;
	    break;

	  case EDV_HISTORY_RECYCLED_OBJECT_RECOVER:
	    type_name = "Recover";
	    pixmap = lw->recobj_recover_pixmap;
	    mask = lw->recobj_recover_mask;
	    break;
	  case EDV_HISTORY_RECYCLED_OBJECT_PURGE:
	    type_name = "Purge";
	    pixmap = lw->recobj_purge_pixmap;
	    mask = lw->recobj_purge_mask;
	    break;
	  case EDV_HISTORY_RECYCLED_OBJECT_PURGE_ALL:
	    type_name = "Purge All";
	    pixmap = lw->recobj_purge_pixmap;
	    mask = lw->recobj_purge_mask;
	    break;

	  case EDV_HISTORY_ARCHIVE_OBJECT_ADD:
	    type_name = "Archive Add";
	    pixmap = lw->arcobj_add_pixmap;
	    mask = lw->arcobj_add_mask;
	    break;
	  case EDV_HISTORY_ARCHIVE_OBJECT_EXTRACT:
	    type_name = "Archive Extract";
	    pixmap = lw->arcobj_extract_pixmap;
	    mask = lw->arcobj_extract_mask;
	    break;
	  case EDV_HISTORY_ARCHIVE_OBJECT_DELETE:
	    type_name = "Archive Delete";
	    pixmap = lw->arcobj_delete_pixmap;
	    mask = lw->arcobj_delete_mask;
	    break;
	  case EDV_HISTORY_ARCHIVE_FIX:
	    type_name = "Archive Fix";
	    pixmap = lw->archive_fix_pixmap;
	    mask = lw->archive_fix_mask;
	    break;

	  default:
	    type_name = "Other";
	    pixmap = lw->other_operation_pixmap;
	    mask = lw->other_operation_mask;
	    break;
	}


	/* Is the specified row in bounds? */
	if((row >= 0) && (row < nrows))
	{
	    gint column;
	    gtk_clist_freeze(clist);

	    /* Set the cell values */

	    /* Type */
	    column = 0;
	    if(column < ncolumns)
	    {
		if(pixmap != NULL)
		    gtk_clist_set_pixtext(
			clist, row, column,
			type_name,
			EDV_LIST_PIXMAP_TEXT_SPACING,
			pixmap, mask
		    );
		else
		    gtk_clist_set_text(
			clist, row, column,
			type_name
		    );
	    }
	    /* Set column 1 as the Source */
	    column = 1;
	    if((column < ncolumns) && !STRISEMPTY(h->src_path))
	    {
		gchar	*s,
			*path = STRDUP(h->src_path),
			*name = (path != NULL) ? g_basename(path) : NULL;
		mask = NULL;
		pixmap = NULL;
		if((path != NULL) ? g_path_is_absolute(path) : FALSE)
		{
		    struct stat lstat_buf;
		    if(!lstat((const char *)path, &lstat_buf))
		    {
			edv_object_struct *obj = EDVObjectNew();
			if(obj != NULL)
			{
			    EDVObjectSetPath(obj, path);
			    EDVObjectSetStat(obj, &lstat_buf);
			    EDVMatchObjectIcon(
				core->device, core->total_devices,
				core->mimetype, core->total_mimetypes,
				obj->type,
				obj->full_path,
				TRUE,		/* Assume link valid */
				obj->permissions,
				0,		/* Small icons */
				&pixmap, &mask,
				NULL, NULL,
				NULL, NULL,
				NULL, NULL
			    );
			    EDVObjectDelete(obj);
			}
		    }
		}
		/* If icon was not obtained then use a generic icon
		 * for certain operation types
		 */
		if(pixmap == NULL)
		{
		    switch(type)
		    {
		      case EDV_HISTORY_SYNC_RECBIN:
		      case EDV_HISTORY_DISK_OBJECT_CREATE:
		      case EDV_HISTORY_DISK_OBJECT_COPY:
		      case EDV_HISTORY_DISK_OBJECT_MOVE:
		      case EDV_HISTORY_DISK_OBJECT_LINK:
		      case EDV_HISTORY_DISK_OBJECT_CHOWN:
		      case EDV_HISTORY_DISK_OBJECT_CHMOD:
		      case EDV_HISTORY_DISK_OBJECT_CHTIME:
		      case EDV_HISTORY_DISK_OBJECT_DOWNLOAD:
		      case EDV_HISTORY_DISK_OBJECT_DELETE:
		      case EDV_HISTORY_RECYCLED_OBJECT_RECOVER:
		      case EDV_HISTORY_RECYCLED_OBJECT_PURGE:
		      case EDV_HISTORY_ARCHIVE_OBJECT_ADD:
		      case EDV_HISTORY_ARCHIVE_OBJECT_EXTRACT:
		      case EDV_HISTORY_ARCHIVE_OBJECT_DELETE:
		      case EDV_HISTORY_ARCHIVE_FIX:
			EDVMatchObjectIcon(
			    NULL, 0,
			    core->mimetype, core->total_mimetypes,
			    EDV_OBJECT_TYPE_FILE,
			    path,
			    TRUE,
			    0x00000000,
			    0,          /* Samll icons */
			    &pixmap, &mask,
			    NULL, NULL,
			    NULL, NULL,
			    NULL, NULL
			);
			break;
		      default:
			break;
		    }
		}

		/* Get cell value */
		if(STRISEMPTY(name))
		    s = STRDUP("");
		else
		    s = EDVShortenPath(name, 32);

		/* Set cell */
		if(pixmap != NULL)
		    gtk_clist_set_pixtext(
			clist, row, column, s,
			EDV_LIST_PIXMAP_TEXT_SPACING,
			pixmap, mask
		    );
		else
		    gtk_clist_set_text(
		        clist, row, column, s
		    );
		g_free(s);
		g_free(path);
	    }
	    /* Set column 2 as the Target */
	    column = 2;
	    if((column < ncolumns) && !STRISEMPTY(h->tar_path))
	    {
		gchar	*s,
			*path = STRDUP(h->tar_path),
			*name = (path != NULL) ? g_basename(path) : NULL;
		mask = NULL;
		pixmap = NULL;
		if((path != NULL) ? g_path_is_absolute(path) : FALSE)
		{
		    struct stat lstat_buf;
		    if(!lstat((const char *)path, &lstat_buf))
		    {
			edv_object_struct *o = EDVObjectNew();
			EDVObjectSetPath(o, path);
			EDVObjectSetStat(o, &lstat_buf);
			EDVMatchObjectIcon(
			    core->device, core->total_devices,
			    core->mimetype, core->total_mimetypes,
			    o->type,
			    o->full_path,
			    TRUE,		/* Assume link valid */
			    o->permissions,
			    0,			/* Small icons */
			    &pixmap, &mask,
			    NULL, NULL,
			    NULL, NULL,
			    NULL, NULL
			);
			EDVObjectDelete(o);
		    }
		}
		/* If icon was not obtained then use a generic icon
		 * for certain operation types
		 */
		if(pixmap == NULL)
		{
		    switch(type)
		    {
		      case EDV_HISTORY_DISK_OBJECT_CREATE:
		      case EDV_HISTORY_DISK_OBJECT_COPY:
		      case EDV_HISTORY_DISK_OBJECT_MOVE:
		      case EDV_HISTORY_DISK_OBJECT_LINK:
		      case EDV_HISTORY_DISK_OBJECT_DOWNLOAD:
		      case EDV_HISTORY_DISK_OBJECT_DELETE:
		      case EDV_HISTORY_RECYCLED_OBJECT_RECOVER:
		      case EDV_HISTORY_ARCHIVE_OBJECT_ADD:
		      case EDV_HISTORY_ARCHIVE_OBJECT_EXTRACT:
		      case EDV_HISTORY_ARCHIVE_OBJECT_DELETE:
		      case EDV_HISTORY_ARCHIVE_FIX:
			EDVMatchObjectIcon(
			    NULL, 0,
			    core->mimetype, core->total_mimetypes,
			    EDV_OBJECT_TYPE_FILE,
			    path,
			    TRUE,
			    0x00000000,
			    0,          /* Samll icons */
			    &pixmap, &mask,
			    NULL, NULL,
			    NULL, NULL,
			    NULL, NULL
			);
			break;
		      default:
			break;
		    }
		}

		/* Get cell value */
		if(STRISEMPTY(name))
		    s = STRDUP("");
		else
		    s = EDVShortenPath(name, 32);

		/* Set cell */
		if(pixmap != NULL)
		    gtk_clist_set_pixtext(
			clist, row, column, s,
			EDV_LIST_PIXMAP_TEXT_SPACING,
			pixmap, mask
		    );
		else
		    gtk_clist_set_text(
			clist, row, column, s
		    );
		g_free(s);
		g_free(path);
	    }
	    /* Set column 3 as the Status */
	    column = 3;
	    if(column < ncolumns)
	    {
		const gchar *s;
		GtkStyle *style;
		switch(h->status)
		{
		  case 0:
		    s = "Success";
		    style = lw->cell_style[EDV_HISTORY_CELL_STYLE_SUCCESS];
		    break;
		  case -1:
		    s = "Error";
		    style = lw->cell_style[EDV_HISTORY_CELL_STYLE_ERROR];
		    break;
		  case -2:
		    s = "Invalid Value";
		    style = lw->cell_style[EDV_HISTORY_CELL_STYLE_ERROR];
		    break;
		  case -3:
		    s = "System Error";
		    style = lw->cell_style[EDV_HISTORY_CELL_STYLE_ERROR];
		    break;
		  case -4:
		    s = "User Aborted";
		    style = lw->cell_style[EDV_HISTORY_CELL_STYLE_CANCEL];
		    break;
		  case -5:
		    s = "User Canceled";
		    style = lw->cell_style[EDV_HISTORY_CELL_STYLE_CANCEL];
		    break;
		  case -6:
		    s = "Operation In Progress";
		    style = lw->cell_style[EDV_HISTORY_CELL_STYLE_ERROR];
		    break;
		  default:
		    s = "Unknown Error";
		    style = NULL;
		    break;
		}
		gtk_clist_set_text(
		    clist, row, column, s
		);
		if(style != NULL)
		    gtk_clist_set_cell_style(clist, row, column, style);
	    }
	    /* Set column 4 as the Start Time */
	    column = 4;
	    if(column < ncolumns)
	    {
		const gchar *s = EDVDateFormatString(
		    h->time_start, date_format, date_relativity
		);
		gtk_clist_set_text(
		    clist, row, column,
		    (s != NULL) ? s : ""
		);
	    }
	    /* Set column 5 as the End Time */
	    column = 5;
	    if(column < ncolumns)
	    {
		const gchar *s = EDVDateFormatString(
		    h->time_end, date_format, date_relativity
		);
		gtk_clist_set_text(
		    clist, row, column,
		    (s != NULL) ? s : ""
		);
	    }


	    /* Set this row's data as the history item */
	    gtk_clist_set_row_data_full(
		clist, row,
		EDVHistoryCopy(h), EDVHistoryListItemDestroyCB
	    );

	    gtk_clist_thaw(clist);
	}
}

/*
 *	Updates the History List Window's history list.
 */
void EDVHistoryWinUpdateList(edv_history_win_struct *lw)
{
	gint i, max_events, row, ncolumns, nhistory_events;
	gchar **strv;
	GList *glist, *history_list;
	GtkRcStyle *lists_rcstyle;
	GtkAdjustment *adj;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	edv_history_struct *h;
	edv_core_struct *core;

	if(lw == NULL)
	    return;

	clist = GTK_CLIST(lw->history_clist);
	core = lw->core;
	cfg_list = core->cfg_list;
	lists_rcstyle = core->lists_rcstyle;
	max_events = EDV_GET_I(EDV_CFG_PARM_HISTORY_EVENTS_MAX);

	/* Create the row cell values */
	ncolumns = MAX(clist->columns, 1);
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	gtk_clist_freeze(clist);

	/* Clear the list */
	gtk_clist_clear(clist);

	/* Open the history items from file */
	history_list = EDVHistoryListOpen(
	    EDV_GET_S(EDV_CFG_PARM_FILE_HISTORY),
	    NULL,
	    core
	);

	nhistory_events = g_list_length(history_list);

	/* Get the start of the history list, if max_events is positive
	 * then start at the last nhistory_events - max_events item
	 */
	if(max_events > 0)
	    glist = g_list_nth(
		history_list,
		MAX(nhistory_events - max_events, 0)
	    );
	else
	    glist = history_list;

	/* Iterate through the history list and append each history
	 * event to the GtkCList
	 */
	while(glist != NULL)
	{
	    h = EDV_HISTORY(glist->data);
	    if(h == NULL)
	    {
		glist = g_list_next(glist);
		continue;
	    }

	    /* Append a new row */
	    row = gtk_clist_append(clist, strv);
	    if(row < 0)
		break;

	    EDVHistoryWinSetRow(lw, row, h);

	    glist = g_list_next(glist);
	}

	/* Delete the history list */
	if(history_list != NULL)
	{
	    g_list_foreach(
		history_list, (GFunc)EDVHistoryDelete, NULL
	    );
	    g_list_free(history_list);
	}

	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);

	/* Scroll to the end */
	adj = clist->vadjustment;
	if(adj != NULL)
	    gtk_adjustment_set_value(
		adj,
		MAX(
		    (adj->upper - adj->page_size),
		    adj->lower
		)
	    );

	/* Set RC style after column heading mapping */
	if(lists_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		GTK_WIDGET(clist), lists_rcstyle
	    );

	/* Delete the row cell values */
	g_free(strv);
}


/*
 *      Sets values on the given history list window to the
 *	configuration list.
 */
void EDVHistoryWinSyncConfiguration(edv_history_win_struct *lw)
{
	GtkWidget *w;
	edv_core_struct *core;
	cfg_item_struct *cfg_list;

	if(lw == NULL)
	    return;

	core = lw->core;
	cfg_list = core->cfg_list;

	/* Toplevel geometry */
	w = lw->toplevel;
	if(w != NULL)
	{
#if 0
	    GdkWindow *window = w->window;
	    gint x = 0, y = 0;

	    if(window != NULL)
		gdk_window_get_root_origin(window, &x, &y);

	    EDV_SET_I(EDV_CFG_PARM_HISTORYWIN_X, x);
	    EDV_SET_I(EDV_CFG_PARM_HISTORYWIN_Y, y);
	    EDV_SET_I(EDV_CFG_PARM_HISTORYWIN_WIDTH, w->allocation.width);
	    EDV_SET_I(EDV_CFG_PARM_HISTORYWIN_HEIGHT, w->allocation.height);
#endif
	}
}


/*
 *	Recreates the History List Window's list styles.
 */
void EDVHistoryWinListStylesRegenerate(edv_history_win_struct *lw)
{
	gint i;
	edv_history_win_cell_style n;
	GdkColormap *colormap;
	GtkStyle *style, *src_style;
	GtkWidget *w;

	if(lw == NULL)
	    return;

	w = lw->history_clist;
	if(w == NULL)
	    return;

	/* Unref existing cell styles */
	for(i = 0; i < EDV_HISTORY_TOTAL_CELL_STYLES; i++)
	{
	    GTK_STYLE_UNREF(lw->cell_style[i]);
	    lw->cell_style[i] = NULL;
	}

	/* Get source style from contents clist as a template */
	colormap = gtk_widget_get_colormap(w);
	src_style = gtk_widget_get_style(w);
	if(src_style == NULL)
	    return;

	/* Begin creating cell styles */

#define REALLOC_COLOR(_ct_,_cs_)	{			\
 GDK_COLORMAP_FREE_COLOR(colormap, (_ct_));			\
 memcpy((_ct_), (_cs_), sizeof(GdkColor));			\
 GDK_COLORMAP_ALLOC_COLOR(colormap, (_ct_));			\
}

	/* Success */
	n = EDV_HISTORY_CELL_STYLE_SUCCESS;
	lw->cell_style[n] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
	    const GdkColor      cs = EDV_GDKCOLOR_MEDIUM_GREEN,
				cs_sel = EDV_GDKCOLOR_GREEN,
				cs_insen = EDV_GDKCOLOR_DARK_GREEN;

	    GtkStateType state = GTK_STATE_NORMAL;
	    REALLOC_COLOR(&style->fg[state], &cs);

	    state = GTK_STATE_ACTIVE;
	    REALLOC_COLOR(&style->fg[state], &cs);

	    state = GTK_STATE_PRELIGHT;
	    REALLOC_COLOR(&style->fg[state], &cs);

	    state = GTK_STATE_SELECTED;
	    REALLOC_COLOR(&style->fg[state], &cs_sel);

	    state = GTK_STATE_INSENSITIVE;
	    REALLOC_COLOR(&style->fg[state], &cs_insen);
	}

	/* Error */
	n = EDV_HISTORY_CELL_STYLE_ERROR;
	lw->cell_style[n] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
	    const GdkColor      cs = EDV_GDKCOLOR_RED,
				cs_sel = EDV_GDKCOLOR_RED,
				cs_insen = EDV_GDKCOLOR_DARK_RED;

	    GtkStateType state = GTK_STATE_NORMAL;
	    REALLOC_COLOR(&style->fg[state], &cs);

	    state = GTK_STATE_ACTIVE;
	    REALLOC_COLOR(&style->fg[state], &cs);

	    state = GTK_STATE_PRELIGHT;
	    REALLOC_COLOR(&style->fg[state], &cs);

	    state = GTK_STATE_SELECTED;
	    REALLOC_COLOR(&style->fg[state], &cs_sel);

	    state = GTK_STATE_INSENSITIVE;
	    REALLOC_COLOR(&style->fg[state], &cs_insen);
	}

	/* Cancel */
	n = EDV_HISTORY_CELL_STYLE_CANCEL;
	lw->cell_style[n] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
	    const GdkColor      cs = EDV_GDKCOLOR_ORANGE,
				cs_sel = EDV_GDKCOLOR_ORANGE,
				cs_insen = EDV_GDKCOLOR_DARK_ORANGE;

	    GtkStateType state = GTK_STATE_NORMAL;
	    REALLOC_COLOR(&style->fg[state], &cs);

	    state = GTK_STATE_ACTIVE;
	    REALLOC_COLOR(&style->fg[state], &cs);

	    state = GTK_STATE_PRELIGHT;
	    REALLOC_COLOR(&style->fg[state], &cs);

	    state = GTK_STATE_SELECTED;
	    REALLOC_COLOR(&style->fg[state], &cs_sel);

	    state = GTK_STATE_INSENSITIVE;
	    REALLOC_COLOR(&style->fg[state], &cs_insen);
	}

#undef REALLOC_COLOR
}


/*
 *	Creates a new History List Window.
 */
edv_history_win_struct *EDVHistoryWinNew(edv_core_struct *core)
{
	const gint border_major = 5;
	gchar *heading[6];
	gpointer mclient_data;
	gpointer entry_rtn;
	GdkWindow *window;
	GtkRcStyle *standard_rcstyle = NULL, *lists_rcstyle = NULL;
	GtkAccelGroup *accelgrp;
	GtkStyle *style;
	GtkWidget	*w, *menu,
			*parent, *parent2,
			*toplevel;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	edv_history_win_struct *lw;

	if(core == NULL)
	    return(NULL);

	cfg_list = core->cfg_list;

	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	/* Create the History Window */
	lw = EDV_HISTORY_WIN(g_malloc0(sizeof(edv_history_win_struct)));
	if(lw == NULL)
	    return(NULL);

	lw->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	lw->accelgrp = accelgrp = gtk_accel_group_new();
	lw->processing = FALSE;
	lw->busy_count = 0;
	lw->freeze_count = 0;
	lw->core = core;

	lw->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
        gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
/* gtk_widget_set_uposition() */
	gtk_widget_set_usize(w, LISTWIN_WIDTH, LISTWIN_HEIGHT);
	gtk_window_set_title(GTK_WINDOW(w), LISTWIN_TITLE);
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "history", PROG_NAME
	);
	gtk_widget_realize(w);
	style = gtk_widget_get_style(w);
	window = w->window;
	if(window != NULL)
	{
#if 1
            GdkGeometry geo;

            geo.min_width = 100;
            geo.min_height = 70;
            geo.base_width = 0;
            geo.base_height = 0;
            geo.width_inc = 1;
            geo.height_inc = 1;
            gdk_window_set_geometry_hints(
                window, &geo,
                GDK_HINT_MIN_SIZE |
                GDK_HINT_BASE_SIZE |
                GDK_HINT_RESIZE_INC
            );
#else
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
		GDK_DECOR_MINIMIZE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
	    );
#endif
	    GUISetWMIcon(window, (guint8 **)icon_trace_48x48_xpm);
	}
        gtk_widget_add_events(
            w,
            GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK
        );
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVHistoryWinDeleteEventCB), lw
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;


	/* Load the pixmaps */
#define LOAD_PIXMAP(_d_,_n_,_pixmap_rtn_,_mask_rtn_) { \
 EDVLoadPixmap(				\
  core,					\
  (guint8 **)(_d_), (_n_),		\
  (_pixmap_rtn_), (_mask_rtn_)		\
 );					\
}
	LOAD_PIXMAP(
	    icon_executable_20x20_xpm,
	    "icon_executable_20x20_xpm",
	    &lw->other_operation_pixmap,
	    &lw->other_operation_mask
	);
	LOAD_PIXMAP(
	    icon_trash_20x20_xpm,
	    "icon_trash_20x20_xpm",
	    &lw->recycle_bin_pixmap,
	    &lw->recycle_bin_mask
	);
	LOAD_PIXMAP(
	    icon_new_20x20_xpm,
	    "icon_new_20x20_xpm",
	    &lw->object_create_pixmap,
	    &lw->object_create_mask
	);
	LOAD_PIXMAP(
	    icon_copy_file_20x20_xpm,
	    "icon_copy_file_20x20_xpm",
	    &lw->object_copy_pixmap,
	    &lw->object_copy_mask
	);
	LOAD_PIXMAP(
	    icon_move_file_20x20_xpm,
	    "icon_move_file_20x20_xpm",
	    &lw->object_move_pixmap,
	    &lw->object_move_mask
	);
	LOAD_PIXMAP(
	    icon_link2_20x20_xpm,
	    "icon_link2_20x20_xpm",
	    &lw->object_link_pixmap,
	    &lw->object_link_mask
	);
	LOAD_PIXMAP(
	    icon_owned_20x20_xpm,
	    "icon_owned_20x20_xpm",
	    &lw->object_chown_pixmap,
	    &lw->object_chown_mask
	);
	LOAD_PIXMAP(
	    icon_chmod_20x20_xpm,
	    "icon_chmod_20x20_xpm",
	    &lw->object_chmod_pixmap,
	    &lw->object_chmod_mask
	);
	LOAD_PIXMAP(
	    icon_time_stamp_20x20_xpm,
	    "icon_time_stamp_20x20_xpm",
	    &lw->object_chtime_pixmap,
	    &lw->object_chtime_mask
	);
	LOAD_PIXMAP(
	    icon_download_file_20x20_xpm,
	    "icon_download_file_20x20_xpm",
	    &lw->object_download_pixmap,
	    &lw->object_download_mask
	);
	LOAD_PIXMAP(
	    icon_cancel_20x20_xpm,
	    "icon_cancel_20x20_xpm",
	    &lw->object_delete_pixmap,
	    &lw->object_delete_mask
	);

	LOAD_PIXMAP(
	    icon_recover_20x20_xpm,
	    "icon_recover_20x20_xpm",
	    &lw->recobj_recover_pixmap,
	    &lw->recobj_recover_mask
	);
	LOAD_PIXMAP(
	    icon_purge_20x20_xpm,
	    "icon_purge_20x20_xpm",
	    &lw->recobj_purge_pixmap,
	    &lw->recobj_purge_mask
	);

	LOAD_PIXMAP(
	    icon_archive_add_20x20_xpm,
	    "icon_archive_add_20x20_xpm",
	    &lw->arcobj_add_pixmap,
	    &lw->arcobj_add_mask
	);
	LOAD_PIXMAP(
	    icon_archive_extract_20x20_xpm,
	    "icon_archive_extract_20x20_xpm",
	    &lw->arcobj_extract_pixmap,
	    &lw->arcobj_extract_mask
	);
	LOAD_PIXMAP(
	    icon_cancel_20x20_xpm,
	    "icon_cancel_20x20_xpm",
	    &lw->arcobj_delete_pixmap,
	    &lw->arcobj_delete_mask
	);
	LOAD_PIXMAP(
	    icon_aid_20x20_xpm,
	    "icon_aid_20x20_xpm",
	    &lw->archive_fix_pixmap,
	    &lw->archive_fix_mask
	);

	LOAD_PIXMAP(
	    icon_object_misc_20x20_xpm,
	    "icon_object_misc_20x20_xpm",
	    &lw->object_other_small_pixmap,
	    &lw->object_other_small_mask
	);
	LOAD_PIXMAP(
	    icon_object_misc_32x32_xpm,
	    "icon_object_misc_32x32_xpm",
	    &lw->object_other_medium_pixmap,
	    &lw->object_other_medium_mask
	);
	LOAD_PIXMAP(
	    icon_error_20x20_xpm,
	    "icon_error_20x20_xpm",
	    &lw->error_small_pixmap,
	    &lw->error_small_mask
	);
	LOAD_PIXMAP(
	    icon_error_32x32_xpm,
	    "icon_error_32x32_xpm",
	    &lw->error_medium_pixmap,
	    &lw->error_medium_mask
	);
#undef LOAD_PIXMAP


	/* Main GtkVBox */
	lw->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Find Prompt GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	w = GUIPromptBar(
	    (guint8 **)icon_search_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"El Hallazgo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Recherche"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fund"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trovare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vondst"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ache"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Funn"
#else
"Find"
#endif
	    ":", NULL, &entry_rtn
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	lw->find_entry = w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "activate",
	    GTK_SIGNAL_FUNC(EDVHistoryWinFindCB), lw
	);
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);


	/* History clist GtkScrolledWindow */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* History GtkCList */
#if defined(PROG_LANGUAGE_SPANISH)
	heading[0] = "El Tipo";
	heading[1] = "La Fuente";
	heading[2] = "El Objetivo";
	heading[3] = "La Posicin";
	heading[4] = "El Comienzo Del Tiempo";
	heading[5] = "El Fin Del Tiempo.";
#elif defined(PROG_LANGUAGE_FRENCH)
	heading[0] = "Type";
	heading[1] = "Source";
	heading[2] = "Destination";
	heading[3] = "Etat";
	heading[4] = "Heure Dbut";
	heading[5] = "Heure Fin";
#elif defined(PROG_LANGUAGE_GERMAN)
	heading[0] = "Typ";
	heading[1] = "Quelle";
	heading[2] = "Ziel";
	heading[3] = "Status";
	heading[4] = "Zeit Anfang";
	heading[5] = "Zeit Ende";
#elif defined(PROG_LANGUAGE_ITALIAN)
	heading[0] = "Il Tipo";
	heading[1] = "La Fonte";
	heading[2] = "Il Bersaglio";
	heading[3] = "Lo Stato";
	heading[4] = "L'Inizio Di Tempo";
	heading[5] = "La Fine Di Tempo";
#elif defined(PROG_LANGUAGE_DUTCH)
	heading[0] = "Type";
	heading[1] = "Bron";
	heading[2] = "Doel";
	heading[3] = "Status";
	heading[4] = "Regeel Start";
	heading[5] = "Regeel Einde";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	heading[0] = "O Tipo";
	heading[1] = "A Fonte";
	heading[2] = "O Alvo";
	heading[3] = "O Estado";
	heading[4] = "O Tempo Comea";
	heading[5] = "O Fim De Tempo";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	heading[0] = "Type";
	heading[1] = "Kilde";
	heading[2] = "Ml";
	heading[3] = "Status";
	heading[4] = "TidsStart";
	heading[5] = "TidsSlutt";
#else
	heading[0] = "Type";
	heading[1] = "Source";
	heading[2] = "Target";
	heading[3] = "Status";
	heading[4] = "Time Start";
	heading[5] = "Time End";
#endif
	lw->history_clist = w = gtk_clist_new_with_titles(6, heading);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(EDVCListMotionEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVHistoryWinKeyEventCB), lw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVHistoryWinKeyEventCB), lw
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVHistoryWinButtonPressEventCB), lw
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, TRUE);
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 1, TRUE);
	gtk_clist_set_column_resizeable(clist, 1, TRUE);
	gtk_clist_set_column_justification(
	    clist, 1, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 2, TRUE);
	gtk_clist_set_column_resizeable(clist, 2, TRUE);
	gtk_clist_set_column_justification(
	    clist, 2, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 3, TRUE);
	gtk_clist_set_column_resizeable(clist, 3, TRUE);
	gtk_clist_set_column_justification(
	    clist, 3, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 4, TRUE);
	gtk_clist_set_column_resizeable(clist, 4, TRUE);
	gtk_clist_set_column_justification(
	    clist, 4, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 5, TRUE);
	gtk_clist_set_column_resizeable(clist, 5, TRUE);
	gtk_clist_set_column_justification(
	    clist, 5, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(EDVHistoryWinSelectRowCB), lw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(EDVHistoryWinUnselectRowCB), lw
	);
	gtk_widget_show(w);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Refresh Button */
	lw->refresh_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_reload_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Refresque"
#elif defined(PROG_LANGUAGE_FRENCH)
"Rafrachir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Erfrischen Sie"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Rinfrescare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verfris"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Refresque-se"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forfrisk"
#else
"Refresh"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVHistoryWinRefreshCB), lw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_F5, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_r, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_r);
	gtk_widget_show(w);

#if 0
	/* Save As Button */
	lw->save_as_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_save_as_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Salve Como"
#elif defined(PROG_LANGUAGE_FRENCH)
"Enregister sous:"
#elif defined(PROG_LANGUAGE_GERMAN)
"Auer Als"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Risparmiare Come"
#elif defined(PROG_LANGUAGE_DUTCH)
"Behalve Als"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Exceto Como"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Untatt Som"
#else
"Save As"
#endif
	    "...", NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVHistoryWinSaveAsCB), lw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_s);
	gtk_widget_show(w);
#endif

	/* Clear Button */
	lw->clear_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_clear_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Claro"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klar"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Chiaro"
#elif defined(PROG_LANGUAGE_DUTCH)
"Helder"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Claro"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klar"
#else
"Clear"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVHistoryWinClearCB), lw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_l, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_l);
	gtk_widget_show(w);


	/* Horizontal GtkSeparator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* History Event Display Parent GtkVBox */
	lw->display_parent = w = gtk_vbox_new(FALSE, 0);
	gtk_widget_set_usize(w, -1, LISTWIN_DISPLAY_HEIGHT);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Client vbox that is parented to the display_parent, leave
	 * this NULL, it will be created when needed
	 */
	lw->display_client = NULL;


	/* Horizontal GtkSeparator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons GtkHbox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Close button */
	lw->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Quitter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVHistoryWinCloseCB), lw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);



	/* Right click menu */
	lw->menu = menu = GUIMenuCreate();
	mclient_data = lw;
	if(menu != NULL)
	{
	    guint8 **icon;
	    const gchar *label;
	    guint accel_key, accel_mods;
	    void (*func_cb)(GtkWidget *, gpointer);

#define DO_ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
  icon, label, accel_key, accel_mods, NULL,	\
  mclient_data, func_cb				\
 );						\
}
#define DO_ADD_MENU_SEP		{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}
	    icon = (guint8 **)icon_reload_20x20_xpm;
	    label =
#if defined(PROG_LANGUAGE_SPANISH)
"Refresque"
#elif defined(PROG_LANGUAGE_FRENCH)
"Rafrachir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Erfrischen Sie"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Rinfrescare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verfris"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Refresque-se"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forfrisk"
#else
"Refresh"
#endif
	    ;
	    accel_key = GDK_F5;
	    accel_mods = 0;
	    func_cb = EDVHistoryWinRefreshCB;
	    DO_ADD_MENU_ITEM_LABEL
	    lw->refresh_mi = w;

	    DO_ADD_MENU_SEP

	    icon = (guint8 **)icon_save_as_20x20_xpm;
	    label =
#if defined(PROG_LANGUAGE_SPANISH)
"Salve Como"
#elif defined(PROG_LANGUAGE_FRENCH)
"Enregister sous"
#elif defined(PROG_LANGUAGE_GERMAN)
"Auer Als"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Risparmiare Come"
#elif defined(PROG_LANGUAGE_DUTCH)
"Behalve Als"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Exceto Como"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Untatt Som"
#else
"Save As"
#endif
		"...";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVHistoryWinSaveAsCB;
	    DO_ADD_MENU_ITEM_LABEL
	    lw->save_as_mi = w;

	    DO_ADD_MENU_SEP

	    icon = (guint8 **)icon_clear_20x20_xpm;
	    label =
#if defined(PROG_LANGUAGE_SPANISH)
"Claro"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klar"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Chiaro"
#elif defined(PROG_LANGUAGE_DUTCH)
"Helder"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Claro"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klar"
#else
"Clear"
#endif
	    ;
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVHistoryWinClearCB;
	    DO_ADD_MENU_ITEM_LABEL
	    lw->clear_mi = w;


#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SEP
	}


	/* Set initial RC styles */
	if(standard_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		lw->toplevel, standard_rcstyle
	    );
	if(lists_rcstyle != NULL)
	    gtk_widget_modify_style(
		lw->history_clist, lists_rcstyle
	    );
	EDVHistoryWinListStylesRegenerate(lw);


	/* Set the initial display */
	EDVHistoryWinUpdateDisplay(lw, NULL);


	EDVHistoryWinUpdateMenus(lw);

	lw->freeze_count--;

	return(lw);
}

/*
 *	Updates menus and other widgets on the given history list
 *	window to reflect its current data.
 */
void EDVHistoryWinUpdateMenus(edv_history_win_struct *lw)
{
	gint sel_row;
	GtkCList *clist;

	if(lw == NULL)
	    return;

	/* Get selected row on clist */
	clist = (GtkCList *)lw->history_clist;
	sel_row = EDVCListGetSelectedLast(clist, NULL);

}

/*
 *	Marks the history list window as busy or ready.
 */
void EDVHistoryWinSetBusy(
	edv_history_win_struct *lw, const gboolean is_busy
)
{
	GdkCursor *cursor;
	GtkWidget *w;
	edv_core_struct *core;

	if(lw == NULL)
	    return;

	w = lw->toplevel;
	core = lw->core;

	if(is_busy)
	{
	    /* Increase busy count */
	    lw->busy_count++;

	    /* If already busy then don't change anything */
	    if(lw->busy_count > 1)
		return;

	    cursor = EDVGetCursor(core, EDV_CURSOR_CODE_BUSY);
	}
	else
	{
	    /* Reduce busy count */
	    lw->busy_count--;
	    if(lw->busy_count < 0)
		lw->busy_count = 0;

	    /* If still busy do not change anything */
	    if(lw->busy_count > 0)
		return;

	    cursor = NULL;		/* Use default cursor */
	}
	gdk_window_set_cursor(w->window, cursor);
	gdk_flush();
}


/*
 *	Checks if the History List Window is mapped.
 */
gboolean EDVHistoryWinIsMapped(edv_history_win_struct *lw)
{
	if(lw == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(lw->toplevel));
}

/*
 *	Maps the History List Window.
 */
void EDVHistoryWinMap(edv_history_win_struct *lw)
{
	GtkWidget *w;

	if(lw == NULL)
	    return;

	gtk_widget_show_raise(lw->toplevel);

	/* Grab focus and default on the Close button */
	w = lw->close_btn;
	if(w != NULL)
	{
	    gtk_widget_grab_focus(w);
	    gtk_widget_grab_default(w);
	}
}

/*
 *	Unmaps the History List Window.
 */
void EDVHistoryWinUnmap(edv_history_win_struct *lw)
{
	if(lw == NULL)
	    return;

	gtk_widget_hide(lw->toplevel);
}

/*
 *	Deletes the History List Window.
 */
void EDVHistoryWinDelete(edv_history_win_struct *lw)
{
	gint i;

	if(lw == NULL)
	    return;

	EDVHistoryWinUnmap(lw);

	lw->freeze_count++;

	GTK_WIDGET_DESTROY(lw->menu);

	GTK_WIDGET_DESTROY(lw->toplevel);

	GTK_ACCEL_GROUP_UNREF(lw->accelgrp);

	GDK_PIXMAP_UNREF(lw->other_operation_pixmap);
	GDK_BITMAP_UNREF(lw->other_operation_mask);

	GDK_PIXMAP_UNREF(lw->recycle_bin_pixmap);
	GDK_BITMAP_UNREF(lw->recycle_bin_mask);

	GDK_PIXMAP_UNREF(lw->object_create_pixmap);
	GDK_BITMAP_UNREF(lw->object_create_mask);
	GDK_PIXMAP_UNREF(lw->object_copy_pixmap);
	GDK_BITMAP_UNREF(lw->object_copy_mask);
	GDK_PIXMAP_UNREF(lw->object_move_pixmap);
	GDK_BITMAP_UNREF(lw->object_move_mask);
	GDK_PIXMAP_UNREF(lw->object_link_pixmap);
	GDK_BITMAP_UNREF(lw->object_link_mask);
	GDK_PIXMAP_UNREF(lw->object_chown_pixmap);
	GDK_BITMAP_UNREF(lw->object_chown_mask);
	GDK_PIXMAP_UNREF(lw->object_chmod_pixmap);
	GDK_BITMAP_UNREF(lw->object_chmod_mask);
	GDK_PIXMAP_UNREF(lw->object_chtime_pixmap);
	GDK_BITMAP_UNREF(lw->object_chtime_mask);
	GDK_PIXMAP_UNREF(lw->object_download_pixmap);
	GDK_BITMAP_UNREF(lw->object_download_mask);
	GDK_PIXMAP_UNREF(lw->object_delete_pixmap);
	GDK_BITMAP_UNREF(lw->object_delete_mask);

	GDK_PIXMAP_UNREF(lw->recobj_recover_pixmap);
	GDK_BITMAP_UNREF(lw->recobj_recover_mask);
	GDK_PIXMAP_UNREF(lw->recobj_purge_pixmap);
	GDK_BITMAP_UNREF(lw->recobj_purge_mask);

	GDK_PIXMAP_UNREF(lw->arcobj_add_pixmap);
	GDK_BITMAP_UNREF(lw->arcobj_add_mask);
	GDK_PIXMAP_UNREF(lw->arcobj_extract_pixmap);
	GDK_BITMAP_UNREF(lw->arcobj_extract_mask);
	GDK_PIXMAP_UNREF(lw->arcobj_delete_pixmap);
	GDK_BITMAP_UNREF(lw->arcobj_delete_mask);
	GDK_PIXMAP_UNREF(lw->archive_fix_pixmap);
	GDK_BITMAP_UNREF(lw->archive_fix_mask);

	GDK_PIXMAP_UNREF(lw->object_other_small_pixmap);
	GDK_BITMAP_UNREF(lw->object_other_small_mask);
	GDK_PIXMAP_UNREF(lw->object_other_medium_pixmap);
	GDK_BITMAP_UNREF(lw->object_other_medium_mask);

	GDK_PIXMAP_UNREF(lw->error_small_pixmap);
	GDK_BITMAP_UNREF(lw->error_small_mask);
	GDK_PIXMAP_UNREF(lw->error_medium_pixmap);
	GDK_BITMAP_UNREF(lw->error_medium_mask);

	/* Unref all cell styles */
	for(i = 0; i < EDV_HISTORY_TOTAL_CELL_STYLES; i++)
	    GTK_STYLE_UNREF(lw->cell_style[i]);

	lw->freeze_count--;

	g_free(lw);
}
