#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <unistd.h>

#include "../include/string.h"

#include "guiutils.h"
#include "pulist.h"
#include "cdialog.h"
#include "fb.h"

#include "url.h"
#include "edv_types.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "mime_types_list_win.h"
#include "endeavour2.h"
#include "edv_cb.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_list_cb.h"
#include "config.h"
#include "edv_cfg_list.h"

#include "images/icon_add_20x20.xpm"
#include "images/icon_edit_20x20.xpm"
#include "images/icon_remove_20x20.xpm"
#include "images/icon_browse_20x20.xpm"
#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_mimetypes_48x48.xpm"


typedef struct _CmdDlg			CmdDlg;
#define CMD_DLG(p)			((CmdDlg *)(p))


/* Callbacks */
static void EDVMimeTypeEditDlgCommandDestroyCB(gpointer data);
static void EDVMimeTypeEditDlgIconChangeCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgIconPrevCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgIconNextCB(
	GtkWidget *widget, gpointer data
);

static gint EDVMimeTypeEditDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVMimeTypeEditDlgAnyChangedCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgClassChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
);
static void EDVMimeTypeEditDlgHandlerChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
);
static gint EDVMimeTypeEditDlgCommandsCListEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data 
);
static gboolean EDVMimeTypeEditDlgCommandsCListDragMotionCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y, guint t,
	gpointer data
);
static void EDVMimeTypeEditDlgCommandsCListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void EDVMimeTypeEditDlgCommandsCListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void EDVMimeTypeEditDlgCommandsCListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);
static void EDVMimeTypeEditDlgCommandsCListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void EDVMimeTypeEditDlgCommandsCListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void EDVMimeTypeEditDlgCommandAddCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgCommandEditCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgCommandRemoveCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgCommandShiftUpCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgCommandShiftDownCB(
	GtkWidget *widget, gpointer data
);

static void EDVMimeTypeEditDlgOKCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgApplyCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgCancelCB(
	GtkWidget *widget, gpointer data
);

void EDVMimeTypeEditDlgMimeTypeRemovedCB(
	edv_mime_type_edit_dlg_struct *d, gint mt_num
);

static gint CmdDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void CmdDlgChangedCB(GtkWidget *widget, gpointer data);
static void CmdDlgCListDragDataGet(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void CmdDlgCListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void CmdDlgCListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void CmdDlgTokensCB(GtkWidget *widget, gpointer data);
static void CmdDlgInsertTokenCB(GtkWidget *widget, gpointer data);
static void CmdDlgOKCB(GtkWidget *widget, gpointer data);
static void CmdDlgCancelCB(GtkWidget *widget, gpointer data);
static void CmdDlgCloseCB(GtkWidget *widget, gpointer data);

/* Utilities */
static edv_mime_type_class EDVMimeTypeEditDlgGetClass(
	edv_mime_type_edit_dlg_struct *d
);
static edv_mime_type_handler EDVMimeTypeEditDlgGetHandler(
	edv_mime_type_edit_dlg_struct *d
);
static void EDVMimeTypeEditDlgIconUpdateDisplay(
	edv_mime_type_icon_picker_struct *ei,
	const edv_mime_type_icon_state state
);
static gint EDVMimeTypeEditDlgCheckMIMEType(
	edv_core_struct *core,
	edv_mime_type_struct **list, const gint total,
	const gint mt_num,
	const gboolean verbose, GtkWidget *toplevel
);

/* Get/set values */
void EDVMimeTypeEditDlgGetValues(
	edv_mime_type_edit_dlg_struct *d, const gint mt_num
);
void EDVMimeTypeEditDlgSetValues(
	edv_mime_type_edit_dlg_struct *d, const gint mt_num,
	const gboolean verbose
);

/* Commands Add/Edit/Remove/Etc */
static void EDVMimeTypeEditDlgCommandAdd(
	edv_mime_type_edit_dlg_struct *d
);
static CmdDlg *CmdDlgNew(
	edv_core_struct *core,
	GtkWidget *ref_toplevel,
	edv_mime_type_command_struct *cmd,
	const gboolean is_new
);
static void CmdDlgDelete(CmdDlg *d);
static gboolean EDVMimeTypeEditDlgCommandEdit(
	edv_mime_type_edit_dlg_struct *d,
	const gboolean is_new
);
static void EDVMimeTypeEditDlgCommandRemove(
	edv_mime_type_edit_dlg_struct *d, const gboolean confirm
);
static void EDVMimeTypeEditDlgCommandShiftUp(
	edv_mime_type_edit_dlg_struct *d
);
static void EDVMimeTypeEditDlgCommandShiftDown(
	edv_mime_type_edit_dlg_struct *d
);

/* Commands List & Items */
static gint EDVMimeTypeEditDlgCommandsListInsert(
	edv_mime_type_edit_dlg_struct *d,
	const gint cmd_num,
	edv_mime_type_command_struct *cmd
);
static gint EDVMimeTypeEditDlgCommandsListAppend(
	edv_mime_type_edit_dlg_struct *d,
	edv_mime_type_command_struct *cmd
);
static void EDVMimeTypeEditDlgCommandsListSet(
	edv_mime_type_edit_dlg_struct *d,
	const gint cmd_num,
	edv_mime_type_command_struct *cmd
);

/* Icon Picker */
static edv_mime_type_icon_picker_struct *EDVMimeTypeEditDlgIconNew(
	edv_mime_type_edit_dlg_struct *d,
	GtkWidget *parent, const gchar *title,
	const gint icon_width, const gint icon_height,
	const gint btn_width, const gint btn_height
);
static void EDVMimeTypeEditDlgIconDelete(edv_mime_type_icon_picker_struct *ei);

/* MIME Type Edit Dialog */
edv_mime_type_edit_dlg_struct *EDVMimeTypeEditDlgNew(
	edv_core_struct *core, edv_mime_types_list_win_struct *lw
);
void EDVMimeTypeEditDlgResetHasChanges(
	edv_mime_type_edit_dlg_struct *d,
	const gboolean has_changes
);
void EDVMimeTypeEditDlgUpdateMenus(edv_mime_type_edit_dlg_struct *d);
gboolean EDVMimeTypeEditDlgIsMapped(edv_mime_type_edit_dlg_struct *d);
void EDVMimeTypeEditDlgMap(edv_mime_type_edit_dlg_struct *d);
void EDVMimeTypeEditDlgUnmap(edv_mime_type_edit_dlg_struct *d);
void EDVMimeTypeEditDlgDelete(edv_mime_type_edit_dlg_struct *d);


struct _CmdDlg {
	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count;
	gboolean	has_changes,
			got_response;
	edv_core_struct	*core;
	GtkWidget	*name_entry,
			*command_entry,
			*tokens_btn,
			*tokens_box,
			*tokens_clist,
			*tokens_insert_btn,
			*ok_btn,
			*cancel_btn,
			*close_btn;
};


#define EDITWIN_WIDTH		500
#define EDITWIN_HEIGHT		-1

#define EDITWIN_TITLE		"MIME Type"


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Commands GtkCList item destroy signal callback.
 */
static void EDVMimeTypeEditDlgCommandDestroyCB(gpointer data)
{
	EDVMimeTypeCommandDelete(EDV_MIME_TYPE_COMMAND(data));
}

/*
 *	Icon change callback.
 */
static void EDVMimeTypeEditDlgIconChangeCB(
	GtkWidget *widget, gpointer data
)
{
	gboolean response;
	gint nftypes = 0, npaths = 0;
	const gchar *old_path;
	gchar *old_parent_path, **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	edv_mime_type_icon_state state;
	edv_mime_type_edit_dlg_struct *d;
	edv_core_struct *core;
	edv_mime_type_icon_picker_struct *ei =
	    EDV_MIME_TYPE_ICON_PICKER(data);
	if((ei == NULL) || FileBrowserIsQuery())
	    return;

	/* Get icon state */
	state = (edv_mime_type_icon_state)GTK_ADJUSTMENT_GET_VALUE(ei->adj);
	if((state < 0) || (state >= EDV_MIME_TYPE_TOTAL_ICON_STATES))
	    return;

	d = EDV_MIME_TYPE_EDIT_DLG(ei->d);
	toplevel = d->toplevel;
	core = d->core;

	if(d->freeze_count > 0)
	    return;

	/* Get the current path to the icon file */
	old_path = ei->path[state];
	if(old_path != NULL)
	    old_parent_path = g_dirname(old_path);
	else
	    old_parent_path = NULL;

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    ".xpm", "X Pixmap"
	);
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Current old_path must be considered invalid at this point */
	old_path = NULL;

	/* Query the user for an image */
	d->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Icono Selecto",
"Selecto",
"Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
"Icne Privilgie",
"Privilgi",
"Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
"Erlesenes Abbild",
"Erlesen",
"Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scegliere L'Icona",
"Scegliere",
"Annullare",
#elif defined(PROG_LANGUAGE_DUTCH)
"Uitgezochte Beeld",
"Uitgezocht",
"Annuleer",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Selecione Icone",
"Selecione",
"Cancelamento",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Utvalgt Ikon",
"Utvalgt",
"Kanseller",
#else
"Select Icon",
"Select",
"Cancel",
#endif
	    old_parent_path,
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
	d->freeze_count--;

	/* Got user response? */
	if(response)
	{
	    const gchar *new_path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(new_path != NULL)
	    {
		/* Load new pixmap and mask */
		GdkBitmap *mask;
		GdkPixmap *pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(
		    &mask, new_path
		);
		if(pixmap != NULL)
		{
		    /* New pixmap and mask opened successfully */

		    /* Unref the old and set the new pixmap and mask */
		    GDK_PIXMAP_UNREF(ei->pixmap[state]);
		    GDK_BITMAP_UNREF(ei->mask[state]);
		    EDVResizePixmap(
			pixmap, mask,
			ei->icon_width, ei->icon_height,
			&ei->pixmap[state], &ei->mask[state]
		    );
		    GDK_PIXMAP_UNREF(pixmap);
		    GDK_BITMAP_UNREF(mask);

		    /* Set the new icon file path */
		    g_free(ei->path[state]);
		    ei->path[state] = STRDUP(new_path);
		}
		else
		{   
		    GDK_PIXMAP_UNREF(ei->pixmap[state]);
		    ei->pixmap[state] = NULL;

		    GDK_BITMAP_UNREF(ei->mask[state]);
		    ei->mask[state] = NULL;

		    g_free(ei->path[state]);
		    ei->path[state] = NULL;
		}

		/* Update the displayed icon and mark has changes */
		EDVMimeTypeEditDlgIconUpdateDisplay(ei, state);
		EDVMimeTypeEditDlgResetHasChanges(d, TRUE);
	    }
	}


	/* Delete file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	g_free(old_parent_path);
}

/*
 *      Prev icon edit window icon callback.
 */
static void EDVMimeTypeEditDlgIconPrevCB(
	GtkWidget *widget, gpointer data
)
{
	GtkAdjustment *adj;
	edv_mime_type_edit_dlg_struct *d;
	edv_mime_type_icon_picker_struct *ei =
	    EDV_MIME_TYPE_ICON_PICKER(data);
	if(ei == NULL)
	    return;

	d = EDV_MIME_TYPE_EDIT_DLG(ei->d);
	if(d == NULL)
	    return;

	adj = ei->adj;
	if(adj == NULL)
	    return;

	/* Increment adjustment position */
	if(adj->value > adj->lower)
	    adj->value -= adj->step_increment;
	if(adj->value < adj->lower)
	    adj->value = adj->lower;

	EDVMimeTypeEditDlgIconUpdateDisplay(
	    ei, (edv_mime_type_icon_state)adj->value
	);
}

/*
 *	Next icon edit window icon callback.
 */
static void EDVMimeTypeEditDlgIconNextCB(
	GtkWidget *widget, gpointer data
)
{
	GtkAdjustment *adj;
	edv_mime_type_edit_dlg_struct *d;
	edv_mime_type_icon_picker_struct *ei =
	    EDV_MIME_TYPE_ICON_PICKER(data);
	if(ei == NULL)
	    return;

	d = EDV_MIME_TYPE_EDIT_DLG(ei->d);
	if(d == NULL)
	    return;

	adj = ei->adj;
	if(adj == NULL)
	    return;

	/* Increment adjustment position */
	if(adj->value < adj->upper)
	    adj->value += adj->step_increment;
	if(adj->value > adj->upper)
	    adj->value = adj->upper;

	EDVMimeTypeEditDlgIconUpdateDisplay(
	    ei, (edv_mime_type_icon_state)adj->value
	);
}


/*
 *	GtkWindow "delete_event" signal callback.
 */
static gint EDVMimeTypeEditDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return(TRUE);

	EDVMimeTypeEditDlgCancelCB(widget, d);

	return(TRUE);
}

/*
 *	Any GtkWidget changed signal callback.
 */
static void EDVMimeTypeEditDlgAnyChangedCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	if(!d->has_changes)
	    EDVMimeTypeEditDlgResetHasChanges(d, TRUE);

	d->freeze_count--;
}

/*
 *	Class popup list box "changed" signal callback.
 *
 *	This will update the value_label and value_entry widgets.
 */
static void EDVMimeTypeEditDlgClassChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
)
{
	gboolean sensitive = TRUE;
	edv_mime_type_class mt_class;
	const gchar *value_label_str = "";
	GtkWidget *w;
	pulist_struct *pulist;
	pulistbox_struct *pulistbox2;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((pulistbox == NULL) || (d == NULL))
	    return;

	pulist = PUListBoxGetPUList(d->class_pulistbox);
	if(pulist == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	/* Get the new class */
	mt_class = (edv_mime_type_class)PUListGetItemData(pulist, i);

	/* Update the widgets depending on class */
	switch(mt_class)
	{
	  case EDV_MIME_TYPE_CLASS_SYSTEM:
	    sensitive = FALSE;
	    value_label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"El Valor"
#elif defined(PROG_LANGUAGE_FRENCH)
"Valeur"
#elif defined(PROG_LANGUAGE_GERMAN)
"Wert"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Valore"
#elif defined(PROG_LANGUAGE_DUTCH)
"Waarde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Valor"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Verdi"
#else
"Value"
#endif
		":";
	    break;

	  case EDV_MIME_TYPE_CLASS_FORMAT:
	    sensitive = TRUE;
	    value_label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"Las Extensiones"
#elif defined(PROG_LANGUAGE_FRENCH)
"Extensions"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verlngerungen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Le Estensioni"
#elif defined(PROG_LANGUAGE_DUTCH)
"Uitbreidingen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"As Extenses"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Utvidelser"
#else
"Extensions"
#endif
		":";
	    break;

	  case EDV_MIME_TYPE_CLASS_PROGRAM:
	    sensitive = TRUE;
	    value_label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"La Ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ort"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plaats"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Plassering"
#else
"Location"
#endif
		":";
	    break;

	  case EDV_MIME_TYPE_CLASS_UNIQUE:
	    sensitive = TRUE;
	    value_label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"La Ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ort"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plaats"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Plassering"
#else
"Location"
#endif
		":";
	    break;
	}

	gtk_label_set_text(GTK_LABEL(d->value_label), value_label_str);

	gtk_widget_set_sensitive(d->value_entry, sensitive);

	pulistbox2 = d->handler_pulistbox;
	if(pulistbox2 != NULL)
	{
	    w = pulistbox2->toplevel;
	    gtk_widget_set_sensitive(w, sensitive);
	}

	gtk_widget_set_sensitive(d->commands_clist_box, sensitive);

	d->freeze_count--;

	EDVMimeTypeEditDlgAnyChangedCB(pulistbox->toplevel, data);
}

/*
 *	Handler popup list box "changed" signal callback.
 *
 *	This will update the sensitivity or mapping of some widgets
 *	such as the command editable clist.
 */
static void EDVMimeTypeEditDlgHandlerChangedCB(
	pulistbox_struct *pulistbox, gint i, gpointer data
)
{
	gboolean sensitive = FALSE;
	edv_mime_type_class mt_class;
	edv_mime_type_handler handler;
	pulist_struct *pulist;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((pulistbox == NULL) || (d == NULL))
	    return;

	pulist = PUListBoxGetPUList(d->handler_pulistbox);
	if(pulist == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	/* Get the current class and handler values */
	mt_class = EDVMimeTypeEditDlgGetClass(d);
	handler = (edv_mime_type_handler)PUListGetItemData(pulist, i);

	/* Update widgets depending on class */
	switch(handler)
	{
	  case EDV_MIME_TYPE_HANDLER_COMMAND:
	    switch(mt_class)
	    {
	      case EDV_MIME_TYPE_CLASS_SYSTEM:
		sensitive = FALSE;
		break;
	      case EDV_MIME_TYPE_CLASS_FORMAT:
	      case EDV_MIME_TYPE_CLASS_PROGRAM:
	      case EDV_MIME_TYPE_CLASS_UNIQUE:
		sensitive = TRUE;
		break;
	    }
	    break;
	  case EDV_MIME_TYPE_HANDLER_EDV_ARCHIVER:
	  case EDV_MIME_TYPE_HANDLER_EDV_IMAGE_BROWSER:
	  case EDV_MIME_TYPE_HANDLER_EDV_RECYCLE_BIN:
	    sensitive = FALSE;
	    break;
	}

	gtk_widget_set_sensitive(d->commands_clist_box, sensitive);

	d->freeze_count--;

	EDVMimeTypeEditDlgAnyChangedCB(pulistbox->toplevel, data);
}

/*
 *	Commands GtkCList event signal callback.
 */
static gint EDVMimeTypeEditDlgCommandsCListEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data 
)
{
	gint status = FALSE;
	gboolean key_press;
	gint row, column;
	GdkEventKey *key;
	GdkEventButton *button;
	GtkCList *clist;
	cfg_item_struct *cfg_list;
	edv_core_struct *core;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	if(d->freeze_count > 0)
	    return(status);

	core = d->core;
	cfg_list = core->cfg_list;
			   
	switch((gint)event->type)
	{
	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    key_press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	    switch(key->keyval)
	    {
	      case GDK_Return:
		if(key_press)
		    EDVMimeTypeEditDlgCommandEdit(d, FALSE);
		status = TRUE;
		break;

	      case GDK_Delete:
		if(key_press)
		    EDVMimeTypeEditDlgCommandRemove(d, TRUE);
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {              
	      case GDK_BUTTON3:
		clist = GTK_CLIST(d->commands_clist);
		if(!gtk_clist_get_selection_info(
		    clist, button->x, button->y, &row, &column
		))
		{
		    row = -1;
		    column = 0;
		}
		/* Select the row before mapping menu? */
		if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
		   (row >= 0) && (row < clist->rows)
		)
		{
		    gtk_clist_freeze(clist);
		    gtk_clist_select_row(clist, row, column);
		    gtk_clist_thaw(clist);
		}

		/* Map the right-click menu */
		gtk_menu_popup(
		    GTK_MENU(d->commands_menu), NULL, NULL,
		    NULL, NULL,
		    button->button, button->time
		);
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_2BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON1:
		EDVMimeTypeEditDlgCommandEdit(d, FALSE);
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_BUTTON_RELEASE:
	    break;
	}

	return(status);
}


/*
 *	Commands GtkCList "drag_motion" signal callback.
 */
static gboolean EDVMimeTypeEditDlgCommandsCListDragMotionCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y, guint t,
	gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((widget == NULL) || (dc == NULL) || (d == NULL))
	    return(FALSE);

	if(d->freeze_count > 0)
	    return(FALSE);

	if(gtk_drag_get_source_widget(dc) == widget)
	{
	    if(dc->actions & GDK_ACTION_MOVE)
		gdk_drag_status(dc, GDK_ACTION_MOVE, t);
	    else if(dc->actions & GDK_ACTION_COPY)
		gdk_drag_status(dc, GDK_ACTION_COPY, t);
	    else
		gdk_drag_status(dc, 0, t);
	}
	else
	{
	    if(dc->actions & GDK_ACTION_COPY)
		gdk_drag_status(dc, GDK_ACTION_COPY, t);
	    else
		gdk_drag_status(dc, 0, t);
	}

	return(TRUE);
}

/*
 *	Commands GtkCList "drag_data_get" signal callback.
 */
static void EDVMimeTypeEditDlgCommandsCListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	gint row;
	GList *glist, *cmds_list;
	GtkCList *clist;
	edv_mime_type_command_struct *cmd;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((dc == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->commands_clist);

	/* Create a list of selected commands */
	cmds_list = NULL;
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)                             
	{
	    row = (gint)glist->data;
	    cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, row));
	    if(cmd == NULL)
		continue;

	    cmds_list = g_list_append(cmds_list, cmd);
	}                                           
	if(cmds_list != NULL)
	{
	    gint buf_len;
	    guint8 *buf = EDVMimeTypeCommandsListToDDEBuffer(
		cmds_list, &buf_len
	    );

	    g_list_free(cmds_list);

	    if(buf != NULL)
	    {
		gtk_selection_data_set(
		    selection_data,
		    GDK_SELECTION_TYPE_STRING,
		    8,			/* Bits Per Character */
		    buf,		/* Data */
		    buf_len		/* Length */
		);
		data_sent = TRUE;
		g_free(buf);
	    }
	}

	if(!data_sent)
	{
	    const gchar *s = "Error";
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,			/* Bits Per Character */
		s,			/* Data */
		STRLEN(s)		/* Length */
	    );
	}
}

/*
 *	Commands GtkCList "drag_data_received" signal callback.
 */
static void EDVMimeTypeEditDlgCommandsCListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gint row = -1, column = 0;
	GtkCList *clist;
	edv_core_struct *core;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((dc == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->commands_clist);
	core = d->core;

	/* Find the dropped on row and column */
	gtk_clist_get_selection_info(
	    clist,
	    x,    
	    y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
		clist->column_title_area.height +
		clist->column_title_area.y : 0),
	    &row, &column
	);

	/* Handle by the target type
	 *
	 * MIME Type Command
	 */
	if(info == EDV_DND_INFO_MIME_TYPE_COMMAND)
	{
	    /* Get the commands list and insert it to the GtkCList */
	    gint	insert_row = row,
			new_row;
	    GList	*glist,
			*cmds_list = EDVMimeTypeCommandsListFromDDEBuffer(
		selection_data->data, selection_data->length
	    );
	    edv_mime_type_command_struct *cmd;

	    /* Insert each command to the GtkCList */
	    gtk_clist_freeze(clist);
	    for(glist = cmds_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		cmd = EDV_MIME_TYPE_COMMAND(glist->data);
		if(cmd == NULL)
		    continue;

		if(insert_row > -1)
		{
		    new_row = EDVMimeTypeEditDlgCommandsListInsert(
			d, insert_row, cmd
		    );
		    glist->data = NULL;		/* Transfered */
		    insert_row++;
		}
		else
		{
		    new_row = EDVMimeTypeEditDlgCommandsListAppend(
			d, cmd
		    );
		    glist->data = NULL;		/* Transfered */
		}
		if(new_row < 0)
		    break;
	    }
	    gtk_clist_thaw(clist);

	    /* Delete the parsed commands list */
	    if(cmds_list != NULL)
	    {
		g_list_foreach(
		    cmds_list, (GFunc)EDVMimeTypeCommandDelete, NULL
		);
		g_list_free(cmds_list);
	    }

	    EDVMimeTypeEditDlgAnyChangedCB(widget, data);
	    EDVMimeTypeEditDlgUpdateMenus(d);
	}
	/* MIME Type */
	else if(info == EDV_DND_INFO_MIME_TYPE)
	{
	    /* Get the commands list and insert it to the GtkCList */
	    gint	insert_row = row,
			new_row;
	    GList	*glist,
			*mt_list = EDVMimeTypesListFromDDEBuffer(
		core->mimetype, core->total_mimetypes,
		selection_data->data, selection_data->length
	    );
	    edv_mime_type_struct *m;
	    edv_mime_type_command_struct *cmd;

	    /* Iterate through the parsed MIME Types list */
	    gtk_clist_freeze(clist);
	    for(glist = mt_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
		    continue;

		/* Create a new command */
		cmd = EDVMimeTypeCommandNew();
		g_free(cmd->name);
		if((clist->rows == 0) || (insert_row == 0))
		    cmd->name = STRDUP("Default");
		else
		    cmd->name = g_strdup_printf(
			"Command #%i",
			(insert_row > -1) ?
			    (insert_row + 1) : (clist->rows + 1)
		    );
		g_free(cmd->command);
		cmd->command = STRDUP(m->type);

		/* Insert the new command to the GtkCList */
		if(insert_row > -1)
		{
		    new_row = EDVMimeTypeEditDlgCommandsListInsert(
			d, insert_row, cmd
		    );
		    cmd = NULL;
		    insert_row++;
		}
		else
		{
		    new_row = EDVMimeTypeEditDlgCommandsListAppend(
			d, cmd
		    );
		    cmd = NULL;
		}
		if(new_row < 0)
		    break;
	    }
	    gtk_clist_thaw(clist);

	    /* Delete the parsed MIME Types list */
	    if(mt_list != NULL)
	    {
		g_list_foreach(
		    mt_list, (GFunc)EDVMimeTypeDelete, NULL
		);
		g_list_free(mt_list);
	    }

	    EDVMimeTypeEditDlgAnyChangedCB(widget, data);
	    EDVMimeTypeEditDlgUpdateMenus(d);
	}
	/* String */
	else if((info == EDV_DND_INFO_TEXT_PLAIN) ||
		(info == EDV_DND_INFO_TEXT_URI_LIST) ||
		(info == EDV_DND_INFO_STRING)
	)
	{
	    /* Get the commands list and insert it to the GtkCList */
	    gint	insert_row = row,
			new_row;
	    GList	*glist,
			*urls_list = URLDecode(
		(const guint8 *)selection_data->data,
		selection_data->length
	    );
	    url_struct *url;
	    edv_mime_type_command_struct *cmd;

	    /* Iterate through the parsed URLs list */
	    gtk_clist_freeze(clist);
	    for(glist = urls_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		url = URL(glist->data);
		if(url == NULL)
		    continue;

		/* Create a new command */
		cmd = EDVMimeTypeCommandNew();
		g_free(cmd->name);
		if((clist->rows == 0) || (insert_row == 0))
		    cmd->name = STRDUP("Default");
		else
		    cmd->name = g_strdup_printf(
			"Command #%i",
			(insert_row > -1) ?
			    (insert_row + 1) : (clist->rows + 1)
		    );
		g_free(cmd->command);
		cmd->command = g_strconcat(
		    url->path,
		    " \"%p\"",
		    NULL
		);

		/* Insert the new command to the GtkCList */
		if(insert_row > -1)
		{
		    new_row = EDVMimeTypeEditDlgCommandsListInsert(
			d, insert_row, cmd
		    );
		    cmd = NULL;
		    insert_row++;
		}
		else
		{
		    new_row = EDVMimeTypeEditDlgCommandsListAppend(
			d, cmd
		    );
		    cmd = NULL;
		}
		if(new_row < 0)
		    break;
	    }
	    gtk_clist_thaw(clist);

	    /* Delete the URLs list */
	    if(urls_list != NULL)
	    {
		g_list_foreach(
		    urls_list, (GFunc)URLDelete, NULL
		);
		g_list_free(urls_list);
	    }

	    EDVMimeTypeEditDlgAnyChangedCB(widget, data);
	    EDVMimeTypeEditDlgUpdateMenus(d);
	}
}

/*
 *	Commands GtkCList "drag_data_delete" signal callback.
 */
static void EDVMimeTypeEditDlgCommandsCListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	GtkCList *clist;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((dc == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->commands_clist);

	/* Delete the selected commands */
	gtk_clist_freeze(clist);
	while(clist->selection != NULL)
	    gtk_clist_remove(
		clist,
		(gint)clist->selection->data
	    );
	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);
}

/*
 *	Commands GtkCList "select_row" signal callback.
 */
static void EDVMimeTypeEditDlgCommandsCListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((clist == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	if(gtk_clist_row_is_visible(clist, row) !=
	   GTK_VISIBILITY_FULL
	)
	    gtk_clist_moveto(
		clist,
		row, -1,	/* Row, column */
		0.5f, 0.0f	/* Row, column */
	    );

	/* Clear the DND icon */
	GUIDNDSetDragIcon(NULL, NULL, 0, 0);

	EDVMimeTypeEditDlgUpdateMenus(d);
}

/*
 *	Commands GtkCList "unselect_row" signal callback.
 */
static void EDVMimeTypeEditDlgCommandsCListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((clist == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgUpdateMenus(d);
}


/*
 *	Command add callback.
 */
static void EDVMimeTypeEditDlgCommandAddCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;
	
	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgCommandAdd(d);
}

/*
 *	Command edit callback.
 */
static void EDVMimeTypeEditDlgCommandEditCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;
	
	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgCommandEdit(d, FALSE);
}

/*
 *	Command remove callback.
 */
static void EDVMimeTypeEditDlgCommandRemoveCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;
	
	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgCommandRemove(d, TRUE);
}

/*
 *	Command shift up callback.
 */
static void EDVMimeTypeEditDlgCommandShiftUpCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;
	
	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgCommandShiftUp(d);
}

/*
 *	Command shift down callback.
 */
static void EDVMimeTypeEditDlgCommandShiftDownCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;
	
	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgCommandShiftDown(d);
}


/*
 *	OK Callback.
 */
static void EDVMimeTypeEditDlgOKCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgSetValues(d, d->mt_num, TRUE);
	EDVMimeTypeEditDlgUnmap(d);
}

/*
 *	Apply callback.
 */
static void EDVMimeTypeEditDlgApplyCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgSetValues(d, d->mt_num, TRUE);
}

/*
 *	Cancel callback.
 */
static void EDVMimeTypeEditDlgCancelCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgUnmap(d);
}


/*
 *      Notifies the given MIME Types list window that the given MIME Type
 *      has been removed.
 */
void EDVMimeTypeEditDlgMimeTypeRemovedCB(
	edv_mime_type_edit_dlg_struct *d, gint mt_num
)
{
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	/* If the removed MIME Type is the MIME Type that this edit window
	 * is displaying values for then this edit window needs to be
	 * unmapped
	 */
	if(d->mt_num == mt_num)
	{
	    d->mt_num = -1;
	    EDVMimeTypeEditDlgUnmap(d);
	}
}


/*
 *	Command dialog "delete_event" signal callback.
 */
static gint CmdDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	CmdDlgCloseCB(widget, data);
	return(TRUE);
}

/*
 *	Command dialog GtkCList "drag_data_get" signal callback.
 */
static void CmdDlgCListDragDataGet(
	GtkWidget *widget, GdkDragContext *dc,           
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data                                         
)
{
	gint row;
	gchar *s;
	GList *glist;
	GtkCList *clist;
	CmdDlg *d = CMD_DLG(data);
	if((widget == NULL) || (dc == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->tokens_clist);
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

	s = g_strconcat(
	    "\"",
	    (const gchar *)gtk_clist_get_row_data(clist, row),
	    "\"",
	    NULL
	);
	if(s == NULL)
	    s = STRDUP(" ");

	gtk_selection_data_set(
	    selection_data,
	    GDK_SELECTION_TYPE_STRING,
	    8,				/* Bits Per Character */
	    s,				/* Data */
	    STRLEN(s)			/* Length */
	);

	g_free(s);
}

/*
 *	Command dialog GtkCList "select_row" signal callback.
 */
static void CmdDlgCListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	CmdDlg *d = CMD_DLG(data);
	if((clist == NULL) || (event == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	if(gtk_clist_row_is_visible(clist, row) !=
	   GTK_VISIBILITY_FULL
	)
	    gtk_clist_moveto(
		clist,
		row, -1,	/* Row, column */
		0.5f, 0.0f	/* Row, column */
	    );

	/* Clear the DND icon */
	GUIDNDSetDragIcon(NULL, NULL, 0, 0);

	if(event->type == GDK_2BUTTON_PRESS)
	{
	    const gchar *token = (gchar *)gtk_clist_get_row_data(clist, row);
	    if(token != NULL)
	    {
		GtkEditable *editable = GTK_EDITABLE(d->command_entry);
		gint position = gtk_editable_get_position(editable);
		gchar *s = g_strconcat(
		    "\"", token, "\"", NULL
		);
		gtk_editable_insert_text(
		    editable,
		    s,
		    STRLEN(s),
		    &position
		);
		g_free(s);
	    }
	}

	gtk_widget_set_sensitive(d->tokens_insert_btn, TRUE);
}

/*
 *	Command dialog GtkCList "unselect_row" signal callback.
 */
static void CmdDlgCListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	CmdDlg *d = CMD_DLG(data);
	if((clist == NULL) || (event == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	gtk_widget_set_sensitive(d->tokens_insert_btn, FALSE);
}

/*
 *	Command dialog "changed" signal callback.
 */
static void CmdDlgChangedCB(GtkWidget *widget, gpointer data)
{
	CmdDlg *d = CMD_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	if(!d->has_changes)
	{
	    gtk_widget_set_sensitive(d->ok_btn, TRUE);
	    gtk_widget_hide(d->close_btn);
	    gtk_widget_show(d->cancel_btn);
	    d->has_changes = TRUE;
	}
}

/*
 *	Prompt dialog browse command callback.
 */
static void CmdDlgBrowseCB(GtkWidget *widget, gpointer data)
{                                                       
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar *cur_path, **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	CmdDlg *d = CMD_DLG(data);
	if((d == NULL) || FileBrowserIsQuery())
	    return;

	if(d->freeze_count > 0)
	    return;

	toplevel = d->toplevel;

	cur_path = gtk_entry_get_text(GTK_ENTRY(d->command_entry));
	if(cur_path != NULL)
	{
	    if(*cur_path == G_DIR_SEPARATOR)
	    {
		gchar *s;

		cur_path = STRDUP(cur_path);
		s = (gchar *)strpbrk((char *)cur_path, " \t");
		if(s != NULL)
		    *s = '\0';

		s = g_dirname(cur_path);
		g_free(cur_path);
		cur_path = s;
	    }
	    else
	    {
		cur_path = NULL;
	    }
	}

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the new location */
	d->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Program",
	    "Select", "Cancel",
	    cur_path,
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	g_free(cur_path);
	FileBrowserSetTransientFor(NULL);
	d->freeze_count--;

	/* Got user response? */
	if(response)
	{
	    const gchar *path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		gtk_entry_set_text(
		    GTK_ENTRY(d->command_entry),
		    path
		);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);
}

/*
 *	Command dialog tokens callback.
 */
static void CmdDlgTokensCB(GtkWidget *widget, gpointer data)
{
	GtkWidget *w;
	CmdDlg *d = CMD_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	w = d->tokens_box;
	if(GTK_WIDGET_MAPPED(w))
	{
	    GUIButtonPixmapUpdate(
		d->tokens_btn,
		NULL, "Tokens >>>"
	    );
	    gtk_widget_hide(w);
	}
	else
	{
	    GUIButtonPixmapUpdate(
		d->tokens_btn,
		NULL, "Tokens <<<"
	    );
	    gtk_widget_show(w);
	}
}

/*
 *	Command dialog insert token callback.
 */
static void CmdDlgInsertTokenCB(GtkWidget *widget, gpointer data)
{
	gint row;
	const gchar *token;
	GList *glist;
	GtkCList *clist;
	CmdDlg *d = CMD_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->tokens_clist);
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;
	token = (gchar *)gtk_clist_get_row_data(clist, row);
	if(token != NULL)
	{
	    GtkEditable *editable = GTK_EDITABLE(d->command_entry);
	    gint position = gtk_editable_get_position(editable);
	    gchar *s = g_strconcat(
		"\"", token, "\"", NULL
	    );
	    gtk_editable_insert_text(
		editable,
		s,
		STRLEN(s),
		&position
	    );
	    g_free(s);
	}
}

/*
 *	Command dialog ok callback.
 */
static void CmdDlgOKCB(GtkWidget *widget, gpointer data)
{
	CmdDlg *d = CMD_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->got_response = TRUE;
	gtk_main_quit();
}

/*
 *	Command dialog cancel callback.
 */
static void CmdDlgCancelCB(GtkWidget *widget, gpointer data)
{
	CmdDlg *d = CMD_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->got_response = FALSE;
	gtk_main_quit();
}

/*
 *	Command dialog close callback.
 */
static void CmdDlgCloseCB(GtkWidget *widget, gpointer data)
{
	CmdDlg *d = CMD_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->got_response = FALSE;
	gtk_main_quit();
}


/*
 *	Returns the current MIME Type class value set on the dialog.
 */
static edv_mime_type_class EDVMimeTypeEditDlgGetClass(
	edv_mime_type_edit_dlg_struct *d
)
{
	pulist_struct *pulist;
	pulistbox_struct *pulistbox;

	if(d == NULL)
	    return(EDV_MIME_TYPE_CLASS_SYSTEM);

	pulistbox = d->class_pulistbox;
	pulist = PUListBoxGetPUList(pulistbox);
	if(pulist == NULL)
	    return(EDV_MIME_TYPE_CLASS_SYSTEM);

	return((edv_mime_type_class)PUListGetItemData(
	    pulist, PUListBoxGetSelected(pulistbox)
	));
}

/*
 *	Returns the current MIME Type handler value set on the dialog.
 */
static edv_mime_type_handler EDVMimeTypeEditDlgGetHandler(
	edv_mime_type_edit_dlg_struct *d
)
{
	pulist_struct *pulist;
	pulistbox_struct *pulistbox;

	if(d == NULL)
	    return(EDV_MIME_TYPE_HANDLER_COMMAND);

	pulistbox = d->handler_pulistbox;
	pulist = PUListBoxGetPUList(pulistbox);
	if(pulist == NULL)
	    return(EDV_MIME_TYPE_HANDLER_COMMAND);

	return((edv_mime_type_handler)PUListGetItemData(
	    pulist, PUListBoxGetSelected(pulistbox)
	));
}

/*
 *	Updates the icon displayed on the button of the given edit window
 *	icon structure with the corresponding pixmap and mask pair of
 *	the given state.
 */
static void EDVMimeTypeEditDlgIconUpdateDisplay(
	edv_mime_type_icon_picker_struct *ei,
	const edv_mime_type_icon_state state
)
{
	const gchar *label_str = "";
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkWidget *w, *parent;

	if(ei == NULL)
	    return;

	if((state < 0) || (state >= EDV_MIME_TYPE_TOTAL_ICON_STATES))
	    return;

	/* Get references to pixmap and mask pair specified by the
	 * given state
	 */
	pixmap = ei->pixmap[state];
	mask = ei->mask[state];

	/* Get button widget as the parent to create GtkPixmap in */
	parent = ei->button;
	if(parent == NULL)
	    return;

	/* Referenced pixmap valid? */
	if(pixmap != NULL)
	{
	    /* Referenced pixmap is valid, now check if the GtkPixmap
	     * that is to be used to display the pixmap has been
	     * created yet (if not then create a new one)
	     */
	    w = ei->gtk_pixmap;
	    if(w == NULL)
	    {
		ei->gtk_pixmap = w = gtk_pixmap_new(pixmap, mask);
		gtk_container_add(GTK_CONTAINER(parent), w);
		gtk_widget_show(w);
	    }
	    else
	    {
		/* GtkPixmap exists, so just update it */
		gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
		gtk_widget_show(w);
	    }
	}
	else
	{
	    /* No referenced pixmap, so unmap the the GtkPixmap */
	    w = ei->gtk_pixmap;
	    if(w != NULL)
	    {
		gtk_widget_hide(w);
	    }
	}


	/* Update state label, get the currect label string to display
	 * for the state label based on the given state
	 */
	switch(state)
	{
	  case EDV_MIME_TYPE_ICON_STATE_HIDDEN:
	    label_str = "Hidden";
	    break;
	  case EDV_MIME_TYPE_ICON_STATE_EXTENDED:
	    label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"Prolongado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Etendu"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ausgedehnt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Esteso"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verlengde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Estendido"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forlenget"
#else
"Extended"
#endif
	    ;
	    break;
	  case EDV_MIME_TYPE_ICON_STATE_SELECTED:
	    label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"Escogido"
#elif defined(PROG_LANGUAGE_FRENCH)
"Choisi"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ausgewhlt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Scelto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Geselecteerde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Selecionado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Valgt Ut"
#else
"Selected"
#endif
	    ;
	    break;
	  case EDV_MIME_TYPE_ICON_STATE_STANDARD:
	    label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"El Estndar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Normal"
#elif defined(PROG_LANGUAGE_GERMAN)
"Standard"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Norma"
#elif defined(PROG_LANGUAGE_DUTCH)
"Standaard"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Padro"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Standard"
#else
"Standard"
#endif
	    ;
	    break;
	}
	w = ei->state_label;
	if(w != NULL)
	    gtk_label_set_text(GTK_LABEL(w), label_str);


#if 0
	/* Update button sensitivities */
	w = ei->left_btn;
	if(w != NULL)
	    gtk_widget_set_sensitive(
		w,
		(state > 0) ? TRUE : FALSE
	    )
	w = ei->right_btn;
	if(w != NULL)
	    gtk_widget_set_sensitive(
		w,
		(state < (EDV_MIME_TYPE_TOTAL_ICON_STATES - 1)) ? TRUE : FALSE
	    )
#endif
}

/*
 *	Checks all of the specified MIME Type's values to see if they
 *	are any problems
 *
 *	Returns non-zero if there was a problem.
 */
static gint EDVMimeTypeEditDlgCheckMIMEType(
	edv_core_struct *core,
	edv_mime_type_struct **list, const gint total,
	const gint mt_num,
	const gboolean verbose, GtkWidget *toplevel
)
{
	gint status = 0;
	edv_mime_type_class mt_class;
	const gchar *type, *val, *desc;
	edv_mime_type_handler handler;
	const gchar *title =
#if defined(PROG_LANGUAGE_SPANISH)
"MIME Type Advertencia"
#elif defined(PROG_LANGUAGE_FRENCH)
"MIME Type Avertissement"
#elif defined(PROG_LANGUAGE_GERMAN)
"MIME Type Warnen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"MIME Type L'avvertimento"
#elif defined(PROG_LANGUAGE_DUTCH)
"MIME Type Waarschuwen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"MIME Type Aviso"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"MIME Type Varsling"
#else
"MIME Type Warning"
#endif
	;
	edv_mime_type_struct *mt;


	if((core == NULL) || (list == NULL))
	    return(status);

	/* Get this the MIME Type from the MIME Types list */
	mt = ((mt_num >= 0) && (mt_num < total)) ?
	    list[mt_num] : NULL;
	/* Not in the list? */
	if(mt == NULL)
	{
	    status = -1;
	    if(verbose)
	    {
		gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"MIME Type con el ndice #%i no est en la lista"
#elif defined(PROG_LANGUAGE_FRENCH)
"MIME Type avec l'index %i n'est pas dans la liste"
#elif defined(PROG_LANGUAGE_GERMAN)
"MIME Type mit index #%i ist nicht in der liste"
#elif defined(PROG_LANGUAGE_ITALIAN)
"MIME Type con #%i di indice non  nell'elenco"
#elif defined(PROG_LANGUAGE_DUTCH)
"MIME Type met index #%i is niet in de lijst"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"MIME Type com #%i de ndice no est na lista"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"MIME Type med indeks #%i er ikke i listen"
#else
"The MIME Type with index #%i is not in the list"
#endif
		    , mt_num
		);
		EDVPlaySoundError(core);
		EDVMessageError(title, msg, NULL, toplevel);
		g_free(msg);
	    }
	    return(status);
	}

	/* Get MIME Type's values */
	mt_class = mt->mt_class;
	type = mt->type;
	val = mt->value;
	desc = mt->description;
	handler = mt->handler;

	/* Check the MIME Type's type and value
	 *
	 * Handle by the MIME Type class
	 */
	switch(mt_class)
	{
	  case EDV_MIME_TYPE_CLASS_SYSTEM:
	    /* Is the Type defined? */
	    if(STRISEMPTY(type))
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg = STRDUP(
"This MIME Type's Type was not set, there will be no way to\n\
identify it."
		    );
		    EDVPlaySoundWarning(core);
		    EDVMessageWarning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    /* Value is ignored */
	    break;

	  case EDV_MIME_TYPE_CLASS_FORMAT:
	    /* Is the Type defined? */
	    if(STRISEMPTY(type))
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg = STRDUP(
"This MIME Type's Type was not set, there will be no way to\n\
identify it."
		    );
		    EDVPlaySoundWarning(core);
		    EDVMessageWarning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    /* Is the Value defined? */
	    if(STRISEMPTY(val))
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg = STRDUP(
"This MIME Type's Value was not set, this MIME Type will not\n\
identify any formats."
		    );
		    EDVPlaySoundWarning(core);
		    EDVMessageWarning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    /* Check the handler
	     *
	     * Command?
	     */
	    if(handler == EDV_MIME_TYPE_HANDLER_COMMAND)
	    {
		const gchar *name, *command;
		GList *glist;
		edv_mime_type_command_struct *cmd;

		/* Check all the commands */
		for(glist = mt->commands_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    cmd = EDV_MIME_TYPE_COMMAND(glist->data);
		    if(cmd == NULL)
			continue;

		    name = cmd->name;
		    command = cmd->command;
		    if(STRISEMPTY(command))
			continue;

		    /* Reference to another MIME Type? */
		    if(*command != G_DIR_SEPARATOR)
		    {
			/* Check if the referenced MIME Type exists */
			edv_mime_type_struct *m_ref = EDVMimeTypesListMatchType(
			    list, total, NULL, command, TRUE
			);
			if(m_ref == NULL)
			{
			    /* Referenced MIME Type was not found */
			    status = -2;
			    if(verbose)
			    {
			        gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Ordene reffers de \"%s\" a un MIME Type\n\
inexistente llam \"%s\"."
#elif defined(PROG_LANGUAGE_FRENCH)
"la commande \"%s\" fait rfrence  un MIME Type\n\
inexistant nomm \"%s\"."
#elif defined(PROG_LANGUAGE_GERMAN)
"Befehlen Sie \"%s\" reffers zu einem nicht\n\
vorhandenen MIME Type hat gerufen \"%s\"."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Comandare il reffers di \"%s\" a un MIME Type\n\
non-esistente ha chiamato \"%s\"."
#elif defined(PROG_LANGUAGE_DUTCH)
"Beveel aan een onbestaande MIME Type\n\
\"%s\" reffers riep \"%s\"."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comande reffers de \"%s\" a um MIME Type\n\
inexistente chamou \"%s\"."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kommander \"%s\" reffers til en\n\
ikke-eksisterende MIME Type kalt \"%s\"."
#else
"This MIME Type's command \"%s\" refers to another MIME Type:\n\
\n\
    %s\n\
\n\
Which could not be found."
#endif
				    , name, command
				);
				EDVPlaySoundWarning(core);
				EDVMessageWarning(title, msg, NULL, toplevel);
			        g_free(msg);
			    }
			}
		    }
		    else
		    {
			/* Check if the program exists */
			gchar *prog;
			const gchar *s;

			/* Get the program from the command */
			s = (const gchar *)strpbrk(
			    (const char *)command, " \t"
			);
			if(s != NULL)
			{
			    const gint len = s - command;
			    prog = (gchar *)g_malloc((len + 1) * sizeof(gchar));
			    if(len > 0)
				memcpy(prog, command, len * sizeof(gchar));
			    prog[len] = '\0';
			}
			else
			{
			    prog = STRDUP(command);
			}

			/* Got the program from the command? */
			if(prog != NULL)
			{
			    struct stat stat_buf;

			    /* Check if the program does not exist */
			    if(stat((const char *)prog, &stat_buf))
			    {
				const gint error_code = (gint)errno;
				status = -2;
				if(verbose)
				{
				    gchar *msg = g_strdup_printf(
"This MIME Type's command \"%s\" refers to a program:\n\
\n\
    %s\n\
\n\
Which could not be found, %s.",
					name,
					prog,
					g_strerror(error_code)
				    );
				    EDVPlaySoundWarning(core);
				    EDVMessageWarning(title, msg, NULL, toplevel);
				    g_free(msg);
				}
			    }

			    g_free(prog);
			}
		    }
		}	/* Check all the commands */ 
	    }
	    break;

	  case EDV_MIME_TYPE_CLASS_PROGRAM:
	    /* Is the Type defined? */
	    if(STRISEMPTY(type))
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg = STRDUP(
"This MIME Type's Type was not set, there will be no way to\n\
identify it."
		    );
		    EDVPlaySoundWarning(core);
		    EDVMessageWarning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    /* Is the Value defined and does it refer to a valid
	     * program?
	     */
	    if(!STRISEMPTY(val))
	    {
		struct stat stat_buf;
		if(stat((const char *)val, &stat_buf))
		{
		    const gint error_code = (gint)errno;
		    status = -1;
		    if(verbose)
		    {
			gchar *msg = g_strdup_printf(
"Unable to find the location of the program:\n\
\n\
    %s\n\
\n\
%s.",
			    val, g_strerror(error_code)
			);
			EDVPlaySoundWarning(core);
			EDVMessageWarning(title, msg, NULL, toplevel);
			g_free(msg);
		    }
		}
	    }
	    else
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg;
#if defined(PROG_LANGUAGE_SPANISH)
		    msg = STRDUP(
"La ubicacin de objeto de aplicacin no conjunto"
		    );
#elif defined(PROG_LANGUAGE_FRENCH)
		    msg = STRDUP(
"L'emplacement d'objet d'application pas srie"
		    );
#else
		    msg = STRDUP(
"The location of the application object was not set."
		    );
#endif
		    EDVPlaySoundWarning(core);
		    EDVMessageWarning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    /* Check the handler
	     *
	     * Command?
	     */
	    if(handler == EDV_MIME_TYPE_HANDLER_COMMAND)
	    {
		/* At least one command defined? */
		if(mt->commands_list != NULL)
		{
		    const gchar *s, *name, *command;
		    gchar *prog;
		    GList *glist;
		    edv_mime_type_command_struct *cmd;

		    /* Check all the commands */
		    for(glist = mt->commands_list;
		        glist != NULL;
		        glist = g_list_next(glist)
		    )
		    {
			cmd = EDV_MIME_TYPE_COMMAND(glist->data);
			if(cmd == NULL)
			    continue;

			name = cmd->name;
			command = cmd->command;
			if(STRISEMPTY(command))
			    continue;

			/* Check if the program exists, get the program
			 * from the command
			 */
			s = (const gchar *)strpbrk(
			    (const char *)command, " \t"
			);
			if(s != NULL)
			{
			    const gint len = s - command;
			    prog = (gchar *)g_malloc((len + 1) * sizeof(gchar));
			    if(len > 0)
				memcpy(prog, command, len * sizeof(gchar));
			    prog[len] = '\0';
			}
			else
			{
			    prog = STRDUP(command);
			}

			/* Got the program from the command? */
			if(prog != NULL)
			{
			    struct stat stat_buf;

			    /* Check if the program does not exist */
			    if(stat((const char *)prog, &stat_buf))
			    {
				const gint error_code = (gint)errno;
				status = -2;
				if(verbose)
				{
				    gchar *msg = g_strdup_printf(
"This MIME Type's command \"%s\" refers to a program:\n\
\n\
    %s\n\
\n\
Which could not be found, %s.",
					name,
					prog,
					g_strerror(error_code)
				    );
				    EDVPlaySoundWarning(core);
				    EDVMessageWarning(title, msg, NULL, toplevel);
				    g_free(msg);
				}
			    }

			    g_free(prog);
			}
		    }	/* Check all the commands */
		}
		else
		{
		    status = -2;
		    if(verbose)
		    {
			gchar *msg;
#if defined(PROG_LANGUAGE_SPANISH)
			msg = STRDUP(
"Ninguna orden se ha definido en esta clase de la\n\
aplicacin MIME Type, este MIME Type no ser capaz de\n\
ejecutar la aplicacin reffers a."
			);
#elif defined(PROG_LANGUAGE_FRENCH)
			msg = STRDUP(
"Aucune commande n'a t dfinie pour cette classe d'application MIME Type\n\
, le MIME Type ne pourra pas donc\n\
excuter l'application auquel il se rfre."
			);
#else
			msg = STRDUP(
"No command has been defined on this application\n\
class MIME Type, this MIME Type will not be able to\n\
execute the application that it refers to."
			);
#endif
			EDVPlaySoundWarning(core);
			EDVMessageWarning(title, msg, NULL, toplevel);
			g_free(msg);
		    }
		}
	    }
	    break;

	  case EDV_MIME_TYPE_CLASS_UNIQUE:
	    /* Is the Type defined? */
	    if(STRISEMPTY(type))
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg = STRDUP(
"This MIME Type's Type was not set, there will be no way to\n\
identify it."
		    );
		    EDVPlaySoundWarning(core);
		    EDVMessageWarning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    /* Is the Value defined and does it refer to a valid
	     * object?
	     */
	    if(!STRISEMPTY(val))
	    {
		struct stat stat_buf;
		if(stat((const char *)val, &stat_buf))
		{
		    const gint error_code = (gint)errno;
		    status = -1;
		    if(verbose)
		    {
			gchar *msg = g_strdup_printf(
"Unable to find the location of the unique object:\n\
\n\
    %s\n\
\n\
%s.",
			    val, g_strerror(error_code)
			);
			EDVPlaySoundWarning(core);
			EDVMessageWarning(title, msg, NULL, toplevel);
			g_free(msg);
		    }
		}
	    }
	    else
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg;

#if defined(PROG_LANGUAGE_SPANISH)
		    msg = STRDUP(
"La ubicacin de objeto extraordinario no conjunto"
		    );
#elif defined(PROG_LANGUAGE_FRENCH)
		    msg = STRDUP(
"L'emplacement de l'objet spcifique n'est pas dfini"
		    );
#else
		    msg = STRDUP(
"The location of the unique object was not set."
		    );
#endif
		    EDVPlaySoundWarning(core);
		    EDVMessageWarning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    break;

	  default:
	    status = -2;
	    if(verbose)
	    {
		gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Cdigo `%i' no apoyado de clase de MIME Type."
#elif defined(PROG_LANGUAGE_FRENCH)
"Le code de la classe MIME Type `%i' n'est pas support."
#else
"Unsupported MIME Type class code `%i'."
#endif
		    , mt_class
		);
		EDVPlaySoundError(core);
		EDVMessageError(title, msg, NULL, toplevel);
		g_free(msg);
	    }
	    break;
	}

	return(status);
}

/*
 *	Gets the values from the MIME Type and sets them to the
 *	dialog's widgets.
 *
 *	The mt_num specifies the MIME Type.
 */
void EDVMimeTypeEditDlgGetValues(
	edv_mime_type_edit_dlg_struct *d, const gint mt_num
)
{
	GtkEntry *entry;
	GtkCList *clist;
	pulist_struct *pulist;
	pulistbox_struct *pulistbox;
	edv_mime_type_struct *mt;
	edv_core_struct *core;
	edv_mime_type_icon_picker_struct *ei;

	if(d == NULL)
	    return;

	core = d->core;
	if(core == NULL)
	    return;

	/* Get pointer to MIME Type structure on core's MIME Types list */
	if((mt_num < 0) || (mt_num >= core->total_mimetypes))
	    return;
	else
	    mt = core->mimetype[mt_num];
	if(mt == NULL)
	    return;


	/* Begin fetching values */

	/* Set MIME Type index on core structure from the given mt_num */
	d->mt_num = mt_num;

	/* Class */
	pulistbox = d->class_pulistbox;
	pulist = PUListBoxGetPUList(pulistbox);
	if(pulist != NULL)
	{
	    gint i;
	    const gint m = PUListGetTotalItems(pulist);
	    edv_mime_type_class mt_class;

	    for(i = 0; i < m; i++)
	    {
		mt_class = (edv_mime_type_class)PUListGetItemData(pulist, i);
		if(mt_class == mt->mt_class)
		{
		    PUListBoxSelect(pulistbox, i);
		    EDVMimeTypeEditDlgClassChangedCB(pulistbox, i, d);
		    break;
		}
	    }
	}

	/* Type */
	entry = (GtkEntry *)d->type_entry;
	if(entry != NULL)
	{
	    gtk_entry_set_text(
		entry,
		(mt->type != NULL) ? mt->type : ""
	    );
	}

	/* Value */
	entry = (GtkEntry *)d->value_entry;
	if(entry != NULL)
	{
	    gtk_entry_set_text(
		entry,
		(mt->value != NULL) ? mt->value : ""
	    );
	}

	/* Description */
	entry = (GtkEntry *)d->description_entry;
	if(entry != NULL)
	{
	    gtk_entry_set_text(
		entry,
		(mt->description != NULL) ? mt->description : ""
	    );
	}

	/* Handler */
	pulistbox = d->handler_pulistbox;
	pulist = PUListBoxGetPUList(pulistbox);
	if(pulist != NULL)
	{
	    gint i;
	    const gint m = PUListGetTotalItems(pulist);
	    edv_mime_type_handler handler;

	    for(i = 0; i < m; i++)
	    {
		handler = (edv_mime_type_handler)PUListGetItemData(pulist, i);
		if(handler == mt->handler)
		{
		    PUListBoxSelect(pulistbox, i);
		    EDVMimeTypeEditDlgHandlerChangedCB(pulistbox, i, d);
		    break;
		}
	    }
	}

	/* Commands */
	clist = GTK_CLIST(d->commands_clist);
	if(clist != NULL)
	{
	    gint row;
	    GList *glist;
	    edv_mime_type_command_struct *cmd;

	    gtk_clist_freeze(clist);

	    /* Clear the list */
	    gtk_clist_clear(clist);

	    /* Append the commands */
	    for(glist = mt->commands_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		cmd = EDV_MIME_TYPE_COMMAND(glist->data);
		if(cmd == NULL)
		    continue;

		row = EDVMimeTypeEditDlgCommandsListAppend(
		    d, EDVMimeTypeCommandCopy(cmd)
		);
	    }

	    gtk_clist_thaw(clist);
	}


	/* Begin loading icons */

#define DO_LOAD_ICON	{				\
 gint i;						\
 for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++) {	\
  /* Unref existing pixmap and mask on the Edit		\
   * Window's Icon Picker				\
   */							\
  if(ei->pixmap[i] != NULL)				\
   gdk_pixmap_unref(ei->pixmap[i]);			\
  if(ei->mask[i] != NULL)				\
   gdk_bitmap_unref(ei->mask[i]);			\
							\
  /* Increase ref count on new pixmap and mask and set	\
   * if to the Edit Window's Icon Picker		\
   */							\
  if(pixmap[i] != NULL)					\
   gdk_pixmap_ref(pixmap[i]);				\
  if(mask[i] != NULL)					\
   gdk_bitmap_ref(mask[i]);				\
							\
  ei->pixmap[i] = pixmap[i];				\
  ei->mask[i] = mask[i];				\
							\
  /* Set icon path */					\
  g_free(ei->path[i]);					\
  ei->path[i] = STRDUP(path[i]);			\
 }							\
}

	/* Small icons */
	ei = d->icon_small;
	if(ei != NULL)
	{
	    GdkPixmap **pixmap = mt->small_pixmap;
	    GdkBitmap **mask = mt->small_mask;
	    gchar **path = mt->small_icon_file;

	    if(ei->adj != NULL)
		ei->adj->value = 0.0f;

	    DO_LOAD_ICON

	    if(ei->adj != NULL)
		EDVMimeTypeEditDlgIconUpdateDisplay(
		    ei, (edv_mime_type_icon_state)ei->adj->value
		);
	}

	/* Medium icons */
	ei = d->icon_medium;
	if(ei != NULL)
	{
	    GdkPixmap **pixmap = mt->medium_pixmap;
	    GdkBitmap **mask = mt->medium_mask;
	    gchar **path = mt->medium_icon_file;

	    if(ei->adj != NULL)
		ei->adj->value = 0.0f;

	    DO_LOAD_ICON

	    if(ei->adj != NULL)
		EDVMimeTypeEditDlgIconUpdateDisplay(
		    ei, (edv_mime_type_icon_state)ei->adj->value
		);
	}

	/* Large icons */
	ei = d->icon_large;
	if(ei != NULL)
	{
	    GdkPixmap **pixmap = mt->large_pixmap;
	    GdkBitmap **mask = mt->large_mask;
	    gchar **path = mt->large_icon_file;

	    if(ei->adj != NULL)
		ei->adj->value = 0.0f;

	    DO_LOAD_ICON

	    if(ei->adj != NULL)
		EDVMimeTypeEditDlgIconUpdateDisplay(
		    ei, (edv_mime_type_icon_state)ei->adj->value
		);
	}

#undef DO_LOAD_ICON


	EDVMimeTypeEditDlgUpdateMenus(d);
}

/*
 *	Applies the values from the dialog's widgets to the MIME
 *	Type.
 *
 *	The mt_num specifies the MIME Type.
 */
void EDVMimeTypeEditDlgSetValues(
	edv_mime_type_edit_dlg_struct *d, const gint mt_num,
	const gboolean verbose
)
{
	GtkWidget *toplevel;
	GtkEntry *entry;
	GtkCList *clist;
	edv_mime_type_struct *mt;
	edv_mime_types_list_win_struct *lw;
	edv_mime_type_icon_picker_struct *ei;
	edv_core_struct *core;

	if(d == NULL)
	    return;

	toplevel = d->toplevel;
	lw = d->list_win;
	core = d->core;

	/* Get the MIME Type being edited */
	mt = ((mt_num >= 0) && (mt_num < core->total_mimetypes)) ?
	    core->mimetype[mt_num] : NULL;

	/* Does the MIME Type no longer exist? */
	if(mt == NULL)
	{
	    if(verbose)
	    {
		EDVPlaySoundError(core);
		EDVMessageError(
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique Fallado",
"Este MIME Type no ms largo existe, estos valores se tirarn.",
#elif defined(PROG_LANGUAGE_FRENCH)
"Appliquer Echou",
"Ce MIME Type n'existe plu, les valeurs seront rejetes.",
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden Sie Versagt".
"Dieser MIME Type werden nicht mehr, diese werte\n\
ausgeschieden werden existiert.",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare Fallito".
"Questo MIME Type non pi esistere, questi valori saranno scartati.",
#elif defined(PROG_LANGUAGE_DUTCH)
"Pas Geverzuimenene Toe".
"Deze MIME Type zullen, deze waarde niet langer weggegooid\n\
worden bestaat.",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique Fracassado".
"Este MIME Type no existe mais, estes valores sero descartados.",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld Sviktet".
"Denne MIME Type finnes ikke mer, disse verdiene kastet.",
#else
"Apply Failed",
"This MIME Type no longer exists, these values will be discarded.",
#endif
		    NULL,
		    toplevel
		);
	    }
	    return;
	}

	/* MIME Type marked as read only? */
	if(mt->read_only)
	{
	    if(verbose)
	    {
		gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"MIME Type \"%s\" no se puede redactar, porque era ni\n\
creado internamente ni cargado de una configuracin global."
#elif defined(PROG_LANGUAGE_FRENCH)
"Le Type MIME \"%s\" ne peut pas tre dit, parce qu'il\n\
ou a t soit cr intrieurement soit charg par la configuration\n\
globale."
#elif defined(PROG_LANGUAGE_GERMAN)
"MIME Type \"%s\" kann nicht redigiert werden, weil es\n\
war, entweder geschaffen inner oder kann von einer globalen\n\
konfiguration beladen werden."
#elif defined(PROG_LANGUAGE_ITALIAN)
"MIME Type \"%s\" non pu essere redatto, perch era sia\n\
creato internamente o caricato da una configurazione globale."
#elif defined(PROG_LANGUAGE_DUTCH)
"MIME Type \"%s\" kun niet is geredigeerd, omdat het ofwel\n\
gecreerde inwendig ofwel gelaade van een globale\n\
configuratie was."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"\"%s\" de MIME Type no pode ser editado, porque ele\n\
qualquer um foi criado internamente nem foi carregado de uma\n\
configurao global."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"MIME Type \"%s\" redigert ikke, fordi det var hver\n\
skape innvendig eller lastet fra en global konfigurasjon."
#else
"MIME Type \"%s\" cannot be edited, because it\n\
was either created internally or loaded from a global\n\
configuration."
#endif
		    , mt->type
		);
		EDVPlaySoundWarning(core);
		EDVMessageWarning(
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique Fallado"
#elif defined(PROG_LANGUAGE_FRENCH)
"appliquer Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden Sie Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Pas Geverzuimenene Toe"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique Fracassado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld Sviktet"
#else
"Apply Failed"
#endif
		    , msg, NULL,
		    toplevel
		);
		g_free(msg);
	    }
	    return;
	}


	/* Begin applying the values */

	/* Class */
	mt->mt_class = EDVMimeTypeEditDlgGetClass(d);

	/* Type */
	entry = GTK_ENTRY(d->type_entry);
	if(entry != NULL)
	{
	    const gchar *s = gtk_entry_get_text(entry);
	    if(s != NULL)
	    {
		g_free(mt->type);
		mt->type = STRDUP(s);
	    }
	}

	/* Value */
	entry = GTK_ENTRY(d->value_entry);
	if(entry != NULL)
	{
	    const gchar *s = gtk_entry_get_text(entry);
	    if(s != NULL)
	    {
		g_free(mt->value);
		mt->value = STRDUP(s);
	    }
	}

	/* Description */
	entry = GTK_ENTRY(d->description_entry);
	if(entry != NULL)
	{
	    const gchar *s = gtk_entry_get_text(entry);
	    if(s != NULL)
	    {
		g_free(mt->description);
		mt->description = STRDUP(s);
	    }
	}

	/* Handler */
	mt->handler = EDVMimeTypeEditDlgGetHandler(d);

	/* Commands */
	clist = GTK_CLIST(d->commands_clist);
	if(clist != NULL)
	{
	    gint i, n;
	    edv_mime_type_command_struct *cmd;

	    /* Delete the old commands */
	    if(mt->commands_list != NULL)
	    {
		g_list_foreach(
		    mt->commands_list, (GFunc)EDVMimeTypeCommandDelete, NULL
		);
		g_list_free(mt->commands_list);
		mt->commands_list = NULL;
	    }

	    /* Adjust the total commands on MIME Type structure and
	     * reallocate the command and command_name pointer arrays
	     * to the new size
	     */
	    n = clist->rows;
	    for(i = 0; i < n; i++)
	    {
		cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(
		    clist, i
		));
		if(cmd == NULL)
		    continue;

		mt->commands_list = g_list_append(
		    mt->commands_list,
		    EDVMimeTypeCommandCopy(cmd)
		);
	    }
	}

	/* Begin applying the icon file paths */

	/* Small icon */
	ei = d->icon_small;
	if(ei != NULL)
	{
	    gint i;
	    const gchar *s;

	    for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	    {
		s = ei->path[i];

		g_free(mt->small_icon_file[i]);
		mt->small_icon_file[i] = STRDUP(s);
	    }
	}

	/* Medium icon */
	ei = d->icon_medium;
	if(ei != NULL)
	{
	    gint i;
	    const gchar *s;

	    for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	    {
		s = ei->path[i];

		g_free(mt->medium_icon_file[i]);
		mt->medium_icon_file[i] = STRDUP(s);
	    }
	}

	/* Large icon */
	ei = d->icon_large;
	if(ei != NULL)
	{
	    gint i;
	    const gchar *s;

	    for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	    {
		s = ei->path[i];

		g_free(mt->large_icon_file[i]);
		mt->large_icon_file[i] = STRDUP(s);
	    }
	}

	/* Update time stamps */
	mt->access_time =
	mt->modify_time =
	mt->change_time = time(NULL);

	/* Check all of the MIME Type's values to see if they are any
	 * problems
	 */
	EDVMimeTypeEditDlgCheckMIMEType(
	    core,
	    core->mimetype, core->total_mimetypes,
	    mt_num,
	    TRUE,			/* Verbose */
	    toplevel
	);


	/* Reset the has changes marker */
	EDVMimeTypeEditDlgResetHasChanges(d, FALSE);


	/* Re-realize the MIME Type */
	EDVMimeTypeRealize(mt, TRUE);

	/* Send MIME Type modified signal to all of endeavour's resources */
	EDVMimeTypeModifiedEmit(core, mt_num, mt);
}


/*
 *	Add a new command.
 */
static void EDVMimeTypeEditDlgCommandAdd(
	edv_mime_type_edit_dlg_struct *d
)
{
	gint row, new_row;
	GList *glist;
	GtkCList *clist;
	cfg_item_struct *cfg_list;
	edv_mime_type_command_struct *cmd;
	edv_core_struct *core;

	if(d == NULL)
	    return;

	core = d->core;
	cfg_list = core->cfg_list;
	clist = GTK_CLIST(d->commands_clist);
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

	cmd = EDVMimeTypeCommandNew();
	if(cmd == NULL)
	    return;

	/* Set the default name */
	g_free(cmd->name);
	if((row == 0) || (clist->rows == 0))
	    cmd->name = STRDUP("Default");
	else if(row > -1)
	    cmd->name = g_strdup_printf(
		"Command #%i",
		row + 1
	    );
	else
	    cmd->name = g_strdup_printf(
		"Command #%i",
		clist->rows + 1
	    );

	/* Set the default command */
	g_free(cmd->command);
	cmd->command = STRDUP(
	    EDV_GET_S(EDV_CFG_PARM_PROG_DEF_VIEWER)
	);

	new_row = EDVMimeTypeEditDlgCommandsListInsert(
	    d, row, cmd
	);
	cmd = NULL;
	if(new_row < 0)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_select_row(clist, new_row, 0);
	gtk_clist_thaw(clist);

	EDVMimeTypeEditDlgAnyChangedCB(d->command_add_btn, d);
	EDVMimeTypeEditDlgUpdateMenus(d);

	/* Edit the new item */
	if(!EDVMimeTypeEditDlgCommandEdit(d, TRUE))
	{
	    EDVMimeTypeEditDlgCommandRemove(d, FALSE);
	}
}

/*
 *	Creates a new command dialog.
 */
static CmdDlg *CmdDlgNew(
	edv_core_struct *core,
	GtkWidget *ref_toplevel,
	edv_mime_type_command_struct *cmd,
	const gboolean is_new
)
{
	const gint      border_major = 5,
			border_minor = 2;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4, *parent5,
			*toplevel;
	GtkCList *clist;
	CmdDlg *d = CMD_DLG(g_malloc0(sizeof(CmdDlg)));
	if(d == NULL)
	    return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->freeze_count = 0;
	d->has_changes = FALSE;
	d->got_response = FALSE;
	d->core = core;

	d->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_widget_set_usize(w, 500, -1);
	gtk_window_set_title(GTK_WINDOW(w), "Command");
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	}
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(CmdDlgDeleteEventCB), d
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
/*	gtk_window_set_modal(GTK_WINDOW(w), TRUE); */
	gtk_window_set_transient_for(
	    GTK_WINDOW(w), GTK_WINDOW(ref_toplevel)
	);
	parent = w;

	/* Main GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Name */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("Name:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	d->name_entry = w = gtk_entry_new();
	gtk_widget_set_usize(w, 200, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_entry_set_text(
	    GTK_ENTRY(w),
	    (cmd->name != NULL) ? cmd->name : ""
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(CmdDlgChangedCB), d
	);
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
"Enter the name of this command"
	);
	gtk_widget_grab_focus(w);
	gtk_widget_show(w);

	/* Command */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("Command:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	d->command_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_entry_set_text(
	    GTK_ENTRY(w),
	    (cmd->command != NULL) ? cmd->command : ""
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(CmdDlgChangedCB), d
	);
	EDVEntrySetDND(core, w);
	EDVEntrySetCompletePath(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
 "Enter the shell command or a MIME Type reference for this\
 command. All shell commands start with a '/' character and MIME Type\
 references do not. MIME Type references should be the exact name\
 of the referenced MIME Type."
	);
	gtk_widget_show(w);

	w = GUIButtonPixmap(
	    (guint8 **)icon_browse_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgBrowseCB), d
	);
	GUISetWidgetTip(w, "Browse");
	gtk_widget_show(w);

	d->tokens_btn = w = GUIButtonPixmapLabelH(
	    NULL, "Tokens >>>", NULL
	);
	gtk_widget_set_usize(
	    w,
	    -1, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgTokensCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_t, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_t);
	GUISetWidgetTip(
	    w,
"Click this to display the list of tokens that you can use"
	);
	gtk_widget_show(w);

	/* Tokens */
	d->tokens_box = w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	parent3 = w;

	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	d->tokens_clist = w = gtk_clist_new(2);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(  
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(EDVCListMotionEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(CmdDlgCListSelectRowCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(CmdDlgCListUnselectRowCB), d
	);
	gtk_widget_set_usize(w, -1, 150);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, FALSE);
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 1, TRUE);
	gtk_clist_set_column_resizeable(clist, 1, FALSE);
	gtk_clist_set_column_justification(
	    clist, 1, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	if(w != NULL)
	{
	    const GtkTargetEntry dnd_src_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
	    };
	    GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY,		/* Actions */
		GDK_BUTTON1_MASK,		/* Buttons */
		NULL,
		CmdDlgCListDragDataGet,
		NULL,
		NULL,
		d
	    );
	}
	gtk_widget_show(w);

#define ADD_LINE(_s1_,_s2_)	{		\
 gint row;					\
 gchar *strv[2];				\
 strv[0] = (_s1_);				\
 strv[1] = (_s2_);				\
 row = gtk_clist_append(clist, strv);		\
 if(row > -1) {					\
  gtk_clist_set_row_data_full(			\
   clist, row,					\
   STRDUP(_s1_), (GtkDestroyNotify)g_free	\
  );						\
 }						\
}

	gtk_clist_freeze(clist);
	ADD_LINE(
"%cwd", "Current working directory."
	);
	ADD_LINE(
"%display", "Calling process' display address."
	);
	ADD_LINE(
"%first_name", "Name of the first selected object (without the full path)."
	);
	ADD_LINE(
"%first_name_wo_ext", "Name of the first selected object (without the full path or extension)."
	);
	ADD_LINE(
"%first_path", "Full path to the first selected object."
	);
	ADD_LINE(
"%first_path_wo_ext", "Full path to the first selected object without its extension."
	);
	ADD_LINE(
"%home", "User's home directory."
	);
	ADD_LINE(
"%name", "Name of the last selected object (without the full path)."
	);
	ADD_LINE(
"%name_wo_ext", "Name of the last selected object (without the full path or extension)."
	);
	ADD_LINE(
"%names", "Name(s) of all the selected object(s) (without the full path)."
	);
	ADD_LINE(
"%path", "Full path to the last selected object."
	);
	ADD_LINE(
"%path_wo_ext", "Full path to the last selected object without its extension."
	);
	ADD_LINE(
"%paths", "Full path(s) to all the selected object(s)."
	);
	ADD_LINE(
"%pid", "Calling process' process ID."
	);
	ADD_LINE(
"%pe", "Same as %path_wo_ext."
	);
	ADD_LINE(
"%p", "Same as %path."
	);
	ADD_LINE(
"%s", "Same as %path."
	);
#undef ADD_LINE
	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_label_new(
"Remember to put double quotes (\") around tokens so that the\n\
substituted values which have spaces don't get mistaken for
multiple arguments (ie \"%paths\")."
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Insert Token Button */
	w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_end(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	d->tokens_insert_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_add_20x20_xpm, "Insert", NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgInsertTokenCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_i, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_i);
	GUISetWidgetTip(
	    w,
"Click this to insert the selected token into the command"
	);
	gtk_widget_set_sensitive(w, FALSE);
	gtk_widget_show(w);


	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* OK button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "Ok", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgOKCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);
	gtk_widget_set_sensitive(w, is_new);
	gtk_widget_show(w);

	/* Cancel button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm, "Cancel", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	if(is_new)
	    gtk_widget_show(w);

	/* Close button */
	d->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm, "Close", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgCloseCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	if(!is_new)
	    gtk_widget_show(w);

	d->freeze_count--;

	gtk_widget_show_raise(d->toplevel);

	return(d);
}

/*
 *	Creates a new command dialog.
 */
static void CmdDlgDelete(CmdDlg *d)
{
	GtkCList *clist;

	if(d == NULL)
	    return;

	gtk_widget_hide(d->toplevel);
/*	gtk_window_set_modal(GTK_WINDOW(d->toplevel), FALSE); */
	gtk_window_set_transient_for(GTK_WINDOW(d->toplevel), NULL);

	d->freeze_count++;

	clist = GTK_CLIST(d->tokens_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);

	gtk_widget_destroy(d->toplevel);

	d->freeze_count--;

	g_free(d);
}

/*
 *	Edit the selected command.
 */
static gboolean EDVMimeTypeEditDlgCommandEdit(
	edv_mime_type_edit_dlg_struct *d,
	const gboolean is_new
)
{
	gboolean status, sensitive;
	gint row, cmd_num;
	GList *glist;
	GtkWidget *toplevel;
	GtkCList *clist;
	edv_mime_type_command_struct *cmd;
	edv_core_struct *core;
	CmdDlg *cmddlg;

	if(d == NULL)
	    return(FALSE);

	toplevel = d->toplevel;
	core = d->core;
	clist = GTK_CLIST(d->commands_clist);
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;
	if(row < 0)
	    return(FALSE);

	cmd_num = row;
	cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, row));
	if(cmd == NULL)
	    return(FALSE);

	/* Create a new command dialog and query the user */
	cmddlg = CmdDlgNew(core, toplevel, cmd, is_new);
	if(cmddlg == NULL)
	    return(FALSE);

	d->freeze_count++;

	sensitive = FALSE;
	gtk_widget_set_sensitive(d->command_add_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_add_mi, sensitive);
	gtk_widget_set_sensitive(d->command_edit_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_edit_mi, sensitive);
	gtk_widget_set_sensitive(d->command_remove_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_remove_mi, sensitive);
	gtk_widget_set_sensitive(d->command_shift_up_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_shift_up_mi, sensitive);
	gtk_widget_set_sensitive(d->command_shift_down_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_shift_down_mi, sensitive);

	/* Wait for user response */
	gtk_main();

	d->freeze_count--;

	if(cmddlg->got_response)
	{
	    /* Create a new command with the new values */
	    edv_mime_type_command_struct *cmd = EDVMimeTypeCommandNew();
	    if(cmd != NULL)
	    {
		cmd->name = STRDUP(gtk_entry_get_text(
		    GTK_ENTRY(cmddlg->name_entry)
		));
		cmd->command = STRDUP(gtk_entry_get_text(
		    GTK_ENTRY(cmddlg->command_entry)
		));
		EDVMimeTypeEditDlgCommandsListSet(
		    d, cmd_num, cmd
		);
		cmd = NULL;
	    }

	    EDVMimeTypeEditDlgAnyChangedCB(d->command_edit_btn, d);

	    status = TRUE;
	} 
	else
	{
	    status = FALSE;
	}

	/* Delete the command dialog */
	CmdDlgDelete(cmddlg);

	EDVMimeTypeEditDlgUpdateMenus(d);

	return(status);
}

/*
 *	Remove the selected handler(s).
 */
static void EDVMimeTypeEditDlgCommandRemove(
	edv_mime_type_edit_dlg_struct *d, const gboolean confirm
)
{
	gint row;
	GList *glist, *cmds_list;
	GtkWidget *toplevel;
	GtkCList *clist;
	edv_mime_type_command_struct *cmd;

	if(d == NULL)
	    return;

	clist = GTK_CLIST(d->commands_clist);
	toplevel = gtk_widget_get_toplevel(d->toplevel);

	/* Create a list of selected commands */
	cmds_list = NULL;
	glist = clist->selection;
	while(glist != NULL)
	{
	    row = (gint)glist->data;
	    cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, row));
	    if(cmd != NULL)
		cmds_list = g_list_append(cmds_list, cmd);
	    glist = g_list_next(glist);
	}

	/* Nothing to remove? */
	if(cmds_list == NULL)
	    return;

	/* Confirm remove? */
	if(confirm)
	{
	    const gint ncmds = g_list_length(cmds_list);
	    gint response;
	    gchar *msg;
	    if(ncmds == 1)
	    {
		cmd = EDV_MIME_TYPE_COMMAND(cmds_list->data);
		msg = g_strdup_printf(
"Remove command \"%s\"?",
		    (cmd != NULL) ? cmd->name : "(null)"
		);
	    }     
	    else
	    {
		msg = g_strdup_printf(
"Remove %i selected commands?",
		    ncmds
		);
	    }
	    CDialogSetTransientFor(toplevel);
	    response = CDialogGetResponse(
		"Confirm Remove",
		msg,
		NULL,
		CDIALOG_ICON_QUESTION, 
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	    );
	    g_free(msg);
	    CDialogSetTransientFor(NULL);
		   
	    if(response != CDIALOG_RESPONSE_YES)
	    {
		g_list_free(cmds_list);
		return;
	    }
	}    
	 
	/* Begin removing the selected commands */
	gtk_clist_freeze(clist);
	for(glist = cmds_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{          
	    cmd = EDV_MIME_TYPE_COMMAND(glist->data);
	    row = gtk_clist_find_row_from_data(clist, cmd);
	    if(row < 0)
		continue;

	    EDVMimeTypeEditDlgAnyChangedCB(d->command_remove_btn, d);

	    gtk_clist_remove(clist, row);
	}
	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);

	g_list_free(cmds_list);

	EDVMimeTypeEditDlgUpdateMenus(d);
}

/*
 *	Shift the selected handler(s) up.
 */
static void EDVMimeTypeEditDlgCommandShiftUp(
	edv_mime_type_edit_dlg_struct *d
)
{
	gint i, n, row;
	GList *glist, *cmds_list;
	GtkCList *clist;
	edv_mime_type_command_struct *cmd;

	if(d == NULL)
	    return;

	clist = GTK_CLIST(d->commands_clist);

	/* Create a list of selected commands */
	cmds_list = NULL;
	glist = clist->selection;
	while(glist != NULL)
	{
	    row = (gint)glist->data;
	    cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, row));
	    if(cmd != NULL)
		cmds_list = g_list_append(cmds_list, cmd);
	    glist = g_list_next(glist);
	}

	/* Nothing to shift? */
	if(cmds_list == NULL)
	    return;

	n = clist->rows;

	/* First row selected? */
	i = 0;
	cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, i));
	if(g_list_find(cmds_list, cmd) != NULL)
	{
	    /* The first row may not be shifted up any further */
	    g_list_free(cmds_list);
	    return;
	}

	/* Begin shifting the selected rows up */
	gtk_clist_freeze(clist);
	for(i = 1; i < n; i++)
	{
	    cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, i));
	    if(g_list_find(cmds_list, cmd) != NULL)
	    {
		gtk_clist_swap_rows(clist, i, i - 1);
		EDVMimeTypeEditDlgAnyChangedCB(d->command_shift_up_btn, d);
	    }                    
	}
	gtk_clist_thaw(clist);

	g_list_free(cmds_list);

	EDVMimeTypeEditDlgUpdateMenus(d);
}

/*
 *	Shift the selected handler(s) down.
 */
static void EDVMimeTypeEditDlgCommandShiftDown(
	edv_mime_type_edit_dlg_struct *d
)
{
	gint i, n, row;
	GList *glist, *cmds_list;
	GtkCList *clist;
	edv_mime_type_command_struct *cmd;

	if(d == NULL)
	    return;

	clist = GTK_CLIST(d->commands_clist);

	/* Create a list of selected commands */
	cmds_list = NULL;
	glist = clist->selection;
	while(glist != NULL)
	{
	    row = (gint)glist->data;
	    cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, row));
	    if(cmd != NULL)
		cmds_list = g_list_append(cmds_list, cmd);
	    glist = g_list_next(glist);
	}

	/* Nothing to shift? */
	if(cmds_list == NULL)
	    return;

	n = clist->rows;

	/* Last row selected? */
	i = n - 1;
	cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, i));
	if(g_list_find(cmds_list, cmd) != NULL)
	{
	    /* The last row may not be shifted down any further */
	    g_list_free(cmds_list);
	    return;
	}

	/* Begin shifting the selected rows down */
	gtk_clist_freeze(clist);
	for(i = n - 2; i >= 0; i--)
	{
	    cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, i));
	    if(g_list_find(cmds_list, cmd) != NULL)
	    {
		gtk_clist_swap_rows(clist, i, i + 1);
		EDVMimeTypeEditDlgAnyChangedCB(d->command_shift_down_btn, d);
	    }
	}
	gtk_clist_thaw(clist);

	g_list_free(cmds_list);

	EDVMimeTypeEditDlgUpdateMenus(d);
}


/*
 *	Inserts a new handler item to the list.
 *
 *	The cmd_num specifies the insert index.
 *
 *	The cmd specifies the command that will be transfered and
 *	should not be referenced again after this call.
 *
 *	Returns the new command index or -1 on error.
 */
static gint EDVMimeTypeEditDlgCommandsListInsert(
	edv_mime_type_edit_dlg_struct *d,
	const gint cmd_num,
	edv_mime_type_command_struct *cmd
)
{
	gint i, row, new_row, ncolumns;
	gchar **strv;
	GtkCList *clist;

	if(d == NULL)
	{
	    EDVMimeTypeCommandDelete(cmd);
	    return(-1);
	}

	clist = GTK_CLIST(d->commands_clist);
	ncolumns = MAX(clist->columns, 1);
	row = cmd_num;

	gtk_clist_freeze(clist);

	/* Allocate row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	/* Insert or append row */
	if(row > -1)
	    new_row = gtk_clist_insert(clist, row, strv);
	else
	    new_row = gtk_clist_append(clist, strv);

	/* Delete row cell values */
	g_free(strv);

	/* Failed to create new row? */
	if(new_row < 0)
	{
	    gtk_clist_thaw(clist);
	    EDVMimeTypeCommandDelete(cmd);
	    return(-1);
	}              

	/* Set the row values */
	EDVMimeTypeEditDlgCommandsListSet(d, new_row, cmd);

	gtk_clist_thaw(clist);

	return(new_row);
}

/*
 *	Appends a new handler item to the list.
 *
 *	The cmd specifies the command that will be transfered and
 *	should not be referenced again after this call.
 *
 *	Returns the new command index or -1 on error.
 */
static gint EDVMimeTypeEditDlgCommandsListAppend(
	edv_mime_type_edit_dlg_struct *d,
	edv_mime_type_command_struct *cmd
)
{
	return(EDVMimeTypeEditDlgCommandsListInsert(
	    d, -1, cmd
	));
}

/*
 *	Sets the handler item's values.
 *
 *	The cmd_num specifies the handler item.
 *
 *	The cmd specifies the command that will be transfered and
 *	should not be referenced again after this call, it also
 *	specifies the item's values.
 */
static void EDVMimeTypeEditDlgCommandsListSet(
	edv_mime_type_edit_dlg_struct *d,
	const gint cmd_num,
	edv_mime_type_command_struct *cmd
)
{
	gint row, nrows, column, ncolumns;
	GtkCList *clist;

	if(d == NULL)
	{
	    EDVMimeTypeCommandDelete(cmd);
	    return;
	}

	clist = GTK_CLIST(d->commands_clist);
	ncolumns = clist->columns;
	nrows = clist->rows;

	row = cmd_num;
	if((row < 0) || (row >= nrows))
	{
	    EDVMimeTypeCommandDelete(cmd);
	    return;
	}

	gtk_clist_freeze(clist);

	if(cmd != NULL)
	{
	    column = 0;
	    if(column < ncolumns)
		gtk_clist_set_text(
		    clist, row, column,
		    (cmd->name != NULL) ? cmd->name : ""
		);
	    column = 1;
	    if(column < ncolumns)
		gtk_clist_set_text(
		    clist, row, column,
		    (cmd->command != NULL) ? cmd->command : ""
		);
	}
	else
	{
	    column = 0;
	    if(column < ncolumns)
		gtk_clist_set_text(
		    clist, row, column, ""
		);
	    column = 1;
	    if(column < ncolumns)
		gtk_clist_set_text(
		    clist, row, column, ""
		);
	}

	/* Set this row's data as the command */
	gtk_clist_set_row_data_full(
	    clist, row,
	    cmd, EDVMimeTypeEditDlgCommandDestroyCB
	);

	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);
}


/*
 *	Creates a new Icon Picker.
 */
static edv_mime_type_icon_picker_struct *EDVMimeTypeEditDlgIconNew(
	edv_mime_type_edit_dlg_struct *d,
	GtkWidget *parent, const gchar *title,
	const gint icon_width, const gint icon_height,
	const gint btn_width, const gint btn_height
)
{
	gint bw = 20, bh = 20;
	GtkAdjustment *adj;
	GtkWidget *w, *parent2, *parent3, *parent4;
	edv_mime_type_icon_picker_struct *ei;

	if(d == NULL)
	    return(NULL);

	ei = EDV_MIME_TYPE_ICON_PICKER(
	    g_malloc0(sizeof(edv_mime_type_icon_picker_struct))
	);
	if(ei == NULL)
	    return(NULL);

	ei->d = d;
	ei->icon_width = icon_width;
	ei->icon_height = icon_height;

	/* Create adjustment for keeping track of displayed icon */
	ei->adj = adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,		/* Value */
	    0.0f,		/* Lower */
	    (gfloat)EDV_MIME_TYPE_TOTAL_ICON_STATES - 1.0f,	/* Upper */
	    1.0f, 1.0f,		/* Step and page increments */
	    1.0f		/* Page size */
	);

	/* Toplevel GtkVBox */
	ei->toplevel = w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Title GtkHBox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Label */
	w = gtk_label_new(title);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Button GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Create button */
	ei->button = w = gtk_button_new();
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgIconChangeCB), ei
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
#if 0
	ei->gtk_pixmap = w = gtk_pixmap_new(NULL, NULL);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_show(w);
#endif


	/* Hbox for buttons and labels */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Left button */
	ei->left_btn = w = gtk_button_new();
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgIconPrevCB), ei
	);
	gtk_widget_set_usize(w, bw, bh);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Arrow */
	w = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);

	/* Frame for state label */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_widget_set_usize(w, 80, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* State label */
	ei->state_label = w = gtk_label_new("");
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);

	/* Right button */
	ei->right_btn = w = gtk_button_new();
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgIconNextCB), ei
	);
	gtk_widget_set_usize(w, bw, bh);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Arrow */
	w = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);


	return(ei);
}

/*
 *	Deletes the Icon Picker.
 */
static void EDVMimeTypeEditDlgIconDelete(edv_mime_type_icon_picker_struct *ei)
{
	gint i;

	if(ei == NULL)
	    return;

	/* Unref all loaded pixmap and mask pairs */
	for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	{
	    GDK_PIXMAP_UNREF(ei->pixmap[i])
	    GDK_BITMAP_UNREF(ei->mask[i])
	    g_free(ei->path[i]);
	}

	/* Begin destroying widgets */
	GTK_WIDGET_DESTROY(ei->gtk_pixmap)
	GTK_WIDGET_DESTROY(ei->button)
	GTK_WIDGET_DESTROY(ei->left_btn)
	GTK_WIDGET_DESTROY(ei->right_btn)
	GTK_WIDGET_DESTROY(ei->state_label)
	GTK_WIDGET_DESTROY(ei->toplevel)

	GTK_OBJECT_UNREF(ei->adj)

	g_free(ei);
}


/*
 *	Creates a new MIME Types edit window.
 */
edv_mime_type_edit_dlg_struct *EDVMimeTypeEditDlgNew(
	edv_core_struct *core, edv_mime_types_list_win_struct *lw
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gchar *heading[2];
	gpointer entry_rtn, label_rtn;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4,
			*parent5, *parent6, *parent7,
			*toplevel;
	GtkCList *clist;
	pulist_struct *pulist;
	pulistbox_struct *pulistbox;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(
	    g_malloc0(sizeof(edv_mime_type_edit_dlg_struct))
	);
	if((d == NULL) || (core == NULL))
	{
	    g_free(d);
	    return(NULL);
	}

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->freeze_count = 0;
	d->busy_count = 0;
	d->has_changes = FALSE;
	d->title = STRDUP(EDITWIN_TITLE);
	d->core = core;
	d->list_win = lw;
	d->mt_num = -1;
	d->browse_path_rtn = NULL;

	d->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_widget_set_usize(w, EDITWIN_WIDTH, EDITWIN_HEIGHT);
	gtk_window_set_title(GTK_WINDOW(w), d->title);
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_mimetypes_48x48_xpm);
	}
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgDeleteEventCB), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;


	/* Main GtkVBox */
	d->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* GtkHBox to make two columns */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left column GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Class GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"La Clase"
#elif defined(PROG_LANGUAGE_FRENCH)
"Classe"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klasse"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Classe"
#elif defined(PROG_LANGUAGE_DUTCH)
"Klas"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Classe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klasse"
#else
"Class"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Class popup list box */
	d->class_pulistbox = pulistbox = PUListBoxNew(
	    parent4, -1, -1
	);
	pulist = PUListBoxGetPUList(pulistbox);
        if(core->run_flags & EDV_RUN_SAFE_MODE)
            PUListSetShadowStyle(pulist, PULIST_SHADOW_NONE);
	PUListBoxSetChangedCB(
	    pulistbox,
	    EDVMimeTypeEditDlgClassChangedCB, d
	);
	if(pulist != NULL)
	{
	    gint i;

	    i = PUListAddItem(
		pulist,
		"System Object Type"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_CLASS_SYSTEM);
	    i = PUListAddItem(
		pulist,
		"File Format"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_CLASS_FORMAT);
	    i = PUListAddItem(
		pulist,
		"Application"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_CLASS_PROGRAM);
	    i = PUListAddItem(
		pulist,
		"Specific Object"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_CLASS_UNIQUE);

	    PUListBoxSetLinesVisible(pulistbox, i + 1);
	}
	PUListBoxMap(pulistbox);


	/* Identification GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Identificacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Identification"
#elif defined(PROG_LANGUAGE_GERMAN)
"Identifikation"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Identificazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Identificatie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Identificao"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Identification"
#else
"Identification"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Type GtkEntry */
	w = GUIPromptBar(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"El Tipo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Type"
#elif defined(PROG_LANGUAGE_GERMAN)
"Typ"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Tipo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Type"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Tipo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Type"
#else
"Type"
#endif
	    ":", NULL, &entry_rtn
	);
	d->type_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgAnyChangedCB), d
	);
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);

	/* Value GtkEntry */
	w = GUIPromptBar(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"El Valor"
#elif defined(PROG_LANGUAGE_FRENCH)
"Valeur"
#elif defined(PROG_LANGUAGE_GERMAN)
"Wert"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Valore"
#elif defined(PROG_LANGUAGE_DUTCH)
"Waarde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Valor"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Verdi"
#else
"Value"
#endif
	    ":", &label_rtn, &entry_rtn
	);
	d->value_label = (GtkWidget *)label_rtn;
	d->value_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgAnyChangedCB), d
	);
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);


	/* Description GtkEntry */
	w = GUIPromptBar(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"La Descripcin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Description"
#elif defined(PROG_LANGUAGE_GERMAN)
"Beschreibung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Descrizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beschrijving"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Descrio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Beskrivelse"
#else
"Description"
#endif
	    ":", NULL, &entry_rtn
	);
	d->description_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgAnyChangedCB), d
	);
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);


	/* Handler GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Tratante"
#elif defined(PROG_LANGUAGE_FRENCH)
"Agent"
#elif defined(PROG_LANGUAGE_GERMAN)
"Behandler"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Addestratore"
#elif defined(PROG_LANGUAGE_DUTCH)
"Handler"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Manipulador"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Behandler"
#else
"Handler"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;


	/* Handle By GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Label */
	w = gtk_label_new("Handle By:");
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Popup list box */
	d->handler_pulistbox = pulistbox = PUListBoxNew(
	    parent5, -1, -1
	);
	pulist = PUListBoxGetPUList(pulistbox);
        if(core->run_flags & EDV_RUN_SAFE_MODE)
            PUListSetShadowStyle(pulist, PULIST_SHADOW_NONE);
	PUListBoxSetChangedCB(
	    pulistbox,
	    EDVMimeTypeEditDlgHandlerChangedCB, d
	);
	if(pulist != NULL)
	{
	    gint i;

	    i = PUListAddItem(
		pulist,
		"Command"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_HANDLER_COMMAND);
	    i = PUListAddItem(
		pulist,
		"Endeavour Archiver"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_HANDLER_EDV_ARCHIVER);
	    i = PUListAddItem(
		pulist,
		"Endeavour Image Browser"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_HANDLER_EDV_IMAGE_BROWSER);
	    i = PUListAddItem(
		pulist,
		"Endeavour Recycle Bin"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_HANDLER_EDV_RECYCLE_BIN);

	    PUListBoxSetLinesVisible(pulistbox, i + 1);
	}
	PUListBoxMap(pulistbox);

	/* Handler GtkHBox */
	d->commands_clist_box = w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent5), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent6 = w;

	/* Create the GtkScrolledWindow for the GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent7 = w;

	/* Commands GtkCList */
#if defined(PROG_LANGUAGE_SPANISH)
heading[0] = "El Nombre";
heading[1] = "La Orden";
#elif defined(PROG_LANGUAGE_FRENCH)
heading[0] = "Nom";
heading[1] = "Commande";
#elif defined(PROG_LANGUAGE_GERMAN)
heading[0] = "Name";
heading[1] = "Befehl";
#elif defined(PROG_LANGUAGE_ITALIAN)
heading[0] = "Il Nome";
heading[1] = "Il Comando";
#elif defined(PROG_LANGUAGE_DUTCH)
heading[0] = "Naam";
heading[1] = "Bevel";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
heading[0] = "O Nome";
heading[1] = "O Comando";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
heading[0] = "Navn";
heading[1] = "Kommando";
#else
heading[0] = "Name";
heading[1] = "Command";
#endif
	d->commands_clist = w = gtk_clist_new_with_titles(
	    sizeof(heading) / sizeof(gchar *),
	    heading
	);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(  
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(EDVCListMotionEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandsCListEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandsCListEventCB), d
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandsCListEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandsCListSelectRowCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandsCListUnselectRowCB), d
	);
	gtk_widget_set_usize(w, -1, 140);
	gtk_container_add(GTK_CONTAINER(parent7), w);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, TRUE);
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 1, TRUE);
	gtk_clist_set_column_resizeable(clist, 1, TRUE);
	gtk_clist_set_column_justification(
	    clist, 1, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	if(w != NULL)
	{
	    const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING},
{EDV_DND_TARGET_MIME_TYPE,	0,	EDV_DND_INFO_MIME_TYPE},
{EDV_DND_TARGET_MIME_TYPE_COMMAND, 0,	EDV_DND_INFO_MIME_TYPE_COMMAND}
	    };
	    const GtkTargetEntry dnd_src_types[] = {
{EDV_DND_TARGET_MIME_TYPE_COMMAND, 0,	EDV_DND_INFO_MIME_TYPE_COMMAND}
	    };
	    GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE,	/* Actions */
		GDK_BUTTON1_MASK,			/* Buttons */
		NULL,
		EDVMimeTypeEditDlgCommandsCListDragDataGetCB,
		EDVMimeTypeEditDlgCommandsCListDragDataDeleteCB,
		NULL,
		d
	    );
	    GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE,	/* Actions */
		GDK_ACTION_MOVE,		/* Default action if same */
		GDK_ACTION_COPY,		/* Default action */
		EDVMimeTypeEditDlgCommandsCListDragDataReceivedCB,
		d
	    );
	    gtk_signal_connect_after(  
		GTK_OBJECT(w), "drag_motion",
		GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandsCListDragMotionCB), d
	    );
	}
	gtk_widget_show(w);

	/* Command buttons GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent7 = w;

	/* Add button */
	d->command_add_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_add_20x20_xpm,
	    "Add",
	    NULL
	);
	gtk_widget_set_usize(
	    w, GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandAddCB), d
	);
	GUISetWidgetTip(
	    w,
	    "Add a new command"
	);
	gtk_widget_show(w);

	/* Edit button */
	d->command_edit_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_edit_20x20_xpm,
	    "Edit",
	    NULL
	);
	gtk_widget_set_usize(
	    w, GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandEditCB), d
	);
	GUISetWidgetTip(
	    w,
	    "Edit the selected command"
	);
	gtk_widget_show(w);

	/* Remove button */
	d->command_remove_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_remove_20x20_xpm,
	    "Remove",
	    NULL
	);
	gtk_widget_set_usize(
	    w, GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandRemoveCB), d
	);
	GUISetWidgetTip(
	    w,
	    "Remove the selected command(s)"
	);
	gtk_widget_show(w);

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent7 = w;

	/* Shift up button */
	d->command_shift_up_btn = w = GUIButtonArrowLabelH(
	    GTK_ARROW_UP, 20, 20,
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Arriba"
#elif defined(PROG_LANGUAGE_FRENCH) 
"Changer Haut"
#elif defined(PROG_LANGUAGE_GERMAN)
"Spostare Su"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Spostare Su"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verplaats Op"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Cima"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forskyv Opp"
#else
"Shift Up"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w, 
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandShiftUpCB), d
	);
	GUISetWidgetTip(
	    w,
	    "Shift the selected command(s) up"
	);
	gtk_widget_show(w);

	/* Shift down button */
	d->command_shift_down_btn = w = GUIButtonArrowLabelH(
	    GTK_ARROW_DOWN, 20, 20,
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Abajo"
#elif defined(PROG_LANGUAGE_FRENCH) 
"Changer Bas"
#elif defined(PROG_LANGUAGE_GERMAN)
"Spostare Gi"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Spostare Gi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verplaats Beneden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Baixo"      
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forskyv Ned"
#else       
"Shift Down"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w, 
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandShiftDownCB), d
	);
	GUISetWidgetTip(
	    w,
	    "Shift the selected command(s) down"
	);
	gtk_widget_show(w);


	/* Create the commands menu */
#define DO_ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
  icon, label, accel_key, accel_mods, NULL,	\
  menu_d, func_cb				\
 );						\
}
#define DO_ADD_MENU_SEP		{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}
	d->commands_menu = w = GUIMenuCreate();
	if(w != NULL)
	{
	    GtkAccelGroup *accelgrp = NULL;
	    GtkWidget *menu = w;
	    guint8 **icon;
	    const gchar *label;
	    guint accel_key, accel_mods;
	    void (*func_cb)(GtkWidget *w, gpointer);
	    gpointer menu_d = d;

	    icon = (guint8 **)icon_add_20x20_xpm;
	    label = "Add...";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVMimeTypeEditDlgCommandAddCB;
	    DO_ADD_MENU_ITEM_LABEL
	    d->commands_menu_add_mi = w;

	    icon = (guint8 **)icon_edit_20x20_xpm;
	    label = "Edit...";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVMimeTypeEditDlgCommandEditCB;
	    DO_ADD_MENU_ITEM_LABEL
	    d->commands_menu_edit_mi = w;

	    icon = (guint8 **)icon_remove_20x20_xpm;
	    label = "Remove";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVMimeTypeEditDlgCommandRemoveCB;
	    DO_ADD_MENU_ITEM_LABEL
	    d->commands_menu_remove_mi = w;

	    DO_ADD_MENU_SEP

	    icon = NULL;
	    label = "Shift Up";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVMimeTypeEditDlgCommandShiftUpCB;
	    DO_ADD_MENU_ITEM_LABEL
	    d->commands_menu_shift_up_mi = w;

	    icon = NULL;
	    label = "Shift Down";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVMimeTypeEditDlgCommandShiftDownCB;
	    DO_ADD_MENU_ITEM_LABEL
	    d->commands_menu_shift_down_mi = w;
	}
#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SEP

	/* Right side vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Icons frame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Iconos"
#elif defined(PROG_LANGUAGE_FRENCH)
"Icne"
#elif defined(PROG_LANGUAGE_GERMAN)
"Abbilder"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Le Icone"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beelden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Os cones"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ikoner"
#else
"Icons"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;


	/* Create icon structures */
	d->icon_large = EDVMimeTypeEditDlgIconNew(
	    d, parent4,
#if defined(PROG_LANGUAGE_SPANISH)
"Grande"
#elif defined(PROG_LANGUAGE_FRENCH)
"Grand"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gro"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Grande"
#elif defined(PROG_LANGUAGE_DUTCH)
"Groot"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Grande"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stor"
#else
"Large"
#endif
	    " 48x48:",
	    48, 48,
	    54, 54
	);
	d->icon_medium = EDVMimeTypeEditDlgIconNew(
	    d, parent4,
#if defined(PROG_LANGUAGE_SPANISH)
"El Medio"
#elif defined(PROG_LANGUAGE_FRENCH)
"Moyen"
#elif defined(PROG_LANGUAGE_GERMAN)
"Medium"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Mezzo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Middel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mdio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Middel"
#else
"Medium"
#endif
	    " 32x32:",
	    32, 32,
	    38, 38
	);
	d->icon_small = EDVMimeTypeEditDlgIconNew(
	    d, parent4,
#if defined(PROG_LANGUAGE_SPANISH)
"Pequeo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Petit"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Piccolo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Klein"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Pequeno"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Liten"
#else
"Small"
#endif
	    " 20x20:",
	    20, 20,
	    26, 26
	);


	/* Horizontal GtkSeparator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* OK button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "OK", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgOKCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);

	/* Apply button */
	d->apply_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_select_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique"
#elif defined(PROG_LANGUAGE_FRENCH)
"Appliquer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toepas"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld"
#else
"Apply"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgApplyCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_a);

	/* Cancel button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cancele"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Heben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Annullare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Annuleer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cancelamento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kanseller"
#else
"Cancel"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);

	/* Close button */
	d->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Quitter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);


	EDVMimeTypeEditDlgUpdateMenus(d);

	d->freeze_count--;

	return(d);
}

/*
 *	Resets the MIME Types Edit Window's has changes marker.
 */
void EDVMimeTypeEditDlgResetHasChanges(
	edv_mime_type_edit_dlg_struct *d,
	const gboolean has_changes
)
{
	if(d == NULL)
	    return;

	if(d->has_changes == has_changes)
	    return;

	d->has_changes = has_changes;
	EDVMimeTypeEditDlgUpdateMenus(d);
}

/*
 *	Updates the MIE Types Edit Window's widgets to reflect current
 *	values.
 */
void EDVMimeTypeEditDlgUpdateMenus(edv_mime_type_edit_dlg_struct *d)
{
	gboolean	sensitive,
			has_changes,
			read_only;
	gint nrows, nselected, sel_row, mt_num;
	const edv_mime_type_struct *m;
	gchar *s;
	GList *glist;
	GtkCList *clist;
	edv_core_struct *core;

	if(d == NULL)
	    return;

	d->freeze_count++;

	core = d->core;

	has_changes = d->has_changes;
	mt_num = d->mt_num;
	if((mt_num >= 0) && (mt_num < core->total_mimetypes))
	    m = core->mimetype[mt_num];
	else
	    m = NULL;
	read_only = (m != NULL) ? m->read_only : FALSE;

	/* Title */
	s = g_strconcat(
	    (d->title != NULL) ? d->title : "",
	    (has_changes) ? " (*)" : "",
	    NULL
	);
	gtk_window_set_title(GTK_WINDOW(d->toplevel), s);
	g_free(s);

	/* Commands GtkCList */
	clist = GTK_CLIST(d->commands_clist);
	nrows = clist->rows;
	nselected = g_list_length(clist->selection);
	glist = clist->selection_end;
	sel_row = (glist != NULL) ? (gint)glist->data : -1;
	sensitive = (nselected <= 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->command_add_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_add_mi, sensitive);
	sensitive = (nselected == 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->command_edit_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_edit_mi, sensitive);
	sensitive = (nselected >= 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->command_remove_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_remove_mi, sensitive);
	sensitive = ((g_list_find(clist->selection, (gpointer)0) == NULL) &&
	    (nselected > 0)) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->command_shift_up_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_shift_up_mi, sensitive);
	sensitive = ((g_list_find(clist->selection, (gpointer)(nrows - 1)) == NULL) &&
	    (nselected > 0)) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->command_shift_down_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_shift_down_mi, sensitive);

	/* OK, Apply, Cancel, Close Buttons */
	sensitive = (has_changes && !read_only) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->ok_btn, sensitive);
	gtk_widget_set_sensitive(d->apply_btn, sensitive);
	sensitive = has_changes;
	gtk_widget_set_sensitive(d->cancel_btn, sensitive);
	sensitive = (has_changes && !read_only) ? FALSE : TRUE;
	gtk_widget_set_sensitive(d->close_btn, sensitive);
	if(has_changes)
	{
	    gtk_widget_show(d->ok_btn);
	    gtk_widget_show(d->apply_btn);
	    gtk_widget_show(d->cancel_btn);
	    gtk_widget_hide(d->close_btn);
	}
	else
	{
	    gtk_widget_show(d->ok_btn);
	    gtk_widget_show(d->apply_btn);
	    gtk_widget_hide(d->cancel_btn);
	    gtk_widget_show(d->close_btn);
	}

	d->freeze_count--;
}

/*
 *	Checks if the MIME Types Edit Window is mapped.
 */
gboolean EDVMimeTypeEditDlgIsMapped(
	edv_mime_type_edit_dlg_struct *d
)
{
	if(d == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(d->toplevel));
}

/*
 *	Maps the MIME Types Edit Window.
 */
void EDVMimeTypeEditDlgMap(
	edv_mime_type_edit_dlg_struct *d
)
{
	pulistbox_struct *pulistbox;

	if(d == NULL)
	    return;

	gtk_widget_show_raise(d->toplevel);

	pulistbox = d->class_pulistbox;
	if(pulistbox != NULL)
	    gtk_widget_grab_focus(pulistbox->da);
}

/*
 *	Unmaps the MIME Types Edit Window.
 */
void EDVMimeTypeEditDlgUnmap(
	edv_mime_type_edit_dlg_struct *d
)
{
	if(d == NULL)
	    return;

	gtk_widget_hide(d->toplevel);
}

/*
 *	Deletes the MIME Types Edit Window.
 */
void EDVMimeTypeEditDlgDelete(
	edv_mime_type_edit_dlg_struct *d
)
{
	GtkCList *clist;

	if(d == NULL)
	    return;

	EDVMimeTypeEditDlgUnmap(d);

	clist = GTK_CLIST(d->commands_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);

	d->freeze_count++;

	gtk_widget_destroy(d->commands_menu);

	EDVMimeTypeEditDlgIconDelete(d->icon_small);
	EDVMimeTypeEditDlgIconDelete(d->icon_medium);
	EDVMimeTypeEditDlgIconDelete(d->icon_large);

	PUListBoxDelete(d->class_pulistbox);
	PUListBoxDelete(d->handler_pulistbox);

	gtk_widget_destroy(d->toplevel);
	gtk_accel_group_unref(d->accelgrp);

	g_free(d->title);
	g_free(d->browse_path_rtn);

	d->freeze_count--;

	g_free(d);
}
