#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <glib.h>
#include <unistd.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "edv_types.h"
#include "edv_obj.h"
#include "edv_archive_obj.h"
#include "edv_utils.h"


edv_archive_object_struct *EDVArchObjectNew(void);
edv_archive_object_struct *EDVArchObjectCopy(
	edv_archive_object_struct *obj
);
void EDVArchObjectSetStat(
	edv_archive_object_struct *obj,
	struct stat *lstat_buf
);
void EDVArchObjectSetToStat(
	struct stat *lstat_buf,
	edv_archive_object_struct *obj
);
void EDVArchObjectDelete(edv_archive_object_struct *obj);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Create a new Archive Object.
 */
edv_archive_object_struct *EDVArchObjectNew(void)
{
	return(EDV_ARCHIVE_OBJECT(
	    g_malloc0(sizeof(edv_archive_object_struct))
	));
}


/*
 *	Coppies the Archive Object.
 */
edv_archive_object_struct *EDVArchObjectCopy(
	edv_archive_object_struct *obj
)
{
	edv_archive_object_struct *new_obj;

	if(obj == NULL)
	    return(NULL);

	new_obj = EDVArchObjectNew();
	if(new_obj == NULL)
	    return(NULL);

	new_obj->type = obj->type;

	new_obj->index = obj->index;

	new_obj->name = STRDUP(obj->name);
	new_obj->full_path = STRDUP(obj->full_path);

	new_obj->size = obj->size;
	new_obj->compressed_size = obj->compressed_size;

	new_obj->link_target = STRDUP(obj->link_target);

	new_obj->permissions = obj->permissions;

	new_obj->access_time = obj->access_time;
	new_obj->modify_time = obj->modify_time;
	new_obj->change_time = obj->change_time;

	new_obj->owner_name = STRDUP(obj->owner_name);
	new_obj->group_name = STRDUP(obj->group_name);

	new_obj->device_type = obj->device_type;

	new_obj->encryption = STRDUP(obj->encryption);
	new_obj->compression_ratio = obj->compression_ratio;
	new_obj->method = STRDUP(obj->method);
	new_obj->crc = STRDUP(obj->crc);

	return(new_obj);
}

/*
 *	Sets/converts the struct stat information to the archive
 *	object.
 *
 *	The obj specifies the target archive object containing the
 *	information to set/convert to.
 *
 *	The lstat_buf specifies the source struct stat to set/convert
 *	the information from.
 *
 *	Note that the struct stat contains less information than the
 *	object so not all the information will be set/converted.
 */
void EDVArchObjectSetStat(
	edv_archive_object_struct *obj,
	struct stat *lstat_buf
)
{
	mode_t m;

	if((lstat_buf == NULL) || (obj == NULL))
	    return;

	m = lstat_buf->st_mode;

	obj->type = EDVStatModeToObjectType(m);

	obj->index = (guint)lstat_buf->st_ino;

	obj->size = (gulong)lstat_buf->st_size;

	obj->permissions = EDVStatModeToEDVPermissions(m);

	obj->access_time = (gulong)lstat_buf->st_atime;
	obj->modify_time = (gulong)lstat_buf->st_mtime;
	obj->change_time = (gulong)lstat_buf->st_ctime;

	obj->device_type = (gint)lstat_buf->st_rdev;
}

/*
 *	Sets/converts the archive object information to the struct
 *	stat.
 *
 *	The lstat_buf specifies the target struct stat to set/convert
 *	the information to.
 *
 *	The obj specifies the source archive object containing the
 *	information to set/convert from.
 *
 *	Note that the struct stat contains less information than the
 *	object so not all the information will be set/converted.
 */
void EDVArchObjectSetToStat(
	struct stat *lstat_buf,
	edv_archive_object_struct *obj
)
{
	if((obj == NULL) || (lstat_buf == NULL))
	    return;

	lstat_buf->st_dev = (dev_t)0;
	lstat_buf->st_ino = (ino_t)obj->index;

	lstat_buf->st_mode = EDVObjectTypeToStatType(obj->type);
	lstat_buf->st_mode |= EDVEDVPermissionsToStatMode(obj->permissions);

	lstat_buf->st_nlink = (nlink_t)0;

	lstat_buf->st_uid = (uid_t)0;
	lstat_buf->st_gid = (gid_t)0;

	lstat_buf->st_rdev = (dev_t)obj->device_type;

	lstat_buf->st_size = (off_t)obj->size;

#if !defined(_WIN32)
	lstat_buf->st_blksize = (unsigned long)0;
	lstat_buf->st_blocks = (unsigned long)0;
#endif

	lstat_buf->st_atime = (time_t)obj->access_time;
	lstat_buf->st_mtime = (time_t)obj->modify_time;
	lstat_buf->st_ctime = (time_t)obj->change_time;
}

/*
 *	Deletes the Archive Object.
 */
void EDVArchObjectDelete(edv_archive_object_struct *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->full_path);

	g_free(obj->link_target);

	g_free(obj->owner_name);
	g_free(obj->group_name);

	g_free(obj->encryption);

	g_free(obj->method);
	g_free(obj->crc);

	g_free(obj);
}
