#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <fnmatch.h>
#include <sys/stat.h>
#include <glib.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "edv_types.h"
#include "edv_obj.h"
#include "edv_recycled_obj.h"
#include "edv_recbin_index.h"
#include "edv_archive_obj.h"
#include "edv_archive_stat.h"
#include "edv_find.h"


/* Grep */
static gchar *EDVFindGrepExcerptFile(
	const gchar *path, const gchar *expression,
	const gboolean case_sensitive, gint *line_index
);


/* Archive Object */
static void EDVFindArchiveObjectByNameIterate(
	edv_core_struct *core,
	const gchar *arch_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
);
gint EDVFindArchiveObjectByName(
	edv_core_struct *core,
	const gchar *arch_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
);

static void EDVFindArchiveObjectBySizeIterate(
	edv_core_struct *core,
	const gchar *arch_path,
	const gulong size,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint EDVFindArchiveObjectBySize(
	edv_core_struct *core,
	const gchar *arch_path,
	const gulong size,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
);

static void EDVFindArchiveObjectByModifyTimeIterate(
	edv_core_struct *core,
	const gchar *arch_path,
	const gulong t,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint EDVFindArchiveObjectByModifyTime(
	edv_core_struct *core,
	const gchar *arch_path,
	const gulong t,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
);


/* Recycled Object */
static void EDVFindRecycledObjectByNameIterate(
	const gchar *index_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint EDVFindRecycledObjectByName(
	const gchar *index_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
);

static void EDVFindRecycledObjectByContentIterate(
	const gchar *index_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, struct stat *,
		const gchar *, const gint,
		gpointer
	),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
);
gint EDVFindRecycledObjectByContent(
	const gchar *index_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, struct stat *,
		const gchar *, const gint,
		gpointer
	),
	gpointer match_data
);

static void EDVFindRecycledObjectBySizeIterate(
	const gchar *index_path,
	const gulong size,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint EDVFindRecycledObjectBySize(
	const gchar *index_path,
	const gulong size,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
);

static void EDVFindRecycledObjectByModifyTimeIterate(
	const gchar *index_path,
	const gulong t,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint EDVFindRecycledObjectByModifyTime(
	const gchar *index_path,
	const gulong t,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
);


/* Object */
static void EDVFindObjectByNameIterate(
	const gchar *path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint EDVFindObjectByName(
	const gchar *start_path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
);

static void EDVFindObjectByContentIterate(
	const gchar *path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, struct stat *,
		const gchar *, const gint,
		gpointer
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint EDVFindObjectByContent(
	const gchar *start_path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, struct stat *,
		const gchar *, const gint,
		gpointer
	),
	gpointer match_data
);

static void EDVFindObjectBySizeIterate(
	const gchar *path,
	const gulong size,
	const gboolean recursive,
	const gboolean follow_links,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint EDVFindObjectBySize(
	const gchar *start_path,
	const gulong size,
	const gboolean recursive,
	const gboolean follow_links,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
);

static void EDVFindObjectByModifyTimeIterate(
	const gchar *path,
	const gulong t,
	const gboolean recursive,
	const gboolean follow_links,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
);
gint EDVFindObjectByModifyTime(
	const gchar *start_path,
	const gulong t,
	const gboolean recursive,
	const gboolean follow_links,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
);


#define EDV_FIND_EXCERPT_PAD_MAX	40


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))
#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))


/*
 *	Opens the file specified by path and searches for a string
 *	within it that matches the given expression.
 *
 *	If a match is made then a dynamically allocated string will be
 *	returned containing the excerpt of the matched line.
 */
static gchar *EDVFindGrepExcerptFile(
	const gchar *path, const gchar *expression,
	const gboolean case_sensitive, gint *line_index
)
{
	FILE *fp;
	gint c;
	gulong line_start_pos;
	const gchar *expre_ptr;
	gchar *excerpt;

	if(line_index != NULL)
	    *line_index = 0;

	if(STRISEMPTY(path) || STRISEMPTY(expression))
	    return(NULL);

	/* Open the file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
	    return(NULL);

	/* Set the expression to the beginning */
	expre_ptr = expression;

	line_start_pos = 0l;
	excerpt = NULL;

	/* Search for the expression in the file */
	while(TRUE)
	{
	    /* Get the next character */
	    c = (gint)fgetc(fp);
	    if((int)c == EOF)
		break;

	    /* If not performing case sensitive search then make the
	     * character that was just read uppercase
	     */
	    if(!case_sensitive)
		c = (gint)toupper((int)c);

	    /* If this character marks the end of the line then record
	     * the last line starting position as one character ahead
	     * of the current position
	     */
	    if(ISCR(c))
	    {
		if(line_index != NULL)
		    *line_index = (*line_index) + 1;

		/* Record the line start position */
		line_start_pos = (gulong)ftell(fp);
	    }

	    /* Does this character matches the current character of the
	     * expression?
	     */
	    if(((gchar)c == *expre_ptr) && ((gchar)c != '\0'))
	    {
		/* Increment to the next character in the expression */
		expre_ptr++;
	    }
	    /* Reset expression back to the beginning as needed */
	    else if(expre_ptr != expression)
	    {
		expre_ptr = expression;
	    }

	    /* Matched the entire expression?
	     *
	     * Test if the expression pointer has been incremented all
	     * the way to the end which suggests that the entire
	     * expression has been matched
	     */
	    if(*expre_ptr == '\0')
	    {
		gchar *s;

		/* Seek to the start of the line */
		fseek(fp, (long)line_start_pos, SEEK_SET);

		/* Get current line as a string */
		s = (gchar *)FGetStringLiteral(fp);
		if(s != NULL)
		{
		    /* Seek to the expression in the excerpt */
		    gchar *s_exp_start = (case_sensitive) ?
			(gchar *)strstr((char *)s, (const char *)expression) :
			(gchar *)strcasestr((char *)s, (const char *)expression);
		    if(s_exp_start != NULL)
		    {
			const gint	left_pad_max = EDV_FIND_EXCERPT_PAD_MAX,
					right_pad_max = left_pad_max;
			gchar	*s2,
				*s_exp_end = s_exp_start + STRLEN(expression);

			/* Sanitize and limit length of the excerpt */
			for(s2 = s; *s2 != '\0'; s2++)
			{
			    /* Non-ACSII character? */
			    if(!isascii((int)(*s2)))
				*s2 = ' ';	/* Replace with space */

			    /* Exceeded the maximum excerpt length? */
			    if(s2 > s_exp_end)
			    {
				if((gint)(s2 - s_exp_end) >= right_pad_max)
				{
				    /* Cap the excerpt and break */
				    *s2 = '\0';
				    break;
				}
			    }
			}

			/* Copy the sanitized excerpt to the return */
			g_free(excerpt);
			if((s_exp_start - s) > left_pad_max)
			    excerpt = STRDUP(s_exp_start - left_pad_max);
			else
			    excerpt = STRDUP(s);
		    }

		    g_free(s);
		}

		break;	/* Break after the first match */

	    }	/* Matched the entire expression? */
	}

	/* Close the file */
	fclose(fp);

	return(excerpt);
}


/*
 *	Called by EDVFindArchiveObjectByName().
 */
static void EDVFindArchiveObjectByNameIterate(
	edv_core_struct *core,
	const gchar *arch_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
)
{
	gint i, nobjs;
	GList *glist, *obj_list;
	edv_archive_object_struct *obj;

	if(*stop_find)
	    return;

	/* Get the list of all the objects in the archive */
	obj_list = EDVArchStatList(
	    core,
	    arch_path,
	    NULL,			/* Get all the objects */
	    NULL,			/* No filter */
	    NULL,			/* No password */
	    NULL, NULL			/* No progress callback */
	);

	/* Iterate through all the objects in the archive */
	nobjs = g_list_length(obj_list);
	for(glist = obj_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
	    obj = EDV_ARCHIVE_OBJECT(glist->data);
	    if(obj == NULL)
		continue;

	    /* User aborted? */
	    if(*stop_find)
		break;

	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    obj->name,
		    (gfloat)i / (gfloat)nobjs,
		    progress_data
		))
		{
		    *stop_find = TRUE;
		    continue;
		}
	    }

	    /* Match name */
	    if(!STRISEMPTY(obj->name))
	    {
		gchar *s = STRDUP(obj->name);
		if(!case_sensitive)
		    g_strup(s);

		/* Names match? */
		if(!fnmatch((const char *)expression, (const char *)s, 0))
		{
		    /* Convert the object in the archive stats */
		    struct stat *lstat_buf = (struct stat *)g_malloc(
			sizeof(struct stat)
		    );
		    if(lstat_buf != NULL)
			EDVArchObjectSetToStat(lstat_buf, obj);

		    /* Count this match */
		    *nmatches = (*nmatches) + 1;

		    /* Report this match */
		    if(match_cb != NULL)
			match_cb(
			    obj->full_path,	/* Path */
			    lstat_buf,		/* Stats */
			    match_data		/* Data */
			);

		    g_free(lstat_buf);
		}

		g_free(s);
	    }
	}

	/* Delete the archive object stats */
	if(obj_list != NULL)
	{
	    g_list_foreach(obj_list, (GFunc)EDVArchObjectDelete, NULL);
	    g_list_free(obj_list);
	}
}

/*
 *	Find an archive object by name.
 *
 *	The arch_path specifies the full path to the archive.
 *
 *	The expression specifies the string describing the object's
 *	name to find. Wildcards are allowed.
 *
 *	If case_sensitive is TRUE then only case sensitive matches
 *	will be made.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be
 *	reported to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindArchiveObjectByName(
	edv_core_struct *core,
	const gchar *arch_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *lexpression;

	if((core == NULL) || STRISEMPTY(arch_path) ||
	   STRISEMPTY(expression)
	)
	    return(0);

	if(!strcmp((const char *)expression, "*"))
	    return(0);

	/* Convert expression to upper case if not performing case
	 * sensitive matching
	 */
	lexpression = STRDUP(expression);
	if(!case_sensitive)
	    g_strup(lexpression);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindArchiveObjectByNameIterate(
	    core,
	    arch_path,
	    lexpression,
	    case_sensitive,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches,
	    &stop_find
	);

	g_free(lexpression);

	return(nmatches);
}


/*
 *	Called by EDVFindArchiveObjectBySize().
 */
static void EDVFindArchiveObjectBySizeIterate(
	edv_core_struct *core,
	const gchar *arch_path,
	const gulong size,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint i, nobjs;
	GList *glist, *obj_list;
	edv_archive_object_struct *obj;

	if(*stop_find)
	    return;

	/* Get the list of all the objects in the archive */
	obj_list = EDVArchStatList(
	    core,
	    arch_path,
	    NULL,			/* Get all the objects */
	    NULL,			/* No filter */
	    NULL,			/* No password */
	    NULL, NULL			/* No progress callback */
	);

	/* Iterate through all the objects in the archive */
	nobjs = g_list_length(obj_list);
	for(glist = obj_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
	    obj = EDV_ARCHIVE_OBJECT(glist->data);
	    if(obj == NULL)
		continue;

	    /* User aborted? */
	    if(*stop_find)
		break;

	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    obj->name,
		    (gfloat)i / (gfloat)nobjs,
		    progress_data
		))
		{
		    *stop_find = TRUE;
		    continue;
		}
	    }

	    /* Match size */
	    if(TRUE)
	    {
		gboolean got_match = FALSE;

		/* Match the size by the size relativity */
		switch(relativity)
		{
		  case EDV_FIND_EQUAL_TO:
		    if(size == obj->size)
			got_match = TRUE;
		    break;

		  case EDV_FIND_NOT_EQUAL_TO:
		    if(size != obj->size)
			got_match = TRUE;
		    break;

		  case EDV_FIND_LESS_THAN:
		    if(size >= obj->size)
			got_match = TRUE;
		    break;

		  case EDV_FIND_GREATER_THAN:
		    if(size <= obj->size)
			got_match = TRUE;
		    break;
		}
		if(got_match)
		{
		    /* Convert the object in the archive stats */
		    struct stat *lstat_buf = (struct stat *)g_malloc(
			sizeof(struct stat)
		    );
		    if(lstat_buf != NULL)
			EDVArchObjectSetToStat(lstat_buf, obj);

		    /* Count this match */
		    *nmatches = (*nmatches) + 1;

		    /* Report this match */
		    if(match_cb != NULL)
			match_cb(
			    obj->full_path,	/* Path */
			    lstat_buf,		/* Stats */
			    match_data		/* Data */
			);

		    g_free(lstat_buf);
		}
	    }
	}

	/* Delete the archive object stats */
	if(obj_list != NULL)
	{
	    g_list_foreach(obj_list, (GFunc)EDVArchObjectDelete, NULL);
	    g_list_free(obj_list);
	}
}

/*
 *	Find an archive object by size.
 *
 *	The arch_path specifies the full path to the archive.
 *
 *	The size specifies the size in bytes to find.
 *
 *	The relativity specifies the size relativity, one of
 *	EDV_FIND_*.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be
 *	reported to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindArchiveObjectBySize(
	edv_core_struct *core,
	const gchar *arch_path,
	const gulong size,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;

	if(STRISEMPTY(arch_path))
	    return(0);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindArchiveObjectBySizeIterate(
	    core,
	    arch_path,
	    size,
	    relativity,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches,
	    &stop_find
	);

	return(nmatches);
}


/*
 *	Called by EDVFindArchiveObjectByModifyTime().
 */
static void EDVFindArchiveObjectByModifyTimeIterate(
	edv_core_struct *core,
	const gchar *arch_path,
	const gulong t,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint i, nobjs;
	GList *glist, *obj_list;
	edv_archive_object_struct *obj;

	if(*stop_find)
	    return;

	/* Get the list of all the objects in the archive */
	obj_list = EDVArchStatList(
	    core,
	    arch_path,
	    NULL,			/* Get all the objects */
	    NULL,			/* No filter */
	    NULL,			/* No password */
	    NULL, NULL			/* No progress callback */
	);

	/* Iterate through all the objects in the archive */
	nobjs = g_list_length(obj_list);
	for(glist = obj_list, i = 0;
	    glist != NULL;
	    glist = g_list_next(glist), i++
	)
	{
	    obj = EDV_ARCHIVE_OBJECT(glist->data);
	    if(obj == NULL)
		continue;

	    /* User aborted? */
	    if(*stop_find)
		break;

	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    obj->name,
		    (gfloat)i / (gfloat)nobjs,
		    progress_data
		))
		{
		    *stop_find = TRUE;
		    continue;
		}
	    }

	    /* Match modify time */
	    if(TRUE)
	    {
		gboolean got_match = FALSE;

		/* Match the size by the size relativity */
		switch(relativity)
		{
		  case EDV_FIND_EQUAL_TO:
		    if(t == obj->modify_time)
			got_match = TRUE;
		    break;

		  case EDV_FIND_NOT_EQUAL_TO:
		    if(t != obj->modify_time)
			got_match = TRUE;
		    break;

		  case EDV_FIND_LESS_THAN:
		    if(t >= obj->modify_time)
			got_match = TRUE;
		    break;

		  case EDV_FIND_GREATER_THAN:
		    if(t <= obj->modify_time)
			got_match = TRUE;
		    break;
		}
		if(got_match)
		{
		    /* Convert the object in the archive stats */
		    struct stat *lstat_buf = (struct stat *)g_malloc(
			sizeof(struct stat)
		    );
		    if(lstat_buf != NULL)
			EDVArchObjectSetToStat(lstat_buf, obj);

		    /* Count this match */
		    *nmatches = (*nmatches) + 1;

		    /* Report this match */
		    if(match_cb != NULL)
			match_cb(
			    obj->full_path,	/* Path */
			    lstat_buf,		/* Stats */
			    match_data		/* Data */
			);

		    g_free(lstat_buf);
		}
	    }
	}

	/* Delete the archive object stats */
	if(obj_list != NULL)
	{
	    g_list_foreach(obj_list, (GFunc)EDVArchObjectDelete, NULL);
	    g_list_free(obj_list);
	}
}

/*
 *	Find an archive object by modify time.
 *
 *	The arch_path specifies the full path to the archive.
 *
 *	The t specifies the time in seconds since EPOCH to find.
 *
 *	The relativity specifies the time relativity, one of
 *	EDV_FIND_*.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be
 *	reported to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindArchiveObjectByModifyTime(
	edv_core_struct *core,
	const gchar *arch_path,
	const gulong t,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;

	if(STRISEMPTY(arch_path))
	    return(0);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindArchiveObjectByModifyTimeIterate(
	    core,
	    arch_path,
	    t,
	    relativity,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches,
	    &stop_find
	);

	return(nmatches);
}


/*
 *	Called by EDVFindRecycledObjectByName().
 */
static void EDVFindRecycledObjectByNameIterate(
	const gchar *index_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches, gboolean *stop_find
)
{
	gint i, nobjs;
	gchar *recbin_path;
	edv_recycled_object_struct *obj;
	edv_recbin_index_struct *rbi_ptr;

	if(*stop_find)
	    return;

	/* Get the recycle bin directory */
	recbin_path = EDVRecBinIndexGetRecycleBinDirectory(index_path);

	/* Get the total number of recycled objects */
	nobjs = EDVRecBinIndexGetTotal(index_path);

	/* Get recycled objects directory index pointer used to iterate
	 * through all recycled objects
	 */
	i = 0;
	rbi_ptr = EDVRecBinIndexOpen(index_path);
	while(!EDVRecBinIndexNext(rbi_ptr))
	{
	    obj = rbi_ptr->obj;
	    if(obj == NULL)
	    {
		i++;
		continue;
	    }

	    /* User aborted? */
	    if(*stop_find)
		break;

	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    obj->name,
		    (gfloat)i / (gfloat)nobjs,
		    progress_data
		))
		{
		    *stop_find = TRUE;
		    i++;
		    continue;
		}
	    }

	    /* Match name */
	    if(!STRISEMPTY(obj->name))
	    {
		gchar *s = STRDUP(obj->name);
		if(!case_sensitive)
		    g_strup(s);

		/* Names match? */
		if(!fnmatch((const char *)expression, (const char *)s, 0))
		{
		    /* Got match, Count this match and report
		     * the match
		     */
		    gchar *index_str = g_strdup_printf("#%u", obj->index);

		    /* Convert the recycled object's stats */
		    struct stat *lstat_buf = (struct stat *)g_malloc(
			sizeof(struct stat)
		    );
		    if(lstat_buf != NULL)
			EDVRecycledObjectSetToStat(lstat_buf, obj);

		    /* Count this match */
		    *nmatches = (*nmatches) + 1;

		    /* Report this match */
		    if(match_cb != NULL)
			match_cb(
			    index_str,		/* Index as path */
			    lstat_buf,		/* Stats */
			    match_data		/* Data */
			);

		    g_free(lstat_buf);
		    g_free(index_str);
		}

		g_free(s);
	    }
	}

	EDVRecBinIndexClose(rbi_ptr);

	g_free(recbin_path);
}

/*
 *	Find a recycled object by name.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	The expression specifies the string describing the object's
 *	name to find. Wildcards are allowed.
 *
 *	If case_sensitive is TRUE then only case sensitive matches
 *	will be made.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be
 *	reported to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindRecycledObjectByName(
	const gchar *index_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *lexpression;

	if(STRISEMPTY(index_path) || STRISEMPTY(expression))
	    return(0);
	if(!strcmp((const char *)expression, "*"))
	    return(0);

	/* Convert expression to upper case if not matching case
	 * sensitive
	 */
	lexpression = STRDUP(expression);
	if(!case_sensitive)
	    g_strup(lexpression);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindRecycledObjectByNameIterate(
	    index_path,
	    lexpression,
	    case_sensitive,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches,
	    &stop_find
	);

	g_free(lexpression);

	return(nmatches);
}


/*
 *	Called by EDVFindRecycledObjectByContent() to perform one find
 *	iteration on the given path.
 */
static void EDVFindRecycledObjectByContentIterate(
	const gchar *index_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, struct stat *,
		const gchar *, const gint,
		gpointer
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint i, nobjs;
	gchar *recbin_path;
	edv_recycled_object_struct *obj;
	edv_recbin_index_struct *rbi_ptr;

	if(*stop_find)
	    return;

	/* Get the recycle bin directory */
	recbin_path = EDVRecBinIndexGetRecycleBinDirectory(index_path);

	/* Get the total number of recycled objects */
	nobjs = EDVRecBinIndexGetTotal(index_path);

	/* Get recycled objects directory index pointer used to iterate
	 * through all recycled objects
	 */
	i = 0;
	rbi_ptr = EDVRecBinIndexOpen(index_path);
	while(!EDVRecBinIndexNext(rbi_ptr))
	{
	    obj = rbi_ptr->obj;
	    if(obj == NULL)
	    {
		i++;
		continue;
	    }

	    /* User aborted? */
	    if(*stop_find)
		break;

	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    obj->name,
		    (gfloat)i / (gfloat)nobjs,
		    progress_data
		))
		{
		    *stop_find = TRUE;
		    i++;
		    continue;
		}
	    }

	    /* Match content */
	    if(TRUE)
	    {
		gint line_index;
		gchar *full_path = EDVRecBinIndexGetRecycledObjectPath(
		    index_path,
		    obj->index
		);

		/* Contents match? */
		gchar *excerpt = EDVFindGrepExcerptFile(
		    full_path,
		    expression,
		    case_sensitive,
		    &line_index
		);
		if(excerpt != NULL)
		{
		    /* Got match, count this match and report
		     * the match
		     */
		    gchar *index_str = g_strdup_printf("#%u", obj->index);

		    /* Convert the recycled object's stats */
		    struct stat *lstat_buf = (struct stat *)g_malloc(
			sizeof(struct stat)
		    );
		    if(lstat_buf != NULL)
			EDVRecycledObjectSetToStat(lstat_buf, obj);

		    /* Count this match */
		    *nmatches = (*nmatches) + 1;

		    /* Report this match */
		    if(match_cb != NULL)
			match_cb(
			    index_str,		/* Path as index string */
			    lstat_buf,		/* Stats */
			    excerpt,		/* Excerpt */
			    line_index,		/* Line Index */
			    match_data		/* Data */
			); 

		    g_free(lstat_buf);
		    g_free(index_str);
		    g_free(excerpt);
		}

		g_free(full_path);
	    }
	}

	EDVRecBinIndexClose(rbi_ptr);

	g_free(recbin_path);
}

/*
 *	Find a recycled object by content.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	The expression specifies the string to find within the
 *	object's contents.
 *
 *	If case_sensitive is TRUE then only case sensitive matches
 *	will be made.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be
 *	reported to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindRecycledObjectByContent(
	const gchar *index_path,
	const gchar *expression,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, struct stat *,
		const gchar *, const gint,
		gpointer
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *lexpression;

	if(STRISEMPTY(index_path) || STRISEMPTY(expression))
	    return(0);

	/* Convert expression to upper case if not performing case
	 * sensitive matching
	 */
	lexpression = STRDUP(expression);
	if(!case_sensitive)
	    g_strup(lexpression);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindRecycledObjectByContentIterate(
	    index_path,
	    lexpression,
	    case_sensitive,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches,
	    &stop_find
	);

	g_free(lexpression);

	return(nmatches);
}


/*
 *	Called by EDVFindRecycledObjectBySize().
 */
static void EDVFindRecycledObjectBySizeIterate(
	const gchar *index_path,
	const gulong size,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint i, nobjs;
	gchar *recbin_path;
	edv_recycled_object_struct *obj;
	edv_recbin_index_struct *rbi_ptr;

	if(*stop_find)
	    return;

	/* Get the recycle bin directory */
	recbin_path = EDVRecBinIndexGetRecycleBinDirectory(index_path);

	/* Get the total number of recycled objects */
	nobjs = EDVRecBinIndexGetTotal(index_path);

	/* Get recycled objects directory index pointer used to iterate
	 * through all recycled objects
	 */
	i = 0;
	rbi_ptr = EDVRecBinIndexOpen(index_path);
	while(!EDVRecBinIndexNext(rbi_ptr))
	{
	    obj = rbi_ptr->obj;
	    if(obj == NULL)
	    {
		i++;
		continue;
	    }

	    /* User aborted? */
	    if(*stop_find)
		break;

	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    obj->name,
		    (gfloat)i / (gfloat)nobjs,
		    progress_data
		))
		{
		    *stop_find = TRUE;
		    i++;
		    continue;
		}
	    }

	    /* Match size */
	    if(TRUE)
	    {
		gboolean got_match = FALSE;

		/* Match the size by the size relativity */
		switch(relativity)
		{
		  case EDV_FIND_EQUAL_TO:
		    if(size == obj->size)
			got_match = TRUE;
		    break;

		  case EDV_FIND_NOT_EQUAL_TO:
		    if(size != obj->size)
			got_match = TRUE;
		    break;

		  case EDV_FIND_LESS_THAN:
		    if(size >= obj->size)
			got_match = TRUE;
		    break;

		  case EDV_FIND_GREATER_THAN:
		    if(size <= obj->size)
			got_match = TRUE;
		    break;
		}
		if(got_match)
		{
		    /* Got match, Count this match and report
		     * the match
		     */
		    gchar *index_str = g_strdup_printf("#%u", obj->index);

		    /* Convert the recycled object's stats */
		    struct stat *lstat_buf = (struct stat *)g_malloc(
			sizeof(struct stat)
		    );
		    if(lstat_buf != NULL)
			EDVRecycledObjectSetToStat(lstat_buf, obj);

		    /* Count this match */
		    *nmatches = (*nmatches) + 1;

		    /* Report this match */
		    if(match_cb != NULL)
			match_cb(
			    index_str,		/* Path as index string */
			    lstat_buf,		/* Stats */
			    match_data		/* Data */
			); 

		    g_free(lstat_buf);
		    g_free(index_str);
		}
	    }
	}

	EDVRecBinIndexClose(rbi_ptr);

	g_free(recbin_path);
}

/*
 *	Find a recycled object by size.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	The size specifies the size in bytes to find.
 *
 *	The relativity specifies the size relativity, one of
 *	EDV_FIND_*.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindRecycledObjectBySize(
	const gchar *index_path,
	const gulong size,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;

	if(STRISEMPTY(index_path))
	    return(0);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindRecycledObjectBySizeIterate(
	    index_path,
	    size,
	    relativity,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches,
	    &stop_find
	);

	return(nmatches);
}


/*
 *	Called by EDVFindRecycledObjectByModifyTime().
 */
static void EDVFindRecycledObjectByModifyTimeIterate(
	const gchar *index_path,
	const gulong t,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint i, nobjs;
	gchar *recbin_path;
	edv_recycled_object_struct *obj;
	edv_recbin_index_struct *rbi_ptr;

	if(*stop_find)
	    return;

	/* Get the recycle bin directory */
	recbin_path = EDVRecBinIndexGetRecycleBinDirectory(index_path);

	/* Get the total number of recycled objects */
	nobjs = EDVRecBinIndexGetTotal(index_path);

	/* Get recycled objects directory index pointer used to iterate
	 * through all recycled objects
	 */
	i = 0;
	rbi_ptr = EDVRecBinIndexOpen(index_path);
	while(!EDVRecBinIndexNext(rbi_ptr))
	{
	    obj = rbi_ptr->obj;
	    if(obj == NULL)
	    {
		i++;
		continue;
	    }

	    /* User aborted? */
	    if(*stop_find)
		break;

	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    obj->name,
		    (gfloat)i / (gfloat)nobjs,
		    progress_data
		))
		{
		    *stop_find = TRUE;
		    i++;
		    continue;
		}
	    }

	    /* Match modify time */
	    if(TRUE)
	    {
		gboolean got_match = FALSE;

		/* Match the modify time by the time relativity */
		switch(relativity)
		{
		  case EDV_FIND_EQUAL_TO:
		    if(t == obj->modify_time)
			got_match = TRUE;
		    break;

		  case EDV_FIND_NOT_EQUAL_TO:
		    if(t != obj->modify_time)
			got_match = TRUE;
		    break;

		  case EDV_FIND_LESS_THAN:
		    if(t >= obj->modify_time)
			got_match = TRUE;
		    break;

		  case EDV_FIND_GREATER_THAN:
		    if(t <= obj->modify_time)
			got_match = TRUE;
		    break;
		}
		if(got_match)
		{
		    /* Got match, Count this match and report
		     * the match
		     */
		    gchar *index_str = g_strdup_printf("#%u", obj->index);

		    /* Convert the recycled object's stats */
		    struct stat *lstat_buf = (struct stat *)g_malloc(
			sizeof(struct stat)
		    );
		    if(lstat_buf != NULL)
			EDVRecycledObjectSetToStat(lstat_buf, obj);

		    /* Count this match */
		    *nmatches = (*nmatches) + 1;

		    /* Report this match */
		    if(match_cb != NULL)
			match_cb(
			    index_str,		/* Path as index string */
			    lstat_buf,		/* Stats */
			    match_data		/* Data */
			); 

		    g_free(lstat_buf);
		    g_free(index_str);
		}
	    }
	}

	EDVRecBinIndexClose(rbi_ptr);

	g_free(recbin_path);
}

/*
 *	Find a recycled object by modify time.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	The t specifies the modify time in seconds since EPOCH to find.
 *
 *	The relativity specifies the time relativity, one of
 *	EDV_FIND_*.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindRecycledObjectByModifyTime(
	const gchar *index_path,
	const gulong t,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;

	if(STRISEMPTY(index_path))
	    return(0);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindRecycledObjectByModifyTimeIterate(
	    index_path,
	    t,
	    relativity,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches,
	    &stop_find
	);

	return(nmatches);
}


/*
 *	Called by EDVFindObjectByName().
 */
static void EDVFindObjectByNameIterate(
	const gchar *path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint strc;
	gchar **strv;

	if(*stop_find)
	    return;

	/* Get the listing of objects at the specified location */
	strv = (gchar **)GetDirEntNames2((const char *)path, (int *)&strc);
	if(strv != NULL)
	{
	    struct stat stat_buf;
	    gint i, status;
	    const gchar *name;
	    gchar *full_path = NULL;

	    strv = (gchar **)StringQSort((char **)strv, (int)strc);
	    if(strv == NULL)
		return;

	    for(i = 0; i < strc; i++)
	    {
#define FREE_CONTINUE	{		\
 g_free(full_path);			\
 full_path = NULL;			\
 g_free(strv[i]);			\
 strv[i] = NULL;			\
 continue;				\
}
		name = strv[i];
		if(STRISEMPTY(name))
		    FREE_CONTINUE

		/* User requested stop? */
		if(*stop_find)
		    FREE_CONTINUE

		/* Skip special directory notations */
		if(!strcmp((const char *)name, "..") ||
		   !strcmp((const char *)name, ".")
		)
		    FREE_CONTINUE

		/* Get the full path */
		full_path = STRDUP(PrefixPaths(
		    (const char *)path, (const char *)name
		));
		if(full_path == NULL)
		    FREE_CONTINUE

		/* Report progress */
		if(progress_cb != NULL)
		{
		    if(progress_cb(
			full_path,
			recursive ? -1.0f : ((gfloat)i / (gfloat)strc),
			progress_data
		    ))
		    {
			*stop_find = TRUE;
			FREE_CONTINUE
		    }
		}

		/* Get the object's statistics */
		if(follow_links)
		    status = (gint)stat((const char *)full_path, &stat_buf);
		else
		    status = (gint)lstat((const char *)full_path, &stat_buf);
		if(status == 0)
		{
		    /* Get/copy the object's name and make it upper case
		     * if not matching case sensitive
		     */
		    gchar *s = STRDUP(name);
		    if(!case_sensitive)
			g_strup(s);

		    /* Match the object's name */
		    if(!fnmatch((const char *)expression, (const char *)s, 0))
		    {
			*nmatches = (*nmatches) + 1;
			if(match_cb != NULL)
			    match_cb(
				full_path,		/* Path */
				&stat_buf,		/* Stats */
				match_data		/* Data */
			    );
		    }

		    g_free(s);
		}

		/* Is it a directory and should we recurse into it? */
#ifdef S_ISDIR
		if(recursive && S_ISDIR(stat_buf.st_mode))
#else
		if(FALSE)
#endif
		{
		    EDVFindObjectByNameIterate(
			full_path,
			expression,
			recursive,
			follow_links,
			case_sensitive,
			progress_cb, progress_data,
			match_cb, match_data,
			nmatches,
			stop_find
		    );
		}

		FREE_CONTINUE
#undef FREE_CONTINUE
	    }

	    g_free(strv);
	}
}

/*
 *	Find an object by name.
 *
 *	The start_path specifies the full path to the location where
 *	the search is to be made.
 *
 *	The expression specifies the string describing the object's
 *	name to find. Wildcards are allowed.
 *
 *	If case_sensitive is TRUE then only case sensitive matches
 *	will be made.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindObjectByName(
	const gchar *start_path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *lexpression;

	if(STRISEMPTY(start_path) || STRISEMPTY(expression))
	    return(0);
	if(!strcmp((const char *)expression, "*"))
	    return(0);

	/* Convert the expression to upper case if not matching case
	 * sensitive
	 */
	lexpression = STRDUP(expression);
	if(!case_sensitive)
	    g_strup(lexpression);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindObjectByNameIterate(
	    start_path,
	    lexpression,
	    recursive,
	    follow_links,
	    case_sensitive,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches,
	    &stop_find
	);

	g_free(lexpression);

	return(nmatches);
}


/*
 *	Called by EDVFindObjectByContent().
 */
static void EDVFindObjectByContentIterate(
	const gchar *path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, struct stat *,
		const gchar *, const gint,
		gpointer
	),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint strc;
	gchar **strv;

	if(*stop_find)
	    return;

	/* Get the listing of objects at the specified location */
	strv = (gchar **)GetDirEntNames2((const char *)path, (int *)&strc);
	if(strv != NULL)
	{
	    struct stat stat_buf;
	    gint i, status;
	    const gchar *name;
	    gchar *full_path = NULL;

	    strv = (gchar **)StringQSort((char **)strv, (int)strc);
	    if(strv == NULL)
		return;

	    for(i = 0; i < strc; i++)
	    {
#define FREE_CONTINUE	{		\
 g_free(full_path);			\
 full_path = NULL;			\
 g_free(strv[i]);			\
 strv[i] = NULL;			\
 continue;				\
}
		name = strv[i];
		if(STRISEMPTY(name))
		    FREE_CONTINUE

		/* User requested stop? */
		if(*stop_find)
		    FREE_CONTINUE

		/* Skip special directory notations */
		if(!strcmp((const char *)name, "..") ||
		   !strcmp((const char *)name, ".")
		)
		    FREE_CONTINUE

		/* Get the full path */
		full_path = STRDUP(PrefixPaths(
		    (const char *)path, (const char *)name
		));
		if(full_path == NULL)
		    FREE_CONTINUE

		/* Report progress */
		if(progress_cb != NULL)
		{
		    if(progress_cb(
			full_path,
			recursive ? -1.0f : ((gfloat)i / (gfloat)strc),
			progress_data
		    ))
		    {
			*stop_find = TRUE;
			FREE_CONTINUE
		    }
		}

		/* Get the object's statistics */
		if(follow_links)
		    status = (gint)stat((const char *)full_path, &stat_buf);
		else
		    status = (gint)lstat((const char *)full_path, &stat_buf);
		if(status == 0)
		{
		    /* Match the object's contents */
		    gint line_index;
		    gchar *excerpt = EDVFindGrepExcerptFile(
			full_path, expression,
			case_sensitive, &line_index
		    );
		    if(excerpt != NULL)
		    {
			*nmatches = (*nmatches) + 1;
			if(match_cb != NULL)
			    match_cb(
				full_path,		/* Path */
				&stat_buf,		/* Stats */
				excerpt,		/* Excerpt */
				line_index,		/* Line Index */
				match_data		/* Data */
			    );

			g_free(excerpt);
		    }
		}

		/* Is it a directory and should we recurse into it? */
#ifdef S_ISDIR
		if(recursive && S_ISDIR(stat_buf.st_mode))
#else
		if(FALSE)
#endif
		{
		    EDVFindObjectByContentIterate(
			full_path,
			expression,
			recursive,
			follow_links,
			case_sensitive,
			progress_cb, progress_data,
			match_cb, match_data,
			nmatches,
			stop_find
		    );
		}

		FREE_CONTINUE
#undef FREE_CONTINUE
	    }

	    g_free(strv);
	}
}

/*
 *	Find an object by its content.
 *
 *	The start_path specifies the full path to the location where
 *	the search is to be made.
 *
 *	The expression specifies the string to find within the
 *	object's contents.
 *
 *	If case_sensitive is TRUE then only case sensitive matches
 *	will be made.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be
 *	reported to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindObjectByContent(
	const gchar *start_path,
	const gchar *expression,
	const gboolean recursive,
	const gboolean follow_links,
	const gboolean case_sensitive,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(
		const gchar *, struct stat *,
		const gchar *, const gint,
		gpointer
	),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;
	gchar *lexpression;

	if(STRISEMPTY(start_path) || STRISEMPTY(expression))
	    return(0);

	/* Convert the expression to upper case if not matching
	 * case sensitive
	 */
	lexpression = STRDUP(expression);
	if(!case_sensitive)
	    g_strup(lexpression);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindObjectByContentIterate(
	    start_path,
	    lexpression,
	    recursive,
	    follow_links,
	    case_sensitive,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches,
	    &stop_find
	);

	g_free(lexpression);

	return(nmatches);
}


/*
 *	Called by EDVFindObjectBySize().
 */
static void EDVFindObjectBySizeIterate(
	const gchar *path,
	const gulong size,
	const gboolean recursive,
	const gboolean follow_links,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint strc;
	gchar **strv;

	if(*stop_find)
	    return;

	/* Get the listing of objects at the specified location */
	strv = (gchar **)GetDirEntNames2((const char *)path, (int *)&strc);
	if(strv != NULL)
	{
	    struct stat stat_buf;
	    gint i, status;
	    const gchar *name;
	    gchar *full_path = NULL;

	    strv = (gchar **)StringQSort((char **)strv, (int)strc);
	    if(strv == NULL)
		return;

	    for(i = 0; i < strc; i++)
	    {
#define FREE_CONTINUE	{		\
 g_free(full_path);			\
 full_path = NULL;			\
 g_free(strv[i]);			\
 strv[i] = NULL;			\
 continue;				\
}
		name = strv[i];
		if(STRISEMPTY(name))
		    FREE_CONTINUE

		/* User requested stop? */
		if(*stop_find)
		    FREE_CONTINUE

		/* Skip special directory notations */
		if(!strcmp((const char *)name, "..") ||
		   !strcmp((const char *)name, ".")
		)
		    FREE_CONTINUE

		/* Get the full path */
		full_path = STRDUP(PrefixPaths(
		    (const char *)path, (const char *)name
		));
		if(full_path == NULL)
		    FREE_CONTINUE

		/* Report progress */
		if(progress_cb != NULL)
		{
		    if(progress_cb(
			full_path,
			recursive ? -1.0f : ((gfloat)i / (gfloat)strc),
			progress_data
		    ))
		    {
			*stop_find = TRUE;
			FREE_CONTINUE
		    }
		}

		/* Get the object's statistics */
		if(follow_links)
		    status = (gint)stat((const char *)full_path, &stat_buf);
		else
		    status = (gint)lstat((const char *)full_path, &stat_buf);
		if(status == 0)
		{
		    /* Match the size by the size relativity */
		    switch(relativity)
		    {
		      case EDV_FIND_EQUAL_TO:
			if(size == (gulong)stat_buf.st_size)
			{
			    *nmatches = (*nmatches) + 1;
			    if(match_cb != NULL)
				match_cb(
				    full_path,		/* Path */
				    &stat_buf,		/* Stats */
				    match_data		/* Data */
				);
			}
			break;

		      case EDV_FIND_NOT_EQUAL_TO:
			if(size != (gulong)stat_buf.st_size)
			{
			    *nmatches = (*nmatches) + 1;
			    if(match_cb != NULL)
				match_cb(
				    full_path,		/* Path */
				    &stat_buf,		/* Stats */
				    match_data		/* Data */
				);
			}
			break;

		      case EDV_FIND_LESS_THAN:
			if(size >= (gulong)stat_buf.st_size)
			{
			    *nmatches = (*nmatches) + 1;
			    if(match_cb != NULL)
				match_cb(
				    full_path,		/* Path */
				    &stat_buf,		/* Stats */
				    match_data		/* Data */
				);
			}
			break;

		      case EDV_FIND_GREATER_THAN:
			if(size <= (gulong)stat_buf.st_size)
			{
			    *nmatches = (*nmatches) + 1;
			    if(match_cb != NULL)
				match_cb(
				    full_path,		/* Path */
				    &stat_buf,		/* Stats */
				    match_data		/* Data */
				);
			}
			break;
		    }
		}

		/* Is it a directory and should we recurse into it? */
#ifdef S_ISDIR
		if(recursive && S_ISDIR(stat_buf.st_mode))
#else
		if(FALSE)
#endif
		{
		    EDVFindObjectBySizeIterate(
			full_path,
			size,
			recursive,
			follow_links,
			relativity,
			progress_cb, progress_data,
			match_cb, match_data,
			nmatches,
			stop_find
		    );
		}

		FREE_CONTINUE
#undef FREE_CONTINUE
	    }

	    g_free(strv);
	}
}

/*
 *	Find an object by size.
 *
 *	The start_path specifies the full path to the location where
 *	the search is to be made.
 *
 *	The size specifies the size in bytes to find.
 *
 *	The relativity specifies the size relativity, one of
 *	EDV_FIND_*.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindObjectBySize(
	const gchar *start_path,
	const gulong size,
	const gboolean recursive,
	const gboolean follow_links,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;

	if(STRISEMPTY(start_path))
	    return(0);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindObjectBySizeIterate(
	    start_path,
	    size,
	    recursive,
	    follow_links,
	    relativity,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches,
	    &stop_find
	);

	return(nmatches);
}


/*
 *	Called by EDVFindObjectByModifyTime().
 */
static void EDVFindObjectByModifyTimeIterate(
	const gchar *path,
	const gulong t,
	const gboolean recursive,
	const gboolean follow_links,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data,
	gint *nmatches,
	gboolean *stop_find
)
{
	gint strc;
	gchar **strv;

	if(*stop_find)
	    return;

	/* Get the listing of objects at the specified location */
	strv = (gchar **)GetDirEntNames2((const char *)path, (int *)&strc);
	if(strv != NULL)
	{
	    struct stat stat_buf;
	    gint i, status;
	    const gchar *name;
	    gchar *full_path = NULL;

	    strv = (gchar **)StringQSort((char **)strv, (int)strc);
	    if(strv == NULL)
		return;

	    for(i = 0; i < strc; i++)
	    {
#define FREE_CONTINUE	{		\
 g_free(full_path);			\
 full_path = NULL;			\
 g_free(strv[i]);			\
 strv[i] = NULL;			\
 continue;				\
}
		name = strv[i];
		if(STRISEMPTY(name))
		    FREE_CONTINUE

		/* User requested stop? */
		if(*stop_find)
		    FREE_CONTINUE

		/* Skip special directory notations */
		if(!strcmp((const char *)name, "..") ||
		   !strcmp((const char *)name, ".")
		)
		    FREE_CONTINUE

		/* Get the full path */
		full_path = STRDUP(PrefixPaths(
		    (const char *)path, (const char *)name
		));
		if(full_path == NULL)
		    FREE_CONTINUE

		/* Report progress */
		if(progress_cb != NULL)
		{
		    if(progress_cb(
			full_path,
			recursive ? -1.0f : ((gfloat)i / (gfloat)strc),
			progress_data
		    ))
		    {
			*stop_find = TRUE;
			FREE_CONTINUE
		    }
		}

		/* Get the object's statistics */
		if(follow_links)
		    status = (gint)stat((const char *)full_path, &stat_buf);
		else
		    status = (gint)lstat((const char *)full_path, &stat_buf);
		if(status == 0)
		{
		    /* Match the modify time by the relativity */
		    switch(relativity)
		    {
		      case EDV_FIND_EQUAL_TO:
			if(t == (gulong)stat_buf.st_mtime)
			{
			    *nmatches = (*nmatches) + 1;
			    if(match_cb != NULL)
				match_cb(
				    full_path,		/* Path */
				    &stat_buf,		/* Stats */
				    match_data		/* Data */
				);
			}
			break;

		      case EDV_FIND_NOT_EQUAL_TO:
			if(t != (gulong)stat_buf.st_mtime)
			{
			    *nmatches = (*nmatches) + 1;
			    if(match_cb != NULL)
				match_cb(
				    full_path,		/* Path */
				    &stat_buf,		/* Stats */
				    match_data		/* Data */
				);
			}
			break;

		      case EDV_FIND_LESS_THAN:
			if(t >= (gulong)stat_buf.st_mtime)
			{
			    *nmatches = (*nmatches) + 1;
			    if(match_cb != NULL)
				match_cb(
				    full_path,		/* Path */
				    &stat_buf,		/* Stats */
				    match_data		/* Data */
				);
			}
			break;

		      case EDV_FIND_GREATER_THAN:
			if(t <= (gulong)stat_buf.st_mtime)
			{
			    *nmatches = (*nmatches) + 1;
			    if(match_cb != NULL)
				match_cb(
				    full_path,		/* Path */
				    &stat_buf,		/* Stats */
				    match_data		/* Data */
				);
			}
			break;
		    }
		}

		/* Is it a directory and should we recurse into it? */
#ifdef S_ISDIR
		if(recursive && S_ISDIR(stat_buf.st_mode))
#else
		if(FALSE)
#endif
		{
		    EDVFindObjectByModifyTimeIterate(
			full_path,
			t,
			recursive,
			follow_links,
			relativity,
			progress_cb, progress_data,
			match_cb, match_data,
			nmatches,
			stop_find
		    );
		}

		FREE_CONTINUE
#undef FREE_CONTINUE
	    }

	    g_free(strv);
	}
}

/*
 *	Find an object by modify time.
 *
 *	The start_path specifies the full path to the location where
 *	the search is to be made.
 *
 *	The t specifies the modify time in seconds since EPOCH to find.
 *
 *	The relativity specifies the time relativity, one of
 *	EDV_FIND_*.
 *
 *	If progress_cb is not NULL then it will be called frequently to
 *	update progress. If progress_cb returns non-zero then the
 *	operation will end prematurly.
 *
 *	If match_cb is not NULL then each matched object will be reported
 *	to it.
 *
 *	Returns the number of occurances found.
 */
gint EDVFindObjectByModifyTime(
	const gchar *start_path,
	const gulong t,
	const gboolean recursive,
	const gboolean follow_links,
	const edv_find_relativity relativity,
	gint (*progress_cb)(const gchar *, const gfloat, gpointer),
	gpointer progress_data,
	void (*match_cb)(const gchar *, struct stat *, gpointer),
	gpointer match_data
)
{
	gboolean stop_find;
	gint nmatches;

	if(STRISEMPTY(start_path))
	    return(0);

	/* Begin the search */
	nmatches = 0;
	stop_find = FALSE;
	EDVFindObjectByModifyTimeIterate(
	    start_path,
	    t,
	    recursive,
	    follow_links,
	    relativity,
	    progress_cb, progress_data,
	    match_cb, match_data,
	    &nmatches,
	    &stop_find
	);

	return(nmatches);
}
