#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"
#include "pulist.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_date.h"
#include "edv_obj.h"
#include "edv_find_bar.h"
#include "endeavour2.h"
#include "edv_find.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_stop_20x20.xpm"


/* Callbacks */
static gint EDVFindBarProgressCB(
	const gchar *path, const gfloat progress,
	gpointer data
);
static void EDVFindBarMatchExcerptCB(
	const gchar *path, struct stat *lstat_buf,
	const gchar *excerpt, const gint line_index,
	gpointer data
);
static void EDVFindBarMatchCB(
	const gchar *path, struct stat *lstat_buf,
	gpointer data
);

static void EDVFindBarFindCB(GtkWidget *widget, gpointer data);
static void EDVFindBarStopCB(GtkWidget *widget, gpointer data);
static gint EDVFindBarCrossingCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

static void EDVFindBarFindByChangedCB(
        pulistbox_struct *pulistbox, gint i, gpointer data
);


/* Find Bar */
void EDVFindBarSetSearch(
	edv_find_bar_struct *fb,
	const gchar *s,
	const gboolean record_history
);
edv_find_bar_find_by EDVFindBarCurrentFindBy(edv_find_bar_struct *fb);

edv_find_bar_struct *EDVFindBarNew(
	edv_core_struct *core,
	GtkWidget *parent,
	const edv_location_type location_type,
	const gchar *(*get_location_cb)(
		edv_find_bar_struct *,
		gpointer
	),
	void (*start_cb)(
		edv_find_bar_struct *,
		gpointer
	),
	void (*end_cb)(
		edv_find_bar_struct *, const gint,
		gpointer
	),
	void (*match_cb)(
	    const gchar *, struct stat *,
	    const gchar *, const gint,
	    gpointer
	),
	void (*status_message_cb)(const gchar *, gpointer),
	void (*status_progress_cb)(const gfloat, gpointer),
	gpointer data
);
void EDVFindBarUpdateMenus(edv_find_bar_struct *fb);
void EDVFindBarMap(edv_find_bar_struct *fb);
void EDVFindBarUnmap(edv_find_bar_struct *fb);
void EDVFindBarDelete(edv_find_bar_struct *fb);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Progress callback, used for EDVFindObjectByName() and similar
 *	functions.
 */
static gint EDVFindBarProgressCB(
	const gchar *path, const gfloat progress,
	gpointer data
)
{
	edv_find_bar_struct *fb = EDV_FIND_BAR(data);
	if(fb == NULL)
	    return(0);

	if(fb->stop_count > 0)
	    return(1);

	if(fb->status_message_cb != NULL)
	    fb->status_message_cb(path, fb->data);
	if(fb->status_progress_cb != NULL)
	    fb->status_progress_cb(progress, fb->data);

	while(gtk_events_pending() > 0)
	    gtk_main_iteration();

	return(0);
}

/*
 *      Got match callback, used for EDVFindObjectByName() and similar
 *      functions.
 */
static void EDVFindBarMatchExcerptCB(
	const gchar *path, struct stat *lstat_buf,
	const gchar *excerpt, const gint line_index,
	gpointer data
)
{
	edv_find_bar_struct *fb = EDV_FIND_BAR(data);
	if(fb == NULL)
	    return;

	if(fb->match_cb != NULL)
	    fb->match_cb(
		path,			/* Path */
		lstat_buf,		/* Stats */
		excerpt,		/* Excerpt */
		line_index,		/* Line Index */
		fb->data		/* Data */
	    );

	while(gtk_events_pending() > 0)
	    gtk_main_iteration();
}

static void EDVFindBarMatchCB(
	const gchar *path, struct stat *lstat_buf,
	gpointer data
)
{
	EDVFindBarMatchExcerptCB(path, lstat_buf, NULL, -1, data);
}

/*
 *	Find "activate" signal callback.
 *
 *	This starts the search process.
 */
static void EDVFindBarFindCB(GtkWidget *widget, gpointer data)
{
	gboolean case_sensitive = FALSE;
	const gchar *s;
	gchar *location, *search_string;
	gint nmatches;
	GtkWidget *w;
	const cfg_item_struct *cfg_list;
	edv_find_bar_find_by operation;
	edv_core_struct *core;
	edv_find_bar_struct *fb = EDV_FIND_BAR(data);
	if(fb == NULL)
	    return;

	if(fb->freeze_count > 0)
	    return;

	core = fb->core;
	cfg_list = core->cfg_list;

	fb->freeze_count++;
	fb->processing = TRUE;

	/* Reset the find values */
	fb->stop_count = 0;
	EDVFindBarUpdateMenus(fb);

	/* Get the operation */
	operation = EDVFindBarCurrentFindBy(fb);

	/* Get/copy the location */
	s = NULL;
	switch(fb->location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	  case EDV_LOCATION_TYPE_ARCHIVE:
	    if(fb->get_location_cb != NULL)
		s = fb->get_location_cb(fb, fb->data);
	    break;
	  case EDV_LOCATION_TYPE_RECBIN:
	    s = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	    break;
	}
	location = STRDUP((s != NULL) ? s : "/");

	/* Get/copy the search string */
	s = NULL;
	w = fb->search_combo;
	if(w != NULL)
	{
	    GtkCombo *combo = GTK_COMBO(w);
	    GtkEntry *entry = GTK_ENTRY(combo->entry);
	    s = gtk_entry_get_text(entry);
	}
	search_string = g_strdup((s != NULL) ? s : "");
	if(search_string == NULL)
	{
	    g_free(location);
	    fb->processing = FALSE;
	    fb->freeze_count--;
	    return;
	}

	/* No wild cards in the search string and finding by name? */
	if((strchr((char *)search_string, '*') == NULL) &&
	   (strchr((char *)search_string, '?') == NULL) &&
	   (operation == EDV_FIND_BAR_FIND_BY_NAME)
	)
	{
	    /* Tack on two '*' characters to the beginning and end of
	     * the search string.
	     */
	    gchar *s = g_strdup_printf("*%s*", search_string);
	    g_free(search_string);
	    search_string = s;
	}
	/* Record the search string history */
	EDVFindBarSetSearch(fb, search_string, TRUE);

	/* Case sensitive */
	case_sensitive = GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    fb->case_sensitive_check
	);


	/* Notify about the start of the find */
	if(fb->start_cb != NULL)
	    fb->start_cb(fb, fb->data);


	/* Perform the search by the operation */
	nmatches = 0;
	switch(operation)
	{
	  case EDV_FIND_BAR_FIND_BY_NAME:
	    switch(fb->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectByName(
		    location,
		    search_string,
		    FALSE,			/* Do not recurse */
		    FALSE,			/* Do not follow links */
		    case_sensitive,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;

	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectByName(
		    location,			/* Recycled objects index file */
 		    search_string,
		    case_sensitive,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectByName(
		    core,
		    location,			/* Archive */
		    search_string,
		    case_sensitive,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	    }
	    break;

	  case EDV_FIND_BAR_FIND_BY_CONTENT:
	    switch(fb->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectByContent(
		    location,
		    search_string,
		    FALSE,			/* Do not recurse */
		    FALSE,			/* Do not follow links */
		    case_sensitive,		/* Case sensitive */
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchExcerptCB, fb
		);
		break;

	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectByContent(
		    location,			/* Recycled objects index file */
		    search_string,
		    case_sensitive,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchExcerptCB, fb
		);
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		/* Not supported */
		break;
	    }
	    break;

	  case EDV_FIND_BAR_FIND_BY_SIZE_EQUAL_TO:
	    switch(fb->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectBySize(
		    location,
		    ATOL(search_string),	/* Size */
		    FALSE,			/* Do not recurse */
		    FALSE,			/* Do not follow links */
		    EDV_FIND_EQUAL_TO,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectBySize(
		    location,			/* Recycled objects index file */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_EQUAL_TO,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectBySize(
		    core,
		    location,			/* Archive */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_EQUAL_TO,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	    }
	    break;

	  case EDV_FIND_BAR_FIND_BY_SIZE_NOT_EQUAL_TO:
	    switch(fb->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectBySize(
		    location,
		    ATOL(search_string),	/* Size */
		    FALSE,			/* Do not recurse */
		    FALSE,			/* Do not follow links */
		    EDV_FIND_NOT_EQUAL_TO,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectBySize(
		    location,			/* Recycled objects index file */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_NOT_EQUAL_TO,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectBySize(
		    core,
		    location,			/* Archive */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_NOT_EQUAL_TO,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	    }
	    break;

	  case EDV_FIND_BAR_FIND_BY_SIZE_LESS_THAN:
	    switch(fb->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectBySize(
		    location,
		    ATOL(search_string),	/* Size */
		    FALSE,			/* Do not recurse */
		    FALSE,			/* Do not follow links */
		    EDV_FIND_LESS_THAN,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectBySize(
		    location,			/* Recycled objects index file */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_LESS_THAN,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectBySize(
		    core,
		    location,			/* Archive */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_LESS_THAN,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	    }
	    break;

	  case EDV_FIND_BAR_FIND_BY_SIZE_GREATER_THAN:
	    switch(fb->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectBySize(
		    location,
		    ATOL(search_string),	/* Size */
		    FALSE,			/* Do not recurse */
		    FALSE,			/* Do not follow links */
		    EDV_FIND_GREATER_THAN,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectBySize(
		    location,			/* Recycled objects index file */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_GREATER_THAN,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectBySize(
		    core,
		    location,			/* Archive */
		    ATOL(search_string),	/* Size */
		    EDV_FIND_GREATER_THAN,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	    }
	    break;

	  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_EQUAL_TO:
	    switch(fb->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectByModifyTime(
		    location,
		    EDVDateParseEPOCH(search_string),	/* Time */
		    FALSE,			/* Do not recurse */
		    FALSE,			/* Do not follow links */
		    EDV_FIND_EQUAL_TO,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectByModifyTime(
		    location,			/* Recycled objects index file */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_EQUAL_TO,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectByModifyTime(
		    core,
		    location,			/* Archive */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_EQUAL_TO,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	    }
	    break;

	  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
	    switch(fb->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectByModifyTime(
		    location,
		    EDVDateParseEPOCH(search_string),	/* Time */
		    FALSE,			/* Do not recurse */
		    FALSE,			/* Do not follow links */
		    EDV_FIND_NOT_EQUAL_TO,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectByModifyTime(
		    location,			/* Recycled objects index file */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_NOT_EQUAL_TO,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectByModifyTime(
		    core,
		    location,			/* Archive */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_NOT_EQUAL_TO,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	    }
	    break;

	  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_LESS_THAN:
	    switch(fb->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectByModifyTime(
		    location,
		    EDVDateParseEPOCH(search_string),	/* Time */
		    FALSE,			/* Do not recurse */
		    FALSE,			/* Do not follow links */
		    EDV_FIND_LESS_THAN,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectByModifyTime(
		    location,			/* Recycled objects index file */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_LESS_THAN,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectByModifyTime(
		    core,
		    location,			/* Archive */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_LESS_THAN,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	    }
	    break;

	  case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_GREATER_THAN:
	    switch(fb->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		nmatches = EDVFindObjectByModifyTime(
		    location,
		    EDVDateParseEPOCH(search_string),	/* Time */
		    FALSE,			/* Do not recurse */
		    FALSE,			/* Do not follow links */
		    EDV_FIND_GREATER_THAN,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		nmatches = EDVFindRecycledObjectByModifyTime(
		    location,			/* Recycled objects index file */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_GREATER_THAN,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		nmatches = EDVFindArchiveObjectByModifyTime(
		    core,
		    location,			/* Archive */
		    EDVDateParseEPOCH(search_string),	/* Time */
		    EDV_FIND_GREATER_THAN,
		    EDVFindBarProgressCB, fb,
		    EDVFindBarMatchCB, fb
		);
		break;
	    }
	    break;
	}


	/* Tally */
	if(fb->status_progress_cb != NULL)
	    fb->status_progress_cb(0.0f, fb->data);

	if(nmatches > 0)
	{
	    gchar *msg = g_strdup_printf(
		"Matched %i %s",
		nmatches,
		(nmatches == 1) ? "object" : "objects"
	    );
	    if(fb->status_message_cb != NULL)
		fb->status_message_cb(
		    msg,
		    fb->data
		);
	    g_free(msg);
	}
	else
	{
	    if(fb->status_message_cb != NULL)
		fb->status_message_cb(
		    "No objects found",
		    fb->data
		);
	}

	fb->processing = FALSE;
	fb->freeze_count--;

	EDVFindBarUpdateMenus(fb);


	/* Call end find callback to notify about end of find */
	if(fb->end_cb != NULL)
	    fb->end_cb(fb, nmatches, fb->data);


	g_free(location);
	g_free(search_string);
}

/*
 *	Stop callback.
 */
static void EDVFindBarStopCB(GtkWidget *widget, gpointer data)
{
	edv_find_bar_struct *fb = EDV_FIND_BAR(data);
	if(fb == NULL)
	    return;

	fb->stop_count++;
}

/*
 *	Any GtkWidget "enter_notify_event" or "leave_notify_event"
 *	signal callback.
 */
static gint EDVFindBarCrossingCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	GtkWidget *w;
	const gchar *msg = NULL;
	edv_find_bar_struct *fb = EDV_FIND_BAR(data);
	if((widget == NULL) || (crossing == NULL) || (fb == NULL))
	    return(status);

	/* Get event type */
	etype = (gint)crossing->type;
	w = widget;

	/* Handle by event type */
	switch(etype)
	{
	  case GDK_ENTER_NOTIFY:
	    if(w == fb->case_sensitive_check)
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Especifica si los iguales de cuerda no deben ser el casos sensibles ni"
#elif defined(PROG_LANGUAGE_FRENCH)
"Spcifie si les allumettes de ficelle doivent tre des cas sensibles ou pas"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gibt an, wenn Schnur Gegenstcke Fall empfindlich oder nicht sein sollen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Non specifica se i fiammiferi di cordicella dovrebbero essere dei casi sensibili o"
#elif defined(PROG_LANGUAGE_DUTCH)
"Specificeert indien koord stellen geval gevoelig of niet zouden moeten zijn"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Especifica se partidas de barbante devem ser caso sensvel ou no"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Spesifiserer om snorkamper er tilfelle sensitiv eller ikke"
#else
"Specifies if string matches should be case sensitive or not"
#endif
		;
	    else if(w == fb->stop_btn)
		msg =
#if defined(PROG_LANGUAGE_SPANISH)
"Especifica si el proceso de la bsqueda debe recurse en subdirectorios"
#elif defined(PROG_LANGUAGE_FRENCH)
"Spcifie si le procd de recherche devrait recurse dans subdirectories"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gibt an, wenn das suche verfahren recurse in unterverzeichnisse soll"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Specifica se il processo di ricerca dovrebbe il recurse nelle directory secondarie"
#elif defined(PROG_LANGUAGE_DUTCH)
"Specificeert indien het zoektocht proces recurse in subdirectories zal"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Especifica se o processo de busca devia recurse em subdirectories"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Spesifiserer om letingssom prosessen skal recurse inn i subdirectories"
#else
"Specifies if the search process should recurse into subdirectories"
#endif
		;
	    break;
	}

	if(fb->status_message_cb != NULL)
	    fb->status_message_cb(msg, fb->data);

	return(status);
}


/*
 *	Find by method changed callback.
 */
static void EDVFindBarFindByChangedCB(
        pulistbox_struct *pulistbox, gint i, gpointer data
)
{
        edv_find_bar_struct *fb = EDV_FIND_BAR(data);
        if(fb == NULL)
            return;

        if(fb->freeze_count > 0)
            return;

	fb->freeze_count++;

        EDVFindBarUpdateMenus(fb);

	fb->freeze_count--;
}


/*
 *	Sets the search string.
 *
 *	The s specifies the new search string.
 *
 *	If record_history is TRUE then the current search string will
 *	be recorded on the search list history.
 */
void EDVFindBarSetSearch(
	edv_find_bar_struct *fb,
	const gchar *s,
	const gboolean record_history
)
{
	const gchar *cur_s;
	gchar *new_s;
	GtkCombo *combo;

	if((fb == NULL) || (s == NULL))
	    return;

	combo = GTK_COMBO(fb->search_combo);

	/* Copy the search string */
	new_s = STRDUP(s);
	if(new_s == NULL)
	    return;

	/* Check for no change in value */
	cur_s = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	if(cur_s != NULL)
	{
	    /* No change in value? */
	    if(!strcmp((const char *)cur_s, (const char *)new_s))
	    {
		g_free(new_s);
		return;
	    }
	}

	/* Get record the current search string on to the search
	 * history list?
	 */
	if(record_history)
	{
	    gchar *old_s = STRDUP(cur_s);
	    if(old_s != NULL)
	    {
		GUIComboAddItem(GTK_WIDGET(combo), old_s);
		g_free(old_s);
	    }
	}

	/* Set the new search string */
	gtk_entry_set_text(GTK_ENTRY(combo->entry), new_s);

	g_free(new_s);
}


/*
 *	Gets the current find by method.
 */
edv_find_bar_find_by EDVFindBarCurrentFindBy(edv_find_bar_struct *fb)
{
	pulist_struct *pulist;

	if(fb == NULL)
	    return(EDV_FIND_BAR_FIND_BY_NAME);

	pulist = PUListBoxGetPUList(fb->find_by_pulistbox);
	if(pulist == NULL)
	    return(EDV_FIND_BAR_FIND_BY_NAME);

	return((edv_find_bar_find_by)PUListGetItemData(
	    pulist,
	    PUListGetSelectedLast(pulist)
	));
}



/*
 *	Creates a new find bar.
 */
edv_find_bar_struct *EDVFindBarNew(
	edv_core_struct *core,
	GtkWidget *parent,
	const edv_location_type location_type,
	const gchar *(*get_location_cb)(
		edv_find_bar_struct *,
		gpointer
	),
	void (*start_cb)(
		edv_find_bar_struct *,
		gpointer
	),
	void (*end_cb)(
		edv_find_bar_struct *, const gint,
		gpointer
	),
	void (*match_cb)(
	    const gchar *, struct stat *,
	    const gchar *, const gint,
	    gpointer
	),
	void (*status_message_cb)(const gchar *, gpointer),
	void (*status_progress_cb)(const gfloat, gpointer),
	gpointer data
)
{
	const gint border_minor = 2;
	gint i;
	gpointer combo_rtn;
	GList *glist;
	GtkWidget *w, *parent2, *parent3, *toplevel;
	GtkCombo *combo;
	pulist_struct *pulist;
	pulistbox_struct *pulistbox;
	edv_find_bar_struct *fb;

	if(core == NULL)
	    return(NULL);

	fb = EDV_FIND_BAR(g_malloc0(sizeof(edv_find_bar_struct)));
	if(fb == NULL)
	    return(NULL);

	fb->toplevel = toplevel = gtk_hbox_new(FALSE, border_minor);
	fb->processing = FALSE;
	fb->freeze_count = 0;
	fb->stop_count = 0;
	fb->core = core;
	fb->location_type = location_type;
	fb->get_location_cb = get_location_cb;
	fb->start_cb = start_cb;
	fb->end_cb = end_cb;
	fb->match_cb = match_cb;
	fb->status_message_cb = status_message_cb;
	fb->status_progress_cb = status_progress_cb;
	fb->data = data;

	fb->freeze_count++;

	/* Toplevel GtkHBox */
	w = toplevel;
	gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	gtk_container_add(GTK_CONTAINER(parent), w);
	parent2 = w;

	/* Find Operation GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Buscar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dcouverte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fund"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trovare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vondst"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ache"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Funn"
#else
"Find"
#endif
	    ":"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	fb->find_by_pulistbox = pulistbox = PUListBoxNew(
	    parent3,
	    170, -1
	);
        PUListBoxSetChangedCB(
            pulistbox,
            EDVFindBarFindByChangedCB,
            fb
        );
	pulist = PUListBoxGetPUList(pulistbox);
	if(core->run_flags & EDV_RUN_SAFE_MODE)
	    PUListSetShadowStyle(pulist, PULIST_SHADOW_NONE);
	i = PUListAddItem(pulist, "Name");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_NAME);
	i = PUListAddItem(pulist, "Content");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_CONTENT);
	i = PUListAddItem(pulist, "Size Equal To");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_SIZE_EQUAL_TO);
	i = PUListAddItem(pulist, "Size Not Equal To");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_SIZE_NOT_EQUAL_TO);
	i = PUListAddItem(pulist, "Size Less Than");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_SIZE_LESS_THAN);
	i = PUListAddItem(pulist, "Size Greater Than");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_SIZE_GREATER_THAN);
	i = PUListAddItem(pulist, "Date Modified On");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_MODIFY_TIME_EQUAL_TO);
	i = PUListAddItem(pulist, "Date Modified Not On");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO);
	i = PUListAddItem(pulist, "Date Modified Before");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_MODIFY_TIME_LESS_THAN);
	i = PUListAddItem(pulist, "Date Modified After");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_BAR_FIND_BY_MODIFY_TIME_GREATER_THAN);
	PUListBoxSetLinesVisible(pulistbox, MIN((i + 1), 10));
	PUListBoxMap(pulistbox);

	/* Search string GtkCombo */
	glist = NULL;
	w = GUIComboCreate(
#if defined(PROG_LANGUAGE_SPANISH)
"Palabra"
#elif defined(PROG_LANGUAGE_FRENCH)
"Egaler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Anpassend"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Uguagliare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Passend"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Combinar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Passende"
#else
"Matching"
#endif
	    ":",
	    "",			/* No initial value */
	    glist,		/* List */
	    25,			/* Max items */
	    &combo_rtn,
	    fb,
	    EDVFindBarFindCB,
	    NULL
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)combo_rtn;
	fb->search_combo = w;
	combo = GTK_COMBO(w);
	gtk_combo_set_case_sensitive(combo, TRUE);

	w = combo->entry;
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);


	/* Case sensitive GtkCheckButton */
	fb->case_sensitive_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
"Sensible May/min."
#elif defined(PROG_LANGUAGE_FRENCH)
"Reconnatre Sensible"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fall Empfindlich"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Imballare Sensibile"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sluit Gevoelig In"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Caso Sensvel"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfelle Sensitiv"
#else
"Case Sensitive"
#endif
	);
	GTK_TOGGLE_BUTTON(w)->active = FALSE;
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindBarCrossingCB), fb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindBarCrossingCB), fb
	);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Verifique esto para discriminar Mayusculas y minusculas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vrifier ceci pour galer des ficelles reconnaissent sensiblement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Prfen sie dies, schnren fall empfindlich anzupassen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Controllare questo per uguagliare le cordicelle imballano sensibilmente"
#elif defined(PROG_LANGUAGE_DUTCH)
"Controleer dit om bij koorden geval gevoelig te passen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Verifique isto combinar caso de barbantes sensivelmente"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sjekk dette passe snortilfelle sensitivt"
#else
"Check this to match strings case sensitively"
#endif
	);
	gtk_widget_show(w);

	w = gtk_vseparator_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Stop button */
	fb->stop_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_stop_20x20_xpm
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVFindBarStopCB), fb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindBarCrossingCB), fb
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindBarCrossingCB), fb
	);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Pare el procedimiento actual del hallazgo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Arrter la procdure actuelle de dcouverte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Halten Sie das jetzige Fund Verfahren auf"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Fermare la corrente trova la procedura"
#elif defined(PROG_LANGUAGE_DUTCH)
"Stop de huidig vondst procedure"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Pare a corrente achar procedimento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stans den nvrendee funnprosedyre"
#else
"Stop the current find procedure"
#endif
	);
	gtk_widget_show(w);



	EDVFindBarUpdateMenus(fb);

	fb->freeze_count--;

	return(fb);
}


/*
 *	Updates the Find Bar's widgets to reflect current values.
 */
void EDVFindBarUpdateMenus(edv_find_bar_struct *fb)
{
	gboolean sensitive, processing;
	edv_find_bar_find_by find_by;
	edv_core_struct *core;

	if(fb == NULL)
	    return;

	processing = fb->processing;
	find_by = EDVFindBarCurrentFindBy(fb);
	core = fb->core;

	/* Find By */
	sensitive = !processing;
	gtk_widget_set_sensitive(
	    PUListBoxGetToplevel(fb->find_by_pulistbox),
	    sensitive
	);

	/* Search String */
	sensitive = !processing;
	gtk_widget_set_sensitive(fb->search_combo, sensitive);

	/* Case Sensitive */
	if(processing)
	{
	    sensitive = FALSE;
	}
	else
	{
            switch(find_by)
            {
              case EDV_FIND_BAR_FIND_BY_NAME:
              case EDV_FIND_BAR_FIND_BY_CONTENT:
                sensitive = TRUE;
                break;
              case EDV_FIND_BAR_FIND_BY_SIZE_EQUAL_TO:
              case EDV_FIND_BAR_FIND_BY_SIZE_NOT_EQUAL_TO:
              case EDV_FIND_BAR_FIND_BY_SIZE_LESS_THAN:
              case EDV_FIND_BAR_FIND_BY_SIZE_GREATER_THAN:
              case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_EQUAL_TO:
              case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
              case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_LESS_THAN:
              case EDV_FIND_BAR_FIND_BY_MODIFY_TIME_GREATER_THAN:
                sensitive = FALSE;
                break;
            }
	}
	gtk_widget_set_sensitive(fb->case_sensitive_check, sensitive);

	/* Stop */
	sensitive = processing;
	gtk_widget_set_sensitive(fb->stop_btn, sensitive);
}

/*
 *	Maps the Find Bar.
 */
void EDVFindBarMap(edv_find_bar_struct *fb)
{
	if(fb == NULL)
	    return;

	gtk_widget_show(fb->toplevel);
}

/*
 *	Unmaps the Find Bar.
 */
void EDVFindBarUnmap(edv_find_bar_struct *fb)
{
	if(fb == NULL)
	    return;

	gtk_widget_hide(fb->toplevel);
}

/*
 *	Deletes the Find Bar.
 */
void EDVFindBarDelete(edv_find_bar_struct *fb)
{
	if(fb == NULL)
	    return;

	EDVFindBarUnmap(fb);

	fb->freeze_count++;

	PUListBoxDelete(fb->find_by_pulistbox);
	gtk_widget_destroy(fb->toplevel);

	fb->freeze_count--;

	g_free(fb);
}
