#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include "../../include/string.h"



static void SARPrintHelp(void);

static char *SARReplaceBuf(
        char *buf,
        const char *pattern,
        const char *value,
	const int verbose,
        const int case_sensitive,
        int *nreplacements_rtn
);

static int SARReplaceFile(
        const char *path,
        const char *pattern,
        const char *value,
	const int verbose,
        const int case_sensitive
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : 1)


static void SARPrintHelp(void)
{
	puts(
"Usage: sar <pattern> <value> <file(s)...>"
	);





}


static char *SARReplaceBuf(
	char *buf,
	const char *pattern,
	const char *value,
	const int verbose,
	const int case_sensitive,
	int *nreplacements_rtn
)
{
	const int	pattern_len = strlen(pattern),
			value_len = strlen(value);
	char *s, *prev_s;
	int i = 0;

        s = (case_sensitive) ? 
            strstr(buf + i, pattern) : (char *)strcasestr(buf, pattern);
	while(s != NULL)
	{
	    i = (int)(s - buf);
	    buf = strdelchrs(buf, i, pattern_len);
	    buf = strinsstr(buf, i, value);

	    if(nreplacements_rtn != NULL)
		*nreplacements_rtn = (*nreplacements_rtn) + 1;

	    i += value_len;
	    prev_s = s;
	    s = (case_sensitive) ?
		strstr(buf + i, pattern) : (char *)strcasestr(buf, pattern);
	    if(prev_s == s)
		break;
	}

	return(buf);
}


static int SARReplaceFile(
	const char *path,
        const char *pattern,
        const char *value,
	const int verbose,
        const int case_sensitive
)
{
	int nreplacements;
	size_t len;
	struct stat stat_buf;
	char *buf;
	FILE *fp = fopen(path, "rb");
	if(fp == NULL)
	{
	    const int error_code = errno;
	    if(1)
		fprintf(
		    stderr,
"%s: %s.\n",
		    path, strerror(error_code)
		);
	    return(1);
	}

	if(fstat(fileno(fp), &stat_buf))
	{
	    len = 0;
	}
	else
	{
	    len = stat_buf.st_size;
	}

	buf = (char *)malloc((len + 1) * sizeof(char));
	if(buf == NULL)
	{
	    const int error_code = errno;
            if(1)
                fprintf(
                    stderr,
"%s: %s.\n",
                    path, strerror(error_code)
                );
	    fclose(fp);
	    return(3);
	}

	fread(buf, sizeof(char), len, fp);
	buf[len] = '\0';

	fclose(fp);


	nreplacements = 0;
	buf = SARReplaceBuf(
	    buf, pattern, value,
	    verbose, case_sensitive,
	    &nreplacements
	);
	if(buf == NULL)
	    return(1);

	/* No replacements? */
	if(nreplacements == 0)
	{
	    free(buf);
	    return(0);
	}

	/* Get the new length of the buffer after replacements*/
	len = strlen(buf);

        if(verbose)
            printf(
"%s\n",
                path
            );

	fp = fopen(path, "wb");
	if(fp == NULL)
	{
	    const int error_code = errno;
	    free(buf);
            if(1)
                fprintf(
                    stderr,
"%s: %s.\n",
                    path, strerror(error_code)
                );
	    return(1);
	}

	fwrite(buf, sizeof(char), len, fp);

	free(buf);

	fclose(fp);

	return(0);
}


int main(int argc, char *argv[])
{
	int	verbose = 1,
		case_sensitive = 1;
	const char	*pattern = NULL,
			*value = NULL;
	int i, str_arg_num, first_file_arg_num, status;
	const char *arg;


	first_file_arg_num = 0;
	str_arg_num = 0;
	for(i = 1; i < argc; i++)
	{
	    arg = argv[i];
	    if(arg == NULL)
		continue;

	    if(!strcasecmp(arg, "--help") ||
               !strcasecmp(arg, "-help") ||
               !strcasecmp(arg, "--h") ||
               !strcasecmp(arg, "-h") ||
               !strcasecmp(arg, "-?")
	    )
	    {
		SARPrintHelp();
                return(0);
	    }
	    else if((*arg != '-') && (*arg != '+'))
	    {
		switch(str_arg_num)
		{
		  case 0:
		    pattern = arg;
		    break;
		  case 1:
		    value = arg;
		    break;
		  default:
		    if(first_file_arg_num == 0)
			first_file_arg_num = i;
		    break;
		}
		str_arg_num++;
	    }
	}

	if((pattern == NULL) || (value == NULL))
	{
	    SARPrintHelp();
	    return(0);
	}

	status = 0;
	for(i = first_file_arg_num; i < argc; i++)
	{
	    arg = argv[i];
	    if(arg == NULL)
		continue;

	    if((*arg != '-') && (*arg != '+'))
	    {
		const char *path = arg;

		status = SARReplaceFile(
		    path, pattern, value, verbose, case_sensitive
		);
		if(status)
		    break;
	    }
	}

	return(status);
}
