/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: world.cc,v 1.72.2.13 2003/10/11 16:45:37 dheck Exp $
 */
#include "world.hh"
#include "objects.hh"
#include "actors.hh"
#include "items.hh"
#include "laser.hh"
#include "stones.hh"
#include "display.hh"
#include "player.hh"
#include "sound.hh"
#include "px/px.hh"
#include "px/dict.hh"
#include "lua.hh"
#include "options.hh"
#include "game.hh"

#include <iostream>
#include <algorithm>
#include <functional>
#include <numeric>
#include <cassert>

// remove comment from define below to switch on verbose messaging
// note: VERBOSE_MESSAGES is defined in multiple source files!
// #define VERBOSE_MESSAGES

using namespace std;
using namespace world;
using namespace px;

#include "world_internal.hh"


Object *SimpleSignal::get_target() 
{
    return GetObject(destloc);
}


void SimpleSignal::emit_from (Object *src, int value) 
{
    if (source == src) {
        Object *dst = get_target();

#if defined(VERBOSE_MESSAGES)
        src->warning("emit_from: msg='%s' dest=%i/%i obj=%p",
                     message.c_str(),
                     destloc.pos.x, destloc.pos.y,
                     dst);
#endif
        SendMessage (dst, message, value);
    }
#if defined(VERBOSE_MESSAGES)
    else {
        fprintf(stderr,
                "emit_from source object does not match\n" 
                "signal '%s' dropped (expected=%p=%s found=%p=%s)\n",
                message.c_str(), source, source->get_kind(), src, src->get_kind());
    }
#endif
}

void SignalList::emit_from (Object *source, int value) 
{
    // signals may have side effects. To minimize them
    //   1. collect all targets and then
    //   2. emit signals to targets

    size_t          size = m_signals.size();
    vector<Object*> targets;
    targets.resize(size);

    for (unsigned i=0; i<size; ++i) {
        targets[i] = m_signals[i]->get_target();
    }

    for (unsigned i=0; i<size; ++i) {
        if (m_signals[i]->get_target() == targets[i])
            m_signals[i]->emit_from (source, value);
    }
}

Object *SignalList::find_single_destination(Object *src) 
{
    Object *found = 0;
    size_t size = m_signals.size();

    for (unsigned i = 0; i<size; ++i) {
        if (m_signals[i]->get_source() == src) {
            Object *candidate = m_signals[i]->get_target();
            if (candidate) {
                if (!found)
                    found = candidate;
                else if (candidate != found) {
                    found = 0; // multiple targets
                    break;
                }
            }
        }
    }

    return found;
}

//----------------------------------------
// Rubber band
//----------------------------------------

RubberBand::RubberBand (Actor *a1, Actor *a2, double strength_, double length_)
: actor(a1), actor2(a2), stone(0),
  model(display::AddRubber(get_p1(),get_p2())),
  strength(strength_), length(length_)
{
    assert(actor);
    assert(length >= 0);
}

RubberBand::RubberBand (Actor *a1, Stone *st, double strength_, double length_)
: actor(a1), actor2(0), stone(st), model(0),
  strength(strength_), length(length_)
{
    assert(actor);
    assert(length >= 0);
    model = display::AddRubber(get_p1(), get_p2());
}

RubberBand::~RubberBand() {
    model.kill();
}

void RubberBand::tick(double)
{
    V2 v = get_p2()-get_p1();
    double vv = px::length(v);

    if (vv > length) {
        V2 force = v * strength*(vv-length)/vv;
        V2 f(force[0],force[1]);
        actor->add_force(f);
        if (actor2)
            actor2->add_force(-f);
    }

    model.update_first (get_p1());
    model.update_second (get_p2());
}

V2 RubberBand::get_force(Actor */*a*/)
{
    return V2();
}

V2 RubberBand::get_p1() const
{
    return V2(actor->get_pos()[0], actor->get_pos()[1]);
}

V2 RubberBand::get_p2() const
{
    if (!stone)
        return V2(actor2->get_pos()[0], actor2->get_pos()[1]);
    else
        return stone->get_pos().center();
}

//----------------------------------------
// Field
//----------------------------------------

Field::Field()
{
    floor=0;
    item=0;
    stone=0;
}

Field::~Field()
{
    DisposeObject(floor);
    DisposeObject(item);
    DisposeObject(stone);
}


//----------------------------------------
// StoneContact
//----------------------------------------

StoneContact::StoneContact(Actor *a, GridPos p,
                           const V2 &cp, const V2 &n)
: actor(a), stonepos(p),
  response(STONE_PASS),
  contact_point(cp),
  normal(n),
  is_collision(false),
  fake_collision(false),
  new_collision(false),
  is_contact(true)
{}

StoneContact::StoneContact()
: is_collision(false),
  fake_collision(false),
  new_collision(false),
  is_contact(false)
{}

DirectionBits
world::contact_faces(const StoneContact &sc)
{
    using namespace enigma;

    int dirs = NODIRBIT;

    if (sc.normal[0] < 0)
        dirs |= WESTBIT;
    else if (sc.normal[0] > 0)
        dirs |= EASTBIT;
    if (sc.normal[1] < 0)
        dirs |= NORTHBIT;
    else if (sc.normal[1] > 0)
        dirs |= SOUTHBIT;

    return DirectionBits(dirs);
}

Direction
world::contact_face(const StoneContact &sc)
{
    using namespace enigma;
    if (sc.normal == V2(-1,0))
        return WEST;
    else if (sc.normal == V2(1,0))
        return EAST;
    else if (sc.normal == V2(0,-1))
        return NORTH;
    else if (sc.normal == V2(0,1))
        return SOUTH;
    else
        return NODIR;
}


WorldImpl::WorldImpl(int ww, int hh) : fields(ww,hh)
{
    w = ww;
    h = hh;

    scrambleIntensity = options::Difficulty == DIFFICULTY_EASY ? 3 : 10;
}

WorldImpl::~WorldImpl()
{
    fields = FieldArray(0,0);
    for_each(actorlist.begin(), actorlist.end(), mem_fun(&Actor::dispose));
    delete_sequence (rubbers.begin(), rubbers.end());
}

void WorldImpl::remove (ForceField *ff)
{
    ForceList::iterator i=find (forces.begin(), forces.end(), ff);
    if (i != forces.end())
        forces.erase(i);
}

Object *WorldImpl::get_named (const string &name)
{
    px::Dict<Object*>::iterator found = m_objnames.find(name);
    return (found != m_objnames.end()) ? found->second : 0;
}

void WorldImpl::name_object (Object *obj, const std::string &name)
{
    m_objnames.insert(name, obj); // [name] = obj;
    obj->set_attrib("name", name);
}

void WorldImpl::unname (Object *obj)
{
    assert(obj);
    string name;
    if (obj->string_attrib("name", &name)) {
        m_objnames.remove(name);
        obj->set_attrib("name", "");
    }
}

void WorldImpl::tick (double dtime)
{
    mouseforce.tick (dtime);
    for_each (forces.begin(), forces.end(),
              bind2nd(mem_fun(&ForceField::tick), dtime));

    for_each (rubbers.begin(), rubbers.end(),
              bind2nd(mem_fun(&RubberBand::tick), dtime));
}

enigma::Timer world::g_timer;

//======================================================================
//      PUZZLE SCRAMBLING
//======================================================================

void WorldImpl::add_scramble(GridPos p, Direction dir)
{
    // fprintf(stderr, "scrambles.size()=%i ; add_scramble(%i, %i, %s)\n", 
    //         scrambles.size(), p.x, p.y, to_suffix(dir).c_str());
    scrambles.push_back(Scramble(p, dir, scrambleIntensity));
}

void WorldImpl::scramble_puzzles()
{
    // fprintf(stderr, "scramble_puzzles: scrambles.size()=%i\n", scrambles.size());
    while (!scrambles.empty()) {
        list<Scramble>::iterator i = scrambles.begin();
        list<Scramble>::iterator e = scrambles.end();

        for (; i != e; ++i) {
            Stone *puzz = GetStone(i->pos);
            if (puzz && i->intensity) {
                // fprintf(stderr, "Sending 'scramble' to stone at pos %i/%i\n", i->pos.x, i->pos.y);
                SendMessage(puzz, "scramble", Value(double(i->dir)));
                --i->intensity;
            }
            else {
                fprintf(stderr, "no stone found for scramble at %i/%i\n", i->pos.x, i->pos.y);
                i->intensity = 0;
            }
        }

        scrambles.remove_if(mem_fun_ref(&Scramble::expired));
    }
}


//======================================================================
// VARIABLES
//======================================================================

namespace
{
    auto_ptr<WorldImpl> level;
}



//======================================================================
// PHYSICS SIMULATION
//======================================================================

/* Calculate the total acceleration on an actor A at time TIME.  The
   actor's current position X and velocity V are also passed.  [Note
   that the position and velocity entries in ActorInfo will be updated
   only after a *successful* time step, so they cannot be used
   here.] */
static V2
get_accel (Actor *a, const V2 & x, const V2 & v, double time)
{
    V2 f;
    GridPos p((int)x[0], (int)x[1]);

    if (a->is_on_floor()) {
        if (Floor *floor = GetFloor(p)) {
            double friction;

            // Mouse force
            if (a->has_spikes()) {
                f += level->mouseforce.get_force(a) * 1.0;
                friction = 1.0;
            }
            else {
                f += floor->process_mouseforce(a, level->mouseforce.get_force(a));
                friction = floor->friction();
            }

            // Friction
            double vv=length(v);
            if (vv > 0) {
                V2 frictionf = v * (enigma::FrictionFactor*friction);
                frictionf /= vv;
                frictionf *= pow(vv, 0.8);
                f -= frictionf;
            }

            f += floor->get_force(a);
        }

        if (Item *item = GetItem(p)) {
            f += item->get_force(a);
        }
    }

    // Electrostatic forces between actors.
    if (double q = a->get_charge()) {
        for (ActorList::iterator i=level->actorlist.begin();
             i != level->actorlist.end(); ++i)
        {
            Actor *a2 = *i;
            if (a2 == a) continue;
            if (double q2 = a2->get_charge()) {
                V2 distv = a->get_pos() - a2->get_pos();
                if (double dist = distv.normalize())
//                    f += enigma::ElectricForce * q * q2 / (dist*dist) * distv;
                    f += enigma::ElectricForce * q * q2 / (dist) * distv;
            }
        }
    }

    // All other force fields.
    for (ForceList::iterator i=level->forces.begin();
         i != level->forces.end(); ++i)
    {
        f += (*i)->get_force(a, x, v, time);
    }

    ActorInfo *ai = a->get_actorinfo();
    f += ai->force;

    return f / ai->mass;
}


/* This function performs one step in the numerical integration of an
   actor's equation of motion.  TIME ist the current absolute time and
   H the size of the integration step. */
static void
advance_actor(Actor *a, double time, double h)
{
    const double MAXVEL = 70;

    ActorInfo &ai = *a->get_actorinfo();
    V2 accel = get_accel(a, ai.pos, ai.vel, time);

    // If the actor is currently in contact with other objects, remove
    // the force components in the direction of these objects.
    for (unsigned i=0; i<ai.contact_normals.size(); ++i)
    {
        const V2 &normal = ai.contact_normals[i];
        double normal_component = normal * accel;
        if (normal_component < 0)
            accel -= normal_component * normal;
    }


    ai.vel += h*accel;

    // If the actor is currently in contact with other objects, remove
    // the force components in the direction of these objects.
//     for (unsigned i=0; i<ai.contact_normals.size(); ++i)
//     {
//         const V2 &normal = ai.contact_normals[i];
//         double normal_component = normal * ai.vel;
//         if (normal_component < 0)
//             ai.vel -= normal_component * normal;
//     }

    ai.pos += h*ai.vel;

    // Limit maximum velocity
    double q = length(ai.vel) / MAXVEL;
    if (q > 1)
        ai.vel /= q;
}

/* If the actor and the stone are not in contact, `contact' is filled
   with information about the closest feature on the stone and
   `is_contact' is set to false. 
*/
static void
find_contact_with_stone (Actor *a, StoneContact &contact, int x, int y)
{
    contact.actor      = a;
    contact.stonepos   = GridPos(x, y);
    contact.is_contact = false;

    Stone *stone = world::GetStone(contact.stonepos);
    if (!stone)
        return;

    const ActorInfo &ai = *a->get_actorinfo();
    double r = ai.radius;
    double ax = ai.pos[0];
    double ay = ai.pos[1];
    const double contact_e = 0.01;
    const double erad = 2.0/32; // edge radius

    // Closest feature == north or south face of the stone?
    if (ax>x+erad && ax<x+1-erad) {
        double dist = r+5;

        // south
        if (ay>y+1) {
            contact.contact_point = V2(ax, y+1);
            contact.normal        = V2(0,+1);
            dist                  = ay-(y+1);
        }
        // north
        else if (ay<y) {
            contact.contact_point = V2(ax, y);
            contact.normal        = V2(0,-1);
            dist                  = y-ay;
        }
	contact.is_contact = (dist-r < contact_e);
    }
    // Closest feature == west or east face of the stone?
    else if (ay>y+erad && ay<y+1-erad) {
        double dist=r+5;
        if (ax>x+1) { // east
            contact.contact_point = V2(x+1,ay);
            contact.normal        = V2(+1,0);
            dist                  = ax-(x+1);
        }
        else if (ax<x) { // west
            contact.contact_point = V2(x,ay);
            contact.normal        = V2(-1,0);
            dist                  = x-ax;
        }
	contact.is_contact = (dist-r < contact_e);
    }
    // Closest feature == any of the four corners
    else {
        int xcorner=(ax >= x+1-erad);
        int ycorner=(ay >= y+1-erad);
        double cx[2] = {erad, -erad};

        V2 corner(x+xcorner+cx[xcorner], y+ycorner+cx[ycorner]);
        V2 b=V2(ax,ay) - corner;

        contact.is_contact    = (length(b)-r-erad < contact_e);
        contact.normal        = normalize(b);
        contact.contact_point = corner + contact.normal*erad;
    }

    contact.is_collision  = contact.normal*ai.vel < 0;
    contact.response = stone->collision_response(contact);
    contact.sound = stone->collision_sound();
}

/* Enigma's stones have rounded corners; this leads to realistic
   behaviour when a marble rebounds from the corner of a single stone.
   But it's annoying when there are two adjacent stones and a marble
   rebounds from any one (or even both) corners, because it changes
   the direction of the marble quite unpredictably.

   This function modifies the contact information for two adjacent
   stones in such a way that the two collisions are treated as a
   single collision with a flat surface.
 */
static void
maybe_join_contacts (StoneContact & a, StoneContact &b)
{
//    double maxd = 4.0/32;   // Max distance between joinable collisions

    if ((a.is_contact || b.is_contact)  &&
	a.is_collision && b.is_collision &&
        a.response==STONE_REBOUND && b.response==STONE_REBOUND) // &&
//        length(a.contact_point - b.contact_point) <= maxd)
    {
        if (a.is_contact && b.is_contact)
            b.fake_collision = true; // Don't rebound from `b'

        DirectionBits fa = contact_faces(a);
        DirectionBits fb = contact_faces(b);

        StoneContact &c = (a.is_contact) ? a : b;
        switch (fa & fb) {
        case NORTHBIT: c.normal = V2(0,-1); break;
        case EASTBIT: c.normal = V2(1,0); break;
        case SOUTHBIT: c.normal = V2(0,1); break;
        case WESTBIT: c.normal = V2(-1,0); break;
        case NODIRBIT:
            //fprintf(stderr, "Strange: contacts have no direction in common\n");
            break;
        default:
            //fprintf(stderr, "Strange: contacts have multiple directions in common\n");
            break;
        }
    }
}

static void
find_stone_contacts (Actor *a, StoneContactList &cl)
{
    ActorInfo &ai = *a->get_actorinfo();
    px::Rect r(int(ai.pos[0]-0.5), int(ai.pos[1]-0.5),
	       int(ai.pos[0]+0.5), int(ai.pos[1]+0.5));
    r.w -= r.x;
    r.h -= r.y;

    StoneContact contacts[2][2];

    // Test for collisions with the nearby stones
    for (int i=0; i<=r.w; i++)
        for (int j=0; j<=r.h; j++)
            find_contact_with_stone(a, contacts[i][j], r.x+i, r.y+j);

    maybe_join_contacts (contacts[0][0], contacts[1][0]);
    maybe_join_contacts (contacts[0][0], contacts[0][1]);
    maybe_join_contacts (contacts[1][0], contacts[1][1]);
    maybe_join_contacts (contacts[0][1], contacts[1][1]);

    for (int i=0; i<=r.w; i++)
        for (int j=0; j<=r.h; j++)
            if (contacts[i][j].is_contact)
                cl.push_back(contacts[i][j]);
}


/* This function is called for every contact between an actor and a
   stone.  It is here that the stones are informed about contacts.
   Note that a stone may kill itself during a call to actor_hit() or
   actor_contact() so we must not refer to it after having called
   these functions.  (Incidentally, this is why StoneContact only
   contains a 'stonepos' and no 'stone' entry.)
*/
static void
handle_stone_contact (StoneContact &sc, double total_dtime)
{
    Actor     *a           = sc.actor;
    ActorInfo &ai          = *a->get_actorinfo();
    double     restitution = 1.0; //0.95;

    if (sc.is_collision) {
        if (!sc.fake_collision && sc.response == STONE_REBOUND) {
            if (length(ai.vel) > 0.1) {
		if (Stone *stone = world::GetStone(sc.stonepos))
		    stone->actor_hit(sc);

                display::AddEffect (sc.contact_point, "ring-anim");
                sound::PlaySound(sc.sound.c_str(), sc.contact_point);
            }
            ai.vel -= (1 + restitution)*(ai.vel*sc.normal)*sc.normal;
            ai.contact_normals.push_back (sc.normal);
        }
    }
    else if (sc.is_contact) {
        if (Stone *stone = world::GetStone(sc.stonepos))
            stone->actor_contact(sc.actor);
    }
}

static void
handle_contacts(Actor *actor1, double total_dtime)
{
    ActorInfo &a1 = *actor1->get_actorinfo();

    // List of current contacts is updated in this function
    a1.contact_normals.clear();

    /*
     * Handle contacts with stones
     */
    StoneContactList cl;
    find_stone_contacts(actor1, cl);
    for (StoneContactList::iterator i=cl.begin();
         i != cl.end(); ++i)
    {
        handle_stone_contact (*i, total_dtime);
    }

    /*
     * Handle contacts with other actors
     */
    V2     p1 = a1.pos;
    double m1 = a1.mass;

    unsigned s=level->actorlist.size();
    for (unsigned j=0; j<s; ++j)
    {
	Actor     *actor2 = level->actorlist[j];
        ActorInfo &a2     = *actor2->get_actorinfo();

        if (actor2 == actor1 || a2.grabbed)
	    continue;

        V2     p2      = a2.pos;
        double overlap = a1.radius + a2.radius - length(p1-p2); 
        if (overlap > 0)
        {
            V2 n  = normalize(p1-p2);	// normal to contact surface
            V2 v1 = n * (a1.vel*n);
            V2 v2 = n * (a2.vel*n);

            bool collisionp = (a1.vel-a2.vel)*n < 0;
            if (collisionp) 
            {
                a1.pos = a1.oldpos;		// Reset actor position

		actor1->on_hit(actor2);
		actor2->on_hit(actor1);

                bool reboundp = (actor1->is_movable() && actor2->is_movable() && 
                                 (actor1->is_on_floor() == actor2->is_on_floor()));

                if (reboundp) {
                    double m2 = a2.mass;
                    double M = m1+m2;

                    double restitution = 0.95;

		    a1.vel += -v1 + (v2*(2*m2) + v1*(m1-m2)) / M;
		    a2.vel += -v2 + (v1*(2*m1) + v2*(m2-m1)) / M;
		    
		    a1.vel *= restitution;
		    a2.vel *= restitution;

		    sound::PlaySound("ballcollision");

                    a1.contact_normals.push_back (-n);
		}
            }
        }
    }
}

static void
tick_actor(Actor *a, double dtime)
{
    ActorInfo &ai = * a->get_actorinfo();

    ai.last_pos = ai.pos;

    double rest_time = dtime;
    double dt = 0.0025; // Size of one time step -- do not change!

//    handle_contacts(a, 0.0);
    while (rest_time > 0 && !ai.grabbed)
    {
        advance_actor(a, dtime-rest_time, dt);
        handle_contacts(a, dtime-rest_time);
        a->move();
        rest_time -= dt;
    }
}


//----------------------------------------
// Functions
//----------------------------------------

void world::Reset ()
{
    /* Restart the Lua environment so symbol definitions from
       different levels do not get in each other's way.*/
    lua::Shutdown();
    lua::Init();
    lua::Dofile("init.lua");

    enigma::GameReset();
    player::NewWorld();

    g_timer.clear();
}

void world::Create (int w, int h)
{
    level.reset (new WorldImpl(w,h));
    display::NewWorld(w, h);
}

void world::Shutdown()
{
    level.reset();
    Repos_Shutdown();
    lua::Shutdown();
}

bool world::InitWorld()
{
    level->scramble_puzzles();

    laser::RecalcLight();
    laser::RecalcLightNow();    // recalculate laser beams if necessary

    bool seen_player0 = false;

    for (ActorList::iterator i=level->actorlist.begin();
         i != level->actorlist.end(); ++i)
    {
        Actor *a = *i;
        a->on_creation(a->get_actorinfo()->pos);

        int iplayer;
        if (a->int_attrib("player", &iplayer)) {
            player::AddActor(iplayer,a);
            if (iplayer == 0) seen_player0 = true;
        }
    }

    if (!seen_player0) {
        fprintf(stderr, "Error: No player 0 defined!\n");
        return false;
    }

    world::BroadcastMessage("init", Value(),
                            // currently only EasyModeItem listens to "init" :
                            static_cast<GridLayerBits>(GRID_ITEMS_BIT));

    player::InitMoveCounter();
    display::StatusBar *sb = display::GetStatusBar();
    sb->show_move_counter (enigma::ShowMoves);

    return true;
}

void world::SetMouseForce(V2 f) 
{
    level->mouseforce.add_force(f);
}

void world::NameObject(Object *obj, const std::string &name) 
{
    string old_name;
    if (obj->string_attrib("name", &old_name)) {
        obj->warning("name '%s' overwritten by '%s'",
                     old_name.c_str(), name.c_str());
        UnnameObject(obj);
    }
    level->name_object (obj, name);
}

void world::UnnameObject(Object *obj) 
{
    level->unname(obj);
}

void world::TransferObjectName (Object *source, Object *target) 
{
    string name;
    if (source->string_attrib("name", &name)) {
        UnnameObject(source);
        string targetName;
        if (target->string_attrib("name", &targetName)) {
            target->warning("name '%s' overwritten by '%s'",
                            targetName.c_str(), name.c_str());
            UnnameObject(target);
        }
        NameObject(target, name);
    }
}

Object * world::GetNamedObject (const std::string &name) 
{
    return level->get_named (name);
}

bool world::IsLevelBorder(GridPos p) 
{
    return level->is_border(p);
}

//----------------------------------------
// Force fields
//----------------------------------------

void world::AddForceField(ForceField *ff)
{
    level->forces.push_back(ff);
}

void world::RemoveForceField(ForceField *ff) {
    level->remove (ff);
}

//----------------------------------------
// Rubber bands
//----------------------------------------

void world::AddRubberBand (Actor *a, Stone *st, double strength,double length)
{
    level->rubbers.push_back(new RubberBand (a, st, strength, length));
}

void world::AddRubberBand (Actor *a, Actor *a2, double strength,double length)
{
    level->rubbers.push_back(new RubberBand (a, a2, strength, length));
}

void world::KillRubberBand (Actor *a, Stone *st)
{
    assert(a);
    for (unsigned i=0; i<level->rubbers.size(); ) {
        RubberBand &r = *level->rubbers[i];
        if (r.get_actor() == a && r.get_stone() != 0)
            if (r.get_stone()==st || st==0) {
                delete &r;
                level->rubbers.erase(level->rubbers.begin()+i);
                continue;       // don't increment i
            }
        ++i;
    }
}

void world::KillRubberBand (Actor *a, Actor *a2)
{
    assert (a);
    for (unsigned i=0; i<level->rubbers.size(); ) {
        RubberBand &r = *level->rubbers[i];
        if (r.get_actor() == a && r.get_actor2() != 0)
            if (r.get_actor2()==a2 || a2==0) {
                delete &r;
                level->rubbers.erase(level->rubbers.begin()+i);
                continue;       // don't increment i
            }
        ++i;
    }
}

void world::KillRubberBands (Stone *st)
{
   for (unsigned i=0; i<level->rubbers.size(); ) {
        RubberBand &r = *level->rubbers[i];
        if (r.get_stone() != 0 && r.get_stone()==st) {
            delete &r;
            level->rubbers.erase(level->rubbers.begin()+i);
            continue;       // don't increment i
        }
        ++i;
    }
}

void
world::GiveRubberBands (Stone *st, vector<Rubber_Band_Info> &rubs)
{
   for (unsigned i=0; i<level->rubbers.size(); ) {
        RubberBand &r = *level->rubbers[i];
        if (r.get_stone() == st) {
	    Rubber_Band_Info rbi;
	    rbi.act = r.get_actor();
	    rbi.strength = r.get_strength();
	    rbi.length = r.get_length();
	    rubs.push_back(rbi);
	}
        ++i;
    }
}

bool world::HasRubberBand (Actor *a, Stone *st)
{
    for (unsigned i=0; i<level->rubbers.size(); ++i) {
        RubberBand &r = *level->rubbers[i];
        if (r.get_actor() == a && r.get_stone() == st)
            return true;
    }
    return false;
}

//----------------------------------------
// Signals
//----------------------------------------

void world::AddSignal (GridLoc srcloc, GridLoc dstloc, const string &msg)
{
#if defined(VERBOSE_MESSAGES)
    fprintf(stderr, "AddSignal src=%i/%i dest=%i/%i msg='%s'\n",
            srcloc.pos.x, srcloc.pos.y, dstloc.pos.x, dstloc.pos.y, msg.c_str());
#endif // VERBOSE_MESSAGES

    if (Object *src = GetObject(srcloc)) {
        src->set_attrib("action", "signal");
        level->signals.add (new SimpleSignal (src, dstloc, msg));
    }
    else {
        Log << "AddSignal: Invalid signal source\n";
    }
}


void world::EmitSignals (Object *src, int value)
{
    level->signals.emit_from (src, value);
}

/* Searches all signals for a signal starting at 'src' and returns the
   target object.  If multiple signals start from 'src' this function
   returns 0.
*/
Object *world::FindSignalDestination(Object *src)
{
    return level->signals.find_single_destination(src);
}

namespace
{
//----------------------------------------
// Layer
//
// Changes to a layer are cached and only applied at the end of a
// tick. [### not implemented]
//----------------------------------------
    template <class T>
    class Layer {
        T *defaultval;
    public:
        Layer(T* deflt = 0) : defaultval(deflt) {}
        virtual ~Layer() {}

        T *get(GridPos p) {
            if (Field *f=level->get_field(p))
                return raw_get(*f);
            else
                return defaultval;
        }

        T *yield(GridPos p) {
            if (Field *f=level->get_field(p)) {
                T *x = raw_get(*f);
                if (x) {
                    raw_set(*f, 0);
                    x->removal();
                }
                return x;
            } else
                return defaultval;
        }

        void set(GridPos p, T *x) {
            if (x) {
                if (Field *f=level->get_field(p)) {
                    dispose(raw_get(*f));
                    raw_set(*f, x);
                    x->creation(p);
                }
                else
                    dispose(x);
            }
        }
        void kill(GridPos p) { dispose(yield(p)); }

        virtual T* raw_get (Field &) = 0;
        virtual void raw_set (Field &, T *) = 0;

    private:
        virtual void dispose(T *x) { if (x) DisposeObject(x); }
    };

    /*
    ** Floor layer
    */
    class FloorLayer : public Layer<Floor> {
    private:
        Floor *raw_get (Field &f) { return f.floor; }
        void raw_set (Field &f, Floor *x) { f.floor = x;}
    };


    /*
    ** Item layer
    */
    class ItemLayer : public Layer<Item> {
    private:
        Item *raw_get (Field &f) { return f.item; }
        void raw_set (Field &f, Item *x) { f.item = x;}
    };

    /*
    ** Stone layer
    */
    class StoneLayer : public Layer<Stone>
    {
    public:
        StoneLayer() : Layer<Stone>(&borderstone) {}
    private:
        Stone *raw_get (Field &f) { return f.stone; }
        void raw_set (Field &f, Stone *st) { f.stone = st;}
        void dispose (Stone *st) {
            if (st) {
                KillRubberBands(st);
                DisposeObject(st);
            }
        }

        /* This stone is used as the virtual border of the playing area.
           It is immovable and indestructible and makes sure the player's
           marble cannot leave the level. */
        class BorderStone : public Stone {
        public:
            BorderStone() : Stone("borderstone") {}
            Object *clone() { return this; }
            void dispose() {}
        };

        BorderStone borderstone;
    };

}

namespace
{
    FloorLayer      fl_layer;
    ItemLayer       it_layer;
    StoneLayer      st_layer;
    vector<GridPos> changed_stones;

    // delayed impulses :

    ImpulseList delayed_impulses;
}

Object *world::GetObject (const GridLoc &l)
{
    switch (l.layer) {
    case GRID_FLOOR:  return GetFloor(l.pos);
    case GRID_ITEMS:  return GetItem(l.pos);
    case GRID_STONES: return GetStone(l.pos);
    default: return 0;
    }
}

//----------------------------------------
// Floor manipulation.
//----------------------------------------

void world::KillFloor(GridPos p) {fl_layer.kill(p);}
Floor *world::GetFloor(GridPos p) {return fl_layer.get(p);}
void world::SetFloor(GridPos p, Floor* fl) {
    fl_layer.set(p,fl);
    Stone *st = GetStone(p);
    if (st)
        st->floor_change();
}

//----------------------------------------
// Stone manipulation.
//----------------------------------------

namespace
{
    void stone_change(GridPos p) {
        Stone *st = GetStone(p);
        if (st)
            st->floor_change();

        if (Item *it = GetItem(p))
            it->stone_change(st);

        if (Floor *fl = GetFloor(p))
            fl->stone_change(st);

        laser::MaybeRecalcLight(p);
    }
}

void
world::BroadcastMessage (const std::string& msg, const Value& value, GridLayerBits grids) 
{
    int  width     = level->w;
    int  height    = level->h;
    bool to_floors = grids & GRID_FLOOR_BIT;
    bool to_items  = grids & GRID_ITEMS_BIT;
    bool to_stones = grids & GRID_STONES_BIT;

    for (int y = 0; y<height; ++y) {
        for (int x = 0; x<width; ++x) {
            GridPos p(x, y);
            if (to_floors) SendMessage (GetFloor(p), msg, value);
            if (to_items)  SendMessage (GetItem(p),  msg, value);
            if (to_stones) SendMessage (GetStone(p), msg, value);
        }
    }
}

Stone *
world::GetStone(GridPos p) 
{
    return st_layer.get(p);
}

void
world::KillStone(GridPos p)
{
    st_layer.kill(p);
    changed_stones.push_back(p);
}

Stone *
world::YieldStone(GridPos p)
{
    Stone *st = st_layer.yield(p);
    changed_stones.push_back(p);
    return st;
}

void
world::SetStone(GridPos p, Stone* st)
{
    st_layer.set(p,st);
    changed_stones.push_back(p);
}

void
world::ReplaceStone(GridPos p, Stone* st)
{
    Stone *old = st_layer.get(p);
    if (old) {
        TransferObjectName(old, st);
        st_layer.kill(p);
    }
    SetStone(p, st);
}

void 
world::MoveStone(GridPos oldPos, GridPos newPos) 
{
    SetStone(newPos, YieldStone(oldPos));
}

void
world::SetScrambleIntensity(int intensity) {
    level->scrambleIntensity = intensity;
}

void
world::AddScramble(GridPos p, Direction d) {
    level->add_scramble(p, d);
}


//----------------------------------------
// Item manipulation.
//----------------------------------------

void world::KillItem(GridPos p) {
    laser::MaybeRecalcLight(p);
    it_layer.kill(p);
}

Item *world::GetItem(GridPos p) {
    return it_layer.get(p);
}

Item *world::YieldItem(GridPos p) {
    laser::MaybeRecalcLight(p);
    return it_layer.yield(p);
}

void world::SetItem(GridPos p, Item* it) {
    laser::MaybeRecalcLight(p);
    it_layer.set(p,it);
}

void world::SetItemAsStone(GridPos p, Item* it) {
    Stone *st = world::ConvertToStone(it);
    if (st)
        SetStone(p, st);
    else
        fprintf(stderr, "Could not SetItemAsStone(%s)\n", it->get_kind());
}

// ----------------------------------
//      Item <-> Stone conversion
// ----------------------------------

// These conversions assume that the name of the item starts with
// 'it-' and the name of stone starts with 'st-'. The rest of
// the name has to be identical.

// If the conversion fails, a message is printed to stderr
// and the source (item or stone) is not disposed.

Stone *world::ConvertToStone(Item *it) {
    string kind = it->get_kind();
    if (kind[0] == 'i') kind[0] = 's';

    Stone *st = MakeStone(kind.c_str());
    if (st) it->dispose();
    return st;
}

Item *world::ConvertToItem(Stone *st) {
    string kind = st->get_kind();
    if (kind[0] == 's') kind[0] = 'i';

    Item *it = MakeItem(kind.c_str());
    if (it) st->dispose();

    return it;
}

//----------------------------------------
// Actor manipulation.
//----------------------------------------

void
world::AddActor(double x, double y, Actor* a)
{
    V2 pos(x,y);
    level->actorlist.push_back(a);
    a->get_actorinfo()->pos = pos;

    // now done in AddActor(int, Actor*) :
    //     a->on_creation(pos);
    //     ReleaseActor(a);
}

Actor *
world::YieldActor(Actor *a)
{
    ActorList::iterator i=find(level->actorlist.begin(), level->actorlist.end(), a);
    if (i!=level->actorlist.end()) {
        level->actorlist.erase(i);
        GrabActor(a);
        return a;
    }
    return 0;
}

void world::WarpActor(Actor *a, double newx, double newy, bool fast)
{
    V2 newpos = V2(newx, newy);
    a->get_actorinfo()->vel = V2();
    a->warp(newpos);
}

void world::RespawnActor(Actor *a) {
    a->respawn();
}

void world::GrabActor(Actor *a)
{
    a->get_actorinfo()->grabbed = true;
}

void world::ReleaseActor(Actor *a)
{
    a->get_actorinfo()->grabbed = false;
}

bool world::GetActorsInRange (px::V2 center, double range,
                              vector<Actor*>& actors)
{
    ActorList &al = level->actorlist;
    for (ActorList::iterator i=al.begin(); i!=al.end(); ++i) {
        Actor *a = *i;
        if (length(a->get_pos()-center) < range)
            actors.push_back(a);
    }
    return !actors.empty();
}

//----------------------------------------
// handling of delayed impulses
//----------------------------------------

void
world::addDelayedImpulse(const Impulse& impulse, double delay, const Stone *estimated_receiver) {

    // @@@ FIXME: is a special handling necessary if several impulses hit same destination ?

    delayed_impulses.push_back(DelayedImpulse(impulse, delay, estimated_receiver));
}


static void
handle_delayed_impulses (double dtime)
{
    // Handle delayed impulses
    for (ImpulseList::iterator i = delayed_impulses.begin(); i != delayed_impulses.end(); ) 
    {
        // shall the impulse take effect now ?
        if (i->tick(dtime)) {
            if (Stone *st = GetStone(i->destination()))
                i->send_impulse(st);
            i = delayed_impulses.erase(i);
        }
        else
            ++i;
    }

}

//----------------------------------------
// Tick
//----------------------------------------

void
world::Tick(double dtime)
{
    const double timestep = 15.0/1000;
    static double timeaccu = 0.0;

    timeaccu += dtime;

    while (timeaccu >= timestep)
    {
	// Move actors
        for (ActorList::iterator i=level->actorlist.begin(); i!=level->actorlist.end(); ++i) {
            ActorInfo *ai = (*i)->get_actorinfo();
            if (!ai->grabbed && !(*i)->is_dead())
            {
		ai->force= ai->forceacc;
		ai->forceacc = V2();
                tick_actor(*i, timestep);
            }
	    (*i)->think(timestep);
        }

        handle_delayed_impulses (timestep);

        // Tell floors and items about new stones.
        for (unsigned i=0; i<changed_stones.size(); ++i)
            stone_change(changed_stones[i]);
        changed_stones.clear();

        level->tick(timestep);

        g_timer.tick(timestep);
        laser::RecalcLightNow();   // recalculate laser beams if necessary
        timeaccu -= timestep;
        timeaccu = 0;
    }
}

void world::Init()
{
    actors::Init();
    laser::Init();
    items::Init();
    stones::Init();
}
