#------------------------------------------------------------------------------
#
#  Define the base Enable object traits
#
#  Written by: David C. Morrill
#
#  Date: 09/24/2003
#
#  (c) Copyright 2003 by Enthought, Inc.
#
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
#  Imports:
#------------------------------------------------------------------------------

import base

from enthought.util.numerix import arange
from types      import ListType, TupleType
from base       import LEFT, RIGHT, TOP, BOTTOM, TOP_LEFT, TOP_RIGHT, \
                       BOTTOM_LEFT, BOTTOM_RIGHT, VCENTER, HCENTER, \
                       standard_colors, transparent_color, default_font_name, \
                       str_to_font, engraving_style, gc_image_for

from enthought.enable.traits.rgba_color_trait import RGBAColor
from enthought.kiva.fonttools import Font
from enthought.kiva.traits.kiva_font_trait import KivaFont
from enthought.traits.api    import HasTraits, Trait, TraitError, Range, Undefined,\
    TraitPrefixList, TraitPrefixMap, TraitHandler, Delegate, Str, Float
from enthought.traits.ui.api import ImageEnumEditor, FileEditor, TupleEditor, \
    TextEditor, Handler


#------------------------------------------------------------------------------
#  Constants:
#------------------------------------------------------------------------------

# Numeric 'array' type:
ArrayType = type( arange( 1.0 ) )

# Basic sequence types:
basic_sequence_types = ( ListType, TupleType )

# Sequence types:
sequence_types = [ ArrayType, ListType, TupleType ]

# Valid pointer shape names:
pointer_shapes = [
   'arrow', 'right arrow', 'blank', 'bullseye', 'char', 'cross', 'hand',
   'ibeam', 'left button', 'magnifier', 'middle button', 'no entry',
   'paint brush', 'pencil', 'point left', 'point right', 'question arrow',
   'right button', 'size top', 'size bottom', 'size left', 'size right',
   'size top right', 'size bottom left', 'size top left', 'size bottom right',
   'sizing', 'spray can', 'wait', 'watch', 'arrow wait'
]

alignment_positions = {
   'top':    TOP_LEFT,
   'left':   TOP_LEFT,
   'bottom': BOTTOM_RIGHT,
   'right':  BOTTOM_RIGHT,
   'center': VCENTER + HCENTER
}

# Trait editor version of 'alignment_positions':
edit_alignment_positions = {
   TOP_LEFT:          '1:left',
   VCENTER + HCENTER: '2:center',
   BOTTOM_RIGHT:      '3:right'
}

# Relative positions:
all_positions =  {
   'top left':      TOP_LEFT,
   'left top':      TOP_LEFT,
   'top right':     TOP_RIGHT,
   'right top':     TOP_RIGHT,
   'top center':    TOP + HCENTER,
   'center top':    TOP + HCENTER,
   'center left':   VCENTER + LEFT,
   'left center':   VCENTER + LEFT,
   'center right':  VCENTER + RIGHT,
   'right center':  VCENTER + RIGHT,
   'bottom left':   BOTTOM_LEFT,
   'left bottom':   BOTTOM_LEFT,
   'bottom right':  BOTTOM_RIGHT,
   'right bottom':  BOTTOM_RIGHT,
   'bottom center': BOTTOM + HCENTER,
   'center bottom': BOTTOM + HCENTER,
   'center':        VCENTER + HCENTER,
   'c':             VCENTER + HCENTER,
   'tl':            TOP_LEFT,
   'lt':            TOP_LEFT,
   'tr':            TOP_RIGHT,
   'rt':            TOP_RIGHT,
   'bl':            BOTTOM_LEFT,
   'lb':            BOTTOM_LEFT,
   'br':            BOTTOM_RIGHT,
   'rb':            BOTTOM_RIGHT,
   'top':           TOP + HCENTER,
   't':             TOP + HCENTER,
   'bottom':        BOTTOM + HCENTER,
   'b':             BOTTOM + HCENTER,
   'left':          VCENTER + LEFT,
   'l':             VCENTER + LEFT,
   'right':         VCENTER + RIGHT,
   'r':             VCENTER + RIGHT
}

# Trait editor version of 'all_positions':
edit_all_positions = {
   TOP_LEFT:          '1:top left',
   TOP + HCENTER:     '2:top center',
   TOP_RIGHT:         '3:top right',
   VCENTER + LEFT:    '4:center left',
   VCENTER + HCENTER: '5:center',
   VCENTER + RIGHT:   '6:center right',
   BOTTOM_LEFT:       '7:bottom left',
   BOTTOM + HCENTER:  '8:bottom center',
   BOTTOM_RIGHT:      '9:bottom right'
}

# Layout styles:
layout_styles = [ 'horizontal', 'vertical' ]

# Cursor styles:
CURSOR_X = 1
CURSOR_Y = 2

cursor_styles = {
    'default':    -1,
    'none':       0,
    'horizontal': CURSOR_Y,
    'vertical':   CURSOR_X,
    'both':       CURSOR_X | CURSOR_Y
}

# Layout modes:
layout_modes = [ 'immediate', 'deferred' ]

# Selection state:
selection_state = [ 'unselected', 'selected', 'coselected' ]

#-------------------------------------------------------------------------------
#  'ComponentHandler' class:
#-------------------------------------------------------------------------------

class ComponentHandler ( Handler ):

    #---------------------------------------------------------------------------
    #  Handles setting a specified object trait's value:
    #---------------------------------------------------------------------------

    def setattr ( self, object, name, value ):
        """ Handles setting a specified object trait's value.
        """
        setattr( object, name, value )
        object.redraw()

# Create the trait sheet handler:
component_handler = ComponentHandler()

#-------------------------------------------------------------------------------
#  'TraitImage' class:
#-------------------------------------------------------------------------------

class TraitImage ( TraitHandler ):

    def __init__ ( self, allow_none = True ):
        self.allow_none = allow_none

    def validate ( self, object, name, value ):
        if self.allow_none and ((value is None) or (value == '')):
            setattr( object, '_' + name, None )
            return None
        path   = ''
        image  = value
        prefix = image[:1]
        if prefix == '=':
            path  = object
            image = image[1:]
        elif prefix == '.':
            path  = None
            image = image[1:]
        image_ = gc_image_for( image, path )
        if image_ is not None:
            setattr( object, '_' + name, image_ )
            return value
        self.error( object, name, self.repr( value ) )

    def info ( self ):
        return 'the name of an image file (e.g a .png, .jpg, .gif file)'

### LGV HACK!!!
class TraitColor ( TraitHandler ):

    def validate ( self, object, name, value ):
        try:
            type_value = type( value )
            if isinstance(type_value, basestring):
                return standard_colors[ value ]
            if type_value in sequence_types:
                if ((len( value ) == 4)      and
                    (0.0 <= value[0] <= 1.0) and
                    (0.0 <= value[1] <= 1.0) and
                    (0.0 <= value[2] <= 1.0) and
                    (0.0 <= value[3] <= 1.0)):
                    return value
                raise TraitError
            if value is None:
                return transparent_color
            num = int( value )
            return ( ((num / 0x10000) & 0xFF) / 255.0,
                     ((num / 0x100)   & 0xFF) / 255.0,
                     (num             & 0xFF) / 255.0,
                     1.0 )
        except:
            self.error( object, name, self.repr( value ) )

    def setattr ( self, object, name, value, default ):
        mapped_value = self.validate( object, name, value )
        old_value    = object.__dict__.get( name, Undefined )
        if value != old_value:
            setattr( object, name + '_', mapped_value )
        object._set_trait_value( object, name, value, default )
        return value

    def is_mapped ( self ):
        return True

    def info ( self ):
        keys = standard_colors.keys()
        keys.sort()
        return ('a valid color, which can be a hex number of the form '
                '0xrrggbb, where rr is red, gg is green, and bb is blue, or a '
                'sequence of 4 elements ( rr, gg, bb, aa ), where aa is alpha, '
                'with each element a number in the range from 0.0 to 1.0, or '
                'one of the standard color names: %s' % (', '.join( keys )))

#-------------------------------------------------------------------------------
#  Verify that a bounds is a correctly formed 4-tuple:
#-------------------------------------------------------------------------------

def valid_bounds ( object, name, value ):
    try:
        if (type( value ) is TupleType) and (len( value ) == 4):
            return ( float( value[0] ), float( value[1] ),
                     float( min( max( value[2], object.min_width ),
                                                object.max_width ) ),
                     float( min( max( value[3], object.min_height ),
                                                object.max_height ) ) )
    except:
        pass
    raise TraitError

valid_bounds.info = 'an (x,y,width,height) position tuple'

#-------------------------------------------------------------------------------
#  Trait Editor definitions:
#-------------------------------------------------------------------------------

bounds_editor      = TupleEditor( traits = Float,
                                  labels = [ 'x', 'y', 'width', 'height' ],
                                  cols   = 2 )
eval_editor        = TextEditor( auto_set = False,
                                 evaluate = eval )
alignment_editor   = ImageEnumEditor(
                         values = edit_alignment_positions,
                         suffix = '_align',
                         cols   = 3,
                         module = base )
position_editor    = ImageEnumEditor(
                         values = edit_all_positions,
                         suffix = '_position',
                         cols   = 3,
                         module = base )
border_size_editor = ImageEnumEditor(
                         values = [ x for x in range( 9 ) ],
                         suffix = '_weight',
                         cols   = 3,
                         module = base )

#-------------------------------------------------------------------------------
#  Delegates:
#-------------------------------------------------------------------------------

simple_delegate = Delegate( 'container' )

#-------------------------------------------------------------------------------
#  Trait definitions:
#-------------------------------------------------------------------------------

# Color traits:
color_trait           = RGBAColor( 'white' )
white_color_trait     = color_trait
black_color_trait     = RGBAColor( 'black' )
cyan_color_trait      = RGBAColor( 'cyan' )
red_color_trait       = RGBAColor( 'red' )
green_color_trait     = RGBAColor( 'green' )
blue_color_trait      = RGBAColor( 'blue' )
yellow_color_trait    = RGBAColor( 'yellow' )
grey_color_trait      = RGBAColor( 'grey' )
clear_color_trait     = RGBAColor( 'clear' )

# Font trait:
font_trait            = KivaFont( default_font_name )

# Bounds trait:
bounds_trait          = Trait( ( 0.0, 0.0, 20.0, 20.0 ), valid_bounds,
                               editor = bounds_editor )

# Component minimum size trait:
ComponentMinSize      = Range( 0.0, 99999.0 )
ComponentMaxSize      = ComponentMinSize( 99999.0 )

# Pointer shape trait:
Pointer = Trait( 'arrow', TraitPrefixList( pointer_shapes ) )

# Cursor style trait:
cursor_style_trait    = Trait( 'default', TraitPrefixMap( cursor_styles ) )

# Edge alignment trait:
alignment_trait       = Trait( 'center',
                               TraitPrefixMap( alignment_positions ),
                               editor = alignment_editor )

# Relative position trait:
center_position_trait = Trait( 'center', all_positions,
                               editor = position_editor )
left_position_trait   = Trait( 'left', center_position_trait )

# Text engraving style:
engraving_trait       = Trait ( 'none', TraitPrefixMap( engraving_style ),
                                cols = 4 )

spacing_trait         = Range( 0, 63, value = 4 )
padding_trait         = Range( 0, 63, value = 4 )
margin_trait          = Range( 0, 63 )
border_size_trait     = Range( 0,  8, editor = border_size_editor )

# Simple image trait:
image_trait           = Trait( None, TraitImage(), editor = FileEditor )
string_image_trait    = Str( editor = FileEditor )

# Layout style trait:
layout_style_trait    = Trait( 'vertical', TraitPrefixList( layout_styles ) )

# Layout mode trait:
layout_mode_trait     = Trait( 'deferred', TraitPrefixList( layout_modes ) )

# Time interval trait:
TimeInterval          = Trait( None, None, Range( 0.0, 3600.0 ) )

# Stretch traits:
Stretch               = Range( 0.0, 1.0, value = 1.0 )
NoStretch             = Stretch( 0.0 )

# Grid trait:
grid_trait            = Range( 0, 512, value = 16 )

# Selection state traits:
selection_state_trait = Trait( 'unselected',
                               TraitPrefixList( selection_state ) )

