#-------------------------------------------------------------------------------
#  
#  Defines the OMTrack class of the Enable 'om' (Object Model) package. 
#
#  An OMTrack is a subcomponent of an OMTrackContactManager object that manages
#  a horizontally or vertically laid out set of Contacts.
#
#  Written by: David C. Morrill
#  
#  Date: 02/01/2005
#  
#  (c) Copyright 2005 by Enthought, Inc.
#  
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:  
#-------------------------------------------------------------------------------

from om_component     import OMComponent
from om_traits        import ContactCategory, OffsetXY, SizeXY, StyleDelegate
from om_contact       import OMContact
from om_track_group   import OMTrackGroup, OMTrackGroupStyle, \
                             default_track_group_style

from enthought.traits.api import HasPrivateTraits, HasStrictTraits, Trait, Str, \
                             List, Range, Property, Instance

#-------------------------------------------------------------------------------
#  Trait definitions:  
#-------------------------------------------------------------------------------
    
# Sorting order for the Contacts ('add' = order the Contacts are added in):
ADD  = 0
NAME = 1

TrackOrder = Trait( 'add', { 'add': ADD, 'name': NAME } )
    
# Sorting direction for the Contacts:
ASCENDING  = 0
DESCENDING = 1

TrackSort  = Trait( 'ascending', { 'ascending':  ASCENDING, 
                                   'descending': DESCENDING } )

# Layout orientation of the track:
VERTICAL   = 0
HORIZONTAL = 1

TrackOrientation = Trait( 'vertical', { 'vertical':   VERTICAL, 
                                        'horizontal': HORIZONTAL } )
                                        
# Position of the track relative to the associated component:
OUTER_LEFT   = 0
OUTER_RIGHT  = 1
OUTER_TOP    = 2
OUTER_BOTTOM = 3
INNER_LEFT   = 4
INNER_RIGHT  = 5
INNER_TOP    = 6
INNER_BOTTOM = 7
                                 
# Map Track positions to Contact positions:                                  
track_to_contact_map = [ 'left', 'right', 'top', 'bottom',
                         'left', 'right', 'top', 'bottom' ]

TrackPosition = Trait( 'left', { 'left':         OUTER_LEFT,
                                 'right':        OUTER_RIGHT,
                                 'top':          OUTER_TOP,
                                 'bottom':       OUTER_BOTTOM,
                                 'inner left':   INNER_LEFT,
                                 'inner right':  INNER_RIGHT,
                                 'inner top':    INNER_TOP,
                                 'inner bottom': INNER_BOTTOM } )
                                 
# Contact/Group spacing:
TrackSpacing = Range( 0, 31, 3 )

# Margin around the sides of the track:
TrackMargin = Range( 0, 31, 0 )

#-------------------------------------------------------------------------------
#  'OMTrackStyle' class:  
#-------------------------------------------------------------------------------

class OMTrackStyle ( HasStrictTraits ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------
    
    # Sorting order for the Contacts ('add' = order the Contacts are added in):
    order           = TrackOrder
    
    # Sorting direction for the Contacts:
    sort            = TrackSort
    
    # Amount of space between contacts in the same group:
    contact_spacing = TrackSpacing( 3 )
    
    # Amount of space between groups:
    group_spacing   = TrackSpacing( 2 )
    
    # Amount of margin added to left side of track:
    left_margin     = TrackMargin
    
    # Amount of margin added to right side of track:
    right_margin    = TrackMargin
    
    # Amount of margin added to top side of track:
    top_margin      = TrackMargin
    
    # Amount of margin added to bottom side of track:
    bottom_margin   = TrackMargin
    
# Create a default track style:    
default_track_style = OMTrackStyle()    
        
#-------------------------------------------------------------------------------
#  'OMTrack' class:  
#-------------------------------------------------------------------------------

class OMTrack ( HasPrivateTraits ):
   
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------

    # The component the track is managing contacts for:
    component       = Instance( OMComponent )
    
    # List of track groups containing the Contacts that the track manages:
    groups          = List( OMTrackGroup )
    
    # List of all Contacts that the track manages:
    contacts        = Property

    # Layout orientation of the track:
    orientation     = TrackOrientation
    
    # Position of the track relative to the associated component:
    position        = TrackPosition
    
    # Name of the track:
    name            = Str
    
    # Category of Contacts that the track manages:
    category        = ContactCategory
    
    # The style to use for new track groups when they are created:
    group_style     = Instance( OMTrackGroupStyle, default_track_group_style )
    
    # The style to use for the track:
    style           = Instance( OMTrackStyle, default_track_style )
    
    # Sorting order for the Contacts ('add' = order the Contacts are added in):
    order           = StyleDelegate
    
    # Sorting direction for the Contacts:
    sort            = StyleDelegate
    
    # Amount of space between contacts in the same group:
    contact_spacing = StyleDelegate
    
    # Amount of space between groups:
    group_spacing   = StyleDelegate
    
    # Amount of margin added to left side of track:
    left_margin     = StyleDelegate
    
    # Amount of margin added to right side of track:
    right_margin    = StyleDelegate
    
    # Amount of margin added to top side of track:
    top_margin      = StyleDelegate
    
    # Amount of margin added to bottom side of track:
    bottom_margin   = StyleDelegate
    
    # Origin of the track relative to its containing component:
    origin          = OffsetXY
    
    # Total size of the track and all of its contacts:
    size            = SizeXY
    
    # Offset to the center of the track:
    center          = OffsetXY
    
    # Minimum size of the track and all of its contacts:
    min_size        = Property
                       
    # List of all links connected to the associated component:                       
    links           = Property  

#-- Property Implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  Implementation of the 'contacts' property:
    #---------------------------------------------------------------------------
    
    def _get_contacts ( self ):
        result = []
        for group in self.groups:
            result.extend( group.contacts )
        return result
    
    #---------------------------------------------------------------------------
    #  Implementation of the 'links' property:  
    #---------------------------------------------------------------------------

    def _get_links ( self ):
        result = []
        for contact in self.contacts:
            result.extend( contact.links )
        return result
        
    #---------------------------------------------------------------------------
    #  Implementation of the 'min_size' property:  
    #---------------------------------------------------------------------------
                
    def _get_min_size ( self ):
        if self._min_size is None:
            self._compute_min_size()
        return self._min_size
        
#-- Event Handlers -------------------------------------------------------------
        
    #---------------------------------------------------------------------------
    #  Handles the 'origin' being changed: 
    #---------------------------------------------------------------------------
                
    def _origin_changed ( self, old, new ):
        dx = new[0] - old[0]
        dy = new[1] - old[1]
        for group in self.groups:
            cx, cy = group.origin
            group.origin = ( cx + dx, cy + dy )
        for contact in self.contacts:
            cx, cy         = contact.origin
            contact.origin = ( cx + dx, cy + dy )
        
    #---------------------------------------------------------------------------
    #  Handles the 'size' being changed:  
    #---------------------------------------------------------------------------
                
    def _size_changed ( self ):
        self._layout_track()
        
#-- Public Methods -------------------------------------------------------------        
    
    #---------------------------------------------------------------------------
    #  Adds a list of Contacts to the track:
    #---------------------------------------------------------------------------
    
    def add_contacts ( self, *contacts ):
        """ Adds a list of Contacts to the track.
        """
        # Process each Contact being added:
        for contact in contacts:
            
            # Make sure argument is a Contact object:
            if not isinstance( contact, OMContact ):
                raise ValueError, 'Each argument must be an OMContact object'
                
            # Set the Contact's position based on the Track's position:
            contact.position = track_to_contact_map[ self.position_ ]
                
            for group in self.groups:
                # Check if the contact should be added to this group:
                if contact.group == group.name:
                    
                    # Only add if it is not already in the group:
                    if contact not in group.contacts:
                        if self.order_ == ADD:
                            # Add the contact in 'add' order:
                            if self.sort_ == ASCENDING:
                                group.contacts.append( contact )
                            else:
                                group.contacts[0:0] = contact
                        else:
                            # Else add it in 'name' order:
                            group.contacts.append( contact )
                            if self.sort_ == ASCENDING:
                                group.contacts.sort( 
                                    lambda l,r: cmp( l.name, r.name ) )
                            else:
                                group.contacts.sort( 
                                    lambda l,r: cmp( r.name, l.name ) )
                    break
            else:
                # Does not belong in any current group, so add a new group:
                self.groups.append(
                    OMTrackGroup( contacts  = [ contact ],
                                  style     = self.group_style,
                                  container = self.component ) )
        
    #---------------------------------------------------------------------------
    #  Removes a list of Contacts from the tracks:  
    #---------------------------------------------------------------------------
            
    def remove_contacts ( self, *contacts ):
        """ Removes a list of Contacts from the tracks.
        """
        if len( contacts ) == 0:
            del self.groups[:]
            return
            
        for contact in contacts:
            for i, group in enumerate( self.groups ):
                group_contacts = group.contacts
                for j, a_contact in enumerate( group_contacts[:] ):
                    if contact is a_contact:
                        del group_contacts[j]
                        if len( group_contacts ) == 0:
                            del self.groups[i]
                        break
        
    #---------------------------------------------------------------------------
    #  Gets the list of Contacts in the track that match an optionally specified 
    #  name, category and/or group. If no arguments are specified, it returns 
    #  all Contacts associated with the track; otherwise it returns all Contacts 
    #  whose name, category or group match the specified value:
    #---------------------------------------------------------------------------
                
    def get_contacts ( self, name = None, category = None, group = None ):
        """ Gets the list of Contacts in the track that match an optionally 
            specified name, category and/or group. If no arguments are 
            specified, it returns all Contacts associated with the track; 
            otherwise it returns all Contacts whose name, category or group 
            match the specified value.
        """
        return [ contact for contact in self.contacts 
                 if (((name     is None) or (name     == contact.name))     and
                     ((category is None) or (category == contact.category)) and
                     ((group    is None) or (group    == contact.group))) ]
        
    #---------------------------------------------------------------------------
    #  Draws all of the groups:  
    #---------------------------------------------------------------------------
    
    def draw ( self, gc ):
        """ Draws all of the groups.
        """
        for group in self.groups:
            group.draw( gc )
        for contact in self.contacts:
            contact.draw( gc )
                       
    #---------------------------------------------------------------------------
    #  Returns the components that contain a specified (x,y) point:
    #---------------------------------------------------------------------------
       
    def components_at ( self, x, y ):
        """ Returns the components that contain a specified (x,y) point.
        """
        for contact in self.contacts:
            result = contact.components_at( x, y )
            if len( result ) > 0:
                return result
        return []

#-- Private Helper Methods -----------------------------------------------------

    #---------------------------------------------------------------------------
    #  Computes the amount of space that the track and all its groups and 
    #  contacts require:  
    #---------------------------------------------------------------------------

    def _compute_min_size ( self ):
        """ Computes the amount of space that the track and all its groups and
            contacts require.
        """
        tdx    = self.left_margin + self.right_margin 
        tdy    = self.top_margin  + self.bottom_margin
        groups = self.groups
        pos    = self.position_
        
        if self.orientation_ == VERTICAL:
            # Compute the size for a vertical layout of the contacts:
            tdy += self.group_spacing * (len( groups ) - 1)
            lx   = rx = gpbs = 0
            for group in groups:
                if group.visible:
                    gpbs = group.padding + 1  # 1 == border size
                    tdy += gpbs + gpbs
                tdy += self.contact_spacing * (len( group.contacts ) - 1)
                for contact in group.contacts:
                    cdx, cdy = contact.size
                    cx, cy   = contact.center
                    tdy     += cdy
                    lx       = max( lx, cx )
                    rx       = max( rx, cdx - cx )
            tdx += (lx + rx + gpbs)
            if (pos == INNER_RIGHT) or (pos == OUTER_RIGHT):
                lx += gpbs
            self.center = ( lx + self.left_margin, 0 )
        else:
            # Compute the size for a horizontal layout of the contacts:
            tdx += self.group_spacing * len( groups )
            ty   = by = gpbs = 0
            for group in groups:
                if group.visible:
                    gpbs = group.padding + 1  # 1 == border size
                    tdy += gpbs + gpbs
                tdx += self.contact_spacing * (len( group.contacts ) - 1)
                for contact in group.contacts:
                    cdx, cdy = contact.size
                    cx, cy   = contact.center
                    tdx     += cdx
                    ty       = max( ty, cdy - cy + gpbs )
                    by       = max( by, cy )
            tdy += (ty + by + gpbs)
            if (pos == INNER_TOP) or (pos == OUTER_TOP):
                by += gpbs
            self.center = ( 0, by + self.bottom_margin )
            
        self._min_size = ( tdx, tdy )
        
    #---------------------------------------------------------------------------
    #  Performs a layout of the track contents (i.e. groups and contacts):  
    #---------------------------------------------------------------------------
    
    def _layout_track ( self ):
        """ Performs a layout of the track contents (i.e. groups and contacts).
        """
        tx, ty     = self.origin
        ctx, cty   = self.center
        tdx, tdy   = self.size
        mtdx, mtdy = self._min_size
        
        # Calculate the amount of 'extra space' in each direction:
        ex = float( tdx - mtdx )
        ey = float( tdy - mtdy )
        
        # Calculate the number of inter-contact 'gaps':
        groups = self.groups
        gaps   = 0
        for group in groups:
            gaps += len( group.contacts )
        
        if gaps != 0:
            g_spacing = self.group_spacing
            pos       = self.position_
            if self.orientation_ == VERTICAL:
                # Determine if the track is on the left or right:
                left = (pos == INNER_LEFT) or (pos == OUTER_LEFT)
                lm   = self.left_margin
                rm   = self.right_margin
                
                # Lay the contacts out vertically:
                extra     = ey / gaps
                c_spacing = self.contact_spacing + extra
                tx       += ctx
                ty       += self.bottom_margin
                
                for i, group in enumerate( groups ):
                    nc = len( group.contacts ) - 1
                    
                    if group.visible:
                        base_ty = ty
                        gpbs    = group.padding + 1  # 1 == border size
                        ty      += gpbs
                      
                    ty += extra / 2.0
                    for j, contact in enumerate( group.contacts ):
                        cdx, cdy       = contact.size
                        cx, cy         = contact.center
                        contact.origin = ( tx - cx, int( ty ) )
                        ty += cdy
                        if j < nc:
                            ty += c_spacing
                    ty += extra / 2.0
                            
                    if group.visible:
                        ty += gpbs
                        gdy = int( round( ty ) - round( base_ty ) )
                        if left:
                            group.size   = ( tdx - ctx - rm, gdy )
                            group.origin = ( tx, int( round( base_ty ) ) )
                        else:
                            gdx          = ctx - lm
                            group.size   = ( gdx, gdy )
                            group.origin = ( tx - gdx, int( round( base_ty ) ) )
                            
                    ty += g_spacing
            else:
                # Determine if the track is on the top or bottom:
                bottom = (pos == INNER_BOTTOM) or (pos == OUTER_BOTTOM)
                tm     = self.top_margin
                bm     = self.bottom_margin
                
                # Lay the contacts out horizontally:
                extra     = ex / gaps
                c_spacing = self.contact_spacing + extra
                tx       += self.left_margin
                ty       += cty
                
                for i, group in enumerate( groups ):
                    nc = len( group.contacts ) - 1
                    
                    if group.visible:
                        base_tx = tx
                        gpbs    = group.padding + 1  # 1 == border size
                        tx     += gpbs
                        
                    tx += extra / 2.0
                    for j, contact in enumerate( group.contacts ):
                        cdx, cdy       = contact.size
                        cx, cy         = contact.center
                        contact.origin = ( int( tx ), ty - cy )
                        tx += cdx
                        if j < nc:
                            tx += c_spacing
                    tx += extra / 2.0
                            
                    if group.visible:
                        tx += gpbs
                        gdx = int( round( tx ) - round( base_tx ) )
                        if bottom:
                            group.size   = ( gdx, tdy - cty - tm )
                            group.origin = ( int( round( base_tx ) ), ty )
                        else:
                            gdy          = cty - bm
                            group.size   = ( gdx, gdy )
                            group.origin = ( int( round( base_tx ) ), ty - gdy )
                            
                    tx += g_spacing
        
