#-------------------------------------------------------------------------------
#  
#  Defines the OMTrackGroup class of the Enable 'om' (Object Model) package. 
#
#  An OMTrackGroup is a subcomponent of an OMTrack object that manages the set
#  of OMContact objects within a single track group.
#
#  Written by: David C. Morrill
#  
#  Date: 04/27/2005
#  
#  (c) Copyright 2005 by Enthought, Inc.
#  
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:  
#-------------------------------------------------------------------------------

from om_traits        import StyleDelegate, ERGBAColor, OffsetXY, SizeXY
from om_contact       import OMContact

from enthought.enable      import Component
from enthought.enable.base import transparent_color, filled_rectangle

from enthought.traits.api import HasStrictTraits, List, Range, Property, Instance, \
                             false

#-------------------------------------------------------------------------------
#  Trait definitions:  
#-------------------------------------------------------------------------------
 
# Amount of padding to allow around the edge of a track group:
GroupPadding = Range( 0, 31, value = 2 )    

#-------------------------------------------------------------------------------
#  'OMTrackGroupStyle' class:  
#-------------------------------------------------------------------------------

class OMTrackGroupStyle ( HasStrictTraits ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------

    # The track group background color:
    bg_color     = ERGBAColor( ( 1.0, 1.0, 1.0, 1.0 ) )
    
    # The track group border color:
    border_color = ERGBAColor( ( 0.0, 0.0, 0.0, 1.0 ) )
    
    # Amount of padding to add around the edge of the track group:
    padding      = GroupPadding
    
    # Is the track group visible?
    visible      = false
    
# Create a default track style:    
default_track_group_style = OMTrackGroupStyle()    
        
#-------------------------------------------------------------------------------
#  'OMTrackGroup' class:  
#-------------------------------------------------------------------------------

class OMTrackGroup ( Component ):
   
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------
    
    # Name of the track group:
    name         = Property
    
    # List of contacts that the track group manages:
    contacts     = List( OMContact )
    
    # The location and size of the track group (override of the Enable Component 
    # definition):
    bounds       = Property
    
    # Origin of the track group relative to its containing component:
    origin       = OffsetXY
    
    # Size of the track group:
    size         = SizeXY
    
    # The style to use for the track group:
    style        = Instance( OMTrackGroupStyle, default_track_group_style )

    # The track group background color:
    bg_color     = StyleDelegate
    
    # The rack group border color:
    border_color = StyleDelegate
    
    # Amount of padding to add around the edge of the track group:
    padding      = StyleDelegate
    
    # Is the track group visible? (overrides the Enable Component definition)
    visible      = StyleDelegate
    
#-- Property implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  Implementation of the 'name' property:  
    #---------------------------------------------------------------------------

    def _get_name ( self ):
        if len( self.contacts ) > 0:
            return self.contacts[0].group
        return 'default'

    #---------------------------------------------------------------------------
    #  Implementation of the 'bounds' property:  
    #---------------------------------------------------------------------------

    def _get_bounds ( self ):
        x, y, dx, dy = self.container.bounds
        ox, oy       = self.origin
        return ( x + ox, y + oy ) + self.size
        
    def _set_bounds ( self, bounds ):
        x, y, dx, dy        = self._bounds = bounds
        cx, cy, cdx, cdy    = self.container.bounds
        self.origin         = ( int( x - cx ), int( y - cy ) )
        self.size           = ( int( dx ), int( dy ) )
        self._drag_location = None

#-- Enable Component method overrides ------------------------------------------

    #---------------------------------------------------------------------------
    #  Draws the track group:  
    #---------------------------------------------------------------------------
    
    def _draw ( self, gc ):
        """ Draws the track group.
        """
        filled_rectangle( gc, self.bounds, self.bg_color_, self.border_color_ )
            
