"""
Shows how to implement a simple drag operation with transitory visual
side-effects that have to be cleaned up when the drag completes.
"""
from enthought.util.numerix import array

from enthought.traits.api import Array, Float, Trait, Tuple

from enthought.enable import Component, Container
from enthought.enable.enable_traits import Pointer
from enthought.enable.wx import Window

# Relative imports
from demo_base import DemoFrame, demo_main


class Circle(Component):
    """
    The circle moves with the mouse cursor but leaves a translucent version of
    itself in its original position until the mouse button is released.
    """
    color = (0.0, 0.0, 1.0, 1.0)
    
    # used for drawing the shadow image when moving
    line_dash = Trait(None, None, Array)
    
    normal_pointer = Pointer("arrow")
    moving_pointer = Pointer("hand")
    
    original_position = Tuple  # (x,y)
    original_radius = Float
    
    offset_x = Float
    offset_y = Float
    
    def _draw(self, gc):
        gc.save_state()
        gc.set_fill_color(self.color)
        x, y, dx, dy = self.bounds
        gc.arc(x+dx/2.0, y+dy/2.0, min(dx/2.0, dy/2.0), 0.0, 2*3.14159)
        gc.fill_path()
        if self.event_state == "moving":
            gc.arc(self.original_position[0], self.original_position[1], self.original_radius,
                   0.0, 2*3.14159)
            if self.line_dash is None:
                gc.set_fill_color(self.color[0:3] + (self.color[3]*0.3,))
                gc.fill_path()
            else:
                gc.set_stroke_color(self.color[0:3] + (self.color[3]*0.8,))
                gc.set_line_dash(self.line_dash)
                gc.stroke_path()
        gc.restore_state()
        return
    
    def normal_left_down(self, event):
        self.event_state = "moving"
        self.pointer = self.moving_pointer
        x, y, dx, dy = self.bounds
        self.offset_x = event.x - x
        self.offset_y = event.y - y
        self.original_position = (x+dx/2.0, y+dx/2.0)
        self.original_radius = min(dx/2.0, dy/2.0)
        return

    def moving_mouse_move(self, event):
        self.bounds = (event.x-self.offset_x, event.y-self.offset_y, self.bounds[2], self.bounds[3])
        self.redraw()
        return

    def moving_left_up(self, event):
        self.event_state = "normal"
        self.pointer = self.normal_pointer
        self.window.redraw()   # for some reason self.redraw() doesn't work...
        return


class MyFrame(DemoFrame):
    def _create_window(self):
        circle1 = Circle(bounds=(50,50,75,75), line_dash=array([2.0, 2.0]))
        circle2 = Circle(bounds=(200,50,75,75))
        container = Container(bounds=(0,0,500,500))
        container.add(circle1)
        container.add(circle2)
        return Window(self, -1, component=container)
        
if __name__ == "__main__":
    demo_main(MyFrame, title="Click and drag to move the circles")
    
# EOF
