#------------------------------------------------------------------------------
# Copyright (c) 2005, Enthought, Inc.
# All rights reserved.
# 
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
# 
# Author: Enthought, Inc.
# Description: <Enthought pyface package component>
#------------------------------------------------------------------------------
""" A splash screen. """


# Standard library imports.
import logging

# Major package imports.
import wx

# Enthought library imports.
from enthought.traits.api import Bool, Font, Instance, Int, Str, Tuple
from enthought.traits.ui.wx.color_trait import WxColor

# Local imports.
from image_resource import ImageResource
from splash_screen_log_handler import SplashScreenLogHandler
from util.font_helper import new_font_like
from window import Window


# Create a logger for this module.
logger = logging.getLogger(__name__)


class SplashScreen(Window):
    """ A splash screen. """

    #### 'SplashScreen' interface #############################################
    
    # The image to display on the splash screen.
    image = Instance(ImageResource, ImageResource('splash'))

    # If log messages are to be displayed then this is the logging level. See
    # the Python documentation for the 'logging' module for more details.
    log_level = Int(logging.DEBUG)

    # Should the splash screen display log messages in the splash text?
    show_log_messages = Bool(True)

    # handler will
    # Optional text to display on top of the splash image.
    text = Str

    # The text color.
    text_color = WxColor('black')

    # The text font.
    text_font = Font(None)

    # The x, y location where the text will be drawn.
    text_location  = Tuple(5, 5)
    
    ###########################################################################
    # 'Window' interface.
    ###########################################################################

    def open(self):
        """ Creates the toolkit-specific control for the widget. """

        super(SplashScreen, self).open()

        if self.show_log_messages:
            self._log_handler = SplashScreenLogHandler(self)
            self._log_handler.setLevel(self.log_level)
            
            logger.addHandler(self._log_handler)

        return

    def close(self):
        """ Close the window. """

        if self.show_log_messages:
            logger.removeHandler(self._log_handler)

        super(SplashScreen, self).close()
        
        return

    ###########################################################################
    # Protected 'Window' interface.
    ###########################################################################

    def _create_control(self, parent):
        """ Create the toolkit-specific control that represents the window.

        This method is intended to be overridden!  By default we just create an
        empty frame.

        """

        # Get the splash screen image.
        image = self.image.create_image()
        
        splash_screen = wx.SplashScreen(
            # The bitmap to display on the splash screen.
            image.ConvertToBitmap(),
            # Splash Style.
            wx.SPLASH_NO_TIMEOUT | wx.SPLASH_CENTRE_ON_SCREEN,
            # Timeout in milliseconds (we don't currently timeout!).
            0,
            # The parent of the splash screen.
            parent,
            # wx Id.
            -1,
            # Window style.
            style = wx.SIMPLE_BORDER | wx.FRAME_NO_TASKBAR
        )

        # By default we create a font slightly bigger and slightly more italic
        # than the normal system font ;^)  The font is used inside the event
        # handler for 'EVT_PAINT'.
        if self.text_font is None:
            self.text_font = self._default_text_font()

        # This allows us to write status text on the splash screen.
        wx.EVT_PAINT(splash_screen, self._on_paint)

        return splash_screen

    def _create_contents(self, parent):
        """ Creates the window contents. """

        # In this case, 'wx.SplashScreen' does it all for us.
        pass
        
    ###########################################################################
    # Private interface.
    ###########################################################################

    def _default_text_font(self):
        """ Creates the default text font. """

        font = new_font_like(
            wx.NORMAL_FONT,
            point_size = wx.NORMAL_FONT.GetPointSize() + 1,
            style      = wx.ITALIC
        )

        return font
        
    #### Trait event handlers #################################################

    def _text_changed(self):
        """ Called when the splash screen text has been changed. """

        if self.control is not None:
            # Passing 'False' to 'Refresh' means "do not erase the background".
            self.control.Refresh(False)
            self.control.Update()
            wx.Yield()
        
        return

    #### wx event handlers ####################################################
    
    def _on_paint(self, event):
        """ Called when the splash window is being repainted. """

        if self.control is not None:
            # Get the window that the splash image is drawn in.
            window = self.control.GetSplashWindow()

            dc = wx.PaintDC(window)
            dc.SetFont(self.text_font)
            dc.SetTextForeground(self.text_color)

            x, y = self.text_location
            dc.DrawText(self.text, x, y)

        # Let the normal wx paint handling do its stuff.
        event.Skip()
        
        return

#### EOF ######################################################################
