#-------------------------------------------------------------------------------
#
#  Debugger kernel module, based on the 'bdb.py' module distributed with Python.
#
#  Written by:  Python Development Team
#  Modified by: David C. Morrill
#
#  Date: 07/23/2006
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

import sys
import os
import types

from enthought.traits.api \
    import HasPrivateTraits, Any, File, Int, Str, Enum, Property, false, true

from enthought.developer.helper.read_file \
    import read_file

#-------------------------------------------------------------------------------
#  Constants:
#-------------------------------------------------------------------------------

# Template for 'Log' break point code:
logger_template = """import logging
logger = logging.getLogger(__name__)
logger.debug( "%s: %%s" %% (%s) )
"""

#-------------------------------------------------------------------------------
#  Trait definitions:
#-------------------------------------------------------------------------------

BPType = Enum( 'Breakpoint', 'Temporary', 'Count', 'Trace', 'Print', 'Log' )

#-------------------------------------------------------------------------------
#  'BdbQuit' class:
#-------------------------------------------------------------------------------

class BdbQuit ( Exception ):
    """Exception to give up completely"""

#-------------------------------------------------------------------------------
#  'Bdb' class:
#-------------------------------------------------------------------------------

class Bdb:
    """Generic Python debugger base class.

       This class takes care of details of the trace facility; a derived class
       should implement user interaction. The standard debugger class (pdb.Pdb)
       is an example.
    """

    def __init__ ( self ):
        self.breaks  = {}
        self.fncache = {}

    def canonic ( self, file_name ):
        if file_name == ("<" + file_name[1:-1] + ">"):
            return file_name

        canonic = self.fncache.get( file_name )
        if not canonic:
            canonic = os.path.abspath( file_name )
            canonic = os.path.normcase( canonic )
            self.fncache[ file_name ] = canonic

        return canonic

    def reset ( self ):
        import linecache

        linecache.checkcache()
        self.botframe    = None
        self.stopframe   = None
        self.returnframe = None
        self.quitting    = 0

    def trace_dispatch ( self, frame, event, arg ):
        if self.quitting:
            return # None

        if event == 'line':
            return self.dispatch_line( frame )

        if event == 'call':
            return self.dispatch_call( frame, arg )

        if event == 'return':
            return self.dispatch_return( frame, arg )

        if event == 'exception':
            return self.dispatch_exception( frame, arg )

        print 'bdb.Bdb.dispatch: unknown debugging event:', `event`

        return self.trace_dispatch

    def dispatch_line ( self, frame ):
        if self.stop_here ( frame ) or self.break_here( frame ):
            self.user_line( frame )
            if self.quitting:
                raise BdbQuit

        return self.trace_dispatch

    def dispatch_call ( self, frame, arg ):
        # XXX 'arg' is no longer used
        if self.botframe is None:
            # First call of dispatch since reset()
            self.botframe = frame.f_back # (CT) Note that this may also be None!
            return self.trace_dispatch

        if not (self.stop_here( frame ) or self.break_anywhere( frame )):
            # No need to trace this function
            return # None

        self.user_call( frame, arg )
        if self.quitting:
            raise BdbQuit

        return self.trace_dispatch

    def dispatch_return ( self, frame, arg ):
        if self.stop_here( frame ) or (frame == self.returnframe):
            self.user_return( frame, arg )
            if self.quitting:
                raise BdbQuit

        return self.trace_dispatch

    def dispatch_exception ( self, frame, arg ):
        if self.stop_here( frame ):
            self.user_exception( frame, arg )
            if self.quitting:
                raise BdbQuit

        return self.trace_dispatch

    #---------------------------------------------------------------------------
    # Normally derived classes don't override the following methods, but they
    # may if they want to redefine the definition of stopping and breakpoints:
    #---------------------------------------------------------------------------

    def stop_here ( self, frame ):
        # (CT) stopframe may now also be None, see dispatch_call.
        # (CT) the former test for None is therefore removed from here.
        if frame is self.stopframe:
            return True

        while (frame is not None) and (frame is not self.stopframe):
            if frame is self.botframe:
                return True

            frame = frame.f_back

        return False

    def break_here ( self, frame ):
        file_name = self.canonic( frame.f_code.co_filename )
        if file_name not in self.breaks:
            return False

        line = frame.f_lineno
        if line not in self.breaks[ file_name ]:
            return False

        return effective( file_name, line, frame )

    def break_anywhere ( self, frame ):
        return self.breaks.has_key( self.canonic( frame.f_code.co_filename ) )

    #---------------------------------------------------------------------------
    #  Derived classes should override the user_* methods to gain control:
    #---------------------------------------------------------------------------

    def user_call ( self, frame, argument_list ):
        """This method is called when there is the remote possibility
        that we ever need to stop in this function."""
        pass

    def user_line ( self, frame ):
        """This method is called when we stop or break at this line."""
        pass

    def user_return ( self, frame, return_value ):
        """This method is called when a return trap is set here."""
        pass

    def user_exception ( self, frame, ( exc_type, exc_value, exc_traceback ) ):
        """This method is called if an exception occurs,
        but only if we are to stop at or just below this level."""
        pass

    #---------------------------------------------------------------------------
    #  Derived classes and clients can call the following methods to affect the
    #  stepping state:
    #---------------------------------------------------------------------------

    def set_step ( self ):
        """Stop after one line of code."""
        self.stopframe   = None
        self.returnframe = None
        self.quitting    = 0

    def set_next ( self, frame ):
        """Stop on the next line in or below the given frame."""
        self.stopframe   = frame
        self.returnframe = None
        self.quitting    = 0

    def set_return ( self, frame ):
        """Stop when returning from the given frame."""
        self.stopframe   = frame.f_back
        self.returnframe = frame
        self.quitting    = 0

    def set_trace ( self ):
        """Start debugging from here."""
        frame = sys._getframe().f_back
        self.reset()
        while frame:
            frame.f_trace = self.trace_dispatch
            self.botframe = frame
            frame         = frame.f_back

        self.set_step()
        sys.settrace( self.trace_dispatch )

    def set_continue ( self ):
        # Don't stop except at breakpoints or when finished
        self.stopframe   = self.botframe
        self.returnframe = None
        self.quitting    = 0

        if not self.breaks:
            # no breakpoints; run without debugger overhead
            sys.settrace( None )
            frame = sys._getframe().f_back
            while frame and (frame is not self.botframe):
                del frame.f_trace
                frame = frame.f_back

    def set_quit ( self ):
        self.stopframe   = self.botframe
        self.returnframe = None
        self.quitting    = 1
        sys.settrace( None )

    #---------------------------------------------------------------------------
    #  Derived classes and clients can call the following methods to manipulate
    #  breakpoints.  These methods return an error message is something went
    #  wrong, None if all is well. Set_break prints out the breakpoint line and
    #  file:lineno. Call self.get_*break*() to see the breakpoints:
    #---------------------------------------------------------------------------

    def set_break ( self, file_name, line, bp_type = 'Breakpoint', code = '' ):
        file_name = self.canonic( file_name )
        list      = self.breaks.setdefault( file_name, [] )
        if line not in list:
            list.append( line )

        # Return the break point:
        return Breakpoint( file      = file_name,
                           line      = line,
                           bp_type   = bp_type,
                           code      = code )

    def restore_break ( self, bp ):
        list = self.breaks.setdefault( bp.file, [] )
        if bp.line not in list:
            list.append( bp.line )

    def clear_break ( self, file_name, line ):
        file_name = self.canonic( file_name )
        breaks    = self.breaks.get( file_name )
        if (breaks is None) or (line not in breaks):
            return []

        # If there's only one bp in the list for that file,line pair, then
        # remove the breaks entry:
        bps = Breakpoint.bp_list[ ( file_name, line ) ][:]
        for bp in bps:
            bp.delete_me()

        if not Breakpoint.bp_list.has_key( ( file_name, line ) ):
            breaks.remove( line )

        if len( breaks ) == 0:
            del self.breaks[ file_name ]

        return bps

    def clear_all_file_breaks ( self, file_name ):
        file_name = self.canonic( file_name )
        if file_name in self.breaks:
            for line in self.breaks[ file_name ]:
                for bp in Breakpoint.bp_list[ ( file_name, line ) ][:]:
                    bp.delete_me()

            del self.breaks[ file_name ]

    def clear_all_breaks ( self ):
        for bps in Breakpoint.bp_list.values():
            for bp in bps:
                bp.delete_me()

        self.breaks = {}

    def clear_bp ( self, bp ):
        file, line = bp.file, bp.line
        bp.delete_me()
        breaks = self.breaks.get( file )
        if not Breakpoint.bp_list.has_key( ( file, line ) ):
            breaks.remove( line )

        if len( breaks ) == 0:
            del self.breaks[ file ]

    def get_break ( self, file_name, line ):
        return (( self.canonic( file_name ), line ) in Breakpoint.bp_list)

    def get_breaks ( self, file_name, line ):
        file_name = self.canonic( file_name )
        return Breakpoint.bp_list.get( ( file_name, line ), [] )

    def get_file_breaks ( self, file_name ):
        file_name = self.canonic( file_name )
        if file_name in self.breaks:
            return self.breaks[ file_name ]

        return []

    def get_all_breaks ( self ):
        return self.breaks

    #---------------------------------------------------------------------------
    #  The following two methods can be called by clients to use a debugger to
    #  debug a statement, given as a string:
    #---------------------------------------------------------------------------

    def run ( self, cmd, globals = None, locals = None ):
        if globals is None:
            import __main__
            globals = __main__.__dict__

        if locals is None:
            locals = globals

        self.reset()
        sys.settrace( self.trace_dispatch )
        if not isinstance( cmd, types.CodeType ):
            cmd += '\n'

        try:
            try:
                exec cmd in globals, locals
            except BdbQuit:
                pass
        finally:
            self.quitting = 1
            sys.settrace( None )

    def runeval ( self, expr, globals = None, locals = None ):
        if globals is None:
            import __main__
            globals = __main__.__dict__

        if locals is None:
            locals = globals

        self.reset()
        sys.settrace( self.trace_dispatch )
        if not isinstance( expr, types.CodeType ):
            expr += '\n'

        try:
            try:
                return eval( expr, globals, locals )
            except BdbQuit:
                pass
        finally:
            self.quitting = 1
            sys.settrace( None )

    def runctx ( self, cmd, globals, locals ):
        # B/W compatibility
        self.run( cmd, globals, locals )

    #---------------------------------------------------------------------------
    #  This method is more useful to debug a single function call:
    #---------------------------------------------------------------------------

    def runcall ( self, func, *args ):
        self.reset()
        sys.settrace( self.trace_dispatch )
        res = None
        try:
            try:
                res = func( *args )
            except BdbQuit:
                pass
        finally:
            self.quitting = 1
            sys.settrace( None )
        return res

def set_trace():
    Bdb().set_trace()

#-------------------------------------------------------------------------------
#  'Breakpoint' class:
#-------------------------------------------------------------------------------

class Breakpoint ( HasPrivateTraits ):

    #---------------------------------------------------------------------------
    #  Class variables:
    #---------------------------------------------------------------------------

    # Indexed by (file, lineno) tuple:
    bp_list = {}

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    # The owner of this break point:
    owner = Any

    # The fully qualified name of the file the break point is in:
    file = File

    # The module name:
    module = Str

    # The source file path:
    path = Str

    # The line number the break point is on:
    line = Int

    # Break point type:
    bp_type = BPType

    # Is the break point enabled:
    enabled = true

    # Optional code associated with the break point:
    code = Str

    # The number of times the break point should be ignored before tripping:
    ignore = Int

    # The number of times the break point has been hit:
    hits = Int

    # The number of times this breakpoint has been counted:
    count = Int

    # Source line:
    source = Property

    #---------------------------------------------------------------------------
    #  Initializes the object:
    #---------------------------------------------------------------------------

    def __init__ ( self, **traits ):
        """ Initializes the object.
        """
        super( Breakpoint, self ).__init__( **traits )
        self.register()

    #---------------------------------------------------------------------------
    #  Registers a break point:
    #---------------------------------------------------------------------------

    def register ( self ):
        """ Registers a break point.
        """
        Breakpoint.bp_list.setdefault( ( self.file, self.line ), []
                                     ).append( self )

    #---------------------------------------------------------------------------
    #  Returns the persistent state of a break point:
    #---------------------------------------------------------------------------

    def __getstate__ ( self ):
        return self.get( 'file', 'line', 'bp_type', 'enabled', 'code',
                         'source' )

    #---------------------------------------------------------------------------
    #  Implementation of the 'source' property:
    #---------------------------------------------------------------------------

    def _get_source ( self ):
        if self._source is None:
            try:
                self._source = read_file( self.file ).split( '\n'
                                                 )[ self.line - 1 ].strip()
            except:
                self._source = '???'
        return self._source

    def _set_source ( self, source ):
        self._source = source

    #---------------------------------------------------------------------------
    #  Handles any trait on the break point being changed:
    #---------------------------------------------------------------------------

    def _anytrait_changed ( self, name, old, new ):
        """ Handles any trait on the break point being changed.
        """
        if ((self.owner is not None) and
            (name in ( 'bp_type', 'enabled', 'code', 'ignore' ))):
            self.owner.modified = True

    #---------------------------------------------------------------------------
    #  Handles the 'file' trait being changed:
    #---------------------------------------------------------------------------

    def _file_changed ( self ):
        """ Handles the 'file' trait being changed.
        """
        self.path, module = os.path.split( self.file )
        self.module       = os.path.splitext( module )[0]

    #---------------------------------------------------------------------------
    #  Handles the 'code' trait being changed:
    #---------------------------------------------------------------------------

    def _code_changed ( self ):
        """ Handles the 'code' trait being changed.
        """
        self._code = None

    #---------------------------------------------------------------------------
    #  Handles the 'bp_type' trait being changed:
    #---------------------------------------------------------------------------

    def _bp_type_changed ( self ):
        """ Handles the 'bp_type' trait being changed.
        """
        self._code = None

    #---------------------------------------------------------------------------
    #  Restores the correct line value based on current source file contents:
    #---------------------------------------------------------------------------

    def restore ( self ):
        """ Restores the correct line value based on current source file
            contents.
        """
        self._file_changed()
        try:
            lines = read_file( self.file ).split( '\n' )
        except:
            return False
        n      = len( lines )
        line   = self.line - 1
        source = self.source

        # Search outward from the last known location of the source for a
        # match:
        for i in range( 100 ):
            if ((line - i) < 0) and ((line + i) >= n):
                break

            j = line + i
            if (j < n) and (source == lines[j].strip()):
                self.line = j + 1
                self.register()
                return True

            j = line - i
            if (j >= 0) and (source == lines[j].strip()):
                self.line = j + 1
                self.register()
                return True

        # Indicate source line could not be found:
        self.line = 0

        return False

    #---------------------------------------------------------------------------
    #  Deletes the breakpoint:
    #---------------------------------------------------------------------------

    def delete_me ( self ):
        """ Deletes the breakpoint.
        """
        index = ( self.file, self.line )

        # No longer in list:
        bp_list = Breakpoint.bp_list
        bp_list[ index ].remove( self )
        if len( bp_list[ index ] ) == 0:
            # No more bp for this file:line combo:
            del bp_list[ index ]

#-------------------------------------------------------------------------------
#  Returns whether there is an effective (active) breakpoint at this line of
#  code.
#-------------------------------------------------------------------------------

def effective ( file, line, frame ):
    """ Returns whether there is an effective (active) breakpoint at this line
        of code.
    """
    result = False
    for bp in Breakpoint.bp_list[ file, line ]:
        if not bp.enabled:
            continue

        # Count every hit when bp is enabled:
        bp.hits += 1

        code    = bp.code.strip()
        bp_type = bp.bp_type
        if bp_type in ( 'Trace', 'Print', 'Log' ):
            if bp._code is None:
                if code != '':
                    if bp_type == 'Print':
                        code = "print '%s:', %s" % ( code, code )
                    elif bp_type == 'Log':
                        code = logger_template % ( code, code )
                    try:
                        bp._code = compile( code, '<string>', 'exec' )
                    except:
                        continue

            if bp.ignore <= 0:
                try:
                    exec bp._code in frame.f_globals, frame.f_locals
                except:
                    print ('Error executing break point code on line %d '
                           'in %s' % ( bp.line, bp.file ))
            else:
                bp.ignore -= 1

            continue

        if code == '':
            # If unconditional, and ignoring, go on to next, else break:
            if bp.ignore > 0:
                bp.ignore -= 1
                continue

            # If this is a counter, count it and continue:
            if bp_type == 'Count':
                bp.count += 1
                continue

            # Breakpoint and marker that's ok to delete if temporary:
            result = True
            continue

        # Conditional bp.
        # Ignore count applies only to those bpt hits where the condition
        # evaluates to true.
        if bp._code is None:
            try:
                bp._code = compile( code, '<string>', 'eval' )
            except:
                result = True
                continue

        try:
            val = eval( bp._code, frame.f_globals, frame.f_locals )
            if val:
                if bp.ignore <= 0:

                    # If this is a counter, count it and continue:
                    if bp_type == 'Count':
                        bp.count += 1
                        continue

                    result = True
                    continue

                bp.ignore -= 1
        except:
            # If eval fails, most conservative thing is to stop on breakpoint
            # regardless of ignore count.
            result = True

    return result

