// ePDFView - A lightweight PDF Viewer.
// Copyright (C) 2006 Emma's Software.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>
#include <time.h>
#include <glib/poppler.h>
#include "epdfview.h"

using namespace ePDFView;

// Constants.
static const gint PIXBUF_BITS_PER_SAMPLE = 8;
static const gint DATE_LENGTH = 100;

// Forward declarations.
static PageLayout convertPageLayout (gint pageLayout);
static PageMode convertPageMode (gint pageMode);
static gchar *getAbsoluteFileName (const gchar *fileName);

///
/// @brief Constructs a new PDFDocument object.
///
PDFDocument::PDFDocument ():
    IDocument ()
{
    m_Document = NULL;
}

///
/// @brief Deletes all dynamically created objects of PDFDocument.
///
PDFDocument::~PDFDocument ()
{
    clearCache ();
    if ( NULL != m_Document )
    {
        g_object_unref (G_OBJECT (m_Document));
        m_Document = NULL;
    }
}

GList *
PDFDocument::findTextInPage (gint pageNum, const gchar *textToFind)
{
    GList *results = NULL;
    
    if ( NULL == m_Document )
    {    
        return results;
    }

    PopplerPage *page = poppler_document_get_page (m_Document, pageNum - 1);
    if ( NULL != page )
    {
        gdouble height = 1.0;
        poppler_page_get_size (page, NULL, &height);
        GList *matches = poppler_page_find_text (page, textToFind);
        for ( GList *match = g_list_first (matches) ;
              NULL != match ;
              match = g_list_next (match) )
        {
            PopplerRectangle *matchRect = (PopplerRectangle *)match->data;
            DocumentRectangle *rect = 
                new DocumentRectangle (matchRect->x1,
                                       (height - matchRect->y2),
                                       matchRect->x2,
                                       (height - matchRect->y1));
            results = g_list_prepend (results, rect);
        }
    }

    return g_list_reverse (results);
}

///
/// @brief Checks if the document has been loaded.
///
/// @return TRUE if the document has been loaded, FALSE otherwise.
///
gboolean
PDFDocument::isLoaded ()
{
    return (NULL != m_Document);
}

///
/// @brief Loads a PDF file.
///
/// Tries to open the PDF file @a filename using the password in @a password.
///
/// @param filename The name of the file name to open. It must be an absolute
///                 path.
/// @param password The password to use to open @a filename.
/// @param error Location to store the error occurring or NULL to ignore 
///              errors.
///
/// @return TRUE if the file could be opened, FALSE otherwise.
///
gboolean
PDFDocument::loadFile (const gchar *filename, const gchar *password, 
                    GError **error)
{
    g_assert (NULL != filename && "Tried to load a NULL file name");

    gchar *absoluteFileName = getAbsoluteFileName (filename);
    gchar *filename_uri = g_filename_to_uri (absoluteFileName, NULL, error);
    g_free (absoluteFileName);
    if ( NULL == filename_uri )
    {
        return FALSE;
    }
    // Try to open the PDF document.
    GError *loadError = NULL;
    PopplerDocument *newDocument = 
        poppler_document_new_from_file (filename_uri, password, &loadError);
    g_free (filename_uri);
    // Check if the document couldn't be opened successfully and why.
    if ( NULL == newDocument )
    {
        // Poppler's glib wrapper passes the Poppler error code unless the
        // error is that the file is encrypted. We want to set our own
        // error code in this case.
        DocumentError errorCode = DocumentErrorNone;
        if ( POPPLER_ERROR == loadError->domain )
        {
            errorCode = DocumentErrorEncrypted; 
        }
        else
        {
            // OK, the glib's wrapper don't pass the error code directly
            // from Poppler. Instead returns G_FILE_ERROR_FAILED and a 
            // non translated string.
            // Maybe I'm wrong (very probable) but that's a wrong way.
            // So I'm reading the error code from the error string...
            sscanf (loadError->message, "Failed to load document (error %d)", 
                    (gint *)&errorCode);
        }
        g_error_free (loadError);
        // Get our error message.
        gchar *errorMessage = IDocument::getErrorMessage (errorCode);
        g_set_error (error, 
                     EPDFVIEW_DOCUMENT_ERROR, errorCode,
                     _("Failed to load document '%s'.\n%s\n"),
                     filename, errorMessage);
        g_free (errorMessage);

        return FALSE;
    }
   
    // Set the used filename and password to let the user reload the
    // document.
    setFileName (filename);    
    setPassword (password);
    if ( NULL != m_Document )
    {
        g_object_unref (G_OBJECT (m_Document));
        m_Document = NULL;
    }
    m_Document = newDocument;
    // Load the document's information and outline.
    loadMetadata ();
    PopplerIndexIter *outline = poppler_index_iter_new (m_Document);
    m_Outline = new DocumentOutline ();
    setOutline (m_Outline, outline);

    return TRUE;
}

///
/// @brief Reads the document's meta data.
///
/// After each successful load of a PDF file, its meta data is read and
/// keep in member variables of this class, so a call to get*() function
/// will return it.
///
/// Also resets the rotation degree to 0 and the scale level to 1.0f.
///
void
PDFDocument::loadMetadata (void)
{
    g_assert (NULL != m_Document && "The document has not been loaded.");

    gchar *author = NULL;
    GTime creationDate;
    gchar *creator = NULL;
    gchar *format = NULL;
    gchar *keywords = NULL;
    PopplerPageLayout layout = POPPLER_PAGE_LAYOUT_UNSET;
    gchar *linearized = NULL;
    GTime modDate;
    PopplerPageMode mode = POPPLER_PAGE_MODE_UNSET;
    gchar *producer = NULL;
    gchar *subject = NULL;
    gchar *title = NULL;
   
    g_object_get (m_Document,
            "author", &author,
            "creation-date", &creationDate,
            "creator", &creator,
            "format", &format,
            "keywords", &keywords,
            "page-layout", &layout,
            "linearized", &linearized,
            "mod-date", &modDate,
            "page-mode", &mode,
            "producer", &producer,
            "subject", &subject,
            "title", &title,
            NULL);
    setAuthor (author);
    if ( 0 < creationDate )
    {
        struct tm *tmpTime = localtime ((const time_t *)&creationDate);
        gchar *date = g_strnfill (DATE_LENGTH + 1, 0);
        strftime (date, DATE_LENGTH, "%Y-%m-%d %H:%M:%S", tmpTime);
        setCreationDate (date);
    }
    else
    {
        setCreationDate (NULL);
    }
    setCreator (creator);
    setFormat (format);
    setKeywords (keywords);
    setLinearized (linearized);
    if ( 0 < modDate )
    {
        struct tm *tmpTime = localtime ((const time_t *)&modDate);
        gchar *date = g_strnfill (DATE_LENGTH + 1, 0);
        strftime (date, DATE_LENGTH, "%Y-%m-%d %H:%M:%S", tmpTime);
        setModifiedDate (date);
    }
    else
    {
        setModifiedDate (NULL);
    }
    setProducer (producer);
    setSubject (subject);
    setTitle (title);

    // For the page mode and layout we need the enumerator value
    GEnumValue *pageLayout = g_enum_get_value (
            (GEnumClass *)g_type_class_peek (POPPLER_TYPE_PAGE_LAYOUT), layout);
    setPageLayout (convertPageLayout (pageLayout->value));
    GEnumValue *pageMode = g_enum_get_value (
            (GEnumClass *)g_type_class_peek (POPPLER_TYPE_PAGE_MODE), mode);
    setPageMode (convertPageMode (pageMode->value));

    // Get the number of pages and set the current to the first.
    setNumPages (poppler_document_get_n_pages (m_Document));
}

///
/// @brief Sets the pages links.
///
/// This function adds all links from a page to the rendered page
/// image of it.
///
/// @param renderedPage The rendered page to add the links to.
/// @param popplerPage The page to get the links from.
///
void
PDFDocument::setLinks (DocumentPage *renderedPage, PopplerPage *popplerPage)
{
    gdouble pageHeight = 1.0;
    // Get the height, to calculate the Y position as the document's origin
    // is at the bottom-left corner, not the top-left as the screen does.
    poppler_page_get_size (popplerPage, NULL, &pageHeight);
    // We'll already calculate the positions scaled.
    gdouble scale = getZoom ();
    GList *pageLinks = poppler_page_get_link_mapping (popplerPage); 
    for (GList *pageLink = g_list_first (pageLinks) ;
         NULL != pageLink ;
         pageLink = g_list_next (pageLink) )
    {
        PopplerLinkMapping *link = (PopplerLinkMapping *)pageLink->data;
        PopplerAction *action = link->action;
        // Only internal links.
        if ( POPPLER_ACTION_GOTO_DEST == action->type )
        {
            PopplerActionGotoDest *actionGoTo = (PopplerActionGotoDest *)action;
            DocumentLink *documentLink = 
                new DocumentLink (link->area.x1 * scale, 
                                  (pageHeight - link->area.y2) * scale,
                                  link->area.x2 * scale,
                                  (pageHeight - link->area.y1) * scale,
                                  actionGoTo->dest->page_num);
            renderedPage->addLink (documentLink);
        }
    }
    poppler_page_free_link_mapping (pageLinks);
}

///
/// @brief Sets the document's outline.
///
/// This is a recursive function that adds child outlines
/// from the PDF's outline to the passed @a outline.
///
/// @param outline The outline to set the nodes to. The first
///                call must be set to the root DocumentOutline.
/// @param childrenList The list of children for to set to @a outline.
///                     The first line must be the returned valued of
///                     poppler_index_iter_new().
///
void
PDFDocument::setOutline (DocumentOutline *outline, 
                         PopplerIndexIter *childrenList)
{
    if ( NULL != childrenList )
    {
        do
        {
            PopplerAction *action = 
                poppler_index_iter_get_action (childrenList);
            if ( POPPLER_ACTION_GOTO_DEST == action->type )
            {
                PopplerActionGotoDest *actionGoTo = 
                    (PopplerActionGotoDest *)action;
                DocumentOutline *child = new DocumentOutline ();
                child->setParent (outline);
                child->setTitle (actionGoTo->title);
                PopplerDest *destination = actionGoTo->dest;
                child->setDestination (destination->page_num);
                outline->addChild (child);
                PopplerIndexIter *childIter = 
                    poppler_index_iter_get_child (childrenList);
                setOutline (child, childIter);
            }
        } 
        while ( poppler_index_iter_next (childrenList) );

        poppler_index_iter_free (childrenList);
    }
}

///
/// @brief Gets a document's page's unscaled size.
///
/// Retrieves the width and height of a document's page before to scale, but
/// after rotation.
/// 
/// @param pageNum The page to get its size.
/// @param width The output pointer to save the page's width.
/// @param height The output pointer to save the page's height.
///
void
PDFDocument::getPageSizeForPage (gint pageNum, gdouble *width, gdouble *height)
{
    g_assert (NULL != m_Document && "Tried to get size of a NULL document.");
    g_assert (NULL != width && "Tried to save the page's width to NULL.");
    g_assert (NULL != height && "Tried to save the page's height to NULL.");

    PopplerPage *page = poppler_document_get_page (m_Document, pageNum - 1);

    gdouble pageWidth;
    gdouble pageHeight;
    // Check which rotation has the document's page to know what is width
    // and what is height.
    gint rotate = getRotation ();
    if ( 90 == rotate || 270 == rotate )
    {
        poppler_page_get_size (page, &pageHeight, &pageWidth);
    }
    else 
    {
        poppler_page_get_size (page, &pageWidth, &pageHeight);
    }

    *width = pageWidth;
    *height = pageHeight;

    g_object_unref (G_OBJECT (page));
}

///
/// @brief Renders a document's page.
///
/// Rendering a document's page means to get the pixels for the page,
/// given the current rotation level and scale.
///
/// @param pageNum The page to render.
///
/// @return A DocumentPage with the image. The returned page must be freed
///         by calling delete.
///
DocumentPage *
PDFDocument::renderPage (gint pageNum)
{
    if ( NULL == m_Document )
    {
        return NULL;
    }
    
    // First create the document's page.
    gdouble pageWidth;
    gdouble pageHeight;
    getPageSizeForPage (pageNum, &pageWidth, &pageHeight);
    gint width = MAX((gint) ((pageWidth * getZoom ()) + 0.5), 1);
    gint height = MAX((gint) ((pageHeight * getZoom ()) + 0.5), 1);
    DocumentPage *renderedPage = new DocumentPage ();
    renderedPage->newPage (width, height);

    // Create the pixbuf from the data and render to it.
    GdkPixbuf *pixbuf = gdk_pixbuf_new_from_data (renderedPage->getData (),
                                                  GDK_COLORSPACE_RGB,
                                                  FALSE,
                                                  PIXBUF_BITS_PER_SAMPLE,
                                                  width, height,
                                                  renderedPage->getRowStride (),
                                                  NULL, NULL);
    PopplerPage *page = poppler_document_get_page (m_Document, pageNum - 1);
    poppler_page_render_to_pixbuf (page, 0, 0, width, height, getZoom (),
                                   getRotation (), pixbuf);
    g_object_unref (pixbuf);
    setLinks (renderedPage, page);
    g_object_unref (G_OBJECT (page));
    
    return (renderedPage);
}

///
/// @brief Gets the document's page layout from Poppler's page layout.
///
/// @param pageLayout Is the page layout that Poppler's glib wrapper gives.
///
/// @return The PageLayout based on @a pageLayout.
///
PageLayout
convertPageLayout (gint pageLayout)
{
    PageLayout layout = PageLayoutUnset;
    switch (pageLayout)
    {
        case POPPLER_PAGE_LAYOUT_SINGLE_PAGE:
            layout = PageLayoutSinglePage;
            break;
        case POPPLER_PAGE_LAYOUT_ONE_COLUMN:
            layout = PageLayoutOneColumn;
            break;
        case POPPLER_PAGE_LAYOUT_TWO_COLUMN_LEFT:
            layout = PageLayoutTwoColumnLeft;
            break;
        case POPPLER_PAGE_LAYOUT_TWO_COLUMN_RIGHT:
            layout = PageLayoutTwoColumnRight;
            break;
        case POPPLER_PAGE_LAYOUT_TWO_PAGE_LEFT:
            layout = PageLayoutTwoPageLeft;
            break;
        case POPPLER_PAGE_LAYOUT_TWO_PAGE_RIGHT:
            layout = PageLayoutTwoPageRight;
            break;
        case POPPLER_PAGE_LAYOUT_UNSET:
        default:
            layout = PageLayoutUnset;
    }

    return layout;
}

///
/// @brief Get the document's page mode.
///
/// @param pageLayout Is the page mode that Poppler's catalog gives.
///
/// @return The PageLayout based on @a pageMode.
///
PageMode
convertPageMode (gint pageMode)
{
    PageMode mode = PageModeUnset;
    switch (pageMode)
    {
        case POPPLER_PAGE_MODE_USE_OUTLINES:
            mode = PageModeOutlines;
            break;
        case POPPLER_PAGE_MODE_USE_THUMBS:
            mode = PageModeThumbs;
            break;
        case POPPLER_PAGE_MODE_FULL_SCREEN:
            mode = PageModeFullScreen;
            break;
        case POPPLER_PAGE_MODE_USE_OC:
            mode = PageModeOC;
            break;
        case POPPLER_PAGE_MODE_USE_ATTACHMENTS:
            mode = PageModeAttach;
            break;
        case POPPLER_PAGE_MODE_NONE:
        case POPPLER_PAGE_MODE_UNSET:
        default:
            mode = PageModeUnset;
    }

    return mode;
}

///
/// @brief Gets the absolute path of a filename.
///
/// This function checks if the given @a fileName is an absolute path. If
/// it is then it returns a copy of it, otherwise it prepends the current
/// working directory to it.
///
/// @param fileName The filename to get the absolute path from.
///
/// @return A copy of the absolute path to the file name. This copy must be
///         freed when no longer needed.
///
gchar *
getAbsoluteFileName (const gchar *fileName)
{
    gchar *absoluteFileName = NULL;
    if ( g_path_is_absolute (fileName) )
    {
        absoluteFileName = g_strdup (fileName);
    }
    else
    {
        gchar *currentDir = g_get_current_dir ();
        absoluteFileName = g_build_filename (currentDir, fileName, NULL);
        g_free (currentDir);
    }

    return absoluteFileName;
}
