# -*- coding: utf-8 -*-

# Copyright (c) 2002 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#
#
# It is inspired by the cvs log form code found in kdevelop 3.0
#

"""
Module implementing a dialog to show the output of the cvs log command process.
"""

import os

from qt import *

from LogForm import LogForm
from DiffDialog import CvsDiffDialog

import Utilities

class CvsLogDialog(LogForm):
    """
    Module implementing a dialog to show the output of the cvs log command process.
    
    The dialog is nonmodal. Clicking a link in the upper text pane shows 
    a diff of the versions.
    """
    def __init__(self, vcs, parent = None):
        """
        Constructor
        
        @param vcs reference to the vcs object
        @param parent parent widget (QWidget)
        """
        LogForm.__init__(self, parent)
        
        self.saveButton.hide()
        
        QWhatsThis.add(self.contents,self.tr("<b>CVS Log</b>\n"
            "<p>This shows the output of the cvs log command."
            " By clicking on the links you may show the difference"
            " between versions.</p>"
        ))
        QWhatsThis.add(self.errors,self.tr("<b>CVS log errors</b>\n"
            "<p>This shows possible error messages of the cvs log"
            " command.</p>"
        ))
        
        self.setWFlags(self.getWFlags() | Qt.WDestructiveClose)
        self.process = QProcess(self)
        self.vcs = vcs
        
        self.connect(self.process, SIGNAL('readyReadStdout()'),
            self.handleReadStdout)
        self.connect(self.process, SIGNAL('readyReadStderr()'),
            self.handleReadStderr)
        self.connect(self.process, SIGNAL('processExited()'),
            self.handleProcessExited)
        self.disconnect(self.contents, SIGNAL('linkClicked(const QString&)'),
            self.contents, SLOT('setSource(const QString&)'))
        self.connect(self.contents, SIGNAL('linkClicked(const QString&)'),
            self.handleLinkClicked)
        self.connect(self.contents, SIGNAL('anchorClicked(const QString&, const QString&)'),
            self.handleAnchorClicked)
            
        self.rx_sep = QRegExp('\\-+')
        self.rx_sep2 = QRegExp('=+')
        self.rx_date = QRegExp('date: .* author: .* state: .* lines: .*')
        self.rx_rev = QRegExp('revision ((\\d+\\.?)+)')
        # "revision" followed by one or more decimals followed by an optional dot
        self.rx_file = QRegExp('Working file: (.*)')
        # "Working file: " followed by the file name
        
        self.revisions = QStringList() # stack of remembered revisions
        self.buf = QStringList()    # buffer for stdout
        
    def closeEvent(self, e):
        """
        Private slot implementing a close event handler.
        
        @param e close event (QCloseEvent)
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        e.accept()
        
    def start(self, fn):
        """
        Public slot to start the cvs log command.
        
        @param fn filename to be diffed (string)
        """
        self.intercept = 0
        self.filename = fn
        self.dname, self.fname = self.vcs.splitPath(fn)
        
        self.process.kill()
        self.contents.clear()
        
        self.process.clearArguments()
        self.process.addArgument('cvs')
        self.vcs.addArguments(self.process, self.vcs.options['global'])
        self.process.addArgument('log')
        self.vcs.addArguments(self.process, self.vcs.options['log'])
        self.process.addArgument(self.fname)
        self.process.setWorkingDirectory(QDir(self.dname))
        
        self.process.start()
        self.setCaption(self.trUtf8('CVS Log %1').arg(self.filename))
        
    def handleProcessExited(self):
        """
        Private slot to handle the processExited signal.
        
        After the process has exited, diff links are inserted into the contents pane.
        """
        self.closeButton.setFocus()
        self.inputGroup.setEnabled(0)
        
        self.contents.clear()
        lvers = 1
        for s in self.buf:
            if self.rx_file.exactMatch(s):
                self.fname = self.rx_file.cap(1)
                self.contents.append(s)
            elif self.rx_rev.exactMatch(s):
                ver = self.rx_rev.cap(1)
                dstr = QString('<b>%s</b' % unicode(s))
                try:
                    lv = self.revisions[lvers]
                    lvers += 1
                except IndexError:
                    lv = None
                
                if lv is not None:
                    url = QUrl()
                    url.setProtocol("file")
                    url.setFileName(self.filename)
                    query = QString()
                    query.append(lv).append('_').append(ver)
                    url.setQuery(query)
                    dstr.append(' [<a href="')\
                        .append(url.toString())\
                        .append('" name="')\
                        .append(query)\
                        .append('">')\
                        .append(self.trUtf8('diff to %1').arg(lv))\
                        .append('</a>]')
                    
                self.contents.append(dstr)
                
            elif self.rx_date.exactMatch(s):
                self.contents.append('<i>%s</i>' % unicode(s))
                
            elif self.rx_sep.exactMatch(s) or self.rx_sep2.exactMatch(s):
                self.contents.append('<hr>')
                
            else:
                self.contents.append(Utilities.html_encode(unicode(s)))
        
        self.contents.setCursorPosition(0, 0)
        self.contents.ensureCursorVisible()
        
    def handleReadStdout(self):
        """
        Private slot to handle the readyReadStdout signal.
        
        It reads the output of the process and inserts it into a buffer.
        """
        while self.process.canReadLineStdout():
            s = self.process.readLineStdout()
            self.buf.append(s)
            if self.rx_rev.exactMatch(s):
                ver = self.rx_rev.cap(1)
                # save revision number for later use
                self.revisions.append(ver)
        
    def handleReadStderr(self):
        """
        Private slot to handle the readyReadStderr signal. 
        
        It reads the error output of the process and inserts it into the
        error pane.
        """
        if self.process is not None:
            s = unicode(self.process.readStderr())
            self.errors.moveCursor(QTextEdit.MoveEnd, 0)
            self.errors.insert(s)
        
    def handleLinkClicked(self, link):
        """
        Private slot to handle the linkClicked signal of the contents pane.
        
        @param link the link that was clicked (QString)
        """
        self.contents.setSource('')
        link = QUrl(link)
        filename = link.path()
        ver = link.query()
        v1 = ver.section('_', 0, 0)
        v2 = ver.section('_', 1, 1)
        if v1.isEmpty() or v2.isEmpty():
            return
            
        self.diff = CvsDiffDialog(self.vcs)
        self.diff.show()
        self.diff.start(filename, [v1, v2])
        
    def handleAnchorClicked(self, name, link):
        """
        Private slot to handle the anchorClicked signal of the contents pane.
        
        @param name name of the anchor that was clicked (QString)
        @param link the link that was clicked (QString)
        """
        self.contents.scrollToAnchor(name)
        
    def passwordMode(self, isOn):
        """
        Private slot to handle the password checkbox toggled.
        
        @param isOn flag indicating the status of the check box (boolean)
        """
        if isOn:
            self.input.setEchoMode(QLineEdit.Password)
        else:
            self.input.setEchoMode(QLineEdit.Normal)
        
    def sendInput(self):
        """
        Private slot to send the input to the subversion process.
        """
        input = self.input.text()
        input.append(os.linesep)
        
        self.errors.moveCursor(QTextEdit.MoveEnd, 0)
        if self.passwordCheckBox.isChecked():
            self.errors.insert(os.linesep)
        else:
            self.errors.insert(input)
        
        self.proc.writeToStdin(input)
        
        self.passwordCheckBox.setChecked(0)
        self.input.clear()
        
    def returnPressed(self):
        """
        Private slot to handle the press of the return key in the input field.
        """
        self.intercept = 1
        self.sendInput()
        
    def keyPressEvent(self, evt):
        """
        Protected slot to handle a key press event.
        
        @param evt the key press event (QKeyEvent)
        """
        if self.intercept:
            self.intercept = 0
            evt.accept()
            return
        LogForm.keyPressEvent(self, evt)
