# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Package implementing various functions/classes needed everywhere within eric3. 
"""

import os
import sys
import re
import string
import fnmatch

from qt import QRegExp, QString, QMimeSourceFactory, QTranslator, QDir

from eric3config import getConfig

coding_re = re.compile(r"coding[:=]\s*([-\w_.]+)")
UTF8_BOM = '\xef\xbb\xbf'
codecs = ['utf-8', 'iso8859-1',  'iso8859-15', 'koi8-r', 'koi8-u',
          'iso8859-2', 'iso8859-3', 'iso8859-4', 'iso8859-5', 
          'iso8859-6', 'iso8859-7', 'iso8859-8', 'iso8859-9', 
          'iso8859-10', 'iso8859-13', 'iso8859-14', 'latin-1', 
          'utf-16']

def get_coding(text):
    """
    Function to get the coding of a text.
    
    @param text text to inspect (string)
    @return coding string
    """
    for l in text.splitlines()[:2]:
        m = coding_re.search(l)
        if m:
            return m.group(1)
    return None

def decode(text):
    """
    Function to decode a text.
    
    @param text text to decode (string)
    @return decoded text
    """
    try:
        if text.startswith(UTF8_BOM):
            # UTF-8 with BOM
            return unicode(text[3:], 'utf-8'), 'utf-8-bom'
        coding = get_coding(text)
        if coding:
            return unicode(text, coding), coding
    except (UnicodeError, LookupError):
        pass
    # Assume Latin-1
    return unicode(text, "latin-1"), 'latin-1-guessed'

def encode(text, orig_coding):
    """
    Function to encode a text.
    
    @param text text to encode (string)
    @param orig_coding type of the original coding (string)
    @return encoded text
    """
    if orig_coding == 'utf-8-bom':
        return UTF8_BOM + text.encode("utf-8")
    # Try saving as ASCII
    try:
        return text.encode('ascii')
    except UnicodeError:
        pass
    # Try declared coding spec
    coding = get_coding(text)
    try:
        if coding:
            return text.encode(coding)
    except (UnicodeError, LookupError):
        # Error: Declared encoding is incorrect
        pass
    try:
        if orig_coding == 'latin-1-guessed':
            return text.encode('latin-1')
    except UnicodeError:
        pass
    # Save as UTF-8 without BOM
    return text.encode('utf-8')
    
def toUnicode(s):
    """
    Private method to convert a string to unicode.
    
    If the passed in string is of type QString, it is
    simply returned unaltered, assuming, that it is already
    a unicode string. For all other strings, various codes
    are tried until one converts the string without an error.
    If all codecs fail, the string is returned unaltered.
    
    @param s string to be converted (string or QString)
    @return converted string (unicode or QString)
    """
    if isinstance(s, QString):
        return s
    
    if type(s) is type(u""):
        return s
    
    for codec in codecs:
        try:
            u = unicode(s, codec)
            return u
        except UnicodeError:
            pass
        except TypeError:
            break
    
    # we didn't succeed
    return s
    
def escape(data):
    """
    Function to escape &, <, and > in a string of data.
    
    @param data data to be escaped (string)
    @return the escaped data (string)
    """

    # must do ampersand first
    data = data.replace("&", "&amp;")
    data = data.replace(">", "&gt;")
    data = data.replace("<", "&lt;")
    return data

_escape = re.compile(eval(r'u"[&<>\"\u0080-\uffff]"'))

_escape_map = {
    "&": "&amp;",
    "<": "&lt;",
    ">": "&gt;",
    '"': "&quot;",
}

def escape_entities(m, map=_escape_map):
    """
    Function to encode html entities.
    
    @param m the match object
    @param map the map of entioties to encode
    @return the converted text (string)
    """
    char = m.group()
    text = map.get(char)
    if text is None:
        text = "&#%d;" % ord(char)
    return text
    
def html_encode(text, pattern=_escape):
    """
    Function to correctly encode a text for html.
    
    @param text text to be encoded (string)
    @param pattern search pattern for text to be encoded (string)
    @return the encoded text (string)
    """
    if not text:
        return ""
    text = pattern.sub(escape_entities, text)
    return text.encode("ascii")

_uescape = re.compile(eval(r'u"[\u0080-\uffff]"'))

def escape_uentities(m):
    """
    Function to encode html entities.
    
    @param m the match object
    @return the converted text (string)
    """
    char = m.group()
    text = "&#%d;" % ord(char)
    return text
    
def html_uencode(text, pattern=_uescape):
    """
    Function to correctly encode a unicode text for html.
    
    @param text text to be encoded (string)
    @param pattern search pattern for text to be encoded (string)
    @return the encoded text (string)
    """
    if not text:
        return ""
    text = unicode(text)
    text = pattern.sub(escape_uentities, text)
    return text.encode("ascii")

def normcasepath(path):
    """
    Function returning a path, that is normalized with respect to its case and references.
    
    @param path file path (string)
    @return case normalized path (string)
    """
    return os.path.normcase(os.path.normpath(path))
    
def normabspath(path):
    """
    Function returning a normalized, absolute path.
    
    @param path file path (string)
    @return absolute, normalized path (string)
    """
    return os.path.abspath(path)
    
def normjoinpath(a, *p):
    """
    Function returning a normalized path of the joined parts passed into it.
    
    @param a first path to be joined (string)
    @param p variable number of path parts to be joind (string)
    @return normalized path (string)
    """
    return os.path.normpath(os.path.join(a, *p))
    
def normabsjoinpath(a, *p):
    """
    Function returning a normalized, absolute path of the joined parts passed into it.
    
    @param a first path to be joined (string)
    @param p variable number of path parts to be joind (string)
    @return absolute, normalized path (string)
    """
    return os.path.abspath(os.path.join(a, *p))
    
def isinpath(file):
    """
    Function to check for an executable file.
    
    @param file filename of the executable to check (string)
    @return flag to indicate, if the executable file is accessible
        via the searchpath defined by the PATH environment variable.
    """
    if os.path.isabs(file):
        return os.access(file, os.X_OK)
        
    path = os.getenv('PATH')
    
    # environment variable not defined
    if path is None:
        return 0
        
    dirs = string.split(path, os.pathsep)
    for dir in dirs:
        if os.access(os.path.join(dir, file), os.X_OK):
            return 1
            
    return 0
    
def getExecutablePath(file):
    """
    Function to build the full path of an executable file from the environment.
    
    @param file filename of the executable to check (string)
    @return full executable name, if the executable file is accessible
        via the searchpath defined by the PATH environment variable, or an
        empty string otherwise.
    """
    if os.path.isabs(file):
        if os.access(file, os.X_OK):
            return file
        else:
            return ""
        
    path = os.getenv('PATH')
    
    # environment variable not defined
    if path is None:
        return ""
        
    dirs = string.split(path, os.pathsep)
    for dir in dirs:
        exe = os.path.join(dir, file)
        if os.access(exe, os.X_OK):
            return exe
            
    return ""
    
def samepath(f1, f2):
    """
    Function to compare two paths.
    
    @param f1 first path for the compare (string)
    @param f2 second path for the compare (string)
    @return flag indicating whether the two paths represent the
        same path on disk.
    """
    if f1 is None or f2 is None:
        return 0
        
    if normcasepath(f1) == normcasepath(f2):
        return 1
        
    return 0
        
try:
    EXTSEP = os.extsep
except AttributeError:
    EXTSEP = "."

def splitPath(name):
    """
    Function to split a pathname into a directory part and a file part.
    
    @param name path name (string)
    @return a tuple of 2 strings (dirname, filename).
    """
    fi = unicode(name)
    if os.path.isdir(fi):
        dn = os.path.abspath(fi)
        fn = "."
    else:
        dn, fn = os.path.split(fi)
    return (dn, fn)

def joinext(prefix, ext):
    """
    Function to join a file extension to a path.
    
    The leading "." of ext is replaced by a platform specific extension
    separator if neccessary.
    
    @param prefix the basepart of the filename (string)
    @param ext the extension part (string)
    @return the complete filename (string)
    """
    if ext[0] != ".":
        ext = ".%s" % ext # require leading separator, to match os.path.splitext
    return prefix + EXTSEP + ext[1:]

def direntries(path, filesonly=0, pattern=None, followsymlinks=1):
    """
    Function returning a list of all files and directories.
    
    @param path root of the tree to check
    @param filesonly flag indicating that only files are wanted
    @param pattern a filename pattern to check against
    @param followsymlinks flag indicating whether symbolic links
            should be followed
    @return list of all files and directories in the tree rooted
        at path. The names are expanded to start with path. 
    """
    if filesonly:
        files = []
    else:
        files = [path]
    entries = os.listdir(path)
    for entry in entries:
        if entry in ['CVS', '.svn']:
            continue
            
        fentry = os.path.join(path, entry)
        if pattern and \
        not os.path.isdir(fentry) and \
        not fnmatch.fnmatch(entry, pattern):
            # entry doesn't fit the given pattern
            continue
            
        if os.path.isdir(fentry):
            if os.path.islink(fentry) and not followsymlinks:
                continue
            files += direntries(fentry, filesonly, pattern)
        else:
            files.append(fentry)
    return files

def getDirs(path, excludeDirs):
    """
    Function returning a list of all directories below path.
    
    @param path root of the tree to check
    @param excludeDirs basename of directories to ignore
    @return list of all directories found
    """
    try:
        names = os.listdir(path)
    except:
        return

    dirs = []
    for name in names:
        if os.path.isdir(os.path.join(path, name)) and \
          not os.path.islink(os.path.join(path, name)):
            exclude = 0
            for e in excludeDirs:
                if name.split(os.sep,1)[0] == e:
                    exclude = 1
                    break
            if not exclude:
                dirs.append(os.path.join(path, name))

    for name in dirs[:]:
        if not os.path.islink(name):
            dirs = dirs + getDirs(name, excludeDirs)

    return dirs

def getTestFileName(fn):
    """
    Function to build the filename of a unittest file.
    
    The filename for the unittest file is built by prepending
    the string "test" to the filename passed into this function.
    
    @param fn filename basis to be used for the unittest filename (string)
    @return filename of the corresponding unittest file (string)
    """
    dn, fn = os.path.split(fn)
    return os.path.join(dn, "test%s" % fn)
    
def parseOptionString(s):
    """
    Function used to convert an option string into a list of options.
    
    @param s option string (string or QString)
    @return list of options (list of strings)
    """
    rx = QRegExp(r"""\s([^\s]+|"[^"]+"|'[^']+')""")
    return parseString(s, rx)
    
def parseEnvironmentString(s):
    """
    Function used to convert an environment string into a list of environment settings.
    
    @param s environment string (string or QString)
    @return list of environment settings (list of strings)
    """
    rx = QRegExp(r"""\s(\w+=[^\s]+|\w+="[^"]+"|\w+='[^']+')""")
    return parseString(s, rx)

def parseString(s, rx):
    """
    Function used to convert a string into a list.
    
    @param s string to be parsed (string or QString)
    @param rx regex defining the parse pattern (QRegExp)
    @return list of parsed data (list of strings)
    """
    olist = []
    qs = QString(s)
    if not qs.startsWith(' '):
        # prepare the  string to fit our pattern
        qs = qs.prepend(' ')
        
    pos = rx.search(qs)
    while pos != -1:
        cs = unicode(rx.cap(1))
        if cs.startswith('"') or cs.startswith("'"):
            cs = cs[1:-1]
        olist.append(cs)
        pos += rx.matchedLength()
        pos = rx.search(qs, pos)
        
    return olist

def getPythonLibPath():
    """
    Function to determine the path to Pythons library.
    
    @return path to the Python library (string)
    """
    pyFullVers = string.split(sys.version)[0]

    vl = string.split(re.findall("[0-9.]*",pyFullVers)[0],".")
    major = vl[0]
    minor = vl[1]

    pyVers = major + "." + minor
    pyVersNr = int(major) * 10 + int(minor)

    if sys.platform == "win32":
        libDir = sys.prefix + "\\Lib"
    else:
        libDir = sys.prefix + "/lib/python" + pyVers
        
    return libDir
    
def getPythonVersion():
    """
    Function to get the Python version (major, minor) as an integer value.
    
    @return An integer representing major and minor version number (integer)
    """
    return sys.hexversion >> 16
    
def compile(file):
    """
    Function to compile one Python source file to Python bytecode.
    
    @param file source filename (string)
    @return A tuple indicating status (1 = an error was found), the
        filename, the linenumber, the code string and the error message
        (boolean, string, string, string, string). The values are only
        valid, if the status equals 1.
    """
    import __builtin__
    f = open(file)
    codestring = f.read()
    # If parsing from a string, line breaks are \n (see parsetok.c:tok_nextc)
    # Replace will return original string if pattern is not found, so
    # we don't need to check whether it is found first.
    codestring = codestring.replace("\r\n","\n")
    codestring = codestring.replace("\r","\n")
    f.close()
    if codestring and codestring[-1] != '\n':
        codestring = codestring + '\n'
        
    try:
        if file.endswith('.ptl'):
            try:
                import quixote.ptl_compile
            except ImportError:
                return (0, None, None, None, None)
            template = quixote.ptl_compile.Template(codestring, str(file))
            template.compile()
            codeobject = template.code
        else:
            codeobject = __builtin__.compile(codestring, file, 'exec')
    except SyntaxError, detail:
        import traceback, re
        lines = traceback.format_exception_only(SyntaxError, detail)
        match = re.match('\s*File "(.+)", line (\d+)', 
            lines[0].replace('<string>', '%s' % file))
        fn, line = match.group(1, 2)
        code = re.match('(.+)', lines[1]).group(1)
        error = ""
        for seLine in lines[2:]:
            if seLine.startswith('SyntaxError:'):
                error = re.match('SyntaxError: (.+)', seLine).group(1)
        return (1, fn, line, code, error)
        
    return (0, None, None, None, None)

def getConfigDir():
    """
    Module function to get the name of the directory storing the config data.
    
    @return directory name of the config dir (string)
    """
    if sys.platform == "win32":
        cdn = "_eric3"
    else:
        cdn = ".eric3"
        
    hp = QDir.homeDirPath()
    dn = QDir(hp)
    dn.mkdir(cdn)
    hp.append("/").append(cdn)
    return unicode(QDir.convertSeparators(hp))
    
def win32_Kill(pid):
    """
    Function to provide an os.kill equivalent for Win32.
    
    @param pid process id
    """
    import win32api
    handle = win32api.OpenProcess(1, 0, pid)
    return (0 != win32api.TerminateProcess(handle, 0))
