# -*- coding: utf-8 -*-

# Copyright (c) 2004 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the writer class for writing an XML project session file.
"""

import os
import time

from qt import qApp

from Utilities import escape
from XMLWriterBase import XMLWriterBase
from Config import sessionFileFormatVersion

import Preferences

class SessionWriter(XMLWriterBase):
    """
    Class implementing the writer class for writing an XML project session file..
    """
    def __init__(self, file, projectName):
        """
        Constructor
        
        @param file open file (like) object for writing
        @param projectName name of the project (string)
        """
        XMLWriterBase.__init__(self, file)
        
        self.name = projectName
        self.project = qApp.mainWidget().getProject()
        self.vm = qApp.mainWidget().getViewManager()
        self.dbg = qApp.mainWidget().getDebugger()
        
    def writeXML(self):
        """
        Public method to write the XML to the file.
        """
        XMLWriterBase.writeXML(self)
        
        self._write('<!DOCTYPE Session SYSTEM "Session-%s.dtd">' % sessionFileFormatVersion)
        
        # add some generation comments
        self._write("<!-- eric3 session file for project %s -->" % self.name)
        self._write("<!-- This file was generated automatically, do not edit. -->")
        self._write("<!-- Saved: %s -->" % time.strftime('%Y-%m-%d, %H:%M:%S'))
        
        # add the main tag
        self._write('<Session version="%s">' % sessionFileFormatVersion) 
        
        # step 1: save all open project filenames and the active window
        allOpenFiles = self.vm.getOpenFilenames()
        self._write("  <Filenames>")
        for of in allOpenFiles:
            if unicode(of).startswith(self.project.ppath):
                ed = self.vm.getOpenEditor(of)
                if ed is not None:
                    line, index = ed.getCursorPosition()
                else:
                    line, index = 0, 0
                self._write('    <Filename cline="%d" cindex="%d">%s</Filename>' % \
                    (line, index, of))
        self._write("  </Filenames>")
        
        aw = self.vm.getActiveName()
        if aw and unicode(aw).startswith(self.project.ppath):
            ed = self.vm.getOpenEditor(aw)
            if ed is not None:
                line, index = ed.getCursorPosition()
            else:
                line, index = 0, 0
            self._write('  <ActiveWindow cline="%d" cindex="%d">%s</ActiveWindow>' % \
                (line, index, aw))
        
        # step 2: save all breakpoints
        if Preferences.getProject("SessionAllBreakpoints"):
            bpDict = self.dbg.getAllBreakpoints()
        else:
            bpDict = self.dbg.getProjectBreakpoints()
        self._write("  <Breakpoints>")
        for fname, bpList in bpDict.items():
            for (lineno, cond, temp, enabled, count) in bpList:
                self._write("    <Breakpoint>")
                self._write("      <BpFilename>%s</BpFilename>" % fname)
                self._write('      <Linenumber value="%d" />' % lineno)
                self._write("      <Condition>%s</Condition>" % escape(str(cond)))
                self._write('      <Temporary value="%d" />' % temp)
                self._write('      <Enabled value="%d" />' % enabled)
                self._write('      <Count value="%d" />' % count)
                self._write("    </Breakpoint>")
        self._write("  </Breakpoints>")
        
        # step 3: save the debug info
        self._write("  <DebugInfo>")
        self._write("    <CommandLine>%s</CommandLine>" % escape(str(self.project.dbgCmdline)))
        self._write("    <WorkingDirectory>%s</WorkingDirectory>" % self.project.dbgWd)
        self._write("    <Environment>%s</Environment>" % self.project.dbgEnv)
        self._write('    <ReportExceptions value="%d" />' % self.project.dbgReportExceptions)
        self._write("    <Exceptions>")
        for exc in self.project.dbgExcList:
            self._write("      <Exception>%s</Exception>" % exc)
        self._write("    </Exceptions>")
        self._write('    <TracePython value="%d" />' % self.project.dbgTracePython)
        self._write("    <CovexcPattern>%s</CovexcPattern>" % escape(self.project.dbgCovexcPattern))
        self._write("  </DebugInfo>")
        
        # step 4: save bookmarks of all open project files
        self._write("  <Bookmarks>")
        for of in allOpenFiles:
            if unicode(of).startswith(self.project.ppath):
                editor = self.vm.getOpenEditor(of)
                for bookmark in editor.getBookmarks():
                    self._write("    <Bookmark>")
                    self._write("      <BmFilename>%s</BmFilename>" % of)
                    self._write('      <Linenumber value="%d" />' % bookmark)
                    self._write("    </Bookmark>")
        self._write("  </Bookmarks>")
        
        self._write("</Session>")
