# -*- coding: utf-8 -*-

# Copyright (c) 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog for entering multiple template variables.
"""

from qt import *

class TemplateMultipleVariablesDialog(QDialog):
    """
    Class implementing a dialog for entering multiple template variables.
    """
    def __init__(self, variables, parent = None):
        """
        Constructor
        
        @param variables list of template variable names (list of strings)
        @param parent parent widget of this dialog (QWidget)
        """
        QDialog.__init__(self, parent, None, 1)

        TemplateMultipleVariablesDialogLayout = \
            QVBoxLayout(self, 6, 6,"TemplateMultipleVariablesDialogLayout")

        # generate the scrollview
        self.variablesView = QScrollView(self, "variablesView")
        self.variablesView.setFrameStyle(QFrame.NoFrame)
        TemplateMultipleVariablesDialogLayout.addWidget(self.variablesView)
        
        self.variablesView.setResizePolicy(QScrollView.AutoOneFit)
        
        top = QWidget(self.variablesView.viewport())
        self.variablesView.addChild(top)
        grid = QGridLayout(top, len(variables), 2)
        grid.setMargin(6)
        grid.setSpacing(6)

        # populate the scrollview with labels and text edits
        self.variablesEntries = {}
        row = 0
        for var in variables:
            l = QLabel("%s:" % var, top)
            grid.addWidget(l, row, 0, Qt.AlignTop)
            if var.find(":") >= 0:
                formatStr = var[1:-1].split(":")[1]
                if formatStr in ["ml"]:
                    t = QTextEdit(top)
                    t.setTabChangesFocus(1)
                else:
                    t = QLineEdit(top)
            else:
                t = QLineEdit(top)
            grid.addWidget(t, row, 1)
            self.variablesEntries[var] = t
            row += 1
        self.variablesEntries[variables[0]].setFocus()

        # generate the buttons
        layout1 = QHBoxLayout(None,0,6,"layout1")
        spacer1 = QSpacerItem(40,20,QSizePolicy.Expanding,QSizePolicy.Minimum)
        layout1.addItem(spacer1)

        self.okButton = QPushButton(self,"okButton")
        self.okButton.setDefault(1)
        layout1.addWidget(self.okButton)

        self.cancelButton = QPushButton(self,"cancelButton")
        layout1.addWidget(self.cancelButton)
        spacer2 = QSpacerItem(40,20,QSizePolicy.Expanding,QSizePolicy.Minimum)
        layout1.addItem(spacer2)
        TemplateMultipleVariablesDialogLayout.addLayout(layout1)

        # set the texts of the standard widgets
        self.setCaption(self.trUtf8("Enter Template Variables"))
        self.okButton.setText(self.trUtf8("&OK"))
        self.okButton.setAccel(self.trUtf8("Alt+O"))
        self.cancelButton.setText(self.trUtf8("&Cancel"))
        self.cancelButton.setAccel(self.trUtf8("Alt+C"))

        # polish up the dialog
        self.resize(QSize(400,480).expandedTo(self.minimumSizeHint()))
        self.clearWState(Qt.WState_Polished)

        self.connect(self.okButton,SIGNAL("clicked()"),self.accept)
        self.connect(self.cancelButton,SIGNAL("clicked()"),self.reject)

    def getVariables(self):
        """
        Public method to get the values for all variables.
        
        @return dictionary with the variable as a key and its value (string)
        """
        values = {}
        for var, textEdit in self.variablesEntries.items():
            values[var] = unicode(textEdit.text())
        return values
