# -*- coding: utf-8 -*-

# Copyright (c) 2004 - 2006 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Compatibility module to use the KDE Message Box instead of the Qt Message Box.
"""

try:
    import os
    if os.getenv("e3nokde"):
        raise ImportError
        
    from kdeui import KMessageBox, KGuiItem
    from qt import QString
    
    def information(parent, caption, text, button0Text = QString.null,
                    button1Text = QString.null, button2Text = QString.null,
                    defaultButton = 0, escapeButton = -1):
        """
        Function to show a modal information message box.
        
        @param parent parent widget of the message box
        @param caption caption of the message box
        @param text text to be shown by the message box
        @param button0Text text of the first button (ignored, it is always 'OK') 
        @param button1Text text of the second button (ignored)
        @param button2Text text of the third button (ignored)
        @param defaultButton index of the default button (0, 1 or 2) (ignored)
        @param escapeButton index of the escape button (-1, 0, 1 or 2) (ignored)
        @return number of the button that was pressed (integer, always 0)
        """
        KMessageBox.information(parent, text, caption)
        return 0
    
    def warning(parent, caption, text, button0Text = QString.null,
                button1Text = QString.null, button2Text = QString.null,
                defaultButton = 0, escapeButton = -1):
        """
        Function to show a modal warning message box.
        
        @param parent parent widget of the message box
        @param caption caption of the message box
        @param text text to be shown by the message box
        @param button0Text text of the first button 
        @param button1Text text of the second button
        @param button2Text text of the third button
        @param defaultButton index of the default button (0, 1 or 2) (ignored)
        @param escapeButton index of the escape button (-1, 0, 1 or 2) (ignored)
        @return number of the button that was pressed (integer, always 0)
        """
        if button0Text is QString.null or button1Text is QString.null:
            KMessageBox.sorry(parent, text, caption)
            return 0
            
        if button2Text is QString.null:
            res = KMessageBox.warningYesNo(parent, text, caption,
                    KGuiItem(button0Text), KGuiItem(button1Text))
            if res == KMessageBox.Yes:
                return 0
            else:
                return 1
            
        res = KMessageBox.warningYesNoCancel(parent, text, caption,
                KGuiItem(button0Text), KGuiItem(button1Text)) # button 3 is always 'Cancel'
        if res == KMessageBox.Yes:
            return 0
        elif res == KMessageBox.No:
            return 1
        else:
            return 2
    
    def critical(parent, caption, text, button0Text = QString.null,
                 button1Text = QString.null, button2Text = QString.null,
                 defaultButton = 0, escapeButton = -1):
        """
        Function to show a modal critical message box.
        
        @param parent parent widget of the message box
        @param caption caption of the message box
        @param text text to be shown by the message box
        @param button0Text text of the first button 
        @param button1Text text of the second button
        @param button2Text text of the third button
        @param defaultButton index of the default button (0, 1 or 2) (ignored)
        @param escapeButton index of the escape button (-1, 0, 1 or 2) (ignored)
        @return number of the button that was pressed (integer, always 0)
        """
        return warning(parent, caption, text, button0Text, button1Text,
                       button2Text, defaultButton, escapeButton)
    
    def question(parent, caption, text, button0Text = QString.null,
                 button1Text = QString.null, button2Text = QString.null,
                 defaultButton = 0, escapeButton = -1):
        """
        Function to show a modal question message box.
        
        @param parent parent widget of the message box
        @param caption caption of the message box
        @param text text to be shown by the message box
        @param button0Text text of the first button 
        @param button1Text text of the second button
        @param button2Text text of the third button
        @param defaultButton index of the default button (0, 1 or 2) (ignored)
        @param escapeButton index of the escape button (-1, 0, 1 or 2) (ignored)
        @return number of the button that was pressed (integer, always 0)
        """
        if button0Text is QString.null or button1Text is QString.null:
            KMessageBox.questionYesNo(parent, text, caption)
            return 0
            
        if button2Text is QString.null:
            res = KMessageBox.questionYesNo(parent, text, caption,
                    KGuiItem(button0Text), KGuiItem(button1Text))
            if res == KMessageBox.Yes:
                return 0
            else:
                return 1
            
        res = KMessageBox.questionYesNoCancel(parent, text, caption,
                KGuiItem(button0Text), KGuiItem(button1Text)) # button 3 is always 'Cancel'
        if res == KMessageBox.Yes:
            return 0
        elif res == KMessageBox.No:
            return 1
        else:
            return 2
        
    def about(parent, caption, text):
        """
        Function to show a modal about message box.
        
        @param parent parent widget of the message box
        @param caption caption of the message box
        @param text text to be shown by the message box
        """
        from qt import QStringList, qApp
        from kdeui import KDialogBase, KStdGuiItem
        _caption = QString(caption)
        if _caption.isEmpty():
            _caption = qApp.translate("KQMessageBox", "About")
        
        try:
            dialog = KDialogBase(
                caption,
                KDialogBase.Yes,
                KDialogBase.Yes,
                KDialogBase.Yes,
                parent,
                "about", 1, 1, KStdGuiItem.ok())
            
            ret = parent.icon()
            if ret.isNull():
                ret = qApp.mainWidget().icon()
            dialog.setIcon(ret)
            KMessageBox.createKMessageBox(\
                dialog, ret, text, QStringList(), QString.null, 0) 
        except AttributeError:
            KMessageBox.about(parent, text, caption)

except (ImportError, RuntimeError):
    import qt
    
    information = qt.QMessageBox.information
    warning = qt.QMessageBox.warning
    critical = qt.QMessageBox.critical
    try:
        question = qt.QMessageBox.question
    except:
        question = qt.QMessageBox.information
    about = qt.QMessageBox.about
