# -*- coding: utf-8 -*-

# Copyright (c) 2005 - 2007 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to show the output of the packager process.
"""

import os.path

from qt import *

from KdeQt import KQMessageBox

from PackagersExecForm import PackagersExecForm

class PackagersExecDialog(PackagersExecForm):
    """
    Module implementing a dialog to show the output of the packager process.
    
    This class starts a QProcess and displays a dialog that
    shows the output of the packager command process.
    """
    def __init__(self, cmdname, parent = None):
        """
        Constructor
        
        @param cmdname name of the packager (string)
        @param parent parent widget of this dialog (QWidget)
        """
        PackagersExecForm.__init__(self, parent)
        
        self.process = None
        self.cmdname = cmdname
        
    def start(self, args, fn):
        """
        Public slot to start the packager command.
        
        @param args commandline arguments for packager program (QStringList)
        @param fn main script name to be processed by by the packager
            (string or QString)
        @return flag indicating the successful start of the process
        """
        script = unicode(fn)
        dname = os.path.dirname(script)
        script = os.path.basename(script)
        
        self.contents.clear()
        self.errors.clear()
        
        self.process = QProcess()
        self.process.setArguments(args)
        self.process.addArgument(script)
        self.process.setWorkingDirectory(QDir(dname))
        
        self.connect(self.process, SIGNAL('readyReadStdout()'),
            self.handleReadStdout)
        self.connect(self.process, SIGNAL('readyReadStderr()'),
            self.handleReadStderr)
        self.connect(self.process, SIGNAL('processExited()'),
            self.finish)
            
        self.setCaption(self.trUtf8('%1 - %2').arg(self.cmdname).arg(script))
        self.contents.append(self.process.arguments().join(' '))
        procStarted = self.process.start()
        if not procStarted:
            arg0 = self.process.arguments()[0]
            KQMessageBox.critical(None,
                self.trUtf8('Process Generation Error'),
                self.trUtf8(
                    'The process %1 could not be started.'
                    'Ensure, that it is in the search path.'
                ).arg(arg0),
                self.trUtf8('&OK'))
        return procStarted
        
    def buttonPressed(self):
        """
        Private slot connected to the button clicked signal.
        """
        if self.process is None:
            self.accept()
        else:
            self.finish()
            
    def finish(self):
        """
        Private slot called when the process finished.
        
        It is called when the process finished or
        the user pressed the button.
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        self.button.setText(self.trUtf8('&Close'))
        self.button.setDefault(1)
        
        self.process = None
        
        self.contents.append(self.trUtf8('\n%1 finished.\n').arg(self.cmdname))
        
    def handleReadStdout(self):
        """
        Private slot to handle the readyReadStdout signal. 
        
        It reads the output of the process, formats it and inserts it into
        the contents pane.
        """
        self.contents.setTextFormat(QTextBrowser.PlainText)
        
        while self.process.canReadLineStdout():
            s = self.process.readLineStdout()
            self.contents.append(s)
        
    def handleReadStderr(self):
        """
        Private slot to handle the readyReadStderr signal. 
        
        It reads the error output of the process and inserts it into the
        error pane.
        """
        while self.process.canReadLineStderr():
            s = self.process.readLineStderr()
            self.errors.append(s)
