# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2007 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog for the configuration of a keyboard shortcut.
"""

from qt import *

from ShortcutForm import ShortcutForm


class ShortcutDialog(ShortcutForm):
    """
    Class implementing a dialog for the configuration of a keyboard shortcut.
    
    @signal shortcutChanged(QKeySequence) emitted after the OK button
            was pressed
    """
    def __init__(self,parent = None,name = None,modal = 0,fl = 0):
        """
        Constructor
        
        @param parent The parent widget of this dialog. (QWidget)
        @param name The name of this dialog. (QString)
        @param modal Flag indicating a modal dialog. (boolean)
        @param fl The window flags.
        """
        ShortcutForm.__init__(self,parent,name,modal,fl)
        
        self.keyIndex = 0
        self.keys = [0, 0, 0, 0]
        
        self.shortcutsButtonGroup.installEventFilter(self)
        self.primaryButton.installEventFilter(self)
        self.alternateButton.installEventFilter(self)
        self.primaryClearButton.installEventFilter(self)
        self.alternateClearButton.installEventFilter(self)

    def setKeys(self, key, alternateKey):
        """
        Public method to set the key to be configured.
        
        @param key key sequence to be changed (QKeySequence)
        @param alternateKey alternate key sequence to be changed (QKeySequence)
        """
        self.keyIndex = 0
        self.keys = [0, 0, 0, 0]
        self.keyLabel.setText(QString(key))
        self.alternateKeyLabel.setText(QString(alternateKey))
        self.primaryButton.setChecked(1)
        
    def handleOk(self):
        """
        Private slot to handle the OK button press.
        """
        self.hide()
        self.emit(PYSIGNAL('shortcutChanged'), 
                  (QKeySequence(self.keyLabel.text()),
                   QKeySequence(self.alternateKeyLabel.text())))

    def handleClear(self):
        """
        Private slot to handle the Clear button press.
        """
        self.keyIndex = 0
        self.keys = [0, 0, 0, 0]
        self.setKeyLabelText("")
        
    def handleTypeChanged(self):
        """
        Private slot to handle the change of the shortcuts type.
        """
        self.keyIndex = 0
        self.keys = [0, 0, 0, 0]
        
    def setKeyLabelText(self, txt):
        """
        Private method to set the text of a key label.
        
        @param txt text to be set (QString)
        """
        if self.primaryButton.isChecked():
            self.keyLabel.setText(txt)
        else:
            self.alternateKeyLabel.setText(txt)
        
    def eventFilter(self, watched, event):
        """
        Method called to filter the event queue.
        
        @param watched the QObject being watched
        @param event the event that occurred
        @return always 0
        """
        if event.type() == QEvent.KeyPress:
            self.keyPressEvent(event)
            return True
            
        return False
        
    def keyPressEvent(self, evt):
        """
        Private method to handle a key press event.
        
        @param evt the key event (QKeyEvent)
        """
        if evt.key() == Qt.Key_Control:
            return;
        if evt.key() == Qt.Key_Meta:
            return;
        if evt.key() == Qt.Key_Shift:
            return;
        if evt.key() == Qt.Key_Alt:
            return;
        if evt.key() == Qt.Key_Menu:
            return;
    
        if self.keyIndex == 4:
            self.keyIndex = 0
            self.keys = [0, 0, 0, 0]
    
        if evt.key() == Qt.Key_Backtab and evt.state() & Qt.ShiftButton:
            self.keys[self.keyIndex] = Qt.Key_Tab
        else:
            self.keys[self.keyIndex] = evt.key();
        
        if evt.state() & Qt.ShiftButton:
            self.keys[self.keyIndex] += Qt.SHIFT
        if evt.state() & Qt.ControlButton:
            self.keys[self.keyIndex] += Qt.CTRL
        if evt.state() & Qt.AltButton:
            self.keys[self.keyIndex] += Qt.ALT
        # the following is a workaround for a bug in PyQt 3.7
        try:
            if evt.state() & Qt.MetaButton:
                self.keys[self.keyIndex] += Qt.META
        except AttributeError:
            if evt.state() & 0x0800:
                self.keys[self.keyIndex] += Qt.META
        
        self.keyIndex += 1
        
        if self.keyIndex == 1:
            self.setKeyLabelText(QString(QKeySequence(self.keys[0])))
        elif self.keyIndex == 2:
            self.setKeyLabelText(QString(QKeySequence(self.keys[0], self.keys[1])))
        elif self.keyIndex == 3:
            self.setKeyLabelText(QString(QKeySequence(self.keys[0], self.keys[1],
                self.keys[2])))
        elif self.keyIndex == 4:
            self.setKeyLabelText(QString(QKeySequence(self.keys[0], self.keys[1],
                self.keys[2], self.keys[3])))
