# -*- coding: utf-8 -*-

# Copyright (c) 2005 - 2007 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the templates properties dialog.
"""

from qt import *

from KdeQt import KQMessageBox

from TemplatePropertiesForm import TemplatePropertiesForm


class TemplatePropertiesDialog(TemplatePropertiesForm):
    """
    Class implementing the templates properties dialog.
    """
    def __init__(self, parent, groupMode = 0, itm = None):
        """
        Constructor
        
        @param parent the parent widget (QWidget)
        @param groupMode flag indicating group mode (boolean)
        @param itm item (TemplateEntry or TemplateGroup) to
            read the data from
        """
        TemplatePropertiesForm.__init__(self, parent, None, 1)
        
        self.languages = [
            ("All", self.trUtf8("All")),
            ("Bash", self.trUtf8("Bash")),
            ("Batch", self.trUtf8("Batch")),
            ("C/C++", self.trUtf8("C++")),
            ("C#", self.trUtf8("C#")),
            ("HTML/XML", self.trUtf8("HTML/XML")),
            ("IDL", self.trUtf8("IDL")),
            ("Java", self.trUtf8("Java")),
            ("JavaScript", self.trUtf8("JavaScript")),
            ("Lua", self.trUtf8("Lua")),
            ("Makefile", self.trUtf8("Makefile")),
            ("Perl", self.trUtf8("Perl")),
            ("Properties", self.trUtf8("Properties")),
            ("Python", self.trUtf8("Python")),
            ("Ruby", self.trUtf8("Ruby")),
            ("SQL", self.trUtf8("SQL")),
            ("TeX", self.trUtf8("TeX")),
        ]
        
        self.groupMode = groupMode
        if groupMode:
            langList = QStringList()
            for lang, langDisp in self.languages:
                langList.append(langDisp)
            
            self.groupLabel.setText(self.trUtf8("Language:"))
            self.groupCombo.insertStringList(langList)
            self.templateLabel.setEnabled(0)
            self.templateEdit.setEnabled(0)
            self.templateEdit.setText(self.trUtf8("GROUP"))
            self.helpButton.setEnabled(0)
        else:
            groups = QStringList()
            for group in parent.getGroupNames():
                groups.append(group)
            self.groupCombo.insertStringList(groups)
        
        if itm is not None:
            self.nameEdit.setText(itm.getName())
            if groupMode:
                lang = itm.getLanguage()
                for l, d in self.languages:
                    if l == lang:
                        self.groupCombo.setCurrentText(d)
                        break
            else:
                self.groupCombo.setCurrentText(itm.getGroupName())
                self.templateEdit.setText(itm.getTemplateText())

    def handleHelp(self):
        """
        Public slot to show some help.
        """
        KQMessageBox.information(self,
            self.trUtf8("Template Help"),
            self.trUtf8("""<p>To use variables in a template, you just have to enclose"""
                        """ the variablename with $-characters. When you use the template,"""
                        """ you will then be asked for a value for this variable.</p>"""
                        """<p>Example template: This is a $VAR$</p>"""
                        """<p>When you use this template you will be prompted for a value"""
                        """ for the variable $VAR$. Any occurences of $VAR$ will then be"""
                        """ replaced with whatever you've entered.</p>"""
                        """<p>If you need a single $-character in a template, which is not"""
                        """ used to enclose a variable, type $$(two dollar characters)"""
                        """ instead. They will automatically be replaced with a single"""
                        """ $-character when you use the template.</p>"""
                        """<p>If you want a variables contents to be treated specially,"""
                        """ the variablename must be followed by a ':' and one formatting"""
                        """ specifier (e.g. $VAR:ml$). The supported specifiers are:"""
                        """<table>"""
                        """<tr><td>ml</td><td>Specifies a multiline formatting."""
                        """ Each line of the variable contents is prefixed with the string"""
                        """ occuring before the variable on the same line of the template."""
                        """</td></tr>"""
                        """</table></p>"""
                        """<p>If you want to change the default delimiter to anything"""
                        """ different, please use the configuration dialog to do so.</p>"""),
            self.trUtf8("&OK"),
            QString.null,
            QString.null,
            0, -1)

    def setSelectedGroup(self, name):
        """
        Public method to select a group.
        
        @param name name of the group to be selected (string or QString)
        """
        self.groupCombo.setCurrentText(name)

    def getData(self):
        """
        Public method to get the data entered into the dialog.
        
        @return a tuple of two strings (name, language), if the dialog is in group mode,
            and a tuple of three strings (name, group name, template) otherwise.
        """
        if self.groupMode:
            return (unicode(self.nameEdit.text()),
                    self.languages[self.groupCombo.currentItem()][0]
                   )
        else:
            return (unicode(self.nameEdit.text()),
                    unicode(self.groupCombo.currentText()),
                    unicode(self.templateEdit.text())
                   )
