#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2002-2009 Detlev Offenbach <detlev@die-offenbachs.de>
#
# This is the install script for eric4.

"""
Installation script for the eric4 IDE and all eric4 related tools.
"""

import sys
import os
import re
import compileall
import py_compile
import glob
import shutil
import fnmatch
import distutils.sysconfig

# Define the globals.
progName = None
modDir = None
pyModDir = None
platBinDir = None
distDir = None
qtDataDir = None
doCleanup = True
doCompile = True
specialInstall = False
cfg = {}
progLanguages = ["Python", "Ruby"]

def usage(rcode = 2):
    """
    Display a usage message and exit.

    @param rcode the return code passed back to the calling process.
    """
    global progName, platBinDir, modDir, distDir

    print
    print "Usage:"
    print "    %s [-chxz] [-b dir] [-d dir] [-f file] [-i dir]" % (progName)
    print "where:"
    print "    -h        display this help message"
    print "    -b dir    where the binaries will be installed"
    print "              (default: %s)" % (platBinDir)
    print "    -d dir    where eric4 python files will be installed"
    print "              (default: %s)" % (modDir)
    print "    -f file   configuration file naming the various installation paths"
    if sys.platform <> "win32":
        print "    -i dir    temporary install prefix"
        print "              (default: %s)" % (distDir)
    print "    -x        don't perform dependency checks (use on your own risk)"
    print "    -c        don't cleanup old installation first"
    print "    -z        don't compile the installed python files"
    print
    print "The file given to the -f option must be valid Python code defining a"
    print "dictionary called 'cfg' with the keys 'ericDir', 'ericPixDir', 'ericIconDir',"
    print "'ericDTDDir', 'ericCSSDir', 'ericStylesDir', 'ericDocDir', 'ericExamplesDir',"
    print "'ericTranslationsDir', 'ericTemplatesDir', 'ericCodeTemplatesDir',"
    print "'ericOthersDir','bindir' and 'mdir'."
    print "These define the directories for the installation of the various parts of"\
          " eric4."

    sys.exit(rcode)


def initGlobals():
    """
    Sets the values of globals that need more than a simple assignment.
    """
    global platBinDir, modDir, pyModDir

    if sys.platform == "win32":
        platBinDir = sys.exec_prefix
    else:
        platBinDir = "/usr/local/bin"

    modDir = distutils.sysconfig.get_python_lib(True)
    pyModDir = modDir


def copyToFile(name, text):
    """
    Copy a string to a file.

    @param name the name of the file.
    @param text the contents to copy to the file.
    """
    f = open(name,"w")
    f.write(text)
    f.close()


def wrapperName(dname, wfile):
    """
    Create the platform specific name for the wrapper script.
    
    @param dname name of the directory to place the wrapper into
    @param wfile basename (without extension) of the wrapper script
    @return the name of the wrapper script
    """
    if sys.platform == "win32":
        wname = dname + "\\" + wfile + ".bat"
    else:
        wname = dname + "/" + wfile

    return wname


def createPyWrapper(pydir, wfile):
    """
    Create an executable wrapper for a Python script.

    @param pydir the name of the directory where the Python script will eventually
        be installed
    @param wfile the basename of the wrapper
    @return the platform specific name of the wrapper
    """
    # all kinds of Windows systems
    if sys.platform == "win32":
        wname = wfile + ".bat"
        wrapper = \
'''@"%s\\pythonw" "%s\\%s.py" %%1 %%2 %%3 %%4 %%5 %%6 %%7 %%8 %%9
''' % (platBinDir, pydir, wfile)

    # Mac OS X
    elif sys.platform == "darwin":
        wname = wfile
        wrapper = \
'''#!/bin/sh

exec "%s/bin/pythonw" "%s/%s.py" "$@"
''' % (sys.exec_prefix, pydir, wfile)

    # *nix systems
    else:
        wname = wfile
        wrapper = \
'''#!/bin/sh

exec "%s" "%s/%s.py" "$@"
''' % (sys.executable, pydir, wfile)

    copyToFile(wname, wrapper)
    os.chmod(wname, 0755)

    return wname


def copyTree(src, dst, filters, excludeDirs=[]):
    """
    Copy Python, translation, documentation, wizards configuration,
    designer template files and DTDs of a directory tree.
    
    @param src name of the source directory
    @param dst name of the destination directory
    @param filters list of filter pattern determining the files to be copied
    @param excludeDirs list of (sub)directories to exclude from copying
    """
    try:
        names = os.listdir(src)
    except OSError:
        return      # ignore missing directories (most probably the i18n directory)
    
    for name in names:
        srcname = os.path.join(src, name)
        dstname = os.path.join(dst, name)
        for filter in filters:
            if fnmatch.fnmatch(srcname, filter):
                if not os.path.isdir(dst):
                    os.makedirs(dst)
                shutil.copy2(srcname, dstname)
                break
        else:
            if os.path.isdir(srcname) and not srcname in excludeDirs:
                copyTree(srcname, dstname, filters)


def createGlobalPluginsDir():
    """
    Create the global plugins directory, if it doesn't exist.
    """
    global cfg, distDir
    
    pdir = os.path.join(cfg['mdir'], "eric4plugins")
    fname = os.path.join(pdir, "__init__.py")
    if not os.path.exists(fname):
        if not os.path.exists(pdir):
            os.mkdir(pdir,  0755)
        f = open(fname, "wb")
        f.write(\
'''# -*- coding: utf-8 -*-

"""
Package containing the global plugins.
"""
'''        
        )
        f.close()


def getAPIsPath(progLanguage = None):
    """
    Get the name of the directory containing the API files.
    
    @param progLanguage programming language of the API files (string)
    @return name of the directory containing the API files (string)
    """
    global cfg, qtDataDir
    
    if not specialInstall and qtDataDir is None:
        try:
            from PyQt4 import pyqtconfig
            qtDataDir = pyqtconfig._pkg_config["qt_data_dir"]
        except (AttributeError, ImportError):
            print
            print "Please enter the name of the Qt data directory."
            print "(That is the one containing the 'qsci' subdirectory.)"
            qtDataDir = raw_input(">>> ")
    if qtDataDir:
        scidir = os.path.join(qtDataDir, "qsci")
        if progLanguage is None:
            apidir = os.path.join(scidir, "api")
        else:
            apidir = os.path.join(scidir, "api", progLanguage.lower())
    else:
        apidir = cfg['ericOthersDir']
    
    return apidir


def cleanUp():
    """
    Uninstall the old eric files.
    """
    try:
        from eric4config import getConfig
    except ImportError:
        # eric4 wasn't installed previously
        return
    
    global pyModDir, progLanguages
    
    # Remove the wrapper scripts
    rem_wnames = [
        "eric4-api", "eric4-compare",
        "eric4-configure", "eric4-diff",
        "eric4-doc", "eric4-helpviewer",
        "eric4-qregexp", "eric4-re", 
        "eric4-trpreviewer", "eric4-uipreviewer",
        "eric4-unittest", "eric4",
        "eric4-tray", "eric4-editor", 
        "eric4-plugininstall", "eric4-pluginuninstall", 
        "eric4-pluginrepository", 
    ]
    for rem_wname in rem_wnames:
        rwname = wrapperName(getConfig('bindir'), rem_wname)
        if os.path.exists(rwname):
            os.remove(rwname)
    
    # Cleanup our config file
    for name in ['eric4config.py', 'eric4config.pyc']:
        e4cfile = os.path.join(pyModDir, name)
        if os.path.exists(e4cfile):
            os.remove(e4cfile)
        
    # Cleanup the install directories
    for name in ['ericExamplesDir', 'ericDocDir', 'ericDTDDir', 'ericCSSDir',
                 'ericIconDir', 'ericPixDir', 'ericDir', 'ericTemplatesDir',
                 'ericCodeTemplatesDir', 'ericOthersDir', 'ericStylesDir']:
        if os.path.exists(getConfig(name)):
            shutil.rmtree(getConfig(name), True)
    
    # Cleanup translations
    for name in glob.glob(os.path.join(getConfig('ericTranslationsDir'), 'eric4_*.qm')):
        if os.path.exists(name):
            os.remove(name)
    
    # Cleanup API files
    try:
        apidir = getConfig('apidir')
        for progLanguage in progLanguages:
            for name in getConfig('apis'):
                apiname = os.path.join(apidir, progLanguage, name)
                if os.path.exists(apiname):
                    os.remove(apiname)
    except AttributeError:
        pass


def installEric():
    """
    Actually perform the installation steps.
    """
    global distDir, doCleanup, cfg, progLanguages
    
    # Create the platform specific wrappers.
    wnames = []
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-api"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-compare"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-configure"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-diff"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-doc"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-helpviewer"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-qregexp"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-re"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-trpreviewer"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-uipreviewer"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-unittest"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-tray"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-editor"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-plugininstall"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-pluginuninstall"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4-pluginrepository"))
    wnames.append(createPyWrapper(cfg['ericDir'], "eric4"))
    
    # set install prefix, if not None
    if distDir:
        for key in cfg.keys():
            cfg[key] = os.path.normpath(distDir + os.sep + cfg[key])
    
    try:
        # Install the files
        # make the install directories
        for key in cfg.keys():
            if not os.path.isdir(cfg[key]):
                os.makedirs(cfg[key])
        
        # copy the eric4 config file
        if distDir:
            shutil.copy('eric4config.py', cfg['mdir'])
            if os.path.exists('eric4config.pyc'):
                shutil.copy('eric4config.pyc', cfg['mdir'])
        else:
            shutil.copy('eric4config.py', modDir)
            if os.path.exists('eric4config.pyc'):
                shutil.copy('eric4config.pyc', modDir)
        
        # copy the eric4.pth file
        if distDir:
            shutil.copy('eric%seric4.pth' % os.sep, cfg['mdir'])
        else:
            shutil.copy('eric%seric4.pth' % os.sep, modDir)
        
        # copy the various parts of eric4
        copyTree('eric', cfg['ericDir'], ['*.py', '*.pyc', '*.pyo', '*.pyw'], 
            ['eric%sExamples' % os.sep])
        copyTree('eric', cfg['ericDir'], ['*.rb'], 
            ['eric%sExamples' % os.sep])
        copyTree('eric%sPlugins' % os.sep, '%s%sPlugins' % (cfg['ericDir'], os.sep), 
            ['*.png'])
        copyTree('eric%sDocumentation' % os.sep, cfg['ericDocDir'], ['*.html'])
        copyTree('eric%sDTDs' % os.sep, cfg['ericDTDDir'], ['*.dtd'])
        copyTree('eric%sCSSs' % os.sep, cfg['ericCSSDir'], ['*.css'])
        copyTree('eric%sStyles' % os.sep, cfg['ericStylesDir'], ['*.qss'])
        copyTree('eric%si18n' % os.sep, cfg['ericTranslationsDir'], ['*.qm'])
        copyTree('eric%sicons' % os.sep, cfg['ericIconDir'], ['*.png', 'LICENSE*.*'])
        copyTree('eric%spixmaps' % os.sep, cfg['ericPixDir'], ['*.png', '*.xpm', '*.ico'])
        copyTree('eric%sDesignerTemplates' % os.sep, cfg['ericTemplatesDir'], ['*.tmpl'])
        copyTree('eric%sCodeTemplates' % os.sep, cfg['ericCodeTemplatesDir'], ['*.tmpl'])
        copyTree('eric%sExamples' % os.sep, cfg['ericExamplesDir'], 
                 ['*.py', '*.pyc', '*.pyo'])
        
        # copy the wrappers
        for wname in wnames:
            shutil.copy(wname, cfg['bindir'])
        
        # copy the license file
        shutil.copy('eric%sLICENSE.GPL3' % os.sep, cfg['ericDir'])
        
        # create the global plugins directory
        createGlobalPluginsDir()
        
    except IOError, msg:
        sys.stderr.write('IOError: %s\nTry install as root.\n' % msg)
        sys.exit(7)
    
    # copy some text files to the doc area
    for name in ["LICENSE.GPL3", "THANKS", "changelog"]:
        try:
            shutil.copy('eric%s%s' % (os.sep, name), cfg['ericDocDir'])
        except:
            print "Could not install 'eric%s%s'." % (os.sep, name)
    for name in glob.glob(os.path.join('eric', 'README*.*')):
        try:
            shutil.copy(name, cfg['ericDocDir'])
        except:
            print "Could not install 'eric%s%s'." % (os.sep, name)
   
    # copy some more stuff
    for name in ['default.e4k']:
        try:
            shutil.copy('eric%s%s' % (os.sep, name), cfg['ericOthersDir'])
        except:
            print "Could not install 'eric%s%s'." % (os.sep, name)
    
    # install the API file
    for progLanguage in progLanguages:
        apidir = getAPIsPath(progLanguage)
        if not os.path.exists(apidir):
            os.makedirs(apidir)
        for apiName in glob.glob(os.path.join("eric", "APIs", progLanguage, "*.api")):
            try:
                shutil.copy(apiName, apidir)
            except:
                try:
                    shutil.copy(apiName, cfg['ericOthersDir'])
                except:
                    print "Could not install '%s'." % apiName


def createInstallConfig():
    """
    Create the installation config dictionary.
    """
    global modDir, platBinDir, cfg
        
    ericdir = os.path.join(modDir, "eric4")
    cfg = {
        'ericDir'              : ericdir,
        'ericPixDir'           : os.path.join(ericdir, "pixmaps"),
        'ericIconDir'          : os.path.join(ericdir, "icons"),
        'ericDTDDir'           : os.path.join(ericdir, "DTDs"),
        'ericCSSDir'           : os.path.join(ericdir, "CSSs"),
        'ericStylesDir'        : os.path.join(ericdir, "Styles"),
        'ericDocDir'           : os.path.join(ericdir, "Documentation"),
        'ericExamplesDir'      : os.path.join(ericdir, "Examples"),
        'ericTranslationsDir'  : os.path.join(ericdir, "i18n"),
        'ericTemplatesDir'     : os.path.join(ericdir, "DesignerTemplates"),
        'ericCodeTemplatesDir' : os.path.join(ericdir, 'CodeTemplates'),
        'ericOthersDir'        : ericdir,
        'bindir'               : platBinDir,
        'mdir'                 : modDir,
    }
configLength = 14
    
def createConfig():
    """
    Create a config file with the respective config entries.
    """
    global cfg
    
    apis = []
    for progLanguage in progLanguages:
        for apiName in glob.glob(os.path.join("eric", "APIs", progLanguage, "*.api")):
            apis.append(os.path.basename(apiName))
    
    fn = 'eric4config.py'
    config = \
"""# -*- coding: utf-8 -*-
#
# This module contains the configuration of the individual eric4 installation
#

_pkg_config = {
    'ericDir'              : r'%s',
    'ericPixDir'           : r'%s',
    'ericIconDir'          : r'%s',
    'ericDTDDir'           : r'%s',
    'ericCSSDir'           : r'%s',
    'ericStylesDir'        : r'%s',
    'ericDocDir'           : r'%s',
    'ericExamplesDir'      : r'%s',
    'ericTranslationsDir'  : r'%s',
    'ericTemplatesDir'     : r'%s',
    'ericCodeTemplatesDir' : r'%s',
    'ericOthersDir'        : r'%s',
    'bindir'               : r'%s',
    'mdir'                 : r'%s',
    'apidir'               : r'%s',
    'apis'                 : %s,
}

def getConfig(name):
    '''
    Module function to get a configuration value.

    @param name the name of the configuration value (string).
    '''
    try:
        return _pkg_config[name]
    except KeyError:
        pass

    raise AttributeError, '"%%s" is not a valid configuration value' %% name
""" % (cfg['ericDir'], cfg['ericPixDir'], cfg['ericIconDir'], 
       cfg['ericDTDDir'], cfg['ericCSSDir'], 
       cfg['ericStylesDir'], cfg['ericDocDir'],
       cfg['ericExamplesDir'], cfg['ericTranslationsDir'],
       cfg['ericTemplatesDir'],
       cfg['ericCodeTemplatesDir'], cfg['ericOthersDir'],
       cfg['bindir'], cfg['mdir'], 
       getAPIsPath(), apis)
    copyToFile(fn, config)

def doDependancyChecks():    
    """
    Perform some dependancy checks.
    """
    # perform dependency checks
    if sys.version_info < (2,4,0):
        print 'Sorry, you must have Python 2.4.0 or higher.'
        sys.exit(5)
    
    try:
        from PyQt4.QtCore import qVersion, PYQT_VERSION
    except ImportError, msg:
        print 'Sorry, please install PyQt4.'
        print 'Error: %s' % msg
        sys.exit(1)
    
    try:
        from PyQt4 import Qsci
    except ImportError, msg:
        print "Sorry, please install QScintilla2 and"
        print "it's PyQt4 wrapper."
        print 'Error: %s' % msg
        sys.exit(1)
    
    # check version of Qt
    qtMajor = int(qVersion().split('.')[0])
    qtMinor = int(qVersion().split('.')[1])
    if qtMajor < 4 or (qtMajor == 4 and qtMinor < 3):
        print 'Sorry, you must have Qt version 4.3.0 or higher.'
        sys.exit(2)
    
    #check version of PyQt
    from PyQt4.QtCore import PYQT_VERSION_STR
    pyqtVersion = PYQT_VERSION_STR
    # always assume, that snapshots are new enough
    if pyqtVersion.find("snapshot-") == -1:
        while pyqtVersion.count('.') < 2:
            pyqtVersion += '.0'
        (maj, min, pat) = pyqtVersion.split('.')
        maj = int(maj)
        min = int(min)
        pat = int(pat)
        if maj < 4 or (maj == 4 and min < 3):
            print 'Sorry, you must have PyQt 4.3.0 or higher or' \
                  ' a recent snapshot release.'
            sys.exit(3)
    
    #check version of QScintilla
    from PyQt4.Qsci import QSCINTILLA_VERSION_STR
    scintillaVersion = QSCINTILLA_VERSION_STR
    # always assume, that snapshots are new enough
    if scintillaVersion.find("snapshot-") == -1:
        while scintillaVersion.count('.') < 2:
            scintillaVersion += '.0'
        (maj, min, pat) = scintillaVersion.split('.')
        maj = int(maj)
        min = int(min)
        pat = int(pat)
        if maj < 2 or (maj == 2 and min < 2):
            print 'Sorry, you must have QScintilla 2.2.0 or higher or' \
                  ' a recent snapshot release.'
            sys.exit(4)

def main(argv):
    """
    The main function of the script.

    @param argv the list of command line arguments.
    """
    import getopt

    # Parse the command line.
    global progName, modDir, doCleanup, doCompile, distDir, cfg, specialInstall
    progName = os.path.basename(argv[0])

    initGlobals()

    try:
        if sys.platform == "win32":
            optlist, args = getopt.getopt(argv[1:],"chxzb:d:f:")
        else:
            optlist, args = getopt.getopt(argv[1:],"chxzb:d:f:i:")
    except getopt.GetoptError:
        usage()

    global platBinDir
    
    depChecks = True

    for opt, arg in optlist:
        if opt == "-h":
            usage(0)
        elif opt == "-b":
            platBinDir = arg
        elif opt == "-d":
            modDir = arg
            specialInstall = True
        elif opt == "-i":
            distDir = os.path.normpath(arg)
        elif opt == "-x":
            depChecks = False
        elif opt == "-c":
            doCleanup = False
        elif opt == "-z":
            doCompile = False
        elif opt == "-f":
            try:
                execfile(arg, globals())
                if len(cfg) != configLength:
                    print "The configuration dictionary in '%s' is incorrect. Aborting"\
                        % arg
                    sys.exit(6)
            except:
                cfg = {}
    
    if len(cfg) == 0:
        createInstallConfig()
    
    if depChecks:
        doDependancyChecks()
    
    # get rid of development config file, if it exists
    try:
        os.remove(os.path.join("eric", "eric4config.py"))
    except:
        pass
    
    # cleanup old installation
    try:
        if doCleanup:
            if distDir:
                shutil.rmtree(distDir, True)
            else:
                cleanUp()
    except IOError, msg:
        sys.stderr.write('IOError: %s\nTry install as root.\n' % msg)

    # Create a config file and delete the default one
    createConfig()

    if doCompile:
        print "Compiling debugger files."
        if distDir:
            compileall.compile_dir("eric", 
                                   ddir = os.path.join(distDir, modDir, cfg['ericDir']))
            py_compile.compile("eric4config.py", 
                               dfile = os.path.join(distDir, modDir, "eric4config.py"))
        else:
            compileall.compile_dir("eric", 
                                   ddir = os.path.join(modDir, cfg['ericDir']))
            py_compile.compile("eric4config.py", 
                               dfile = os.path.join(modDir, "eric4config.py"))
    print "\nInstalling eric4 ..."
    installEric()
    print "\nInstallation complete."
    print
    
    # check PyXML version and output a message for broken PyXML (< 0.8.6)
    try:
        import _xmlplus
        v = _xmlplus.version_info
        if v < (0, 8, 6):
            from eric4.patch_pyxml import isPatched, patchPyXML
            if not isPatched():
                print "NOTE:"
                print "    Found PyXML %d.%d.%d, which needs a patch to work correctly" % \
                    (v[0], v[1], v[2])
                print "    with foreign characters. Please see 'README-PyXML.txt' for"
                print "    details."
                res = raw_input("    Shall pyXML be patched now (y/n)? ")
                if res in ["Y", "y"]:
                    patchPyXML()
    except:
        pass
    
    
if __name__ == "__main__":
    try:
        main(sys.argv)
    except SystemExit:
        raise
    except:
        print \
"""An internal error occured.  Please report all the output of the program,
including the following traceback, to eric4-bugs@die-offenbachs.de.
"""
        raise

