# -*- coding: utf-8 -*-

# Copyright (c) 2002, 2003 Detlev Offenbach <detlev@die-offenbachs.de>
# Copyright (c) 2000 Phil Thompson <phil@river-bank.demon.co.uk>
#

"""
Module implementing a debug client base class.
"""

import sys
import socket
import select
import codeop
import traceback
import os
import atexit
import time
import imp


from DebugProtocol import *
from DebugBase import printerr
from AsyncFile import *
from Config import ConfigVarTypeStrings
from FlexCompleter import Completer


DebugClientInstance = None

def DebugClientRawInput(prompt):
    """
    Replacement for the standard raw_input builtin.
    
    This function works with the split debugger.
    
    @param prompt The prompt to be shown. (string)
    """
    if DebugClientInstance is None:
        return DebugClientOrigRawInput(prompt)

    return DebugClientInstance.raw_input(prompt)

# Use our own raw_input().
try:
    DebugClientOrigRawInput = __builtins__.__dict__['raw_input']
    __builtins__.__dict__['raw_input'] = DebugClientRawInput
except:
    import __main__
    DebugClientOrigRawInput = __main__.__builtins__.__dict__['raw_input']
    __main__.__builtins__.__dict__['raw_input'] = DebugClientRawInput


class DebugClientBase:
    """
    Class implementing the client side of the debugger.

    It provides access to the Python interpeter from a debugger running in another
    process whether or not the Qt event loop is running.

    The protocol between the debugger and the client assumes that there will be
    a single source of debugger commands and a single source of Python
    statements.  Commands and statement are always exactly one line and may be
    interspersed.

    The protocol is as follows.  First the client opens a connection to the
    debugger and then sends a series of one line commands.  A command is either
    &gt;Load&lt;, &gt;Step&lt;, &gt;StepInto&lt;, ... or a Python statement. 
    See DebugProtocol.py for a listing of valid protocol tokens.

    A Python statement consists of the statement to execute, followed (in a
    separate line) by &gt;OK?&lt;.  If the statement was incomplete then the response
    is &gt;Continue&lt;.  If there was an exception then the response is &gt;Exception&lt;.
    Otherwise the response is &gt;OK&lt;.  The reason for the &gt;OK?&lt; part is to
    provide a sentinal (ie. the responding &gt;OK&lt;) after any possible output as a
    result of executing the command.

    The client may send any other lines at any other time which should be
    interpreted as program output.

    If the debugger closes the session there is no response from the client.
    The client may close the session at any time as a result of the script
    being debugged closing or crashing.
    
    <b>Note</b>: This class is meant to be subclassed by individual DebugClient classes.
    Do not instantiate it directly.
    """
    def __init__(self):
        """
        Constructor
        """
        self.breakpoints = {}

        # The next couple of members are needed for the threaded version.
        # For this base class they contain static values for the non threaded
        # debugger
        
        # dictionary of all threads running
        self.threads = {}
        
        # the "current" thread, basically the thread we are at a breakpoint for.
        self.currentThread = self
        
        # special objects representing the main scripts thread and frame
        self.mainThread = self
        self.mainFrame = None
        
        # The context to run the debugged program in.
        self.debugMod = imp.new_module('__main__')

        # The list of complete lines to execute.
        self.buffer = ''

        self.pendingResponse = ResponseOK
        self.fncache = {}
        self.dircache = []
        self.inRawMode = 0
        self.mainProcStr = None     # used for the passive mode
        self.passive = 0            # used to indicate the passive mode
        self.running = None
        self.test = None
        self.tracePython = 0

        self.readstream = None
        self.writestream = None
        self.errorstream = None
        
        # So much for portability.
        if sys.platform == 'win32':
            self.skipdir = sys.prefix
        else:
            self.skipdir = os.path.join(sys.prefix,'lib/python' + sys.version[0:3])
            
        self.variant = 'You should not see this'
        
        # commandline completion stuff
        self.complete = Completer(self.debugMod.__dict__).complete

    def attachThread(self, target=None, args=None, kwargs=None, mainThread=0):
        """
        Public method to setup a thread for DebugClient to debug.
        
        If mainThread is non-zero, then we are attaching to the already 
        started mainthread of the app and the rest of the args are ignored.
        
        This is just an empty function and is overridden in the threaded
        debugger.
        
        @param target the start function of the target thread (i.e. the user code)
        @param args arguments to pass to target
        @param kwargs keyword arguments to pass to target
        @param mainThread non-zero, if we are attaching to the already 
              started mainthread of the app
        @return The identifier of the created thread
        """
        pass

    def raw_input(self,prompt):
        """
        Public method to implement raw_input() using the event loop.
        
        @param prompt the prompt to be shown (string)
        @return the entered string
        """
        self.write("%s%s\n" % (ResponseRaw, prompt))
        self.inRawMode = 1
        self.eventLoop()
        return self.rawLine

    def handleException(self):
        """
        Private method called in the case of an exception
        
        It ensures that the debug server is informed of the raised exception.
        """
        self.pendingResponse = ResponseException
    
    def sessionClose(self):
        """
        Private method to close the session with the debugger and terminate.
        """
        try:
            self.set_quit()
        except:
            pass

        # clean up asyncio.
        self.disconnect()
        
        # make sure we close down our end of the socket
        # might be overkill as normally stdin, stdout and stderr
        # SHOULD be closed on exit, but it does not hurt to do it here
        self.readstream.close()
        self.writestream.close()
        self.errorstream.close()

        # Ok, go away.
        sys.exit()

    def handleLine(self,line):
        """
        Private method to handle the receipt of a complete line.

        It first looks for a valid protocol token at the start of the line. Thereafter
        it trys to execute the lines accumulated so far.
        
        @param ine the received line
        """
        # Remove any newline.
        if line[-1] == '\n':
            line = line[:-1]

##~         printerr(line)          ##debug

        eoc = line.find('<')

        if eoc >= 0 and line[0] == '>':
            # Get the command part and any argument.
            cmd = line[:eoc + 1]
            arg = line[eoc + 1:]
            
            if cmd == RequestVariables:
                frmnr, scope, filter = eval(arg)
                self.dumpVariables(int(frmnr), int(scope), filter)
                return
                
            if cmd == RequestVariable:
                var, frmnr, scope, filter = eval(arg)
                self.dumpVariable(var, int(frmnr), int(scope), filter)
                return
                
            if cmd == RequestStep:
                self.currentThread.step(1)
                self.eventExit = 1
                return

            if cmd == RequestStepOver:
                self.currentThread.step(0)
                self.eventExit = 1
                return
                
            if cmd == RequestStepOut:
                self.currentThread.stepOut()
                self.eventExit = 1
                return
                
            if cmd == RequestStepQuit:
                if self.passive:
                    self.progTerminated(42)
                else:
                    self.set_quit()
                    self.eventExit = 1
                return

            if cmd == RequestContinue:
                self.currentThread.go()
                self.eventExit = 1
                return

            if cmd == RequestOK:
                self.write(self.pendingResponse + '\n')
                self.pendingResponse = ResponseOK
                return

            if cmd == RequestLoad:
                self.fncache = {}
                self.dircache = []
                sys.argv = []
                wd, fn, args, tracePython = arg.split('|')
                sys.argv.append(fn)
                sys.argv.extend(args.split())
                sys.path[0] = os.path.dirname(sys.argv[0])
                if wd == '':
                    os.chdir(sys.path[0])
                else:
                    os.chdir(wd)
                tracePython = int(tracePython)
                self.running = sys.argv[0]
                self.mainFrame = None
                self.inRawMode = 0
                
                self.threads.clear()
                self.attachThread(mainThread = 1)
                
                # set the system exception handling function to ensure, that
                # we report on all unhandled exceptions
                sys.excepthook = self.unhandled_exception
                
                # clear all old breakpoints, they'll get set after we have started
                self.mainThread.clear_all_breaks()
                
                self.mainThread.tracePython = tracePython
                
                # This will eventually enter a local event loop.
                # Note the use of backquotes to cause a repr of self.running. The
                # need for this is on Windows os where backslash is the path separator.
                # They will get inadvertantly stripped away during the eval causing IOErrors
                # if self.running is passed as a normal str.
                sys.modules['__main__'] = self.debugMod
                self.mainThread.run('execfile(' + `self.running` + ')',self.debugMod.__dict__)
                return

            if cmd == RequestRun:
                sys.argv = []
                wd, fn, args = arg.split('|')
                sys.argv.append(fn)
                sys.argv.extend(args.split())
                sys.path[0] = os.path.dirname(sys.argv[0])
                if wd == '':
                    os.chdir(sys.path[0])
                else:
                    os.chdir(wd)

                # set the system exception handling function to ensure, that
                # we report on all unhandled exceptions
                sys.excepthook = self.unhandled_exception
                
                sys.modules['__main__'] = self.debugMod
                execfile(sys.argv[0], self.debugMod.__dict__)
                return

            if cmd == RequestCoverage:
                import PyCoverage
                sys.argv = []
                wd, fn, args, erase = arg.split('|')
                sys.argv.append(fn)
                sys.argv.extend(args.split())
                sys.path[0] = os.path.dirname(sys.argv[0])
                if wd == '':
                    os.chdir(sys.path[0])
                else:
                    os.chdir(wd)

                # set the system exception handling function to ensure, that
                # we report on all unhandled exceptions
                sys.excepthook = self.unhandled_exception
                
                # generate a coverage object
                self.cover = PyCoverage.coverage(sys.argv[0])
                
                # register an exit handler to save the collected data
                atexit.register(self.cover.save)

                if int(erase):
                    self.cover.erase()
                sys.modules['__main__'] = self.debugMod
                self.cover.start()
                execfile(sys.argv[0], self.debugMod.__dict__)
                return

            if cmd == RequestProfile:
                import PyProfile
                sys.argv = []
                wd, fn, args, erase = arg.split('|')
                sys.argv.append(fn)
                sys.argv.extend(args.split())
                sys.path[0] = os.path.dirname(sys.argv[0])
                if wd == '':
                    os.chdir(sys.path[0])
                else:
                    os.chdir(wd)

                # set the system exception handling function to ensure, that
                # we report on all unhandled exceptions
                sys.excepthook = self.unhandled_exception
                
                # generate a profile object
                self.prof = PyProfile.PyProfile(sys.argv[0])
                
                if int(erase):
                    self.prof.erase()
                sys.modules['__main__'] = self.debugMod
                self.prof.run('execfile(' + `sys.argv[0]` + ',' + `self.debugMod.__dict__` + ')')
                return

            if cmd == RequestCyclops:
                from Cyclops import RunCyclops
                sys.argv = []
                wd, fn, args, modfunc, reports = arg.split('|')
                sys.argv.append(fn)
                sys.argv.extend(args.split())
                sys.path[0] = os.path.dirname(sys.argv[0])
                if wd == '':
                    os.chdir(sys.path[0])
                else:
                    os.chdir(wd)

                # set the system exception handling function to ensure, that
                # we report on all unhandled exceptions
                sys.excepthook = self.unhandled_exception
                
                sys.modules['__main__'] = self.debugMod
                RunCyclops.run(fn, modfunc, int(reports), self)
                return

            if cmd == RequestShutdown:
                self.sessionClose()
                return
                
            if cmd == RequestBreak:
                fn, line, temporary, set, cond = arg.split(',')
                line = int(line)
                set = int(set)
                temporary = int(temporary)

                if set:
                    if cond == 'None':
                        cond = None
                    self.mainThread.set_break(fn, line, temporary, cond)
                else:
                    self.mainThread.clear_break(fn, line)

                return
                
            if cmd == RequestBreakEnable:
                fn, line, enable = arg.split(',')
                line = int(line)
                enable = int(enable)
                
                bp = self.mainThread.get_break(fn, line)
                if bp is not None:
                    if enable:
                        bp.enable()
                    else:
                        bp.disable()
                        
                return
                
            if cmd == RequestBreakIgnore:
                fn, line, count = arg.split(',')
                line = int(line)
                count = int(count)
                
                bp = self.mainThread.get_break(fn, line)
                if bp is not None:
                    bp.ignore = count
                    
                return
                
            if cmd == RequestEval:
                try:
                    value = eval(arg, self.currentThread.getCurrentFrame().f_globals,
                                      self.currentThread.getCurrentFrame().f_locals)
                except:
                    # Report the exception and the traceback
                    try:
                        type, value, tb = sys.exc_info()
                        sys.last_type = type
                        sys.last_value = value
                        sys.last_traceback = tb
                        tblist = traceback.extract_tb(tb)
                        del tblist[:1]
                        list = traceback.format_list(tblist)
                        if list:
                            list.insert(0, "Traceback (innermost last):\n")
                            list[len(list):] = traceback.format_exception_only(type, value)
                    finally:
                        tblist = tb = None

                    map(self.write,list)

                    self.write(ResponseException + '\n')
                    
                else:
                    self.write(str(value) + '\n')
                    self.write(ResponseOK + '\n')
                    
                return
            
            if cmd == RequestExec:
                globals = self.currentThread.getCurrentFrame().f_globals
                locals = self.currentThread.getCurrentFrame().f_locals
                try:
                    code = compile(arg + '\n', '<stdin>', 'single')
                    exec code in globals, locals
                except:
                    # Report the exception and the traceback
                    try:
                        type, value, tb = sys.exc_info()
                        sys.last_type = type
                        sys.last_value = value
                        sys.last_traceback = tb
                        tblist = traceback.extract_tb(tb)
                        del tblist[:1]
                        list = traceback.format_list(tblist)
                        if list:
                            list.insert(0, "Traceback (innermost last):\n")
                            list[len(list):] = traceback.format_exception_only(type, value)
                    finally:
                        tblist = tb = None

                    map(self.write,list)

                    self.write(ResponseException + '\n')
                    
                return
            
            if cmd == RequestBanner:
                self.write('%s%s\n' % (ResponseBanner, 
                    str((sys.version, sys.platform, self.variant))))
                return
            
            if cmd == RequestCompletion:
                self.completionList(arg)
                return
            
            if cmd == RequestUTPrepare:
                fn, tn, cov, covname, erase = arg.split('|')
                sys.path.insert(0,os.path.dirname(os.path.abspath(fn)))

                # set the system exception handling function to ensure, that
                # we report on all unhandled exceptions
                sys.excepthook = self.unhandled_exception
                
                try:
                    import unittest
                    self.test = unittest.defaultTestLoader.loadTestsFromName(tn)
                except:
                    exc_type, exc_value, exc_tb = sys.exc_info()
                    self.write('%s%s\n' % (ResponseUTPrepared,
                        str((0, str(exc_type), str(exc_value)))))
                    self.handleException()
                    return
                    
                # generate a coverage object
                if int(cov):
                    import PyCoverage
                    self.cover = PyCoverage.coverage(covname)
                    if int(erase):
                        self.cover.erase()
                else:
                    self.cover = None
                
                self.write('%s%s\n' % (ResponseUTPrepared,
                    str((self.test.countTestCases(), "", ""))))
                return
                
            if cmd == RequestUTRun:
                from DCTestResult import DCTestResult
                self.testResult = DCTestResult(self)
                if self.cover:
                    self.cover.start()
                self.test.run(self.testResult)
                if self.cover:
                    self.cover.stop()
                    self.cover.save()
                self.write('%s\n' % ResponseUTFinished)
                return
                
            if cmd == RequestUTStop:
                self.testResult.stop()
                return
            
        # If we are handling raw mode input then reset the mode and break out
        # of the current event loop.
        if self.inRawMode:
            self.inRawMode = 0
            self.rawLine = line
            self.eventExit = 1
            return

        if self.buffer:
            self.buffer = self.buffer + '\n' + line
        else:
            self.buffer = line

        try:
            code = codeop.compile_command(self.buffer,self.readstream.name)
        except (OverflowError, SyntaxError, ValueError):
            # Report the exception
            sys.last_type, sys.last_value, sys.last_traceback = sys.exc_info()
            map(self.write,traceback.format_exception_only(sys.last_type,sys.last_value))
            self.buffer = ''

            self.handleException()
        else:
            if code is None:
                self.pendingResponse = ResponseContinue
            else: 
                self.buffer = ''

                try:
                    if self.running is None:
                        exec code in self.debugMod.__dict__
                    else:
                        globals = self.currentThread.getCurrentFrame().f_globals
                        locals = self.currentThread.getCurrentFrame().f_locals
                        exec code in globals, locals
                except SystemExit:
                    self.sessionClose()
                except:
                    # Report the exception and the traceback
                    try:
                        type, value, tb = sys.exc_info()
                        sys.last_type = type
                        sys.last_value = value
                        sys.last_traceback = tb
                        tblist = traceback.extract_tb(tb)
                        del tblist[:1]
                        list = traceback.format_list(tblist)
                        if list:
                            list.insert(0, "Traceback (innermost last):\n")
                            list[len(list):] = traceback.format_exception_only(type, value)
                    finally:
                        tblist = tb = None

                    map(self.write,list)

                    self.handleException()

    def write(self,s):
        """
        Private method to write data to the output stream.
        
        @param s data to be written (string)
        """
        self.writestream.write(s)
        self.writestream.flush()

    def interact(self):
        """
        Private method to Interact with  the debugger.
        """
        global DebugClientInstance

        self.setDescriptors(self.readstream,self.writestream)
        DebugClientInstance = self

        if not self.passive:
            # At this point simulate an event loop.
            self.eventLoop()

    def eventLoop(self):
        """
        Private method implementing our event loop.
        """
        self.eventExit = None

        while self.eventExit is None:
            wrdy = []

            if AsyncPendingWrite(self.writestream):
                wrdy.append(self.writestream)

            if AsyncPendingWrite(self.errorstream):
                wrdy.append(self.errorstream)

            rrdy, wrdy, xrdy = select.select([self.readstream],wrdy,[])

            if self.readstream in rrdy:
                self.readReady(self.readstream.fileno())

            if self.writestream in wrdy:
                self.writeReady(self.writestream.fileno())

            if self.errorstream in wrdy:
                self.writeReady(self.errorstream.fileno())

        self.eventExit = None

    def connectDebugger(self,port,remoteAddress=None):
        """
        Public method to establish a session with the debugger. 
        
        It opens a network connection to the debugger, connects it to stdin, 
        stdout and stderr and saves these file objects in case the application
        being debugged redirects them itself.
        
        @param port the port number to connect to (int)
        @param remoteAddress the network address of the debug server host (string)
        """
        sock = socket.socket(socket.AF_INET,socket.SOCK_STREAM)
        if remoteAddress is None:
            sock.connect((DebugAddress,port))
        else:
            sock.connect((remoteAddress,port))
        sock.setblocking(0)

        sys.stdin = AsyncFile(sock,sys.stdin.mode,sys.stdin.name)
        sys.stdout = AsyncFile(sock,sys.stdout.mode,sys.stdout.name)
        sys.stderr = AsyncFile(sock,sys.stderr.mode,sys.stderr.name)
        
        # save the stream in case a program redirects them
        self.readstream = sys.stdin
        self.writestream = sys.stdout
        self.errorstream = sys.stderr
        
        # attach to the main thread here
        self.attachThread(mainThread=1)

    def unhandled_exception(self, exctype, excval, exctb):
        """
        Private method called to report an uncaught exception.
        
        @param exctype the type of the exception
        @param excval data about the exception
        @param exctb traceback for the exception
        """
        self.mainThread.user_exception(None, (exctype,excval,exctb), 1)
        
    def absPath(self,fn):
        """
        Private method to convert a filename to an absolute name.

        sys.path is used as a set of possible prefixes. The name stays 
        relative if a file could not be found.
        
        @param fn filename (string)
        @return the converted filename (string)
        """
        if os.path.isabs(fn):
            return fn

        # Check the cache.
        if self.fncache.has_key(fn):
            return self.fncache[fn]

        # Search sys.path.
        for p in sys.path:
            afn = os.path.abspath(os.path.join(p,fn))

            if os.path.exists(afn):
                self.fncache[fn] = afn
                d = os.path.dirname(afn)
                if (d not in sys.path) and (d not in self.dircache):
                    self.dircache.append(d)
                return afn

        # Search the additional directory cache
        for p in self.dircache:
            afn = os.path.abspath(os.path.join(p,fn))
            
            if os.path.exists(afn):
                self.fncache[fn] = afn
                return afn
                
        # Nothing found.
        return fn

    def shouldSkip(self, fn):
        """
        Public method to check if a file should be skipped.
        
        @param fn filename to be checked
        @return non-zero if fn represents a file we are 'skipping', zero otherwise.
        """
        if self.mainThread.tracePython:     # trace into Python library
            return 0
            
        # Eliminate anything that is part of the Python installation.
        afn = self.absPath(fn)
        if afn.find(self.skipdir) == 0:
            return 1
            
        return 0
        
    def getRunning(self):
        """
        Public method to return the main script we are currently running.
        """
        return self.running

    def progTerminated(self,status):
        """
        Private method to tell the debugger that the program has terminated.
        
        @param status the return status
        """
        if status is None:
            status = 0
        else:
            try:
                int(status)
            except:
                status = 1

        self.set_quit()
        self.running = None
        self.write('%s%d\n' % (ResponseExit,status))

    def dumpVariables(self, frmnr, scope, filter):
        """
        Private method to return the variables of a frame to the debug server.
        
        @param frmnr distance of frame reported on. 0 is the current frame (int)
        @param scope 1 to report global variables, 0 for local variables (int)
        @param filter the indices of variable types to be filtered (list of int)
        """
        f = self.currentThread.getCurrentFrame()
        
        while f is not None and frmnr > 0:
            f = f.f_back
            frmnr -= 1
        
        if f is None:
            return
        
        if scope:
            dict = f.f_globals
        else:
            dict = f.f_locals
            
            if f.f_globals is f.f_locals:
                scope = -1
                
        varlist = [scope]
        
        if scope != -1:
            keylist = dict.keys()
            
            vlist = self.formatVariablesList(keylist, dict, filter)
            varlist.extend(vlist)
            
        self.write('%s%s\n' % (ResponseVariables, str(varlist)))
    
    def dumpVariable(self, var, frmnr, scope, filter):
        """
        Private method to return the variables of a frame to the debug server.
        
        @param var list encoded name of the requested variable (list of strings)
        @param frmnr distance of frame reported on. 0 is the current frame (int)
        @param scope 1 to report global variables, 0 for local variables (int)
        @param filter the indices of variable types to be filtered (list of int)
        """
        f = self.currentThread.getCurrentFrame()
        
        while f is not None and frmnr > 0:
            f = f.f_back
            frmnr -= 1
        
        if f is None:
            return
        
        if scope:
            dict = f.f_globals
        else:
            dict = f.f_locals
            
            if f.f_globals is f.f_locals:
                scope = -1
                
        varlist = [scope, var]
        
        if scope != -1:
            # search the correct dictionary
            i = 0
            while i < len(var):
                ndict = {}
                # this has to be in line with VariablesViewer.indicators
                if var[i][-2:] in ["[]", "()", "{}"]:
                    try:
                        exec 'ndict.update(dict["%s"][%s].__dict__)' % (var[i][:-2], var[i+1])
                    except:
                        pass
                    try:
                        cdict = {}
                        exec 'slv = dict["%s"][%s].__slots__' % (var[i][:-2], var[i+1])
                        for v in slv:
                            exec 'cdict[v] = dict["%s"][%s].%s' % (var[i][:-2], var[i+1], v)
                        ndict.update(cdict)
                    except:
                        pass
                    i += 1
                else:
                    try:
                        ndict.update(dict[var[i]].__dict__)
                    except:
                        pass
                    try:
                        cdict = {}
                        slv = dict[var[i]].__slots__
                        for v in slv:
                            exec 'cdict[v] = dict[var[i]].%s' % v
                        ndict.update(cdict)
                    except:
                        pass
                i += 1
                dict = ndict
                        
            # format the dictionary found
            vlist = self.formatVariablesList(dict.keys(), dict, filter)
            varlist.extend(vlist)
            
        self.write('%s%s\n' % (ResponseVariable, str(varlist)))
        
    def formatVariablesList(self, keylist, dict, filter = [], classdict = 0, prefix = ''):
        """
        Private method to produce a formated variables list.
        
        The dictionary passed in to it is scanned. If classdict is false,
        it builds a list of all class instances in dict. If it is
        true, we are formatting a class dictionary. In this case
        we prepend prefix to the variable names. Variables are
        only added to the list, if their type is not contained 
        in the filter list. The formated variables list (a list of 
        tuples of 3 values) and the list of class instances is returned.
        
        @param keylist keys of the dictionary
        @param dict the dictionary to be scanned
        @param filter the indices of variable types to be filtered. Variables are
              only added to the list, if their type is not contained 
              in the filter list.
        @param classdict boolean indicating the formating of a class or
              module dictionary. If classdict is false,
              it builds a list of all class instances in dict. If it is
              true, we are formatting a class dictionary. In this case
              we prepend prefix to the variable names.
        @param prefix prefix to prepend to the variable names (string)
        @return A tuple consisting of a list of formatted variables and a list of
            class instances. Each variable entry is a tuple of three elements,
            the variable name, its type and value.
        """
        varlist = []
        
        for key in keylist:
            # filter hidden attributes (filter #0)
            if 0 in filter and str(key)[:2] == '__':
                continue
            
            # special handling for '__builtins__' (it's way too big)
            if key == '__builtins__':
                rvalue = '<module __builtin__ (built-in)>'
                valtype = 'module'
            else:
                value = dict[key]
                valtypestr = str(type(value))[1:-1]
                    
                if valtypestr.split(' ',1)[0] == 'class':
                    # handle new class type of python 2.2+
                    if ConfigVarTypeStrings.index('instance') in filter:
                        continue
                    valtype = valtypestr[7:-1]
                else:
                    valtype = valtypestr[6:-1]
                    try:
                        if ConfigVarTypeStrings.index(valtype) in filter:
                            continue
                    except ValueError:
                        if ConfigVarTypeStrings.index('other') in filter:
                            continue
                    
                try:
                    rvalue = repr(value)
                except:
                    rvalue = ''
                    
            varlist.append((key, valtype, rvalue))
        
        return varlist
        
    def completionList(self, text):
        """
        Private slot to handle the request for a commandline completion list.
        
        @param text the text to be completed (string)
        """
        completerDelims = ' \t\n`~!@#$%^&*()-=+[{]}\\|;:\'",<>/?'
        
        completions = []
        state = 0
        # find position of last delim character
        pos = -1
        while pos >= -len(text):
            if text[pos] in completerDelims:
                if pos == -1:
                    text = ''
                else:
                    text = text[pos+1:]
                break
            pos -= 1
        
        try:
            comp = self.complete(text, state)
        except:
            comp = None
        while comp is not None:
            completions.append(comp)
            state += 1
            try:
                comp = self.complete(text, state)
            except:
                comp = None
            
        self.write("%s%s||%s\n" % (ResponseCompletion, str(completions), text))

    def startDebugger(self, filename=None, host=None, port=None, enableTrace=1):
        """
        Method used to start the remote debugger.
        
        @param filename the program to be debugged (string)
        @param host hostname of the debug server (string)
        @param port portnumber of the debug server (int)
        @param enableTrace flag to enable the tracing function (boolean)
        """
        global debugClient
        if host is None:
            host = os.getenv('ERICHOST', 'localhost')
        if port is None:
            port = os.getenv('ERICPORT', 42424)
            
        self.connectDebugger(port, socket.gethostbyname(host))
        if filename is not None:
            self.running = os.path.abspath(filename)
        else:
            try:
                self.running = os.path.abspath(sys.argv[0])
            except:
                pass
        self.passive = 1
        self.write("%s%s\n" % (PassiveStartup, self.running))
        self.interact()
        
        # setup the debugger variables
        self.fncache = {}
        self.dircache = []
        self.mainFrame = None
        self.inRawMode = 0
        
        self.attachThread(mainThread=1)
        
        # set the system exception handling function to ensure, that
        # we report on all unhandled exceptions
        sys.excepthook = self.unhandled_exception
        
        # now start debugging
        if enableTrace:
            self.mainThread.set_trace()
        
    def startProgInDebugger(self, progargs, wd = '', host=None, port=None):
        """
        Method used to start the remote debugger.
        
        @param progargs commandline for the program to be debugged 
            (list of strings)
        @param wd working directory for the program execution (string)
        @param host hostname of the debug server (string)
        @param port portnumber of the debug server (int)
        """
        if host is None:
            host = os.getenv('ERICHOST', 'localhost')
        if port is None:
            port = os.getenv('ERICPORT', 42424)
            
        self.connectDebugger(port, socket.gethostbyname(host))
        
        self.fncache = {}
        self.dircache = []
        sys.argv = progargs[:]
        sys.argv[0] = os.path.abspath(sys.argv[0])
        sys.path[0] = os.path.dirname(sys.argv[0])
        if wd == '':
            os.chdir(sys.path[0])
        else:
            os.chdir(wd)
        self.running = sys.argv[0]
        self.mainFrame = None
        self.inRawMode = 0
        
        self.passive = 1
        self.write("%s%s\n" % (PassiveStartup, self.running))
        self.interact()
        
        self.attachThread(mainThread = 1)
        
        # set the system exception handling function to ensure, that
        # we report on all unhandled exceptions
        sys.excepthook = self.unhandled_exception
        
        # This will eventually enter a local event loop.
        # Note the use of backquotes to cause a repr of self.running. The
        # need for this is on Windows os where backslash is the path separator.
        # They will get inadvertantly stripped away during the eval causing IOErrors
        # if self.running is passed as a normal str.
        sys.modules['__main__'] = self.debugMod
        res = self.mainThread.run('execfile(' + `self.running` + ')',self.debugMod.__dict__)
        self.progTerminated(res)

    def run_call(self, scriptname, func, *args):
        """
        Public method used to start the remote debugger and call a function.
        
        @param scriptname name of the script to be debugged (string)
        @param func function to be called
        @param *args arguments being passed to func
        @return result of the function call
        """
        self.startDebugger(scriptname, enableTrace = 0)
        res = self.mainThread.runcall(func, *args)
        self.progTerminated(0)
        return res
        
    def main(self):
        """
        Public method implementing the main method.
        """
        if '--' in sys.argv:
            args = sys.argv[1:]
            host = None
            port = None
            wd = ''
            while args[0]:
                if args[0] == '-h':
                    host = args[1]
                    del args[0]
                    del args[1]
                elif args[0] == '-p':
                    port = args[1]
                    del args[0]
                    del args[1]
                elif args[0] == '-w':
                    wd = args[1]
                    del args[0]
                    del args[1]
                elif args[0] == '--':
                    del args[0]
                    break
            if not args:
                print "No program given. Aborting!"
            else:
                self.startProgInDebugger(args, wd, host, port)
        else:
            try:
                port = int(sys.argv[1])
            except:
                port = -1
            try:
                remoteAddress = sys.argv[2]
            except:
                remoteAddress = None
            sys.argv = ['']
            sys.path.insert(0, '')
            if port >= 0:
                self.connectDebugger(port, remoteAddress)
                self.interact()
