# -*- coding: utf-8 -*-

# Copyright (c) 2002, 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the project properties dialog.
"""

import os
import string

from qt import *

from PropertiesForm import PropertiesForm
import Preferences

class PropertiesDialog(PropertiesForm):
    """
    Class implementing the project properties dialog.
    """
    def __init__(self,project,new = 1,parent = None,name = None,modal = 0,fl = 0):
        """
        Constructor
        
        @param project reference to the project object
        @param new flag indicating the generation of a new project
        @param parent parent widget of this dialog (QWidget)
        @param name name of this dialog (string or QString)
        @param modal flag indicating a modal dialog
        @param fl window flags
        """
        PropertiesForm.__init__(self,parent,name,1,fl)
        
        self.project = project
        
        if not new:
            (name, dummy) = os.path.splitext(self.project.pfile)
            self.nameEdit.setText(os.path.basename(name))
            self.dirEdit.setText(self.project.ppath)
            try:
                self.versionEdit.setText(self.project.pdata["VERSION"][0])
            except IndexError:
                pass
            try:
                self.mainscriptEdit.setText(self.project.pdata["MAINSCRIPT"][0])
            except IndexError:
                pass
            try:
                self.authorEdit.setText(self.project.pdata["AUTHOR"][0])
            except IndexError:
                pass
            try:
                self.emailEdit.setText(self.project.pdata["EMAIL"][0])
            except IndexError:
                pass
            try:
                self.descriptionEdit.setText(\
                    string.join(self.project.pdata["DESCRIPTION"], '\n'))
            except:
                pass
            if self.project.vcs is not None:
                self.vcsCheckBox.setChecked(1)
            self.vcsCheckBox.setEnabled(0)
        else:
            hp = os.getcwd()
            hp = hp + os.sep
            self.dirEdit.setText(hp)
            self.versionEdit.setText('0.1')
            
    def handleDirDialog(self):
        """
        Private slot to display a directory selection dialog.
        """
        directory = QFileDialog.getExistingDirectory(self.dirEdit.text(),
            self, None, self.trUtf8("Select project directory"), 1)
            
        if not directory.isNull():
            self.dirEdit.setText(QDir.convertSeparators(directory))
        
    def handleFileDialog(self):
        """
        Private slot to display a file selection dialog.
        """
        dir = self.dirEdit.text()
        if dir.isEmpty():
            dir = QDir.currentDirPath()
        fn = QFileDialog.getOpenFileName(dir,
            self.trUtf8("Python Files (*.py);;All Files (*)"),
            self, None, self.trUtf8("Select main script file"))
            
        if not fn.isNull():
            ppath = self.dirEdit.text()
            if not ppath.isEmpty():
                ppath = QDir(ppath).absPath()
                ppath.append(QDir.separator())
                fn.replace(QRegExp(ppath), "")
            self.mainscriptEdit.setText(QDir.convertSeparators(fn))

    def storeData(self):
        """
        Public method to store the entered/modified data.
        """
        self.project.ppath = os.path.abspath(str(self.dirEdit.text()))
        fn = self.nameEdit.text()
        if not fn.isEmpty():
            self.project.name = str(fn)
            if Preferences.getProject("CompressedProjectFiles"):
                fn = "%s.e3pz" % str(fn)
            else:
                fn = "%s.e3p" % str(fn)
            self.project.pfile = os.path.join(self.project.ppath, fn)
        else:
            self.project.pfile = ""
        self.project.pdata["VERSION"] = [str(self.versionEdit.text())]
        fn = self.mainscriptEdit.text()
        if not fn.isEmpty():
            fn = str(fn).replace(self.project.ppath+os.sep, "")
            self.project.pdata["MAINSCRIPT"] = [fn]
            self.project.translationsRoot, dummy = os.path.splitext(fn)
        self.project.pdata["AUTHOR"] = [str(self.authorEdit.text())]
        self.project.pdata["EMAIL"] = [str(self.emailEdit.text())]
        text = str(self.descriptionEdit.text())
        self.project.pdata["DESCRIPTION"] = text.split('\n')
        
    def vcsSelected(self):
        """
        Public method to retrieve the vcs system support.
        
        @return flag indicating whether VCS support is selected
        """
        return self.vcsCheckBox.isChecked()
