# -*- coding: utf-8 -*-

# Copyright (c) 2002, 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to enter options used to start a project in the VCS.
"""

import os

from qt import *

from OptionsForm import vcsOptionsForm
from Config import ConfigVcsSystems

class vcsOptionsDialog(vcsOptionsForm):
    """
    Class implementing a dialog to enter options used to start a project in the VCS.
    """
    def __init__(self, project, archiveName, archiveEditable=0, parent=None):
        """
        Constructor
        
        @param project reference to the project object
        @param archiveName name of the archive (string or QString)
        @param archiveEditable flag indicating that the archives name may be changed
                (boolean)
        @param parent parent widget (QWidget)
        """
        vcsOptionsForm.__init__(self, parent, None, 1)
        
        self.project = project
        
        vcsSystems = QStringList()
        for vcssys in ConfigVcsSystems:
            vcsSystems.append(vcssys)
        self.vcsComboBox.insertStringList(vcsSystems)
        
        hd = QDir.convertSeparators(QDir.homeDirPath())
        hd = os.path.join(str(hd), ConfigVcsSystems[0].lower() + 'root')
        self.vcsDirectoryEdit.setText(hd)
        
        self.vcsArchiveEdit.setText(archiveName)
        self.vcsArchiveEdit.setReadOnly(not archiveEditable)
        
        self.vcsVendorEdit.setText(archiveName)
        
        self.cvsTagValidator = QRegExpValidator(QRegExp(r"[a-zA-Z][a-zA-Z0-9_-]*"), self)
        
    def handleDirectory(self):
        """
        Private slot to display a directory selection dialog.
        """
        directory = QFileDialog.getExistingDirectory(self.vcsDirectoryEdit.text(),
            self, None, self.trUtf8("Select VCS-Directory"), 1)
            
        if not directory.isNull():
            self.vcsDirectoryEdit.setText(QDir.convertSeparators(directory))
            
    def handleComboBox(self, vcsSystem):
        """
        Private slot to set the vcs root directory entry.
        
        @param vcsSystem name of the VCS system
        """
        hd = QDir.homeDirPath()
        hd.append('/').append(vcsSystem.lower()).append('root')
        self.vcsDirectoryEdit.setText(hd)
        
        if str(vcsSystem.lower()) == "cvs":
            txt = self.vcsVendorEdit.text()
            txt.replace(QRegExp(r"[^a-zA-Z0-9_-]"), "_")
            self.vcsVendorEdit.setText(txt)
            self.vcsVendorEdit.setValidator(self.cvsTagValidator)
            self.vcsReleaseEdit.setValidator(self.cvsTagValidator)
        else:
            self.vcsVendorEdit.setText(self.vcsArchiveEdit.text())
            self.vcsVendorEdit.clearValidator()
            self.vcsReleaseEdit.clearValidator()
        
    def getData(self):
        """
        Public slot to retrieve the data entered into the dialog.
        
        @return a tuple of 6 strings and one boolean (vcs system, vcs directory,
            archive name, vendor label, log message, release label,
            command options).
        """
        return (str(self.vcsComboBox.currentText()),
            str(self.vcsDirectoryEdit.text()),
            str(self.vcsArchiveEdit.text()),
            str(self.vcsVendorEdit.text()),
            str(self.vcsLogEdit.text()),
            str(self.vcsReleaseEdit.text()),
            self.optionsCheckBox.isChecked()
        )
