# -*- coding: utf-8 -*-

# Copyright (c) 2002, 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to show the output of the cvs status command process.
"""

from qt import *

from LogForm import LogForm

class CvsStatusDialog(LogForm):
    """
    Module implementing a dialog to show the output of the cvs status command process.
    """
    def __init__(self, vcs, parent = None):
        """
        Constructor
        
        @param vcs reference to the vcs object
        @param parent parent widget (QWidget)
        """
        LogForm.__init__(self, parent)
        
        self.contentsLabel.setText(self.trUtf8('Status:'))
        self.setCaption(self.trUtf8('CVS Status'))
        self.saveButton.hide()
        
        QWhatsThis.add(self.contents,self.tr("<b>CVS Status</b>\n"
            "<p>This shows the output of the cvs status command.</p>"
        ))
        QWhatsThis.add(self.errors,self.tr("<b>CVS status errors</b>\n"
            "<p>This shows possible error messages of the cvs status"
            " command.</p>"
        ))
        
        self.setWFlags(self.getWFlags() | Qt.WDestructiveClose)
        self.process = QProcess(self)
        self.vcs = vcs
        
        self.connect(self.process, SIGNAL('readyReadStdout()'),
            self.handleReadStdout)
        self.connect(self.process, SIGNAL('readyReadStderr()'),
            self.handleReadStderr)
        self.connect(self.process, SIGNAL('processExited()'),
            self.handleProcessExited)
            
    def closeEvent(self, e):
        """
        Private slot implementing a close event handler.
        
        @param e close event (QCloseEvent)
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        e.accept()
        
    def start(self, fn):
        """
        Public slot to start the cvs status command.
        
        @param fn filename to be diffed (string)
        """
        self.filename = fn
        dname, fname = self.vcs.splitPath(fn)
        
        self.process.kill()
        self.contents.clear()
        
        self.process.clearArguments()
        self.process.addArgument('cvs')
        self.vcs.addArguments(self.process, self.vcs.options['global'])
        self.process.addArgument('status')
        self.vcs.addArguments(self.process, self.vcs.options['status'])
        self.process.addArgument(fname)
        self.process.setWorkingDirectory(QDir(dname))
        
        self.process.start()
        self.setCaption(self.trUtf8('CVS Status %1').arg(self.filename))
        
    def handleProcessExited(self):
        """
        Private slot to handle the processExited signal.
        
        Just do nothing.
        """
        pass
        
    def handleReadStdout(self):
        """
        Private slot to handle the readyReadStdout signal.
        
        It reads the output of the process, formats it and inserts it into
        the contents pane.
        """
        self.contents.setTextFormat(QTextBrowser.PlainText)
        
        while self.process.canReadLineStdout():
            s = self.process.readLineStdout()
            self.contents.append(s)
                
    def handleReadStderr(self):
        """
        Private slot to handle the readyReadStderr signal.
        
        It reads the error output of the process and inserts it into the
        error pane.
        """
        while self.process.canReadLineStderr():
            s = self.process.readLineStderr()
            self.errors.append(s)
