/* packet-juniper.c
 * Routines for Juniper Networks, Inc. packet disassembly
 * Copyright 2005 Hannes Gredler <hannes@juniper.net>
 *
 * $Id: packet-juniper.c 15081 2005-07-26 08:59:32Z etxrab $
 *
 * Ethereal - Network traffic analyzer
 * By Gerald Combs <gerald@ethereal.com>
 * Copyright 1998 Gerald Combs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <glib.h>
#include <epan/packet.h>
#include "etypes.h"
#include <epan/prefs.h>
#include <epan/addr_resolv.h>
#include "ppptypes.h"
#include "packet-ppp.h"
#include "packet-ip.h"

#define JUNIPER_FLAG_PKT_OUT        0x00     /* Outgoing packet */
#define JUNIPER_FLAG_PKT_IN         0x01     /* Incoming packet */
#define JUNIPER_FLAG_NO_L2          0x02     /* L2 header stripped */
#define JUNIPER_ATM2_PKT_TYPE_MASK  0x70
#define JUNIPER_ATM2_GAP_COUNT_MASK 0x3F
#define JUNIPER_PCAP_MAGIC          0x4d4743

#define JUNIPER_ATM1   1
#define JUNIPER_ATM2   2

#define JUNIPER_HDR_SNAP   0xaaaa03
#define JUNIPER_HDR_NLPID  0xfefe03
#define JUNIPER_HDR_CNLPID 0x03
#define JUNIPER_HDR_PPP    0xff03

#define ML_PIC_COOKIE_LEN 2
#define LS_PIC_COOKIE_LEN 4
#define AS_PIC_COOKIE_LEN 8

#define GSP_SVC_REQ_APOLLO 0x40
#define GSP_SVC_REQ_LSQ    0x47

#define LSQ_COOKIE_RE         (1 << 3)
#define LSQ_COOKIE_DIR        (1 << 2)
#define LSQ_L3_PROTO_SHIFT     4
#define LSQ_L3_PROTO_MASK     (0x17 << LSQ_L3_PROTO_SHIFT)
#define LSQ_L3_PROTO_IPV4     (0 << LSQ_L3_PROTO_SHIFT)
#define LSQ_L3_PROTO_IPV6     (1 << LSQ_L3_PROTO_SHIFT)
#define LSQ_L3_PROTO_MPLS     (2 << LSQ_L3_PROTO_SHIFT)
#define LSQ_L3_PROTO_ISO      (3 << LSQ_L3_PROTO_SHIFT)

static const value_string juniper_direction_vals[] = {
    {JUNIPER_FLAG_PKT_OUT, "Out"},
    {JUNIPER_FLAG_PKT_IN,  "In"},
    {0,                    NULL}
};

static const value_string juniper_l2hdr_presence_vals[] = {
    { 0, "Present"},
    { 2, "none"},
    {0,                    NULL}
};

static int proto_juniper = -1;

static int hf_juniper_magic = -1;
static int hf_juniper_direction = -1;
static int hf_juniper_l2hdr_presence = -1;
static int hf_juniper_atm1_cookie = -1;
static int hf_juniper_atm2_cookie = -1;
static int hf_juniper_mlpic_cookie = -1;
static int hf_juniper_lspic_cookie = -1;
static int hf_juniper_aspic_cookie = -1;

static gint ett_juniper = -1;

static dissector_handle_t ipv4_handle;
static dissector_handle_t ipv6_handle;
static dissector_handle_t mpls_handle;
static dissector_handle_t llc_handle;
static dissector_handle_t eth_handle;
static dissector_handle_t ppp_handle;
static dissector_handle_t data_handle;

static dissector_table_t osinl_subdissector_table;
static dissector_table_t osinl_excl_subdissector_table;

int dissect_juniper_header(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, proto_item *ti, guint8 *flags);
int dissect_juniper_payload_proto(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree,proto_item *ti, guint proto, guint offset);
static void dissect_juniper_atm(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, guint16 atm_pictype);
static gboolean ppp_heuristic_guess(guint16 proto);
static guint ip_heuristic_guess(guint8 ip_header_byte);
static guint juniper_svc_cookie_len (guint64 cookie);
static guint juniper_svc_cookie_proto (guint64 cookie);

enum {
    PROTO_UNKNOWN = 0,
    PROTO_IP,
    PROTO_IP6,
    PROTO_PPP,
    PROTO_ISO,
    PROTO_MPLS,
    PROTO_LLC,
    PROTO_LLC_SNAP,
    PROTO_ETHER,
    PROTO_OAM,
};

static const value_string juniper_proto_vals[] = {
    {PROTO_IP, "IPv4"},
    {PROTO_IP6, "IPv6"},
    {PROTO_PPP, "PPP"},
    {PROTO_ISO, "OSI"},
    {PROTO_MPLS, "MPLS"},
    {PROTO_LLC, "LLC"},
    {PROTO_LLC_SNAP, "LLC/SNAP"},
    {PROTO_ETHER, "Ethernet"},
    {PROTO_OAM, "ATM OAM Cell"},
    {0,                    NULL}
};

/* the first subtree is accessed by several routines */
static proto_tree *juniper_subtree = NULL;

/* generic juniper header dissector  */
int
dissect_juniper_header(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, proto_item *ti, guint8 *flags)
{
  proto_item *tisub;
  guint8     direction,l2hdr_presence,proto;
  guint32    magic_number;

  tvbuff_t   *next_tvb;

  magic_number = tvb_get_ntoh24(tvb, 0);
  *flags = tvb_get_guint8(tvb, 3);
  direction = *flags & JUNIPER_FLAG_PKT_IN;
  l2hdr_presence = *flags & JUNIPER_FLAG_NO_L2;

  juniper_subtree = proto_item_add_subtree(ti, ett_juniper);          
  tisub = proto_tree_add_text (juniper_subtree, tvb, 0, 3,
                            "Magic-Number: 0x%06x (%scorrect)", 
                            magic_number,
                            (magic_number == JUNIPER_PCAP_MAGIC) ?  "" : "in" );
    
  if (magic_number != JUNIPER_PCAP_MAGIC)
     return -1;
  
  tisub = proto_tree_add_uint_format (juniper_subtree, hf_juniper_direction, tvb, 3, 1,
                            direction, "Direction: %s",
                            val_to_str(direction,juniper_direction_vals,"Unknown"));
  
  tisub = proto_tree_add_uint_format (juniper_subtree, hf_juniper_l2hdr_presence, tvb, 3, 1,
                            l2hdr_presence, "L2-header: %s",
                            val_to_str(l2hdr_presence,juniper_l2hdr_presence_vals,"Unknown"));

  if ((*flags & JUNIPER_FLAG_NO_L2) == JUNIPER_FLAG_NO_L2) { /* no link header present ? */
      next_tvb = tvb_new_subset(tvb, 8, -1, -1);
      proto = ip_heuristic_guess(tvb_get_guint8(tvb, 8)); /* try IP */
      if (proto != PROTO_UNKNOWN) {
          dissect_juniper_payload_proto(tvb, pinfo, tree, ti, proto, 8);
      }
      return -1;
  }

  return 4; /* 4 bytes parsed */

}

/* print the payload protocol  */
int
dissect_juniper_payload_proto(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree,
                              proto_item *ti, guint proto, guint offset)
{

    tvbuff_t   *next_tvb;
    guint8     nlpid; 

    ti = proto_tree_add_text (juniper_subtree, tvb, offset, 0, "Payload Type: %s",
                            val_to_str(proto,juniper_proto_vals,"Unknown"));

    next_tvb = tvb_new_subset(tvb, offset, -1, -1);  
  
    switch (proto) {
    case PROTO_IP:
        call_dissector(ipv4_handle, next_tvb, pinfo, tree);
        break;
    case PROTO_IP6:
        call_dissector(ipv6_handle, next_tvb, pinfo, tree);  
        break;
    case PROTO_PPP:
        call_dissector(ppp_handle, next_tvb, pinfo, tree);
        break;
    case PROTO_ETHER:
        call_dissector(eth_handle, next_tvb, pinfo, tree);
        break;
    case PROTO_LLC:
    case PROTO_LLC_SNAP:
        call_dissector(llc_handle, next_tvb, pinfo, tree);
        break;
    case PROTO_ISO:
        nlpid = tvb_get_guint8(tvb, offset);
        if(dissector_try_port(osinl_subdissector_table, nlpid, next_tvb, pinfo, tree))
            return 0;
        next_tvb = tvb_new_subset(tvb, offset+1, -1, -1);
        if(dissector_try_port(osinl_excl_subdissector_table, nlpid, next_tvb, pinfo, tree))
            return 0;
        break;
    case PROTO_OAM: /* FIXME call OAM disector without leading HEC byte */
    default:
        call_dissector(data_handle, next_tvb, pinfo, tree);  
        break;
    }

    return 0;
}


/* MLPPP dissector */
static void
dissect_juniper_mlppp(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree)
{
  proto_item *ti;
  guint      offset;
  int        bytes_processed;
  guint8     flags;
  guint64    aspic_cookie;
  guint32    lspic_cookie;
  guint16    mlpic_cookie;
  guint      proto,cookie_len;

  if (check_col(pinfo->cinfo, COL_PROTOCOL))
      col_set_str(pinfo->cinfo, COL_PROTOCOL, "Juniper MLPPP");
  if (check_col(pinfo->cinfo, COL_INFO))
      col_clear(pinfo->cinfo, COL_INFO);

  offset = 0;

  ti = proto_tree_add_text (tree, tvb, offset, 4, "Juniper MLPPP");

  /* parse header, match mgc, extract flags and build first tree */
  bytes_processed = dissect_juniper_header(tvb, pinfo, tree, ti, &flags);

  if(bytes_processed == -1)
      return;
  else
      offset+=bytes_processed;

  aspic_cookie = tvb_get_ntoh64(tvb,offset);
  proto = juniper_svc_cookie_proto(aspic_cookie);
  cookie_len = juniper_svc_cookie_len(aspic_cookie);

  if (cookie_len == AS_PIC_COOKIE_LEN)
      ti = proto_tree_add_uint64(juniper_subtree, hf_juniper_aspic_cookie,
                                 tvb, offset, AS_PIC_COOKIE_LEN, aspic_cookie);
  if (cookie_len == LS_PIC_COOKIE_LEN) {
      lspic_cookie = tvb_get_ntohl(tvb,offset);
      ti = proto_tree_add_uint(juniper_subtree, hf_juniper_lspic_cookie,
                               tvb, offset, LS_PIC_COOKIE_LEN, lspic_cookie);
  }

  /* no cookie pattern identified - lets guess from now on */

  /* child link of an LS-PIC bundle ? */
  if (cookie_len == 0 && tvb_get_ntohs(tvb, offset) == JUNIPER_HDR_PPP) {
      proto = PROTO_PPP;
      offset += 2;
  }

  /* ML-PIC ? */
  if (cookie_len == 0 && ppp_heuristic_guess(tvb_get_ntohs(tvb, offset+2))) {
      proto = PROTO_PPP;
      cookie_len = 2;
      mlpic_cookie = tvb_get_ntohs(tvb, offset);
      ti = proto_tree_add_uint(juniper_subtree, hf_juniper_mlpic_cookie,
                               tvb, offset, ML_PIC_COOKIE_LEN, mlpic_cookie);
  }

  /* child link of an ML-PIC bundle ? */
  if (cookie_len == 0 && ppp_heuristic_guess(tvb_get_ntohs(tvb, offset))) {
      proto = PROTO_PPP;
  }

  ti = proto_tree_add_text (juniper_subtree, tvb, offset, 0, "Cookie length %u",cookie_len);
  offset += cookie_len;

  dissect_juniper_payload_proto(tvb, pinfo, tree, ti, proto, offset);
      
}


/* PPPoE dissector */
static void
dissect_juniper_pppoe(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree)
{
  proto_item *ti;
  guint      offset;
  int        bytes_processed;
  guint8     flags;

  if (check_col(pinfo->cinfo, COL_PROTOCOL))
      col_set_str(pinfo->cinfo, COL_PROTOCOL, "Juniper PPPoE");
  if (check_col(pinfo->cinfo, COL_INFO))
      col_clear(pinfo->cinfo, COL_INFO);

  offset = 0;

  ti = proto_tree_add_text (tree, tvb, offset, 4, "Juniper PPPoE PIC");

  /* parse header, match mgc, extract flags and build first tree */
  bytes_processed = dissect_juniper_header(tvb, pinfo, tree, ti, &flags);

  if(bytes_processed == -1)
      return;
  else
      offset+=bytes_processed;

  dissect_juniper_payload_proto(tvb, pinfo, tree, ti, PROTO_ETHER, offset);

}


/* wrapper for passing the PIC type to the generic ATM dissector */
static void
dissect_juniper_atm1(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree)
{ 
    dissect_juniper_atm(tvb,pinfo,tree, JUNIPER_ATM1);
}

/* wrapper for passing the PIC type to the generic ATM dissector */
static void
dissect_juniper_atm2(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree)
{ 
    dissect_juniper_atm(tvb,pinfo,tree, JUNIPER_ATM2);
}

/* generic ATM dissector */
static void
dissect_juniper_atm(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree, guint16 atm_pictype)
{
  proto_item *ti,*tisub;
  guint8     next_proto = PROTO_UNKNOWN,atm1_header_len,atm2_header_len,flags;
  guint32    cookie1, proto;
  guint64    cookie2;
  guint      offset = 0;
  int        bytes_processed;
  tvbuff_t   *next_tvb;

  switch (atm_pictype) {
  case JUNIPER_ATM1:
      if (check_col(pinfo->cinfo, COL_PROTOCOL))
          col_set_str(pinfo->cinfo, COL_PROTOCOL, "Juniper ATM1");
      break;
  case JUNIPER_ATM2:
      if (check_col(pinfo->cinfo, COL_PROTOCOL))
          col_set_str(pinfo->cinfo, COL_PROTOCOL, "Juniper ATM2");
      break;
  default: /* should not happen */
      if (check_col(pinfo->cinfo, COL_PROTOCOL))
          col_set_str(pinfo->cinfo, COL_PROTOCOL, "Juniper ATM unknown");
      return;

  }

  if (check_col(pinfo->cinfo, COL_INFO))
      col_clear(pinfo->cinfo, COL_INFO);

  switch (atm_pictype) {
  case JUNIPER_ATM1:
      ti = proto_tree_add_text (tree, tvb, 0, 4 , "Juniper ATM1 PIC");
      break;
  case JUNIPER_ATM2:
      ti = proto_tree_add_text (tree, tvb, 0, 4 , "Juniper ATM2 PIC");
      break;
  default: /* should not happen */
      ti = proto_tree_add_text (tree, tvb, 0, 0 , "Juniper unknown ATM PIC");
      return;
  }

  /* parse header, match mgc, extract flags and build first tree */
  bytes_processed = dissect_juniper_header(tvb, pinfo, tree, ti, &flags);
  if(bytes_processed == -1)
      return;
  else
      offset+=bytes_processed;

  if ((flags & JUNIPER_FLAG_NO_L2) == JUNIPER_FLAG_NO_L2) {
      atm1_header_len = 4;
      atm2_header_len = 4;
  }
  else {
      atm1_header_len = 4;
      atm2_header_len = 8;
  }

  cookie1 = tvb_get_ntohl(tvb,4);
  cookie2 = tvb_get_ntoh64(tvb,4);

  switch (atm_pictype) {
  case JUNIPER_ATM1:
      tisub = proto_tree_add_uint(juniper_subtree, hf_juniper_atm1_cookie, tvb, 4, 4, cookie1);
      offset += atm1_header_len;
      if (cookie1 & 0x80000000) /* OAM cell ? */
          next_proto = PROTO_OAM;
      break;
  case JUNIPER_ATM2:
      tisub = proto_tree_add_uint64(juniper_subtree, hf_juniper_atm2_cookie, tvb, 4, 8, cookie2);
      offset += atm2_header_len;
      if (cookie2 & 0x7000) /* OAM cell ? */
          next_proto = PROTO_OAM;
      break;
  default: /* should not happen */
      return;  
  }

  next_tvb = tvb_new_subset(tvb, offset, -1, -1);  

  if (next_proto == PROTO_OAM) {
      dissect_juniper_payload_proto(tvb, pinfo, tree, ti, PROTO_OAM, offset);
      return;
  }

  proto = tvb_get_ntoh24(tvb, offset); /* first try: 24-Bit guess */

  if (proto == JUNIPER_HDR_NLPID) {
      /*
       * This begins with something that appears to be an LLC header for
       * OSI; is this LLC-multiplexed traffic?
       */
      dissect_juniper_payload_proto(tvb, pinfo, tree, ti, PROTO_LLC, offset);
      return;
  }

  if (proto == JUNIPER_HDR_SNAP) {
      /*
       * This begins with something that appears to be an LLC header for
       * SNAP; is this LLC-multiplexed traffic?
       */
      dissect_juniper_payload_proto(tvb, pinfo, tree, ti, PROTO_LLC_SNAP, offset);
      return;
  }

  if ((flags & JUNIPER_FLAG_PKT_IN) != JUNIPER_FLAG_PKT_IN && /* ether-over-1483 encaps ? */
      (cookie1 & JUNIPER_ATM2_GAP_COUNT_MASK) &&
      atm_pictype != JUNIPER_ATM1) {
      dissect_juniper_payload_proto(tvb, pinfo, tree, ti, PROTO_ETHER, offset);
      return;
  }

  proto = tvb_get_ntohs(tvb, offset); /* second try: 16-Bit guess */

  if ( ppp_heuristic_guess( (guint16) proto) && 
       atm_pictype != JUNIPER_ATM1) {
      /*
       * This begins with something that appears to be a PPP protocol
       * type; is this VC-multiplexed PPPoA?
       * That's not supported on ATM1 PICs.
       */
      ti = proto_tree_add_text (juniper_subtree, tvb, offset, 0, "Encapsulation Type: VC-MUX");
      dissect_juniper_payload_proto(tvb, pinfo, tree, ti, PROTO_PPP , offset);
      return;
  }

  proto = tvb_get_guint8(tvb, offset); /* third try: 8-Bit guess */

  if ( proto == JUNIPER_HDR_CNLPID ) {
      /*
       * Cisco style NLPID encaps?
       * Is the 0x03 an LLC UI control field?
       */
      ti = proto_tree_add_text (juniper_subtree, tvb, offset, 1, "Encapsulation Type: Cisco NLPID");
      dissect_juniper_payload_proto(tvb, pinfo, tree, ti, PROTO_ISO , offset+1);
      return;
  }

  next_proto = ip_heuristic_guess( (guint8) proto);
  if (next_proto != PROTO_UNKNOWN) { /* last resort: VC-MUX encaps ? */
      /*
       * This begins with something that might be the first byte of
       * an IPv4 or IPv6 packet; is this VC-multiplexed IP?
       */
      ti = proto_tree_add_text (juniper_subtree, tvb, offset, 0, "Encapsulation Type: VC-MUX");
      dissect_juniper_payload_proto(tvb, pinfo, tree, ti, next_proto , offset);
      return;
  }

  /* could not figure what it is */
  ti = proto_tree_add_text (juniper_subtree, tvb, offset, -1, "Payload Type: unknown");
  call_dissector(data_handle, next_tvb, pinfo, tree);  
}

/* list of Juniper supported PPP proto IDs */
static gboolean
ppp_heuristic_guess(guint16 proto) {

    switch(proto) {
    case PPP_IP :
    case PPP_OSI :
    case PPP_MPLS_UNI :
    case PPP_MPLS_MULTI :
    case PPP_IPCP :
    case PPP_OSICP :
    case PPP_MPLSCP :
    case PPP_LCP :
    case PPP_PAP :
    case PPP_CHAP :
    case PPP_MP :
    case PPP_IPV6 :
    case PPP_IPV6CP :
        return TRUE;
        break;

    default:
        return FALSE; /* did not find a ppp header */
        break;
    }
}

/*
 * return the IP version number based on the first byte of the IP header
 * returns 0 if it does not match a valid first IPv4/IPv6 header byte
 */
static guint
ip_heuristic_guess(guint8 ip_header_byte) {

    switch(ip_header_byte) {
    case 0x45:
    case 0x46:
    case 0x47:
    case 0x48:
    case 0x49:
    case 0x4a:
    case 0x4b:
    case 0x4c:
    case 0x4d:
    case 0x4e:
    case 0x4f:
        return PROTO_IP;
        break;
    case 0x60:
    case 0x61:
    case 0x62:
    case 0x63:
    case 0x64:
    case 0x65:
    case 0x66:
    case 0x67:
    case 0x68:
    case 0x69:
    case 0x6a:
    case 0x6b:
    case 0x6c:
    case 0x6d:
    case 0x6e:
    case 0x6f:
        return PROTO_IP6;
        break;
    default:
        return PROTO_UNKNOWN; /* did not find a ip header */
    }
}

/* return cookie length dep. on cookie SVC id */
static
guint juniper_svc_cookie_len (guint64 cookie) {

    guint8 svc_cookie_id;
    svc_cookie_id = (guint8)(cookie >> 56) & 0xff;

    switch(svc_cookie_id) {
    case 0x54:
        return LS_PIC_COOKIE_LEN;
    case GSP_SVC_REQ_APOLLO:
    case GSP_SVC_REQ_LSQ:
        return AS_PIC_COOKIE_LEN;
    default:
        return 0;
    }
}

/* return the next-level protocol based on cookie input */
static guint
juniper_svc_cookie_proto (guint64 cookie) {

    guint8 svc_cookie_id;
    guint16 lsq_proto;
    guint8 lsq_dir;

    svc_cookie_id = (guint8)(cookie >> 56) & 0xff;
    lsq_proto = (guint16)((cookie >> 16) & LSQ_L3_PROTO_MASK);
    lsq_dir = (guint8)(cookie >> 24) & 0xff;


    switch (svc_cookie_id) {
    case 0x54:
        return PROTO_PPP;
    case GSP_SVC_REQ_APOLLO:
    case GSP_SVC_REQ_LSQ:
        switch(lsq_proto) {
        case LSQ_L3_PROTO_IPV4:
            /* IP traffic going to the RE would not have a cookie
             * -> this must be incoming IS-IS over PPP
             */
            if (lsq_dir == (LSQ_COOKIE_RE|LSQ_COOKIE_DIR))
                return PROTO_PPP;
            else
                return PROTO_IP;
        case LSQ_L3_PROTO_IPV6:
            return PROTO_IP6;
        case LSQ_L3_PROTO_MPLS:
            return PROTO_MPLS;
        case LSQ_L3_PROTO_ISO:
            return PROTO_ISO;
        default:
            return PROTO_UNKNOWN;
        }
    default:
        return PROTO_UNKNOWN;
    }
}


void
proto_register_juniper(void)
{
  static hf_register_info hf[] = {
    { &hf_juniper_magic,
      { "Magic Number", "juniper.magic-number", FT_UINT24, BASE_HEX,
        NULL, 0x0, "", HFILL }},
    { &hf_juniper_direction,
      { "Direction", "juniper.direction", FT_UINT8, BASE_HEX,
        VALS(juniper_direction_vals), 0x0, "", HFILL }},
    { &hf_juniper_l2hdr_presence,
      { "L2 header presence", "juniper.l2hdr", FT_UINT8, BASE_HEX,
        VALS(juniper_l2hdr_presence_vals), 0x0, "", HFILL }},
    { &hf_juniper_atm2_cookie,
      { "Cookie", "juniper.atm2.cookie", FT_UINT64, BASE_HEX,
        NULL, 0x0, "", HFILL }},
    { &hf_juniper_atm1_cookie,
      { "Cookie", "juniper.atm1.cookie", FT_UINT32, BASE_HEX,
        NULL, 0x0, "", HFILL }},
    { &hf_juniper_mlpic_cookie,
    { "Cookie", "juniper.mlpic.cookie", FT_UINT16, BASE_HEX,
        NULL, 0x0, "", HFILL }},
    { &hf_juniper_lspic_cookie,
    { "Cookie", "juniper.lspic.cookie", FT_UINT32, BASE_HEX,
        NULL, 0x0, "", HFILL }},
    { &hf_juniper_aspic_cookie,
    { "Cookie", "juniper.aspic.cookie", FT_UINT64, BASE_HEX,
        NULL, 0x0, "", HFILL }},
  };

  static gint *ett[] = {
    &ett_juniper,
  };

  proto_juniper = proto_register_protocol("Juniper", "Juniper", "juniper");
  proto_register_field_array(proto_juniper, hf, array_length(hf));
  proto_register_subtree_array(ett, array_length(ett));

}

void
proto_reg_handoff_juniper(void)
{
  dissector_handle_t juniper_atm1_handle;
  dissector_handle_t juniper_atm2_handle;
  dissector_handle_t juniper_pppoe_handle;
  dissector_handle_t juniper_mlppp_handle;

  osinl_subdissector_table = find_dissector_table("osinl");
  osinl_excl_subdissector_table = find_dissector_table("osinl.excl");
  eth_handle = find_dissector("eth_withoutfcs");
  ppp_handle = find_dissector("ppp");
  llc_handle = find_dissector("llc");
  ipv4_handle = find_dissector("ip");
  ipv6_handle = find_dissector("ipv6");
  mpls_handle = find_dissector("mpls");
  data_handle = find_dissector("data");

  juniper_atm2_handle = create_dissector_handle(dissect_juniper_atm2, proto_juniper);
  juniper_atm1_handle = create_dissector_handle(dissect_juniper_atm1, proto_juniper);
  juniper_pppoe_handle = create_dissector_handle(dissect_juniper_pppoe, proto_juniper);
  juniper_mlppp_handle = create_dissector_handle(dissect_juniper_mlppp, proto_juniper);
  dissector_add("wtap_encap", WTAP_ENCAP_JUNIPER_ATM2, juniper_atm2_handle);
  dissector_add("wtap_encap", WTAP_ENCAP_JUNIPER_ATM1, juniper_atm1_handle);
  dissector_add("wtap_encap", WTAP_ENCAP_JUNIPER_PPPOE, juniper_pppoe_handle);
  dissector_add("wtap_encap", WTAP_ENCAP_JUNIPER_MLPPP, juniper_mlppp_handle);
}

