/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-account-sel.c,v 1.6 2003/12/10 23:16:45 hoa Exp $
 */

#include "etpan-account-sel.h"
#include "etpan-subapp.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-errors.h"
#include "etpan-tools.h"
#include "etpan-cfg-vfolder.h"
#include "etpan-cfg-account.h"
#include "etpan-cfg-common.h"
#include <stdlib.h>
#include "etpan-thread-manager.h"
#include "etpan-subapp-thread.h"
#include "etpan-account-common.h"
#include "etpan-account-edit.h"
#include "etpan-help-viewer.h"

static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static int display_init(struct etpan_subapp * app);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);

static struct etpan_subapp_driver etpan_account_sel_app_driver = {
  .name = "account-sel",
  .always_handle_key = 0,
  .always_on_top = 1,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  struct etpan_account_common_state common_state;
  
  /* account to replace */
  clistiter * replace;

  /* upcall */
  void (* upcall)(struct etpan_subapp *, int, void *);
  void * upcall_data;
};


static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;

  state = app->data;
  
  etpan_account_common_display(app, &state->common_state, w);
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  r = etpan_account_common_init(&state->common_state);
  if (r != NO_ERROR)
    goto free;
  
  state->replace = NULL;
  
  state->upcall = NULL;
  state->upcall_data = NULL;
  
  subapp->data = state;
  
  return NO_ERROR;
  
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}


struct etpan_subapp * etpan_account_sel_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_account_sel_app_driver);
}

void etpan_account_sel_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_account_common_flush(&state->common_state);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  etpan_account_sel_app_flush(app);
}

int etpan_account_sel_app_set(struct etpan_subapp * app,
    struct etpan_app_config * config,
    struct etpan_account_info * account,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * upcall_data)
{
  struct app_state * state;
  int r;

  state = app->data;
  
  r = etpan_account_common_set(&state->common_state,
    config, account);
  if (r != NO_ERROR)
    return r;
  
  state->upcall = upcall;
  state->upcall_data = upcall_data;
  
  return NO_ERROR;
}

static int delete_account(struct etpan_subapp * app);

static int add_account(struct etpan_subapp * app);

static int edit_account(struct etpan_subapp * app);

static int set_default_account(struct etpan_subapp * app);

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  
  state = app->data;

  etpan_account_common_handle_key(app, &state->common_state, key);

  switch (key) {
  case 'd':
    delete_account(app);
    break;

  case 'e':
    edit_account(app);
    break;

  case 'a':
    add_account(app);
    break;

  case 's':
    set_default_account(app);
    break;

  case KEY_CTRL('G'):
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_ACCOUNT_SEL_CANCEL, state->upcall_data);
    break;
    
  case '\n':
    if (state->upcall != NULL)
      state->upcall(app, ETPAN_ACCOUNT_SEL_VALID, state->upcall_data);
    break;

  case KEY_F(1):
  case '?':
    show_help(app);
    break;
  }
}

static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, "etPan! - select an account");
  return etpan_app_subapp_display_init(app);
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_account_common_set_color(app, &state->common_state);
}

static void done(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_account_common_done(&state->common_state);
  free(state);
}


static void
recursive_folder_replace_account(struct etpan_app_config * config,
    struct mailfolder * folder,
    struct etpan_account_info * old_account,
    struct etpan_account_info * new_account)
{
  unsigned int i;
  struct etpan_account_info * account;
  
  account = etpan_vfolder_get_account(config->vfolder_config,
    config->account_config, folder);
  
  if (account == config->account_config->default_account)
    account = NULL;
  
  if (account == old_account)
    etpan_vfolder_set_account(config->vfolder_config,
        folder, new_account);
  
  for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
    struct mailfolder * child;
    
    child = carray_get(folder->fld_children, i);
    recursive_folder_replace_account(config, child, old_account, new_account);
  }
}

static int delete_account(struct etpan_subapp * app)
{
  struct app_state * state;
  clistiter * cur;
  struct etpan_account_info * account;
  int r;
  
  state = app->data;
  
  account = etpan_account_common_get_account(&state->common_state);
  if (account == NULL) {
    ETPAN_APP_LOG((app->app, "no account selected"));
    return ERROR_INVAL;
  }
  
  for(cur = clist_begin(state->common_state.config->account_config->list) ;
      cur != NULL ; cur = clist_next(cur)) {
    if (clist_content(cur) == account) {
      clist_delete(state->common_state.config->account_config->list, cur);
      break;
    }
  }
  
  recursive_folder_replace_account(state->common_state.config,
      state->common_state.config->vfolder_config->root,
      account, NULL);
  
  if (state->common_state.config->account_config->default_account == account)
    state->common_state.config->account_config->default_account = NULL;
  
  etpan_account_info_free(account);
  
  r = etpan_account_common_set(&state->common_state,
      state->common_state.config, NULL);
  if (r != NO_ERROR) {
    /* ignore errors */
  }
  
  return NO_ERROR;
}

struct etpan_account_info *
etpan_account_sel_app_get_account(struct etpan_subapp * app)
{
  struct app_state * state;

  state = app->data;
  
  return etpan_account_common_get_account(&state->common_state);
}

static int open_account_editor(struct etpan_subapp * app,
    struct etpan_account_info * account);

static int add_account(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;

  state->replace = NULL;
  
  return open_account_editor(app, NULL);
}


static int edit_account(struct etpan_subapp * app)
{
  struct app_state * state;
  clistiter * cur;
  struct etpan_account_info * account;

  state = app->data;
  
  account = etpan_account_common_get_account(&state->common_state);
  if (account == NULL)
    return ERROR_INVAL;
  
  state->replace = NULL;
  for(cur = clist_begin(state->common_state.config->account_config->list) ;
      cur != NULL ; cur = clist_next(cur)) {
    if (clist_content(cur) == account) {
      state->replace = cur;
      break;
    }
  }
  
  return open_account_editor(app, account);
}


static void
account_editor_upcall(struct etpan_subapp * account_editor_app,
    int valid, void * data)
{
  struct etpan_account_info * account;
  struct etpan_subapp * app;
  struct app_state * state;
  int r;
  
  app = data;
  state = app->data;
  
  if (valid == ETPAN_ACCOUNT_EDIT_CANCEL) {
    etpan_app_quit_subapp(account_editor_app);
    return;
  }
  
  account = etpan_account_editor_app_get_account(account_editor_app);
  if (account == NULL)
    goto err;
  
  if (state->replace != NULL) {
    struct etpan_account_info * old_account;
    
    old_account = clist_content(state->replace);

    recursive_folder_replace_account(state->common_state.config,
        state->common_state.config->vfolder_config->root,
        old_account, account);
    
    etpan_account_info_free(old_account);
    
    if (state->common_state.config->account_config->default_account ==
        old_account)
      state->common_state.config->account_config->default_account = account;
    
    state->replace->data = account;
  }
  else {
    r = clist_append(state->common_state.config->account_config->list,
        account);
    if (r < 0) {
      etpan_account_info_free(account);
      goto err;
    }
  }
  
  r = etpan_account_common_set(&state->common_state,
      state->common_state.config, account);
  /* ignore errors */

  etpan_app_quit_subapp(account_editor_app);
  return;
  
 err:
  ETPAN_APP_LOG((app->app, "account editor - could not create account"));
  etpan_app_quit_subapp(account_editor_app);
  return;
}


static int open_account_editor(struct etpan_subapp * app,
    struct etpan_account_info * account)
{
  struct etpan_subapp * account_app;
  
  account_app = etpan_app_find_subapp(app->app, "account-edit",
      0, NULL, NULL);
  if (account_app == NULL)
    account_app = etpan_account_editor_app_new(app->app);
  if (account_app == NULL)
    return ERROR_MEMORY;
  
  etpan_account_editor_app_set(account_app, account,
      account_editor_upcall, app);
  
  etpan_subapp_set_parent(account_app, app);
  etpan_app_switch_subapp(account_app, 0);
  
  return NO_ERROR;
}


static int set_default_account(struct etpan_subapp * app)
{
  struct app_state * state;
  struct etpan_account_info * account;
  
  state = app->data;
  
  account = etpan_account_common_get_account(&state->common_state);
  if (account == NULL)
    return ERROR_INVAL;
  
  state->common_state.config->account_config->default_account = account;
  
  return NO_ERROR;
}


#define HELP_TEXT \
"\
Help for account selection dialog\n\
---------------------------------\n\
\n\
This application will let you select an account.\n\
\n\
- arrow keys : scroll\n\
\n\
- a          : add an account\n\
- e          : edit account\n\
- d          : delete account\n\
- s          : select default account\n\
\n\
- [Enter]    : select account\n\
- Ctrl-G     : cancel\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
