/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-app-subapp.c,v 1.18 2004/11/14 04:37:38 hoa Exp $
 */

#include "etpan-app-subapp.h"

#include <ncurses.h>
#include "etpan-app-types.h"
#include "etpan-subapp.h"
#include "etpan-cfg-color.h"
#include "etpan-ncurses.h"
#include <signal.h>
#include "etpan-errors.h"
#include <pthread.h>
#include <locale.h>
#include "etpan-app.h"
#include <stdlib.h>
#include <libetpan/libetpan.h>
#include "etpan-subapp-thread.h"
#include <string.h>

/* ******************************************* */
/* color management */

/* color_available_tab[0] is not used */

static void color_init(struct etpan_app * app)
{
  int i;
  
  for(i = 0 ; i < ETPAN_MAX_COLORS ; i ++)
    app->color_available_tab[i] = 1;
  
  if (has_colors()) {
    start_color();
    app->color_enabled = 1;
  }
  else {
    app->color_enabled = 0;
  }
}

static int color_get_id(struct etpan_app * app)
{
  int i;
  
  if (!app->color_enabled)
    return -1;
  
  for(i = 1 ; i < COLOR_PAIRS ; i ++) {
    if (app->color_available_tab[i]) {
      app->color_available_tab[i] = 0;
      return i;
    }
  }
  
  return -1;
}

static void color_reset(struct etpan_app * app)
{
  int i;

  for(i = 1 ; i < COLOR_PAIRS ; i ++) {
    app->color_available_tab[i] = 1;
  }
  chash_clear(app->color_pairs_hash);
}

struct color_alloc {
  short foreground;
  short background;
};

void etpan_app_set_color(struct etpan_app * app,
    char * name, int * attr, int default_attr)
{
  struct etpan_color_element * color;
  int color_id;
  chashdatum key;
  chashdatum value;
  struct color_alloc color_key;
  int r;

  color = etpan_color_config_get_color(app->config.color_config, name);
  if (color == NULL) {
    * attr = default_attr;
    return;
  }
  
  color_key.foreground = color->foreground;
  color_key.background = color->background;
  key.data = &color_key;
  key.len = sizeof(color_key);
  r = chash_get(app->color_pairs_hash, &key, &value);
  if (r == 0) {
    memcpy(&color_id, value.data, sizeof(color_id));
    * attr = COLOR_PAIR(color_id) | color->attr;
  }
  else {
    color_id = color_get_id(app);
    if (color_id == -1) {
      * attr = default_attr;
      return;
    }
    
    value.data = &color_id;
    value.len = sizeof(color_id);
    r = chash_set(app->color_pairs_hash, &key, &value, NULL);
    if (r < 0) {
      /* ignore errors */
    }
    
    init_pair(color_id, color->foreground, color->background);
    
    * attr = COLOR_PAIR(color_id) | color->attr;
  }
}


/* ******************************************* */
/* SIGWINCH management */
/* could only have one per process */

static struct sigaction resize_handler_last_handler;
static int resize_handler_winch_received = 0;

static void resize_handler_winch_handler(int signal)
{
  resize_handler_winch_received = 1;
}

static void alloc_buffer(struct etpan_app * app)
{
  if (app->width + 1 > app->buf_len) {
    char * buffer;
    char * fill;
    char * output;

    buffer = realloc(app->buffer, app->width + 1);
    if (buffer != NULL)
      app->buffer = buffer;
    output = realloc(app->output, app->width + 1);
    if (output != NULL)
      app->output = output;
    fill = realloc(app->fill, app->width + 1);
    if (fill != NULL) {
      int i;
      
      for(i = 0 ; i < app->width ; i ++)
        fill[i] = ' ';
      fill[app->width] = '\0';
      app->fill = fill;
    }
    
    if ((fill != NULL) && (buffer != NULL) && (fill != NULL))
      app->buf_len = app->width + 1;
    else
      app->buf_len = 0;
  }
}

static void resize_handler_update_window(struct etpan_app * app)
{
  unsigned int app_index;

  if (!resize_handler_winch_received)
    return;
  
  etpan_resize_term(&app->height, &app->width);
  
  for(app_index = 0 ; app_index < carray_count(app->zorder) ;
      app_index ++) {
    struct etpan_subapp * subapp;
    
    subapp = carray_get(app->zorder, app_index);
    etpan_subapp_handle_resize(subapp);
    
    if (subapp->height < 3)
      subapp->height = 3;
    if (subapp->width < 3)
      subapp->width = 3;
    etpan_subapp_set_relative_coord(subapp);
    
    etpan_subapp_handle_resize(subapp);
  }
  
  alloc_buffer(app);
  
  resize_handler_winch_received = 0;
}

static void resize_handler_init(struct etpan_app * app)
{
  struct sigaction act;
  
  etpan_resize_term(&app->height, &app->width);
  
  act.sa_handler = resize_handler_winch_handler;
  sigemptyset(&act.sa_mask);
  act.sa_flags = 0;
  
  sigaction(SIGWINCH, &act, &resize_handler_last_handler);
}

static void resize_handler_done(struct etpan_app * app)
{
  sigaction(SIGWINCH, &resize_handler_last_handler, NULL);
}

/* ******************************************* */
/* display management */
/* could only have one per process */

void etpan_app_display_init(struct etpan_app * app)
{
  initscr();
  raw();
  keypad(stdscr, TRUE);
  noecho();
  color_init(app);
  resize_handler_init(app);

  alloc_buffer(app);
}

void etpan_app_display_done(struct etpan_app * app)
{
  unsigned int app_index;

  for(app_index = 0 ; app_index < carray_count(app->subapp_list) ;
      app_index ++) {
    struct etpan_subapp * subapp;
    
    subapp = carray_get(app->subapp_list, app_index);
    etpan_subapp_display_done(subapp);
  }
  
  resize_handler_done(app);
  endwin();
}

static int get_subapp_index(carray * app_list,
    struct etpan_subapp * subapp);

struct etpan_subapp *
etpan_app_find_subapp(struct etpan_app * app, char * drivername,
    int enabled,
    int (* compare)(struct etpan_subapp *, void *), void * data);

void fill_with_app_child(carray * apps, struct etpan_subapp * subapp)
{
  unsigned int i;
  
  carray_add(apps, subapp, NULL);
#if 0
  for(i = 0 ; i < carray_count(subapp->app->subapp_list) ; i ++) {
#endif
  for(i = 0 ; i < carray_count(subapp->app->switch_order) ; i ++) {
    struct etpan_subapp * child;
    
#if 0
    child = carray_get(subapp->app->subapp_list, i);
#endif
    child = carray_get(subapp->app->switch_order, i);
    
    if (child->enabled && (child->parent == subapp))
      fill_with_app_child(apps, child);
  }
}

static void switch_to_next_app(struct etpan_app * app, int forward)
{
  carray * apps;
  int app_index;
  struct etpan_subapp * root_app;
  
  root_app = app->current_subapp;
  while (root_app->parent != NULL)
    root_app = root_app->parent;
  
#if 0
  apps = carray_new(carray_count(app->subapp_list));
#endif
  apps = carray_new(carray_count(app->switch_order));
  
  fill_with_app_child(apps, root_app);
  
  app_index = get_subapp_index(apps, app->current_subapp);
  if (app_index == -1)
    return;
  
  if (forward)
    app_index ++;
  else
    app_index --;
  
  if ((app_index >= 0) && ((unsigned int) app_index < carray_count(apps))) {
    struct etpan_subapp * subapp;
    
    subapp = carray_get(apps, app_index);
    etpan_app_switch_subapp(subapp, 0);
  }
  
  carray_free(apps);
}

/* ******************************************* */
/* loop management */

enum {
  MODE_NORMAL,
  MODE_MOVE,
  MODE_RESIZE,
};

void etpan_app_run(struct etpan_app * app, struct etpan_subapp * subapp)
{
  int mode;
  
  mode = MODE_NORMAL;
  
  etpan_app_display_init(app);
  
  if (subapp != NULL)
    etpan_app_switch_subapp(subapp, 0);

  while (1) {
    fd_set fds;
    int max_fd;
    unsigned int app_index;
    int r;
    int end;
    int i;
    int min_idle_time;
    
    pthread_mutex_lock(&app->end_lock);
    end = app->end;
    pthread_mutex_unlock(&app->end_lock);
    if (end)
      break;
    
    for(app_index = 0 ; app_index < carray_count(app->subapp_list) ;
        app_index ++) {
      struct etpan_subapp * subapp;
      
      subapp = carray_get(app->subapp_list, app_index);
      if (subapp->enabled)
        etpan_subapp_idle(subapp);
    }
    
    /* display */
    resize_handler_update_window(app);
    
    for(i = 0 ; i < app->width ; i ++)
      app->fill[i] = ' ';
    app->fill[app->width] = '\0';
    
    for(i = 0 ; i < app->height ; i ++)
      mvaddstr(i, 0, app->fill);
    
    for(app_index = 0 ; app_index < carray_count(app->zorder) ;
        app_index ++) {
      struct etpan_subapp * subapp;
      
      subapp = carray_get(app->zorder, app_index);
      if (subapp->enabled) {
        etpan_subapp_display(subapp);
      }
    }
    
    for(i = 0 ; i < app->width ; i ++)
      app->fill[i] = ' ';
    app->fill[app->width] = '\0';
    
    switch (mode) {
    case MODE_RESIZE:
      move(app->height - 1, 0);
      mvprintw(app->height - 1, 0, "%s", app->fill);
      mvprintw(app->height - 1, 0, "resize window");
      break;
    case MODE_MOVE:
      move(app->height - 1, 0);
      mvprintw(app->height - 1, 0, "%s", app->fill);
      mvprintw(app->height - 1, 0, "move window");
      break;
    }
    
    refresh();
    
    FD_ZERO(&fds);
    FD_SET(0, &fds);
    max_fd = 0;
    
    /* file descriptor set */
    for(app_index = 0 ; app_index < carray_count(app->subapp_list) ;
        app_index ++) {
      struct etpan_subapp * subapp;
      
      subapp = carray_get(app->subapp_list, app_index);
      if (subapp->enabled)
        etpan_subapp_set_fd(subapp, &fds, &max_fd);
    }
    
    /* minimum idle time */
    min_idle_time = -1;
    for(app_index = 0 ; app_index < carray_count(app->subapp_list) ;
        app_index ++) {
      struct etpan_subapp * subapp;
      int idle_time;
      
      subapp = carray_get(app->subapp_list, app_index);
      if (subapp->enabled) {
        idle_time = etpan_subapp_get_idle_delay(subapp);
        if (idle_time != -1) {
          if (min_idle_time == -1) {
            min_idle_time = idle_time;
          }
          else {
            if (idle_time < 0) {
              ETPAN_APP_DEBUG((app, "BUG detected - idle delay of %i by %s",
                                  idle_time, subapp->driver->name));
            }
            if (idle_time < min_idle_time)
              min_idle_time = idle_time;
          }
        }
      }
    }
    /* fix bug in idle delay */
    if (min_idle_time != -1) {
      if (min_idle_time <= 0)
        min_idle_time = 1;
    }
    
    /* set cursor */
    if (app->current_subapp != NULL)
      curs_set(app->current_subapp->show_cursor);
    
    if (min_idle_time != -1) {
      struct timeval timeout;
      
      timeout.tv_sec = min_idle_time;
      timeout.tv_usec = 0;
      r = select(max_fd + 1, &fds, NULL, NULL, &timeout);
    }
    else {
      /* if no idle time */
      
      r = select(max_fd + 1, &fds, NULL, NULL, NULL);
    }
    if (r > 0) {
      if (FD_ISSET(0, &fds)) {
        int ch;
        struct etpan_subapp * current_subapp;
        int switched;
        
        ch = getch();
        
        current_subapp = app->current_subapp;

        switched = 0;
        /* handle key */
        if (current_subapp != NULL) {
          if (ch == KEY_CTRL('z')) {
            etpan_app_display_done(app);
            kill(getpid(), SIGSTOP);
            etpan_app_display_init(app);
          }
          
          switch (mode) {
          case MODE_NORMAL:
            switch (ch) {
            case KEY_CTRL('s'):
              mode = MODE_RESIZE;
              ETPAN_APP_DEBUG((app, "resize & move"));
              break;
            }
            
            if (!current_subapp->driver->always_on_top) {
              switch (ch) {
              case KEY_CTRL('x'):
                switch_to_next_app(app, 1);
                switched = 1;
                break;
                
              case KEY_CTRL('w'):
                switch_to_next_app(app, 0);
                switched = 1;
                break;
              }
            }
            
            /* handle key by subapp */
            if (!switched) {
              for(app_index = 0 ; app_index < carray_count(app->subapp_list) ;
                  app_index ++) {
                struct etpan_subapp * subapp;
              
                subapp = carray_get(app->subapp_list, app_index);
                if (subapp->enabled) {
                  if (subapp->driver->always_handle_key &&
                      (current_subapp != subapp)) {
                    etpan_subapp_handle_key(subapp, ch);
                  }
                }
              }

              if (current_subapp != NULL) {
                etpan_subapp_handle_key(current_subapp, ch);
              }
            }
            
            break;
            
          case MODE_MOVE:
            switch (ch) {
            case KEY_UP:
            case 'k':
              if (current_subapp->top > 0) {
                current_subapp->top --;
                etpan_subapp_set_relative_coord(current_subapp);
                etpan_subapp_handle_resize(current_subapp);
              }
              break;
            case KEY_DOWN:
            case 'j':
              if (current_subapp->top + current_subapp->height < app->height) {
                current_subapp->top ++;
                etpan_subapp_set_relative_coord(current_subapp);
                etpan_subapp_handle_resize(current_subapp);
              }
              break;
            case KEY_LEFT:
            case 'h':
              if (current_subapp->left > 0) {
                current_subapp->left --;
                etpan_subapp_set_relative_coord(current_subapp);
                etpan_subapp_handle_resize(current_subapp);
              }
              break;
            case KEY_RIGHT:
            case 'l':
              if (current_subapp->left + current_subapp->width < app->width) {
                current_subapp->left ++;
                etpan_subapp_set_relative_coord(current_subapp);
                etpan_subapp_handle_resize(current_subapp);
              }
              break;
            case KEY_CTRL('s'):
              mode = MODE_RESIZE;
              break;
            case '\n':
              mode = MODE_NORMAL;
              break;
            }
            break;

          case MODE_RESIZE:
            switch (ch) {
            case 'k':
            case KEY_UP:
              if (current_subapp->height > 3) {
                current_subapp->height --;
                etpan_subapp_set_relative_coord(current_subapp);
                etpan_subapp_handle_resize(current_subapp);
              }
              break;
            case KEY_DOWN:
            case 'j':
              if (current_subapp->top + current_subapp->height < app->height) {
                current_subapp->height ++;
                etpan_subapp_set_relative_coord(current_subapp);
                etpan_subapp_handle_resize(current_subapp);
              }
              break;
            case KEY_LEFT:
            case 'h':
              if (current_subapp->width > 3) {
                current_subapp->width --;
                etpan_subapp_set_relative_coord(current_subapp);
                etpan_subapp_handle_resize(current_subapp);
              }
              break;
            case KEY_RIGHT:
            case 'l':
              if (current_subapp->left + current_subapp->width < app->width) {
                current_subapp->width ++;
                etpan_subapp_set_relative_coord(current_subapp);
                etpan_subapp_handle_resize(current_subapp);
              }
              break;
            case KEY_CTRL('s'):
              mode = MODE_MOVE;
              break;
            case '\n':
              mode = MODE_NORMAL;
              break;
            }
            etpan_subapp_set_relative_coord(current_subapp);
            break;
          }
        }
      }

      /* handle fd */
      for(app_index = 0 ; app_index < carray_count(app->subapp_list) ;
          app_index ++) {
        struct etpan_subapp * subapp;
        
        subapp = carray_get(app->subapp_list, app_index);
        if (subapp->enabled)
          etpan_subapp_handle_fd(subapp, &fds);
      }
    }
  }
  
  etpan_app_display_done(app);
}

void etpan_app_set_colors(struct etpan_app * app)
{
  unsigned int i;

  etpan_app_set_color(app, "sel-title", &app->sel_title_attr,
      A_REVERSE | A_BOLD);
  etpan_app_set_color(app, "title", &app->title_attr,
      A_REVERSE | A_BOLD);
  
  for(i = 0 ; i < carray_count(app->subapp_list) ; i ++) {
    struct etpan_subapp * cur_subapp;
    
    cur_subapp = carray_get(app->subapp_list, i);

    /*    
    if (cur_subapp->enabled || cur_subapp->driver->always_display)
    */
    if (cur_subapp->enabled)
      etpan_subapp_set_color(cur_subapp);
  }
}


void etpan_app_leave_subapp(struct etpan_subapp * subapp,
    struct etpan_subapp * new_subapp)
{
  int app_index;
  
  while (1) {
    struct etpan_subapp * child;
    
    child = etpan_app_find_child_subapp(subapp, 1);
    if (child != NULL)
      etpan_app_leave_subapp(child, new_subapp);
    else
      break;
  }
  
  etpan_subapp_thread_cancel_all(subapp);
  
  etpan_subapp_leave(subapp, new_subapp);
  
#if 0
  if (!subapp->driver->always_display) {
#endif
    /* remove old app from zorder */
    app_index = get_subapp_index(subapp->app->zorder, subapp);
    if (app_index != -1)
      carray_delete_slow(subapp->app->zorder, app_index);
    
    /* remove old app from the switch order */
    app_index = get_subapp_index(subapp->app->switch_order, subapp);
    if (app_index != -1)
      carray_delete_slow(subapp->app->switch_order, app_index);
#if 0
  }
#endif
}


static int carray_insert(carray * array, unsigned int index, void * data)
{
  if (index >= carray_count(array)) {
    return carray_add(array, data, NULL);
  }
  else {
    unsigned int i;
    int r;
    
    r = carray_add(array, NULL, NULL);
    if (r < 0)
      return r;
    
    for(i = carray_count(array) - 1 ; i >= index + 1 ; i --) {
      void * other_data;
      
      other_data = carray_get(array, i - 1);
      carray_set(array, i, other_data);
    }
    carray_set(array, index, data);
    
    return NO_ERROR;
  }
}

#if 0
static int get_subapp_last_child(carray * app_list,
    struct etpan_subapp * parent)
{
  int i;
  int result;
  
  result = -1;
  for(i = 0 ; i < (int) carray_count(app_list) ; i ++) {
    struct etpan_subapp * child;
    
    child = carray_get(app_list, i);
    if (child->parent == parent)
      result = i;
  }
  
  return result;
}
#endif

/* switch app */

void etpan_app_switch_subapp(struct etpan_subapp * subapp, int leave_last)
{
  struct etpan_subapp * old_subapp;
  int app_index;

  color_reset(subapp->app);
  
  old_subapp = subapp->app->current_subapp;
  if (old_subapp != NULL) {
    if (leave_last)
      etpan_app_leave_subapp(old_subapp, subapp);
  }
  
  etpan_app_set_colors(subapp->app);
  
  subapp->app->current_subapp = subapp;
  
  if (!subapp->enabled) {
    int inserted;
    
    /* if the application was not enabled */
    app_index = get_subapp_index(subapp->app->switch_order, subapp);
    if (app_index != -1)
      carray_delete_slow(subapp->app->switch_order, app_index);
    
    inserted = 0;
    if (!leave_last && (old_subapp != NULL)) {
      app_index = get_subapp_index(subapp->app->switch_order,
          old_subapp);
      if (app_index != -1) {
        carray_insert(subapp->app->switch_order, app_index + 1, subapp);
        inserted = 1;
      }
    }
    if (!inserted)
      carray_add(subapp->app->switch_order, subapp, NULL);
  }
  
  /* join the subapp */
  etpan_subapp_enter(subapp, old_subapp);
  
  /* add new application to zorder */
  app_index = get_subapp_index(subapp->app->zorder, subapp);
  if (app_index != -1)
    carray_delete_slow(subapp->app->zorder, app_index);
  carray_add(subapp->app->zorder, subapp, NULL);
  
  etpan_subapp_set_color(subapp);
}

int etpan_app_add_subapp(struct etpan_app * app,
    struct etpan_subapp * subapp)
{
  int r;
  
  r = carray_add(app->subapp_list, subapp, NULL);
  if (r < 0)
    return ERROR_MEMORY;

  return NO_ERROR;
}

void etpan_app_remove_subapp(struct etpan_app * app,
    struct etpan_subapp * subapp)
{
  int app_index;
  
  app_index = get_subapp_index(app->subapp_list, subapp);
  if (app_index != -1)
    carray_delete_slow(app->subapp_list, app_index);
  
#if 0
  for(i = 0 ; i < carray_count(app->subapp_list) ; i ++) {
    if (carray_get(app->subapp_list, i) == subapp) {
      carray_delete(app->subapp_list, i);
      break;
    }
  }
#endif
}

void etpan_app_stop(struct etpan_app * app)
{
  pthread_mutex_lock(&app->end_lock);
  app->end = 1;
  pthread_mutex_unlock(&app->end_lock);
}

struct etpan_subapp *
etpan_app_find_subapp(struct etpan_app * app, char * drivername,
    int enabled,
    int (* compare)(struct etpan_subapp *, void *), void * data)
{
  struct etpan_subapp * subapp;
  unsigned int i;
  
  /* find a matching interface */
  for(i = 0 ; i < carray_count(app->subapp_list) ; i ++) {
    int match;
    
    subapp = carray_get(app->subapp_list, i);
    
    match = 1;
    
    if (subapp->enabled != enabled)
      match = 0;
    
    if (match) {
      if (drivername != NULL)
        if (strcasecmp(subapp->driver->name, drivername) != 0)
          match = 0;
    }
    
    if (match) {
      if (compare != NULL) {
        if (!compare(subapp, data))
          match = 0;
      }
    }
    
    if (match)
      return subapp;
  }
  
  return NULL;
}

struct etpan_subapp *
etpan_app_find_child_subapp(struct etpan_subapp * app, int enabled)
{
  struct etpan_subapp * subapp;
  unsigned int i;
  
  for(i = 0 ; i < carray_count(app->app->subapp_list) ; i ++) {
    subapp = carray_get(app->app->subapp_list, i);
    if (etpan_subapp_get_parent(subapp) == app) {
      if (subapp->enabled == enabled) {
        return subapp;
      }
    }
  }
  
  return NULL;
}

#define CONSOLE_HEIGHT 1

int etpan_app_subapp_display_init(struct etpan_subapp * app)
{
  app->left = 0;
  app->top = 0;
  app->width = app->app->width - app->left;
  
  if (app->app->console != NULL)
    app->height = app->app->height - app->top - CONSOLE_HEIGHT;
  else
    app->height = app->app->height - app->top;
  
  etpan_subapp_set_relative_coord(app);
  
  return NO_ERROR;
}

static int get_subapp_index(carray * app_list,
    struct etpan_subapp * subapp)
{
  unsigned int app_index;
  
  for(app_index = 0 ; app_index < carray_count(app_list) ;
      app_index ++) {
    struct etpan_subapp * cur_subapp;
    
    cur_subapp = carray_get(app_list, app_index);
    if (cur_subapp == subapp)
      return app_index;
  }
  
  return -1;
}



void etpan_app_quit_subapp(struct etpan_subapp * app)
{
  struct etpan_subapp * parent;
  
  parent = etpan_subapp_get_parent(app);
  if (parent != NULL) {
    etpan_app_switch_subapp(app, 0);
    switch_to_next_app(app->app, 0);
    etpan_app_leave_subapp(app, 0);
  }
  else {
    etpan_app_leave_subapp(app, 0);
    etpan_app_stop(app->app);
  }
}

