/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-app.c,v 1.19 2004/08/22 16:27:37 hoa Exp $
 */

#include "etpan-app.h"

#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <stdarg.h>

#include "etpan-config.h"
#include "etpan-tools.h"
#include "etpan-errors.h"
#include "etpan-thread-manager.h"
#include "etpan-subapp.h"
#include "etpan-console-app.h"
#include "etpan-app-subapp.h"
#include <libetpan/libetpan.h>
#include <sys/types.h>
#include <unistd.h>

struct etpan_app * etpan_app_new(char * config_path)
{
  struct etpan_app * app;
  int r;
  unsigned int i;
  char tmp_dir[PATH_MAX];
  
  app = malloc(sizeof(struct etpan_app));
  if (app == NULL)
    goto err;
  
  strncpy(app->config_path, config_path, sizeof(app->config_path));
  app->config_path[sizeof(app->config_path) - 1] = '\0';
  app->pid = getpid();
  
  app->end = 0;
  r = pthread_mutex_init(&app->end_lock, NULL);
  if (r != 0)
    goto free;
  
  app->console = NULL;
  r = pthread_mutex_init(&app->log_lock, NULL);
  if (r != 0)
    goto free_end_lock;
  
  app->current_subapp = NULL;
  app->subapp_list = carray_new(16);
  if (app->subapp_list == NULL) {
    goto free_log_lock;
  }
  app->zorder = carray_new(16);
  if (app->zorder == NULL) {
    goto free_subapp_list;
  }
  app->switch_order = carray_new(16);
  if (app->switch_order == NULL) {
    goto free_zorder;
  }
  app->width = 0;
  app->height = 0;
  
  for(i = 0 ; i < ETPAN_MAX_COLORS ; i ++)
    app->color_available_tab[i] = 1;
  
  app->color_pairs_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (app->color_pairs_hash == NULL)
    goto free_switch_order;
  
  app->color_enabled = 0;
  
#if 0
  app->security = etpan_security_new();
  if (app->security == NULL)
    goto free_thread_manager;
  
  r = etpan_pgp_init(app);
  if (r != NO_ERROR)
    goto free_security;

  r = etpan_smime_init(app);
  if (r != NO_ERROR)
    goto done_pgp;
#endif
  
  snprintf(tmp_dir, sizeof(tmp_dir), "%s/%s", etpan_get_home_dir(),
      ETPAN_APP_DATA_PATH);
  r = etpan_mkdir(tmp_dir);
  if (r < 0) {
    fprintf(stderr, "could not create %s\n", tmp_dir);
    goto free_color_hash;
  }
  
  snprintf(tmp_dir, sizeof(tmp_dir), "%s/%s", etpan_get_home_dir(),
      ETPAN_TMP_PATH);
  r = etpan_mkdir(tmp_dir);
  if (r < 0) {
    fprintf(stderr, "could not create %s\n", tmp_dir);
    goto free_color_hash;
  }
  
  app->privacy = mailprivacy_new(tmp_dir, 1);
  if (app->privacy == NULL)
    goto free_color_hash;

  r = mailprivacy_gnupg_init(app->privacy);
  if (r != MAIL_NO_ERROR)
    goto free_privacy;
  
  r = mailprivacy_smime_init(app->privacy);
  if (r != MAIL_NO_ERROR)
    goto done_gnupg;
  
  app->thread_manager = etpan_thread_manager_new(app);
  if (app->thread_manager == NULL)
    goto done_smime;

  r = etpan_thread_manager_start(app->thread_manager);
  if (r != NO_ERROR)
    goto free_thread_manager;

#if 0  
  engine_app = app;
#endif
  
  memset(&app->config, 0, sizeof(app->config));
  
  r = etpan_app_config_init(config_path, app, &app->config);
  if (r != NO_ERROR)
    goto free_thread_manager;
  
  app->buffer = NULL;
  app->fill = NULL;
  app->output = NULL;
  app->buf_len = 0;
  
  return app;

 free_thread_manager:
  etpan_thread_manager_stop(app->thread_manager);
  etpan_thread_manager_free(app->thread_manager);
 done_smime:
  mailprivacy_smime_done(app->privacy);
 done_gnupg:
  mailprivacy_gnupg_done(app->privacy);
 free_privacy:
  mailprivacy_free(app->privacy);
#if 0
 free_security:
  etpan_security_free(app->security);
#endif
 free_color_hash:
  chash_free(app->color_pairs_hash);
 free_switch_order:
  carray_free(app->switch_order);
 free_zorder:
  carray_free(app->zorder);
 free_subapp_list:
  carray_free(app->subapp_list);
 free_log_lock:
  pthread_mutex_destroy(&app->log_lock);
 free_end_lock:
  pthread_mutex_destroy(&app->end_lock);
 free:
  free(app);
 err:
  return NULL;
}

void etpan_app_free(struct etpan_app * app)
{
  int r;

  ETPAN_APP_DEBUG((app, "destroying %i applications",
                      carray_count(app->subapp_list)));
  
  /* do this because application will not be left in order */
  while (1) {
    struct etpan_subapp * subapp;
    
    subapp = etpan_app_find_subapp(app, NULL, 1, NULL, NULL);
    if (subapp == NULL)
      break;
    
    etpan_app_leave_subapp(subapp, NULL);
  }
  
  ETPAN_APP_DEBUG((app, "waiting for thread to finish"));
  r = etpan_thread_manager_stop(app->thread_manager);
  if (r != NO_ERROR) {
    ETPAN_APP_DEBUG((app, "could not stop thread manager"));
  }
  else {
    ETPAN_APP_DEBUG((app, "thread finished"));
  }
  
  mailprivacy_debug(app->privacy, stderr);
  libetpan_engine_debug(app->thread_manager->engine, stderr);
  
  while (carray_count(app->subapp_list) > 0) {
    struct etpan_subapp * subapp;
    
    subapp = carray_get(app->subapp_list, 0);
    
    while (subapp->parent != NULL)
      subapp = subapp->parent;
    
    etpan_subapp_free(subapp);
  }
  
  if (app->output != NULL)
    free(app->output);
  if (app->buffer != NULL)
    free(app->buffer);
  if (app->fill != NULL)
    free(app->fill);

#if 0
  ETPAN_APP_DEBUG((app, "mime ref %i",
                      chash_count(app->security->mime_ref)));
  ETPAN_APP_DEBUG((app, "mmapstr %i",
                      chash_count(app->security->mmapstr)));
  ETPAN_APP_DEBUG((app, "msg ref %i",
                      chash_count(app->security->msg_ref)));
#endif

#if 0  
  etpan_app_config_done(&app->config);
#endif
  
  etpan_thread_manager_free(app->thread_manager);
  mailprivacy_smime_done(app->privacy);
  mailprivacy_gnupg_done(app->privacy);
  mailprivacy_free(app->privacy);
#if 0
  etpan_smime_done(app);
  etpan_pgp_done(app);
  etpan_security_free(app->security);
#endif
  chash_free(app->color_pairs_hash);
  carray_free(app->switch_order);
  carray_free(app->zorder);
  carray_free(app->subapp_list);
  pthread_mutex_destroy(&app->log_lock);
  pthread_mutex_destroy(&app->end_lock);
  free(app);
}

#define MAX_LOG_OUTPUT 256
#define MAX_CTIME_OUTPUT 30


void etpan_app_debug_file(struct etpan_app * app, char * format, ...)
{
  char path[PATH_MAX];
  char output[PATH_MAX];
  FILE * f;
  char timestamped[MAX_LOG_OUTPUT];
  char timestamp[MAX_CTIME_OUTPUT];
  struct tm time_info;
  time_t now;
  va_list ap;

  va_start(ap, format);
  vsnprintf(output, sizeof(output), format, ap);
  
  now = time(NULL);
  localtime_r(&now, &time_info);
  strftime(timestamp, sizeof(timestamp), "%H:%M:%S", &time_info);
    
  snprintf(timestamped, sizeof(timestamped), "%s %s", timestamp, output);
  
  snprintf(path, PATH_MAX, "%s/%s/console-%i.log",
      etpan_get_home_dir(), ETPAN_LOG_PATH, app->pid);
  pthread_mutex_lock(&app->log_lock);
  f = fopen(path, "a");
  fprintf(f, "%s\n", timestamped);
  fclose(f);
  pthread_mutex_unlock(&app->log_lock);
}

void etpan_app_log(struct etpan_app * app, char * format, ...)
{
  char output[MAX_LOG_OUTPUT];
  va_list ap;

  va_start(ap, format);
  vsnprintf(output, sizeof(output), format, ap);
  
  pthread_mutex_lock(&app->log_lock);
#ifdef DEBUG_ETPAN_APP
  {
    char path[PATH_MAX];
    FILE * f;
    char timestamped[MAX_LOG_OUTPUT];
    char timestamp[MAX_CTIME_OUTPUT];
    struct tm time_info;
    time_t now;
    
    now = time(NULL);
    localtime_r(&now, &time_info);
    strftime(timestamp, sizeof(timestamp), "%H:%M:%S", &time_info);
    
    snprintf(timestamped, sizeof(timestamped), "%s %s", timestamp, output);
    
    snprintf(path, PATH_MAX, "%s/%s/console-%i.log",
        etpan_get_home_dir(), ETPAN_LOG_PATH, app->pid);
    f = fopen(path, "a");
    if (f != NULL) {
      fprintf(f, "%s\n", timestamped);
      fclose(f);
    }
  }
#endif
  
  if (app->console != NULL)
    etpan_console_log(app->console, output);
  else
    fprintf(stderr, "%s\n", output);
  pthread_mutex_unlock(&app->log_lock);
}

void etpan_app_set_console(struct etpan_app * app,
    struct etpan_subapp * console)
{
  pthread_mutex_lock(&app->log_lock);
  app->console = console;
  pthread_mutex_unlock(&app->log_lock);
}
