/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-cfg-account.c,v 1.9 2003/12/15 16:08:53 hoa Exp $
 */

#include "etpan-cfg-account.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <libetpan/libetpan.h>
#include <libetpan/charconv.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "etpan-cfg-common.h"
#include "etpan-errors.h"
#include "etpan-db-read.h"

struct etpan_account_info * etpan_account_info_new(char * id,
    char * name, char * addr, char * signature,
    char * privacy_driver, char * privacy_encryption)
{
  struct etpan_account_info * account;
  
  account = malloc(sizeof(struct etpan_account_info));
  if (account == NULL) {
    goto err;
  }

  account->id = id;
  account->name = name;
  account->addr = addr;
  account->signature = signature;
  account->privacy_driver = privacy_driver;
  account->privacy_encryption = privacy_encryption;
  
  return account;

 err:
  return NULL;
}

void etpan_account_info_free(struct etpan_account_info * account)
{
  if (account->privacy_encryption != NULL)
    free(account->privacy_encryption);
  if (account->privacy_driver != NULL)
    free(account->privacy_driver);
  if (account->signature != NULL)
    free(account->signature);
  if (account->addr != NULL)
    free(account->addr);
  if (account->name != NULL)
    free(account->name);
  if (account->id != NULL)
    free(account->id);
  free(account);
}

int etpan_account_config_read(char * filename,
    struct etpan_account_config ** result)
{
  int r;
  int res;
  clist * list;
  struct etpan_db * db;
  struct etpan_account_config * account_config;
  unsigned int i;
  struct etpan_account_info * default_account;
  char * default_account_name;
  char * name;
  char * display;
  char * addr;
  char * signature;
  char * privacy_driver;
  char * privacy_encryption;
  
  r = etpan_read_config(filename, &db);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  list = clist_new();
  if (list == NULL) {
    res = ERROR_MEMORY;
    goto free_db;
  }
  
  default_account = NULL;
  default_account_name = NULL;
  
  for(i = 0 ; i < carray_count(db->data) ; i ++) {
    chash * entry;
    char * id;
    
    entry = carray_get(db->data, i);
    if (entry == NULL)
      continue;
    
    id = etpan_db_entry_get_value(entry, "id");
    if (strcasecmp(id, "default") != 0)
      continue;

    default_account_name = etpan_db_entry_get_value(entry, "account");
    
    break;
  }
  
  for(i = 0 ; i < db->data->len ; i ++) {
    chash * entry;
    struct etpan_account_info * account;

    entry = carray_get(db->data, i);
    if (entry == NULL)
      continue;

    /* account info */

    name = etpan_db_entry_get_value(entry, "name");
    if (name == NULL)
      continue;

    display = etpan_db_entry_get_value(entry, "display");

    addr = etpan_db_entry_get_value(entry, "addr");
    if (addr == NULL)
      continue;

    signature = etpan_db_entry_get_value(entry, "signature");
    privacy_driver =
      etpan_db_entry_get_value(entry, "privacy-driver");
    privacy_encryption = 
      etpan_db_entry_get_value(entry, "privacy-encryption");
    
    name = strdup(name);
    if (name == NULL) {
      res = ERROR_MEMORY;
      goto free_list;
    }
    
    if (display != NULL) {
      display = strdup(display);
      if (display == NULL) {
        res = ERROR_MEMORY;
        goto free_name;
      }
    }
    
    if (addr != NULL) {
      addr = strdup(addr);
      if (addr == NULL) {
        res = ERROR_MEMORY;
        goto free_display;
      }
    }
    
    if (signature != NULL) {
      signature = strdup(signature);
      if (signature == NULL) {
        res = ERROR_MEMORY;
        goto free_addr;
      }
    }
    
    if (privacy_driver != NULL) {
      privacy_driver = strdup(privacy_driver);
      if (privacy_driver == NULL) {
        res = ERROR_MEMORY;
        goto free_signature;
      }
    }

    if (privacy_encryption != NULL) {
      privacy_encryption = strdup(privacy_encryption);
      if (privacy_encryption == NULL) {
        res = ERROR_MEMORY;
        goto free_privacy_driver;
      }
    }
    
    account = etpan_account_info_new(name, display, addr, signature,
        privacy_driver, privacy_encryption);
    if (account == NULL) {
      res = ERROR_MEMORY;
      goto free_privacy_encryption;
    }
    
    r = clist_append(list, account);
    if (r < 0) {
      etpan_account_info_free(account);
      res = ERROR_MEMORY;
      goto free_list;
    }
    
    if (default_account_name != NULL) {
      if (strcmp(name, default_account_name) == 0)
        default_account = account;
    }
  }
  
  account_config = etpan_account_config_new(list, default_account);
  if (account_config == NULL) {
    free(default_account);
    res = ERROR_MEMORY;
    goto free_list;
  }

  etpan_db_free(db);
  
  * result = account_config;
  
  return NO_ERROR;
  
 free_privacy_encryption:
  free(privacy_encryption);
 free_privacy_driver:
  free(privacy_driver);
 free_signature:
  free(signature);
 free_addr:
  free(addr);
 free_display:
  free(display);
 free_name:
  free(name);
 free_list:
  clist_foreach(list, (clist_func) etpan_account_info_free, NULL);
  clist_free(list);
 free_db:
  etpan_db_free(db);
 err:
  return res;
}


struct etpan_account_info *
etpan_account_get(struct etpan_account_config * config,
    char * name)
{
  clistiter * cur;

  for(cur = clist_begin(config->list) ; cur != NULL ; cur = cur->next) {
    struct etpan_account_info * account;

    account = cur->data;

    if (strcasecmp(account->id, name) == 0)
      return account;
  }

  return NULL;
}

struct etpan_account_config *
etpan_account_config_new(clist * list,
    struct etpan_account_info * default_account)
{
  struct etpan_account_config * account_config;

  account_config = malloc(sizeof(struct etpan_account_config));
  if (account_config == NULL)
    goto err;

  account_config->list = list;
  account_config->default_account = default_account;

  return account_config;

 err:
  return NULL;
}

void etpan_account_config_free(struct etpan_account_config * config)
{
  clist_foreach(config->list, (clist_func) etpan_account_info_free, NULL);
  clist_free(config->list);
  free(config);
}


static int account_write(FILE * f, int id,
    struct etpan_account_info * account)
{
  fprintf(f, "id = %i\n", id);
  
  fprintf(f, "name = %s\n", account->id);
  if (account->name != NULL)
    fprintf(f, "display = %s\n", account->name);
  if (account->addr != NULL)
    fprintf(f, "addr = %s\n", account->addr);
  if (account->signature != NULL)
    fprintf(f, "signature = %s\n", account->signature);
  if (account->privacy_driver != NULL)
    fprintf(f, "privacy-driver = %s\n", account->privacy_driver);
  if (account->privacy_encryption != NULL)
    fprintf(f, "privacy-encryption = %s\n", account->privacy_encryption);
  
  fprintf(f, "\n");
  
  return NO_ERROR;
}

int etpan_cfg_account_write(char * filename,
    struct etpan_account_config * config)
{
  clistiter * cur;
  int res;
  FILE * f;
  int id;
  int r;
  mode_t old_mask;

  old_mask = umask(0077);
  f = fopen(filename, "w");
  umask(old_mask);
  if (f == NULL) {
    res = ERROR_FILE;
    goto err;
  }
  
  if (config->default_account != NULL) {
    fprintf(f, "id = default\n");
    fprintf(f, "account = %s\n", config->default_account->id);
    fprintf(f, "\n");
  }
  
  id = 0;
  for(cur = clist_begin(config->list) ; cur != NULL ; cur = clist_next(cur)) {
    struct etpan_account_info * account;
    
    account = clist_content(cur);
    r = account_write(f, id, account);
    if (r != NO_ERROR) {
      res = r;
      goto close;
    }
    id ++;
  }
  
  fclose(f);
  
  return NO_ERROR;
  
 close:
  fclose(f);
  unlink(filename);
 err:
  return res;
}
