/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-cfg-color.c,v 1.6 2004/08/22 16:27:37 hoa Exp $
 */

#include "etpan-cfg-color.h"

#include "etpan-errors.h"
#include "etpan-db-read.h"

#include <stdlib.h>
#include <ncurses.h>
#include <string.h>

static int get_color_value(chash * entry, char * name)
{
  char * value_str;
  int value;
  
  value_str = etpan_db_entry_get_value(entry, name);
  if (value_str == NULL)
    value = -1;
  else
    value = strtol(value_str, NULL, 10);
  
  return value;
}

static int get_color_attr(chash * entry, char * name,
    int all_attr, int attr)
{
  char * value_str;
  int value;
  
  value_str = etpan_db_entry_get_value(entry, name);
  if (value_str != NULL) {
    value = strtol(value_str, NULL, 10);
    if (value != 0) {
      all_attr |= attr;
    }
  }
  
  return all_attr;
}

static void color_entry_read(chash * entry,
    struct etpan_color_element * color)
{
  int fg;
  int bg;
  int attr;
  
  fg = get_color_value(entry, "foreground");
  bg = get_color_value(entry, "background");
  
  attr = 0;
  attr = get_color_attr(entry, "blink", attr, A_BLINK);
  attr = get_color_attr(entry, "bold", attr, A_BOLD);
  attr = get_color_attr(entry, "dim", attr, A_DIM);
  attr = get_color_attr(entry, "reverse", attr, A_REVERSE);
  attr = get_color_attr(entry, "standout", attr, A_STANDOUT);
  attr = get_color_attr(entry, "underline", attr, A_UNDERLINE);
  
  color->foreground = fg;
  color->background = bg;
  color->attr = attr;
}

struct etpan_color_config * etpan_color_config_new(chash * color_hash)
{
  struct etpan_color_config * config;
  
  config = malloc(sizeof(* config));
  if (config == NULL)
    return NULL;
  
  config->color_hash = color_hash;
  
  return config;
}

void etpan_color_config_free(struct etpan_color_config * config)
{
  chash_free(config->color_hash);
  free(config);
}

int etpan_color_config_read(char * filename,
    struct etpan_color_config ** result)
{
  struct etpan_db * db;
  int r;
  int res;
  unsigned int i;
  chash * color_hash;
  struct etpan_color_config * config;
  
  r = etpan_read_config(filename, &db);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  color_hash = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL);
  if (color_hash == NULL) {
    res = ERROR_MEMORY;
    goto free_db;
  }

  for(i = 0 ; i < db->data->len ; i ++) {
    chash * entry;
    struct etpan_color_element color;
    char * name;
    chashdatum key;
    chashdatum value;
    
    entry = carray_get(db->data, i);
    if (entry == NULL)
      continue;
    
    name = etpan_db_entry_get_value(entry, "name");
    if (name == NULL)
      continue;
    
    color_entry_read(entry, &color);
    
    key.data = name;
    key.len = strlen(name);
    value.data = &color;
    value.len = sizeof(color);

    r = chash_set(color_hash, &key, &value, NULL);
    if (r < 0) {
      res = ERROR_MEMORY;
      goto free_color_hash;
    }
  }

  config = etpan_color_config_new(color_hash);
  if (config == NULL) {
    res = ERROR_MEMORY;
    goto free_color_hash;
  }
  
  etpan_db_free(db);
  
  * result = config;
  
  return NO_ERROR;

 free_color_hash:
  chash_free(color_hash);
 free_db:
  etpan_db_free(db);
 err:
  return res;
}

struct etpan_color_element *
etpan_color_config_get_color(struct etpan_color_config * color_config,
    char * id)
{
  chashdatum key;
  chashdatum value;
  int r;

  key.data = id;
  key.len = strlen(id);

  r = chash_get(color_config->color_hash, &key, &value);
  if (r < 0)
    return NULL;
  
  return value.data;
}
