/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-cfg-global.c,v 1.18 2004/11/14 19:17:13 hoa Exp $
 */

#include "etpan-cfg-global.h"

#include <stdlib.h>
#include <string.h>
#include <paths.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <libetpan/libetpan.h>

#include "etpan-cfg-common.h"
#include "etpan-errors.h"
#include "etpan-db-read.h"

struct etpan_global_config *
etpan_global_config_new(char * display_charset, char * editor_charset,
    char * message_charset, char * editor, int reply_quote_limit,
    int network_timeout, int poll_delay, char * sendmail_path,
    int thread_type, int across_folder)
{
  struct etpan_global_config * global_config;

  global_config = malloc(sizeof(struct etpan_global_config));
  if (global_config == NULL)
    return NULL;

  global_config->display_charset = display_charset;
  global_config->editor_charset = editor_charset;
  global_config->message_charset = message_charset;
  global_config->editor = editor;
  global_config->reply_quote_limit = reply_quote_limit;
  global_config->poll_delay = poll_delay;
  global_config->network_timeout = network_timeout;
  global_config->sendmail_path = sendmail_path;
  global_config->thread_type = thread_type;
  global_config->across_folder = across_folder;

  return global_config;
}

void etpan_global_config_free(struct etpan_global_config * global_config)
{
  if (global_config->sendmail_path != NULL)
    free(global_config->sendmail_path);
  if (global_config->display_charset != NULL)
    free(global_config->display_charset);
  if (global_config->editor_charset != NULL)
    free(global_config->editor_charset);
  if (global_config->message_charset != NULL)
    free(global_config->message_charset);
  if (global_config->editor != NULL)
    free(global_config->editor);
  free(global_config);
}

int etpan_global_config_read(char * filename,
    struct etpan_global_config ** result)
{
  struct etpan_db * db;
  int r;
  int res;
  unsigned int i;
  struct etpan_global_config * config;
  char * display_charset;
  char * editor_charset;
  char * message_charset;
  char * editor;
  int network_delay;
  int reply_quote_limit;
  int poll_delay;
  char * sendmail_path;
  int thread_type;
  int across_folder;
  
  r = etpan_read_config(filename, &db);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  display_charset = ETPAN_DEFAULT_DISPLAY_CHARSET;
  editor_charset = ETPAN_DEFAULT_EDITOR_CHARSET;
  message_charset = ETPAN_DEFAULT_MESSAGE_CHARSET;
  editor = getenv("EDITOR");
  if (editor == NULL)
    editor = ETPAN_DEFAULT_EDITOR;
  network_delay = ETPAN_DEFAULT_NETWORK_TIMEOUT;
  reply_quote_limit = ETPAN_DEFAULT_REPLY_QUOTE_LIMIT;
  poll_delay = ETPAN_DEFAULT_POLL_DELAY;
  sendmail_path = ETPAN_DEFAULT_SENDMAIL_PATH;
  thread_type = ETPAN_THREAD_TYPE_BY_REFERENCES;
  across_folder = 0;
  
  for(i = 0 ; i < carray_count(db->data) ; i ++) {
    chash * entry;
    char * new_value;

    entry = carray_get(db->data, i);
    if (entry == NULL)
      continue;
    
    new_value = etpan_db_entry_get_value(entry, "display-charset");
    if (new_value != NULL)
      display_charset = new_value;
    
    new_value = etpan_db_entry_get_value(entry, "editor-charset");
    if (new_value != NULL)
      editor_charset = new_value;
    
    new_value = etpan_db_entry_get_value(entry, "message-charset");
    if (new_value != NULL)
      message_charset = new_value;
    
    new_value = etpan_db_entry_get_value(entry, "editor");
    if (new_value != NULL)
      editor = new_value;
    
    new_value = etpan_db_entry_get_value(entry, "sendmail-path");
    if (new_value != NULL)
      sendmail_path = new_value;

    new_value = etpan_db_entry_get_value(entry, "network-timeout");
    if (new_value != NULL)
      network_delay = strtoul(new_value, NULL, 10);
    
    new_value = etpan_db_entry_get_value(entry, "poll-delay");
    if (new_value != NULL)
      poll_delay = strtoul(new_value, NULL, 10);
    
    new_value = etpan_db_entry_get_value(entry, "reply-quote-limit");
    if (new_value != NULL)
      reply_quote_limit = strtoul(new_value, NULL, 10);

    new_value = etpan_db_entry_get_value(entry, "thread-type");
    if (new_value != NULL)
      thread_type = strtoul(new_value, NULL, 10);

    new_value = etpan_db_entry_get_value(entry, "across-folder");
    if (new_value != NULL)
      across_folder = strtoul(new_value, NULL, 10);
  }
  
  r = etpan_cfg_strdup(&display_charset, display_charset);
  if (r != NO_ERROR) {
    res = r;
    goto free_db;
  }

  r = etpan_cfg_strdup(&editor_charset, editor_charset);
  if (r != NO_ERROR) {
    res = r;
    goto free_display_charset;
  }

  r = etpan_cfg_strdup(&message_charset, message_charset);
  if (r != NO_ERROR) {
    res = r;
    goto free_editor_charset;
  }

  r = etpan_cfg_strdup(&editor, editor);
  if (r != NO_ERROR) {
    res = r;
    goto free_message_charset;
  }

  r = etpan_cfg_strdup(&sendmail_path, sendmail_path);
  if (r != NO_ERROR) {
    res = r;
    goto free_editor;
  }

#ifdef LIBETPAN_MAILSTREAM_NETWORK_DELAY
  if ((network_delay == -1) || (network_delay == 0))
    network_delay = ETPAN_DEFAULT_NETWORK_TIMEOUT;
  
  mailstream_network_delay.tv_sec = network_delay;
  mailstream_network_delay.tv_usec = 0;
#endif
  
  if (poll_delay == -1)
    poll_delay = ETPAN_DEFAULT_POLL_DELAY;
  
  config = etpan_global_config_new(display_charset,
      editor_charset, message_charset, editor, reply_quote_limit,
      network_delay, poll_delay, sendmail_path,
      thread_type, across_folder);
  if (config == NULL) {
    res = ERROR_MEMORY;
    goto free_sendmail_path;
  }

  etpan_db_free(db);
  
  * result = config;
  
  return NO_ERROR;

 free_sendmail_path:
  if (sendmail_path != NULL)
    free(sendmail_path);
 free_editor:
  if (editor != NULL)
    free(editor);
 free_message_charset:
  if (message_charset != NULL)
    free(message_charset);
 free_editor_charset:
  if (editor_charset != NULL)
    free(editor_charset);
 free_display_charset:
  if (display_charset != NULL)
    free(display_charset);
 free_db:
  etpan_db_free(db);
 err:
  return res;
}

int etpan_cfg_global_write(char * filename,
    struct etpan_global_config * global_config)
{
  FILE * f;
  int res;
  mode_t old_mask;

  old_mask = umask(0077);
  f = fopen(filename, "w");
  umask(old_mask);
  if (f == NULL) {
    res = ERROR_FILE;
    goto err;
  }
  
  fprintf(f, "id = default\n");
  fprintf(f, "display-charset = %s\n", global_config->display_charset);
  fprintf(f, "editor-charset = %s\n", global_config->editor_charset);
  fprintf(f, "message-charset = %s\n", global_config->message_charset);
  fprintf(f, "editor = %s\n", global_config->editor);
  fprintf(f, "sendmail-path = %s\n", global_config->sendmail_path);
  fprintf(f, "reply-quote-limit = %i\n", global_config->reply_quote_limit);
  fprintf(f, "poll-delay = %i\n", global_config->poll_delay);
  fprintf(f, "network-timeout = %i\n", global_config->network_timeout);
  fprintf(f, "thread-type = %i\n", global_config->thread_type);
  fprintf(f, "across-folder = %i\n", global_config->across_folder);
  
  fclose(f);
  
  return NO_ERROR;
  
 err:
  return res;
}
