/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-compose-app.c,v 1.9 2004/09/15 17:22:34 hoa Exp $
 */

#include "etpan-compose-app.h"

#include "etpan-app-subapp.h"
#include "etpan-subapp.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ncurses.h>
#include "etpan-imf-helper.h"
#include "etpan-address-input.h"
#include "etpan-search-input.h"
#include "etpan-header-input.h"
#include "etpan-app.h"
#include "etpan-msg-new.h"
#include "etpan-subapp-thread.h"
#include "etpan-mime-tools.h"
#include "etpan-header-common.h"
#include "etpan-mime-edit-app.h"
#include "etpan-help-viewer.h"

static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_compose_app_driver = {
  .name = "compose-message",
  .always_handle_key = 0,
  .always_on_top = 0,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};


struct app_state {
  struct etpan_header_common_state common_state;
  mailmessage * msg;
  struct mailmime * edit_part;
  int edit_part_line;
  struct mailfolder * post_from_folder;
  void (* action)(void *);
  void * action_arg;
  void (* cancel_action)(void *);
};

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_header_common_set_color(app, &state->common_state);
}

static void edit_mime(struct etpan_subapp * app);

static int show_help(struct etpan_subapp * app);

static void ask_quit(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  
  state = app->data;

  etpan_header_common_handle_key(app, &state->common_state, key);
  
  switch (key) {
  case KEY_F(1):
  case '?':
    show_help(app);
    break;
    
  case 'y':
    edit_mime(app);
    break;

  case KEY_CTRL('G'):
    if (state->cancel_action != NULL)
      state->cancel_action(state->action_arg);
    etpan_app_quit_subapp(app);
    break;

  case 'q':
    ask_quit(app);
    break;
  }
}


static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, "etPan! - compose message");
  return etpan_app_subapp_display_init(app);
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_header_common_display(app, &state->common_state, w,
      "y: ok  ^G: cancel");
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;

  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  subapp->data = state;
  
  state->msg = NULL;
  state->edit_part = NULL;
  state->edit_part_line = 0;
  state->post_from_folder = NULL;

  state->action = NULL;
  state->action_arg = NULL;
  state->cancel_action = NULL;
  
  r = etpan_common_header_init(subapp, &state->common_state);
  if (r != NO_ERROR)
    goto free;
  
  return NO_ERROR;
  
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * subapp)
{
  struct app_state * state;
  
  state = subapp->data;
  
  etpan_header_common_done(subapp, &state->common_state);
  free(state);
}


void etpan_compose_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_header_common_flush(app, &state->common_state);
  
  if (state->msg != NULL) {
    etpan_message_mime_clear(state->msg->msg_mime);
#if 0
    r = etpan_subapp_thread_op_add(app, THREAD_ID_HEADEREDITOR_FREE_MSG,
        ETPAN_THREAD_MESSAGE_FREE,
        NULL, NULL,
        state->msg, NULL,
        NULL,
        NULL, NULL, NULL);
#endif
    etpan_queue_unref_msg(app, state->msg);
#if 0
    if (r != NO_ERROR) {
      ETPAN_APP_LOG((app->app, "compose message - not enough memory"));
    }
#endif
  }
  
  state->msg = NULL;
  state->edit_part = NULL;
  state->edit_part_line = 0;
  state->post_from_folder = NULL;

  state->action = NULL;
  state->action_arg = NULL;
  state->cancel_action = NULL;
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  etpan_compose_app_flush(app);
}

void etpan_compose_app_set(struct etpan_subapp * app,
    mailmessage * msg,
    struct mailmime * edit_part,
    int edit_part_line,
    struct mailfolder * post_from_folder,
    void (* action)(void *), void * action_arg,
    void (* cancel_action)(void *))
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_compose_app_flush(app);
  
  etpan_header_common_set_fields(app, &state->common_state,
      msg->msg_mime->mm_data.mm_message.mm_fields);
  
  state->msg = msg;
  state->edit_part = edit_part;
  state->edit_part_line = edit_part_line;
  state->post_from_folder = post_from_folder;
  
  state->action = action;
  state->action_arg = action_arg;
  state->cancel_action = cancel_action;
}

mailmessage *
etpan_compose_app_get_msg(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return state->msg;
}


struct etpan_subapp * etpan_compose_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_compose_app_driver);
}


static int etpan_edit_mime_msg(struct etpan_subapp * app,
    struct mailmessage * msg,
    struct mailfolder * post_from_folder,
    void (* action)(void *), void * action_arg,
    void (* cancel_action)(void *));

static void edit_mime(struct etpan_subapp * app)
{
  int r;
  struct app_state * state;
  struct etpan_subapp * parent;

  state = app->data;
  
  etpan_header_common_fields_adjust(app, &state->common_state);
  
  if (state->edit_part != NULL) {
    r = etpan_edit_file(app->app,
        state->edit_part->mm_data.mm_single->dt_data.dt_filename,
        state->edit_part_line,
        NULL);
    if (r == NO_ERROR)
      etpan_set_text_content_type(app->app, state->edit_part);
  }
  
  parent = etpan_subapp_get_parent(app);
  r = etpan_edit_mime_msg(parent,
      state->msg,
      state->post_from_folder,
      state->action, state->action_arg, state->cancel_action);
  if (r == NO_ERROR) {
    /*
      if this is successful, detach message, it is given to the
      MIME structure editor
    */
    state->msg = NULL;
    state->edit_part = NULL;
    state->post_from_folder = NULL;
    
    state->action = NULL;
    state->action_arg = NULL;
    state->cancel_action = NULL;
    
    etpan_app_leave_subapp(app, NULL);
  }
}


static int etpan_edit_mime_msg(struct etpan_subapp * app,
    struct mailmessage * msg,
    struct mailfolder * post_from_folder,
    void (* action)(void *), void * action_arg,
    void (* cancel_action)(void *))
{
  struct etpan_subapp * mimeedit_app;
  int r;
  
  mimeedit_app = etpan_app_find_subapp(app->app, "mime-edit", 0, NULL, NULL);
  if (mimeedit_app == NULL) {
    mimeedit_app = etpan_mime_edit_app_new(app->app);
    if (mimeedit_app == NULL)
      return ERROR_MEMORY;
  }
  
  etpan_subapp_set_parent(mimeedit_app, app);
  
  r = etpan_mime_edit_app_set_msg(mimeedit_app, msg, post_from_folder,
      action, action_arg, cancel_action);
  if (r != NO_ERROR)
    return r;
  
  etpan_app_switch_subapp(mimeedit_app, 0);
  
  return NO_ERROR;
}


static void ask_quit_upcall(struct etpan_subapp * input_app,
    int valid, void * data)
{
  struct app_state * state;
  char * name;
  char * dup_name;
  struct etpan_subapp * app;
  int r;
  char * result;
  int do_quit;

  app = data;
  state = app->data;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  result = etpan_search_input_get_value(input_app);
  if ((* result != 'y') && (* result != 'n')) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  do_quit = (* result == 'y');
  etpan_app_quit_subapp(input_app);
  
  if (do_quit)
    etpan_app_quit_subapp(app);
}


static void ask_quit(struct etpan_subapp * app)
{
  struct app_state * state;
  struct etpan_subapp * input;
  int r;
  
  input = etpan_app_find_subapp(app->app, "search-input",
      0, NULL, NULL);
  if (input == NULL) {
    input = etpan_search_input_new(app->app);
    if (input == NULL)
      goto err;
  }
  
  r = etpan_search_input_set(input,
      "close message (y/n) ? ", 1,
      NULL, 0,
      ask_quit_upcall, app);
  if (input == NULL)
    goto err;
  
  etpan_subapp_set_parent(input, app);
  etpan_app_switch_subapp(input, 0);
  
  return;
  
 err:
  ETPAN_APP_LOG((app->app, "compose - not enough memory"));
}


#define HELP_TEXT \
"\
Help for message compose\n\
------------------------\n\
\n\
The application to compose the message will let you\n\
edit the main header of the message. When you have\n\
finished, you press 'y' and you will edit the remaining\n\
parts of the message.\n\
\n\
- Ctrl-W,\n\
  Ctrl-X     : switch between applications\n\
\n\
- arrow keys : move cursor\n\
\n\
- f          : adds From field   \n\
- t          : adds To field\n\
- c          : adds Cc field\n\
- b          : adds Bcc field\n\
- r          : adds Reply-To field\n\
- F          : adds Resent-From field\n\
- T          : adds Resent-To field\n\
- C          : adds Resent-Cc field\n\
- B          : adds Resent-Bcc field\n\
- s          : adds subject field\n\
- o          : adds custom field\n\
\n\
- [Enter]    : edit current field\n\
- d          : delete current field\n\
\n\
- y          : finished\n\
- Ctrl-G     : cancel\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
